/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2005 Takuro Ashie
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#   include <config.h>
#endif

#include <glib/gi18n-lib.h>

#include "tomoe-handwriting.h"
#include "tomoe-canvas.h"
#include "tomoe-char-table.h"

enum {
    SELECTED_SIGNAL,
    LAST_SIGNAL
};

enum {
    PROP_0,
    PROP_TOMOE_CONTEXT,
    PROP_CANVAS,
    PROP_BUTTON_AREA,
    PROP_CHAR_TABLE
};

typedef struct _TomoeHandwritingPrivate	TomoeHandwritingPrivate;
struct _TomoeHandwritingPrivate
{
    TomoeContext *context;

    GtkWidget *canvas;
    GtkWidget *button_area;
    GtkWidget *find_button;
    GtkWidget *go_back_button;
    GtkWidget *clear_button;
    GtkWidget *normalize_button;
    GtkWidget *candidates_view;
};

#define TOMOE_HANDWRITING_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_HANDWRITING, TomoeHandwritingPrivate))

G_DEFINE_TYPE (TomoeHandwriting, tomoe_handwriting, GTK_TYPE_TABLE)

static void   dispose                       (GObject             *object);
static void   set_property                  (GObject             *object,
                                             guint                prop_id,
                                             const GValue        *value,
                                             GParamSpec          *pspec);
static void   get_property                  (GObject             *object,
                                             guint                prop_id,
                                             GValue              *value,
                                             GParamSpec          *pspec);

static void tomoe_handwriting_set_sensitive (TomoeHandwriting *handwriting);

/* callbacks for child widgets */
static void on_canvas_stroke_added          (TomoeCanvas         *canvas,
                                             gpointer             user_data);
static void on_find_button_clicked          (GtkButton           *button,
                                             gpointer             user_data);
static void on_go_back_button_clicked       (GtkButton           *button,
                                             gpointer             user_data);
static void on_clear_button_clicked         (GtkButton           *button,
                                             gpointer             user_data);
static void on_normalize_button_clicked     (GtkButton           *button,
                                             gpointer             user_data);
#ifdef ENABLE_DUMPSTROKE
static void on_dump_button_clicked          (GtkButton           *button,
                                             gpointer             user_data);
#endif
static void on_candidate_selected           (TomoeCharTable      *table,
                                             gpointer             user_data);

static guint handwriting_signals[LAST_SIGNAL] = { 0 };

static void
tomoe_handwriting_class_init (TomoeHandwritingClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
    GParamSpec *spec;

    gobject_class->dispose      = dispose;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;
    klass->selected             = NULL;

    handwriting_signals[SELECTED_SIGNAL] =
      g_signal_new ("selected",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeHandwritingClass, selected),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);

    spec = g_param_spec_object (
        "tomoe-context",
        N_("Tomoe context"),
        N_("A TomoeContext which stores handwriting dictionaries."),
        TOMOE_TYPE_CONTEXT, G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY);
    g_object_class_install_property (gobject_class, PROP_TOMOE_CONTEXT, spec);

    spec = g_param_spec_object (
        "canvas",
        N_("TomoeCanvas"),
        N_("The TomoeCavas widget inside of the TomoeHandwriting container."),
        GTK_TYPE_WIDGET, G_PARAM_READABLE);
    g_object_class_install_property (gobject_class, PROP_CANVAS, spec);

    spec = g_param_spec_object (
        "button-area",
        N_("ButtonArea"),
        N_("The button area widget inside of the TomoeHandwriting container."),
        GTK_TYPE_WIDGET, G_PARAM_READABLE);
    g_object_class_install_property (gobject_class, PROP_BUTTON_AREA, spec);

    spec = g_param_spec_object (
        "char-table",
        N_("CharTable"),
        N_("A candidates view widget inside of the TomoeHandwriting container."),
        GTK_TYPE_WIDGET, G_PARAM_READABLE);
    g_object_class_install_property (gobject_class, PROP_CHAR_TABLE, spec);

    g_type_class_add_private (gobject_class, sizeof (TomoeHandwritingPrivate));
}

static void
tomoe_handwriting_init (TomoeHandwriting *handwriting)
{
    GtkWidget *main_vbox, *hbox, *vbox, *frame, *alignment;
    GtkWidget *canvas, *button, *table;
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    priv->context = NULL;

    gtk_table_resize (GTK_TABLE (handwriting), 1, 1);
    gtk_table_set_homogeneous (GTK_TABLE (handwriting), TRUE);

    main_vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 8);
    gtk_table_attach_defaults (GTK_TABLE (handwriting), main_vbox, 
                               0, 1, 0, 1);
    gtk_widget_show (main_vbox);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_vbox), hbox,
                        TRUE, TRUE, 0);
    gtk_widget_show (hbox);

    /* canvas */
    frame = gtk_frame_new (NULL);
    gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);
    gtk_widget_show (frame);

    canvas = tomoe_canvas_new ();
    priv->canvas = canvas;
    gtk_widget_set_size_request (canvas, 300, 300);
    gtk_container_add (GTK_CONTAINER (frame), canvas);
    g_signal_connect (G_OBJECT (canvas), "stroke-added",
                      G_CALLBACK (on_canvas_stroke_added),
                      (gpointer) handwriting);
    gtk_widget_show (canvas);

    /* button area */
    alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 8, 0);
    gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    vbox = gtk_vbox_new (FALSE, 0);
    priv->button_area = vbox;
    gtk_container_add (GTK_CONTAINER (alignment), vbox);
    gtk_widget_show (vbox);

    button = gtk_button_new_from_stock (GTK_STOCK_FIND);
    priv->find_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_find_button_clicked),
                      (gpointer) handwriting);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_UNDO);
    priv->go_back_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_go_back_button_clicked),
                      (gpointer) handwriting);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
    priv->clear_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_clear_button_clicked),
                      (gpointer) handwriting);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_ZOOM_FIT);
    priv->normalize_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_normalize_button_clicked),
                      (gpointer) handwriting);
    gtk_widget_show (button);
#ifdef ENABLE_DUMPSTROKE
    button = gtk_button_new_from_stock (GTK_STOCK_SAVE);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_dump_button_clicked),
                      (gpointer) handwriting);
    gtk_widget_show (button);
#endif

    /* candidates view */
    frame = gtk_frame_new (NULL);
    gtk_box_pack_start (GTK_BOX (main_vbox), frame, FALSE, FALSE, 4);
    gtk_widget_show (frame);

    table = tomoe_char_table_new ();
    priv->candidates_view = table;
    tomoe_char_table_set_canvas (TOMOE_CHAR_TABLE (table),
                                 TOMOE_CANVAS (canvas));
    g_signal_connect (G_OBJECT (table), "selected",
                      G_CALLBACK (on_candidate_selected),
                      (gpointer) handwriting);
    gtk_container_add (GTK_CONTAINER (frame), table);
    gtk_widget_show (table);

    tomoe_handwriting_set_sensitive (handwriting);
}

static void
dispose (GObject *object)
{
    TomoeHandwriting *hw = TOMOE_HANDWRITING (object);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (hw);

    if (priv->context) {
        g_object_unref (priv->context);
        priv->context = NULL;
    }

    if (G_OBJECT_CLASS(tomoe_handwriting_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_handwriting_parent_class)->dispose(object);
}

static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
    TomoeHandwriting *hw = TOMOE_HANDWRITING (object);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (hw);

    switch (prop_id) {
    case PROP_TOMOE_CONTEXT:
    {
        TomoeContext *ctx = TOMOE_CONTEXT (g_value_get_object (value));
        if (priv->context)
            g_object_unref (priv->context);
        if (ctx)
            g_object_ref (ctx);
        priv->context = ctx;
        tomoe_canvas_set_context (TOMOE_CANVAS (priv->canvas), ctx);
        break;
    }
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
    TomoeHandwriting *hw = TOMOE_HANDWRITING (object);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (hw);

    switch (prop_id) {
    case PROP_TOMOE_CONTEXT:
        g_value_set_object (value, G_OBJECT (priv->context));
        break;
    case PROP_CANVAS:
        g_value_set_object (value, G_OBJECT (priv->canvas));
        break;
    case PROP_BUTTON_AREA:
        g_value_set_object (value, G_OBJECT (priv->button_area));
        break;
    case PROP_CHAR_TABLE:
        g_value_set_object (value, G_OBJECT (priv->candidates_view));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
tomoe_handwriting_set_sensitive (TomoeHandwriting *handwriting)
{
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);
    gboolean editable = tomoe_canvas_has_stroke (TOMOE_CANVAS (priv->canvas));

    gtk_widget_set_sensitive (priv->find_button,      editable);
    gtk_widget_set_sensitive (priv->go_back_button,   editable);
    gtk_widget_set_sensitive (priv->clear_button,     editable);
    gtk_widget_set_sensitive (priv->normalize_button, editable);
}

GtkWidget *
tomoe_handwriting_new (TomoeContext *context)
{
    g_return_val_if_fail (TOMOE_IS_CONTEXT (context), NULL);
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_HANDWRITING,
                                    "tomoe-context", context,
                                    NULL));
}

GtkWidget *
tomoe_handwriting_get_canvas (TomoeHandwriting *handwriting)
{
    g_return_val_if_fail (TOMOE_IS_HANDWRITING (handwriting), NULL);

    return TOMOE_HANDWRITING_GET_PRIVATE (handwriting)->canvas;
}

GtkWidget *
tomoe_handwriting_get_button_area (TomoeHandwriting *handwriting)
{
    TomoeHandwritingPrivate *priv;

    g_return_val_if_fail (TOMOE_IS_HANDWRITING (handwriting), NULL);
    priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    return priv->button_area;
}

GtkWidget *
tomoe_handwriting_get_char_table (TomoeHandwriting *handwriting)
{
    TomoeHandwritingPrivate *priv;

    g_return_val_if_fail (TOMOE_IS_HANDWRITING (handwriting), NULL);
    priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    return priv->candidates_view;
}

const gchar *
tomoe_handwriting_get_selected_char (TomoeHandwriting *handwriting)
{
    TomoeHandwritingPrivate *priv;
    TomoeChar *c;

    g_return_val_if_fail (TOMOE_IS_HANDWRITING (handwriting), NULL);
    priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    c = tomoe_char_table_get_selected (TOMOE_CHAR_TABLE (priv->candidates_view));
    if (!c) return NULL;

    return tomoe_char_get_utf8 (c);
}

TomoeChar *
tomoe_handwriting_get_selected_tomoe_char (TomoeHandwriting *handwriting)
{
    TomoeHandwritingPrivate *priv;

    g_return_val_if_fail (TOMOE_IS_HANDWRITING (handwriting), NULL);
    priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    return tomoe_char_table_get_selected (TOMOE_CHAR_TABLE (priv->candidates_view));
}

static void
on_canvas_stroke_added (TomoeCanvas *canvas, gpointer user_data)
{
    TomoeHandwriting *handwriting = TOMOE_HANDWRITING (user_data);

    tomoe_handwriting_set_sensitive (handwriting);
}

static void
on_find_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeHandwriting *handwriting = TOMOE_HANDWRITING (user_data);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    g_return_if_fail (TOMOE_IS_HANDWRITING (handwriting));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_find (TOMOE_CANVAS (priv->canvas));
}

static void
on_go_back_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeHandwriting *handwriting = TOMOE_HANDWRITING (user_data);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    g_return_if_fail (TOMOE_IS_HANDWRITING (handwriting));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_revert_stroke (TOMOE_CANVAS (priv->canvas));
    tomoe_canvas_find (TOMOE_CANVAS (priv->canvas));
    tomoe_handwriting_set_sensitive (handwriting);
}

static void
on_clear_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeHandwriting *handwriting = TOMOE_HANDWRITING (user_data);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    g_return_if_fail (TOMOE_IS_HANDWRITING (handwriting));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_clear (TOMOE_CANVAS (priv->canvas));
    tomoe_handwriting_set_sensitive (handwriting);
}

static void
on_normalize_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeHandwriting *handwriting = TOMOE_HANDWRITING (user_data);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);

    g_return_if_fail (TOMOE_IS_HANDWRITING (handwriting));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_normalize (TOMOE_CANVAS (priv->canvas));
    tomoe_handwriting_set_sensitive (handwriting);
}

static void
on_candidate_selected (TomoeCharTable *table,
                       gpointer user_data)
{
    TomoeHandwriting *handwriting = TOMOE_HANDWRITING (user_data);

    g_return_if_fail (TOMOE_IS_HANDWRITING (handwriting));

    g_signal_emit (G_OBJECT (handwriting),
                   handwriting_signals[SELECTED_SIGNAL], 0);
    tomoe_handwriting_set_sensitive (handwriting);
}

#ifdef ENABLE_DUMPSTROKE
static void
on_dump_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeHandwriting *handwriting = TOMOE_HANDWRITING (user_data);
    TomoeHandwritingPrivate *priv = TOMOE_HANDWRITING_GET_PRIVATE (handwriting);
    TomoeWriting *writing;
    GtkWidget *dialog;
    GList *strokes, *list;
    GList *candidates;
    GString *dump_string;
    gchar *filename;

    g_return_if_fail (TOMOE_IS_HANDWRITING (handwriting));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    dialog = gtk_file_chooser_dialog_new (_("Select the file name for dump"),
                                          NULL,
                                          GTK_FILE_CHOOSER_ACTION_SAVE,
                                          GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                          NULL);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_ACCEPT) {
        gtk_widget_destroy (dialog);
        return;
    }
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    gtk_widget_destroy (dialog);

    dump_string = g_string_new (NULL);
    writing = tomoe_canvas_get_writing (TOMOE_CANVAS (priv->canvas));

    /* set candidate data */
    candidates = (GList *) tomoe_canvas_get_candidates (TOMOE_CANVAS (priv->canvas));
    for (list = candidates; list; list = g_list_next (list)) {
        TomoeCandidate *cand = TOMOE_CANDIDATE (list->data);
        TomoeChar *c = tomoe_candidate_get_char (cand);
        if (list != candidates) 
            g_string_append_c (dump_string, ' ');
        g_string_append (dump_string, tomoe_char_get_utf8 (c));
    }
    g_string_append_c (dump_string, '\n');

    /* set stroke data */
    strokes = (GList *) tomoe_writing_get_strokes (writing);
    for (list = strokes; list; list = g_list_next (list)) {
        GList *points = (GList *) list->data;
        GList *point;

        for (point = points; point; point = g_list_next (point)) {
            TomoePoint *p = (TomoePoint *) point->data;
            if (point != points)
                g_string_append (dump_string, ", ");
            g_string_append_printf (dump_string, "%d %d", p->x, p->y);
        }
        g_string_append_c (dump_string, '\n');
    }

    g_file_set_contents (filename, dump_string->str, dump_string->len, NULL);

    g_object_unref (writing);
    g_string_free (dump_string, TRUE);
    g_free (filename);
}
#endif

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
