
/*

    memory.c - memory histograms for dock applet
    Copyright (C) 2002-2004 Mario Pascucci <ilpettegolo (a) yahoo it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    or
    visit the URL http://www.fsf.org/

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dockhelper.h"
#include "memory.h"



/* reads memory and swap status from /proc/meminfo */
void mem_update(struct memory_monitor *mmon)
{
  
  FILE *f;
  char s[255];
  
  if (!(f = fopen ("/proc/meminfo", "r")))
  {
    fprintf(stderr, "Open /proc/meminfo failed.");
    return;
  }
  /* assumes all values in kB, as from procps utils */
  while (!feof (f))
  {
    fgets (s, 254, f);
    if (strstr (s, "MemTotal: "))
    {
      sscanf (s, "MemTotal: %lu", &(mmon->total));
      continue;
    }
    if (strstr (s, "MemFree: "))
    {
      sscanf (s, "MemFree: %lu", &(mmon->mpfree));
      continue;
    }
    if (strstr (s, "MemShared: "))
    {
      sscanf (s, "MemShared: %lu", &(mmon->shared));
      continue;
    }
    if (strstr (s, "Buffers: "))
    {
      sscanf (s, "Buffers: %lu", &(mmon->buffers));
      continue;
    }
    if (strstr (s, "Cached: "))
    {
      sscanf (s, "Cached: %lu", &(mmon->cache));
      continue;
    }
    if (strstr (s, "SwapTotal: "))
    {
      sscanf (s, "SwapTotal: %lu", &(mmon->swap));
      continue;
    }
    if (strstr (s, "SwapFree: "))
    {
      sscanf (s, "SwapFree: %lu", &(mmon->paged));
      continue;
    }
    
  }

  /* the used size is a grand total, here compute the resident size */
  mmon->used = mmon->total - mmon->buffers - mmon->cache - 
	mmon->mpfree - mmon->shared;

  /* calc the used swap from total - free to avoid change meaning of var */
  mmon->paged = mmon->swap - mmon->paged;
  
  /* graph autoscale for system with huge amount of RAM (> 4G) */
  if (mmon->total >= 0x00400000)   /* if there is more RAM than 4G */
  {
    /* divide all values for 1024 */
    mmon->total >>= 10;
    mmon->used >>= 10;
    mmon->buffers >>= 10;
    mmon->shared >>= 10;
    mmon->cache >>= 10;
  }
  if (mmon->swap >= 0x00400000)   /* if there is more swap than 4G */
  {
    /* divide all values for 1024 */
    mmon->swap >>= 10;
    mmon->paged >>= 10;
  }
  fclose (f);
}



/* set size for memory histogram */
int mem_size(struct memory_monitor *mmon) {

  dh_getsize(mmon->handler, &(mmon->xm), &(mmon->ym), &(mmon->wm), &(mmon->hm));
  /* if size is less than 2x10 pixel is unusable */
  if ((mmon->hm < 2) || (mmon->wm < 10)) return -1;
  return 0;
}



/* draw memory/swap histogram */
void mem_draw(struct memory_monitor *mmon) {

  int ux,bx,sx,cx;

  ux = (mmon->used * mmon->wm) / mmon->total;
  bx = (mmon->buffers * mmon->wm) / mmon->total;
  sx = (mmon->shared * mmon->wm) / mmon->total;
  cx = (mmon->cache * mmon->wm) / mmon->total;
  /* the free amount is whole - used, don't you? */
  /* draw main memory histogram */
  dh_color(mmon->used_color);
  dh_fillrect(mmon->xm,mmon->ym,ux,mmon->hm/2-1);
  dh_color(mmon->buffers_color);
  dh_fillrect(mmon->xm+ux,mmon->ym,bx,mmon->hm/2-1);
  dh_color(mmon->shared_color);
  dh_fillrect(mmon->xm+ux+bx,mmon->ym,sx,mmon->hm/2-1);
  dh_color(mmon->cache_color);
  dh_fillrect(mmon->xm+ux+sx+bx,mmon->ym,cx,mmon->hm/2-1);
  dh_color(dh_background());
  dh_fillrect(mmon->xm+ux+bx+sx+cx,mmon->ym,mmon->wm-ux-bx-sx-cx,mmon->hm/2-1);
  /* draw swap histogram */
  if (mmon->swap == 0)    /* no swap (on some configs) */
  {
    ux = 0;
  }
  else
  {
    ux = (mmon->paged * mmon->wm) / mmon->swap;
  }
  dh_color(mmon->paged_color);
  dh_fillrect(mmon->xm,mmon->ym+mmon->hm/2+1,ux,mmon->hm/2-1);
  dh_color(dh_background());
  dh_fillrect(mmon->xm+ux,mmon->ym+mmon->hm/2+1,mmon->wm-ux,mmon->hm/2-1);
}



