/*
 * viaideinfo: Query VIA IDE controllers and display information + statistics
 *
 *     Copyright (C) 2005 Daniel Drake <dsd@gentoo.org>
 *     Copyright (c) 2000-2002 Vojtech Pavlik
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdint.h>
#include <pci/pci.h>
#include <sys/io.h>
#include <getopt.h>
#include "config.h"

typedef u16 word;
typedef u8 byte;

/* Configuration space elements */
#define VIA_IDE_ENABLE		0x40
#define VIA_IDE_CONFIG		0x41
#define VIA_MISC_1		0x44
#define VIA_MISC_3		0x46
#define VIA_DRIVE_TIMING	0x48
#define VIA_ADDRESS_SETUP	0x4c
#define VIA_UDMA_TIMING		0x50
#define VIA_8BIT_TIMING		0x4e

/* ISA Bridge flags */
#define VIA_UDMA		0x007
#define VIA_UDMA_NONE		0x000
#define VIA_UDMA_33		0x001
#define VIA_UDMA_66		0x002
#define VIA_UDMA_100		0x003
#define VIA_UDMA_133		0x004
#define VIA_BAD_PREQ		0x010	/* Crashes if PREQ# till DDACK# set */
#define VIA_BAD_CLK66		0x020	/* 66 MHz clock doesn't work correctly */
#define VIA_SET_FIFO		0x040	/* Needs to have FIFO split set */
#define VIA_NO_UNMASK		0x080	/* Doesn't work with IRQ unmasking on */
#define VIA_BAD_ID		0x100	/* Has wrong vendor ID (0x1107) */
#define VIA_BAD_AST		0x200	/* Don't touch Address Setup Timing */

/* PCI ID's */
#define PCI_VENDOR_ID_VIA               0x1106
#define PCI_DEVICE_ID_VIA_82C576_1      0x1571
#define PCI_DEVICE_ID_VIA_82C586_1      0x0571
#define PCI_DEVICE_ID_VIA_8237          0x3227
#define PCI_DEVICE_ID_VIA_8237A         0x3337
#define PCI_DEVICE_ID_VIA_8251		0x3287
#define PCI_DEVICE_ID_VIA_8235          0x3177
#define PCI_DEVICE_ID_VIA_8233A         0x3147
#define PCI_DEVICE_ID_VIA_8233C_0       0x3109
#define PCI_DEVICE_ID_VIA_8233_0        0x3074
#define PCI_DEVICE_ID_VIA_82C686        0x0686
#define PCI_DEVICE_ID_VIA_8231          0x8231
#define PCI_DEVICE_ID_VIA_82C596        0x0596
#define PCI_DEVICE_ID_VIA_82C586_0      0x0586
#define PCI_DEVICE_ID_VIA_6410		0x3164
#define PCI_DEVICE_ID_VIA_82C576        0x0576
#define PCI_DEVICE_ID_VIA_8237S		0x3372
#define PCI_DEVICE_ID_VIA_SATA_EIDE	0x5324
#define PCI_DEVICE_ID_VIA_CX700		0x8324

#define print_drive(name, format, arg...)\
	printf(name); for (i = 0; i < 4; i++) printf(format, ## arg); printf("\n");

/* VIA IDE Controllers */
static struct pci_device_id {
	word vendor;
	word device;
} via_pci_tbl[] = {
	{ PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_82C576_1 },
	{ PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_82C586_1 },
	{ PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_6410 },
	{ PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_SATA_EIDE },
	{ 0, }
};

/* VIA ISA Bridges */
static struct via_isa_bridge {
	char *name;
	word id;
	unsigned char rev_min;
	unsigned char rev_max;
	u16 flags;
} via_isa_bridges[] = {
	{ "cx700",	PCI_DEVICE_ID_VIA_CX700,    0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt8237s",	PCI_DEVICE_ID_VIA_8237S,    0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt6410",	PCI_DEVICE_ID_VIA_6410,     0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt8251",	PCI_DEVICE_ID_VIA_8251,     0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt8237",	PCI_DEVICE_ID_VIA_8237,     0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt8237a",	PCI_DEVICE_ID_VIA_8237A,    0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt8235",	PCI_DEVICE_ID_VIA_8235,     0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt8233a",	PCI_DEVICE_ID_VIA_8233A,    0x00, 0x2f, VIA_UDMA_133 | VIA_BAD_AST },
	{ "vt8233c",	PCI_DEVICE_ID_VIA_8233C_0,  0x00, 0x2f, VIA_UDMA_100 },
	{ "vt8233",	PCI_DEVICE_ID_VIA_8233_0,   0x00, 0x2f, VIA_UDMA_100 },
	{ "vt8231",	PCI_DEVICE_ID_VIA_8231,     0x00, 0x2f, VIA_UDMA_100 },
	{ "vt82c686b",	PCI_DEVICE_ID_VIA_82C686,   0x40, 0x4f, VIA_UDMA_100 },
	{ "vt82c686a",	PCI_DEVICE_ID_VIA_82C686,   0x10, 0x2f, VIA_UDMA_66 },
	{ "vt82c686",	PCI_DEVICE_ID_VIA_82C686,   0x00, 0x0f, VIA_UDMA_33 | VIA_BAD_CLK66 },
	{ "vt82c596b",	PCI_DEVICE_ID_VIA_82C596,   0x10, 0x2f, VIA_UDMA_66 },
	{ "vt82c596a",	PCI_DEVICE_ID_VIA_82C596,   0x00, 0x0f, VIA_UDMA_33 | VIA_BAD_CLK66 },
	{ "vt82c586b",	PCI_DEVICE_ID_VIA_82C586_0, 0x47, 0x4f, VIA_UDMA_33 | VIA_SET_FIFO },
	{ "vt82c586b",	PCI_DEVICE_ID_VIA_82C586_0, 0x40, 0x46, VIA_UDMA_33 | VIA_SET_FIFO | VIA_BAD_PREQ },
	{ "vt82c586b",	PCI_DEVICE_ID_VIA_82C586_0, 0x30, 0x3f, VIA_UDMA_33 | VIA_SET_FIFO },
	{ "vt82c586a",	PCI_DEVICE_ID_VIA_82C586_0, 0x20, 0x2f, VIA_UDMA_33 | VIA_SET_FIFO },
	{ "vt82c586",	PCI_DEVICE_ID_VIA_82C586_0, 0x00, 0x0f, VIA_UDMA_NONE | VIA_SET_FIFO },
	{ "vt82c576",	PCI_DEVICE_ID_VIA_82C576,   0x00, 0x2f, VIA_UDMA_NONE | VIA_SET_FIFO | VIA_NO_UNMASK },
	{ "vt82c576",	PCI_DEVICE_ID_VIA_82C576,   0x00, 0x2f, VIA_UDMA_NONE | VIA_SET_FIFO | VIA_NO_UNMASK | VIA_BAD_ID },
	{ NULL }
};

static char *via_control3[] = {
	"No limit", "64", "128", "192"
};

static char *via_dma[] = {
	"MWDMA16", "UDMA33", "UDMA66", "UDMA100", "UDMA133"
};

static struct pci_access *pacc;
static struct pci_dev *ide_ctrlr;
static struct pci_dev *isa_bridge;

static struct via_isa_bridge *via_config;
static unsigned int via_clock = 33333;

/* Exit with an error message */
static void __attribute__((noreturn)) die(char *msg, ...)
{
  va_list args;

  va_start(args, msg);
  fputs("viaideinfo: ", stderr);
  vfprintf(stderr, msg, args);
  fputc('\n', stderr);
  exit(1);
}

/* Get the 256 byte configuration space for a particular device */
static byte *get_pci_config(struct pci_dev *p)
{
	byte *config = (byte *) malloc(256);
	if (config == NULL)
		return NULL;
	
	if (!pci_read_block(p, 0, config, 256)) {
		free(config);
		return NULL;
	}

	return config;
}

/*
 * Tells you whether the specified device is one of the supported IDE
 * controllers
 */
static int is_via_ide_ctrlr(struct pci_dev *p)
{
	struct pci_device_id *dev;
	pci_fill_info(p, PCI_FILL_IDENT | PCI_FILL_BASES);

	for (dev = via_pci_tbl; dev->vendor; dev++)
		if (dev->vendor == p->vendor_id && dev->device == p->device_id)
			return 1;

	return 0;
}

/* Scans for a supported ISA bridge, and then for a supported IDE controller */
static void scan_devices(unsigned int bus, unsigned int dev, unsigned int func)
{
	struct pci_dev *p;
	struct via_isa_bridge *bridge;
	byte rev;
	int find_custom = (bus > 0 || dev > 0 || func > 0);

	pci_scan_bus(pacc);

	for (p=pacc->devices; p; p=p->next) {
		pci_fill_info(p, PCI_FILL_IDENT | PCI_FILL_BASES);
		rev = pci_read_word(p, PCI_REVISION_ID);
		for (bridge = via_isa_bridges; bridge->id; bridge++)
			if (p->vendor_id == PCI_VENDOR_ID_VIA + !!(bridge->flags & VIA_BAD_ID)
				&& p->device_id == bridge->id
				&& rev >= bridge->rev_min && rev <= bridge->rev_max)
					break;

		if (bridge->id)
			break;
	}

	if (!bridge->id)
		die("Could not identify a supported VIA ISA bridge");

	isa_bridge = p;
	via_config = bridge;
	
	for (p=pacc->devices; p; p=p->next) {
		if (find_custom) {
			if (p->bus == bus && p->dev == dev && p->func == func)
				break;
		} else if (is_via_ide_ctrlr(p))
			break;
	}

	if (!p) {
		if (find_custom)
			die("Could not find the device '%02x:%02x.%x'", bus, dev, func);
		else
			die("Could not identify a supported VIA IDE controller");
	}

	ide_ctrlr = p;
}

/* Deduces the 40 or 80-wire status */
static int get_80w_status(u32 u)
{
	int i;
	unsigned int via_80w = 0;

	switch (via_config->flags & VIA_UDMA) {
	case VIA_UDMA_66:
		for (i = 24; i >= 0; i -= 8)
			if (((u >> (i & 16)) & 8) &&
			    ((u >> i) & 0x20) &&
			     (((u >> i) & 7) < 2)) {
				/*
				 * 2x PCI clock and
				 * UDMA w/ < 3T/cycle
				 */
				via_80w |= (1 << (1 - (i >> 4)));
			}
		break;

	case VIA_UDMA_100:
		for (i = 24; i >= 0; i -= 8)
			if (((u >> i) & 0x10) ||
			    (((u >> i) & 0x20) &&
			     (((u >> i) & 7) < 4))) {
				/* BIOS 80-wire bit or
				 * UDMA w/ < 60ns/cycle
				 */
				via_80w |= (1 << (1 - (i >> 4)));
			}
		break;

	case VIA_UDMA_133:
		for (i = 24; i >= 0; i -= 8)
			if (((u >> i) & 0x10) ||
			    (((u >> i) & 0x20) &&
			     (((u >> i) & 7) < 6))) {
				/* BIOS 80-wire bit or
				 * UDMA w/ < 60ns/cycle
				 */
				via_80w |= (1 << (1 - (i >> 4)));
			}
		break;
	}

	return via_80w;
}

/* Initializes libpci */
static void init_pcilib()
{
	pacc = pci_alloc();
	pacc->error = die;
	pci_init(pacc);
}

/* Read a byte of the specified configuration data */
static inline byte read_config_byte(byte *data, int index)
{
	return data[index];
}

/* Read a word of the specified configuration data */
static inline word read_config_word(byte *data, int index)
{
	word ret;
	ret  = data[index + 1] << 8;
	ret |= data[index];
	return ret;
}

/* Read a dword of the specified configuration data */
static inline u32 read_config_dword(byte *data, int index)
{
	u32 ret;
	ret  = data[index + 3] << 24;
	ret |= data[index + 2] << 16;
	ret |= data[index + 1] << 8;
	ret |= data[index];
	return ret;
}

/* Find the resource start of the specified device */
static pciaddr_t get_resource_start(struct pci_dev *p, byte *config)
{
	int i;

	for (i = 0; i < 6; i++) {
		pciaddr_t pos = p->base_addr[i];
		pciaddr_t len = (p->known_fields & PCI_FILL_SIZES) ? p->size[i] : 0;
		u32 flg = read_config_dword(config, PCI_BASE_ADDRESS_0 + 4*i);
		if (flg == 0xffffffff)
			flg = 0;
		if (!pos && !flg && !len)
			continue;

		if (flg & PCI_BASE_ADDRESS_SPACE_IO) {
			pciaddr_t a = pos & PCI_BASE_ADDRESS_IO_MASK;
			if (a)
				return a;
		}
	}

	return 0;
}

/* Print the information about a controller */
static void show_via_info()
{
	int i;
	int speed[4], cycle[4], setup[4], active[4], recover[4], den[4],
		 uen[4], udma[4], umul[4], active8b[4], recover8b[4];

	byte *ctrlr_config, *isa_config;
	byte misc1, misc3, ide_config, rev, enable, address_setup, isa_rev;
	word timing_8bit;
	u32 drive_timing, udma_timing;

	unsigned int via_80w;
	unsigned long via_base, simplex = 0;

	ctrlr_config = get_pci_config(ide_ctrlr);
	if (ctrlr_config == NULL)
		die("Could not read PCI configuration data for VIA IDE controller");

	isa_config = get_pci_config(isa_bridge);
	if (isa_config == NULL) {
		free(ctrlr_config);
		die("Could not read PCI configuration data for VIA ISA bridge");
	}

	misc1 = read_config_byte(ctrlr_config, VIA_MISC_1);
	misc3 = read_config_byte(ctrlr_config, VIA_MISC_3);
	ide_config = read_config_byte(ctrlr_config, VIA_IDE_CONFIG);
	rev = read_config_byte(ctrlr_config, PCI_REVISION_ID);
	enable = read_config_byte(ctrlr_config, VIA_IDE_ENABLE);
	address_setup = read_config_byte(ctrlr_config, VIA_ADDRESS_SETUP);
	drive_timing = read_config_dword(ctrlr_config, VIA_DRIVE_TIMING);
	timing_8bit = read_config_word(ctrlr_config, VIA_8BIT_TIMING);
	isa_rev = read_config_byte(isa_config, PCI_REVISION_ID);

	if (via_config->flags & VIA_UDMA)
		udma_timing = read_config_dword(ctrlr_config, VIA_UDMA_TIMING);
	else
		udma_timing = 0;

	via_80w = get_80w_status(udma_timing);
	via_base = get_resource_start(ide_ctrlr, ctrlr_config);
	if (via_base != 0)
		simplex = inb(via_base + 0x02) | (inb(via_base + 0x0a) << 8);

	printf("----------VIA BusMastering IDE Configuration----------------\n");
	printf("viaideinfo Version:                 %s\n", VERSION);

	printf("South Bridge:                       VIA %s Rev 0x%x "
		"(PCI %02x:%02x.%x)\n", via_config->name, isa_rev,
		isa_bridge->bus, isa_bridge->dev, isa_bridge->func);
	printf("IDE Controller:                     Rev 0x%x (PCI %02x:%02x.%x)\n",
		rev, ide_ctrlr->bus, ide_ctrlr->dev, ide_ctrlr->func);
	printf("Highest DMA rate:                   %s\n",
		via_dma[via_config->flags & VIA_UDMA]);

	printf("BM-DMA base:                        %#lx\n", via_base);
	printf("PCI clock:                          %d.%dMHz\n",
		via_clock / 1000, via_clock / 100 % 10);

	printf("Master Read  Cycle IRDY:            %dws\n",
		(misc1 & 64) >> 6);
	printf("Master Write Cycle IRDY:            %dws\n",
		(misc1 & 32) >> 5);
	printf("BM IDE Status Register Read Retry:  %s\n",
		(misc1 & 8) ? "yes" : "no");

	printf("Max DRDY Pulse Width:               %s%s\n",
		via_control3[(misc3 & 0x03)], (misc3 & 0x03) ? " PCI clocks" : "");

	printf("-----------------------Primary IDE-------Secondary IDE------\n");
	printf("Read DMA FIFO flush:   %10s%20s\n",
		(misc3 & 0x80) ? "yes" : "no", (misc3 & 0x40) ? "yes" : "no");
	printf("End Sector FIFO flush: %10s%20s\n",
		(misc3 & 0x20) ? "yes" : "no", (misc3 & 0x10) ? "yes" : "no");

	printf("Prefetch Buffer:       %10s%20s\n",
		(ide_config & 0x80) ? "yes" : "no", (ide_config & 0x20) ? "yes" : "no");
	printf("Post Write Buffer:     %10s%20s\n",
		(ide_config & 0x40) ? "yes" : "no", (ide_config & 0x10) ? "yes" : "no");

	printf("Enabled:               %10s%20s\n",
		(enable & 0x02) ? "yes" : "no", (enable & 0x01) ? "yes" : "no");

	printf("Simplex only:          %10s%20s\n",
		(simplex & 0x80) ? "yes" : "no", (simplex & 0x8000) ? "yes" : "no");

	printf("Cable Type:            %10s%20s\n",
		(via_80w & 1) ? "80w" : "40w", (via_80w & 2) ? "80w" : "40w");

	printf("-------------------drive0----drive1"
		"----drive2----drive3-----\n");

	for (i = 0; i < 4; i++) {

		setup[i]     = ((address_setup >> ((3 - i) << 1)) & 0x3) + 1;
		recover8b[i] = ((timing_8bit >> ((1 - (i >> 1)) << 3)) & 0xf) + 1;
		active8b[i]  = ((timing_8bit >> (((1 - (i >> 1)) << 3) + 4)) & 0xf) + 1;
		active[i]    = ((drive_timing >> (((3 - i) << 3) + 4)) & 0xf) + 1;
		recover[i]   = ((drive_timing >> ((3 - i) << 3)) & 0xf) + 1;
		udma[i]      = ((udma_timing >> ((3 - i) << 3)) & 0x7) + 2;
		umul[i]      = ((udma_timing >> (((3 - i) & 2) << 3)) & 0x8) ? 1 : 2;
		uen[i]       = ((udma_timing >> ((3 - i) << 3)) & 0x20);
		den[i]       = (simplex & ((i & 1) ? 0x40 : 0x20) << ((i & 2) << 2));

		speed[i] = 2 * via_clock / (active[i] + recover[i]);
		cycle[i] = 1000000 * (active[i] + recover[i]) / via_clock;

		if (!uen[i] || !den[i])
			continue;

		switch (via_config->flags & VIA_UDMA) {

			case VIA_UDMA_33:
				speed[i] = 2 * via_clock / udma[i];
				cycle[i] = 1000000 * udma[i] / via_clock;
				break;

			case VIA_UDMA_66:
				speed[i] = 4 * via_clock / (udma[i] * umul[i]);
				cycle[i] = 500000 * (udma[i] * umul[i]) / via_clock;
				break;

			case VIA_UDMA_100:
				speed[i] = 6 * via_clock / udma[i];
				cycle[i] = 333333 * udma[i] / via_clock;
				break;

			case VIA_UDMA_133:
				speed[i] = 8 * via_clock / udma[i];
				cycle[i] = 250000 * udma[i] / via_clock;
				break;
		}
	}

	print_drive("Transfer Mode: ", "%10s",
		den[i] ? (uen[i] ? "UDMA" : "DMA") : "PIO");

	print_drive("Address Setup: ", "%8dns",
		1000000 * setup[i] / via_clock);
	print_drive("Cmd Active:    ", "%8dns",
		1000000 * active8b[i] / via_clock);
	print_drive("Cmd Recovery:  ", "%8dns",
		1000000 * recover8b[i] / via_clock);
	print_drive("Data Active:   ", "%8dns",
		1000000 * active[i] / via_clock);
	print_drive("Data Recovery: ", "%8dns",
		1000000 * recover[i] / via_clock);
	print_drive("Cycle Time:    ", "%8dns",
		cycle[i]);
	print_drive("Transfer Rate: ", "%4d.%dMB/s",
		speed[i] / 1000, speed[i] / 100 % 10);

	free(ctrlr_config);
	free(isa_config);
}

void show_usage()
{
	printf("viaideinfo version " VERSION "\n");
	printf("Display information/statistics about VIA IDE controllers\n\n");
	printf("Usage: viaideinfo [-c <value-in-mhz>] [-d <bus>:<slot>.<func>]\n\n");
	printf("  -c, --ide-clock\tUse a user-specified IDE bus speed (in MHz) for timing\n");
	printf("                 \tcalculations (default 33)\n");
	printf("  -d, --device\t\tQuery this specific PCI device as the IDE controller\n"); 
	printf("              \t\t(default autodetect)\n");
	printf("  -h, --help\t\tShow this help message\n\n");
}

int main(int argc, char **argv)
{
	char *msg;
	int c, clock, ret;
	unsigned int bus = 0;
	unsigned int dev = 0;
	unsigned int func = 0;

	while (1) {
		static struct option long_options[] = {
			{"ide-clock", required_argument, 0, 'c'},
			{"device", required_argument, 0, 'd'},
			{"help", no_argument, 0, 'h'},
			{0, 0, 0, 0}
		};
		/* getopt_long stores the option index here. */
		int option_index = 0;

		c = getopt_long(argc, argv, "c:d:h", long_options, &option_index);

		/* Detect the end of the options. */
		if (c == -1)
			break;

		switch (c) {
		case 'c':
			clock = atoi(optarg);
			switch (clock) {
			case 33:
				via_clock = 33333; break;
			case 37:
				via_clock = 37500; break;
			case 41:
				via_clock = 41666; break;
			default:
				via_clock = clock * 1000;
			}

			if (via_clock < 20000 || via_clock > 50000) {
				printf("PCI clock speed %dMHz is impossible, using 33MHz "
					"instead.\n", clock);
				via_clock = 33333;
			}
			break;

		case 'd':
			ret = sscanf(optarg, "%x:%x.%x", &bus, &dev, &func);
			if (ret != 3)
				die("Could not parse device '%s'", optarg);
			break;

		case 'h':
			show_usage();
			return 0;

		case '?':
			/* getopt_long already printed an error message. */
			break;

		default:
			abort ();
		}
	}

	if (iopl(3) != 0) {
		perror("iopl");
		die("Ensure that you are running with root privileges");
	}

	init_pcilib();

	scan_devices(bus, dev, func);
	show_via_info();

	pci_cleanup(pacc);
}

