/*-*- c++ -*-******************************************************************
**
**  Copyright (c) 1999,2000 Red Hat, Inc. Harald Hoyer (Harald.Hoyer@RedHat.de)
**
**  $Id: autorun.cc,v 1.5 2005/04/19 16:34:27 saturn_de Exp $
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
******************************************************************************/

#include "config.h"

#ifdef HAVE_ARGP_PARSE
#define HAVE_ARGP
#endif

#if STDC_HEADERS
# include <string.h>
#else
# if !HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# endif
char *strchr (), *strrchr ();
# if !HAVE_MEMCPY
#  define memcpy(d, s, n) bcopy ((s), (d), (n))
#  define memmove(d, s, n) bcopy ((s), (d), (n))
# endif
#endif

#include <sys/types.h>
#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#include <unistd.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <sys/file.h>
#include <fcntl.h>

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#include <stdio.h>
#include <stdarg.h>
#include <mntent.h>
#include "cdrom.h"
#include <list>
#include <iostream>
#include <libintl.h>

#ifdef HAVE_ARGP_H
#include <argp.h>
#elif HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <errno.h>

#ifndef MNTTYPE_ISO9660
#define MNTTYPE_ISO9660 "iso9660"
#endif

#ifndef MNTTYPE_UDF
#define MNTTYPE_UDF "udf"
#endif

#ifndef MNTTYPE_AUTO
#define MNTTYPE_AUTO "auto"
#endif

#ifndef MNTTYPE_SUPERMOUNT
#define MNTTYPE_SUPERMOUNT "supermount"
#endif
#define SUPER_OPT_DEV "dev="

#define _(String) strdup(gettext (String))

typedef enum {
  not_avail,
  not_execed,
  execed,
  audio
} execmode_t;
      

struct autocd {
  CDRom *cdrom;
  string mountpath;
  string autopath;
  string audioplayer;
  execmode_t exec_mode;
  bool is_supermount;
};

/* Used by `main' to communicate with `parse_opt'. */
struct arguments
{
  bool silent;
  bool verbose;
  bool allow_eject;
  bool mount_only;
  bool daemon;
  long check_interval;
  string autorun_path;
  string notify_path;
  string notify_insert_path;
  string notify_eject_path;
  string audiocd_player;
  list<autocd*> autolist;
};

bool addDev(struct arguments *arguments, char *arg)
{
  struct mntent * mep=0;
  FILE* vt;
  string devpath(arg);
  char lpath[PATH_MAX];
  struct stat sbuf;
  bool is_supermount = false;

  int len = readlink(arg, lpath, PATH_MAX);
  if( len > 0 && len < PATH_MAX) {
    lpath[len] = 0;
  }

  if ((vt = setmntent("/etc/fstab", "r")) == NULL) 
    ::perror("setmntent");
  else {
    while((mep = ::getmntent(vt)) != NULL) {
      if((::strcmp(devpath.c_str(), mep->mnt_fsname) != 0) &&
	 (len && (::strcmp(lpath, mep->mnt_fsname) != 0)))
	continue;

      stat(mep->mnt_fsname, &sbuf);
      if ((((::strstr(mep->mnt_opts, "owner") || 
	    (::strstr(mep->mnt_opts, "pamconsole"))) && 
	    (sbuf.st_mode & S_IRUSR)) || 
	   ::strstr(mep->mnt_opts, "user")) &&
	  (::strstr(mep->mnt_type, MNTTYPE_ISO9660) ||
	   ::strstr(mep->mnt_type, MNTTYPE_UDF) ||
	   ::strstr(mep->mnt_type, MNTTYPE_AUTO)))
	break;	

      if(::strstr(mep->mnt_opts, MNTTYPE_ISO9660) && 
	 ::strstr(mep->mnt_type, MNTTYPE_SUPERMOUNT)) {
	is_supermount = true;
	if(arguments->verbose)
	  printf(_("Supermount entry found at %s.\n"), mep->mnt_dir);
	break;
      }

    }
    ::endmntent(vt);
  }

  if(mep) {
    autocd *cd = new autocd;
    if(!is_supermount) {
      cd->cdrom = new CDRom(mep->mnt_fsname);
    }
    else {
      char *dev;
      int len = 0;
      dev = ::strstr(mep->mnt_opts, SUPER_OPT_DEV);
      dev += sizeof(SUPER_OPT_DEV)-1;

      while(dev[len] != ',' && dev[len] != ' ' && dev[len] != 0)
	len++;
      if(dev[len] != 0) {
	char devstr[len+1];
	strncpy(devstr, dev, len);
	devstr[len] = 0;
	cd->cdrom = new CDRom(devstr);
      } else
	return false;
    }

    if (cd->cdrom->test() == -1) {
      return false;
    }
    cd->mountpath = mep->mnt_dir;
    cd->autopath = cd->mountpath + arguments->autorun_path;
    cd->audioplayer = arguments->audiocd_player;
    cd->exec_mode = (::strstr(mep->mnt_opts, "exec") == 0)
      ? not_avail : not_execed;
    cd->cdrom->allowEject = arguments->allow_eject;
    cd->is_supermount = is_supermount;
    arguments->autolist.push_back(cd);
    return true;
  }
  return false;
}

const char *argp_program_version = PACKAGE "-" VERSION ;
const char *argp_program_bug_address = "http://bugzilla.redhat.com";

/* A description of the arguments we accept. */
static char args_doc[] = "[cdromdevices...]";

#ifndef HAVE_ARGP
struct argp_option
{
  const char *name;
  int key;
  const char *arg;
  int flags;
  const char *doc;
  int group;
};

struct argp {
  const char *doc;
  struct argp_option *options;
} argp;
  

#endif
     
#define XTOSTR(a) TOSTR(a)
#define TOSTR(a) #a
     
#ifdef HAVE_ARGP

/* Parse a single option. */
static error_t
parse_opt (int key, char *arg, struct argp_state *state)
{
  /* Get the INPUT argument from `argp_parse', which we
     know is a pointer to our arguments structure. */
  struct arguments *arguments = (struct arguments *)state->input;
   
  switch (key)
    {
    case 'q': case 's':
      arguments->silent = true;
      break;
    case 'v':
      arguments->verbose = true;
      break;
    case 'l':
      arguments->allow_eject = false;
      break;
    case 'a':
      arguments->autorun_path = arg;
      break;
    case 'm':
      arguments->mount_only = true;
      break;
    case 'n':
      arguments->notify_path = arg;
      break;
    case 't':
      arguments->notify_insert_path = arg;
      break;
    case 'e':
      arguments->notify_eject_path = arg;
      break;
    case 'c':
      arguments->audiocd_player = arg;
      break;
    case 'i':
      arguments->check_interval = atoi(arg) * 1000;
      break;
   case 'd':
      arguments->daemon = true;
      break;

    case ARGP_KEY_ARG:
      if(!addDev(arguments, arg)) {
	fprintf(stderr, _("%s is not in /etc/fstab or the user/owner option is not set.\n"), arg);
	::argp_usage (state);
      }
      break;
    case ARGP_KEY_END:
      break;
      
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}
     
/* Our argp parser. */
static struct argp argp = { 0, parse_opt, args_doc, };

#else

void print_usage(char *name)
{
  fprintf(stderr, _("Usage: %s  [OPTION...] %s\n"), name, args_doc);
  cerr << argp.doc << endl << endl;
  
  for(int i = 0; argp.options[i].name != 0; i++)
    {
      cerr << "  -" << (char)argp.options[i].key << ", --" << argp.options[i].name;
      if(argp.options[i].arg)
	cerr << "=" << argp.options[i].arg;
      else
	cerr << "\t";
      cerr << "\t " << argp.options[i].doc << endl;
    }
  cerr << _("  -?, --help\t\t Give this help list") << endl
       << _("      --usage\t\t Give a short usage message") << endl
       << _("  -V, --version\t\t Print program version") << endl
       << endl;
  
  cerr << _("Mandatory or optional arguments to long options are also mandatory " \
	    "or optional\nfor any corresponding short options.") << endl;
  fprintf(stderr, _("Report bugs to %s.\n"), argp_program_bug_address);
	
}

void parse_opts (int argc, char **argv, struct arguments *arguments)
{
  if (argc > 1) {
    while (1) {
      int oi = 0, c = 0;
	 
      static struct option long_options[] =
	{
	  {"help",  0, 0, '?'},
	  {"usage",  0, 0, '?'},
	  {"version",  0, 0, 'V'},
	  {"verbose",  0, 0, 'v'},
	  {"quiet",    0, 0, 'q'},
	  {"autorun",  1, 0, 'a'},
	  {"mountonly",  0, 0, 'm'},
	  {"daemon",  0, 0, 'd'},
	  {"notify",  1, 0, 'n'},
	  {"notify-insert",  1, 0, 't'},
	  {"notify-eject",  1, 0, 'e'},
	  {"lock",  0, 0, 'l'},
	  {"cdplayer", 1, 0, 'c'},
	  {"interval", 1, 0, 'i'},
	  {0,}
	};


      c = ::getopt_long (argc, argv, "hVvqsln:t:e:a:c:i:", long_options, &oi);
      if (c == -1)
	break;
	 
      switch (c) {
      case 'q': case 's':
	arguments->silent = true;
	break;
      case 'v':
	arguments->verbose = true;
	break;
      case 'a':
	arguments->autorun_path = optarg;
	break;
      case 'n':
	arguments->notify_path = optarg;
	break;
      case 't':
	arguments->notify_insert_path = optarg;
	break;
      case 'e':
	arguments->notify_eject_path = optarg;
	break;
      case 'c':
	arguments->audiocd_player = optarg;
	break;
      case 'l':
	arguments->allow_eject = false;
	break;
      case 'm':
	arguments->mount_only = true;
	break;
      case 'd':
	arguments->daemon = true;
	break;
      case 'i':
	arguments->check_interval = atoi(optarg) * 1000;
	break;
      case '?':
      default:
	print_usage(argv[0]);
	exit(-1);
	break;
      }
    }
    if (optind < argc)
      {
	for (;optind < argc;optind++) {
	  if(!addDev(arguments, argv[optind])) {
	    fprintf(stderr, _("%s is not in /etc/fstab or the user/owner option is not set.\n"), argv[optind]);
	    print_usage(argv[0]);
	    ::exit(0);
	  }
	}
      }
  }
}

#endif

int exec_string(string& exec_str, autocd *cdp)
{
  CDRom *cdrom = cdp->cdrom;
  pid_t pid;
  if(exec_str.length()) {     
    pid = ::fork();
    if(pid == 0) {

      cdrom->close();
      string::size_type ppos;

      while((ppos = exec_str.find("%P%")) != string::npos) {
	// replace %P% with (mount-)_P_ath
	exec_str.replace(ppos, 3, cdp->mountpath);
      }
      while((ppos = exec_str.find("%D%")) != string::npos ) {
	exec_str.replace(ppos, 3, cdrom->devpath);
      }

      //	 cerr << "Executing: " << exec_str.c_str() << endl;

      char *argv[4];
      argv[0] = "/bin/sh";
      argv[1] = "-c";
      argv[2] = new char[exec_str.length()+1];
      argv[3] = 0;
      strcpy(argv[2], exec_str.c_str());
      ::execv(argv[0], argv);
      cerr << "Could not execute " << exec_str << endl;
      ::exit(1);
    }

    return pid;
  }
  return 0;
}

static char *lockfile_path = NULL;
static int lockfile_fd = 0;

void unlock_file()
{
  if (lockfile_fd) {
    flock(lockfile_fd, LOCK_UN);
    close(lockfile_fd);
  }
  if (lockfile_path)
    unlink(lockfile_path);	  
}

void lock_file()
{
  const char *lockfile="/.autorun.lck";
  lockfile_path = getenv("HOME");

  if (! lockfile_path) {
    fprintf(stderr, _("Environment variable $HOME is not set!\n"));
    perror("getenv");
    exit(10);
  }

  lockfile_path = strdup(lockfile_path);
  lockfile_path = (char *)realloc(lockfile_path, strlen(lockfile_path) + strlen(lockfile) + 1);

  strcat(lockfile_path, lockfile);

  lockfile_fd = open(lockfile_path, O_CREAT|O_RDONLY, 0600);

  if (lockfile_fd == -1) {
    fprintf(stderr, _("cannot open %s!\n"), lockfile_path);
    perror("open");
    exit(10);
  }

  if (flock(lockfile_fd, LOCK_EX|LOCK_NB)) {
    if(errno == EAGAIN)
      fprintf(stderr, _("autorun is already running!\n"));
    else
      perror("flock");
    exit(10);
  }
}

char *dupsnprintf(const char *format, ...)
{
  va_list va;
  char buf[2048];
  va_start(va, format);
  buf[2047] = 0;
  vsnprintf(buf, 2047, format, va);
  va_end(va);
  return strdup(buf);
}

int main (int argc, char **argv)
{
  struct arguments arguments;
  typedef list<autocd*>::iterator ALI;
  int audiopid=0;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  argp.doc = dupsnprintf(_("%s-%s -- the CD-ROM mounter for the lazy user\n\n"
			   "Command strings are parsed and %%P%% is replaced by the mountpoint path.\n"
			   "%%D%% is replaced by the device path. After that the command string will \n"
			   "be executed using /bin/sh -c \"command string\".\n%s"
			   "Copyright (c) 1999-2002 Red Hat, Inc.\n"
			   "Copyright (c) 1998-2002 Harald Hoyer (Harald.Hoyer@RedHat.de).\n\n"
			   "%s comes with ABSOLUTELY NO WARRANTY and is free software, covered\n"
			   "by the GNU General Public License, and you are welcome to change it\n"
			   "and/or distribute copies of it under certain conditions described by the\n"
			   "file COPYING coming with the source code."), 
			 PACKAGE, VERSION, "\v", PACKAGE);

  /* The options we understand. */
  struct argp_option options[] = {
    {"autorun",  'a', "STRING",  0,
     dupsnprintf(_("Command STRING will be executed if the CD changes (default \"%s\")"), AUTORUN_PATH)
    },
    {"cdplayer",'c',"STRING",  0, 
     dupsnprintf(_("Command STRING will be executed if an audio CD is inserted (default \"%s\")"), AUDIOCD_PLAYER)
    },
    {"interval", 'i', "MILLISEC", 0, 
     dupsnprintf(_("The time to wait in MILLISEC between checks (default \"%s\")"), XTOSTR(CHECK_INTERVAL))
    },
    {"lock",     'l', 0,       0,  
     _("Lock the mounted media (unmount by hand)")
    },
    {"notify",   'n', "STRING",  0,
     dupsnprintf(_("Command STRING will be executed if the CD changes (insert and eject) (default \"%s\")"), NOTIFY_PATH)
    },
    {"notify-insert",   't', "STRING",  0,
     dupsnprintf(_("Command STRING will be executed if a CD is inserted (default \"%s\")"), NOTIFY_PATH_INSERT)
    },
    {"notify-eject",   'e', "STRING",  0,
     dupsnprintf(_("Command STRING will be executed if a CD is ejected (default \"%s\")"), NOTIFY_PATH_EJECT)
    },
    {"mountonly", 'm', 0,       0,  
     _("Mount/unmount only (do not execute anything)") },
    {"daemon",   'd', 0,       0,  _("Fork to background immediately")},
    {"quiet",    'q', 0,       0,  _("Don't produce any output") },
    {"verbose",  'v', 0,       0,  _("Produce verbose output") },
    { 0 }
  };

  argp.options = options;
  argp.argp_domain = PACKAGE;

  lock_file();
  ::atexit(unlock_file);

  /* Default values. */
  arguments.check_interval = CHECK_INTERVAL * 1000;
  arguments.autorun_path = AUTORUN_PATH;
  arguments.notify_path = NOTIFY_PATH;
  arguments.notify_insert_path = NOTIFY_PATH_INSERT;
  arguments.notify_eject_path = NOTIFY_PATH_EJECT;
  arguments.audiocd_player = AUDIOCD_PLAYER;
  arguments.verbose=false;
  arguments.silent=true;
  arguments.allow_eject = false;
  arguments.mount_only = false;
  arguments.daemon = false;

#ifdef HAVE_ARGP
  /* Parse our arguments; every option seen by `parse_opt' will
     be reflected in `arguments'. */
  ::argp_parse (&argp, argc, argv, 0, 0, &arguments);
#else
  parse_opts(argc, argv, &arguments);
#endif
   
  if (arguments.daemon) {
    int i=fork();
    if (i<0) exit (1); /* fork error */
    if (i>0) exit (0); /* parent dies */
    /* child continues */
  }
 
  if(arguments.autolist.empty()) {
    struct mntent * mep;
    FILE* vt;

    if ((vt = ::setmntent("/etc/fstab", "r")) == NULL) 
      ::perror("setmntent");
    else {
      while((mep = getmntent(vt))!=NULL) {
	addDev(&arguments, mep->mnt_fsname);
      }
      ::endmntent(vt);
    }
  }


  for(;;) {
     
    if(arguments.autolist.empty()) {
      if(arguments.verbose)
	cout << _("No more devices to check.") << endl; 
      exit(0);
    }

    for(ALI cdi = arguments.autolist.begin(); 
	cdi != arguments.autolist.end(); ++cdi) {	 

      autocd *cdp(*cdi);

      if(arguments.verbose)
	printf(_("Checking %s: "), cdp->mountpath.c_str());

      switch( cdp->cdrom->test() ) {
      case -1:
	/* remove it */
	arguments.autolist.erase(cdi);
	--cdi;
	if(arguments.verbose)
	  printf(_("removing device\n"));
	/* fall through */
      case 0:
	/* try next device */
	continue;
      default:
	/* all ok */
	break;
      }

      if(!cdp->cdrom->isOpen())
	cdp->cdrom->open();

      CDRom::cdstatus_t oldstatus = cdp->cdrom->cdstatus;

      CDRom::cdstatus_t newstatus = cdp->cdrom->checkMedia();

      if(cdp->is_supermount && oldstatus != newstatus) {
	DIR *dir;
	if((dir = opendir(cdp->mountpath.c_str())) != 0)
	  closedir(dir);
      }

      if(oldstatus != newstatus) {
	if(arguments.verbose)  
	  printf(_("disc status changed, executing: %s\n"), 
		  arguments.notify_path.c_str());
	exec_string(arguments.notify_path, cdp);
      }

      switch(newstatus) {
      case CDRom::tray_is_open:
	if(arguments.verbose)
	  cout << _("tray is open") << endl;
	if(oldstatus == CDRom::mounted) {
	  if(! cdp->is_supermount)
	    ::system(string(string("umount ") + cdp->mountpath).c_str());
	}
	if(oldstatus != CDRom::tray_is_open) {
	  if(! arguments.mount_only) {
	    if(arguments.verbose)
	      printf(_("disc ejected, executing: %s\n"),
		       arguments.notify_eject_path.c_str());
	    exec_string(arguments.notify_eject_path, cdp);
	  }
	}	 
	cdp->exec_mode = not_execed;
	break;
      case CDRom::unknown:
	if(arguments.verbose)
	  cout << _("unknown state") << endl;
	if(oldstatus == CDRom::mounted) {
	  if(! cdp->is_supermount)
	    ::system(string(string("umount ") + cdp->mountpath).c_str());
	}
	if(oldstatus != CDRom::tray_is_open) {
	  if(! arguments.mount_only && oldstatus != CDRom::unknown) {
	    if(arguments.verbose)
	      printf(_("disc ejected, executing: %s\n"),
		     arguments.notify_eject_path.c_str());
	    exec_string(arguments.notify_eject_path, cdp);
	  }
	}	 
	cdp->exec_mode = not_execed;
	break;
      case CDRom::unmounted:
	if(arguments.verbose)
	  cout << _("unmounted") << endl;
	if(cdp->exec_mode == not_execed) {
	  if(!cdp->is_supermount) {
	    if(arguments.verbose)
	      cout << _("mounting ");
	    if(cdp->cdrom->mount()) {
	      if(arguments.verbose)
		cout << _("OK") << endl;
	    }
	    else {
	      if(arguments.verbose)
		cout << _("error") << endl;
	    }
	    //		 cdp->exec_mode = execed;
	  }
	}
	break;
      case CDRom::mounted:
	if(arguments.verbose)
	  cout << _("mounted") << endl;
	   
	struct stat buf;
	   
	if(cdp->exec_mode == not_execed) {
	  if(!arguments.mount_only) {
	    if(::stat(cdp->autopath.c_str(), &buf) == 0) {
	      exec_string(cdp->autopath, cdp);
	    }
	    else {
	      if(arguments.verbose)
		printf(_("disc inserted, executing: %s\n"),
		  arguments.notify_insert_path.c_str());
	      exec_string(arguments.notify_insert_path, cdp);
	    }
	  }
	     	     
	  cdp->exec_mode = execed;
	}
	break;
      case CDRom::audio:
	if(arguments.verbose)
	  cout << _("audio") << endl;

	if(cdp->exec_mode == not_execed) {
	  cdp->exec_mode = audio;
	  int status;
	  if(!arguments.mount_only && 
	     ((!audiopid)
	      || (::waitpid(audiopid, &status, WNOHANG) == audiopid))) {
	    audiopid = exec_string(cdp->audioplayer, cdp);
	  }
	}
	break;		    
      }
      cdp->cdrom->close();      
    }
    ::usleep(arguments.check_interval);

    /* Prevent zombies and throw away child status */
    int status;
    if(::waitpid(-1, &status, WNOHANG) == audiopid) {
      audiopid = 0;
    }
  }
}
