/* $Id: xp_pcap_iface.c,v 1.5 2002/02/23 23:09:42 mixter Exp $ */
/*
** Copyright (C) 2001 Fyodor Yarochkin <fygrave@tigerteam.net>,
**                    Ofir Arkin       <ofir@sys-security.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** All material for nonprofit, educational use only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "xprobe.h"

#if defined(DARWIN) && !defined(uint32_t)
#define uint32_t unsigned int
#endif

/* heavily based on R.Stevens examples */

pcap_t *pd;
int datalink;

inter_t interf;

volatile unsigned char lpcap_done;

int init_pcap(char *iname, char *hostname) {
    char cmd[BUFSIZ+1], errbuf[PCAP_ERRBUF_SIZE];
    struct bpf_program fcode;

    if (iname == NULL) {
        if ((iname = pcap_lookupdev(errbuf)) == NULL) {
#ifdef EBUG
            fprintf(stderr,"pcap_lookup: %s\n",errbuf);
#endif
            return -1;
        }
    }
    
    strncpy((char *)interf.iname, iname, sizeof(interf.iname));
    interf.addr = (struct in_addr)get_iface_addr(interf.iname);
#ifdef EBUG
    fprintf(stderr, "Interface: %s/%s\n\n",interf.iname, inet_ntoa(interf.addr));
#endif
    if ((pd = pcap_open_live(iname, LPCAP_SNAPLEN, 0, LPCAP_TIMEOUT,
        errbuf)) == NULL) {
#ifdef EBUG
        fprintf(stderr,"pcap_open_live: %s\n",errbuf);
#endif
        return -1;
    }
    
    if (pcap_lookupnet(iname, (uint32_t *)&(interf.net),
                                 (uint32_t *)&(interf.netmask), errbuf) < 0) {
#ifdef EBUG
        fprintf(stderr,"pcap_lookupnet: %s\n",errbuf);
#endif
        return -1;
    }

    snprintf(cmd, sizeof(cmd), "icmp");

    if (pcap_compile(pd, &fcode, cmd, 0,*((uint32_t*)&interf.netmask)) < 0) {
#ifdef EBUG
        fprintf(stderr,"pcap_compile: %s\n",errbuf);
#endif
        return -1;
    }

    if (pcap_setfilter(pd, &fcode) < 0) {
#ifdef EBUG
        fprintf(stderr,"pcap_setfilter: %s\n",errbuf);
#endif
        return -1;
    }

    if ((datalink = pcap_datalink(pd)) < 0) {
#ifdef EBUG
        fprintf(stderr,"pcap_datalink: %s\n",errbuf);
#endif
        return -1;
    }

    
   return 1; 
}

#ifdef __linux__
/* timeout handling on linux version of libpcap is bloody broken */
/* so we gotta fix it in such a dogdy way */

unsigned char *next_pack(ssize_t *len) {
    unsigned char *ptr = NULL;
    struct pcap_pkthdr hdr;
    int sd;
    fd_set fds;
    struct timeval wait;

    wait.tv_sec = DEF_TIMEOUT+8000;
    wait.tv_usec = 0;
    sd = pcap_fileno(pd);

    while(!lpcap_done && ptr == NULL) {
        FD_ZERO(&fds);
        FD_SET(sd, &fds);
        if (select(sd+1, &fds, (fd_set *)0, (fd_set *)0, &wait) <= 0)
            return NULL;
        ptr = (unsigned char *) pcap_next(pd, &hdr);
    }

    if (ptr != NULL) *len = hdr.caplen;
    return(ptr);
}

#else /* everywhere esles timeout seems to be honored properly */

unsigned char *next_pack(ssize_t *len) {
#ifdef DARWIN
    char *ptr;
#else
    unsigned char *ptr;
#endif
    struct pcap_pkthdr hdr;

    while((ptr = (char *) pcap_next(pd, &hdr)) == NULL) if (lpcap_done) return NULL;

    *len = hdr.caplen;
#ifdef DARWIN
    return (unsigned char*) ptr;
#else
    return(ptr);
#endif
}

#endif

unsigned char *read_icmp(ssize_t  *packlen, int type, int code) {
    unsigned char *ptr;
    struct ether_header *ehp;
    struct ip *ip;
    struct icmp *icmp;

    lpcap_done = 0;
    signal(SIGALRM, read_icmp_timeout);
    alarm(2);

    while(!lpcap_done) {
        if((ptr = next_pack(packlen)) == NULL) return NULL;
#ifdef DEBUG
        hexdump(ptr, *packlen);
#endif                

        switch(datalink) {
            case DLT_RAW:
                break;

            case DLT_NULL:
                *packlen -= 4;
                ptr += 4;
                break;

            case DLT_EN10MB:
                ehp = (struct ether_header *)ptr;
                if (ntohs(ehp->ether_type) != ETHERTYPE_IP) {
#ifdef EBUG
                    fprintf(stderr,">>ERROR: datagram type %x is not IP\n",
                            ntohs(ehp->ether_type));
#endif
                    ptr = NULL;
                    break;
                }
                *packlen -= 14;
                ptr += 14;
                break;

            case DLT_PPP:
                *packlen -= 24;
                ptr += 24;
                break;

            case DLT_SLIP:
                *packlen -= 24;
                ptr += 24;
                break;

           default:     
                fprintf(stderr,"PCAP: Unknown datalink type: %d!\n", datalink);
        }
	    
        if (ptr != NULL && (unsigned)*packlen > (sizeof(ip) + 8)) {
            ip   = (struct ip *) ptr;
            icmp =(struct icmp *)(ptr + (ip->ip_hl<<2));
#ifdef DEBUG
            fprintf(stderr,"Received: icmp code: %d type: %d\n",
                    icmp->icmp_code, icmp->icmp_type);
#endif            

            if (((code == -1) || (code == icmp->icmp_code)) &&
                ((type == -1) || (type == icmp->icmp_type))) 
                return ptr;
        }
    } /* while(.. */
                
    return NULL;
}


void read_icmp_timeout(int sig) {
    lpcap_done = 1;
}
