//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers_output_gnuplot.cc

#include <iostream.h>
#ifndef _TIERS_HH
#include "tiers.hh"
#endif



////////////////////////////////////////////////////////////////////////
// Model:OutputGnuplotHeader
//
// Output the header for a gnuplot script
////////////////////////////////////////////////////////////////////////
bool
Model::OutputGnuplotHeader()
{
  bool ret = true;
  
  unsigned long int TotalNumNodes = SW + (NM*SM) + (NM*NL*SL);
  cout << "#set term postscript portrait \"Helvetica-Bold\" 16" << endl;
  cout << "set term x11" << endl;
  cout << "#set term mif" << endl;

  cout << "set output \"" << outputdir << "/tiers.mif\"" << endl;
  cout << "set title \"" << title << "\"" << endl;

  cout << "set xlabel 'Horizontal Distance"  
	   << "     Number of Nodes: " 
	   << TotalNumNodes << "\'" << endl;
  cout << "set ylabel 'Vertical Distance'" << endl;

  unsigned long int Size = GRID * (WAN_SCALE + MAN_SCALE + 
								   LAN_SCALE);
  //  cout << "set xrange [0:" << Size << "]" << endl;
  // cout << "set yrange [0:" << Size << "]" << endl;

  cout << "#set size square " << endl;
  cout << "set offsets " << Size*0.05 << "," << Size*0.05 
	   << "," << Size*0.05 << "," << Size*0.05 << endl;;

  cout << "set key " << Size*0.85 << "," << Size*0.85 << endl;
  cout << "#set nokey" << endl;

  cout << "set pointsize 1.0" << endl;

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model:OutputGnuplotTrailer
//
// Output the trailer of a gnuplot script
////////////////////////////////////////////////////////////////////////
bool
Model::OutputGnuplotTrailer()
{
  bool ret = true;
  
  cout << "pause -1 \"Wait ... press RETURN to exit after the graph has appeared\"" << endl;
  
  // flush the output
  cout << "set output" << endl;

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model:CreateGlobalNodeList
//
// Create a global list of all the nodes for use when outputing the edges
// Output labels for in the model, in a format suitable for gnuplot
////////////////////////////////////////////////////////////////////////
bool
Model::CreateGlobalNodeList(Node* &GlobalNodeList)
{
  bool ret = true;
  unsigned long int i = 0;
  unsigned long int j = 0;

  // The total number of nodes
  unsigned long int NumNodes = WanList[0].NumNodes(); 
  unsigned long int NumLans = 0;
  unsigned long int CurrentNode = 0;

  // Count the total number of nodes to know how large an array to allocate
  unsigned long int lanindex = 0;
  for (unsigned long int man = 0; man < WanList[0].NumNetworks(); man++)
	{
	  NumNodes += ManList[man].NumNodes();
	  NumLans += ManList[man].NumNetworks();

	  unsigned long int UpperLimit = lanindex + ManList[man].NumNetworks();
	  for (lanindex = lanindex; lanindex < UpperLimit; lanindex++)
	    {
	      NumNodes += LanList[lanindex].NumNodes();
	    }
	}

  GlobalNodeList = new Node[NumNodes];
  if (!GlobalNodeList) 
	{
	  cerr << "Tiers:: failed to allocate space for global node list" << endl;
	  ret = false;
	}


  if (TIERS_LABEL_WAN || TIERS_LABEL_MAN || TIERS_LABEL_LAN)
	{
	  cout << "#" << endl
		   << "# NODES" << endl
		   << "# =====" << endl 
		   << "#" << endl
		   << "# Node Label\tlabelX\tlabelY" << endl
		   << "# ------------------------------------" << endl;
	}

  //
  // Copy each of the node details into nodes
  // and optionally print out the coordinates and a label (the node number)
  //

  // The offset for the label from the node, in the WAN units
  unsigned long int offsetX = 1;
  unsigned long int offsetY = 1;

  // WAN
  for (i = 0; i < 1 /* NW should equal 1 */; i++)
	{
	  for (j = 0; j < WanList[i].NumNodes(); j++)
		{
		  GlobalNodeList[CurrentNode] = WanList[i].nodes[j];

		  if (TIERS_LABEL_WAN)
		    {
			  cout << "set label \"" << CurrentNode << "\" at "
				   << (((WanList[i]).nodes[j]).x + offsetX)*WAN_SCALE << "," 
				   << (((WanList[i]).nodes[j]).y + offsetY)*WAN_SCALE 
				   << " font \"Times,8\""
				   << endl;
			}
		  
		  CurrentNode++;
		}
	}
  
  // MANs
  unsigned long int NumMans = WanList[0].NumNetworks();
  for (i = 0; i < NumMans; i++)
	{
	  for (j = 0; j < ManList[i].NumNodes(); j++)
		{
		  GlobalNodeList[CurrentNode] = ManList[i].nodes[j];

		  if (TIERS_LABEL_MAN)
		    {
			  cout << "set label \"" << CurrentNode << "\" at "
				   << (((ManList[i]).nodes[j]).x) + offsetX << "," 
				   << (((ManList[i]).nodes[j]).y) + offsetY 
				   << " font \"Times,8\""
				   << endl;
			}

		  CurrentNode++;
		}
	}

  // LANs
  for (i = 0; i < NumLans; i++)
	{
	  for (j = 0; j < LanList[i].NumNodes(); j++)
		{
		  GlobalNodeList[CurrentNode] = LanList[i].nodes[j];

		  if (TIERS_LABEL_LAN)
		    {
			  cout << "set label \"" << CurrentNode << "\" at "
				   << (((LanList[i]).nodes[j]).x) + offsetX << "," 
				   << (((LanList[i]).nodes[j]).y) + offsetY 
				   << " font \"Times,8\""
				   << endl;
			}

		  CurrentNode++;
		}
	}
  

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::OutputEdgesGnuplot
//
// Output all the active edges in the graph
// Basically identical to OutputEdges but kept separate for future flexibility
////////////////////////////////////////////////////////////////////////
bool
Model::OutputEdgesGnuplot()
{
  bool ret = true;
  unsigned long int network = 0;
  unsigned long int from = 0;

  Node *GlobalNodeList = 0;
  // Create a list of all the nodes for use when outputing the edges
  // As a side-effect, it may print out labels for each of the nodes
  if (!CreateGlobalNodeList(GlobalNodeList))
    {
	  cerr << "Tiers:: create global node list output error" << endl;
	  ret = false;
    }

  // Write the plot statement into the script, then add the data to be plotted
  // Check that MAN/LAN nodes exist before printing each plot out
  cout << "plot ";
//  if (WanList[0].NumNodes() > 1)
  	cout << "'-' title 'WAN' with linespoints 3";

//  if (ManList[0].NumNodes() > 1)
	cout << ", '-' title 'MAN' with linespoints 2";

//  if (LanList[0].NumNodes() > 1)
	cout << ", '-' title 'LAN' with linespoints 1";
  // And end the line
  cout<< endl;


  if (TIERS_VERBOSE)
	{
	  cout << "#" << endl
		   << "# EDGES" << endl
		   << "# =====" << endl 
		   << "#" << endl
		   << "# From\t(x1, y1)" << endl
		   << "# To\t(x2, y2)" << endl
		   << "# -------------" << endl;
	}

  // Repositioning took care of scaling the coordinates appropriately

  // WANs
  for (network = 0; network < 1 /* NW should equal 1 */; network++)
	{
	  if (TIERS_DEBUG)
		{
		  cout << "#" << endl << "# WAN " << network << endl << "#" << endl;
		}

	  for (from = 0; from < WanList[network].NumNodes(); from++)
		{
		  unsigned long int MaxNumEdges = WanList[network].MaxNumEdges();
		  unsigned long int NumEdges = WanList[network].NumEdges(from);
		  Edge	*edges = WanList[network].newedges;
		  for (unsigned long int to = 0; to < NumEdges; to++)
			{
			  unsigned long int index = from*MaxNumEdges+to;
			  unsigned long int start = edges[index].start;
			  unsigned long int end = edges[index].end;

			  unsigned long int start_scale = WAN_SCALE;
			  if (GlobalNodeList[start].type > WAN)
				{
				  start_scale = 1;	// MAN positions have already been scaled
				}
			  
			  unsigned long int end_scale = WAN_SCALE;
			  if (GlobalNodeList[end].type > WAN)
				{
				  end_scale = 1;	// MAN positions have already been scaled
				}
			  
			  if ((edges[index].state != Model::INACTIVE) && 
				  (WanList[network].test_dup_edges(edges[index].start, 
												   edges[index].end)) )
				{
				  cout << GlobalNodeList[start].x * start_scale << "\t"
					   << GlobalNodeList[start].y * start_scale << endl
					   << GlobalNodeList[end].x * end_scale << "\t"
					   << GlobalNodeList[end].y * end_scale 
					   << endl << endl;
				}
			} // to
		} // from
	} // network

  // End the inline dataset
  cout << "e" << endl;

  // MANs
  for (network = 0; network < WanList[0].NumNetworks(); network++)
	{
	  if (TIERS_DEBUG)
		{
		  cout << "#" << endl << "# MAN " << network << endl << "#" << endl;
		}

	  for (from = 0; from < ManList[network].NumNodes(); from++)
		{
		  unsigned long int MaxNumEdges = ManList[network].MaxNumEdges();
		  unsigned long int NumEdges = ManList[network].NumEdges(from);
		  Edge	*edges = ManList[network].newedges;
		  for (unsigned long int to = 0; to < NumEdges; to++)
			{
			  unsigned long int index = from*MaxNumEdges+to;
			  unsigned long int start = edges[index].start;
			  unsigned long int end = edges[index].end;

			  unsigned long int start_scale = 1;
			  if (GlobalNodeList[start].type == WAN)
				{
				  // WAN positions have not already been scaled
				  start_scale = WAN_SCALE;	
				}
			  
			  unsigned long int end_scale = 1;
			  if (GlobalNodeList[end].type == WAN)
				{
				  // WAN positions have not already been scaled
				  end_scale = WAN_SCALE;	
				}
			  
			  if ((edges[index].state != Model::INACTIVE) &&
				  (ManList[network].test_dup_edges(edges[index].start,
												   edges[index].end)))
				{
				  cout << GlobalNodeList[start].x * start_scale << "\t"
					   << GlobalNodeList[start].y * start_scale << endl
					   << GlobalNodeList[end].x * end_scale << "\t"
					   << GlobalNodeList[end].y * end_scale
					   << endl << endl;
				}
			} // to
		} // from
	} // network

  // End the inline dataset
//  if (ManList[0].NumNodes() > 1)
	cout << "e" << endl;

  // LANs
  // LANs are modelled as a star, with the edges in edges
  // Edges for internetwork redundancy are at the end of edges
  network = 0; // the index of each LAN
  for (unsigned long int supernetwork = 0; 
	   supernetwork < WanList[0].NumNetworks(); 
	   supernetwork++)
	{
	  if (TIERS_DEBUG)
		{
		  cout << "#" << endl << "# LAN " << network << endl << "#" << endl;
		}


	  unsigned long int UpperLimit = network + 
		ManList[supernetwork].NumNetworks();
	  for (network = network; network < UpperLimit; network++)
		{
		  // edge (0,0) is undefined
		  for (unsigned long int to = 1; 
			   to < (LanList[network].NumEdges(0)+1); 
			   to++)
			{
			  Edge	*edges = LanList[network].edges;	// newedges is 0
			  unsigned long int start = edges[to].start;
			  unsigned long int end = edges[to].end;
			  
			  // No scaling needed since LANs only ever connect to MANs and
			  // both have already been scaled when repositioning them

			  if (edges[to].state != Model::INACTIVE)
				{
				  cout << GlobalNodeList[start].x << "\t"
					   << GlobalNodeList[start].y << endl
					   << GlobalNodeList[end].x << "\t"
					   << GlobalNodeList[end].y 
					   << endl << endl;

				  // generate the symmetrical edge for the purposes of output
				  if (!REMOVE_DUP_EDGES)
					{
					  cout << GlobalNodeList[end].x << "\t"
						   << GlobalNodeList[end].y << endl
						   << GlobalNodeList[start].x << "\t"
						   << GlobalNodeList[start].y 
						   << endl << endl;

					}
				} // to
			} // network
		} // supernetwork
	}

  // End the inline dataset
//  if (LanList[0].NumNodes() > 1)
	cout << "e" << endl;

  return ret;
}

// end of file

