// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief D+ -> eta' ell nu
  class BESIII_2024_I2839350 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2024_I2839350);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      declare(UnstableParticles(Cuts::abspid==411), "UFS");

      for(unsigned int ix=0;ix<2;++ix)
        book(_h[ix], "TMP/h_"+toString(ix+1), refData(1, 1, 1+ix));
      book(_c,"TMP/c_Dplus");
    }

    // Calculate the Q2 using mother and daugher meson
    double q2(const Particle& B, int mesonID) {
      FourMomentum q = B.mom() - select(B.children(), Cuts::pid==mesonID)[0];
      return q*q;
    }

    // Check for explicit decay into pdgids
    bool isSemileptonicDecay(const Particle& mother, vector<int> ids) {
      // Trivial check to ignore any other decays but the one in question modulo photons
      const Particles children = mother.children(Cuts::pid!=PID::PHOTON);
      if (children.size()!=ids.size()) return false;
      // Check for the explicit decay
      return all(ids, [&](int i){return count(children, hasPID(i))==1;});
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Loop over D+ mesons
      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles()) {
        // remove particles from radiation/mixing
        bool self=false;
        for(const Particle & child : p.children())
          self |= p.abspid()==child.pid();
        if(self) continue;
        _c->fill();
        if(isSemileptonicDecay(p, {PID::ETAPRIME, PID::EPLUS, PID::NU_E})) {
          _h[1]->fill(q2(p, PID::ETAPRIME));
        }
        else if(isSemileptonicDecay(p, {PID::ETAPRIME, PID::ANTIMUON, PID::NU_MU})) {
          _h[0]->fill(q2(p, PID::ETAPRIME));
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      double tau = 1033e-6;
      for(unsigned int ix=0;ix<2;++ix) {
        Estimate1DPtr  tmp;
        book(tmp, 1, 1, 1+ix);
        barchart(_h[ix],tmp);
        scale(tmp,1./tau/ *_c);
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    Histo1DPtr _h[2];
    CounterPtr _c;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2024_I2839350);

}
