// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief Measurement of $R$ for energies between 39.8 and 45.2 GeV
  class TASSO_1984_I199468 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1984_I199468);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");

      // Book histograms
      book(_c_hadrons1, "/TMP/sigma_hadrons1", refData<YODA::BinnedEstimate<string>>(1,1,1));
      book(_c_muons1,   "/TMP/sigma_muons1"  , refData<YODA::BinnedEstimate<string>>(1,1,1));
      book(_c_hadrons2, "/TMP/sigma_hadrons2", refData(2,1,1));
      book(_c_muons2,   "/TMP/sigma_muons2"  , refData(2,1,1));
      for (const string& en : _c_hadrons1.binning().edges<0>()) {
        double eval = stod(en)*GeV;
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      if (nCount[-13]==1 && nCount[13]==1 && ntotal==2+nCount[22]) {
        // mu+mu- + photons
        _c_muons1->fill(_sqs);
        _c_muons2->fill(sqrtS()/GeV);
      }
      else {
        // everything else
        _c_hadrons1->fill(_sqs);
        _c_hadrons2->fill(sqrtS()/GeV);
      }
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      BinnedEstimatePtr<string> ratio1;
      book(ratio1, 1, 1, 1);
      divide(_c_hadrons1, _c_muons1, ratio1);
      Estimate1DPtr ratio2;
      book(ratio2, 2, 1, 1);
      divide(_c_hadrons2, _c_muons2, ratio2);
    }
    /// @}

    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _c_hadrons1, _c_muons1;
    Histo1DPtr _c_hadrons2, _c_muons2;
    string _sqs = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(TASSO_1984_I199468);
}
