/* SPDX-License-Identifier: GPL-2.0 OR Linux-OpenIB */
/* Copyright (c) Amazon.com, Inc. or its affiliates.
 * All rights reserved.
 */

#ifndef ENA_H
#define ENA_H

#include "kcompat.h"
#include <linux/bitops.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 3, 0)
#include "dim.h"
#else
#include <linux/dim.h>
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(5, 3, 0) */
#include <linux/etherdevice.h>
#include <linux/if_vlan.h>
#include <linux/inetdevice.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/net_tstamp.h>
#ifdef ENA_XDP_SUPPORT
#include <net/xdp.h>
#endif
#ifdef HAS_BPF_HEADER
#include <uapi/linux/bpf.h>
#endif
#include <linux/u64_stats_sync.h>
#ifdef ENA_HAS_DEVLINK_HEADERS
#include <net/devlink.h>
#endif /* ENA_HAS_DEVLINK_HEADERS */
#ifdef ENA_PAGE_POOL_SUPPORT
#include <net/page_pool/helpers.h>
#endif /* ENA_PAGE_POOL_SUPPORT */

#include "ena_com.h"
#include "ena_eth_com.h"

#define DRV_MODULE_GEN_MAJOR	2
#define DRV_MODULE_GEN_MINOR	16
#define DRV_MODULE_GEN_SUBMINOR	1

#define DRV_MODULE_NAME		"ena"
#ifndef DRV_MODULE_GENERATION
#define DRV_MODULE_GENERATION \
	__stringify(DRV_MODULE_GEN_MAJOR) "."	\
	__stringify(DRV_MODULE_GEN_MINOR) "."	\
	__stringify(DRV_MODULE_GEN_SUBMINOR) "g"
#endif

#define DEVICE_NAME	"Elastic Network Adapter (ENA)"

/* 1 for AENQ + ADMIN */
#define ENA_ADMIN_MSIX_VEC		1
#define ENA_MAX_MSIX_VEC(io_queues)	(ENA_ADMIN_MSIX_VEC + (io_queues))

/* The ENA buffer length fields is 16 bit long. So when PAGE_SIZE == 64kB the
 * driver passes 0.
 * Since the max packet size the ENA handles is ~9kB limit the buffer length to
 * 16kB.
 */
#if PAGE_SIZE > SZ_16K
#define ENA_PAGE_SIZE (_AC(SZ_16K, UL))
#else
#define ENA_PAGE_SIZE PAGE_SIZE
#endif

#define ENA_MIN_MSIX_VEC		2

#define ENA_REG_BAR			0
#define ENA_MEM_BAR			2
#define ENA_BAR_MASK (BIT(ENA_REG_BAR) | BIT(ENA_MEM_BAR))

#define ENA_DEFAULT_RING_SIZE		(1024)
#define ENA_DEFAULT_WIDE_LLQ_RING_SIZE	(512)
#define ENA_MIN_RING_SIZE		(256)

#define ENA_MIN_RX_BUF_SIZE (2048)

#define ENA_MIN_NUM_IO_QUEUES	(1)

#define ENA_TX_WAKEUP_THRESH		(MAX_SKB_FRAGS + 2)
#define ENA_DEFAULT_RX_COPYBREAK	(256 - NET_IP_ALIGN)

#define ENA_MIN_MTU		128

#define ENA_NAME_MAX_LEN	20
#define ENA_IRQNAME_SIZE	40

#define ENA_PKT_MAX_BUFS	19

/* The number of tx packet completions that will be handled each NAPI poll
 * cycle is ring_size / ENA_TX_POLL_BUDGET_DIVIDER.
 */
#define ENA_TX_POLL_BUDGET_DIVIDER	4

/* Refill Rx queue when number of required descriptors is above
 * QUEUE_SIZE / ENA_RX_REFILL_THRESH_DIVIDER or ENA_RX_REFILL_THRESH_PACKET
 */
#define ENA_RX_REFILL_THRESH_DIVIDER	8
#define ENA_RX_REFILL_THRESH_PACKET	256

/* Number of queues to check for missing completions / interrupts per timer service */
#define ENA_MONITORED_QUEUES	4
/* Max timeout packets before device reset */
#define MAX_NUM_OF_TIMEOUTED_PACKETS 128

#define ENA_TX_RING_IDX_NEXT(idx, ring_size) (((idx) + 1) & ((ring_size) - 1))

#define ENA_RX_RING_IDX_NEXT(idx, ring_size) (((idx) + 1) & ((ring_size) - 1))
#define ENA_RX_RING_IDX_ADD(idx, n, ring_size) \
	(((idx) + (n)) & ((ring_size) - 1))

#define ENA_IO_TXQ_IDX(q)	(2 * (q))
#define ENA_IO_RXQ_IDX(q)	(2 * (q) + 1)
#define ENA_IO_TXQ_IDX_TO_COMBINED_IDX(q)	((q) / 2)
#define ENA_IO_RXQ_IDX_TO_COMBINED_IDX(q)	(((q) - 1) / 2)

#define ENA_MGMNT_IRQ_IDX		0
#define ENA_IO_IRQ_FIRST_IDX		1
#define ENA_IO_IRQ_IDX(q)		(ENA_IO_IRQ_FIRST_IDX + (q))

#define ENA_ADMIN_POLL_DELAY_US 5000

/* ENA device should send keep alive msg every 1 sec.
 * We wait for 6 sec just to be on the safe side.
 */
#define ENA_DEVICE_KALIVE_TIMEOUT	(6 * HZ)
#define ENA_MAX_NO_INTERRUPT_ITERATIONS 3

#define ENA_MMIO_DISABLE_REG_READ	BIT(0)

#ifdef ENA_LPC_SUPPORT
struct ena_page_cache;

#endif /* ENA_LPC_SUPPORT */
#ifdef ENA_PHC_SUPPORT
struct ena_phc_info;

#endif
struct ena_irq {
	irq_handler_t handler;
	void *data;
	int cpu;
	u32 vector;
	cpumask_t affinity_hint_mask;
	char name[ENA_IRQNAME_SIZE];
};

struct ena_napi {
	unsigned long last_intr_jiffies ____cacheline_aligned;
	u8 interrupts_masked;
	bool lost_interrupt_unmask_handled;
	struct napi_struct napi;
	struct ena_ring *tx_ring;
	struct ena_ring *rx_ring;
	u32 qid;
	struct dim dim;
};

struct ena_tx_buffer {
	union {
		struct sk_buff *skb;
#ifdef ENA_XDP_SUPPORT
		/* XDP buffer structure which is used for sending packets in
		 * the xdp queues
		 */
		struct xdp_frame *xdpf;
#endif /* ENA_XDP_SUPPORT */
	};
	/* num of ena desc for this specific skb
	 * (includes data desc and metadata desc)
	 */
	u32 tx_descs;
	/* num of buffers used by this skb */
	u32 num_of_bufs;

	/* Total size of all buffers in bytes */
	u32 total_tx_size;

	/* Indicate if bufs[0] map the linear data of the skb. */
	u8 map_linear_data;

	/* Used for detecting missed tx packets */
	bool timed_out;
#ifdef ENA_AF_XDP_SUPPORT

	/* used for ordering TX completions when needed (e.g. AF_XDP) */
	u8 acked;

#endif
	/* Save the last jiffies to detect missing tx packets
	 *
	 * sets to non zero value on ena_start_xmit and set to zero on
	 * napi and timer_Service_routine.
	 *
	 * while this value is not protected by lock,
	 * a given packet is not expected to be handled by ena_start_xmit
	 * and by napi/timer_service at the same time.
	 */
	unsigned long tx_sent_jiffies;
	struct ena_com_buf bufs[ENA_PKT_MAX_BUFS];
} ____cacheline_aligned;

struct ena_rx_buffer {
	struct sk_buff *skb;
#ifdef ENA_AF_XDP_SUPPORT
	union {
		struct {
			struct page *page;
			dma_addr_t dma_addr;
		};
		/* XSK pool buffer */
		struct xdp_buff *xdp;
	};
#else
	struct page *page;
	dma_addr_t dma_addr;
#endif /* ENA_AF_XDP_SUPPORT */
	u32 page_offset;
	u32 buf_offset;
	struct ena_com_buf ena_buf;
#ifdef ENA_LPC_SUPPORT
	bool is_lpc_page;
#endif /* ENA_LPC_SUPPORT */
#ifdef ENA_PAGE_POOL_SUPPORT
	/* Used to locally frag a page allocated from page pool via the DRB
	 * mechanism, thus avoiding atomic ops to update the page ref.
	 */
	long pagecnt_bias;
#endif /* ENA_PAGE_POOL_SUPPORT */
} ____cacheline_aligned;

struct ena_stats_tx {
	u64 cnt;
	u64 bytes;
	u64 queue_stop;
	u64 prepare_ctx_err;
	u64 queue_wakeup;
	u64 dma_mapping_err;
	u64 linearize;
	u64 linearize_failed;
	u64 napi_comp;
	u64 tx_poll;
	u64 doorbells;
	u64 bad_req_id;
	u64 llq_buffer_copy;
	u64 missed_tx;
	atomic64_t pending_timedout_pkts;
	u64 unmask_interrupt;
	u64 last_napi_jiffies;
	u64 lost_interrupt;
#ifdef ENA_AF_XDP_SUPPORT
	u64 xsk_cnt;
	u64 xsk_bytes;
	u64 xsk_need_wakeup_set;
	u64 xsk_wakeup_request;
#endif /* ENA_AF_XDP_SUPPORT */
#ifdef ENA_XDP_MB_SUPPORT
	u64 xdp_frags_exceeded;
	u64 xdp_short_linear_part;
#endif /* ENA_XDP_MB_SUPPORT */
};

struct ena_stats_rx {
	u64 cnt;
	u64 bytes;
	u64 rx_copybreak_pkt;
	u64 csum_good;
	u64 refil_partial;
	u64 csum_bad;
	u64 page_alloc_fail;
	u64 skb_alloc_fail;
	u64 dma_mapping_err;
	u64 bad_desc_num;
#ifdef ENA_BUSY_POLL_SUPPORT
	u64 bp_yield;
	u64 bp_missed;
	u64 bp_cleaned;
#endif
	u64 bad_req_id;
	u64 empty_rx_ring;
	u64 csum_unchecked;
#ifdef ENA_XDP_SUPPORT
	u64 xdp_aborted;
	u64 xdp_drop;
	u64 xdp_pass;
	u64 xdp_tx;
	u64 xdp_invalid;
	u64 xdp_redirect;
#endif
#ifdef ENA_LPC_SUPPORT
	u64 lpc_warm_up;
	u64 lpc_full;
	u64 lpc_wrong_numa;
#endif /* ENA_LPC_SUPPORT */
#ifdef ENA_PAGE_POOL_SUPPORT
#ifdef CONFIG_PAGE_POOL_STATS
	u64 pp_alloc_fast;
	u64 pp_alloc_slow;
	u64 pp_alloc_slow_hi_ord;
	u64 pp_alloc_empty;
	u64 pp_alloc_refill;
	u64 pp_alloc_waive;
	u64 pp_cached;
	u64 pp_cache_full;
	u64 pp_ring;
	u64 pp_ring_full;
	u64 pp_released_ref;
#endif /* CONFIG_PAGE_POOL_STATS */
#endif /* ENA_PAGE_POOL_SUPPORT */
#ifdef ENA_AF_XDP_SUPPORT
	u64 xsk_need_wakeup_set;
	u64 zc_queue_pkt_copy;
#endif /* ENA_AF_XDP_SUPPORT */
};

struct ena_ring {
	/* Holds the empty requests for TX/RX
	 * out of order completions
	 */
	u16 *free_ids;

	union {
		struct ena_tx_buffer *tx_buffer_info;
		struct ena_rx_buffer *rx_buffer_info;
	};

	/* cache ptr to avoid using the adapter */
	struct device *dev;
	struct pci_dev *pdev;
	struct napi_struct *napi;
	struct net_device *netdev;
#ifdef ENA_LPC_SUPPORT
	struct ena_page_cache *page_cache;
#endif /* ENA_LPC_SUPPORT */
#ifdef ENA_PAGE_POOL_SUPPORT
	struct page_pool *page_pool;
#endif /* ENA_PAGE_POOL_SUPPORT */
	struct ena_com_dev *ena_dev;
	struct ena_adapter *adapter;
	struct ena_com_io_cq *ena_com_io_cq;
	struct ena_com_io_sq *ena_com_io_sq;
#ifdef ENA_XDP_SUPPORT
	struct bpf_prog *xdp_bpf_prog;
#ifdef ENA_XDP_MB_SUPPORT
	bool xdp_prog_support_frags;
#endif /* ENA_XDP_MB_SUPPORT */
#endif /* ENA_XDP_SUPPORT */
	unsigned long last_checked_last_napi_jiffies;
	bool last_checked_is_cq_empty;
#ifdef ENA_XDP_SUPPORT
	struct xdp_rxq_info xdp_rxq;
	spinlock_t xdp_tx_lock;	/* synchronize XDP TX/Redirect traffic */
	/* Used for rx queues only to point to the xdp tx ring, to
	 * which traffic should be redirected from this rx ring.
	 */
	struct ena_ring *xdp_ring;
#ifdef ENA_AF_XDP_SUPPORT
	struct xsk_buff_pool *xsk_pool;
#endif /* ENA_AF_XDP_SUPPORT */
#endif /* ENA_XDP_SUPPORT */

	u16 next_to_use;
	u16 next_to_clean;
	u16 rx_copybreak;
	u16 rx_headroom;
	u16 qid;
	u16 mtu;
	u16 sgl_size;
	u8 enable_bql;

	/* The maximum header length the device can handle */
	u8 tx_max_header_size;

	bool disable_meta_caching;
	u16 no_interrupt_event_cnt;

	/* cpu and NUMA for TPH */
	int cpu;
	int numa_node;

	/* number of tx/rx_buffer_info's entries */
	int ring_size;

	enum ena_admin_placement_policy_type tx_mem_queue_type;

	struct ena_com_rx_buf_info ena_bufs[ENA_PKT_MAX_BUFS];
	u32 interrupt_interval;
	/* Indicates whether interrupt interval has changed since previous set.
	 * This flag will be kept up, until cleared by the routine which updates
	 * the device with the modified interrupt interval value.
	 */
	bool interrupt_interval_changed;
	u32 per_napi_packets;
	u16 non_empty_napi_events;
	struct u64_stats_sync syncp;
	union {
		struct ena_stats_tx tx_stats;
		struct ena_stats_rx rx_stats;
	};

	u8 *push_buf_intermediate_buf;
	int empty_rx_queue;
#ifdef ENA_BUSY_POLL_SUPPORT
	atomic_t bp_state;
#endif
} ____cacheline_aligned;

#ifdef ENA_BUSY_POLL_SUPPORT
enum ena_busy_poll_state_t {
	ENA_BP_STATE_IDLE = 0,
	ENA_BP_STATE_NAPI,
	ENA_BP_STATE_POLL,
	ENA_BP_STATE_DISABLE
};

#endif
struct ena_keep_alive_stats {
	u64 rx_drops;
	u64 tx_drops;
	u64 rx_overruns;
};

struct ena_stats_dev {
	u64 tx_timeout;
	u64 suspend;
	u64 resume;
	u64 wd_expired;
	u64 interface_up;
	u64 interface_down;
	u64 admin_q_pause;
	u64 reset_fail;
	u64 total_resets;
	u64 bad_tx_req_id;
	u64 bad_rx_req_id;
	u64 bad_rx_desc_num;
	u64 missing_intr;
	u64 suspected_poll_starvation;
	u64 missing_tx_cmpl;
	u64 rx_desc_malformed;
	u64 tx_desc_malformed;
	u64 invalid_state;
	u64 os_netdev_wd;
	u64 missing_admin_interrupt;
	u64 admin_to;
	u64 device_request_reset;
	u64 missing_first_intr;
	struct ena_keep_alive_stats ka_stats;
};

enum ena_flags_t {
	ENA_FLAG_DEVICE_RUNNING,
	ENA_FLAG_DEV_UP,
	ENA_FLAG_LINK_UP,
	ENA_FLAG_MSIX_ENABLED,
	ENA_FLAG_TRIGGER_RESET,
	ENA_FLAG_ONGOING_RESET
};

enum ena_llq_header_size_policy_t {
	/* Intermediate policy until llq configuration is initialized
	 * to either NORMAL or LARGE
	 */
	ENA_LLQ_HEADER_SIZE_POLICY_UNSPECIFIED = 0,
	/* Policy for Normal size LLQ entry (128B) */
	ENA_LLQ_HEADER_SIZE_POLICY_NORMAL,
	/* Policy for Large size LLQ entry (256B) */
	ENA_LLQ_HEADER_SIZE_POLICY_LARGE
};

struct hw_timestamp_state {
	struct hwtstamp_config ts_cfg;
	u8 hw_tx_supported;
	u8 hw_rx_supported;
};

/* adapter specific private data structure */
struct ena_adapter {
	struct ena_com_dev *ena_dev;
	/* OS defined structs */
	struct net_device *netdev;
	struct pci_dev *pdev;

	/* rx packets that are shorter than this len will be copied to the skb
	 * header
	 */
	u32 rx_copybreak;
	u32 max_mtu;

	u32 num_io_queues;
	u32 max_num_io_queues;

#ifdef ENA_LPC_SUPPORT
	/* Local page cache size when it's enabled */
	u32 configured_lpc_size;
	/* Current Local page cache size */
	u32 used_lpc_size;

#endif /* ENA_LPC_SUPPORT */
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 8, 0)
	struct msix_entry *msix_entries;
#endif
	int msix_vecs;

	u32 missing_tx_completion_threshold;

	u32 requested_tx_ring_size;
	u32 requested_rx_ring_size;

	u32 max_tx_ring_size;
	u32 max_rx_ring_size;

	u32 msg_enable;

	/* The policy is used for two purposes:
	 * 1. Indicates who decided on LLQ entry size (user / device)
	 * 2. Indicates whether large LLQ is set or not after device
	 *    initialization / configuration.
	 */
	enum ena_llq_header_size_policy_t llq_policy;
	bool large_llq_header_supported;

	u16 max_tx_sgl_size;
	u16 max_rx_sgl_size;

	u8 mac_addr[ETH_ALEN];

	unsigned long keep_alive_timeout;
	unsigned long missing_tx_completion_to_jiffies;

	char name[ENA_NAME_MAX_LEN];
#ifdef ENA_PHC_SUPPORT

	struct ena_phc_info *phc_info;
#endif

	unsigned long flags;
	/* TX */
	struct ena_ring tx_ring[ENA_MAX_NUM_IO_QUEUES]
		____cacheline_aligned_in_smp;

	/* RX */
	struct ena_ring rx_ring[ENA_MAX_NUM_IO_QUEUES]
		____cacheline_aligned_in_smp;

	struct ena_napi ena_napi[ENA_MAX_NUM_IO_QUEUES];

	struct ena_irq irq_tbl[ENA_MAX_MSIX_VEC(ENA_MAX_NUM_IO_QUEUES)];

	/* timer service */
	struct work_struct reset_task;
	struct timer_list timer_service;

	bool wd_state;
	bool dev_up_before_reset;
	bool disable_meta_caching;
	unsigned long last_keep_alive_jiffies;

	struct u64_stats_sync syncp;
	struct ena_stats_dev dev_stats;

	/* last queue index that was checked for missing completions / interrupts */
	u32 last_monitored_qid;

	enum ena_regs_reset_reason_types reset_reason;

#ifdef ENA_XDP_SUPPORT
	struct bpf_prog *xdp_bpf_prog;
#endif
	u32 xdp_first_ring;
	u32 xdp_num_queues;

	struct hw_timestamp_state hw_ts_state;
	struct ena_keep_alive_stats persistent_ka_stats;

	struct devlink *devlink;
#ifdef ENA_DEVLINK_SUPPORT
	struct devlink_port devlink_port;
#endif /* ENA_DEVLINK_SUPPORT */
#ifdef CONFIG_DEBUG_FS

	struct dentry *debugfs_base;
#endif /* CONFIG_DEBUG_FS */
};

#define ENA_RESET_STATS_ENTRY(reset_reason, stat) \
	[reset_reason] = { \
	.stat_offset = offsetof(struct ena_stats_dev, stat) / sizeof(u64), \
	.has_counter = true \
}

struct ena_reset_stats_offset {
	int stat_offset;
	bool has_counter;
};

static const struct ena_reset_stats_offset resets_to_stats_offset_map[ENA_REGS_RESET_LAST] = {
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_KEEP_ALIVE_TO, wd_expired),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_ADMIN_TO, admin_to),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_MISS_TX_CMPL, missing_tx_cmpl),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_INV_RX_REQ_ID, bad_rx_req_id),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_INV_TX_REQ_ID, bad_tx_req_id),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_TOO_MANY_RX_DESCS, bad_rx_desc_num),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_DRIVER_INVALID_STATE, invalid_state),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_OS_NETDEV_WD, os_netdev_wd),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_MISS_INTERRUPT, missing_intr),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_SUSPECTED_POLL_STARVATION, suspected_poll_starvation),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_RX_DESCRIPTOR_MALFORMED, rx_desc_malformed),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_TX_DESCRIPTOR_MALFORMED, tx_desc_malformed),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_MISSING_ADMIN_INTERRUPT, missing_admin_interrupt),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_DEVICE_REQUEST, device_request_reset),
	ENA_RESET_STATS_ENTRY(ENA_REGS_RESET_MISS_FIRST_INTERRUPT, missing_first_intr),
};

void ena_set_ethtool_ops(struct net_device *netdev);

void ena_dump_stats_to_dmesg(struct ena_adapter *adapter);

void ena_dump_stats_to_buf(struct ena_adapter *adapter, u8 *buf);

struct sk_buff *ena_alloc_skb(struct ena_ring *rx_ring, void *first_frag, u16 len);

#ifdef ENA_LPC_SUPPORT
int ena_set_lpc_state(struct ena_adapter *adapter, bool enabled);

#endif /* ENA_LPC_SUPPORT */
int ena_update_queue_params(struct ena_adapter *adapter,
			    u32 new_tx_size,
			    u32 new_rx_size,
			    u32 new_llq_header_len);

int ena_update_queue_count(struct ena_adapter *adapter, u32 new_channel_count);

int ena_set_rx_copybreak(struct ena_adapter *adapter, u32 rx_copybreak);

/* Increase a stat by cnt while holding syncp seqlock on 32bit machines */
static inline void ena_increase_stat(u64 *statp, u64 cnt,
				     struct u64_stats_sync *syncp)
{
	u64_stats_update_begin(syncp);
	(*statp) += cnt;
	u64_stats_update_end(syncp);
}

int ena_get_sset_count(struct net_device *netdev, int sset);
#ifdef ENA_BUSY_POLL_SUPPORT
static inline void ena_bp_init_lock(struct ena_ring *rx_ring)
{
	/* reset state to idle */
	atomic_set(&rx_ring->bp_state, ENA_BP_STATE_IDLE);
}

/* called from the napi routine to get ownership of the ring */
static inline bool ena_bp_lock_napi(struct ena_ring *rx_ring)
{
	int rc = atomic_cmpxchg(&rx_ring->bp_state, ENA_BP_STATE_IDLE,
				ENA_BP_STATE_NAPI);
	if (rc != ENA_BP_STATE_IDLE) {
		u64_stats_update_begin(&rx_ring->syncp);
		rx_ring->rx_stats.bp_yield++;
		u64_stats_update_end(&rx_ring->syncp);
	}

	return rc == ENA_BP_STATE_IDLE;
}

static inline void ena_bp_unlock_napi(struct ena_ring *rx_ring)
{
	WARN_ON(atomic_read(&rx_ring->bp_state) != ENA_BP_STATE_NAPI);

	/* flush any outstanding Rx frames */
	if (rx_ring->napi->gro_list)
		napi_gro_flush(rx_ring->napi, false);

	/* reset state to idle */
	atomic_set(&rx_ring->bp_state, ENA_BP_STATE_IDLE);
}

/* called from ena_ll_busy_poll() */
static inline bool ena_bp_lock_poll(struct ena_ring *rx_ring)
{
	int rc = atomic_cmpxchg(&rx_ring->bp_state, ENA_BP_STATE_IDLE,
				ENA_BP_STATE_POLL);
	if (rc != ENA_BP_STATE_IDLE) {
		u64_stats_update_begin(&rx_ring->syncp);
		rx_ring->rx_stats.bp_yield++;
		u64_stats_update_end(&rx_ring->syncp);
	}

	return rc == ENA_BP_STATE_IDLE;
}

static inline void ena_bp_unlock_poll(struct ena_ring *rx_ring)
{
	WARN_ON(atomic_read(&rx_ring->bp_state) != ENA_BP_STATE_POLL);

	/* reset state to idle */
	atomic_set(&rx_ring->bp_state, ENA_BP_STATE_IDLE);
}

/* true if a socket is polling, even if it did not get the lock */
static inline bool ena_bp_busy_polling(struct ena_ring *rx_ring)
{
	return atomic_read(&rx_ring->bp_state) == ENA_BP_STATE_POLL;
}

static inline bool ena_bp_disable(struct ena_ring *rx_ring)
{
	int rc = atomic_cmpxchg(&rx_ring->bp_state, ENA_BP_STATE_IDLE,
				ENA_BP_STATE_DISABLE);

	return rc == ENA_BP_STATE_IDLE;
}
#endif /* ENA_BUSY_POLL_SUPPORT */

static inline void ena_reset_device(struct ena_adapter *adapter,
				    enum ena_regs_reset_reason_types reset_reason)
{
	const struct ena_reset_stats_offset *ena_reset_stats_offset =
		&resets_to_stats_offset_map[reset_reason];

	if (ena_reset_stats_offset->has_counter) {
		u64 *stat_ptr = (u64 *)&adapter->dev_stats + ena_reset_stats_offset->stat_offset;

		ena_increase_stat(stat_ptr, 1, &adapter->syncp);
	}

	ena_increase_stat(&adapter->dev_stats.total_resets, 1, &adapter->syncp);
	adapter->reset_reason = reset_reason;
	/* Make sure reset reason is set before triggering the reset */
	smp_mb__before_atomic();
	set_bit(ENA_FLAG_TRIGGER_RESET, &adapter->flags);
}

static inline int ena_tx_map_frags(struct skb_shared_info *sh_info,
				   struct ena_tx_buffer *tx_info,
				   struct ena_ring *tx_ring,
				   struct ena_com_buf *ena_buf,
				   u16 delta)
{
	u32 nr_frags = sh_info->nr_frags;
	dma_addr_t dma;
	u32 frag_len;
	int i, rc;

	for (i = 0; i < nr_frags; i++) {
		const skb_frag_t *frag = &sh_info->frags[i];

		frag_len = skb_frag_size(frag);

		if (unlikely(delta >= frag_len)) {
			delta -= frag_len;
			continue;
		}

		dma = skb_frag_dma_map(tx_ring->dev, frag, delta,
				       frag_len - delta, DMA_TO_DEVICE);
		rc = dma_mapping_error(tx_ring->dev, dma);
		if (unlikely(rc))
			return rc;

		ena_buf->paddr = dma;
		ena_buf->len = frag_len - delta;
		ena_buf++;
		tx_info->num_of_bufs++;
		delta = 0;
	}

	return 0;
}

#ifdef ENA_LPC_SUPPORT
/* Allocate a page and DMA map it
 * @rx_ring: The IO queue pair which requests the allocation
 *
 * @return: address of the mapped page in DMA and allocated page address is
 * succeeded, or NULL
 */
struct page *ena_alloc_map_page(struct ena_ring *rx_ring, dma_addr_t *dma);

#endif /* ENA_LPC_SUPPORT */
int ena_destroy_device(struct ena_adapter *adapter, bool graceful);
int ena_restore_device(struct ena_adapter *adapter);
void ena_get_and_dump_head_tx_cdesc(struct ena_com_io_cq *io_cq);
void ena_get_and_dump_head_rx_cdesc(struct ena_com_io_cq *io_cq);
int handle_invalid_req_id(struct ena_ring *ring, u16 req_id,
			  struct ena_tx_buffer *tx_info);
int validate_tx_req_id(struct ena_ring *tx_ring, u16 req_id);

static inline void ena_ring_tx_doorbell(struct ena_ring *tx_ring)
{
	ena_com_write_tx_sq_doorbell(tx_ring->ena_com_io_sq);
	ena_increase_stat(&tx_ring->tx_stats.doorbells, 1, &tx_ring->syncp);
}

int ena_xmit_common(struct ena_adapter *adapter,
		    struct ena_ring *ring,
		    struct ena_tx_buffer *tx_info,
		    struct ena_com_tx_ctx *ena_tx_ctx,
		    u16 next_to_use,
		    u32 bytes);
void ena_unmap_tx_buff(struct ena_ring *tx_ring,
		       struct ena_tx_buffer *tx_info);
void ena_init_io_rings(struct ena_adapter *adapter,
		       int first_index, int count);
void ena_down(struct ena_adapter *adapter);
int ena_up(struct ena_adapter *adapter);
void ena_unmask_interrupt(struct ena_ring *tx_ring,
			  struct ena_ring *rx_ring,
			  bool lost_interrupt);
void ena_update_ring_numa_node(struct ena_ring *rx_ring);
void ena_unmap_rx_buff_attrs(struct ena_ring *rx_ring,
			     struct ena_rx_buffer *rx_info,
			     unsigned long attrs);
struct sk_buff *ena_rx_skb_copybreak(struct ena_ring *rx_ring,
				     struct ena_rx_buffer *rx_info,
				     u16 len, u8 meta_len, int pkt_offset,
				     void *buf_data_addr);
void ena_fill_rx_frags(struct ena_ring *rx_ring,
		       u32 descs,
		       struct ena_com_rx_buf_info *ena_bufs,
		       struct skb_shared_info *sh_info,
#ifdef ENA_XDP_MB_SUPPORT
		       struct xdp_buff *xdp,
#endif /* ENA_XDP_MB_SUPPORT */
		       struct sk_buff *skb);
void ena_rx_checksum(struct ena_ring *rx_ring,
		     struct ena_com_rx_ctx *ena_rx_ctx,
		     struct sk_buff *skb);
void ena_set_rx_hash(struct ena_ring *rx_ring,
		     struct ena_com_rx_ctx *ena_rx_ctx,
		     struct sk_buff *skb);
int ena_refill_rx_bufs(struct ena_ring *rx_ring, u32 num);

static inline void handle_tx_comp_poll_error(struct ena_ring *tx_ring, u16 req_id, int rc)
{
	if (unlikely(rc == -EINVAL))
		handle_invalid_req_id(tx_ring, req_id, NULL);
	else if (unlikely(rc == -EFAULT)) {
		ena_get_and_dump_head_tx_cdesc(tx_ring->ena_com_io_cq);
		ena_reset_device(tx_ring->adapter,
				 ENA_REGS_RESET_TX_DESCRIPTOR_MALFORMED);
	}
}

static inline u32 get_rss_indirection_table_size(struct ena_adapter *adapter)
{
	struct ena_com_dev *ena_dev = adapter->ena_dev;

	if (!ena_com_indirection_table_config_supported(ena_dev))
		return 0;

	return (1UL << ena_dev->rss.tbl_log_size);
}

static inline void ena_rx_release_packet_buffers(struct ena_ring *rx_ring,
						 int first_to_release,
						 int last_to_release)
{
	int i;

	for (i = first_to_release; i <= last_to_release; i++) {
		int req_id = rx_ring->ena_bufs[i].req_id;

#ifndef ENA_PAGE_POOL_SUPPORT
		/* The page pool mechanism unmaps page when the page is freed
		 * from the pool.
		 */
		ena_unmap_rx_buff_attrs(rx_ring,
					&rx_ring->rx_buffer_info[req_id],
					ENA_DMA_ATTR_SKIP_CPU_SYNC);
#endif /* ENA_PAGE_POOL_SUPPORT */
		rx_ring->rx_buffer_info[req_id].page = NULL;
	}
}

static inline bool ena_too_many_tx_frags(u8 nr_frags, u16 sgl_size,
					 u32 header_len, u8 tx_max_header_size,
					 bool is_llq)
{
	if (likely(nr_frags < sgl_size))
		return false;

	/* If the number of frags is the maximum allowed number of bufs in a
	 * tx packet then usually the buffer holding the linear part of the
	 * skb/xdp_frame will add another buffer to the packet, increasing the
	 * number of buffers in the packet over the allowed limit.
	 *
	 * Except for the following cases:
	 * 1. In LLQ case: If the linear part of the skb/xdp_frame fits the
	 *    header part of the LLQ entry perfectly, the linear part of the
	 *    skb/xdp_frame will not be used as the first buffer.
	 * 2. In non-LLQ case: if the size of the linear part is 0, then the
	 *    linear part will not become an extra buffer.
	 */
	if (unlikely(nr_frags == sgl_size)) {
		if (likely(is_llq && (header_len <= tx_max_header_size)))
			return false;

		if (unlikely(!is_llq && !header_len))
			return false;
	}

	return true;
}

#endif /* !(ENA_H) */
