package main

import (
	"context"
	"flag"
	"io"
	"log"
	"net"
	"os"
	"os/signal"
	"strings"

	"0xacab.org/leap/obfsvpn/client"
)

func main() {
	// Setup logging.
	logger := log.New(os.Stderr, "", log.LstdFlags)
	debug := log.New(io.Discard, "DEBUG ", log.LstdFlags)

	// setup command line flags.
	var (
		hoppingPT       bool
		hopJitter       uint = 5
		kcp             bool
		minHopSeconds   uint  = 5
		portSeed        int64 = 1
		portCount       uint  = 100
		socksPort       string
		socksHost       string
		obfs4Certs      string
		obfs4Remotes    string
		obfs4RemotePort string
		verbose         bool
	)

	socksPort = "8080"
	socksHost = "127.0.0.1"

	flags := flag.NewFlagSet(os.Args[0], flag.ContinueOnError)
	flags.BoolVar(&kcp, "kcp", kcp, "Enable KCP mode")
	flags.BoolVar(&verbose, "v", verbose, "Enable verbose logging")
	flags.BoolVar(&hoppingPT, "h", hoppingPT, "Connect with openvpn over udp in hopping mode")
	flags.StringVar(&obfs4Certs, "c", obfs4Certs, "The remote obfs4 certificates separated by commas. If hopping is not enabled only the first cert will be used")
	flags.StringVar(&obfs4Remotes, "r", obfs4Remotes, "The remote obfs4 endpoint ips (no port) separated by commas. If hopping is not enabled only the first cert will be used")
	flags.StringVar(&obfs4RemotePort, "rp", obfs4RemotePort, "The remote obfs4 endpoint port to use. Only applicable to NON-hopping")
	flags.UintVar(&portCount, "pc", portCount, "The number of ports to try for each remote. Only applicable to hopping")
	flags.Int64Var(&portSeed, "ps", portSeed, "The random seed to generate ports from. Only applicable to hopping")
	flags.UintVar(&minHopSeconds, "m", minHopSeconds, "The minimun number of seconds to wait before hopping. Only applicable to hopping")
	flags.UintVar(&hopJitter, "j", hopJitter, "A random range to wait (on top of the minimum) seconds before hopping. Only applicable to hopping")
	flags.StringVar(&socksPort, "p", socksPort, "The port for the local socks5 proxy (default: 8080)")
	flags.StringVar(&socksHost, "i", socksHost, "The host for the local socks5 proxy (default: localhost)")
	err := flags.Parse(os.Args[1:])
	if err != nil {
		logger.Fatalf("error parsing flags: %v", err)
	}
	// Configure logging.
	if verbose {
		debug.SetOutput(os.Stderr)
	}

	// TODO make sure we're disabling all the crypto options for KCP

	obfs4RemotesList := strings.Split(obfs4Remotes, ",")
	obfs4CertsList := strings.Split(obfs4Certs, ",")

	if len(obfs4CertsList) != len(obfs4RemotesList) {
		logger.Fatalf("number of obfs4 remotes must match number of obfs4 certs")
	}

	socksAddr := net.JoinHostPort(socksHost, socksPort)

	logger.Printf("socksAddr: %+v", socksAddr)

	// Setup graceful shutdown.
	ctx, stop := signal.NotifyContext(context.Background(), os.Interrupt)

	var c client.ObfsClient

	if hoppingPT {
		config := client.HoppingConfig{
			KCP:           kcp,
			ProxyAddr:     socksAddr,
			Remotes:       obfs4RemotesList,
			Certs:         obfs4CertsList,
			PortSeed:      portSeed,
			PortCount:     portCount,
			MinHopSeconds: minHopSeconds,
			HopJitter:     hopJitter,
		}
		c = client.NewHopClient(ctx, stop, config)
	} else {
		c = client.NewClient(ctx, kcp, socksAddr, obfs4CertsList[0])
	}
	if _, err = c.Start(); err != nil {
		log.Fatal(err)
	}
}
