import type {SourceScript, VirtualCode} from '@volar/language-core';
import type {URI} from 'vscode-uri';
import {DocumentsAndMap} from '@volar/language-service/lib/utils/featureWorkers';
import {forEachEmbeddedCode} from '@volar/language-core';
import {LanguageServiceContext} from '@vue/language-server';
import type {ProjectContext} from "@volar/language-service"

export function getDocs(context: LanguageServiceContext, uri: URI): DocumentsAndMap | undefined {
  let sourceScript = context.language.scripts.get(uri);
  if (!sourceScript) return undefined;

  if (sourceScript.generated) {
    let virtualCode = findMainEmbeddedCode(sourceScript.generated.root);
    if (!virtualCode) return undefined;
    return getEmbeddedDocument(context, sourceScript, virtualCode);
  }
  return undefined;
}

function findMainEmbeddedCode(root: VirtualCode): VirtualCode | undefined {
  for (const virtualCode of forEachEmbeddedCode(root)) {
    if (/script_(js|jsx|ts|tsx)/.test(virtualCode.id)) {
      return virtualCode;
    }
  }
}

function getEmbeddedDocument(
  context: LanguageServiceContext,
  sourceScript: SourceScript<URI>,
  embeddedCode: VirtualCode,
): DocumentsAndMap | undefined {
  const embeddedDocumentUri = context.encodeEmbeddedDocumentUri(sourceScript.id, embeddedCode.id);
  if (!context.disabledEmbeddedDocumentUris.get(embeddedDocumentUri)) {
    return [
      context.documents.get(sourceScript.id, sourceScript.languageId, sourceScript.snapshot),
      context.documents.get(embeddedDocumentUri, embeddedCode.languageId, embeddedCode.snapshot),
      context.language.maps.get(embeddedCode, sourceScript),
    ];
  }
}

export function getLanguageService(context: LanguageServiceContext) {
  return context.inject<(import('volar-service-typescript').Provide), "typescript/languageService">("typescript/languageService");
}

export function getCancellationToken(context: LanguageServiceContext) {
  return context.inject<(import('volar-service-typescript').Provide), "typescript/languageServiceHost">("typescript/languageServiceHost")?.getCancellationToken?.();
}

export function asFileName(typescript: ProjectContext["typescript"] & {}, uri: URI): string {
  return typescript.uriConverter.asFileName(uri);
}

export function asUri(typescript: ProjectContext["typescript"] & {}, fileName: string): URI {
  return typescript.uriConverter.asUri(fileName);
}