/***************************************************************************
 *                                                                         *
 *   alignment.cpp  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <iostream>
#include <fstream>
#include <string.h>
#include "vec.h"
#include "ali.h"
#include "ptn.h"
#include "optpairseq.h"
#include "outstream.h"
#include "model.h"
#include "interface.h"
#include "codonmodel.h"
#include "rannum.h"
#include <time.h>

#ifdef PARALLEL
#	include <mpi.h>
#endif // PARALLEL

int nto[NUM_CHAR];
Ali alignment;

using namespace std;


extern int isContinuous;


void createPosGenChar () {
	memset(nto, BS_UNKNOWN, sizeof(nto));

	//here come for Nucleotides
	nto[static_cast<int> ('A')] = BASE_A;
	nto[static_cast<int> ('C')] = BASE_C;
	nto[static_cast<int> ('T')] = BASE_T;
	nto[static_cast<int> ('G')] = BASE_G;



	nto[static_cast<int> ('M')] = BASE_AC;
	nto[static_cast<int> ('R')] = BASE_AG;
	nto[static_cast<int> ('W')] = BASE_AT;

	nto[static_cast<int> ('S')] = BASE_CG;
	nto[static_cast<int> ('Y')] = BASE_CT;
	nto[static_cast<int> ('K')] = BASE_GT;

	nto[static_cast<int> ('B')] = BASE_CGT;
	nto[static_cast<int> ('D')] = BASE_AGT;
	nto[static_cast<int> ('H')] = BASE_ACT;
	nto[static_cast<int> ('V')] = BASE_ACG;

	//here come for AA
	nto[static_cast<int> ('A')] = STATE_A;
	nto[static_cast<int> ('C')] = STATE_C;
	nto[static_cast<int> ('T')] = STATE_T;
	nto[static_cast<int> ('G')] = STATE_G;

	nto[static_cast<int> ('R')] = STATE_R;
	nto[static_cast<int> ('N')] = STATE_N;
	nto[static_cast<int> ('D')] = STATE_D;
	nto[static_cast<int> ('Q')] = STATE_Q;

	nto[static_cast<int> ('E')] = STATE_E;
	nto[static_cast<int> ('H')] = STATE_H;
	nto[static_cast<int> ('I')] = STATE_I;
	nto[static_cast<int> ('L')] = STATE_L;

	nto[static_cast<int> ('K')] = STATE_K;
	nto[static_cast<int> ('M')] = STATE_M;
	nto[static_cast<int> ('F')] = STATE_F;
	nto[static_cast<int> ('P')] = STATE_P;

	nto[static_cast<int> ('S')] = STATE_S;
	nto[static_cast<int> ('W')] = STATE_W;
	nto[static_cast<int> ('Y')] = STATE_Y;
	nto[static_cast<int> ('V')] = STATE_V;
}


//===============================================
void createAmino () {

	/*  for (int count_ = 0; count_ < NUM_CHAR; count_ ++)
	    nto [ count_ ] = BS_UNKNOWN;
	*/
	memset(nto, BS_UNKNOWN, sizeof(nto));

	nto[static_cast<int> ('A')] = STATE_A;
	nto[static_cast<int> ('C')] = STATE_C;
	nto[static_cast<int> ('T')] = STATE_T;
	nto[static_cast<int> ('G')] = STATE_G;

	nto[static_cast<int> ('R')] = STATE_R;
	nto[static_cast<int> ('N')] = STATE_N;
	nto[static_cast<int> ('D')] = STATE_D;
	nto[static_cast<int> ('Q')] = STATE_Q;

	nto[static_cast<int> ('E')] = STATE_E;
	nto[static_cast<int> ('H')] = STATE_H;
	nto[static_cast<int> ('I')] = STATE_I;
	nto[static_cast<int> ('L')] = STATE_L;

	nto[static_cast<int> ('K')] = STATE_K;
	nto[static_cast<int> ('M')] = STATE_M;
	nto[static_cast<int> ('F')] = STATE_F;
	nto[static_cast<int> ('P')] = STATE_P;

	nto[static_cast<int> ('S')] = STATE_S;
	nto[static_cast<int> ('W')] = STATE_W;
	nto[static_cast<int> ('Y')] = STATE_Y;
	nto[static_cast<int> ('V')] = STATE_V;
}


void createDNA () {

	memset(nto, BS_UNKNOWN, sizeof(nto));

	nto[static_cast<int> ('A')] = BASE_A;
	nto[static_cast<int> ('C')] = BASE_C;
	nto[static_cast<int> ('T')] = BASE_T;
	nto[static_cast<int> ('G')] = BASE_G;


	nto[static_cast<int> ('M')] = BASE_AC;
	nto[static_cast<int> ('R')] = BASE_AG;
	nto[static_cast<int> ('W')] = BASE_AT;

	nto[static_cast<int> ('S')] = BASE_CG;
	nto[static_cast<int> ('Y')] = BASE_CT;
	nto[static_cast<int> ('K')] = BASE_GT;

	nto[static_cast<int> ('B')] = BASE_CGT;
	nto[static_cast<int> ('D')] = BASE_AGT;
	nto[static_cast<int> ('H')] = BASE_ACT;
	nto[static_cast<int> ('V')] = BASE_ACG;
}


//construction function
Ali::Ali () {
	longestSeqNameLen_ = -1;
}

//clean all content of this class
void Ali::clean () {
	nSeq_ = 0;
	nSite_ = 0;
	isGenDisMatCmped_ = 0;
	bootstrapped = false;
}

//this function is used for constructing this class
void Ali::doConstructor () {
	clean ();
}

//--------------------------------------------------------------------
//everything is inited here
void Ali::init () {
	if (isContinuous) {
		loadGenDis(out_prefix);
	} else
	cmpGenDis (out_prefix);
}

//=============================================
//set ptnNo to site
void Ali::setPtn (int siteNo, int ptnNo) {
	if (ptnArr_.getSize() < alignment.getNSite())
		ptnArr_.set(alignment.getNSite(), alignment.getNSite());
	ptnArr_[siteNo] = ptnNo;
}

//=============================================
int Ali::getPtn (int siteNo) {
	return ptnArr_[siteNo];
}


//get the order of the sequence whose name is input name
int Ali::getSeqNo (Vec<char> &name) {
	for (int count_ = 0; count_ < nSeq_; count_ ++)
		if (items_[count_].getName () == name)
			return count_;

	return -1;
}

//--------------------------------------------------------------------
int Ali::getLongestSeqNameLen () {
	if (longestSeqNameLen_ > 0)
		return longestSeqNameLen_;

	longestSeqNameLen_ = 3;
	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++) {
		int seqNameLen_ = items_[seqNo_].getName ().getSize ();
		if (seqNameLen_ > longestSeqNameLen_)
			longestSeqNameLen_ = seqNameLen_;
	}

	return longestSeqNameLen_;
}

ostream &Ali::write (ostream &out) {
	out << nSeq_ << "  " << nSite_ << endl;
	//int len = getLongestSeqNameLen()+2;
	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		items_[seqNo_].write (out);
	return out;

}


//--------------------------------------------------------------------
DATA_TYPE Ali::detectDataType () {
	int nState_ = 0;
	int nDNA_ = 0;
	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		for (int siteNo_ = 0; siteNo_ < nSite_; siteNo_ ++)
			if (items_[seqNo_].items_[siteNo_] != '?') {
				nState_ ++;
				if (items_[seqNo_].items_[siteNo_] == 'A' || items_[seqNo_].items_[siteNo_] == 'C'
				        || items_[seqNo_].items_[siteNo_] == 'G' || items_[seqNo_].items_[siteNo_] == 'T')
					nDNA_ ++;
			}

	//cout << (double)nDNA_ / nState_ << endl;

	if (nDNA_ > 0.75 * nState_)
		return NUCLEOTIDE;
	else
		return AMINO_ACID;
}

/**
convert the data input file
*/
void Ali::convertDataType(DATA_TYPE data_type) {

	if (data_type == NUCLEOTIDE)
		createDNA();
	else if (data_type == AMINO_ACID)
		createAmino();

	if (data_type == NUCLEOTIDE || data_type == AMINO_ACID) {

		for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
			for (int siteNo_ = 0; siteNo_ < nSite_; siteNo_ ++) {
				items_[seqNo_].items_[siteNo_] = nto [ (int)items_[seqNo_].items_[siteNo_] ];
			}
	} else { // convert Protein-coding DNA to Codon sites
		createDNA();
		buildCodonInfo();
		convertToCodons();
	}
}

// codon nucleotide order in the map
//const char codon_nt_order[4] = {'T', 'C', 'A', 'G'};

/**
	return order of nucleotide, used when converting to codon
*/
int nucCodonOrder(char nt, bool &error) {
	switch (nt) {
	case BASE_T: return 0;
	case BASE_C: return 1;
	case BASE_A: return 2;
	case BASE_G: return 3;
	default:
		error = true;
		return -1;
	}
}

/**
	return nucleotide character
*/
char codonNucLetter(char nt) {
	switch (nt) {
	case BASE_T: return 'T';
	case BASE_C: return 'C';
	case BASE_A: return 'A';
	case BASE_G: return 'G';
	default:
		return -1;
	}
}

void Ali::convertToCodons() {
	bool error = false;
	// first check the data
	if (nSite_ % 3 != 0) {
		if (!error && isMasterProc())
			cout << "Warning: Length of Protein-coding DNA not divided by 3!" << endl;
		error = true;
	}
	nSite_ /= 3;
	bool *error_site = new bool[nSite_];
	memset(error_site, 0, nSite_);
	int seq, site;

	for (seq = 0; seq < nSeq_; seq++)
		for (site = 0; site < nSite_; site++)
			if (!error_site[site]) {
				int dna_site = site * 3;
				int cpos1 = nucCodonOrder(items_[seq].items_[dna_site], error_site[site]);
				int cpos2 = nucCodonOrder(items_[seq].items_[dna_site+1], error_site[site]);
				int cpos3 = nucCodonOrder(items_[seq].items_[dna_site+2], error_site[site]);
				if (error_site[site]) {
					/*
					if (!error)
						cout << "Error: Protein-coding DNA must only contain T C A or G!" << endl;
					*/
					error = true;
					continue;
				}
				int codon = cpos1 * 16 + cpos2 * 4 + cpos3;
				if (codon_aa_map[codon].amino_acid == STOP_CODON) {
					/*
					if (!error)
						cout << "Error: Stop codon detected!" << endl;
					*/
					error = true;
					error_site[site] = true;
				}
			}


	int remaining_sites = nSite_;

	if (error) {
		remaining_sites = 0;
		for (site = 0; site < nSite_; site++)
			if (!error_site[site])
				remaining_sites ++;
		if (isMasterProc()) {
			const char *ofilename = "out.codon";

			ofstream fout (ofilename);
			fout << nSeq_ << " " << remaining_sites * 3 << endl;
			for (int seq = 0; seq < nSeq_; seq++) {
				fout << items_[seq].getName() << "  ";

				for (int site = 0; site < nSite_; site++)
					if (!error_site[site]) {
						int dna_site = site * 3;
						fout << codonNucLetter(items_[seq].items_[dna_site]);
						fout << codonNucLetter(items_[seq].items_[dna_site+1]);
						fout << codonNucLetter(items_[seq].items_[dna_site+2]);
					}
				fout << endl;
			}
			fout.close();
			cout << "Sequence alignment contains gaps or stop codon!" << endl;
			cout << "Cleaned data is written to file: " << ofilename << endl;
			cout << "Program continues with cleaned data" << endl;
		}
		//exit(1);
	}

	char *newseq = new char[nSite_];

	if (error) {
		origin_site.set(remaining_sites, remaining_sites);
		int count = 0;
		for (site = 0; site < nSite_; site++)
			if (!error_site[site])
				origin_site[count++] = site;
	}

	// now convert data
	for (seq = 0; seq < nSeq_; seq++) {
		int count = 0;
		for (site = 0; site < nSite_; site++)
			if (!error_site[site]) {
				int dna_site = site * 3;
				int cpos1 = nucCodonOrder(items_[seq].items_[dna_site], error);
				int cpos2 = nucCodonOrder(items_[seq].items_[dna_site+1], error);
				int cpos3 = nucCodonOrder(items_[seq].items_[dna_site+2], error);
				int codon = cpos1 * 16 + cpos2 * 4 + cpos3;
				codon = codon_index[codon];
				newseq[count] = (char)codon;
				count++;
			}
		//cout << count << " " << nSite_ << endl;
		items_[seq].set(count, count);
		for (int i = 0; i < count; i++)
			items_[seq][i] = newseq[i];
	}
	nSite_ = remaining_sites;
	delete newseq;
	delete error_site;
}

//--------------------------------------------------------------------
PAM_TYPE Ali::detectDataSource () {
	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		for (int siteNo_ = 0; siteNo_ < nSite_; siteNo_ ++)
			if (items_[seqNo_].items_[siteNo_] == '?')
				return REAL;
	return SIMULATION;
}

//--------------------------------------------------------------------
//get the name of the sequence at certain orther
Vec<char> &Ali::getName (int ndNo) {
	static Vec<char> name_;
	name_.clean ();
	if (ndNo < nSeq_) {//it is a external node
		name_ = items_[ndNo].getName ();
		return name_;
	} else { //it is an internal node
		name_.convert (ndNo);
		return name_;
	}
}

//--------------------------------------------------------------------
//get the name of the sequence at certain orther
void Ali::getName (int ndNo, Vec<char> &name) {
	if (ndNo < nSeq_) //it is a external node
		name = items_[ndNo].getName ();
	else
		name.convert (ndNo);
}

//--------------------------------------------------------------------
//return the content at the given site, only active sequences are taking into account
void Ali::getPtn (const int siteNo, Ptn &ptn) {
	ptn.setSize (nSeq_);
	int len_ = 0;
	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_++) {
		ptn [len_] = items_[seqNo_][siteNo];
		len_ ++;
	};
}

//--------------------------------  ------------------------------------
//return the number sequences of this alignment
int Ali::getNSeq () {
	return nSeq_;
}

//--------------------------------------------------------------------
//return the number of nucleotides in a sequence
int Ali::getNSite () {
	return nSite_;
}

//--------------------------------------------------------------------
//return the sequence index
Seq &Ali::getSeq (const int index) {
	return items_[index];
}

//--------------------------------------------------------------------
//read input from the file fileName
int Ali::readInput (const char *fileName) {
	ifstream in;
	in.open (fileName);
	if (!in)
		return 0;

	int tmpNSeq_ = 0;
	int tmpNSite_ = 0;
	in >> tmpNSeq_ >> tmpNSite_;
	if (tmpNSeq_ == 0 || tmpNSite_ == 0) {
		if (isMasterProc())
			cout << "ERROR: The alignment file must start with #sequences and #sites" << endl;
		return 0;
	}
	else {
		setLimit (tmpNSeq_, tmpNSite_);
	}

	in.ignore (INFINITIVE, '\n');
	for (int count_ = 0; count_ < nSeq_; count_ ++) {
		items_[count_].readInput (in, nSite_, 1);
		items_[count_].setId (count_);
	}

	while (items_[0].getSize () < tmpNSite_) {
		in.ignore (INFINITIVE, '\n');
		int nNu_ = 0;
		for (int count_ = 0; count_ < nSeq_; count_ ++)
			nNu_ = items_[count_].readInput (in, nSite_, 0);
		if (nNu_ == 0) {
			if (isMasterProc()) cout << "ERROR: empty sequence" << endl;
			return 0;
		}
	}
	strcpy (fileName_, fileName);

	for (int i = 0; i < nSeq_; i++)
		if (items_[i].getNSite() < tmpNSite_) {
			//cout << "Sequence name " << items_[i].getName();
			Vec<char> *tmpname = &items_[i].getName();
			int j, cut = 10;
			if (tmpname->getSize() <= cut) break;
			
			int offset = tmpname->getSize() - cut;
			items_[i].setSize(items_[i].getNSite() + offset);
			for (j = items_[i].getNSite()-1; j >= offset; j--)
				items_[i][j] = items_[i][j-offset];
			for (j = cut; j < tmpname->getSize(); j++)
				items_[i][j-cut] = (*tmpname)[j];
				
			tmpname->setSize(cut);
			//(*tmpname)[cut] = 0;
			cout << " -> " << items_[i].getName() << endl;
		}
	for (int i = 0; i < nSeq_; i++)
		if (items_[i].getNSite() < tmpNSite_) {
			cout << "Error: Sequence " << items_[i].getName() << " has too short length of " << items_[i].getNSite() << endl;
			Utl::announceError("Please check the input file again");
		}
	return 1;
}


//--------------------------------------------------------------------
//sort sequences by their name, it serves for deburging
void Ali::sortSeq () {
	for (int seqNo1_ = 0; seqNo1_ < nSeq_; seqNo1_ ++)
		for (int seqNo2_ = seqNo1_ + 1; seqNo2_ < nSeq_; seqNo2_ ++)
			if (items_[seqNo1_].getName ().cmp (items_[seqNo2_].getName () ) == 1) {
				Seq tmpSeq_;
				tmpSeq_ = items_[seqNo1_];
				items_[seqNo1_] = items_[seqNo2_];
				items_[seqNo2_] = tmpSeq_;
			}
}

//--------------------------------------------------------------------
/*return the genetic distance between sequence1 and sequence2,
the distance is cmped by maximum likelihood
*/
double Ali::getGenDis (const int seqNo1, const int seqNo2) {
	if (isGenDisMatCmped_ == 0)
		cmpGenDis (out_prefix);
	return genDisMat_ [seqNo1] [seqNo2];
}

int Ali::checkAli () {
	int seqNo_, seqNo2_;
	for (seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		for (seqNo2_ = seqNo_ + 1; seqNo2_ < nSeq_; seqNo2_ ++)
			if (items_[seqNo_].getName () == items_[seqNo2_].getName ()) {
				//          std::cout <<"Fuck!!!!" << endl;
				//        std::cout << seqNo_ << "/ (" << items_[seqNo_].getName () <<")" << endl;
				//      std::cout << seqNo2_ <<"/ (" <<items_[seqNo2_].getName () <<")" << endl;
				if (isMasterProc())
					std::cout << "The name " << items_[seqNo_].getName () << 
						" appears more than one time in the alignment file ";
				exit (EXIT_FAILURE);
			}
	for (seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		if (items_[seqNo_].getNSite() != getNSite()) {
			cout << "Sequence " << items_[seqNo_].getName() << " has a different length of " << items_[seqNo_].getNSite() << endl;
			Utl::announceError("Bad input alignment");
		}	
	//  std::cout <<"no identical name" << endl;
	return 0;
}



//--------------------------------------------------------------------
//return the address of genetic distance matrix
Mat<double> *Ali::getGenDis () {
	if (isGenDisMatCmped_ == 0)
		cmpGenDis (out_prefix);
	return &genDisMat_;
}

//--------------------------------------------------------------------
//create the distance matrix for any two sequences basing on maximum likelihood method
void Ali::cmpGenDis (char* out_prefix) {

#ifdef PARALLEL

	int cnt;

	if (isMasterProc()) {
		std::cout <<"Waiting for slaves to start computing genetic distance matrix...";
		cout.flush();
	}
	// synchronising the computing of genetic distance
	if (program_mode == PROG_COMPUTE_MATRIX)
		MPI_Barrier(MPI_COMM_WORLD);
	if (isMasterProc())
		std::cout <<"OK" << endl;

	double start_time = MPI_Wtime();


	// calulate the work load for my process
	int total_work = nSeq_*(nSeq_-1) / 2;
	int work_load = (int) ceil( double(total_work) / mpi_size );
	int work_start = work_load * mpi_myrank;
	double *global_vec = new double[work_load * mpi_size];
	double *local_vec = new double[work_load];
	// caculate the starting row and column
	int start_row = 0;
	int upper = nSeq_ - 1;

	// calculating the starting point by binary search
	while (upper - start_row > 1) {
		int middle = (upper + start_row) / 2;
		if ( 2 * work_start < (2*nSeq_ - middle - 1) * (middle))
			upper = middle;
		else
			start_row = middle;
	}
	int start_col = work_start - (2*nSeq_ - start_row - 1) * start_row / 2 + start_row + 1;

	int *row_idx = new int[work_load];
	int *col_idx = new int[work_load];
	int row = start_row, col = start_col;
	for (int no = 0; no < work_load; no ++) {
		row_idx[no] = row;
		col_idx[no] = col;
		//cout << row_idx[no] << " " << col_idx[no] << endl;
		col++;
		if  (col >= nSeq_) {
			row++;
			col = row+1;
		}
	}
	
	//cout << mpi_myrank << "-> row: " << start_row << " col: " << start_col << endl;

	// calculate the local vector
#ifdef _OPENMP
	#pragma omp parallel for private(cnt)
#endif
	for (cnt = 0; cnt < work_load; cnt++) {
		int myrow = row_idx[cnt];
		int mycol = col_idx[cnt];
		if (myrow < nSeq_ && mycol < nSeq_) 
			local_vec[cnt] = opt_pairseq.cmp(Ali::getSeq (myrow), Ali::getSeq (mycol) );
	}

	// gather results from all processes
	MPI_Allgather(local_vec, work_load, MPI_DOUBLE, global_vec, work_load, MPI_DOUBLE, MPI_COMM_WORLD);

	// now calculating the matrix from vector
	start_row = 0;
	start_col = 1;
	for (cnt = 0; cnt < total_work; cnt ++ ) {
		genDisMat_[start_row][start_col] = global_vec[cnt];
		genDisMat_[start_col][start_row] = global_vec[cnt];
		if (start_col < nSeq_ - 1)
			start_col ++;
		else {
			if (start_row == nSeq_ - 2) break;
			start_row ++;
			start_col = start_row + 1;
		}
	}

	for (cnt = 0; cnt < nSeq_; cnt ++)
		genDisMat_[cnt][cnt] = 0.0;

	delete col_idx;
	delete row_idx;
	
	delete local_vec;
	delete global_vec;

#else // not parallel
	cout << "Computing genetic distance matrix..." << endl;
	time_t start_time = time(NULL);
	
	int num_pairs = (nSeq_*(nSeq_-1)) / 2;

	int *row_idx = new int[num_pairs];
	int *col_idx = new int[num_pairs];
	int row = 0, col = 1;
	for (int no = 0; no < num_pairs; no ++) {
		row_idx[no] = row;
		col_idx[no] = col;
		col++;
		if  (col >= nSeq_) {
			row++;
			col = row+1;
		}
	}
	for (row = 0; row < nSeq_; row++)
		genDisMat_[row][row] = 0.0;
	
#ifdef _OPENMP
	#pragma omp parallel for 
#endif

	for (int pair = 0; pair < num_pairs; pair ++) {
		int seqNo1_ = row_idx[pair];
		int seqNo2_ = col_idx[pair];
		
		//for (int seqNo2_ = seqNo1_ + 1; seqNo2_ < nSeq_; seqNo2_ ++) {
			genDisMat_[seqNo1_][seqNo2_] = opt_pairseq.cmp (Ali::getSeq (seqNo1_), Ali::getSeq (seqNo2_) );
			genDisMat_[seqNo2_][seqNo1_] = genDisMat_[seqNo1_][seqNo2_];
		//}
	}
	delete col_idx;
	delete row_idx;
#endif

#ifdef PARALLEL
	if (program_mode == PROG_COMPUTE_MATRIX)
		MPI_Barrier(MPI_COMM_WORLD);

	if (isMasterProc()) {
		cout << "Time computing matrix: " << MPI_Wtime() - start_time << endl;
	}
#else
	cout << "Time computing matrix: " << time(NULL) - start_time << " s" << endl;

#endif

	if (isMasterProc()) {
		ofstream out;

		string distName_;
		getOutFileName(SUF_DIST, distName_);

		out.open (distName_.c_str());
		OutStream::writeGenDis (genDisMat_, out);
		out.close ();
	}

#ifdef PARALLEL
	// all slaves wait until master finishes writing dist file
	if (program_mode == PROG_COMPUTE_MATRIX)
		MPI_Barrier(MPI_COMM_WORLD);
#endif

	if (program_mode == PROG_COMPUTE_MATRIX)
		Finalize(0);

	isGenDisMatCmped_ = 1;
}

/* make all frequencies a little different */
void Ali::convfreq(DVec20 &freqemp, int num_state) {
	int i, j, maxi=0;
	double freq, maxfreq, sum;
	bool zero_detected = false;

	sum = 0.0;
	maxfreq = 0.0;
	for (i = 0; i < num_state; i++) {
		freq = freqemp[i];
		if (freq < MIN_FRQ) { freqemp[i] = MIN_FRQ; zero_detected = true; }
		if (freq > maxfreq) {
			maxfreq = freq;
			maxi = i;
		}

		sum += freqemp[i];
	}
	freqemp[maxi] += 1.0 - sum;

	for (i = 0; i < num_state - 1; i++) {
		for (j = i + 1; j < num_state; j++) {
			if (freqemp[i] == freqemp[j]) {
				freqemp[i] += MIN_FRQ_DIF/2.0;
				freqemp[j] -= MIN_FRQ_DIF/2.0;
			}
		}
	}
	if (zero_detected & isMasterProc()) {
		cout << "Some codon does not appear in the alignment, assume very small frequency." << endl;
	}
} /* convfreq */

//--------------------------------------------------------------------
//compute the state frequency of this alignment
void Ali::estimateStateFrq (DVec20 &stateFrqArr) {
	int stateNo_;
	int nState_ = mymodel.getNState ();


	for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++)
		stateFrqArr [ stateNo_ ] = 1.0 / nState_;

	int NUM_TIME = 8;
	//app = appeareance
	for (int time_ = 0; time_ < NUM_TIME; time_ ++) {
		DVec20 timeAppArr_;
		for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++)
			timeAppArr_[stateNo_] = 0.0;

		for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++) {
			Seq *seq_;
			seq_ = &items_[seqNo_];
			for (int siteNo_ = 0; siteNo_ < nSite_; siteNo_ ++) {
				int stateNo_ = (*seq_)[siteNo_];

				DVec20 siteAppArr_; //App = appearance
				Ali::getApp (stateNo_, siteAppArr_);

				DVec20 newSiteAppArr_;
				double totalSiteApp_ = 0.0;
				for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++) {
					newSiteAppArr_[stateNo_] = stateFrqArr[stateNo_] * siteAppArr_[stateNo_];
					totalSiteApp_ += newSiteAppArr_[stateNo_];
				}

				for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++)
					timeAppArr_[stateNo_] += newSiteAppArr_[stateNo_] / totalSiteApp_;
			}
		}

		double totalTimeApp_ = 0.0;
		int stateNo_;
		for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++)
			totalTimeApp_ += timeAppArr_[stateNo_];


		for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++)
			stateFrqArr[stateNo_] = timeAppArr_[stateNo_] / totalTimeApp_;

	} //end of for time_

	//  std::cout << "state frequency ..." << endl;
	// for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++)
	// std::cout << stateFrqArr[stateNo_] << endl;

}

/**
	estimate the codon frequency
*/
void Ali::estimateCodonFrq(DVec20 &stateFrqArr) {
	if (isMasterProc())
		cout << "Estimating Codon frequencies from data..." << endl;
	int nState_ = mymodel.getNState();

	if (mymodel.baseFrqType_ == EQUAL) {
		for (int codon = 0; codon < NUM_CODON; codon++) {
			stateFrqArr[codon] = 1.0 / NUM_CODON;
		}
		cout << " equal" << endl;
	} else if (mymodel.baseFrqType_ == F3x4) {

		double freq[NUM_BASE][CODON_LENGTH];
		memset(freq, 0, sizeof(freq));
		int seq, pos, site;
		for (seq = 0; seq < nSeq_; seq++) {
			for (site = 0; site < nSite_; site++) {
				char * codon_name = codon_list[(int)items_[seq][site]].name;
				for (pos = 0; pos < CODON_LENGTH; pos++) {
					int baseNr = nto[(int)codon_name[pos]];
					freq[baseNr][pos] ++;
				}
			}
		}

		for (pos = 0; pos < CODON_LENGTH; pos++) {
			double sum = 0;
			int base;
			for (base = 0; base < NUM_BASE; base++)
				sum += freq[base][pos];
			for (base = 0; base < NUM_BASE; base++)
				freq[base][pos] /= sum;
		}

		// print out frequencies
		for (pos = 0; pos < CODON_LENGTH; pos++) {
			int base;
			cout << "position " << pos+1;
			cout.precision(5);
			for (base = 0; base < NUM_BASE; base++) {
				switch (base) {
				case BASE_A: cout << "  A: "; break;
				case BASE_T: cout << "  T: "; break;
				case BASE_G: cout << "  G: "; break;
				case BASE_C: cout << "  C: "; break;
				}
				cout << freq[base][pos];
			}
			cout << endl;
		}

		for (int codon = 0; codon < NUM_CODON; codon++) {
			char *codon_name = codon_list[codon].name;
			double cfreq = 1.0;
			for (pos = 0; pos < CODON_LENGTH; pos++) {
				int base = nto[(int)codon_name[pos]];
				cfreq *= freq[base][pos];
			}
			stateFrqArr[codon] = cfreq;
		}
		double sum = 0;
		int stateNo_;
		for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++) {
			//std::cout << stateFrqArr[stateNo_] << endl;
			sum += stateFrqArr[stateNo_];
		}
		for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++) {
			stateFrqArr[stateNo_] /= sum;
		}
		if (isMasterProc())
			cout << endl;
	} else if (mymodel.baseFrqType_ == F1x4) {

		double freq[NUM_BASE];
		memset(freq, 0, sizeof(freq));
		int seq, pos, site, base;
		for (seq = 0; seq < nSeq_; seq++) {
			for (site = 0; site < nSite_; site++) {
				char * codon_name = codon_list[(int)items_[seq][site]].name;
				for (pos = 0; pos < CODON_LENGTH; pos++) {
					int baseNr = nto[(int)codon_name[pos]];
					freq[baseNr] ++;
				}
			}
		}

		double sum = 0;
		for (base = 0; base < NUM_BASE; base++)
			sum += freq[base];
		for (base = 0; base < NUM_BASE; base++)
			freq[base] /= sum;

		// print out frequencies
		if (isMasterProc()) {
			cout.precision(5);
			for (base = 0; base < NUM_BASE; base++) {
				switch (base) {
				case BASE_A: cout << "  A: "; break;
				case BASE_T: cout << "  T: "; break;
				case BASE_G: cout << "  G: "; break;
				case BASE_C: cout << "  C: "; break;
				}
				cout << freq[base];
			}
			cout << endl;
		}

		for (int codon = 0; codon < NUM_CODON; codon++) {
			char *codon_name = codon_list[codon].name;
			double cfreq = 1.0;
			for (pos = 0; pos < CODON_LENGTH; pos++) {
				int base = nto[(int)codon_name[pos]];
				cfreq *= freq[base];
			}
			stateFrqArr[codon] = cfreq;
		}
		sum = 0;
		int stateNo_;
		for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++) {
			//std::cout << stateFrqArr[stateNo_] << endl;
			sum += stateFrqArr[stateNo_];
		}
		for (stateNo_ = 0; stateNo_ < nState_; stateNo_ ++) {
			stateFrqArr[stateNo_] /= sum;
		}
		if (isMasterProc())
			cout << endl;
	} else {
		estimateStateFrq(stateFrqArr);
	}
	convfreq(stateFrqArr, NUM_CODON);
}


double Ali::estimateProb (BASE baseX, BASE baseY) {
	double totalFrq_ = 0.0;
	for (int siteNo_ = 0; siteNo_ < nSite_; siteNo_ ++) {
		double nBaseX_ = 0.0;
		double nBaseY_ = 0.0;

		for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++) {
			if ((items_[seqNo_].items_[siteNo_]) == baseX)
				nBaseX_ += 1.0;
			if (items_[seqNo_].items_[siteNo_] == baseY)
				nBaseY_ += 1.0;
		}

		totalFrq_ += (nBaseX_ * nBaseY_);
	}

	double prob_ = totalFrq_ / ((double)nSeq_ * (nSeq_ - 1) * nSite_);
	return prob_;
}

//==========================================================================================
void Ali::estimateGenPam (double &tsAG, double &tsCT, double &tvAC, double &tvAT, double &tvCG, double &tvGT) {
	tsAG = estimateProb (BASE_A, BASE_G) * 2.0;
	tsCT = estimateProb (BASE_C, BASE_T)  * 2.0;
	tvAC = estimateProb (BASE_A, BASE_C);
	tvAT = estimateProb (BASE_A, BASE_T);
	tvCG = estimateProb (BASE_C, BASE_G);
	tvGT = estimateProb (BASE_G, BASE_T);
	if (tvGT != 0.0) {
		tsAG /= tvGT;
		tsCT /= tvGT;
		tvAC /= tvGT;
		tvAT /= tvGT;
		tvCG /= tvGT;
		tvGT = 1.0;
	}
}

//==========================================================================================
double Ali::estimateTsTvRatio () {
	double tsAG_ = estimateProb (BASE_A, BASE_G) * 2.0;
	double tsCT_ = estimateProb (BASE_C, BASE_T)  * 2.0;
	double tvAC_ = estimateProb (BASE_A, BASE_C);
	double tvAT_ = estimateProb (BASE_A, BASE_T);
	double tvCG_ = estimateProb (BASE_C, BASE_G);
	double tvGT_ = estimateProb (BASE_G, BASE_T);

	return 2.0 * (tsAG_ + tsCT_) / (tvAC_ + tvAT_ + tvCG_ +tvGT_);
}


//set the memory for this class
void Ali::setLimit (int nSeq, int nSite) {
	//do not need set limit again
	if (nSeq_ == nSeq && nSite_ == nSite)
		return;

	nSeq_ = nSeq;
	nSite_ = nSite;

	genDisMat_.setLimit (nSeq_, nSeq_);

	items_.set (nSeq_, nSeq_);
	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		items_[seqNo_].set (nSite_, nSite_);
}

//--------------------------------------------------------------------
//release all memmory for this class
void Ali::release () {
	genDisMat_.release ();

	int realLimit_ = items_.getLimit ();
	items_.setSize (realLimit_);
	for (int count_ = 0; count_ < realLimit_; count_ ++)
		items_[count_].release ();
	items_.release ();
}

void Ali::getApp (int stateNo, DVec20 &appArr) {
	int nState_ = mymodel.getNState ();
	int count_;
	for (count_ = 0; count_ < nState_; count_ ++)
		appArr[count_] = 0;

	if (mymodel.getDataType () == AMINO_ACID || mymodel.getDataType() == CODON) {
		if (stateNo == BS_UNKNOWN)
			for (count_ = 0; count_ < nState_; count_ ++)
				appArr[count_] = 1;
		else
			appArr[stateNo] = 1;

		return ;
	}

	int ta = 0;
	int tc = 0;
	int tg = 0;
	int tt = 0;
	switch (stateNo) {
	case BASE_A:
		ta = 1;
		break;
	case BASE_C:
		tc = 1;
		break;
	case BASE_G:
		tg = 1;
		break;
	case BASE_T:
		tt = 1;
		break;
	case BASE_AC:
		ta = tc = 1;
		break;
	case BASE_AG:
		ta = tg = 1;
		break;
	case BASE_AT:
		ta = tt = 1;
		break;
	case BASE_CG:
		tc = tg = 1;
		break;
	case BASE_CT:
		tc = tt = 1;
		break;
	case BASE_GT:
		tg = tt = 1;
		break;
	case BASE_ACG:
		ta = tc = tg = 1;
		break;
	case BASE_ACT:
		ta = tc = tt = 1;
		break;
	case BASE_AGT:
		ta = tg = tt = 1;
		break;
	case BASE_CGT:
		tc = tg = tt = 1;
		break;
	default:
		ta = tc = tg = tt = 1;
	} //end of switch

	appArr[BASE_A] = ta;
	appArr[BASE_C] = tc;
	appArr[BASE_G] = tg;
	appArr[BASE_T] = tt;

}

int Ali::getOriginSite(int site) {
	return (origin_site.getSize() > 0) ? origin_site[site] : site;
}

double Ali::calcNumConstSites() {
	int i, j;
	int num_csites = 0;
	int nstate = mymodel.getNState();
	for (i = 0; i < nSite_; i++) {
		bool const_site = true;
		char mych = nstate;
		for (j = 0; j < nSeq_; j++) 
			if (items_[j][i] < nstate) {
				if (mych < nstate && items_[j][i] != mych) {
					const_site = false;
					break;
				}
				mych = items_[j][i];
			}
		if (const_site) num_csites ++;
	}
	return (double)num_csites / nSite_;
}

void Ali::generateBootstrap() {
	//if (bootstrapped) 
		//return;
	Vec<int> sample(nSite_, nSite_);
	//Vec<char> stemp(nSite_, nSite_);
	int i, j;
	if (orig_items_.getSize() == 0) {
		// initialize the original sequences
		orig_items_.set(nSeq_, nSeq_);
		for (i = 0; i < nSeq_; i++) {
			orig_items_[i].set(nSite_, nSite_);			
			for (j = 0; j < nSite_; j++)
				orig_items_[i][j] = items_[i][j];
		}
	}
	for (i = 0; i < nSite_; i++)
		sample[i] = RanNum::getRanInt(nSite_);
	for (i = 0; i < nSeq_; i++) {
		for (j = 0; j < nSite_; j++)
			items_[i][j] = orig_items_[i][sample[j]];
	}
	bootstrapped = true;
}

void Ali::loadBootstrap() {
	int i, j;
	if (orig_items_.getSize() == 0) {
		// initialize the original sequences
		orig_items_.set(nSeq_, nSeq_);
		for (i = 0; i < nSeq_; i++) {
			orig_items_[i].set(nSite_, nSite_);			
			for (j = 0; j < nSite_; j++)
				orig_items_[i][j] = items_[i][j];
		}
	}

	string old_filename = fileName_;
	string filename;
	getOutFileName(SUF_BOOTSAMPLE, filename);
	readInput(filename.c_str());
	strcpy (fileName_, old_filename.c_str());
	bootstrapped = true;
}

bool Ali::isBootstrapped() {
	return bootstrapped;
}

bool str_equal(string &str, Vec<char> &vstr) {
	if ((int)str.length() != vstr.getSize()) return false;
	for (unsigned int i = 0; i < str.length(); i++)
		if (str[i] != vstr[i]) return false;
	return true;
}

void Ali::loadGenDis(char *prefix) {

	ifstream in;

	string distName_;
	getOutFileName(SUF_DIST, distName_);

	in.open (distName_.c_str());
	if (!in.is_open()) {
		cmpGenDis(prefix);
		return;
	}
	if (isMasterProc())
		cout << "Loading genetic distance matrix..." << endl;

	int nseq, seq, seq2;
	in >> nseq;
	if (nseq != nSeq_) {
		Utl::announceError("Distance file has different number of taxa");
	}
	for (seq = 0; seq < nseq; seq++) {
		string str;
		in >> str;
		if (!str_equal(str, items_[seq].getName())) 
			Utl::announceError("Distance file has different sequence name");
		for (seq2 = 0; seq2 < nseq; seq2++)
			in >> genDisMat_[seq][seq2];
		if (genDisMat_[seq][seq] != 0.0) 
			Utl::announceError("A diagonal element is not equal zero");
	}
	for (seq = 1; seq < nseq; seq++)
		for (seq2 = 0; seq2 < seq; seq2++)
			if (genDisMat_[seq][seq2] != genDisMat_[seq2][seq])
				Utl::announceError("Distance matrix is not symmetric");
	in.close ();
	isGenDisMatCmped_ = 1;
}
//--------------------------------------------------------------------
//destruction function
Ali::~Ali () {
	release ();
	//  std::cout << "this is the destructor of ali class " << endl;
}
