/***************************************************************************
 *                                                                         *
 *                                                                         *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "codonyn98.h"
#include <assert.h>
#include "opturtree.h"
#include "model.h"

CodonYN98::CodonYN98()
		: CodonModel() {
	nsSyRatio = 0.2;
}


/**
		calculate the rate parameters (not rate matrix)
*/
void CodonYN98::calcRateParameters() {

	for (int i = 0; i < NUM_CODON; i++)
		for (int j = i+1; j < NUM_CODON; j++) {
			if (codon_trans[i][j].source == 0) { // more than 2 different codon positions
				continue;
			}
			//unitRateMat_[i][j]
			if (codon_trans[i][j].is_transition)
				if (codon_trans[i][j].is_synonymous)
					// synonymous transition
					unitRateMat_[i][j] = tsTvRatio_;
				else // nonsynonymous transition
					unitRateMat_[i][j] = tsTvRatio_ * nsSyRatio;
			else
				if (codon_trans[i][j].is_synonymous)
					// synonymous transversion
					unitRateMat_[i][j] = 1.0;
				else
					// nonsynonymous transversion
					unitRateMat_[i][j] = nsSyRatio;

			unitRateMat_[j][i] = unitRateMat_[i][j];
		}

	/*
	// print info
	cout << "rate mat: " << endl;
	cout.precision(3);
	for (int i = 0; i < NUM_CODON; i++) {
		for (int j = 0; j < NUM_CODON; j++) {
			printf("%2.0f ", unitRateMat_[i][j]);
		}
		cout << endl;
	}*/
	/*
	for (int i = 0; i < NUM_CODON; i++)
		for (int j = i+1; j < NUM_CODON; j++) {
			if (codon_trans[i][j].source == 0) { // more than 2 different codon positions
				continue;
			}
			unitRateMat_[i][j] /= sum;
			unitRateMat_[j][i] /= sum;
		}*/
}

double CodonYN98::cmpNegLogLi (double value) {
	if (isActPam_ == TS_TV_RATIO)
		tsTvRatio_ = value;
	else
		nsSyRatio = value;

	reInit ();

	opt_urtree.cmpLiNd ();
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	return -logLi_;
}

/**
	the target function which needs to be optimized
	@param x the input vector x
	@return the function value at x
*/
double CodonYN98::targetFunk(double x[]) {
	/*
	if (!checkRange(x)) {
		return INFINITIVE;
	}
	*/
	unpackData(x);
	reInit();
	opt_urtree.cmpLiNd ();
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	return -logLi_;
}

/**
	return the number of dimensions
*/
int CodonYN98::getNDim() {
	if (mymodel.getTsTvRatioType() == ESTIMATE)
		return 2;
	else
		return 1;
}

/**
	pack the data to the vector x
*/
void CodonYN98::packData( double x[], double lower[], double upper[], bool bound_check[]) {
	x[1] = nsSyRatio;
	lower[1] = MIN_NS_SY_RATIO;
	upper[1] = MAX_NS_SY_RATIO;
	bound_check[1] = true;
	if (mymodel.getTsTvRatioType() == ESTIMATE) {
		x[2] = tsTvRatio_;
		lower[2] = MIN_TS_TV_RATIO;
		upper[2] = MAX_TS_TV_RATIO;
		bound_check[2] = true;
	}
}

/**
	unpack data from vector x
*/
void CodonYN98::unpackData(double x[]) {
	nsSyRatio = x[1];
	if (mymodel.getTsTvRatioType() == ESTIMATE) 
		tsTvRatio_ = x[2];
}

bool CodonYN98::optPam () {
	double fx_, error_;
	double oriTsTvRatio_ = tsTvRatio_;
	double oriNsSyRatio = nsSyRatio;

	//  double logLi_ = opt_urtree.getLogLi ();
	if (isMasterProc())
		std::cout <<"Optimizing Yang & Nielsen 1998 codon-based model parameters ..." << endl;
	Brent::turnOnOptedPam ();
	double logLi_ = opt_urtree.cmpLogLi ();
	//  std::cout <<"Log likelihood: " << logLi_ << endl;


	if (mymodel.pam_brent == 2 || getNDim() <= 2) {

		if (mymodel.getTsTvRatioType() == ESTIMATE) {
			isActPam_ = TS_TV_RATIO;
			if (isMasterProc())
				std::cout <<"   Optimizing transition/transversion ratio..." << endl;
			tsTvRatio_ = Brent::optOneDim (MIN_TS_TV_RATIO, tsTvRatio_, MAX_TS_TV_RATIO,
			                               EPS_MODEL_PAM_ERROR, &fx_, &error_);
			reInit ();
		}//end of optimizing tsTvRatio

		if (true) {
			isActPam_ = NS_SY_RATIO;
			if (isMasterProc())
				std::cout <<"   Optimizing nonsynonymous/synonymous ratio..." << endl;
			nsSyRatio = Brent::optOneDim (MIN_NS_SY_RATIO, nsSyRatio, MAX_NS_SY_RATIO,
			                              EPS_MODEL_PAM_ERROR, &fx_, &error_);
			reInit ();

		} //end of optimizing pyPuRatio
	} else {
		// optimize by BFGS algorithm
		int ndim = getNDim();

		double *variables = new double[ndim+1];
		double *upper_bound = new double[ndim+1];
		double *lower_bound = new double[ndim+1];
		bool *bound_check = new bool[ndim+1];
		
		packData(variables, lower_bound, upper_bound, bound_check);
		int iter;
		double freturn;
		dfpmin(variables, ndim, lower_bound, upper_bound, EPS_MODEL_FUNC_ERROR, &iter, &freturn);
		unpackData(variables);
		reInit();

		delete bound_check;
		delete lower_bound;
		delete upper_bound;
		delete variables;

	}

	//      logLi_ = opt_urtree.cmpLogLi ();
	//     std::cout <<"Log likelihood: " << logLi_ << endl;
	//------------------------------/------------------------------/





	opt_urtree.cmpLiNd ();
	logLi_ = opt_urtree.getLogLi ();

	std::cout.precision (10);
	//  std::cout << "Log likelihood: " << logLi_ << endl;

	if (isMasterProc()) {

		std::cout.precision (5);
		std::cout << "Transition/transversion ratio  =  " << tsTvRatio_/2.0 << endl;
		std::cout << "Nonsynonymous/synonymous ratio =  " << nsSyRatio << endl;
	}

	Brent::turnOffOptedPam ();
	return (fabs (tsTvRatio_ - oriTsTvRatio_) > EPS_MODEL_PAM_ERROR ||
	        fabs (nsSyRatio - oriNsSyRatio) > EPS_MODEL_PAM_ERROR );
}


/**
	@return Nonsynonymous/Synonymous Ratio
*/
double CodonYN98::getNsSyRatio() {
	return nsSyRatio;
}

/**
	set Nonsynonymous/Synonymous Ratio
*/
void CodonYN98::setNsSyRatio(double ratio) {
	nsSyRatio = ratio;
}

CodonYN98::~CodonYN98() {}



