/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// rate.cpp: implementation of the Rate class.
//
//////////////////////////////////////////////////////////////////////

#include <math.h>
#include <iostream>

#include "rate.h"
#include "opturtree.h"
#include "brent.h"
#include "constant.h"
#include "ptnls.h"
#include "ali.h"
#include "model.h"
#include "outstream.h"

/*
int Rate::nRate_;
Vec<double> Rate::rateArr_;
 
Vec<double> Rate::ptnRateArr_;
 
Vec<SeqPair> Rate::indSeqPairArr_;
 
int Rate::actPtnNo_;
int Rate::isOptedSpecificRate_;
 
double Rate::xxx_gammaShape_;
PAM_TYPE Rate::gammaShapeType_;
RATE_TYPE Rate::rateType_;
 
*/

const double MIN_PROB_INVAR_SITE = 0.0001;
const double MAX_PROB_INVAR_SITE = 0.999;

Rate myrate;

extern int isContinuous;


//extern int nto[NUM_CHAR];

bool atBound(double min, double num, double max) {
//	return (num < min + min*0.1) || (num > max - max*0.1);
	return (num > max - max*0.1);
}



//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


//====================================================================================
//construction function, everything is automatically inited here
void Rate::init () {
	//if (isContinuous == 0) {
		nRate_ = 1;
		xxx_gammaShape_ = 1.0;
		prob_invar_site = 0.1;
	//}
	gamma_scale = 1.0;
	rateArr_.setLimit (nRate_);

	int nPtn_ = ptnlist.getNPtn ();
	ptnRateArr_.set (nPtn_, nPtn_);
	for (int ptnNo_ = 0; ptnNo_ < nPtn_; ptnNo_ ++)
		ptnRateArr_[ptnNo_] = 1.0;

	isOptedSpecificRate_ = 0;
}

void Rate::setGammaShape (const double gammaShape) {
	if (gammaShape < 0.0) {
		gammaShapeType_ = ESTIMATE;
		xxx_gammaShape_ = 1.0;
	} else {
		xxx_gammaShape_ = gammaShape;
		gammaShapeType_ = USER_DEFINED;
	}
}

void Rate::setGammaShapeType (const PAM_TYPE gammaShapeType) {
	gammaShapeType_ = gammaShapeType;
}

//===========================================
PAM_TYPE Rate::getGammaShapeType () {
	return gammaShapeType_;
}

//===========================================
PAM_TYPE Rate::getInvarSiteType () {
	//if (use_invar_site)
		return invar_site_type;
	//else return USER_DEFINED; 
}

double Rate::getInvarRatio () {
	return prob_invar_site;
}

//====================================================================================
//return the rateArr_[rateNo];
double Rate::getRate (int rateNo) {
	return rateArr_ [rateNo];
}

//====================================================================================
//return the number of categories rate
int Rate::getNRate () {
	if (isNsSyHeterogenous()) 
		return nsSy_classes;

	return nRate_/* + use_invar_site*/;

}

//====================================================================================
/*...set number rates of rate heterogeneity
*/
void Rate::setNRate (const int nRate) {
	nRate_ = nRate;
}


//====================================================================================
// get the nRate mean rates of nCatefories different categories
Vec<double> &Rate::getRate () {
	return rateArr_;
}

//====================================================================================

int Rate::isOptedSpecificRate () {
	return isOptedSpecificRate_;
}

bool Rate::isNsSyHeterogenous() {
	return mymodel.getModelType() == Codon_NY98;
	//return isCodonModel(mymodel.getModelType()) && (nsSy_ratio_type == SITE_SPECIFIC);
}

//====================================================================================
//return the shape of gamma distribution
double Rate::getGammaShape () {
	return xxx_gammaShape_;
}


double Rate::cmpNegLogLi(double value) {
	if (flag == USE_GAMMA_SHAPE)
		return cmpNegLogLiGammaShape(value);
	if (flag == USE_GAMMA_SCALE)
		return cmpNegLogLiGammaScale(value);
	else if (flag == USE_PTN)
		return cmpNegLogLiPtn(value);
	else
		return cmpNegLogLiProbInvarSite(value);
}

//--------------------------------------------------------------------
/*compute the log likelihood when given gamma shape parameter
this function is used to optimize the gamma shape parameter of this model
*/
double Rate::cmpNegLogLiProbInvarSite (double prob_invar) {
	prob_invar_site = prob_invar;

	//opt_urtree.cmpLiNd ();
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	return -logLi_;
}


//--------------------------------------------------------------------
/*compute the log likelihood when given gamma shape parameter
this function is used to optimize the gamma shape parameter of this model
*/
double Rate::cmpNegLogLiGammaShape (double gammaShape) {
	std::cout.precision (10);
	xxx_gammaShape_ = gammaShape;
	if (xxx_gammaShape_ < MIN_GAMMA_SHAPE)
		xxx_gammaShape_ = MIN_GAMMA_SHAPE;
	if (xxx_gammaShape_ > MAX_GAMMA_SHAPE)
		xxx_gammaShape_ = MAX_GAMMA_SHAPE;

	doYangDiscreteGamma (xxx_gammaShape_, gamma_scale, nRate_, rateArr_);

	opt_urtree.cmpLiNd ();
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	return -logLi_;
}


//--------------------------------------------------------------------
/*compute the log likelihood when given gamma shape parameter
this function is used to optimize the gamma shape parameter of this model
*/
double Rate::cmpNegLogLiGammaScale (double scale) {
	std::cout.precision (10);
	gamma_scale = scale;

	doYangDiscreteGamma (xxx_gammaShape_, gamma_scale, nRate_, rateArr_);

	opt_urtree.cmpLiNd ();
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	return -logLi_;
}



//====================================================================================
//cmp likelihood when active pattern changes using Sonja Meyer and Arndt von Haeseler, MBE-2003
double Rate::cmpNegLogLiPtn (double ptnRate) {
	double  logLiPtn_ = 0.0;

	int nSeqPair_ = indSeqPairArr_.getSize ();
	DVec20 stateFrqArr_;
	mymodel.getStateFrq (stateFrqArr_);
	int nState_ = mymodel.getNState ();

	for (int pairNo_ = 0; pairNo_ < nSeqPair_; pairNo_ ++) {
		int seqXNo_ = indSeqPairArr_[pairNo_].seqXNo_;
		int seqYNo_ = indSeqPairArr_[pairNo_].seqYNo_;
		double disXY_ = indSeqPairArr_[pairNo_].genDis_;



		int siteNo_ = ptnlist.siteArr_ [actPtnNo_];
		int stateXNo_ = alignment.getSeq (seqXNo_).items_[siteNo_];
		int stateYNo_ = alignment.getSeq (seqYNo_).items_[siteNo_];
		if (stateXNo_ < nState_ && stateYNo_ < nState_) {
			double prob_ = mymodel.cmpProbChange (stateXNo_, stateYNo_, disXY_ * ptnRate);
			logLiPtn_ += log (stateFrqArr_[stateXNo_] * prob_);
		}
	}

	return - logLiPtn_;
}

//====================================================================================
void Rate::optRatePam () {

	double fx_, error_;
	turnOnOptedPam ();

	if (getInvarSiteType() == ESTIMATE && prob_invar_site == 0.0) {
		prob_invar_site = alignment.calcNumConstSites();
		if (isMasterProc())
			cout << "Init p_invar = " << prob_invar_site << endl;
	}
	

	
	if (getGammaShapeType() == ESTIMATE) {
	if (isMasterProc())
		std::cout << "Optimizing gamma shape parameter ... " << endl;

	setOptFlag(USE_GAMMA_SHAPE);
	xxx_gammaShape_ = Brent::optOneDim (MIN_GAMMA_SHAPE, xxx_gammaShape_, MAX_GAMMA_SHAPE,
	                                    EPS_GAMMA_SHAPE_ERROR, &fx_, &error_);
	}
	
	/*									
	if (isMasterProc())
		std::cout << "Optimizing gamma scale parameter ... " << endl;

	setOptFlag(USE_GAMMA_SCALE);
	gamma_scale = Brent::optOneDim (MIN_GAMMA_SCALE, gamma_scale , MAX_GAMMA_SCALE,
	                                    EPS_GAMMA_SHAPE_ERROR, &fx_, &error_);
	*/
	//double logLi_ = -cmpNegLogLiGammaShape (xxx_gammaShape_);

	if (getInvarSiteType() == ESTIMATE) {
		if (isMasterProc())
			std::cout << "Optimizing propotion of invariable sites ... " << endl;
		setOptFlag(USE_PROB_INVAR_SITE);
		prob_invar_site = Brent::optOneDim (MIN_PROB_INVAR_SITE, prob_invar_site, MAX_PROB_INVAR_SITE,
											EPS_GAMMA_SHAPE_ERROR, &fx_, &error_);
	}
	
	turnOffOptedPam ();
	std::cout.precision (10);

	//  std::cout <<"Log likelihood: " << logLi_ << endl;
	std::cout.precision (5);
	if (isMasterProc()) {
		std::cout << "Gamma distribution shape = " << xxx_gammaShape_ << endl;
		//std::cout << "Gamma distribution scale = " << gamma_scale << endl;
		if (invar_site_type == ESTIMATE)
			std::cout << "Proportion of invariable sites = " << prob_invar_site << endl;
	}
}

//====================================================================================
void Rate::setType (RATE_TYPE rateType) {
	rateType_ = rateType;
}

//====================================================================================
RATE_TYPE Rate::getType () {
	return rateType_;
}


//====================================================================================

//optimize specific site substitution rate for PTNNO using Sonja Meyer and Arndt von Haeseler, MBE-2003
double Rate::optPtn (int ptnNo) {
	actPtnNo_ = ptnNo;
	double fx_, error_;

	double ptnRate_ = ptnRateArr_[actPtnNo_];
	setOptFlag(USE_PTN);
	ptnRate_ = optOneDim (MIN_PTN_RATE, ptnRate_, MAX_PTN_RATE,
	                      EPS_PTN_ERROR, &fx_, &error_);
	return ptnRate_;
}


//====================================================================================
//optimize specific site substitution rates using Sonja Meyer and Arndt von Haeseler, MBE-2003
void Rate::scalePtnRate () {
	double totalRate_ = 0.0;

	int nPtn_ = ptnlist.getNPtn ();
	int ptnNo_;
	int count = 0;
	for (ptnNo_ = 0; ptnNo_ < nPtn_; ptnNo_ ++)
		if (!atBound(MIN_PTN_RATE, ptnRateArr_[ptnNo_], MAX_PTN_RATE)) {
			totalRate_ += ptnRateArr_[ptnNo_] * ptnlist.getWeight (ptnNo_);
			count += ptnlist.getWeight(ptnNo_);
		}

	if (totalRate_ < ZERO) {
		for (ptnNo_ = 0; ptnNo_ < nPtn_; ptnNo_ ++)
			ptnRateArr_[ptnNo_] = 1.0;
	} else {
		int nSite_ = count;
		for (ptnNo_ = 0; ptnNo_ < nPtn_; ptnNo_ ++)
			if (!atBound(MIN_PTN_RATE, ptnRateArr_[ptnNo_], MAX_PTN_RATE))
				ptnRateArr_[ptnNo_] = Utl::getMax ( (ptnRateArr_[ptnNo_] / totalRate_) * nSite_, MIN_PTN_RATE);
	}
	//  OutStream::write (ptnRateArr_, std::cout) << endl;
}

//====================================================================================
//optimize specific site substitution rates using Sonja Meyer and Arndt von Haeseler, MBE-2003
void Rate::estimatePtnRate () {

	int nSeqPair_ = indSeqPairArr_.getSize ();
	int nPtn_ = ptnlist.getNPtn ();
	for (int ptnNo_ = 0; ptnNo_ < nPtn_; ptnNo_ ++) {
		double sum_ = 0.0;
		for (int pairNo_ = 0; pairNo_ < nSeqPair_; pairNo_ ++) {
			int seqXNo_ = indSeqPairArr_[pairNo_].seqXNo_;
			int seqYNo_ = indSeqPairArr_[pairNo_].seqYNo_;

			int siteNo_ = ptnlist.siteArr_ [ptnNo_];
			if (alignment.getSeq (seqXNo_).items_[siteNo_] == alignment.getSeq (seqYNo_).items_[siteNo_])
				sum_ += 1.0;

		}

		ptnRateArr_[ptnNo_] = sum_;
	}

	scalePtnRate ();
}


//====================================================================================
//optimize specific site substitution rates using Sonja Meyer and Arndt von Haeseler, MBE-2003
int Rate::optSpecificSite (Vec<SeqPair> &indSeqPairArr) {
	if (isMasterProc())
		std::cout <<"Optimizing site-specific substitution rates ..." << endl;
	Brent::turnOnOptedPam ();

	indSeqPairArr_ = indSeqPairArr;


	int nPtn_ = ptnlist.getNPtn ();
	ptnRateArr_.set (nPtn_, nPtn_);

	//   estimatePtnRate ();

	double oldLogLi_ = opt_urtree.getLogLi ();

	isOptedSpecificRate_ = 1;
	double oldSpecificLogLi_ = oldLogLi_;
	//   std::cout << oldSpecificLogLi_ << endl;

	int nIt_ = 0;
	Vec<double> oriPtnRateArr_;
	//   int nSite_ = alignment.getNSite ();

	//do {
		nIt_ ++;

		oriPtnRateArr_ = ptnRateArr_;
	
		bool close_to_bound = false;

		for (int ptnNo_ = 0; ptnNo_ < nPtn_; ptnNo_ ++) {
			ptnRateArr_[ptnNo_] = optPtn (ptnNo_);
			if (atBound(MIN_PTN_RATE, ptnRateArr_[ptnNo_], MAX_PTN_RATE)) {
				close_to_bound = true;
			}
		}
		if (isMasterProc() && close_to_bound) {
			cout << "Sites with very high rate detected, they are marked as UNKNOWN in rate file." << endl;
		/*
			for (int siteNo = 0; siteNo < alignment.getNSite(); siteNo++) {
				int ptnNo = alignment.getPtn (siteNo);
				if (atBound(MIN_PTN_RATE, ptnRateArr_[ptnNo], MAX_PTN_RATE)) cout << siteNo+1 << "  ";
			}
			cout << endl;*/
		}
			
		scalePtnRate ();

		double newSpecificLogLi_ = opt_urtree.optBranches (MAX_IT_UR_BR_PAM_OPT);
		std::cout.precision (20);

		if (newSpecificLogLi_ <= oldSpecificLogLi_ + EPS_LOGLI_SITE_SPECIFIC_RATE_OPT) {
			ptnRateArr_ = oriPtnRateArr_;
		}
		//      } else
		//      std::cout << "New log likelihood value = " << newSpecificLogLi_ << endl;

		//oldSpecificLogLi_ = newSpecificLogLi_;
	//} while (nIt_ < MAX_IT_SITE_SPECIFIC_RATE_OPT);

	// OutStream::write (ptnRateArr_, std::cout) << endl;
	Brent::turnOffOptedPam ();
	//return (nIt_ > 1);
	return 1;
}

//==================================================================
//optimize specific site substitution rates using Sonja Meyer and Arndt von Haeseler, MBE-2003
Vec<double> *Rate::getPtnRate () {
	return &ptnRateArr_;
}

//==================================================================
//optimize specific site substitution rates using Sonja Meyer and Arndt von Haeseler, MBE-2003
double Rate::getPtnRate (int ptnNo) {
	return ptnRateArr_[ptnNo];
}

//====================================================================================
/* discrete Gamma according to Yang 1994 (JME 39:306-314) */
void Rate::doYangDiscreteGamma (const double shape, const double scale, const int nRate, Vec<double> &rateArr) {
	int rateNo_;
	//rateArr.set (nRate, nRate);
	for (rateNo_ = 0; rateNo_ < nRate; rateNo_ ++) {
		double prob_ = ( 2.0 * rateNo_ + 1 ) / (2.0 * nRate);
		rateArr[ rateNo_ ] = cmpPerPointGamma (prob_, shape, scale);
	}

	//rescale in order to make mean equal to 1.0

	double sumRate_ = 0.0;
	for (rateNo_ = 0; rateNo_ < nRate; rateNo_ ++)
		sumRate_ += rateArr[ rateNo_];

	for (rateNo_ = 0; rateNo_ < nRate; rateNo_ ++)
		rateArr[ rateNo_ ] = rateArr[ rateNo_ ] * nRate_ / sumRate_;

	/* check for very small rates */
	for (rateNo_ = 0; rateNo_ < nRate; rateNo_ ++)
		if (rateArr[rateNo_] < MIN_RATE)
			rateArr[rateNo_] = MIN_RATE;
	/*
	  std::cout << "gamma shape =  " << shape << endl;
	  for (rateNo_ = 0; rateNo_ < nRate; rateNo_ ++) 
	    std::cout << rateArr[rateNo_] << endl;
	*/
}

//====================================================================================
//create the mean rate list using Gamma distribution, using existed parameters
void Rate::create () {
	rateArr_.set (nRate_/* + use_invar_site*/, nRate_/* + use_invar_site*/);


	if (nRate_ == 1) {
		rateArr_[0] = 1.0;
		return;
	}

	doYangDiscreteGamma (xxx_gammaShape_, gamma_scale, nRate_, rateArr_);
	//if (use_invar_site)
		//rateArr_[nRate_] = 0.0;

	//  std::cout << "rates ... " << endl;
	//  for (int rateNo_ = 0; rateNo_ < nRate_; rateNo_ ++)
	//  	std::cout << rateArr_[rateNo_] << endl;
}

//====================================================================================
//get the prob of rateNo
double Rate::getProb () {
	//if (!use_invar_site)
		//return (1.0 / nRate_);
	//else
		return (1.0 - prob_invar_site) / nRate_;
}


//get the prob of rateNo
/*
double Rate::getProb (int rateNo) {
	if (!use_invar_site)
		return (1.0 / nRate_);
	else
		if (rateNo < nRate_)
			return (1.0 - prob_invar_site) / nRate_;
		else 
			return prob_invar_site;
}*/


//====================================================================================
//Normally, beta is assigned equal to shape
double Rate::cmpPerPointGamma (const double prob, const double shape, const double scale) {
	double perPoint_ = cmpPointChi2 (prob, 2.0 * shape) / (2.0 * shape);
	perPoint_ = perPoint_ < 0.0 ? -perPoint_ : perPoint_;
	return perPoint_;
}

/*NUMERICAL SUBROUTINES
**************************************************************************************
 
**************************************************************************************
**************************************************************************************
**************************************************************************************
**************************************************************************************/

/* THE FOLLOWING CODE COMES FROM tools.c in Yang's PAML package */

//----------------------------------------------------------------------------------------
double Rate::cmpLnGamma (double alpha) {
	/* returns ln(gamma(alpha)) for alpha>0, accurate to 10 decimal places.
	   Stirling's formula is used for the central polynomial part of the procedure.
	   Pike MC & Hill ID (1966) Algorithm 291: Logarithm of the gamma function.
	   Communications of the Association for Computing Machinery, 9:684
	*/
	double x=alpha, f=0, z;

	if (x<7) {
		f=1;  z=x-1;
		while (++z<7)  f*=z;
		x=z;   f=-log(f);
	}
	z = 1/(x*x);
	return  f + (x-0.5)*log(x) - x + .918938533204673
	        + (((-.000595238095238*z+.000793650793651)*z-.002777777777778)*z
	           +.083333333333333)/x;
} //end of function cmpLnGamma

//----------------------------------------------------------------------------------------
double Rate::cmpIncompleteGamma (double x, double alpha, double ln_gamma_alpha) {
	/* returns the incomplete gamma ratio I(x,alpha) where x is the upper
		   limit of the integration and alpha is the shape parameter.
	   returns (-1) if in error
	   (1) series expansion     if (alpha>x || x<=1)
	   (2) continued fraction   otherwise
	 
	   RATNEST FORTRAN by
	   Bhattacharjee GP (1970) The incomplete gamma integral.  Applied Statistics,
	   19: 285-287 (AS32)
	*/

	int i;
	double p=alpha, g=ln_gamma_alpha;
	double accurate=1e-8, overflow=1e30;
	double factor, gin=0, rn=0, a=0,b=0,an=0,dif=0, term=0, pn[6];

	if (x==0) return (0);
	if (x<0 || p<=0) return (-1);

	factor=exp(p*log(x)-x-g);
	if (x>1 && x>=p) goto l30;
	/* (1) series expansion */
	gin=1;  term=1;  rn=p;
l20:
	rn++;
	term*=x/rn;   gin+=term;

	if (term > accurate) goto l20;

	gin*=factor/p;
	goto l50;
l30:

	/* (2) continued fraction */
	a=1-p;   b=a+x+1;  term=0;
	pn[0]=1;  pn[1]=x;  pn[2]=x+1;  pn[3]=x*b;
	gin=pn[2]/pn[3];
l32:
	a++;  b+=2;  term++;   an=a*term;
	for (i=0; i<2; i++) pn[i+4]=b*pn[i+2]-an*pn[i];
	if (pn[5] == 0) goto l35;
	rn=pn[4]/pn[5];   dif=fabs(gin-rn);
	if (dif>accurate) goto l34;
	if (dif<=accurate*rn) goto l42;
l34:
	gin=rn;
l35:
	for (i=0; i<4; i++) pn[i]=pn[i+2];
	if (fabs(pn[4]) < overflow) goto l32;
	for (i=0; i<4; i++) pn[i]/=overflow;
	goto l32;
l42:
	gin=1-factor*gin;

l50:
	return (gin);
} //end of function cmpIncompleteGamma


//----------------------------------------------------------------------------------------
/* functions concerning the CDF and percentage points of the gamma and
   Chi2 distribution
*/
double Rate::cmpPointNormal (double prob) {
	/* returns z so that Prob{x<z}=prob where x ~ N(0,1) and (1e-12)<prob<1-(1e-12)
	   returns (-9999) if in error
	   Odeh RE & Evans JO (1974) The percentage points of the normal distribution.
	   Applied Statistics 22: 96-97 (AS70)
	 
	   Newer methods:
	     Wichura MJ (1988) Algorithm AS 241: the percentage points of the
	       normal distribution.  37: 477-484.
	     Beasley JD & Springer SG  (1977).  Algorithm AS 111: the percentage 
	       points of the normal distribution.  26: 118-121.
	 
	*/
	double a0=-.322232431088, a1=-1, a2=-.342242088547, a3=-.0204231210245;
	double a4=-.453642210148e-4, b0=.0993484626060, b1=.588581570495;
	double b2=.531103462366, b3=.103537752850, b4=.0038560700634;
	double y, z=0, p=prob, p1;

	p1 = (p<0.5 ? p : 1-p);

	if (p1<1e-20) return (-9999);

	y = sqrt (log(1/(p1*p1)));
	z = y + ((((y*a4+a3)*y+a2)*y+a1)*y+a0) / ((((y*b4+b3)*y+b2)*y+b1)*y+b0);
	return (p<0.5 ? -z : z);
} //end of function cmpPointNormal



//----------------------------------------------------------------------------------------

double Rate::cmpPointChi2 (double prob, double v) {
	/* returns z so that Prob{x<z}=prob where x is Chi2 distributed with df=v
	   returns -1 if in error.   0.000002<prob<0.999998
	   RATNEST FORTRAN by
	       Best DJ & Roberts DE (1975) The percentage points of the 
	       Chi2 distribution.  Applied Statistics 24: 385-388.  (AS91)
	   Converted into C by Ziheng Yang, Oct. 1993.
	*/
	double e=.5e-6, aa=.6931471805, p=prob, g;
	double xx, c, ch, a=0,q=0,p1=0,p2=0,t=0,x=0,b=0,s1,s2,s3,s4,s5,s6;

	if (p<.000002 || p>.999998 || v<=0) return (-1);

	g = cmpLnGamma (v/2);
	xx=v/2;   c=xx-1;
	if (v >= -1.24*log(p)) goto l1;

	ch=pow((p*xx*exp(g+xx*aa)), 1/xx);
	if (ch-e<0) return (ch);
	goto l4;
l1:
	if (v>.32) goto l3;
	ch=0.4;   a=log(1-p);
l2:
	q=ch;  p1=1+ch*(4.67+ch);  p2=ch*(6.73+ch*(6.66+ch));
	t=-0.5+(4.67+2*ch)/p1 - (6.73+ch*(13.32+3*ch))/p2;
	ch-=(1-exp(a+g+.5*ch+c*aa)*p2/p1)/t;
	if (fabs(q/ch-1)-.01 <= 0) goto l4;
	else                       goto l2;

l3:
	x=cmpPointNormal (p);
	p1=0.222222/v;   ch=v*pow((x*sqrt(p1)+1-p1), 3.0);
	if (ch>2.2*v+6)  ch=-2*(log(1-p)-c*log(.5*ch)+g);
l4:

	do {
		q=ch;   p1=.5*ch;
		if ((t=cmpIncompleteGamma (p1, xx, g))<0) {
			return (-1);
		}
		p2=p-t;
		t=p2*exp(xx*aa+g+p1-c*log(ch));
		b=t/ch;  a=0.5*t-b*c;

		s1=(210+a*(140+a*(105+a*(84+a*(70+60*a))))) / 420;
		s2=(420+a*(735+a*(966+a*(1141+1278*a))))/2520;
		s3=(210+a*(462+a*(707+932*a)))/2520;
		s4=(252+a*(672+1182*a)+c*(294+a*(889+1740*a)))/5040;
		s5=(84+264*a+c*(175+606*a))/2520;
		s6=(120+c*(346+127*c))/5040;
		ch+=t*(1+0.5*t*s1-b*c*(s1-b*(s2-b*(s3-b*(s4-b*(s5-b*s6))))));
	} while (fabs(q/ch-1) > e);

	return (ch);
} //end of function cmpPointChi2


/* THE END OF THE CODES COMMING FROM tools.c in Yang's PAML package */

//--------------------------------------------------------------------------
//release all memory of this class
void Rate::release () {
	rateArr_.release ();
}

//the destructor function
Rate::~Rate () {
	release ();

	//	std::cout << "this is the destructor function of Rate class " << endl;
}
