/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/***************************************************************************
                          urtree.cpp  -  description
                             -------------------
    begin                : Wed Mar 12 2003
    copyright            : (C) 2003 by 
    email                : vinh@cs.uni-duesseldorf.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <iostream>
#include <fstream>

#include "urtree.h"
#include "ali.h"
#include "innd.h"
#include "inndarr.h"
#include "exndarr.h"
#include "brarr.h"
#include "opturtree.h"
#include "outstream.h"
#include "cluster.h"
#include "clusterarr.h"
#include "model.h"
#include "seqpair.h"
#include "constant.h"


#ifdef PARALLEL
#include <mpi.h>
#endif // PARALLEL

using namespace std;

UrTree::UrTree () {
	doConstructor ();
}

//-----------------------------------------------------------
//all things are inited here
void UrTree::doConstructor () {

	maxNExNd_ = alignment.getNSeq ();
	exNdArr_.setLimit (maxNExNd_);

	maxNInNd_ = maxNExNd_ - 2;
	inNdArr_.setLimit (maxNInNd_);

	maxNBr_ = maxNExNd_ * 2 - 3;
	brArr_.setLimit (maxNBr_);

	maxNNd_ = maxNExNd_ + maxNInNd_;

	outGrpNdNo_ = 0;
	isRootedTree_ = 0;
	artInRootNo_ = -1;

	clean ();
	quartetTime_ = 0.0;

	bonusBrArr_ = new int[alignment.getNSeq() * 2];
}

/**
	convert the branch length from the number of nucleotide substitution per nucleotide site
	to number of nucleotide substitution per codon site.
*/
void UrTree::convertBrLenToCodon() {
	for (int i = 0; i < maxNBr_; i++)
		brArr_[i].setLen(brArr_[i].getLen()*3);
}


void UrTree::findNeaExNd (Br<double> &parBr, Vec<int> &neaExNdNoLs, int maxNNeaExNd) {
	neaExNdNoLs.setLimit (maxNExNd_);


	Vec<DBr> brQue_ (maxNBr_);
	brQue_ += parBr;
	int first_ = 0;
	int nNeaExNd_ = 0;

	while (first_ < brQue_.getSize () ) {
		DBr curBr_;
		curBr_ = brQue_[first_];

		int tailNdNo_ = curBr_.getTailNd ();

		if (isExNd (tailNdNo_) == 1) {
			neaExNdNoLs += tailNdNo_;
			nNeaExNd_ ++;
			if (nNeaExNd_ == maxNNeaExNd)
				return ;
		} else {
			DBr lChiBr_;
			DBr rChiBr_;
			getChiBr (curBr_, lChiBr_, rChiBr_);
			brQue_ += lChiBr_;
			brQue_ += rChiBr_;
		}
		first_ ++;
	}
}


//-----------------------------------------------------------
void UrTree::getChiBr (Br<double> &parBr, Br<double> &lChiBr, Br<double> &rChiBr) {
	int tailNdNo_ = parBr.getTailNd ();
	if (isExNd (tailNdNo_) == 1) {
		lChiBr.setId (-1);
		rChiBr.setId (-1);
		return;
	}
	int parHeadNdNo_ = parBr.getHeadNd ();
	int parTailNdNo_ = parBr.getTailNd ();
	int lChiTailNdNo_, rChiTailNdNo_, lBrNo_, rBrNo_;
	inNdArr_[parTailNdNo_].get2RemNeiNdBr (parHeadNdNo_, lChiTailNdNo_, lBrNo_, rChiTailNdNo_, rBrNo_);

	//int lBrNo_ = brArr_.findBr (parTailNdNo_, lChiTailNdNo_);
	lChiBr.set (lBrNo_, parTailNdNo_, lChiTailNdNo_);
	if (parTailNdNo_ < lChiTailNdNo_)
		lChiBr.setDir (StG);
	else
		lChiBr.setDir (GtS);

	//int rBrNo_ = brArr_.findBr (parTailNdNo_, rChiTailNdNo_);
	rChiBr.set (rBrNo_, parTailNdNo_, rChiTailNdNo_);
	if (parTailNdNo_ < rChiTailNdNo_)
		rChiBr.setDir (StG);
	else
		rChiBr.setDir (GtS);
}

void UrTree::mixNeaExNd (int *exNdArr0, int *heiArr0, int *exNdArr1, int *heiArr1, int *exNdArr, int *heiArr) {
	int nNd0_ = 0;
	int nNd1_ = 0;
	int count_;
	for (count_ = 0; count_ < nRep_ ; count_ ++) {
		if (nNd0_ == nRep_  || exNdArr0[nNd0_] == -1) {
			if (nNd1_ == nRep_  || exNdArr1[nNd1_] == -1)
				break;
			else {
				exNdArr[count_] = exNdArr1[nNd1_];
				heiArr[count_] = heiArr1[nNd1_] + 1;
				nNd1_ ++;
			}
		} else {
			if (nNd1_ == nRep_  || exNdArr1[nNd1_] == -1) {
				exNdArr[count_] = exNdArr0[nNd0_];
				heiArr[count_] = heiArr0[nNd0_] + 1;
				nNd0_ ++;
			} else {
				if (heiArr0[nNd0_] < heiArr1[nNd1_]) {
					exNdArr[count_] = exNdArr0[nNd0_];
					heiArr[count_] = heiArr0[nNd0_] + 1;
					nNd0_ ++;
				} else {
					exNdArr[count_] = exNdArr1[nNd1_];
					heiArr[count_] = heiArr1[nNd1_] + 1;
					nNd1_ ++;
				}
			}
		}
	}

	for (int count2_ = count_; count2_ < nRep_ ; count2_ ++)
		exNdArr[count2_] = -1;
}

void UrTree::cmpNeaExNd (int brNo, int isSma, int *neaExNdArr, int *neaHeiArr) {
	Br<double> *br_;
	br_ = &brArr_[brNo];
	if (isSma == 1) {
		if (br_->isSmaCmped_ == 1) {
			for (int count_ = 0; count_ < nRep_ ; count_ ++) {
				neaExNdArr[count_] = br_->smaNeaExNdArr_[count_];
				neaHeiArr[count_] = br_->smaNeaHeiArr_[count_];
			}
		} else {
			int smaNdNo_ = br_->getSmaNd ();
			if (isExNd (smaNdNo_) == 1) {
				for (int count_ = 0; count_ < nRep_ ; count_ ++) {
					neaExNdArr[count_] = -1;
					br_->smaNeaExNdArr_[count_] = -1;
				}
				neaExNdArr[0] = smaNdNo_;
				neaHeiArr[0] = 0;
				br_->smaNeaExNdArr_[0] = smaNdNo_;
				br_->smaNeaHeiArr_[0] = 0;
			} else {
				int lChiBrNo_, rChiBrNo_;
				int lNeaExNdArr_[MAX_NUM_REP + 1];
				int lNeaHeiArr_[MAX_NUM_REP + 1];

				int rNeaExNdArr_[MAX_NUM_REP + 1];
				int rNeaHeiArr_[MAX_NUM_REP + 1];

				inNdArr_[smaNdNo_].get2RemBr (brNo, lChiBrNo_, rChiBrNo_);
				if (brArr_[lChiBrNo_].getGreNd () == smaNdNo_)
					cmpNeaExNd (lChiBrNo_, 1, lNeaExNdArr_, lNeaHeiArr_);
				else
					cmpNeaExNd (lChiBrNo_, 0, lNeaExNdArr_, lNeaHeiArr_);

				if (brArr_[rChiBrNo_].getGreNd () == smaNdNo_)
					cmpNeaExNd (rChiBrNo_, 1, rNeaExNdArr_, rNeaHeiArr_);
				else
					cmpNeaExNd (rChiBrNo_, 0, rNeaExNdArr_, rNeaHeiArr_);

				mixNeaExNd (lNeaExNdArr_, lNeaHeiArr_, rNeaExNdArr_, rNeaHeiArr_, neaExNdArr, neaHeiArr);
				for (int count_ = 0; count_ < nRep_ ; count_ ++) {
					br_->smaNeaExNdArr_[count_] = neaExNdArr[count_];

					br_->smaNeaHeiArr_[count_] = neaHeiArr[count_];
				}
			}
			br_->isSmaCmped_ = 1;
		}
	} else {
		if (br_->isGreCmped_ == 1) {
			for (int count_ = 0; count_ < nRep_ ; count_ ++) {
				neaExNdArr[count_] = br_->greNeaExNdArr_[count_];
				neaHeiArr[count_] = br_->greNeaHeiArr_[count_];
			}
		} else {
			int greNdNo_ = br_->getGreNd ();
			if (isExNd (greNdNo_) == 1) {
				for (int count_ = 0; count_ < nRep_ ; count_ ++) {
					neaExNdArr[count_] = -1;
					br_->greNeaExNdArr_[count_] = -1;
				}
				neaExNdArr[0] = greNdNo_;
				neaHeiArr[0] = 0;
				br_->greNeaExNdArr_[0] = greNdNo_;
				br_->greNeaHeiArr_[0] = 0;
			} else {
				int lChiBrNo_, rChiBrNo_;
				int lNeaExNdArr_[MAX_NUM_REP + 1];
				int lNeaHeiArr_[MAX_NUM_REP + 1];

				int rNeaExNdArr_[MAX_NUM_REP + 1];
				int rNeaHeiArr_[MAX_NUM_REP + 1];

				inNdArr_[greNdNo_].get2RemBr (brNo, lChiBrNo_, rChiBrNo_);
				if (brArr_[lChiBrNo_].getGreNd () == greNdNo_)

					cmpNeaExNd (lChiBrNo_, 1, lNeaExNdArr_, lNeaHeiArr_);
				else
					cmpNeaExNd (lChiBrNo_, 0, lNeaExNdArr_, lNeaHeiArr_);

				if (brArr_[rChiBrNo_].getGreNd () == greNdNo_)
					cmpNeaExNd (rChiBrNo_, 1, rNeaExNdArr_, rNeaHeiArr_);

				else
					cmpNeaExNd (rChiBrNo_, 0, rNeaExNdArr_, rNeaHeiArr_);

				mixNeaExNd (lNeaExNdArr_, lNeaHeiArr_, rNeaExNdArr_, rNeaHeiArr_, neaExNdArr, neaHeiArr);
				for (int count_ = 0; count_ < nRep_ ; count_ ++) {
					br_->greNeaExNdArr_[count_] = neaExNdArr[count_];
					br_->greNeaHeiArr_[count_] = neaHeiArr[count_];
				}
			}
			br_->isGreCmped_ = 1;
		}
	}
}


void UrTree::cmpNeaExNd () {
	Vec<int> *brNoLs_;
	brNoLs_ = brArr_.getBrLs ();



	int nCurBr_ = brNoLs_->getSize ();
	int count_, brNo_;
	for (count_ = 0; count_ < nCurBr_; count_ ++) {
		brNo_ = (*brNoLs_)[count_];
		brArr_[brNo_].isSmaCmped_ = 0;
		brArr_[brNo_].isGreCmped_ = 0;
	}

	int neaExNdArr_[MAX_NUM_REP + 1];
	int neaHeiArr_[MAX_NUM_REP + 1];
	for (count_ = 0; count_ < nCurBr_; count_ ++) {
		brNo_ = (*brNoLs_)[count_];
		if (brArr_[brNo_].isSmaCmped_ == 0)
			cmpNeaExNd (brNo_, 1, neaExNdArr_, neaHeiArr_);

		if (brArr_[brNo_].isGreCmped_ == 0)
			cmpNeaExNd (brNo_, 0, neaExNdArr_, neaHeiArr_);
	}
}

void UrTree::cmpWei4Point (int seqNo0, int seqNo1, int seqNo2, int seqNo3, char &w0, char &w1, char &w2) {
	Mat<double> *genDisMat_;
	genDisMat_ = alignment.getGenDis ();
	double dis0_ =  (*genDisMat_)[seqNo0][seqNo1] + (*genDisMat_)[seqNo2][seqNo3];
	double dis1_ =  (*genDisMat_)[seqNo0][seqNo2] + (*genDisMat_)[seqNo1][seqNo3];
	double dis2_ =  (*genDisMat_)[seqNo0][seqNo3] + (*genDisMat_)[seqNo1][seqNo2];
	w0 = 0;
	w1 = 0;
	w2 = 0;
	if (dis0_ < dis1_ && dis0_ < dis2_)
		w0 = 1;
	else
		if (dis1_ < dis2_)
			w1 = 1;
		else
			w2 = 1;
}

void UrTree::cmpWei (Br<double> &parBr, int ndNo, int *weiArr) {
	int *p_;
	int *l_;
	int *r_;

	int brNo_ = parBr.getId ();
	int lBrNo_, rBrNo_;

	int inNdNo_ = parBr.getTailNd ();
	inNdArr_[inNdNo_].get2RemBr (brNo_, lBrNo_, rBrNo_);

	if (parBr.getSmaNd () == inNdNo_)
		p_ = brArr_[brNo_].greNeaExNdArr_;
	else

		p_ = brArr_[brNo_].smaNeaExNdArr_;

	if (brArr_[lBrNo_].getSmaNd () == inNdNo_)
		l_ = brArr_[lBrNo_].greNeaExNdArr_;
	else
		l_ = brArr_[lBrNo_].smaNeaExNdArr_;

	if (brArr_[rBrNo_].getSmaNd () == inNdNo_)
		r_ = brArr_[rBrNo_].greNeaExNdArr_;
	else
		r_ = brArr_[rBrNo_].smaNeaExNdArr_;

	int limit_ = nRep_ - 1;
	if (p_[nRep_ - 2] == -1 || l_[nRep_ - 2] == -1 || r_[nRep_ - 2] == -1)
		limit_ = nRep_;

	int time_ = 0;
	weiArr[0] = 0;
	weiArr[1] = 0;
	weiArr[2] = 0;
	int MAX_QUARTET = (nRep_ - 1) * (nRep_ - 1) * (nRep_ - 1);
	for (int cp_ = 0; cp_ < limit_; cp_ ++) {
		int pNdNo_ = p_[cp_];
		if (pNdNo_ == -1)
			break;

		for (int cl_ = 0; cl_ < limit_; cl_ ++)
			if (l_[cl_] == -1)
				break;
			else {
				int lNdNo_ = l_[cl_];
				for (int cr_ = 0; cr_ < limit_; cr_ ++)
					if (r_[cr_] == -1)
						break;
					else {
						int rNdNo_ = r_[cr_];
						char w0_, w1_, w2_;

						cmpWei4Point (pNdNo_, ndNo, lNdNo_, rNdNo_, w0_, w2_, w1_);

						weiArr[0] += w0_;
						weiArr[1] += w1_;
						weiArr[2] += w2_;
						time_ ++;
						if (time_ == MAX_QUARTET)
							return ;

					}
			}
	}
}


void UrTree::setBonus (Br<double> &br, int bonusVal) {
	int brNo_ = br.getId ();
	if (br.getGreNd () == br.getHeadNd ())
		brArr_[brNo_].smaBonus_ += bonusVal;

	else

		brArr_[brNo_].greBonus_ += bonusVal;
}



void UrTree::doBonus (int inNdNo, int ndNo) {
	int brNo_ = inNdArr_[inNdNo].getBr (0);
	Br<double> parBr_;
	parBr_ = brArr_[brNo_];
	if (parBr_.getTailNd () != inNdNo)
		parBr_.convert ();
	int weiArr_[3];
	cmpWei (parBr_, ndNo, weiArr_);

	Br<double> lChiBr_;
	Br<double> rChiBr_;
	getChiBr (parBr_, lChiBr_, rChiBr_);

	Br<double> invParBr_;
	invParBr_ = parBr_;
	invParBr_.convert ();
	setBonus (invParBr_, weiArr_[0]);
	setBonus (lChiBr_, weiArr_[1]);
	setBonus (rChiBr_, weiArr_[2]);
}


int UrTree::cmpSmaBonus (Br<double> *br) {
	int brNo_ = br->getId ();
	if (br->isSmaCmped_  == 1)
		return br->smaBonus_;

	int totalBonus_ = br->smaBonus_;
	int greNdNo_ = br->getGreNd ();
	if (isInNd (greNdNo_) == 1) {
		int lBrNo_, rBrNo_;

		inNdArr_[greNdNo_].get2RemBr (brNo_, lBrNo_, rBrNo_);
		Br<double> *lBr_;
		lBr_ = &brArr_[lBrNo_];

		Br<double> *rBr_;
		rBr_ = &brArr_[rBrNo_];

		if (lBr_->getSmaNd () == greNdNo_)
			totalBonus_ += cmpSmaBonus (lBr_);



		else
			totalBonus_ += cmpGreBonus (lBr_);

		if (rBr_->getSmaNd () == greNdNo_)
			totalBonus_ += cmpSmaBonus (rBr_);
		else
			totalBonus_ += cmpGreBonus (rBr_);
	}


	brArr_[brNo_].isSmaCmped_ = 1;
	brArr_[brNo_].smaBonus_ = totalBonus_;
	return totalBonus_;

}
//==============================
int UrTree::cmpGreBonus (Br<double> *br) {
	int brNo_ = br->getId ();
	if (br->isGreCmped_  == 1)
		return br->greBonus_;

	int totalBonus_ = br->greBonus_;
	int smaNdNo_ = br->getSmaNd ();



	if (isInNd (smaNdNo_) == 1) {
		int lBrNo_, rBrNo_;
		inNdArr_[smaNdNo_].get2RemBr (brNo_, lBrNo_, rBrNo_);
		Br<double> *lBr_;
		lBr_ = &brArr_[lBrNo_];

		Br<double> *rBr_;
		rBr_ = &brArr_[rBrNo_];

		if (lBr_->getSmaNd () == smaNdNo_)
			totalBonus_ += cmpSmaBonus (lBr_);
		else
			totalBonus_ += cmpGreBonus (lBr_);

		if (rBr_->getSmaNd () == smaNdNo_)
			totalBonus_ += cmpSmaBonus (rBr_);
		else
			totalBonus_ += cmpGreBonus (rBr_);
	}




	brArr_[brNo_].isGreCmped_ = 1;
	brArr_[brNo_].greBonus_ = totalBonus_;
	return totalBonus_;
}

int UrTree::findBr (int ndNo) {
	Vec<int> *brNoLs_;
	brNoLs_ = brArr_.getBrLs ();

	int nCurBr_ = brNoLs_->getSize ();
	int count_, brNo_;
	for (count_ = 0; count_ < nCurBr_; count_ ++) {
		brNo_ = (*brNoLs_)[count_];
		brArr_[brNo_].smaBonus_ = 0;
		brArr_[brNo_].greBonus_ = 0;
	}

	for (int inNdNo_ = maxNExNd_; inNdNo_ < maxNNd_; inNdNo_ ++)
		if (inNdArr_[inNdNo_].getId () != -1)
			doBonus (inNdNo_, ndNo);

	for (count_ = 0; count_ < nCurBr_; count_ ++) {
		brNo_ = (*brNoLs_)[count_];
		brArr_[brNo_].isSmaCmped_ = 0;
		brArr_[brNo_].isGreCmped_ = 0;
	}

	for (count_ = 0; count_ < nCurBr_; count_ ++) {
		brNo_ = (*brNoLs_)[count_];
		cmpSmaBonus (&brArr_[brNo_]);
		cmpGreBonus (&brArr_[brNo_]);
	}

	int bestBrNo_ = 0;
	int maxBonus_ = 0;
	for (count_ = 0; count_ < nCurBr_; count_ ++) {
		brNo_ = (*brNoLs_)[count_];
		if (brArr_[brNo_].smaBonus_ + brArr_[brNo_].greBonus_ > maxBonus_) {

			maxBonus_ = brArr_[brNo_].smaBonus_ + brArr_[brNo_].greBonus_;
			bestBrNo_ = brNo_;
		}
	}
	return bestBrNo_;
}


void UrTree::insertNd (int ndNo, int nRep) {
	nRep_ = nRep;
	cmpNeaExNd ();

	int brNo_ = findBr (ndNo);
	insertNdBr (ndNo, brNo_);
}


//**************************************************************************************************
//**************************************************************************************************
//**************************************************************************************************
//**************************************************************************************************
//create topological distance vector for this tree topology with respect to seqNo
void UrTree::cmpTopDis (int startSeqNo, Vec<int> &topDisArr) {
	int nSeq_ = alignment.getNSeq ();

	Vec<int> ndQueue_ (nSeq_ * 2, 0);
	topDisArr.set (nSeq_ *  2, nSeq_ * 2);

	int seqNo_;
	for (seqNo_ = 0; seqNo_ < nSeq_ * 2; seqNo_ ++)
		topDisArr[seqNo_] = -1;

	ndQueue_ += startSeqNo;
	topDisArr[startSeqNo] = 0;

	int inNdNo_ = exNdArr_[startSeqNo].getInNd ();
	ndQueue_ += inNdNo_;
	topDisArr[inNdNo_] = 1;

	int curPos_ = 1;
	while (curPos_ < ndQueue_.getSize ()) {
		int ndNo_ = ndQueue_[curPos_];
		curPos_ ++;
		if (isInNd (ndNo_) == 1) {
			Vec<int> neiNdNoLs_;
			neiNdNoLs_ = inNdArr_[ndNo_].getNeiNd ();
			for (int count_ = 0; count_ < neiNdNoLs_.getSize (); count_ ++) {
				int neiNdNo_ = neiNdNoLs_[count_];
				if (topDisArr[neiNdNo_] == -1) {
					topDisArr[neiNdNo_] = topDisArr[ndNo_] + 1;
					ndQueue_ += neiNdNo_;
				} //end if
			}
		} //end of if
	} //end of while
} //end of function


//**************************************************************************************************
//create distance between startSeqNo and other sequences according to the tree
void UrTree::cmpTreeDis (int startSeqNo, Vec<double> &treeDisArr) {
	int nSeq_ = alignment.getNSeq ();

	Vec<int> ndQueue_ (nSeq_ * 2, 0);
	treeDisArr.set (nSeq_ *  2, nSeq_ * 2);

	int seqNo_;
	for (seqNo_ = 0; seqNo_ < nSeq_ * 2; seqNo_ ++)
		treeDisArr[seqNo_] = -1.0;

	ndQueue_ += startSeqNo;
	treeDisArr[startSeqNo] = 0.0;

	int inNdNo_ = exNdArr_[startSeqNo].getInNd ();
	int brNo_ = exNdArr_[startSeqNo].getBr ();
	double brLen_ = brArr_[brNo_].getLen ();

	ndQueue_ += inNdNo_;
	treeDisArr[inNdNo_] = brLen_;

	int curPos_ = 1;
	while (curPos_ < ndQueue_.getSize ()) {
		int ndNo_ = ndQueue_[curPos_];
		curPos_ ++;
		if (isInNd (ndNo_) == 1) {
			Vec<int> neiNdNoLs_;
			neiNdNoLs_ = inNdArr_[ndNo_].getNeiNd ();
			for (int count_ = 0; count_ < neiNdNoLs_.getSize (); count_ ++) {
				int neiNdNo_ = neiNdNoLs_[count_];
				brNo_ = inNdArr_[ndNo_].getBr (count_);
				brLen_ = brArr_[brNo_].getLen ();
				if (treeDisArr[neiNdNo_] < -ZERO) {
					treeDisArr[neiNdNo_] = treeDisArr[ndNo_] + brLen_;
					ndQueue_ += neiNdNo_;
				}
			}
		} //end of if
	} //end of while
}

//**************************************************************************************************
//create topological distance matrix for this tree topology
void UrTree::createTopDis (Mat<int> &topDisMat) {
	int nSeq_ = alignment.getNSeq ();
	topDisMat.setLimit (nSeq_, nSeq_);

	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++) {
		Vec<int> topDisArr_;
		cmpTopDis (seqNo_, topDisArr_);
		for (int seqNo2_ = 0; seqNo2_ < nSeq_; seqNo2_ ++)
			topDisMat[seqNo_][seqNo2_] = topDisArr_[seqNo2_];
	}
}

//**************************************************************************************************
//create distance between sequences according to the tree, not genetic distance
void UrTree::createTreeDis (Mat<double> &treeDisMat) {
	int nSeq_ = alignment.getNSeq ();
	treeDisMat.setLimit (nSeq_, nSeq_);

	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++) {
		Vec<double> treeDisArr_;
		cmpTreeDis (seqNo_, treeDisArr_);
		for (int seqNo2_ = 0; seqNo2_ < nSeq_; seqNo2_ ++)
			treeDisMat[seqNo_][seqNo2_] = treeDisArr_[seqNo2_];
	}
}

//**************************************************************************************************
//create nSeq_ / 2 independent paths for specific-site substitution rate
void UrTree::createApproximateIndSeqPair (Mat<int> &topDisMat, Vec<SeqPair> &indSeqPairArr) {
	int nSeq_ = alignment.getNSeq ();
	Vec<int> markArr_ (nSeq_, nSeq_);
	int seqNo_;
	for (seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		markArr_[seqNo_] = 0;

	int nSeqPair_ = nSeq_ / 2;
	indSeqPairArr.set (nSeqPair_, nSeqPair_);

	for (int seqPairNo_ = 0; seqPairNo_ < nSeqPair_; seqPairNo_ ++) {
		int minTopDis_ = INFINITIVE;
		int minSeqNo_ = -1;
		int minSeqNo2_ = -1;

		for (seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
			if (markArr_[seqNo_] == 0)
				for (int seqNo2_ = seqNo_ + 1; seqNo2_ < nSeq_; seqNo2_ ++)
					if (markArr_[seqNo2_] == 0 && topDisMat[seqNo_][seqNo2_] < minTopDis_) {
						minSeqNo_ = seqNo_;
						minSeqNo2_ = seqNo2_;
						minTopDis_ = topDisMat[minSeqNo_][minSeqNo2_];
					}

		indSeqPairArr[seqPairNo_].seqXNo_ = minSeqNo_;
		indSeqPairArr[seqPairNo_].seqYNo_ = minSeqNo2_;
		indSeqPairArr[seqPairNo_].topDis_ = minTopDis_;
		markArr_[minSeqNo_] = 1;
		markArr_[minSeqNo2_] = 1;
	} //end of for seqPairNo_
}



//**************************************************************************************************
//create nSeq_ / 2 independent paths for specific-site substitution rate
void UrTree::swap4PointCondition (Mat<int> &topDisMat, Vec<SeqPair> &indSeqPairArr) {
	int nSeqPair_ = indSeqPairArr.getSize ();
	int continue_ = 0;
	do {
		continue_ = 0;
		for (int seqPairNo0_ = 0; seqPairNo0_ < nSeqPair_; seqPairNo0_ ++)
			for (int seqPairNo1_ = seqPairNo0_ + 1; seqPairNo1_ < nSeqPair_; seqPairNo1_ ++) {
				int seqNo0_ = indSeqPairArr[seqPairNo0_].seqXNo_;
				int seqNo1_ = indSeqPairArr[seqPairNo0_].seqYNo_;

				int seqNo2_ = indSeqPairArr[seqPairNo1_].seqXNo_;
				int seqNo3_ = indSeqPairArr[seqPairNo1_].seqYNo_;

				int topDis0123_ = topDisMat[seqNo0_][seqNo1_] + topDisMat[seqNo2_][seqNo3_];
				int topDis0213_ = topDisMat[seqNo0_][seqNo2_] + topDisMat[seqNo1_][seqNo3_];
				int topDis0312_ = topDisMat[seqNo0_][seqNo3_] + topDisMat[seqNo1_][seqNo2_];

				int minQuartetTopDis_ = Utl::getMin (topDis0123_, topDis0213_, topDis0312_);
				if (topDis0213_ == minQuartetTopDis_) {
					indSeqPairArr[seqPairNo0_].seqXNo_ = seqNo0_;
					indSeqPairArr[seqPairNo0_].seqYNo_ = seqNo2_;
					indSeqPairArr[seqPairNo1_].seqXNo_ = seqNo1_;
					indSeqPairArr[seqPairNo1_].seqYNo_ = seqNo3_;
					continue_ = 1;
				}
				if (topDis0312_ == minQuartetTopDis_) {
					indSeqPairArr[seqPairNo0_].seqXNo_ = seqNo0_;
					indSeqPairArr[seqPairNo0_].seqYNo_ = seqNo3_;
					indSeqPairArr[seqPairNo1_].seqXNo_ = seqNo1_;
					indSeqPairArr[seqPairNo1_].seqYNo_ = seqNo2_;
					continue_ = 1;
				}
			} //end of for seqPairNo1_
	} while (continue_ == 1);
}

//**************************************************************************************************
//create nSeq_ / 2 independent paths for specific-site substitution rate
void UrTree::createIndSeqPair (Vec<SeqPair> &indSeqPairArr) {
	Mat<int> topDisMat_;
	createTopDis (topDisMat_);
	createApproximateIndSeqPair (topDisMat_, indSeqPairArr);

	swap4PointCondition (topDisMat_, indSeqPairArr);

	Mat<double> treeDisMat_;
	createTreeDis (treeDisMat_);

	int nSeqPair_ = indSeqPairArr.getSize ();
	for (int seqPairNo_ = 0; seqPairNo_ < nSeqPair_; seqPairNo_ ++) {
		int seqNoX_ = indSeqPairArr[seqPairNo_].seqXNo_;
		int seqNoY_ = indSeqPairArr[seqPairNo_].seqYNo_;
		double treeDis_ = treeDisMat_[seqNoX_][seqNoY_];
		//     double genDis_ = alignment.getGenDis (seqNoX_, seqNoY_);
		//     std::cout << genDis_ << " / " << treeDis_ << endl;
		indSeqPairArr[seqPairNo_].genDis_ = treeDis_;
	}
	if (isMasterProc())
		cout << nSeqPair_ << " sequence pairs created" << endl;
}

/**
	create All sequence pairs, AP method in Meyer and von Haeseler 2003
*/
void UrTree::createAllSeqPair(Vec<SeqPair> &indSeqPairArr) {

	int nSeq_ = alignment.getNSeq ();
	int nSeqPair_ = nSeq_*(nSeq_-1) / 2;
	if (isMasterProc())
		cout << nSeqPair_ << " sequence pairs created" << endl;
	indSeqPairArr.set (nSeqPair_, nSeqPair_);
	int count = 0;
	for (int seq1 = 0; seq1 < nSeq_-1; seq1++)
		for (int seq2 = seq1+1; seq2 < nSeq_; seq2++, count++) {
			indSeqPairArr[count].seqXNo_ = seq1;
			indSeqPairArr[count].seqYNo_ = seq2;
		}

	Mat<double> treeDisMat_;
	createTreeDis (treeDisMat_);

	//int nSeqPair_ = indSeqPairArr.getSize ();
	for (int seqPairNo_ = 0; seqPairNo_ < nSeqPair_; seqPairNo_ ++) {
		int seqNoX_ = indSeqPairArr[seqPairNo_].seqXNo_;
		int seqNoY_ = indSeqPairArr[seqPairNo_].seqYNo_;
		double treeDis_ = treeDisMat_[seqNoX_][seqNoY_];
		//     double genDis_ = alignment.getGenDis (seqNoX_, seqNoY_);
		//     std::cout << genDis_ << " / " << treeDis_ << endl;
		indSeqPairArr[seqPairNo_].genDis_ = treeDis_;
	}

}


//**************************************************************************************************
//-----------------------------------------------------------
//clean all the contents of this tree
void UrTree::clean () {
	exNdArr_.clean ();
	inNdArr_.clean ();
	brArr_.clean ();
	isRootedTree_ = 0;
	isOpted_ = 0;
}



int UrTree::isEqual (ClusterArr &bestClusterArr) {
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (brArr_[brNo_].getId () != -1 && brArr_[brNo_].isIn () == 1) {

			Cluster cluster_;
			createCluster (brArr_[brNo_], cluster_);

			int clusterNo_ = bestClusterArr.findCluster (cluster_);

			if (clusterNo_ == -1)
				return 0;
		}
	return 1;
}

int UrTree::cmpDis (ClusterArr &ca2) {
	int dis_ = 0;
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (brArr_[brNo_].getId () != -1 && brArr_[brNo_].isIn () == 1) {
			Cluster cluster_;
			createCluster (brArr_[brNo_], cluster_);
			int clusterNo_ = ca2.findCluster (cluster_);

			if (clusterNo_ == -1)
				dis_ ++;
		}
	return dis_;
}




//-----------------------------------------------------------
//cmp the topological distance between two trees
int UrTree::cmpDis (UrTree &tree2) {
	ClusterArr ca1_;
	createCluster (ca1_);
	int dis_ = tree2.cmpDis (ca1_);
	return dis_ * 2;
}






void UrTree::writeNewickForm (const char *fileName, int isApp, double progTime) {
	createRootedTree (outGrpNdNo_);
	Vec<char> nwtree_;
	std::ofstream out_;
	if (isApp == 1)
		out_.open (fileName, ios::app);
	else
		out_.open (fileName);

	createNeuwickForm (nwtree_);
	double curLogLi_ = getLogLi ();
	out_.precision (10);
	out_ << progTime << endl;
	out_ << curLogLi_ << endl;
	for (int count_ = 0; count_ < nwtree_.getSize (); count_ ++)
		out_ << nwtree_[count_];
	out_ << endl;
	out_.close ();
}

void UrTree::writeNewickForm (std::ostream &out) {
	createRootedTree (outGrpNdNo_);
	//  divideBr (COF_BR);
	Vec<char> nwtree_;

	createNeuwickForm (nwtree_);
	for (int count_ = 0; count_ < nwtree_.getSize (); count_ ++)
		out << nwtree_[count_];
	out << endl;
}

void UrTree::writeTop (const char *fileName, int isApp) {
	createRootedTree (outGrpNdNo_);
	Vec<char> nwtree_;
	std::ofstream out_;
	if (isApp == 1)
		out_.open (fileName, ios::app);
	else {
		out_.open (fileName);
	}
	//  divideBr (COF_BR);

	createNeuwickForm (nwtree_);
	for (int count_ = 0; count_ < nwtree_.getSize (); count_ ++)
		out_ << nwtree_[count_];
	out_ << endl;
	out_.close ();
	//  divideBr (1.0 / COF_BR);
}


//-----------------------------------------------------------
//put the first br into this tree
void UrTree::setFirstBr (Br<double> &br) {
	int smallNdNo_, greatNdNo_;
	br.getNd (smallNdNo_, greatNdNo_);

	//the small nd is an ex nd
	ExNd smallExNd_;
	smallExNd_.setId (smallNdNo_);
	exNdArr_ += smallExNd_;

	//the great nd in this case is also an ex nd
	ExNd greatExNd_;
	greatExNd_.setId (greatNdNo_);

	exNdArr_ += greatExNd_;

	//add the br into the br list of the tree
	brArr_ += br;

}


//-----------------------------------------------------------
//insert a nd into an existing br of the unroot tree
void UrTree::insertNdBr (int ndNo, int brNo) {
	// the special case, when the tree has only one br
	if (brArr_.getCurNBr () == 1)
		insertNdFirstBr (ndNo, brNo);
	else
		if (brArr_[brNo].isEx () )
			insertNdExBr (ndNo, brNo);

		else

			insertNdInBr (ndNo, brNo);
}


//-----------------------------------------------------------
/*
      internal Node
      *       *
   *            *
neiExNd          exNdNo
 
 
 
 
*/




//return the neighbor external node of input external node
int UrTree::getNeiExNd (int exNdNo) {
	int inNdNo_ = exNdArr_[exNdNo].getInNd ();
	InNd *inNd_;
	inNd_ = &inNdArr_[inNdNo_];
	int remNeiNdNo1_, remNeiNdNo2_;
	inNd_->get2RemNeiNd (exNdNo, remNeiNdNo1_, remNeiNdNo2_);
	if (remNeiNdNo1_ < maxNExNd_)
		return remNeiNdNo1_;

	if (remNeiNdNo2_ < maxNExNd_)
		return remNeiNdNo2_;


	return -1;
}

//-----------------------------------------------------------

/*
 
del one external node from this tree
    exNdNo
           *
            *
             inNd * * * neiNdNo2
            *
          *
  neiNdNo1
*/

void UrTree::delExNd (const int exNdNo) {
	//del the internal node which is the neighbor of this external node
	int inNdNo_ = exNdArr_[exNdNo].getInNd ();
	InNd inNd_;
	inNd_ = inNdArr_[inNdNo_];
	inNdArr_ -= inNdNo_;


	//del this external node
	exNdArr_ -= exNdNo;

	//determin two remain nei ndNos, and two remain branches of the internal node
	Vec<int> *neiNdNoLs_;
	neiNdNoLs_  = &inNd_.getNeiNd ();

	Vec<int> *brNoLs_;
	brNoLs_ = &inNd_.getBr ();
	int neiNdNo1_ = -1;
	int neiNdNo2_ = -1;
	int brNo1_, brNo2_;
	inNd_.get2RemNeiNdBr (exNdNo, neiNdNo1_, brNo1_, neiNdNo2_, brNo2_);

	//del all branches contacting to this internal node

	int count_;
	for (count_  = 0; count_ < brNoLs_->getSize (); count_ ++)
		brArr_ -= (*brNoLs_)[count_];

	//create and add new branch into this tree, which contacts two remain nei nodes, neiNdNo1, neiNdNo2
	Br<double> newBr_;
	newBr_.set (brArr_.getNewBr (), neiNdNo1_, neiNdNo2_);
	brArr_ += newBr_;

	//change information of two remain nei nodes of the internal node, neiNdNo1_, neiNdNo2_
	if (inNdArr_.isInNd (neiNdNo1_)) { //if neiNdNo1_ is an internal node

		inNdArr_[neiNdNo1_].changeNeiNd (inNdNo_, neiNdNo2_);
		inNdArr_[neiNdNo1_].changeBr (brNo1_, newBr_.getId ());
	} else {

		exNdArr_[neiNdNo1_].changeBr (newBr_.getId ());
		exNdArr_[neiNdNo1_].changeInNd (neiNdNo2_);

	}

	if (inNdArr_.isInNd (neiNdNo2_)) {
		inNdArr_[neiNdNo2_].changeNeiNd (inNdNo_, neiNdNo1_);
		inNdArr_[neiNdNo2_].changeBr (brNo2_, newBr_.getId ());
	} else {
		exNdArr_[neiNdNo2_].changeBr (newBr_.getId ());
		exNdArr_[neiNdNo2_].changeInNd (neiNdNo1_);
	}


}


//-----------------------------------------------------------
//forward all information of this tree into the opimial unrooted tree
void UrTree::createOptUrTree () {
	outGrpNdNo_ = findOutGrpNd ();

	reCreateRootedTree (outGrpNdNo_);
	opt_urtree.setOutGrpNd (outGrpNdNo_);

	//the artificial internal root of this tree
	opt_urtree.setExNd (exNdArr_);
	opt_urtree.setInNd (inNdArr_);

	opt_urtree.setBr (brArr_);
	opt_urtree.init ();
}

void UrTree::createCluster (Br<double> &br, Cluster &cluster) {
	int nCurExNd_ = exNdArr_.getCurNNd ();
	Vec<int> exNdNoLs_;
	findNeaExNd (br, exNdNoLs_, nCurExNd_);
	double brLen_ = br.getLen ();
	cluster.add (exNdNoLs_, brLen_);
}

void UrTree::createCluster (ClusterArr &clusterArr) {
	clusterArr.nDifCluster_ = 0;
	//  int nCurExNd_ = exNdArr_.getCurNNd ();
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (brArr_[brNo_].getId () != -1 && brArr_[brNo_].isIn () == 1) {
			Cluster cluster;
			createCluster (brArr_[brNo_], cluster);

			clusterArr += cluster;
		}

}

double UrTree::getExBrLen (int exNdNo) {
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (brArr_[brNo_].getSmaNd() == exNdNo) {
			double brLen_ = brArr_[brNo_].getLen ();
			return brLen_;
		}
	return -INFINITIVE;

}


void UrTree::cmpBrLen (UrTree &bestTree, ClusterArr &bestClusterArr) {
	//  bestTree.draw (NON_BR_LEN, 0, std::cout);

	//draw (NON_BR_LEN, 0, std::cout);
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++ ) {
		if (brArr_[brNo_].getId () != -1)

			if (brArr_[brNo_].isEx () == 1) {
				int exNdNo_ = brArr_[brNo_].getSmaNd ();
				double brLen_ = bestTree.getExBrLen (exNdNo_);
				brArr_[brNo_].setLen (brLen_);

				//         std::cout << brLen_ << " ";

			} else {


				Cluster cluster_;
				createCluster (brArr_[brNo_], cluster_);

				int cNo_ = bestClusterArr.findCluster (cluster_);

				double brLen_;
				if (cNo_ == -1)
					brLen_ = -1.0;
				else
					brLen_ = bestClusterArr[cNo_].brLen_;



				brArr_[brNo_].setLen (brLen_);

				//       std::cout << brLen_ << " ";
				//       std::cout << brArr_[brNo_].getLen () << " ";
			}
	}
	//  std::cout << endl;

}

//-----------------------------------------------------------
//assign all diferrent information of this OptUrTree into this tree
void UrTree::getOptUrTree () {
	opt_urtree.getBr (brArr_);

	opt_urtree.getInNd (inNdArr_);
	opt_urtree.getExNd (exNdArr_);
}

//return the log likelihood of this tree
double UrTree::getLogLi () {
	return logLi_;
}

double UrTree::doConOpt (int NNI, bool &topo_changed, int nIt) {
	createOptUrTree ();
	topo_changed = false;
	if (NNI == 1) {
#ifdef PARALLEL
		opt_urtree.turnOnParallelPhase(!PARALLEL_LOGLI);
#endif // PARALLEL

		topo_changed = createLocalTree ();
	} else {
#ifdef PARALLEL
		opt_urtree.turnOnParallelPhase(!PARALLEL_LOGLI);
#endif // PARALLEL

		opt_urtree.optBranches (nIt);
	}

#ifdef PARALLEL
	opt_urtree.turnOffParallelPhase();
#endif // PARALLEL

	getTree ();
	logLi_ = opt_urtree.getLogLi ();
	return logLi_;
}

void UrTree::getTree() {
	opt_urtree.getBr (brArr_);
	opt_urtree.getExNd (exNdArr_);
	opt_urtree.getInNd (inNdArr_);
}





//-----------------------------------------------------------

//this function optimize this tree and return the likelihood of this tree


double UrTree::cmpLogLi (OPT_STATUS status, int NNI, int nIt ) {
	bool topo_changed = false;
	if (isOpted_ == 1)
		return logLi_;
	int brNo_;
	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		brArr_[brNo_].setLen (-INFINITIVE);

	createOptUrTree ();

	// std::cout << (clock () - startTime_) / CLOCKS_PER_SEC << endl;


	if (status == COMPLETE) {

		if (NNI == 0)


			opt_urtree.optBranches (nIt);
		else
			topo_changed = createLocalTree ();
		isOpted_ = 1;
	}

	getTree ();



	logLi_ = opt_urtree.getLogLi ();
	return logLi_;
}

extern ofstream nni_file;
extern ofstream nni_tree;
extern bool nni_lh_record;

bool UrTree::createLocalTree () {
	bool topo_changed;
	opt_urtree.optTopology (topo_changed);
	opt_urtree.optBranches ();
	/*
	if (nni_lh_record) {
		nni_file << opt_urtree.cmpLogLi() << endl;
		nni_file.flush();

	}*/
	return topo_changed;
}

bool UrTree::createLocalTree (ClusterArr &bestClusterArr) {
	bool topo_changed;
	opt_urtree.optTopology (topo_changed);
#ifdef PARALLEL
	if (stopSignalCome())
		return topo_changed;
#endif //PARALLEL
	if (alignment.getNSeq () <= SMALL_DATA_SET) {
		getTree ();
		/*
		    if (isEqual (bestClusterArr) == 1)
		      std::cout << "identical " << endl;
		    else
		      opt_urtree.opt ();
		*/
		if (isEqual (bestClusterArr) == 0)
			opt_urtree.optBranches ();

	} else
		opt_urtree.optBranches ();
	/*
	if (nni_lh_record) {
		nni_file << opt_urtree.cmpLogLi() << endl;
		nni_file.flush();
	}*/
	//nni_file << global_bestLogLi_  << endl;
	return topo_changed;
}

//-----------------------------------------------------------
//this function optimize this tree and return the likelihood of this tree
double UrTree::cmpLogLi (OPT_STATUS status, UrTree &bestTree, ClusterArr &bestClusterArr,int NNI, int nIt) {
	if (isOpted_ == 1)
		return logLi_;

	//  double start_ = clock ();
	cmpBrLen (bestTree, bestClusterArr);
	//  OutStream::writeBrLen (liBrArr_, std::cout);
	//std::cout << "time of init br len" << (clock () - start_) / CLOCKS_PER_SEC << "  / ";
	createOptUrTree ();

#ifdef PARALLEL
	if (stopSignalCome())
		return 0;
#endif

	if (status == COMPLETE) {
		if (NNI == 0)

			opt_urtree.optBranches (nIt);
		else
			createLocalTree (bestClusterArr);
		isOpted_ = 1;
	}

	getTree ();

	logLi_ = opt_urtree.getLogLi ();
	return logLi_;
}

//-----------------------------------------------------------
//this function reOptimize this tree and return the likelihood of this tree
double UrTree::reCmpLogLi (OPT_STATUS status, int NNI, int nIt) {

	isOpted_ = 0;
	logLi_ = cmpLogLi (status, NNI, nIt);
	return logLi_;
}

//-----------------------------------------------------------
//this function reOptimize this tree and return the likelihood of this tree


double UrTree::reCmpLogLi (OPT_STATUS status, UrTree &bestTree, ClusterArr &bestClusterArr, int NNI, int nIt ) {
	isOpted_ = 0;
	logLi_ = cmpLogLi (status, bestTree, bestClusterArr, NNI, nIt);
	return logLi_;
}

//-----------------------------------------------------------
//return the inNdArr_
InNdArr &UrTree::getInNd () {
	return inNdArr_;
}

//-----------------------------------------------------------
//return the exNdArr_
ExNdArr &UrTree::getExNd () {
	return exNdArr_;
}

//-----------------------------------------------------------
//return the brArr_
BrArr<double> &UrTree::getBr () {

	return brArr_;

}

//-----------------------------------------------------------
//return the ourGrpNdNo of this unrooted tree
int UrTree::getOutGrpNd () {

	return outGrpNdNo_;
}

//-----------------------------------------------------------

//return isRootedTree_
int UrTree::isRootedTree () {
	return isRootedTree_;
}

//-----------------------------------------------------------
//returen isOpted_
int UrTree::isOpted () {
	return isOpted_;
}


//-----------------------------------------------------------
//returen artificial internal root no
int UrTree::getArtInRoot () {
	return artInRootNo_;
}

// clone from another tree
void UrTree::clone (UrTree *urTree) {
	this->inNdArr_ = urTree->getInNd ();
	this->exNdArr_ = urTree->getExNd ();
	this->brArr_ =   urTree->getBr ();

	this->isRootedTree_ = urTree->isRootedTree ();

	this->outGrpNdNo_ = urTree->getOutGrpNd ();
	this->artInRootNo_ = urTree->getArtInRoot ();

	this->isOpted_ = urTree->isOpted ();
	this->logLi_ = urTree->getLogLi ();

	this->maxNExNd_ = exNdArr_.getMaxNNd ();
	this->maxNInNd_ = inNdArr_. getMaxNNd ();
	this->maxNBr_ = brArr_.getMaxNBr ();

	this->maxNNd_ = maxNExNd_ + maxNInNd_;
}

//-----------------------------------------------------------
// clone from another tree
void UrTree::clone (UrTree &urTree) {
	this->inNdArr_ = urTree.getInNd ();
	this->exNdArr_ = urTree.getExNd ();
	this->brArr_ =   urTree.getBr ();

	this->isRootedTree_ = urTree.isRootedTree ();

	this->outGrpNdNo_ = urTree.getOutGrpNd ();
	this->artInRootNo_ = urTree.getArtInRoot ();

	this->isOpted_ = urTree.isOpted ();
	this->logLi_ = urTree.getLogLi ();

	this->maxNExNd_ = exNdArr_.getMaxNNd ();
	this->maxNInNd_ = inNdArr_. getMaxNNd ();
	this->maxNBr_ = brArr_.getMaxNBr ();

	this->maxNNd_ = maxNExNd_ + maxNInNd_;
}

//-----------------------------------------------------------
//overload operator =
void UrTree::operator = (UrTree &urTree) {
	clone(urTree);
}

//-----------------------------------------------------------
//check whether or not the input urTree is equal to this tree
int UrTree::operator == (UrTree &tree2) {

	int dis_ = cmpDis (tree2);
	if (dis_ == 0)
		return 1;
	else
		return 0;
}


//-----------------------------------------------------------
//write all information about this tree into file
void UrTree::write (std::ofstream &out) {
	out.precision (10);
	out << "the log likelihood : " << logLi_ << endl;

	OutStream::write (&exNdArr_, &inNdArr_, &brArr_, out);
}

//-----------------------------------------------------------

//write all information about this tree into screen
void UrTree::write () {
	OutStream::write (&exNdArr_, &inNdArr_, &brArr_, std::cout);
}


//--------------------------------------------------------------------
//find an existed external node for being the out group node of this tree
int UrTree::findOutGrpNd () {
	return exNdArr_.findOutGrpNd ();
}
//--------------------------------------------------------------------

//reCreate the tree to be the rooted tree
void UrTree::reCreateRootedTree (int outGrpNdNo) {
	outGrpNdNo_ = outGrpNdNo;

	isRootedTree_ = 0;

	createRootedTree (outGrpNdNo_);

}

void UrTree::createNdOrder (int parNdNo, int inNdNo) {
	inNdArr_[inNdNo].orderNeiNd (parNdNo);
	int chiNdNo0_, chiNdNo1_;
	inNdArr_[inNdNo].get2RemNeiNd (parNdNo, chiNdNo0_, chiNdNo1_);
	if (isInNd (chiNdNo0_) == 1)
		createNdOrder (inNdNo, chiNdNo0_);
	if (isInNd (chiNdNo1_) == 1)
		createNdOrder (inNdNo, chiNdNo1_);

}

void UrTree::createBrDir (Br<double> &parBr) {
	int tailNdNo_ = parBr.getTailNd ();
	int parBrNo_ = parBr.getId ();
	if (parBr.isStG () == 1)
		brArr_[parBrNo_].setDir (StG);
	else
		brArr_[parBrNo_].setDir (GtS);

	if (isInNd (tailNdNo_) == 1) {
		Br<double> chiBr0_;

		Br<double> chiBr1_;
		getChiBr (parBr, chiBr0_, chiBr1_);
		createBrDir (chiBr0_);
		createBrDir (chiBr1_);
	}
}


//--------------------------------------------------------------------
//this will turn this unrooted tree into a rooted tree
void UrTree::createRootedTree (int outGrpNdNo) {
	if (outGrpNdNo_ != outGrpNdNo) {
		outGrpNdNo_ = outGrpNdNo;
		isRootedTree_ = 0;

	} else
		if (isRootedTree_ == 1)
			return;

	if (exNdArr_[outGrpNdNo_].getId () == -1)
		Utl::announceError (OUT_GRP_ND_ERROR);

	artInRootNo_ = exNdArr_[outGrpNdNo_].getInNd ();

	createNdOrder (outGrpNdNo, artInRootNo_);
	int parBrNo_ = inNdArr_[artInRootNo_].findComBr (outGrpNdNo_);

	brArr_[parBrNo_].setDir (StG);

	createBrDir (brArr_[parBrNo_]);
}




//--------------------------------------------------------------------------
//release all memory of this class
void UrTree::release () {
	brArr_.release ();
	inNdArr_.release ();
	exNdArr_.release ();
}

//==================================================================

void UrTree::createRan (int nSeq, double minBrLen, double maxBrLen) {
	Br<double> fb_;

	fb_.set (0, 0, 1);

	setFirstBr (fb_);
	for (int seqNo_ = 2; seqNo_ < nSeq; seqNo_ ++) {
		int brNo_ = rand () % (seqNo_ * 2 - 3);
		insertNdBr (seqNo_, brNo_);
	}
	for (int brNo_ = 0; brNo_ < nSeq * 2 - 3; brNo_ ++) {
		double len_ = maxBrLen - minBrLen;
		double percent_ = rand () % 1000;
		percent_ = percent_ / 1000;
		double brLen_ = minBrLen + len_ * percent_;

		brArr_[brNo_].setLen (brLen_);
	}
}

void UrTree::createNeuwickForm (int ndNo, Vec<char> &tree, bool seq_name) {
	tree.setLimit (MAX_NEWICK_TREE_LEN);
	if (isExNd (ndNo) == 1) {
		if (seq_name)
			alignment.getName (ndNo, tree);
		else {
			tree.convert(ndNo+1);
		}
	} else {
		char colon_ = ':';
		char openBracket_ = '(';
		char comma_ = ',';
		char closeBracket_ = ')';


		Vec<char> rTree_;
		Vec<char> lTree_;
		int lNdNo_ = inNdArr_[ndNo].getNeiNd (1);
		int rNdNo_ = inNdArr_[ndNo].getNeiNd (2);

		createNeuwickForm (lNdNo_, lTree_, seq_name);
		createNeuwickForm (rNdNo_, rTree_, seq_name);

		tree += openBracket_;
		tree += lTree_;

		int lBrNo_ = inNdArr_[ndNo].getBr (1);
		double lBrLen_ = brArr_[lBrNo_].getLen ();
		Vec<char> cLBrLen_;
		cLBrLen_.convert (lBrLen_, 6);
		tree += colon_;
		tree += cLBrLen_;
		tree += comma_;

		tree += rTree_;
		int rBrNo_ = inNdArr_[ndNo].getBr (2);
		double rBrLen_ = brArr_[rBrNo_].getLen ();

		Vec<char> rLBrLen_;
		rLBrLen_.convert (rBrLen_, 6);
		tree += colon_;
		tree += rLBrLen_;
		tree += closeBracket_;
	}
}

void UrTree::createNeuwickForm (Vec<char> &neuwickForm, bool seq_name) {
	createRootedTree (outGrpNdNo_);

	char colon_ = ':';
	char openBracket_ = '(';
	char comma_ = ',';
	//  char closeBracket_ = ')';
	char semiColon_ = ';';

	Vec<char> tree_;
	createNeuwickForm (artInRootNo_, tree_, seq_name);

	neuwickForm.set (maxNExNd_ * 50, 0);

	if (seq_name)
		neuwickForm = alignment.getName (outGrpNdNo_);
	else 
		neuwickForm.convert(outGrpNdNo_+1);
	neuwickForm.insert (0, openBracket_);

	int brNo_ = inNdArr_[artInRootNo_].getBr (0);

	double brLen_ = brArr_[brNo_].getLen ();
	Vec<char> cBrLen_;
	cBrLen_.convert (brLen_, 6);


	neuwickForm += colon_;



	neuwickForm += cBrLen_;
	neuwickForm += comma_;


	tree_.del (0);
	neuwickForm += tree_;
	neuwickForm += semiColon_;
}

void UrTree::writeNewickForm (const char *fileName) {
	std::ofstream fo;
	fo.open (fileName);
	Vec<char> tree_;
	//  divideBr (COF_BR);

	createNeuwickForm (tree_);
	//  divideBr (1.0 / COF_BR);

	for (int count_ = 0; count_ < tree_.getSize (); count_ ++)
		fo << tree_[count_];

	fo.close ();
}

void UrTree::draw (DRAW_TREE_TYPE drawTreeType, int outGrpNdNo, const char *fileName) {
	Mat<char> image_;
	createImage (drawTreeType, outGrpNdNo, image_);
	std::ofstream out_;
	out_.open (fileName);
	OutStream::write (image_, out_);
	out_.close ();
}

void UrTree::draw (DRAW_TREE_TYPE drawTreeType, int outGrpNdNo) {
	Mat<char> image_;
	createImage (drawTreeType, outGrpNdNo, image_);
	OutStream::write (image_, std::cout);
}

void UrTree::createImage (DRAW_TREE_TYPE drawTreeType, int outGrpNdNo, Mat<char> &image) {

	isRootedTree_ = 0;
	createRootedTree (outGrpNdNo);

	Vec<int> abstractBrLenArr_ (maxNBr_, maxNBr_);
	if (drawTreeType == NON_BR_LEN) //it means that all branches length are the same
		for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
			abstractBrLenArr_[brNo_] = MIN_ABSTRACT_BR_LEN;

	else
		createAbstractBrLen (abstractBrLenArr_);


	int artInRootNo_ = exNdArr_[outGrpNdNo_].getInNd ();
	Vec<int> ndRowOrderArr_ (maxNNd_, maxNNd_);
	int firstRowOrder_ = 1;
	createNdRowOrder (artInRootNo_, firstRowOrder_, ndRowOrderArr_);
	ndRowOrderArr_[outGrpNdNo_] = ndRowOrderArr_[artInRootNo_];

	Vec<int> abstractNdHeiArr_ (maxNNd_, maxNNd_);

	//  Vec<char> outGrpName_;
	//  alignment.getName (outGrpNdNo_, outGrpName_);
	//  int outGrpNameLen_ = outGrpName_.getSize ();

	abstractNdHeiArr_[outGrpNdNo_] = 0;
	createAbstractNdHei (artInRootNo_, 0, abstractBrLenArr_, abstractNdHeiArr_);

	int nRow_ = cmpNImageRow (ndRowOrderArr_) + 1;
	int nCol_ = cmpNImageCol (abstractNdHeiArr_) + 1;


	image.setLimit (nRow_, nCol_);

	int row_, col_;

	for (row_ = 0; row_ < nRow_; row_ ++)
		for (col_ = 0; col_ < nCol_; col_ ++)
			image[row_][col_] = ' ';


	createImage (abstractBrLenArr_, ndRowOrderArr_, abstractNdHeiArr_, image);
}


//-----------------------------------------------------------
//destruction function

UrTree::~UrTree () {
	//  std::cout << "this is the destructor of UrTree class " << endl;
	delete bonusBrArr_;
}




/******************************************************************************************************************************************************
******************************************************************************************************************************************************
******************************************************************************************************************************************************/
//-----------------------------------------------------------
/*
insert the third nd into the first br
        the third nd
 
             |
             | 2
             |
          0       1
smallnd *---inNd-----* greatNd
*/

void UrTree::insertNdFirstBr (int ndNo, int brNo) {
	int inNdNo_ = maxNExNd_;
	//get the first br and put it into the br_ variable
	Br<double> br_;
	br_ = brArr_[0];

	//get the small nd and the great nd of the first br
	int smallNdNo_, greatNdNo_;

	br_.getNd (smallNdNo_, greatNdNo_);

	//change information of the ex nd containing the small nd


	ExNd smallExNd_;
	smallExNd_.set (smallNdNo_, inNdNo_, 0);

	exNdArr_.changeItem (smallExNd_);

	//change information of the ex nd containing the great nd
	ExNd greatExNd_;
	greatExNd_.set(greatNdNo_, inNdNo_, 1);


	exNdArr_.changeItem (greatExNd_);

	//add a new ex nd containing the ndNo
	ExNd newExNd_;
	newExNd_.set(ndNo, inNdNo_, 2);
	exNdArr_ += newExNd_;


	//creat the in nd and add it into the innd list
	InNd newInNd_;
	newInNd_.setId (inNdNo_);
	newInNd_.addNeiNd (smallNdNo_, greatNdNo_, ndNo);

	newInNd_.addBr (0, 1, 2);
	inNdArr_ += newInNd_;

	//del the first br from the brArr
	brArr_ -= brNo;

	//create three new branches
	Br<double> smallBr_;
	smallBr_.set (0, smallNdNo_, inNdNo_, EX);
	brArr_ += smallBr_;

	Br<double> greatBr_;
	greatBr_.set (1, greatNdNo_, inNdNo_, EX);
	brArr_ += greatBr_;

	Br<double> newBr_;
	newBr_.set (2, ndNo, inNdNo_, EX);


	brArr_ += newBr_;
} //end of function insertNdFirstBr


//-----------------------------------------------------------
/*insert a nd into an ex br
smallNd
  *
 
 
   * brNo
    *       newBrNo2_
     **************** newExNd
      *
        *newBrNo1_
 
          *
 
        inNdNo_ (great br)
*/
void UrTree::insertNdExBr (int ndNo, int brNo) {
	int inNdNo_ = inNdArr_.getNewNd ();

	//get all information of brNo and put them into variables
	int smallNdNo_, greatNdNo_;
	Br<double> br_;


	br_ = brArr_[brNo];
	br_.getNd (smallNdNo_, greatNdNo_);

	//change the information of the existing br

	Br<double> smallBr_;

	smallBr_.set (brNo, smallNdNo_, inNdNo_, EX);
	brArr_.change (smallBr_);

	//add two new bres
	Br<double> greatBr_;
	int newBrNo1_ = brArr_.getNewBr ();
	greatBr_.set (newBrNo1_, greatNdNo_, inNdNo_, IN);
	brArr_ += greatBr_;


	Br<double> newBr_;
	int newBrNo2_ = brArr_.getNewBr ();


	newBr_.set (newBrNo2_, ndNo, inNdNo_, EX);
	brArr_ += newBr_;




	//change the ex nd containing the small nd
	ExNd smallExNd_;
	smallExNd_.set (smallNdNo_, inNdNo_, brNo);
	exNdArr_.changeItem (smallExNd_);

	//change the in nd containing the great nd
	InNd *greatInNd_;
	greatInNd_ = &inNdArr_[greatNdNo_];
	greatInNd_->changeNeiNd (smallNdNo_, inNdNo_);
	greatInNd_->changeBr (brNo, newBrNo1_);

	//creat the new in nd
	InNd newInNd_;
	newInNd_.setId (inNdNo_);
	newInNd_.addNeiNd (smallNdNo_, greatNdNo_, ndNo);
	newInNd_.addBr (brNo, newBrNo1_, newBrNo2_);


	inNdArr_ += newInNd_;


	//add new ex nd containing the ndNo
	ExNd newExNd_;

	newExNd_.set (ndNo, inNdNo_, newBrNo2_);
	exNdArr_ += newExNd_;

} //end of adding a new seq into an external branch



//-----------------------------------------------------------
/*
                               ndNo
 
                                 *
                                 *
                                 *newBrNo2_
 
                                 *
 
                          brNo   *      newBrNo1_
(internal node )smallNo---------inNdNo------------greatNo (internal node)
*/
//insert a nd into an internal br
void UrTree::insertNdInBr (int ndNo, int brNo) {
	int inNdNo_ = inNdArr_.getNewNd ();
	int smallNdNo_, greatNdNo_;
	Br<double> br_;
	br_ = brArr_[brNo];


	br_.getNd (smallNdNo_, greatNdNo_);


	//change the information of the existing bres
	Br<double> smallBr_;
	smallBr_.set (brNo, smallNdNo_, inNdNo_, IN);
	brArr_.change (smallBr_);

	//creat the new br containing the great nd
	Br<double> greatBr_;

	int newBrNo1_ = brArr_.getNewBr ();
	greatBr_.set (newBrNo1_, greatNdNo_, inNdNo_, IN);
	brArr_ += greatBr_;


	//creat the new br containing the ndNo
	Br<double> newBr_;
	int newBrNo2_ = brArr_.getNewBr ();
	newBr_.set (newBrNo2_, ndNo, inNdNo_, EX);
	brArr_ += newBr_;

	/*
	                               ndNo
	                                 *
	                                 *
	                                 *newBrNo2_
	                                 *
	                          brNo   *      newBrNo1_
	 
	 
	 
	(internal node )smallNo---------inNdNo------------greatNo (internal node)
	*/
	//change the in nd containing the small nd
	InNd *smallInNd_;
	smallInNd_ = &inNdArr_[smallNdNo_];
	smallInNd_->changeNeiNd (greatNdNo_, inNdNo_);

	//change the in nd containing the great nd
	InNd *greatInNd_;

	greatInNd_ = &inNdArr_[greatNdNo_];
	greatInNd_->changeNeiNd (smallNdNo_, inNdNo_);
	greatInNd_->changeBr (brNo, newBrNo1_);

	//creat a new in nd
	InNd newInNd_;
	newInNd_.setId (inNdNo_);


	newInNd_.addNeiNd (smallNdNo_, greatNdNo_, ndNo);
	newInNd_.addBr (brNo, newBrNo1_, newBrNo2_);
	inNdArr_ += newInNd_;


	//creat and add a new ex nd
	ExNd newExNd_;
	newExNd_.set (ndNo, inNdNo_, newBrNo2_);

	exNdArr_ += newExNd_;
} // end of function add a new node into an internal branch


//-----------------------------------------------------------

//check whether or not this node is an external node
int UrTree::isExNd (const int ndNo) {
	if (ndNo < maxNExNd_ )
		return 1;
	else
		return 0;
}


//-----------------------------------------------------------
//check whether or not this node is an internal node
int UrTree::isInNd (const int ndNo) {

	if (ndNo  >= maxNExNd_ )
		return 1;
	else

		return 0;
}


//==================================================================================
//==================================================================================
//==================================================================================
//==================================================================================

//--------------------------------------------------------------------------
//-----------------------------------------------------------
//create the abstract brach len for this urtree
void UrTree::createAbstractBrLen (Vec<int> &abstractBrLenArr) {
	abstractBrLenArr.set (maxNBr_, maxNBr_);

	//find the branch with the longest length
	double maxBrLen_ = 0.0;
	int brNo_;
	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (brArr_[brNo_].getId () != -1 && brArr_[brNo_].getLen () > maxBrLen_ )
			maxBrLen_ = brArr_[brNo_].getLen ();


	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (brArr_[brNo_].getLen () < (maxBrLen_ / NUM_ABSTRACT_BR_LEN_CLASS) )

			abstractBrLenArr[brNo_] = MIN_ABSTRACT_BR_LEN;
		else
			if (brArr_[brNo_].getLen () < (maxBrLen_ * 2.0 / NUM_ABSTRACT_BR_LEN_CLASS) )

				abstractBrLenArr[brNo_] = MIN_ABSTRACT_BR_LEN + 3;
			else
				abstractBrLenArr[brNo_] = MIN_ABSTRACT_BR_LEN + 6;
}

//-----------------------------------------------------------
//create the row order of for all nodes of this tree
int UrTree::createNdRowOrder (int ndNo, int &rowOrder, Vec<int> &ndRowOrderArr) {
	if (ndNo < maxNExNd_) {//it is an external node
		ndRowOrderArr[ndNo] = rowOrder;
		return rowOrder;
	}

	//the case, it is an internal node
	Vec<int> *neiNdNoLs_;
	neiNdNoLs_ = &inNdArr_[ndNo].getNeiNd ();

	int nChi_ = neiNdNoLs_->getSize () - 1;

	int ndRowOrder_ = -1;

	for (int countChi_ = 1; countChi_ <= nChi_; countChi_ ++) {

		int chiNdNo_ = (*neiNdNoLs_)[countChi_];
		int chiNdRowOrder_ = createNdRowOrder (chiNdNo_, rowOrder, ndRowOrderArr);


		if (countChi_ >= static_cast<int> (nChi_ / 2) && (ndRowOrder_ == -1) ) {


			if (nChi_ % 2 == 1) //the number of childrent is old
				ndRowOrder_ = chiNdRowOrder_;
			else { //the case, the number of chidrent is even
				rowOrder ++;
				ndRowOrder_ = rowOrder;

			}
		} // end of if countChi_


		if (countChi_ < nChi_)
			rowOrder ++;
	} // end of for countChi_

	ndRowOrderArr[ndNo] = ndRowOrder_;
	return ndRowOrder_;
}

//-----------------------------------------------------------
//create the abstract height for all nodes of this tree.

void UrTree::createAbstractNdHei (int ndNo, int abstractNdHei,
                                  Vec<int> &abstractBrLenArr, Vec<int> &abstractNdHeiArr) {
	abstractNdHeiArr[ndNo] = abstractNdHei;
	if (ndNo >= maxNExNd_) {// the case of internal node

		Vec<int> *neiNdNoLs_;
		neiNdNoLs_ = &inNdArr_[ndNo].getNeiNd ();

		Vec<int> *brNoLs_;
		brNoLs_ = &inNdArr_[ndNo].getBr ();

		int nChi_ = neiNdNoLs_->getSize () - 1;

		for (int countChi_ = 1; countChi_ <= nChi_; countChi_ ++) {

			int chiNdNo_ = (*neiNdNoLs_)[countChi_];

			int chiBrNo_ = (*brNoLs_)[countChi_];
			//      std::cout << chiNdNo_ << " " << chiBrNo_ << " " << abstractBrLenArr[chiBrNo_] << endl;
			createAbstractNdHei (chiNdNo_, (abstractNdHei + abstractBrLenArr[chiBrNo_]), abstractBrLenArr, abstractNdHeiArr);

		}
	}// end of if

}//end of function createAbstractNdHei


//------------------------------------------------------------------------------
//it serves for createImage
void UrTree::createRowImage (int row, int startCol, int endCol, char ch, Mat<char> &image) {
	for (int col_ = startCol; col_ <= endCol; col_ ++)
		image[row][col_] = ch;
}

//------------------------------------------------------------------------------
//it serves for createImage
void UrTree::createColImage (int col, int startRow, int endRow, char ch, Mat<char> &image) {

	for (int row_ = startRow; row_ <= endRow; row_ ++)
		image[row_][col] = ch;
}

//------------------------------------------------------------------------------

//it serves for createImage
void UrTree::createRowImage (int row, int startCol, int endCol, Vec<char> &name, Mat<char> &image) {
	int realStartCol_ = startCol + static_cast <int> ( (endCol - startCol + 1 - name.getSize ()) / 2 );
	for (int count_ = 0; count_ < name.getSize (); count_ ++)
		image[row][realStartCol_ + count_] = name[count_];
}

//------------------------------------------------------------------------------
//it serves for createImage


void UrTree::createRowImage (int row, int startCol, Vec<char> &name, Mat<char> &image) {
	for (int count_ = 0; count_ < name.getSize (); count_ ++)
		image[row][startCol + count_] = name[count_];
}


//------------------------------------------------------------------------------
//draw the internal node into the image
void UrTree::createImage (Vec<int> &abstractBrLenArr, Vec<int> &ndRowOrderArr, Vec<int> &abstractNdHeiArr, Mat<char> &image) {

	for (int inNdNo_ = maxNExNd_; inNdNo_ < maxNExNd_ + maxNInNd_; inNdNo_ ++)
		if (inNdArr_[inNdNo_].getId () != -1 ) {
			InNd *inNd_;
			inNd_ = &inNdArr_[inNdNo_];
			if (inNd_->getId () != -1) {

				Vec<int> *neiNdNoLs_;

				neiNdNoLs_ = &inNd_->getNeiNd ();

				int col_ = abstractNdHeiArr[inNdNo_];

				for (int count_ = 1; count_ < 3; count_ ++) {
					int chiNdNo_ = (*neiNdNoLs_)[count_];
					int lastCol_ = abstractNdHeiArr[chiNdNo_];
					int row_ = ndRowOrderArr[chiNdNo_];

					createRowImage (row_, col_ + 1 , lastCol_ - 1, '-', image);
					if (isExNd (chiNdNo_) == 1) {
						Vec<char> name_;
						alignment.getName (chiNdNo_, name_);
						createRowImage (row_, lastCol_ + 2, name_, image);
					}
				}

				int row_ = ndRowOrderArr[inNdNo_];

				int firstChiNdNo_ = (*neiNdNoLs_)[1];
				int lastChiNdNo_ = (*neiNdNoLs_)[2];


				int firstRow_ = ndRowOrderArr[firstChiNdNo_];


				int lastRow_ = ndRowOrderArr[lastChiNdNo_];
				createColImage (col_, firstRow_, lastRow_, ':', image);

				Vec<char> inNdName_;
				inNdName_.convert (inNdNo_);
				createRowImage (row_, col_, inNdName_, image);
			}

		} //end if


	int outGrpRow_ = ndRowOrderArr[outGrpNdNo_];
	int brNo_ = exNdArr_[outGrpNdNo_].getBr ();
	Vec<char> sArtName_;
	sArtName_.convert (artInRootNo_);
	createRowImage (outGrpRow_, sArtName_.getSize (), abstractBrLenArr[brNo_], '-', image);

	Vec<char> outGrpName_;
	alignment.getName (outGrpNdNo_, outGrpName_);
	createRowImage (outGrpRow_, abstractBrLenArr[brNo_] + 2, outGrpName_, image);

} //end of function



//------------------------------------------------------------------------------
int UrTree::cmpNImageRow (Vec<int> &ndRowOrderArr) {
	int nRow_ = 0;
	for (int exNdNo_ = 0; exNdNo_ < maxNExNd_; exNdNo_ ++)
		if (exNdArr_[exNdNo_].getId () != -1 && ndRowOrderArr[exNdNo_] > nRow_)

			nRow_ = ndRowOrderArr[exNdNo_];
	return nRow_ + 1;
}

//------------------------------------------------------------------------------
int UrTree::cmpNImageCol (Vec<int> &abstractNdHeiArr) {
	int nCol_ = 0;
	for (int exNdNo_ = 0; exNdNo_ < maxNExNd_; exNdNo_ ++)
		if (exNdArr_[exNdNo_].getId () != -1) {
			Vec<char> name_;
			alignment.getName (exNdNo_, name_);
			int len_ = name_.getSize ();
			//      std::cout << exNdNo_ << " " << len_ << " " << abstractNdHeiArr[exNdNo_] << endl;
			if (abstractNdHeiArr[exNdNo_] + len_ + 1 > nCol_)
				nCol_ = abstractNdHeiArr[exNdNo_] + len_ + 2;
		}
	return nCol_ + 10;
}

