/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/***************************************************************************
                          array.h  -  description
                             -------------------
    begin                : Thu Feb 20 2003
    copyright            : (C) 2003 by Vinh Le Sy and Arndt von Haeseler
    email                : vinh@cs.uni-duesseldorf.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef VEC_H
#define VEC_H

#include <iostream>
#include <fstream>
#include "constant.h"
#include "utl.h"
#include <cstring>

using namespace std;

template <class T>
class Vec {
public :

	//the constructor
	Vec () {
		size_ = 0;
		items_ = 0;
		limit_ = 0;
	}

	//--------------------------------------------------------------------
	//constructor
	Vec (int limit) {
		size_ = 0;
		items_ = 0;
		limit_ = 0;
		setLimit (limit);
	}

	//--------------------------------------------------------------------
	//constructor
	Vec (int limit, int size) {
		size_ = 0;
		items_ = 0;
		limit_ = 0;
		setLimit (limit);
		setSize (size);
	}

	//set both size and limit for this vector
	void set (int limit, int size) {
		setLimit (limit);
		setSize (size);
	}

	//--------------------------------------------------------------------
	//set the size for this vector
	void setSize (const int size) {
		size_ = size;
	}

	//--------------------------------------------------------------------
	//return the size of this vector
	int getSize () {
		return size_;
	}

	//--------------------------------------------------------------------
	//set the limit for this vector, REMAIN all OLD information
	void addLimit (int inc) {
		//create new memory address for this vector
		int newLimit_ = limit_ + inc;
		T *newItems_ = 0;
		newItems_ = new T[newLimit_];

		if (newItems_ == 0)
			Utl::announceError (MEMORY_LOCATION);

		for (int count_ = 0; count_ < limit_; count_ ++)
			newItems_[count_] = items_[count_];

		//release all unused memory
		if (items_ != 0) {
			delete [] items_;
			items_ = 0;
		}

		limit_ = newLimit_;
		items_ = newItems_;
	}

	//--------------------------------------------------------------------
	//set the limit for this vector, REMAIN all OLD information
	void setLimit (int limit) {
		//do not need resetLimit
		if (limit_ == limit || limit <= 0)
			return ;

		release ();

		nVec_ ++;

		size_ = 0;
		limit_ = limit;
		items_ = 0;
		//create new memory address for this vector
		//  std::cout << sizeof (T) << endl;
		items_ = new T[ limit_ ];

		if (items_ == 0)
			Utl::announceError (MEMORY_LOCATION);
	}


	//--------------------------------------------------------------------
	void set (const char *aLine) {
		int len_ = strlen (aLine);
		char ch;
		set (len_, 0);
		for (int pos_ = 0; pos_ < len_; pos_ ++)
			if (Utl::isTreeChar (ch = aLine[pos_]) == 1)
				add (ch);
	}

	//--------------------------------------------------------------------
	//return the limit of this vector
	int getLimit () {
		return limit_;
	}

	//--------------------------------------------------------------------
	//add and item into item list
	void add (T &item) {
		//if limit not enough for add a new item, increase limit up an INC
		if (size_ + 1 > limit_)
			addLimit (INC);

		items_[size_] = item;
		size_++;
	}

	//--------------------------------------------------------------------
	/*
	delete items from position lowerIndex to position upperIndex
	return 1: if all boundary conditions are satisfied
	return 0: Otherwise
	*/
	int del (const int lowerIndex, const int upperIndex) {
		//check boundary conditions
		if (lowerIndex < 0 || lowerIndex >= size_ ||
		        upperIndex < 0 || upperIndex >= size_ )
			return 0;

		int delta_ = upperIndex - lowerIndex + 1;
		for (int i = upperIndex + 1; i< size_; i++)
			items_[i - delta_] = items_[i];

		size_ -= delta_;

		return 1;
	}

	//--------------------------------------------------------------------
	/*
	delete the item at position index
	return 1: if all boundary conditions are satisfied
	return 0: Otherwise
	*/
	int del (const int index) {
		//check boundary conditions
		if (index < 0 || index >= size_) {
			printVecInfo(index, "error inside del()");
			Utl::announceError (BAD_MEMORY_ACCESS);
		}

		for (int count_ = index + 1; count_ < size_; count_ ++)
			items_[count_ - 1] = items_[count_];

		size_--;
		return 1;
	}

	//--------------------------------------------------------------------
	/*
	delete the first item which is equal to given item
	return 1: if there exists an item whose content is equal to value
	return 0: otherwise
	*/
	int delItem (T &item) {
		for (int count_ = 0; count_ < size_; count_ ++)
			if (items_[count_] == item) {
				del (count_);
				return 1;
			}
		return 0;
	}

	//--------------------------------------------------------------------
	/*
	find the first item which is equal to a given oldItem, 
	replace this item by new given item
	return 1: if there exists an item which is equal to given oldItem
	return 0: otherwise
	*/
	int changeItem (T &oldItem, T &newItem) {
		for (int count_ = 0; count_ < size_; count_ ++)
			if (items_[count_] == oldItem) {
				items_[count_] = newItem;
				return 1;
			}
		return 0;
	}






	//--------------------------------------------------------------------
	//copy the vec into this, the vec is passed by reference
	void copyRef (Vec<T> &vec) {
		/*check, change the limit of this if the limit of this
		  is smaller than the limit of vec */
		if (&vec != this) {
			if (limit_ < vec.getLimit () )
				setLimit(vec.getLimit () );

			size_ = vec.getSize ();
			for (int count_ = 0; count_ < size_; count_ ++)
				this->items_[count_] = vec[count_];
		}
	}

	//--------------------------------------------------------------------
	//copy the vec into this, the vec is passed by value, not be reference
	void copyVal (Vec<T> vec) {
		copyRef (vec);
	}

	//--------------------------------------------------------------------

	//overload operator =
	void operator = (Vec<T> &vec) {
		copyRef (vec);
	}

	//--------------------------------------------------------------------

	//return items_[index]
	T& getItem (const int index) {
		//check boundary conditions
		if (index < 0 || index >= size_) {
			printVecInfo(index, "error inside getItem()");
			Utl::announceError (BAD_MEMORY_ACCESS);
		}

		return items_[index];
	}

	//--------------------------------------------------------------------
	//copy all items from start point to end point to the sub vec
	void getItem (int startPoint, int endPoint, Vec<T> &subVec) {
		subVec.set (endPoint - startPoint + 1, 0);
		for (int count_ = startPoint; count_ <= endPoint; count_ ++)
			subVec += items_[count_];
	}

	//--------------------------------------------------------------------
	//return items_
	char &getItem () {
		return items_;
	}

	//--------------------------------------------------------------------
	//overload [], return the items_[index]
	inline T& operator [](const int index) {
		//check boundary conditions
		//  return items_[index];
		if (index < 0 || index >= size_) {
			std::cout << index << endl;
			printVecInfo(index, "error in operator[]");
			Utl::announceError (BAD_MEMORY_ACCESS);
		}
		return items_[index];
	}

	//--------------------------------------------------------------------
	//overload operator +=
	void operator += (T &item) {
		add(item);
	}

	//--------------------------------------------------------------------
	//overload operator +=
	void operator += (Vec<T> &vec) {
		int newSize_ = size_ + vec.getSize ();
		if (newSize_ > limit_)
			addLimit (newSize_ - limit_);

		for (int count_ = 0; count_ < vec.getSize (); count_ ++)
			add (vec[count_]);
	}

	//--------------------------------------------------------------------
	//clean all content of this vector
	void clean() {
		size_ = 0;
	}


	//--------------------------------------------------------------------
	//insert the item into this vector at position index
	void insert (const int index, T &item) {
		if (size_ + 1 > limit_)
			addLimit (INC);


		for (int pos_ = size_; pos_ >= index; pos_ --)
			items_[pos_] = items_[pos_ - 1];
		items_[index] = item;
		size_ ++;
	}

	//--------------------------------------------------------------------
	//write the items list, when this vector is a array of characters
	template <class Tout>
	Tout &writeItem (Tout &out) {
		if (sizeof (T) == sizeof (char) ) {
			for (int count_ = 0; count_ < size_; count_ ++)
				out << items_[count_];
		} else
			for (int count_ = 0; count_ < size_; count_ ++) {
				out.precision (5);
				out.width (12);
				out << items_[count_];
			}
		return out;
	}

	//--------------------------------------------------------------------
	//check if the input vector is qual to this vector
	int operator == (Vec<T> &vec) {

		if (size_ != vec.getSize () )
			return 0;

		for (int count_ = 0; count_ < size_; count_ ++)
			if (items_[count_] != vec[count_])
				return 0;

		return 1;
	}

	//--------------------------------------------------------------------
	//the the index of the input item
	int findItem (T &item) {
		for (int index_ = 0; index_ < size_; index_ ++)
			if (items_[index_] == item)
				return index_;
		return -1;
	}

	//--------------------------------------------------------------------
	//the the index of the input item, BUT RIGHT POINT TO LEFT
	int findRightItem (T &item, int rightPoint) {
		for (int index_ = rightPoint; index_ >= 0; index_ --)
			if (items_[index_] == item)
				return index_;
		return -1;
	}

	//--------------------------------------------------------------------
	int cmp (Vec<T> &vec) {
		if (this->size_ < vec.getSize () )
			return -1;
		if (this->size_ > vec.getSize () )
			return 1;

		for (int count_ = 0; count_ < size_; count_ ++)
			if (this->items_[count_] < vec[count_])
				return -1;
			else
				if (this->items_[count_] > vec[count_])
					return 1;
		return 0;
	}

	//--------------------------------------------------------------------
	//convert from an integer to the char vector
	void convert (int val) {
		val = static_cast<int> (val);
		size_ = 0;
		do {
			char reminder_ = val % 10 + '0';
			this->insert (0, reminder_);
			val = static_cast <int> (val / 10);
		} while (val > 0);
	}


	void convert (double val, int presion) {
		// std::cout << val << endl;
		size_ = 0;
		int intVal_ = static_cast<int> (val);
		convert (intVal_);

		double realVal_ = val - intVal_;
		char colon_ = '.';
		this->add (colon_);
		int count_;
		for (count_ = 0; count_ < presion; count_ ++)
			realVal_ = realVal_ * 10;

		Vec<char> realPart_;
		realPart_.convert (static_cast<int> (realVal_));
		char zero = '0';
		for (count_ = realPart_.getSize (); count_ < presion; count_ ++)
			realPart_.insert (0,zero);
		(*this) += realPart_;
	}

	//--------------------------------------------------------------------
	//release the memory of this class
	void release () {
		if (items_ != 0) {
			nVec_ --;
			delete [] items_;
			limit_ = 0;
			items_ = 0;
		}
	}

	/** print the vector information */
	void printVecInfo(int index, const char *str) {
		cout << str << endl;
		cout << "index " << index << " not in range 0.." << size_ << endl;
		for (int i=0; i < size_; i++) {
			//cout << items_[i];
		}
	}
	
	//--------------------------------------------------------------------
	//destructor, release all unused memory
	virtual ~Vec () {
		release ();
		//  std::cout << "destruction of vector class " << endl;
	}

	//total numbe vecs in the memory
	static int nVec_;

	/*--------------------------------------------------------------------
	--------------------------------------------------------------------
	--------------------------------------------------------------------
	--------------------------------------------------------------------
	--------------------------------------------------------------------
	--------------------------------------------------------------------*/

	//the current size of this Vec
	int size_;

	//the item list of this vector
	T *items_;

	//the limit of this vector
	int limit_;
}
;// end of class template array

template <class T>
int Vec<T>::nVec_ = 0;
#endif

