/* $Id: bmp.c,v 1.1.1.1 2003/01/30 12:22:26 hito Exp $ */
#define BUF_SIZE 1024

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "stimg.h"

typedef struct{
  unsigned short bfType;
  unsigned long bfSize;
  unsigned short bfReserved1;
  unsigned short bfReserved2;
  unsigned long bfOffBits;
} BITMAPFILE;

typedef struct{
  unsigned long	biSize;
  long biWidth;
  long biHeight;
  short	biPlanes;
  short	biBitCount;
  unsigned long	biCompression;
  unsigned long	biSizeImage;
  long biXPelsPerMeter;
  long biYPelsPerMeter;
  unsigned long	biClrUsed;
  unsigned long	biClrImportant;
} BITMAPINFO;

typedef struct{
  unsigned char B;
  unsigned char G;
  unsigned char R;
  unsigned char Reserved;
}RGBQUAD;


static int ReadUnCompress(FILE *fp, STIMG *bmp, RGBQUAD *pal, BITMAPINFO *bi);
static int Read24bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal);
static int Read8bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal);
static int Read4bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal);
static int Read1bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal);
static int ReadCompress(FILE *fp, STIMG *bmp, RGBQUAD *pal, BITMAPINFO *bi);

static BITMAPFILE *read_bitmapfile(FILE *fp);
static BITMAPINFO *read_bitmapinfo(FILE *fp);
static unsigned short read_short(FILE *fp);
static unsigned long read_long(FILE *fp);

STIMG *
load_bmp(char *bmp_file, int check);

STIMG_ANIMATION *
load_animation_bmp(char *bmp_file, int check)
{
  STIMG_ANIMATION *animation;
  STIMG *image;

  image = load_bmp(bmp_file, check);
  if (image == NULL)
    return NULL;

  animation = stimg_animation_new();
  if (animation == NULL) {
    stimg_delete(image);
    return NULL;
  }

  stimg_animation_add_frame(animation, image, 0, 0, 0, 0);
  return animation;
}

STIMG *
load_bmp(char *bmp_file, int check)
{
  int i, palsize;
  BITMAPINFO *bi = NULL;
  BITMAPFILE *bf = NULL;
  RGBQUAD pal[256];
  STIMG *bmp = NULL;
  FILE *fp = NULL;

  if(bmp_file == NULL || (fp = fopen(bmp_file, "r")) == NULL)
    goto ERROR;

  bf = read_bitmapfile(fp);
  if (bf == NULL)
    goto ERROR;

  bi = read_bitmapinfo(fp);
  if (bi == NULL)
    goto ERROR;

  if((bf->bfType != 0x4d42) ||
     (bi->biBitCount != 1 && bi->biBitCount != 4 && bi->biBitCount != 8 && bi->biBitCount != 24) ||
     (bi->biCompression == 1 && bi->biBitCount != 8) ||
     (bi->biCompression == 2 && bi->biBitCount != 4))
    goto ERROR;


  bmp = stimg_new(bi->biWidth, bi->biHeight, 0);
  if (bmp == NULL || check) {
    free(bi);
    free(bf);
    fclose(fp);
    return bmp;
  }
 
  if(bi->biBitCount <= 8){
    palsize = 1 << bi->biBitCount;
    for(i = 0; i < palsize; i++){
#if 0
      unsigned char tmp[4];
      fread(tmp, 1, 4, fp);
      pal[i].B = tmp[0];
      pal[i].G = tmp[1];
      pal[i].R = tmp[2];
      pal[i].Reserved = tmp[3];
#else
      fread(&pal[i], sizeof(pal[i]), 1, fp);
#endif
    }
  }
 
  fseek(fp, bf->bfOffBits, SEEK_SET);

  if(bi->biCompression != 0){
    if (ReadCompress(fp, bmp, pal, bi))
      goto ERROR;
  }else{
    if (ReadUnCompress(fp, bmp, pal, bi))
      goto ERROR;
  }
  free(bi);
  free(bf);
  fclose(fp);
  return bmp;

 ERROR:
  if (bmp)
    stimg_delete(bmp);
  if (bi)
    free(bi);
  if (bf)
    free(bf);
  if (fp)
    fclose(fp);
  return NULL;
}


/************ static functions ************/

static unsigned short read_short(FILE *fp)
{
  unsigned char b[2];
   
  if (fread(b, 1, 2, fp) != 2)
    return 0;
   
  return (b[1] << 8) | b[0];
}

static unsigned long read_long(FILE *fp)
{
  unsigned char b[4];
   
  if (fread(b, 1, 4, fp) != 4)
    return 0;
   
  return (b[3] << 24) | (b[2] << 16) | (b[1] << 8) | b[0];
}

static BITMAPFILE *read_bitmapfile(FILE *fp)
{
  BITMAPFILE *bf;

  bf = malloc(sizeof(BITMAPFILE));
  if(bf == NULL)
    return NULL;

  bf->bfType = read_short(fp);
  bf->bfSize = read_long(fp);
  bf->bfReserved1 = read_short(fp);
  bf->bfReserved2 = read_short(fp);
  bf->bfOffBits = read_long(fp);

  return bf;
}

static BITMAPINFO *read_bitmapinfo(FILE *fp)
{
  BITMAPINFO *bi;
  
  bi = malloc(sizeof(BITMAPINFO));
  if(bi == NULL)
    return NULL;

  bi->biSize = read_long(fp);
  bi->biWidth = read_long(fp);
  bi->biHeight = read_long(fp);
  bi->biPlanes = read_short(fp);
  bi->biBitCount = read_short(fp);
  bi->biCompression = read_long(fp);
  bi->biSizeImage = read_long(fp);
  bi->biXPelsPerMeter = read_long(fp);
  bi->biYPelsPerMeter = read_long(fp);
  bi->biClrUsed = read_long(fp);
  bi->biClrImportant = read_long(fp);

  return bi;
}

static int ReadUnCompress(FILE *fp, STIMG *bmp, RGBQUAD *pal, BITMAPINFO *bi)
{
  int r;

  switch(bi->biBitCount){
  case 24:
    r = Read24bpp(fp, bmp, pal);
    break;
  case 8:
    r = Read8bpp(fp, bmp, pal);
    break;
  case 4:
    r = Read4bpp(fp, bmp, pal);
    break;
  case 1:
    r = Read1bpp(fp, bmp, pal);
    break;
  default:
    r = 1;
  }
  return r;
}

static int Read24bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal)
{
  int x, y, height, width, linesize;
  unsigned char *tmp, *ptr;

  height = bmp->h;
  width = bmp->w;
  ptr = bmp->data + (height - 1) * width * 3;

  linesize = width * 3;
  linesize += (3 - (linesize - 1) % 4);

  tmp = malloc(linesize);
  if (tmp == NULL)
    return 1;

  for(y=0; y < height; y++){
    fread(tmp, 1, linesize, fp);
    for(x=0; x < width; x++){
      *ptr++ = tmp[x * 3 + 2];
      *ptr++ = tmp[x * 3 + 1];
      *ptr++ = tmp[x * 3];
    }
    ptr -= width * 2 * 3;
  }
  free(tmp);
  return 0;
}

static int Read8bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal)
{
  int i, x, y, height, width;
  unsigned long linesize;
  unsigned char *tmp, *ptr;

  height = bmp->h;
  width = bmp->w;
  ptr = bmp->data + (height - 1) * width * 3;

  linesize = width + (3 - (width - 1) % 4);
  tmp = malloc(linesize);
  if (tmp == NULL)
    return 1;

  for(y=0; y < height; y++){
    fread(tmp, 1, linesize, fp);
    for(x=0; x < width; x++){
      i = tmp[x];
      *ptr++ = pal[i].R;
      *ptr++ = pal[i].G;
      *ptr++ = pal[i].B;
    }
    ptr -= width * 2 * 3;
  }
  free(tmp);
  return 0;
}

static int Read4bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal)
{
  int i, x, y, height, width;
  unsigned long linesize;
  unsigned char *tmp, *ptr;


  height = bmp->h;
  width = bmp->w;
  ptr = bmp->data + (height - 1) * width * 3;

  linesize = width / 2 + ((width % 2)?1:0);
  linesize += (3 - (linesize - 1) % 4);
  tmp = malloc(linesize);
  if (tmp == NULL)
    return 1;

  for(y=0; y < height; y++){
    fread(tmp, 1, linesize, fp);
    x=0;
    while(x < width){
      i = tmp[x / 2] >> 4;
      *ptr++ = pal[i].R;
      *ptr++ = pal[i].G;
      *ptr++ = pal[i].B;
      x++;

      if(x >= width)
	break;

      i = tmp[x / 2] & 0x0f;
      *ptr++ = pal[i].R;
      *ptr++ = pal[i].G;
      *ptr++ = pal[i].B;
      x++;
    }
    ptr -= width * 2 * 3;
  }
  free(tmp);
  return 0;
}

static int Read1bpp(FILE *fp, STIMG *bmp, RGBQUAD *pal)
{
  int i, j, x, y, width, height;
  unsigned long linesize;
  unsigned char *tmp, *ptr;

  height = bmp->h;
  width = bmp->w;
  ptr = bmp->data + (height - 1) * width * 3;

  linesize = width / 8 + ((width % 8)?1:0);
  linesize += (3 - (linesize - 1) % 4);
  tmp = malloc(linesize);
  if (tmp == NULL)
    return 1;

  for(y = 0; y < height; y++){
    fread(tmp, 1, linesize, fp);
    x = 0;
    while(x < width){
      for(i = 0x80; i > 0; i >>= 1){
	j = (tmp[x >> 3] & i)? 1: 0;
	*ptr++ = pal[j].R;
	*ptr++ = pal[j].G;
	*ptr++ = pal[j].B;
	x++;
	if(x >= width)
	  break;
      }
    }
    ptr -= width * 2 * 3;
  }
  free(tmp);
  return 0;
}

static int ReadCompress(FILE *fp, STIMG *bmp, RGBQUAD *pal, BITMAPINFO *bi)
{
  int r = 0, i, x, y, bit, compression, literal, linesize, width, height;
  unsigned char data1, data2, *ptr;

  height = bmp->h;
  width = bmp->w;
  ptr = bmp->data + (height - 1) * width * 3;

  bit = 8 / bi->biBitCount;
  linesize = width / bit + ((width % bit)?1:0);
  linesize += (3 - (linesize - 1) % 4);

  compression = bi->biCompression;

  x = y = 0;
  while(1){
    data1 = r = fgetc(fp);
    data2 = r = fgetc(fp);
    if(r == EOF)
      return ferror(fp);

    if(data1 == 0 && data2 == 0){
      if(x != width)
	goto ERR;
      y++;
      x = 0;
      ptr -= width * 2 * 3;
      continue;
    }

    if(data1 == 0 && data2 == 1)		/* EOF */
      break;

    if(data1 == 0 && data2 == 2){		/* MOVE */
      return 1;
    }

    if(data1 == 0 && data2 != 0){
      literal = 1;
      data1 = data2;
    }else{
      literal = 0;
    }

    for(i = 0; i < data1; i += compression){
      if(literal == 1)
	data2 = r = fgetc(fp);

      if(compression == 1){
	*ptr++ = pal[data2].R;
	*ptr++ = pal[data2].G;
	*ptr++ = pal[data2].B;
	x++;
      }else{
	*ptr++ = pal[data2 >> 4].R;
	*ptr++ = pal[data2 >> 4].G;
	*ptr++ = pal[data2 >> 4].B;
	x++;

	if(x >= width) break;

	*ptr++ = pal[data2 & 0x0f].R;
	*ptr++ = pal[data2 & 0x0f].G;
	*ptr++ = pal[data2 & 0x0f].B;
	x++;
      }

      if(x > width)
	goto ERR;
    }

    if(literal == 1){
      if(compression == 2){
	if(((data1 / 2) + (data1 % 2)) % 2 != 0)   /* padding */
	  data2 = r = fgetc(fp);
      }else{
	if(data1 % 2 != 0)                         /* padding */
	  data2 = r = fgetc(fp);
      }
    }
    if (r == EOF)
      return ferror(fp);
  }
  return 0;

 ERR:
  return 1;
}

