/*
 * STAN -- Stream Analyser -- http://www.roqe.org/stan
 * Copyright (c) 2001-2004 Konrad Rieck <kr@roqe.org> 
 * All rights reserved.
 * ---
 * $Id: bits.c,v 1.7 2003/12/10 20:29:29 kr Exp $
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    ``This product includes software developed by Konrad Rieck.''
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ---
 *
 * This file contains all functions that deal with bit analysis. Those
 * functions are needed if we measure the power of a pseudo random number
 * generator.
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <stan.h>
#include <data.h>
#include <stats.h>
#include <bintree.h>

size_t run = 1;
int bit = -1;

extern bnode_t *bintree;

double sum_false, sum_true, bits_ratio;

int get_bit(byte_t b, int pos)
{
   if (b & (int) pow(2, pos))
      return 1;
   return 0;
}

void analyse_byte(byte_t b)
{
   int i;
   for (i = 7; i >= 0; i--) {
      if (bit < 0) {
	 bit = get_bit(b, i);
	 run = 1;
      } else if (bit == get_bit(b, i))
	 run++;
      else {
	 binsert(&bintree, run, bit);
	 run = 1;
	 bit = get_bit(b, i);
      }
   }
}

void calc_sum_bits(bnode_t * bnode)
{
   sum_true += bnode->count_true * bnode->run;
   sum_false += bnode->count_false * bnode->run;
}

void print_runs_ratios()
{
   bnode_t **bnode, **succ;
   double sum_bnode, sum_succ;
   int i;

   printf("\n   Ratio of runs of a length to the next length.\n");

   printf("   Length: ");
   bnode = bmin(&bintree);
   succ = bsuccessor(bnode);
   for (i = 0; i < 8 && *bnode && *succ; i++) {
      printf("%2d ->%2d ", (*bnode)->run, (*succ)->run);
      bnode = succ;
      succ = bsuccessor(bnode);
   }

   printf("\n   Ratio:  ");
   bnode = bmin(&bintree);
   succ = bsuccessor(bnode);
   for (i = 0; i < 8 && *bnode && *succ; i++) {
      sum_bnode = (*bnode)->count_true + (*bnode)->count_false;
      sum_succ = (*succ)->count_true + (*succ)->count_false;
      printf(" %2.2f   ", sum_bnode / sum_succ);
      bnode = succ;
      succ = bsuccessor(bnode);
   }

   printf("\n\n");
   printf("   Ratio of 1/0 runs for each length.\n");

   printf("   Length: ");
   bnode = bmin(&bintree);
   for (i = 0; i < 8 && *bnode; i++) {
      printf(" %d      ", (*bnode)->run);
      bnode = bsuccessor(bnode);
   }

   printf("\n   Ratio:  ");
   bnode = bmin(&bintree);
   for (i = 0; i < 8 && *bnode; i++) {
      printf(" %2.2f   ",
	     (double) (*bnode)->count_true / (*bnode)->count_false);
      bnode = bsuccessor(bnode);
   }

   printf("\n");

}

void calc_bit_ratio()
{
   sum_true = 0;
   sum_false = 0;

   btraverse(&bintree, calc_sum_bits, inorder);
   bits_ratio = sum_true / sum_false;
}

void print_bits_stats()
{
   printf("Bit statistics for the stream, bytes %d, bits %d\n",
	  size_data(), size_data() * 8);

   printf("   Total runs %d(1)/%d(0), different runs %d(1)/"
	  "%d(0)\n",
	  bsize_total(&bintree, 1), bsize_total(&bintree, 0),
	  bsize_bnodes(&bintree, 1), bsize_bnodes(&bintree, 0));


   printf("   Shortest run: ");
   print_double((*bmin(&bintree))->run);
   printf("\n");
   printf("   Longest run:  ");
   print_double((*bmax(&bintree))->run);
   printf("\n");

   calc_bit_ratio();
   printf("   Bit ratio:    ");
   print_double(bits_ratio);
   printf(" = %d(1)/%d(0) ", (int) sum_true, (int) sum_false);
   printf("\n");

   print_runs_ratios();
}
