#include <test_common.h>
#include <igl/curved_hessian_energy.h>
#include <igl/EPS.h>

#include <vector>


TEST_CASE("curved_hessian_energy: cube", "[igl]")
{
  Eigen::MatrixXd V;
  Eigen::MatrixXi F;
  //This is a cube of dimensions 1.0x1.0x1.0
  igl::read_triangle_mesh(test_common::data_path("cube.obj"), V, F);

  Eigen::SparseMatrix<double> Qs;
  igl::curved_hessian_energy(V, F, Qs);
  Eigen::MatrixXd Q(Qs);

  Eigen::MatrixXd Qexact(8,8);
  Qexact << 0x1.3da9e8a554e1cp+4,-0x1.2487ed5110b44p+3,0x1.860a91c16b9b8p+1,-0x1.2487ed5110b44p+3,-0x1.2487ed5110b44p+3,0x1.860a91c16b9b8p+1,-0x1.921fb54442cfap+0,0x1.860a91c16b9b8p+1,-0x1.2487ed5110b44p+3,0x1.5da9e8a554e1cp+4,-0x1.4487ed5110b4ap+3,0x1p+0,0x1p+0,-0x1.4487ed5110b4ap+3,0x1.860a91c16b9b8p+1,0x1.430548e0b5cdcp+1,0x1.860a91c16b9b8p+1,-0x1.4487ed5110b4ap+3,0x1.5da9e8a554e1cp+4,-0x1.4487ed5110b4ap+3,0x1.430548e0b5cdcp+1,0x1p+0,-0x1.2487ed5110b44p+3,0x1p+0,-0x1.2487ed5110b44p+3,0x1p+0,-0x1.4487ed5110b4ap+3,0x1.5da9e8a554e1cp+4,0x1p+0,0x1.430548e0b5cdcp+1,0x1.860a91c16b9b8p+1,-0x1.4487ed5110b4ap+3,-0x1.2487ed5110b44p+3,0x1p+0,0x1.430548e0b5cdcp+1,0x1p+0,0x1.5da9e8a554e1cp+4,-0x1.4487ed5110b4ap+3,0x1.860a91c16b9b8p+1,-0x1.4487ed5110b4ap+3,0x1.860a91c16b9b8p+1,-0x1.4487ed5110b4ap+3,0x1p+0,0x1.430548e0b5cdcp+1,-0x1.4487ed5110b4ap+3,0x1.5da9e8a554e1cp+4,-0x1.2487ed5110b44p+3,0x1p+0,-0x1.921fb54442cfap+0,0x1.860a91c16b9b8p+1,-0x1.2487ed5110b44p+3,0x1.860a91c16b9b8p+1,0x1.860a91c16b9b8p+1,-0x1.2487ed5110b44p+3,0x1.3da9e8a554e1cp+4,-0x1.2487ed5110b44p+3,0x1.860a91c16b9b8p+1,0x1.430548e0b5cdcp+1,0x1p+0,-0x1.4487ed5110b4ap+3,-0x1.4487ed5110b4ap+3,0x1p+0,-0x1.2487ed5110b44p+3,0x1.5da9e8a554e1cp+4;

  test_common::assert_near(Q,Qexact,1e-12);
}


TEST_CASE("curved_hessian_energy: annulus", "[igl]")
{
  Eigen::MatrixXd V(99,3);
  Eigen::MatrixXi F(148,3);
  //This is an annulus
  V.col(0) << 0.29999999999999998889776975374843,0.24270509831248421317440033817547,0.0927050983124842048477276534868,-0.0927050983124842048477276534868,-0.24270509831248421317440033817547,-0.29999999999999998889776975374843,-0.24270509831248421317440033817547,-0.092705098312484301992242308187997,0.0927050983124842048477276534868,0.24270509831248421317440033817547,1,0.98768834059513777035022030759137,0.95105651629515353118193843329209,0.89100652418836789880884907688596,0.80901699437494745126286943559535,0.70710678118654757273731092936941,0.58778525229247313710345679282909,0.45399049973954691550304119118664,0.30901699437494750677402066685318,0.15643446504023089671520096999302,9.999999999999999790977867240346e-17,-0.15643446504023059140386919807497,-0.30901699437494728472941574182187,-0.45399049973954669345843626615533,-0.58778525229247302608115433031344,-0.70710678118654746171500846685376,-0.80901699437494734024056697307969,-0.89100652418836778778654661437031,-0.95105651629515353118193843329209,-0.98768834059513765932791784507572,-1,-0.98768834059513777035022030759137,-0.9510565162951537532265433583234,-0.89100652418836789880884907688596,-0.80901699437494745126286943559535,-0.70710678118654768375961339188507,-0.58778525229247324812575925534475,-0.45399049973954691550304119118664,-0.30901699437494761779632312936883,-0.15643446504023100773750343250867,-1.9999999999999999581955734480692e-16,0.15643446504023070242617166059063,0.30901699437494717370711327930621,0.45399049973954658243613380363968,0.58778525229247291505885186779778,0.70710678118654735069270600433811,0.80901699437494734024056697307969,0.89100652418836778778654661437031,0.95105651629515353118193843329209,0.98768834059513765932791784507572,-0.54653773385274273532985489509883,-0.16778761071338771371230791373819,-0.6640334885500396300272996086278,-0.3755430578136348396256494197587,0.27505167682734571465985595750681,0.61283057248268479888508863950847,0.37554305781363522820370803856349,0.66403348855003985207190453365911,-0.69888569632849417700981575762853,-0.71801469123163186303315796976676,-0.54690508304406848516521222336451,-0.32226059481637447934332385557354,-0.066070994246462080856652221427794,-0.8188400187389239803792406746652,-0.586539425411693127188073049183,-0.056392056768511444886371464235708,-0.32226059481637425729871893054224,0.19658609558397963090747850856133,0.43999996868279372952414973951818,0.64034357918688944444340904738056,0.77800589862411184682855491701048,0.3222605948163745903656263180892,0.5469050830440682631206072983332,0.71801469123163186303315796976676,0.76897255750667192764069568511331,-0.83951158001816739151479396241484,-0.3394941295361795230434154291288,-0.64774834858972418771827506134287,0.066070994246462080856652221427794,0.050978900626839074305962640210055,0.42197972345950862660401980974711,0.1516814321578138502477628435372,0.49407477612085504148353720665909,0.62620665770565242524980931193568,-0.29074511527527352461319765097869,0.47043547857839895787179784747423,-0.47043547857839940196100769753684,-0.26935940260987811800674762707786,-0.45589550848838600316526026290376,-9.7144514654701197287067770957947e-17,0.26935940260987795147329393330438,5.5511151231257827021181583404541e-17,0.28175891956423887441118836250098,0.4328463157366620461807826814038,0.80860600926288284107101844710996,-0.18390862049392966981464780928945,0.59788321616198325525459722484811,-0.43994121336078612083397842980048,-0.47585524604634232836986029724358;
  V.col(1) << 0,0.17633557568774191337546142221981,0.28531695488854602604789079123293,0.28531695488854608155904202249076,0.17633557568774199664218826910655,0,-0.17633557568774191337546142221981,-0.28531695488854602604789079123293,-0.28531695488854608155904202249076,-0.17633557568774199664218826910655,0,0.15643446504023089671520096999302,0.30901699437494739575171820433752,0.45399049973954669345843626615533,0.58778525229247313710345679282909,0.70710678118654746171500846685376,0.80901699437494745126286943559535,0.89100652418836778778654661437031,0.95105651629515353118193843329209,0.98768834059513777035022030759137,1,0.98768834059513777035022030759137,0.95105651629515364220424089580774,0.89100652418836789880884907688596,0.80901699437494745126286943559535,0.70710678118654757273731092936941,0.58778525229247324812575925534475,0.45399049973954691550304119118664,0.30901699437494750677402066685318,0.15643446504023100773750343250867,9.999999999999999790977867240346e-17,-0.15643446504023070242617166059063,-0.30901699437494689615135712301708,-0.45399049973954669345843626615533,-0.58778525229247302608115433031344,-0.70710678118654746171500846685376,-0.80901699437494734024056697307969,-0.89100652418836778778654661437031,-0.95105651629515353118193843329209,-0.98768834059513765932791784507572,-1,-0.98768834059513777035022030759137,-0.95105651629515364220424089580774,-0.89100652418836800983115153940162,-0.809016994374947562285171898111,-0.70710678118654768375961339188507,-0.5877852522924733591480617178604,-0.45399049973954702652534365370229,-0.30901699437494761779632312936883,-0.15643446504023109100423027939541,-0.46678732236060038740532718293252,-0.6988856963284932888313960575033,0.27505167682734582568215842002246,0.61283057248268479888508863950847,-0.6640334885500396300272996086278,-0.37554305781363517269255680730566,0.61283057248268502092969356453978,0.27505167682734588119330965128029,-0.16778761071338740840097614182014,-0.43999996868279378503530097077601,-0.64034357918688944444340904738056,-0.77800589862411184682855491701048,-0.83951158001816683640328164983657,0.1965860955839799917299615117372,0.50095199450696692444751079165144,0.71652901879469466095429197594058,0.77800589862411184682855491701048,-0.8188400187389239803792406746652,-0.71801469123163208507776289479807,-0.5469050830440682631206072983332,-0.32226059481637464587677754934703,0.77800589862411184682855491701048,0.64034357918688966648801397241186,0.43999996868279395156875466454949,0.060519452777067217486006711624213,-0.06607099424646192820098633546877,-0.55400406805393431763917533316999,0.050978900626839240839416333983536,0.83951158001816716947018903738353,-0.64774834858972396567367013631156,-0.49407477612085498597238597540127,0.63179863449392670293747187315603,0.42197972345950890415977596603625,-0.13434633655936117913043403859774,0.40017632013019388548258348237141,0.15285375279687535288175581627002,-0.15285375279687540839290704752784,-0.37074141186115194823713636651519,0.14812943013199278419200766165886,-0.4582615871346303326738791383832,-0.37074141186115194823713636651519,0.47935690537543407252485394565156,0.38780788281971001474346394388704,-0.14064029342468886696337904140819,-0.11824817247245911211450675182277,0.56601253363877979118967687099939,0.036135156012640279321601610718062,-0.32643697968833562672585912878276,0.34133753547532807992581638245611;
  V.col(2) << 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0;
  F.col(0) << 5,71,78,60,33,15,57,33,59,72,76,68,69,41,87,61,62,61,52,49,10,75,63,30,68,67,26,64,1,70,31,70,59,60,3,26,0,53,87,66,66,13,66,88,52,71,89,82,62,44,40,79,62,89,55,21,54,68,68,21,95,80,10,65,24,69,2,47,67,55,71,25,57,78,56,21,32,50,37,56,94,14,63,75,61,72,67,72,72,36,62,57,12,61,4,28,69,63,96,94,77,29,51,6,86,88,81,18,54,7,80,8,2,81,1,82,96,9,3,98,85,82,86,97,6,89,98,52,87,90,89,93,81,91,82,81,93,93,74,70,53,95,57,93,58,76,53,52;
  F.col(1) << 86,78,18,34,59,73,12,34,60,82,50,45,55,42,97,76,39,38,88,94,74,32,27,31,43,42,27,27,85,47,32,69,34,35,84,64,85,66,6,23,22,73,21,77,27,72,87,72,51,45,67,51,61,79,83,65,67,42,69,66,66,54,11,66,64,70,92,69,54,70,17,64,73,19,72,78,59,58,61,71,70,73,75,29,51,16,40,15,73,61,67,74,74,37,98,29,45,77,83,49,75,75,79,87,77,5,78,78,80,89,93,90,91,91,92,92,57,93,91,53,0,85,5,58,86,76,4,77,89,79,90,80,65,65,56,92,9,83,83,94,84,84,96,96,97,97,98,98;
  F.col(2) << 6,81,19,35,32,14,73,59,50,73,60,69,80,67,76,60,40,39,98,48,94,58,28,75,44,68,64,52,92,48,75,47,60,36,4,25,1,24,97,24,23,12,22,86,63,17,76,56,61,68,62,62,39,90,93,78,68,43,80,65,53,68,74,95,53,55,91,46,79,83,18,24,82,20,71,20,58,59,36,81,48,13,77,30,76,17,41,16,15,60,79,12,11,38,88,63,46,52,74,10,58,63,89,7,58,4,65,71,90,8,90,9,3,92,2,85,85,0,95,64,93,57,88,50,97,51,84,88,7,54,8,55,91,95,92,56,90,96,94,83,95,3,74,85,86,50,84,64;

  Eigen::SparseMatrix<double> Qs;
  igl::curved_hessian_energy(V, F, Qs);
  Eigen::MatrixXd Q(Qs);

  std::vector<double> initvec = {0x1.b0a2fea47278ap+7,-0x1.f4ee1c8888958p+5,0x1.588c1ce03556cp-4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.378ec9a5c0038p-5,-0x1.e7a96c6cc3aabp+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.825fb0f25862p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.930db3415cb9p+3,0x0p+0,0x1.4aeb3b082c13dp+3,0x1.908f01caeb6f9p+4,0x0p+0,-0x1.31fe29bc515a8p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0df97df1c2c6bp+2,0x0p+0,0x1.929b5aa49bf18p+2,-0x1.8dd46ca643163p+6,0x0p+0,0x0p+0,0x1.2924877dc5d46p+4,0x0p+0,0x0p+0,-0x1.f4ee1c8888958p+5,0x1.790b14b09d8fep+7,-0x1.deff867aeef18p+5,0x1.24f2b29dd791ap-5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.33eb5043c33a2p-7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b65b9cc416174p+1,0x1.60bdaa1c72424p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3b46a3c04effcp+3,0x1.e7c11b1f30495p+3,0x0p+0,0x0p+0,-0x1.1c9e37fd229fep+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.16363fb0b2c27p+3,-0x1.1266ae5f23fe3p+6,0x1.1ae010713de8bp+3,0x0p+0,0x0p+0,0x1.09a3f30851ca1p+4,0x0p+0,0x0p+0,0x1.588c1ce03556cp-4,-0x1.deff867aeef18p+5,0x1.7b7b64cd3d3e9p+7,-0x1.04c06ccd93994p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5ced6a70385c3p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.652c6ae70b496p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.f4b61ea1b978cp+3,0x1.047e895db06f2p+3,0x0p+0,0x1.b5c0342b93ffbp+3,0x1.867053031adfcp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.04b094aac0a65p+6,-0x1.19be37ebf4616p+6,0x0p+0,0x0p+0,0x1.dee59750b101fp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.24f2b29dd791ap-5,-0x1.04c06ccd93994p+6,0x1.a6fd7502f8919p+7,-0x1.1db7ac6c9fad4p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4de2925585765p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c3173b753d4bap+2,0x1.25d65619dbf6p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.10aada8eb6d08p+4,0x0p+0,0x0p+0,-0x1.78168da8f79e3p+6,0x0p+0,0x0p+0,0x0p+0,0x1.d4d8fe3a90569p+3,0x0p+0,0x0p+0,-0x1.8e456772f54c1p+6,0x1.27a70fc5e21e8p+4,0x0p+0,0x0p+0,0x1.bbe1d89d61de7p+4,0x0p+0,0x0p+0,0x1.4fbff30cdcc37p+4,0x0p+0,0x0p+0,0x0p+0,-0x1.1db7ac6c9fad4p+6,0x1.9ac564c1ff417p+7,-0x1.0a2e257cf90a7p+6,0x1.588c1ce03559p-4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.ddd12e20f357fp+2,0x1.51d8f0ef03965p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dc761ca57e9f6p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.da46fc0511712p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.316132eb599dbp+6,0x0p+0,0x1.2a1d9c60f5e76p+4,0x0p+0,-0x1.715d8d700ce4bp+6,0x0p+0,0x0p+0,0x1.e17e5c670322p+3,0x0p+0,0x0p+0,0x0p+0,0x1.c44eaaad31d5bp+3,0x0p+0,0x0p+0,0x1.25deea0c2c1f9p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.0a2e257cf90a7p+6,0x1.8636da9cfb436p+7,-0x1.2b9db41be397bp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b7c649f8919f1p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.816d906b3141ap+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e7c11b1f30495p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d14f8f0d201abp+3,0x0p+0,-0x1.09988f8e7a76bp+6,0x1.f1769d4365095p+3,-0x1.0fe1b7867330ep+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.84626b7a6f86ep+3,0x1.f0f50b9646f54p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.588c1ce03559p-4,-0x1.2b9db41be397bp+6,0x1.a6134ecd7f649p+7,-0x1.0b6ddb2e2382dp+6,0x1.b7ef91f10e6b7p-6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.11f7d343de10fp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d340f69c7c888p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.61e01c4c8f35cp+4,0x1.4f3fd0814ee9dp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.badb613df8c03p+5,-0x1.6ed42b640a49dp+6,0x1.a98e7f012bdd9p+3,0x1.675d45cb97cf9p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.e32fd7fb044bap+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.0b6ddb2e2382dp+6,0x1.a9dee1afb331fp+7,-0x1.d83216acaa962p+5,0x1.b7ef91f10e7f4p-6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5d80e44cdb6f5p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.791e25fa6443bp+2,0x0p+0,0x0p+0,0x1.93b824faa130dp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c6ea49b3c511dp+3,-0x1.3d14721db598p+6,0x0p+0,-0x1.6c1696b6c3e54p+6,0x1.0d4b80ceb0299p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.79fb1c20b72b2p+3,0x0p+0,0x1.378ec9a5c0038p-5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b7ef91f10e6b7p-6,-0x1.d83216acaa962p+5,0x1.a53d6b4d57b06p+7,-0x1.d774e24df47e3p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.00af507fef52fp-5,0x0p+0,0x0p+0,0x1.5d80e44cdb6f5p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.93b824faa12d5p+3,0x0p+0,0x0p+0,0x1.69ebb7104f8b9p+2,0x1.a5e47f5ac0c02p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a8a362e3b83cep+1,0x0p+0,-0x1.2a8d3e9571221p+6,-0x1.6e58589ef504cp+6,0x0p+0,0x0p+0,0x1.0d9048d272de5p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.e7a96c6cc3aabp+5,0x1.33eb5043c33a2p-7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b7ef91f10e7f4p-6,-0x1.d774e24df47e3p+5,0x1.a422aaf077e01p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.781c1a8013792p-5,0x1.612fff9ebf589p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.93b824faa130dp+3,0x1.696a65a1e5b6ep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.734834f8c538cp+3,0x0p+0,0x0p+0,0x0p+0,0x1.a0b2da8bffce7p+1,-0x1.2a67e1c99dd11p+6,0x0p+0,0x0p+0,-0x1.3fe59c45f93f1p+6,0x0p+0,0x0p+0,0x1.23d72b03b9af9p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b2324d713ea29p+7,-0x1.7422dc93828dfp+6,-0x1.3d8ca589ee8dep+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e114f27104693p+1,-0x1.a3e82cd487f6cp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dcbcbe6afa193p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4cce8b831e0b2p+4,0x0p+0,0x0p+0,0x0p+0,-0x1.ff7057c818d8bp+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.05c806747cc97p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.a7631566c5b74p+4,0x0p+0,0x1.67fa508fe64ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.7422dc93828dfp+6,0x1.9cdceef8350c5p+7,-0x1.eb0982fb7c851p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c3562529b2d94p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c12d99c06f133p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.40b05a9a155bfp+3,-0x1.b6e4fff4551d6p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e9dbe1e3484e5p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.34cfd20a8d5e2p+2,0x0p+0,0x1.af9de00cf4d9bp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.3d8ca589ee8c8p+1,-0x1.eb0982fb7c851p+6,0x1.0a4cf0b0fcdp+8,-0x1.48094738c48c6p+7,0x1.dce4dd08032ecp+1,0x1.4b76b2bb2736p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.858f69544e739p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d84d57687af6dp+0,0x1.b3853934ccc2ep+5,-0x1.12dcb21001dc5p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2a850a6d079e2p+3,0x1.247f6d9bc5dc1p+0,0x0p+0,0x1.c6196ecaf39b8p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.50bdbbb351494p+4,0x0p+0,0x1.1216ea4602118p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.48094738c48c6p+7,0x1.2a1144c7e0aadp+8,-0x1.dc1ccf185db9cp+5,-0x1.127a05423a787p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.012ac2f90596dp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,-0x1.61796938aa8ep+7,0x1.5ad619686e98dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3af00623e68e2p-1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dce4dd08032d5p+1,-0x1.dc1ccf185db9cp+5,0x1.2a4f57f3b0408p+8,-0x1.8dae04169b055p+7,0x1.433c5f1d749b3p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdbe16d5dfb5bp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2cdeff41f6699p+6,-0x1.5ef32254f3327p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.630f16efc0d13p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4b76b2bb2736p+3,-0x1.127a05423a787p+4,-0x1.8dae04169b055p+7,0x1.3faeb4118c294p+8,-0x1.1c0b7a6898f69p+6,-0x1.22bb9e2bb2965p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1dep+2,0x1.a2fb038a8e1d3p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,-0x1.5cb33ac05f743p+7,0x1.36a5e97d20774p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0df7bd7f6c5f7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749b3p+1,-0x1.1c0b7a6898f69p+6,0x1.1fc0d791c183bp+8,-0x1.934f94c2ef2f7p+7,0x1.433c5f1d749c9p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.bb1f34d9de3p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.336d5cd5c2477p+6,-0x1.1e3b8b4cbcdf6p+7,0x1.0f7a5b36a0a08p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.75b55d4c7005fp+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.22bb9e2bb2965p+3,-0x1.934f94c2ef2f7p+7,0x1.5c57d4e866144p+8,-0x1.2be41db626de6p+6,0x1.433c5f1d749b3p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.77d8c2977d902p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.a3022c439e497p+7,0x1.b98865d5af1adp+5,0x1.eadf03585e029p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6f14321576243p+4,0x0p+0,0x0p+0,0x1.241ea42f9723ap+4,0x1.0cc34e5ad024ep+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749c9p+1,-0x1.2be41db626de6p+6,0x1.39507f994ee1p+8,-0x1.8dae04169b055p+7,-0x1.127a05423a76bp+4,0x1.4b76b2bb27398p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0bca3567c6b7ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1b1p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.5302f6e283211p+7,0x1.6d5a90af0bfe7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3e9c52d225e74p+6,0x0p+0,0x0p+0,0x1.0df7bd7f6c5f7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749b3p+1,-0x1.8dae04169b055p+7,0x1.2a4f57f3b0408p+8,-0x1.dc1ccf185dbaap+5,0x1.dce4dd0803292p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdbe16d5dfb77p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2cdeff41f66ap+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.5ef32254f3327p+7,0x0p+0,0x0p+0,0x1.630f16efc0cf7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.127a05423a76bp+4,-0x1.dc1ccf185dbaap+5,0x1.2a1144c7e0aadp+8,-0x1.528f17d706182p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1636643588b02p+6,0x1.5856cfacb7014p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.61298001339bp+7,0x0p+0,0x0p+0,0x1.3af00623e68ebp-1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4b76b2bb27398p+3,0x1.dce4dd08032a8p+1,-0x1.528f17d706182p+7,0x1.318aa754d23fcp+8,-0x1.6af4238005728p+7,-0x1.b85cfcd83ee3bp+2,0x1.4b76b2bb27398p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.63a0d2b1dde83p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.c35eeeef9a96cp+6,0x1.5c8d23c5ba355p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d84d57687afc7p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c10d6128e9affp+5,0x0p+0,0x0p+0,0x1.051af7f5bfaebp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1ff2db8313e8cp+2,0x0p+0,0x0p+0,0x0p+0,0x1.e5b52cf64f397p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.6af4238005728p+7,0x1.373c5bc15f194p+8,-0x1.9c1d74963e05p+5,0x1.dce4dd0803302p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdbe16d5dfb5bp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.27219cb032b2ap+6,-0x1.867a798a6487bp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.336b8f69a2d65p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.697062be35b05p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.b85cfcd83ee3bp+2,-0x1.9c1d74963e05p+5,0x1.2a1144c7e0a9cp+8,-0x1.481077269a949p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.001abf41739e6p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5ad619686e98dp+4,0x1.8c5d7ffe807bcp+4,-0x1.757dc01e66e9p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.421ff3f9efec1p-1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4b76b2bb27398p+3,0x1.dce4dd0803319p+1,-0x1.481077269a949p+7,0x1.076f62167177bp+8,-0x1.ec8f79bc98a2bp+6,-0x1.2cb87b14adc33p+4,0x1.5a21f5dc89467p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2190d230b0af7p-3,-0x1.68e97107527f4p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0fe26bc4f422dp+4,0x1.b684e4859aa7cp+2,0x1.a0bc96c2c2636p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.712188a528fd3p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2451ee3da4063p+2,0x0p+0,0x0p+0,0x1.372a7c92eecb2p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ec8f79bc98a2bp+6,0x1.a44c1d5e8b456p+7,-0x1.cb89ebbba739ep+5,-0x1.2cb87b14adc33p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dcbcbe6afa193p+4,0x1.c65a816b90c96p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.489b5edd9a05dp+6,0x0p+0,0x1.40b05a9a155bfp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.953ae6e6b8987p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.2cb87b14adc33p+4,-0x1.cb89ebbba73acp+5,0x1.a44c1d5e8b456p+7,-0x1.eda23d16a9b5p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b35625caa86bep+4,0x1.dcbcbe6afa193p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.40b05a9a155bfp+3,-0x1.43da47f55feep+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b7935228dacd9p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5a21f5dc89467p+4,-0x1.2cb87b14adc33p+4,-0x1.eda23d16a9b5p+6,0x1.0b9845e4183e6p+8,-0x1.42a8d427d5075p+7,0x1.7971a39ac3286p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.83c8e5a8aca93p+6,0x1.2e617c411f013p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.71ec38d4e89c7p+5,0x1.712377bcf7b39p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.30792c993e62fp+3,0x0p+0,0x1.15f480a0e43cdp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.ffe8cf65490dap+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0861c84586371p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.42a8d427d5075p+7,0x1.1f82c465f1ecep+8,-0x1.1863f2dd60eabp+6,0x1.433c5f1d749b3p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.012ac2f905958p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.473d7782f5a76p+7,0x1.5ad619686e98dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0f7a5b36a0a08p+4,0x0p+0,0x1.0996b02f31a5dp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.7971a39ac324ep+3,-0x1.1863f2dd60eabp+6,0x1.3faeb4118c294p+8,-0x1.8dae04169b078p+7,-0x1.127a05423a76bp+4,0x1.4b76b2bb2736p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.333e6bdc419d8p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1bcp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.86760edcc6cfap+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.36a5e97d2079p+6,0x0p+0,0x1.0df7bd7f6c5f7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749b3p+1,-0x1.8dae04169b078p+7,0x1.2a4f57f3b041ap+8,-0x1.dc1ccf185db9cp+5,0x1.dce4dd08032ecp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdbe16d5dfb93p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2cdeff41f66b5p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.5ef32254f334ap+7,0x0p+0,0x1.630f16efc0cf7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.127a05423a76bp+4,-0x1.dc1ccf185db9cp+5,0x1.2a1144c7e0abfp+8,-0x1.528f17d7061a5p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1636643588b09p+6,0x1.5856cfacb703p+4,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.61298001339d4p+7,0x0p+0,0x1.3af00623e68fep-1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4b76b2bb2736p+3,0x1.dce4dd08032d5p+1,-0x1.528f17d7061a5p+7,0x1.31f4337f5cap+8,-0x1.659c346985074p+7,0x1.304f6f6517462p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.649f5224f4e39p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.bf90776d446cap+6,0x1.676f6e6afec93p+5,0x1.efe2797537973p+2,0x0p+0,0x0p+0,0x1.d84d57687afc7p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c1adf1d54ac56p+5,0x0p+0,0x1.19ab81c1e30f5p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d1496a2db72b9p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6f81647fc2cc7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.659c346985074p+7,0x1.2caddb5f705c6p+8,-0x1.fafd1c4d4ea44p+5,0x1.433c5f1d749c9p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2a6217f1fd2dep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2f53cf7a9a4f8p+6,-0x1.465d98f0aed24p+7,0x1.0dc0b9d0367adp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.336b8f69a2d5ap+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.304f6f651748fp+0,-0x1.fafd1c4d4ea53p+5,0x1.62b60960a35b7p+8,-0x1.934f94c2ef2f7p+7,-0x1.22bb9e2bb293ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.efa26e749c2aap+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.38f6eb8564c79p+5,-0x1.d000ca31d67f9p+7,0x1.bd350ed2e7c0ep+5,0x1.eadf03585e03fp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0cc34e5ad0232p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749c9p+1,-0x1.934f94c2ef2f7p+7,0x1.1fc0d791c183bp+8,-0x1.1c0b7a6898f69p+6,0x1.433c5f1d749b3p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.bb1f34d9de3p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.336d5cd5c2469p+6,-0x1.1e3b8b4cbcdf6p+7,0x1.0f7a5b36a0a08p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.75b55d4c7005fp+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.22bb9e2bb293ep+3,-0x1.1c0b7a6898f7p+6,0x1.3faeb4118c2a5p+8,-0x1.8dae04169b078p+7,-0x1.127a05423a7a3p+4,0x1.4b76b2bb27398p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1e9p+2,0x1.a2fb038a8e1bcp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,-0x1.5cb33ac05f743p+7,0x1.36a5e97d20782p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0df7bd7f6c5f7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749b3p+1,-0x1.8dae04169b078p+7,0x1.2a4f57f3b0408p+8,-0x1.e36bde2ecdd35p+5,-0x1.127a05423a76bp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2cdeff41f66ap+6,-0x1.35304e388bd7p+7,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.630f16efc0d13p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.127a05423a7a3p+4,-0x1.e36bde2ecdd35p+5,0x1.2a4f57f3b0408p+8,-0x1.98b007d3deb24p+7,0x1.433c5f1d749ep+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.bb1f34d9de2e3p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff47p+5,-0x1.38777d0271c3cp+7,0x1.336d5cd5c2462p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4b76b2bb27398p+3,-0x1.127a05423a76bp+4,-0x1.98b007d3deb24p+7,0x1.5c57d4e866133p+8,-0x1.2be41db626dedp+6,0x1.433c5f1d74a0dp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.77d8c2977d902p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fa1680ebdfae3p+5,-0x1.a3022c439e474p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6f14321576243p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0cc34e5ad0232p+4,0x0p+0,0x0p+0,0x1.241ea42f9723ap+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749ep+1,-0x1.2be41db626dedp+6,0x1.39507f994ee1p+8,-0x1.884d9105ab828p+7,-0x1.22bb9e2bb2938p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0bca3567c6b7ep+3,0x0p+0,0x0p+0,0x1.a2fb038a8e1a6p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6d5a90af0bfe7p+4,-0x1.5302f6e2831eep+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1e5545470d9afp+6,0x1.eadf03585e08ep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0df7bd7f6c5dbp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d74a0dp+1,-0x1.884d9105ab828p+7,0x1.1fc0d791c183bp+8,-0x1.1d7e3b67b72cfp+6,0x1.433c5f1d749ep+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d2659d13fd5bp+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2cdeff41f668bp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.1abd285609ac2p+7,0x1.0dc0b9d0367adp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.630f16efc0cdap+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.22bb9e2bb2938p+3,-0x1.1d7e3b67b72cfp+6,0x1.62b60960a35b7p+8,-0x1.98b007d3deb24p+7,-0x1.127a05423a76bp+4,0x1.4b76b2bb27398p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.77d8c2977d8e6p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ac7b3bf4ad53ep+7,0x1.ea29ae41d4cffp+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0cc34e5ad0232p+4,0x1.0cc34e5ad0215p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749ep+1,-0x1.98b007d3deb24p+7,0x1.2a4f57f3b0408p+8,-0x1.e36bde2ecdd35p+5,-0x1.127a05423a787p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.bb1f34d9de2e3p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.336d5cd5c2469p+6,-0x1.38777d0271c3cp+7,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.127a05423a76bp+4,-0x1.e36bde2ecdd27p+5,0x1.2a4f57f3b041ap+8,-0x1.8dae04169b078p+7,0x1.433c5f1d749b3p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,-0x1.35304e388bd7p+7,0x1.2cdeff41f66a7p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.630f16efc0cf7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4b76b2bb27398p+3,-0x1.127a05423a787p+4,-0x1.8dae04169b078p+7,0x1.3faeb4118c2a5p+8,-0x1.1c0b7a6898f7p+6,-0x1.22bb9e2bb294ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1dep+2,0x1.a2fb038a8e1bcp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.36a5e97d20789p+6,-0x1.5cb33ac05f743p+7,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0df7bd7f6c5f7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749b3p+1,-0x1.1c0b7a6898f7p+6,0x1.1fc0d791c183bp+8,-0x1.934f94c2ef2f7p+7,0x1.433c5f1d749ep+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.bb1f34d9de3p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0f7a5b36a0a08p+4,-0x1.1e3b8b4cbcdf6p+7,0x1.336d5cd5c2462p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.75b55d4c7005fp+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.22bb9e2bb294ep+3,-0x1.934f94c2ef2f7p+7,0x1.5c81dea78aafcp+8,-0x1.398861ac4544fp+6,0x1.dcbb59cf2955dp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6fcda221f7043p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.eadf03585e056p+2,0x1.b9150f9b3ca54p+5,-0x1.a5ed2a279ad7ep+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0cc34e5ad0232p+4,0x0p+0,0x0p+0,0x1.0e43b3039f82ep+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.8a6208a1d1cd8p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e114f27104693p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.433c5f1d749ep+1,-0x1.398861ac4544fp+6,0x1.cf3b6d9c81c8ap+7,-0x1.de109afb9a3c1p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a687edff66956p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.83e2126f967cdp+4,-0x1.ea23b4b843637p+6,0x0p+0,0x0p+0,0x0p+0,0x1.0fc1bb0bc577dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.8d4381994be0cp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.f6075245f1fap+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.a3e82cd487f6cp+6,0x1.c3562529b2d94p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dcbb59cf2955dp+2,-0x1.de109afb9a3c1p+6,0x1.bb6de33fd3686p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5c77147a66e86p+5,0x0p+0,0x0p+0,0x0p+0,0x1.068270418ffcdp+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0f6ee503aef53p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.c071f49094b9dp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.11f7d343de10fp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.649f5224f4e2ep+2,0x1.2a6217f1fd2e9p+2,0x1.efa26e749c2aap+4,0x1.bb1f34d9de3p+4,0x1.a2fb038a8e1e9p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e2985cce130cdp+8,0x1.78ed4e4320c1bp+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.31601d6463ef3p+3,-0x1.8ecaa192b2fbcp+7,-0x1.3da7fb9d19081p+7,0x1.70883c618dccbp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.20346851c287ap+2,-0x1.a97787918e255p+5,0x1.347bccb87e1fap+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.16fba18fd70eep+2,0x1.7295c2ca5de59p+3,0x0p+0,0x1.56461e65049a7p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.c3246fd6f6608p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5d80e44cdb6f5p+4,0x1.00af507fef52fp-5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1bcp+2,0x0p+0,0x1.bb1f34d9de2e3p+4,0x1.77d8c2977d902p+4,0x1.0bca3567c6b7ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.78ed4e4320c1bp+0,0x1.9da69a507a44bp+8,0x0p+0,0x0p+0,0x1.a54f1fa182673p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d441972b9fed9p+2,-0x1.28679c3b5d6fp+7,-0x1.4b4cab6bef0bep+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3a927d756840dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.c6b4de12dafa1p+6,0x0p+0,0x0p+0,-0x1.3b360383b0d66p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.ed7c67d73257fp+3,0x0p+0,-0x1.9b6862dc4a1bep+5,0x1.edf4107447fe4p-7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6a598b37e98dcp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.ddd12e20f358bp+2,0x1.b7c649f8919f1p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2190d230b0af7p-3,0x1.dcbcbe6afa193p+4,0x1.b35625caa86bep+4,-0x1.83c8e5a8aca93p+6,0x1.012ac2f905958p+6,0x1.333e6bdc419d8p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.ad125ca15d4p+8,0x1.1556f2c854af6p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.12b1ffe62551cp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ec8dc2da52d8dp+7,-0x1.c78be3b6b0af2p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1f706f9370f46p+4,0x0p+0,-0x1.04690f6bfbf8bp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.eac55195a6054p+3,0x0p+0,0x1.265e30716d912p+3,0x0p+0,-0x1.1179ddd81d415p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.82eedf26dceabp+6,0x0p+0,0x0p+0,0x0p+0,0x1.4de2925585765p+3,0x1.51d8f0ef03965p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.63a0d2b1dde83p+3,0x1.fdbe16d5dfb5bp+4,0x1.001abf41739e6p+6,-0x1.68e97107527f4p+6,0x1.c65a816b90c96p+4,0x1.dcbcbe6afa193p+4,0x1.2e617c411f013p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1556f2c854af6p+3,0x1.bca8c843778c2p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.3eae4b8ae429cp+7,0x1.879612b8e0fcap+4,-0x1.0258ab5d3c282p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.123abd9eeb4bcp+6,0x0p+0,0x0p+0,0x0p+0,0x1.e1f3b43af054fp+2,0x0p+0,0x0p+0,0x1.2e7eeaed1249ap+3,0x0p+0,0x0p+0,0x0p+0,-0x1.9f05c6ed79e51p+6,0x0p+0,0x0p+0,-0x1.7b15f0fd0b439p-3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5d80e44cdb6f5p+4,0x1.781c1a8013792p-5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1a6p+2,0x1.d2659d13fd5bp+0,0x1.77d8c2977d8e6p+4,0x1.bb1f34d9de2e3p+4,0x0p+0,0x1.a2fb038a8e1dep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a54f1fa182673p+2,0x0p+0,0x0p+0,0x1.9db2ba44041fp+8,0x1.afac89295d2d7p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d441972b9fee5p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.29da6cebdba4cp+7,-0x1.420d8fe93b16ep+7,0x1.f860151e0af5p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.d7ff9add37003p+6,-0x1.3c47506560f6p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3bb55dfc4133dp+4,-0x1.95b29f6d49219p+5,0x0p+0,0x0p+0,-0x1.784f9d7ead45p-4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.612fff9ebf589p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1bcp+2,0x1.bb1f34d9de3p+4,0x1.6fcda221f7043p+4,0x1.a687edff66956p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.afac89295d2d7p+2,0x1.a9d2c1af3149p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.697da7c1e95dep+3,-0x1.2b41c7a6f2ee1p+7,-0x1.793fdb13c884fp+7,0x0p+0,0x0p+0,0x0p+0,0x1.9e7e220f9e74ep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.d866e76ffa993p+6,0x0p+0,0x0p+0,-0x1.2cbbfab195999p+6,0x0p+0,0x1.60a4604bea137p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4b36f0d8771f5p+4,0x0p+0,0x0p+0,-0x1.0a39dd05559p+6,0x1.90181e24a5dap+4,0x0p+0,-0x1.7b654d0d2f613p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b65b9cc416174p+1,0x1.5ced6a70385c3p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1dep+2,0x1.bb1f34d9de3p+4,0x1.77d8c2977d902p+4,0x1.0bca3567c6b83p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.7472ba12dda52p+8,0x1.12b1ffe625505p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.38bfd2ca421fdp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.33a804c5344a4p+7,-0x1.3e7008c984447p+7,0x1.01a92acbb1723p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.9f94457b7b81cp+3,0x0p+0,0x0p+0,-0x1.1089d8aa3585cp+6,-0x1.a3365fa416109p+5,0x0p+0,0x0p+0,0x1.0c44f252677fbp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.7d8256eda5c73p+3,-0x1.251e8ec7c5fe3p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.825fb0f25862p+2,0x1.60bdaa1c72424p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dcbcbe6afa193p+4,0x1.c12d99c06f133p+4,-0x1.858f69544e739p+6,0x1.012ac2f905966p+6,0x1.fdbe16d5dfb5bp+4,0x1.a2fb038a8e1d3p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.12b1ffe625505p+1,0x1.b19d39635c679p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.53b7e91f2d15ep+3,-0x1.db5842917e83ap+7,-0x1.260052069bb03p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.19922aa5673b3p+6,0x1.c6ae5292763aep+2,0x0p+0,-0x1.6f5142de8a35cp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.9e514c8d468c8p+3,0x1.6c68fac61e9a8p+2,0x1.a25ae6da149cap+3,0x0p+0,-0x1.289ab4bcf5458p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.816d906b3141ap+3,0x1.d340f69c7c888p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1bcp+2,0x1.fdbe16d5dfb77p+4,0x1.1636643588b09p+6,-0x1.bf90776d446cap+6,0x1.2f53cf7a9a4f8p+6,0x1.38f6eb8564c79p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.31601d6463ef3p+3,0x0p+0,0x1.12b1ffe62551cp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a44ff0f9f2a8bp+8,-0x1.68b9ebad4e7b2p+7,0x1.000f563bf3408p+2,0x0p+0,0x0p+0,0x1.53b7e91f2d196p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.d5f38c4963698p+7,0x1.d3360071445f7p+2,-0x1.040795cd1993bp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.baae9532ab9dbp+6,0x1.c7310a9bb2b58p+2,0x1.d2b5d169804ffp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.dedc1bb71aafap+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5856cfacb703p+4,0x1.676f6e6afec93p+5,-0x1.465d98f0aed24p+7,-0x1.d000ca31d67f9p+7,0x1.336d5cd5c2469p+6,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.8ecaa192b2fbcp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.68b9ebad4e7b2p+7,0x1.2b5504be01f2ep+9,-0x1.e0bd143eb4711p+5,0x1.24b1bd3d57995p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d710c826e26c4p+3,-0x1.4f5ac5e446314p-2,0x1.13b0c50438682p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e3b439dd2b5e9p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.673c77a74b78fp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.efe2797537973p+2,0x1.0dc0b9d0367adp+4,0x1.bd350ed2e7c0ep+5,-0x1.1e3b8b4cbcdf6p+7,-0x1.5cb33ac05f743p+7,0x1.2cdeff41f66ap+6,0x1.062e1a815ff47p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.3da7fb9d19081p+7,0x1.d441972b9fecep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.000f563bf3408p+2,-0x1.e0bd143eb4703p+5,0x1.f4cb46c0e69bep+8,-0x1.c8e9b9c77f2dcp+6,0x1.24b1bd3d57995p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.8a44d2fbbaeb7p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdfdf365ea8d8p+3,0x0p+0,0x1.05c4fded33fe1p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.66ae694485f35p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.eadf03585e03fp+2,0x1.0f7a5b36a0a08p+4,0x1.36a5e97d20782p+6,-0x1.35304e388bd7p+7,-0x1.38777d0271c3cp+7,0x1.fa1680ebdfae3p+5,0x1.6d5a90af0bfe7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.70883c618dccbp+3,-0x1.28679c3b5d6fp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.24b1bd3d57995p+1,-0x1.c8e9b9c77f2dcp+6,0x1.e13a27862864p+8,-0x1.07afc7342f9ecp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdd14ca6b2d81p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.6d4d8caf3b528p+6,0x0p+0,0x0p+0,0x1.55db00cc7254dp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.38c59e0aac1e8p+2,0x0p+0,0x1.1b43f0fb8bd9dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.65d1c781065cdp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,0x1.336d5cd5c2462p+6,-0x1.a3022c439e474p+7,-0x1.5302f6e2831eep+7,0x1.2cdeff41f668bp+6,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.4b4cab6bef0bep+7,0x0p+0,0x0p+0,0x1.d441972b9fefp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.24b1bd3d57995p+1,-0x1.07afc7342f9f3p+6,0x1.19cb7300b57b2p+9,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.dd9ff563e4d17p+6,0x1.24b1bd3d5797ep+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.337b12e5bd3ffp+3,0x0p+0,0x0p+0,-0x1.8e86ab8b3f911p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2b81115181745p+4,0x1.0628d58847693p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.40b05a9a155bfp+3,0x1.71ec38d4e89c7p+5,-0x1.473d7782f5a76p+7,-0x1.86760edcc6cfap+7,0x1.2cdeff41f66b5p+6,0x1.062e1a815ff38p+5,0x1.d84d57687afc7p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ec8dc2da52d8dp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.53b7e91f2d1cep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.21878f5a95f9ep+9,0x1.103e7b2b3201dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.daa7b4ac0ae2p+6,0x0p+0,-0x1.5a48fbd637b2ap+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.75a147b07db7ep+3,0x0p+0,0x1.1a00a34d15a6dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.08c3865d569fp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dc761ca57e9f6p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5ad619686e98dp+4,0x1.0fe26bc4f422dp+4,-0x1.489b5edd9a05dp+6,-0x1.43da47f55feep+6,0x1.712377bcf7b45p+2,0x1.5ad619686e98dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.c78be3b6b0af2p+6,-0x1.3eae4b8ae429cp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.103e7b2b3201dp+4,0x1.848e3d76552ecp+8,0x0p+0,0x1.176b6fb3642e3p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.49a56eb90f8cfp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3edc33049d393p+4,0x0p+0,0x0p+0,0x0p+0,0x1.586940a1c6b95p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.11d6a33035964p+4,0x0p+0,0x0p+0,-0x1.d9711b55b284bp+6,0x0p+0,0x0p+0,0x1.652c6ae70b496p+1,0x1.c3173b753d4bap+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a2fb038a8e1b1p+2,0x1.fdbe16d5dfb77p+4,0x1.1636643588b02p+6,-0x1.c35eeeef9a96cp+6,0x1.27219cb032b2ap+6,0x1.8c5d7ffe807bcp+4,0x1.b684e4859aa7cp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.879612b8e0fcap+4,0x0p+0,0x0p+0,0x1.38bfd2ca421fdp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b6e458bfaf702p+8,-0x1.2093c41199c36p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c21bf95a1c632p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.e648d530a4d18p+7,0x0p+0,0x0p+0,-0x1.ffef8f16f6876p+5,0x0p+0,0x0p+0,0x1.7eb20becbe981p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ef6bfac8a5702p+4,0x1.2709c1a54f935p+3,0x0p+0,0x0p+0,-0x1.19fa7a8b857fep+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.25d65619dbf6p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5856cfacb7014p+4,0x1.5c8d23c5ba355p+5,-0x1.867a798a6487bp+7,-0x1.757dc01e66e9p+7,0x1.a0bc96c2c2636p+5,0x1.40b05a9a155bfp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.0258ab5d3c282p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.176b6fb3642e3p+4,-0x1.2093c41199c36p+7,0x1.2e443b8af6ee9p+9,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1f828803d39d5p+4,0x0p+0,0x0p+0,0x1.fe30e268cac82p+2,0x0p+0,0x0p+0,0x1.0ec4661043521p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1b1598abded16p+3,0x0p+0,0x0p+0,0x0p+0,-0x1.1ea42b0ddb3e8p+4,0x0p+0,0x0p+0,0x1.f28bc361de6a9p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6f14321576243p+4,0x1.1e5545470d9b6p+6,-0x1.1abd285609ac2p+7,-0x1.ac7b3bf4ad53ep+7,0x1.336d5cd5c2469p+6,0x1.062e1a815ff38p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3a927d756840dp+4,0x0p+0,0x0p+0,-0x1.29da6cebdba4cp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdd14ca6b2d81p+2,-0x1.dd9ff563e4d17p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.097dc42abbec3p+9,-0x1.df92a2f8f47d7p+5,0x1.24b1bd3d57995p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.b5e5afbf38fb2p+6,-0x1.2a6db320e2e2fp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e8b2c5abde75ep+3,0x1.1b43f0fb8bd9dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.eadf03585e08ep+2,0x1.0dc0b9d0367adp+4,0x1.ea29ae41d4cffp+5,-0x1.38777d0271c3cp+7,-0x1.35304e388bd7p+7,0x1.36a5e97d20789p+6,0x1.0f7a5b36a0a08p+4,0x1.eadf03585e056p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.420d8fe93b16ep+7,0x1.697da7c1e95dep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.24b1bd3d5797ep+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.df92a2f8f47d7p+5,0x1.e506e5939336cp+8,-0x1.c58d46fa9c079p+6,0x1.24b1bd3d57995p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.39cfb66e33ec5p+4,-0x1.69276278b170ep+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2b80348b7603p+4,0x0p+0,0x0p+0,0x1.04c748a515b99p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,0x1.2cdeff41f66a7p+6,-0x1.5cb33ac05f743p+7,-0x1.1e3b8b4cbcdf6p+7,0x1.b9150f9b3ca54p+5,0x1.83e2126f967cdp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.f860151e0af5p+3,-0x1.2b41c7a6f2ee1p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.24b1bd3d57995p+1,-0x1.c58d46fa9c079p+6,0x1.efb6b002a81dep+8,-0x1.1020b3028fcf5p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.8b5943110649fp+6,0x0p+0,0x0p+0,0x1.47b5c5d25efafp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e31067ad1615fp+3,0x0p+0,0x0p+0,0x1.147c2143acba2p+4,0x1.1733ea712ebf3p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4cce8b831e0b2p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,0x1.336d5cd5c2462p+6,-0x1.a5ed2a279ad7ep+7,-0x1.ea23b4b843637p+6,0x1.5c77147a66e86p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.793fdb13c884fp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.24b1bd3d57995p+1,-0x1.1020b3028fcf5p+6,0x1.1b54000358c81p+9,0x0p+0,0x0p+0,0x0p+0,0x1.c9302e9683ee9p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.48a1a00b30e9dp+3,0x0p+0,0x0p+0,-0x1.9ca44c8792356p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.15fd5af8819eep+4,-0x1.27612238327b6p+7,0x0p+0,0x1.7699501dd8129p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.062e1a815ff38p+5,0x1.336d5cd5c2477p+6,-0x1.a3022c439e497p+7,-0x1.5302f6e283211p+7,0x1.2cdeff41f66ap+6,0x1.062e1a815ff38p+5,0x1.d84d57687afc7p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.33a804c5344a4p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c21bf95a1c632p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.18643e5468c83p+9,-0x1.08c7ed5cd1d5dp+6,0x1.24b1bd3d57995p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.e9cf1a2528d6cp+6,0x0p+0,0x0p+0,-0x1.b117976646999p+6,-0x1.576d872855c0dp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dbb9a12366905p+3,0x1.4f99ec924dc25p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d84d57687af6dp+0,0x1.062e1a815ff38p+5,0x1.2cdeff41f6699p+6,-0x1.5cb33ac05f743p+7,-0x1.1e3b8b4cbcdf6p+7,0x1.b98865d5af1adp+5,0x1.6d5a90af0bfe7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.3e7008c984447p+7,0x1.53b7e91f2d15ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.08c7ed5cd1d5dp+6,0x1.f50049c0ee09dp+8,-0x1.d335e37a1afd8p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fdd14ca6b2d54p+2,0x0p+0,0x0p+0,0x1.93c7cea89b8b1p+3,-0x1.7ca038520b7a6p+6,0x0p+0,0x0p+0,0x1.75a147b07db46p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.719cc888aa1aap+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.40b05a9a155bfp+3,0x1.b3853934ccc3cp+5,-0x1.61796938aa8ep+7,-0x1.5ef32254f3327p+7,0x1.36a5e97d20774p+6,0x1.0f7a5b36a0a08p+4,0x1.eadf03585e029p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.01a92acbb1723p+4,-0x1.db5842917e83ap+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.24b1bd3d57995p+1,-0x1.d335e37a1afd8p+6,0x1.1866a8fa9ddep+9,0x1.0a8e61c97e157p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.6e45f902d3a89p+6,0x0p+0,0x0p+0,0x1.1034e04a36c94p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.00604309308cbp+4,0x0p+0,0x0p+0,0x0p+0,0x1.4d8bd28607cf7p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ff7057c818d8bp+5,-0x1.b6e4fff4551d6p+5,-0x1.12dcb21001dbap+2,0x1.5ad619686e98dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0fc1bb0bc577dp+4,0x1.068270418ffcdp+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.9e7e220f9e74ep+2,0x0p+0,-0x1.260052069bb03p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c9302e9683ee9p+2,0x0p+0,0x0p+0,0x1.0a8e61c97e157p+4,0x1.b5d614dbe0984p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.fa91b83cf08b1p+3,0x1.741298d51f09fp+4,0x0p+0,0x1.ee7facfc6191dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.8987c340b9319p+3,-0x1.6a185ecb2c6ebp+7,0x0p+0,-0x1.557df335b38ddp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.30792c993e62fp+3,0x1.0f7a5b36a0a08p+4,0x1.36a5e97d20797p+6,-0x1.5ef32254f334ap+7,-0x1.61298001339d4p+7,0x1.c1adf1d54ac64p+5,0x1.336b8f69a2d5ap+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.20346851c286ep+2,0x0p+0,0x1.1f706f9370f46p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.d5f38c4963698p+7,0x1.d710c826e26c4p+3,0x0p+0,0x0p+0,0x0p+0,-0x1.daa7b4ac0ae2p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.18435d8ac1178p+9,0x0p+0,-0x1.6c567e16b1686p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0b09af4c7d498p+4,0x0p+0,0x1.00604309308afp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3194fa15f71b7p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.61e01c4c8f35cp+4,0x1.791e25fa6443bp+2,0x1.93b824faa12d5p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0cc34e5ad0232p+4,0x1.75b55d4c7005fp+0,0x1.0df7bd7f6c5f7p+4,0x1.630f16efc0d13p+4,0x0p+0,0x1.0cc34e5ad0232p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.a97787918e255p+5,-0x1.c6b4de12dafa1p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d3360071445f7p+2,-0x1.4f5ac5e4462efp-2,-0x1.8a44d2fbbaeb7p+6,-0x1.6d4d8caf3b528p+6,0x1.337b12e5bd3ffp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.920435520ddf4p+8,0x0p+0,0x0p+0,0x1.0ac12b76c6a4bp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1227f43659a12p+3,-0x1.4d0cefca95161p+7,0x0p+0,-0x1.3e8f3fb0f6bc5p+4,0x1.6cffed1c8ecep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.518fcf93a07d5p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.da46fc0511712p+3,0x1.e7c11b1f30495p+3,0x1.4f3fd0814ee9dp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.15f480a0e43cdp+3,0x1.0996b02f31a5dp+1,0x1.0df7bd7f6c5f7p+4,0x1.630f16efc0cf7p+4,0x1.3af00623e6907p-1,0x1.19ab81c1e30f5p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.347bccb87e1fap+4,0x0p+0,-0x1.04690f6bfbf8bp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.040795cd1993bp+6,0x1.13b0c50438682p+1,0x0p+0,0x0p+0,0x0p+0,-0x1.5a48fbd637b23p+6,0x1.49a56eb90f8cfp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.6c567e16b1686p+6,0x0p+0,0x1.6a30d8f15c5afp+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.caaa8b90d63dbp+5,0x0p+0,-0x1.1a1bfcae926a7p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.50f3e722949a1p+3,0x1.6f982149d385fp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6f14321576243p+4,0x1.3e9c52d225e74p+6,-0x1.5ef32254f3327p+7,-0x1.61298001339bp+7,0x1.c10d6128e9b0dp+5,0x1.336b8f69a2d65p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.9f94457b7b81cp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.e648d530a4d18p+7,0x1.1f828803d39d5p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.e9cf1a2528d6cp+6,0x1.fdd14ca6b2d54p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.19463f28ef3dep+9,0x0p+0,0x0p+0,-0x1.5ec19c30b0bdp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1bea199c721dap+4,0x1.6ae8d25ce7f9p+3,0x0p+0,0x0p+0,0x1.6818185edb00ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.93b824faa130dp+3,0x1.69ebb7104f8cfp+2,0x1.93b824faa130dp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.241ea42f9723ap+4,0x1.0df7bd7f6c5dbp+4,0x1.630f16efc0cdap+4,0x1.0cc34e5ad0232p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.3b360383b0d66p+6,0x0p+0,0x0p+0,-0x1.d7ff9add37003p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.55db00cc7254dp+2,-0x1.8e86ab8b3f911p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.b5e5afbf38fb2p+6,0x1.39cfb66e33ec5p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0ac12b76c6a4bp+4,0x0p+0,0x0p+0,0x1.af6a3e60ae7f3p+8,0x1.0c54b7d30838dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d5c8372b58c7ap+3,0x0p+0,-0x1.8d2aed2ebee58p+7,-0x1.57295300e0aadp+3,0x0p+0,0x0p+0,0x1.6dcf117faf1dbp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.930db3415cb9p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a5e47f5ac0c02p+3,0x1.696a65a1e5b7ap+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0cc34e5ad0215p+4,0x0p+0,0x1.630f16efc0cf7p+4,0x1.0df7bd7f6c5f7p+4,0x1.75b55d4c7005fp+0,0x1.0cc34e5ad0232p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.3c47506560f6p+6,-0x1.d866e76ffa993p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.2a6db320e2e2fp+2,-0x1.69276278b170ep+6,-0x1.8b5943110649fp+6,0x1.48a1a00b30e9dp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0c54b7d30838dp+4,0x1.af39e10a9b275p+8,0x0p+0,0x0p+0,0x1.598d439a51eb9p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d5c8372b58c09p+3,-0x1.90dd7cafe67dcp+7,0x0p+0,0x0p+0,-0x1.9affc4c012efdp+1,0x0p+0,0x0p+0,0x1.53dfc319dc20cp+2,0x0p+0,0x0p+0,0x0p+0,0x1.3b46a3c04f002p+3,0x1.f4b61ea1b978cp+3,0x1.10aada8eb6d08p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.241ea42f9723ap+4,0x1.0df7bd7f6c5f7p+4,0x1.630f16efc0cf7p+4,0x1.3af00623e68ebp-1,0x1.051af7f5bfaebp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.1089d8aa3585cp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ffef8f16f6876p+5,0x1.fe30e268cac82p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.b117976646999p+6,0x1.93c7cea89b8e9p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.5ec19c30b0bdp+6,0x0p+0,0x0p+0,0x1.6d0062c540cc3p+8,0x1.9d89ea8c3ad18p+3,0x0p+0,0x0p+0,0x1.c057012998fe3p+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.18e774793de29p+7,-0x1.084c758ef09ecp+6,0x0p+0,0x0p+0,0x1.7f5ef6fff5e34p+4,0x0p+0,0x0p+0,0x0p+0,0x1.4aeb3b082c13dp+3,0x1.e7c11b1f30495p+3,0x1.047e895db06f2p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2a850a6d079e2p+3,0x1.3af00623e68e2p-1,0x1.630f16efc0d13p+4,0x1.0df7bd7f6c5f7p+4,0x1.75b55d4c7005fp+0,0x1.0cc34e5ad024ep+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.a3365fa416109p+5,-0x1.19922aa5673b3p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.576d872855c24p+1,-0x1.7ca038520b7a6p+6,-0x1.6e45f902d3a89p+6,0x1.fa91b83cf08b1p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.9d89ea8c3ad18p+3,0x1.675c97e3a0867p+8,0x0p+0,0x0p+0,-0x1.02d2bb7c2fd29p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.a89200b841d89p+2,-0x1.f5eadf59e80e4p+6,0x1.a020da6688b39p-1,0x0p+0,0x0p+0,0x1.75821c177cffep+2,0x0p+0,0x0p+0,0x1.908f01caeb6f9p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.05c806747cc97p+3,0x1.e9dbe1e3484e5p+1,0x1.247f6d9bc5dc1p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0e43b3039f82ep+4,0x1.8d4381994be0cp+3,0x1.0f6ee503aef53p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.2cbbfab195999p+6,0x0p+0,0x1.c6ae5292763aep+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.47b5c5d25efafp+2,-0x1.9ca44c8792356p+5,0x0p+0,0x0p+0,0x0p+0,0x1.741298d51f09fp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.598d439a51eb9p+3,0x0p+0,0x0p+0,0x1.b774ab31f9b9bp+8,0x0p+0,0x1.c8b963bb56613p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d05c28417c38fp+3,0x0p+0,0x0p+0,-0x1.25b2cd1d441dbp+7,-0x1.56dbf2a84be09p+7,0x0p+0,-0x1.33dc8118d9621p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b5c0342b93ffbp+3,-0x1.78168da8f79eap+6,-0x1.316132eb599dbp+6,0x1.d14f8f0d201abp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.712188a528fd3p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.eac55195a6054p+3,-0x1.123abd9eeb4c3p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3edc33049d393p+4,0x1.7eb20becbe981p+4,0x1.0ec4661043526p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.aa8ff814fbf32p+8,0x0p+0,0x0p+0,0x0p+0,0x1.b6d6de59c8195p+4,0x0p+0,0x0p+0,0x1.0062cadc4a81dp+5,0x0p+0,0x0p+0,0x0p+0,-0x1.5204c6944681dp+7,0x0p+0,0x0p+0,-0x1.7302f114c6a59p+7,-0x1.31fe29bc515a8p+6,-0x1.1c9e37fd229fep+6,0x1.867053031adfcp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.734834f8c538cp+3,0x0p+0,0x0p+0,0x1.c6196ecaf39b8p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.60a4604bea137p+3,0x1.0c44f252677f5p+3,-0x1.6f5142de8a35cp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.75a147b07db46p+3,0x1.1034e04a36c78p+4,0x1.ee7facfc6191dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c057012998fe3p+1,-0x1.02d2bb7c2fd29p+6,0x1.c8b963bb56613p+3,0x0p+0,0x1.7833d2c078e1dp+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3cc1d15e9b6f2p+2,0x1.adddc6373bdf7p+2,-0x1.7fd7e303ed878p+4,0x1.f9a346cf85383p+2,0x0p+0,0x0p+0,-0x1.94b7dd0410fc2p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2a1d9c60f5e76p+4,-0x1.09988f8e7a772p+6,-0x1.badb613df8c03p+5,0x1.c6ea49b3c511dp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d1496a2db72b9p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.16fba18fd70eep+2,0x0p+0,0x1.265e30716d90dp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.baae9532ab9dbp+6,0x1.e3b439dd2b5e9p+4,0x0p+0,0x0p+0,0x0p+0,0x1.75a147b07db7ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0b09af4c7d498p+4,0x1.1227f43659a12p+3,-0x1.caaa8b90d63dbp+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.62e64c06b35e4p+8,0x1.5c0056abd3da1p+4,-0x1.3f0e501c326c7p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.5b3aea946871ep+7,0x1.66fed59ec062p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.f1769d4365095p+3,-0x1.6ed42b640a49dp+6,-0x1.3d14721db598p+6,0x1.a8a362e3b83cep+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.7295c2ca5de59p+3,0x1.ed7c67d73257fp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.c7310a9bb2b58p+2,0x0p+0,0x1.fdfdf365ea8d8p+3,0x1.38c59e0aac1e8p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.4d0cefca95161p+7,0x0p+0,0x0p+0,0x1.d5c8372b58c7ap+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.5c0056abd3da1p+4,0x1.a8c253549c371p+8,0x0p+0,-0x1.bff0d3b5ee58ap+5,0x1.6c88c2dc2e07dp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.2662b614c91d7p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d4d8fe3a90569p+3,-0x1.715d8d700ce4bp+6,-0x1.0fe1b7867330ep+6,0x1.a98e7f012bdd9p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.ffe8cf65490dap+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.1179ddd81d415p+5,0x1.e1f3b43af054fp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d2b5d16980537p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1a00a34d15a6dp+4,0x1.586940a1c6b95p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.00604309308afp+4,0x0p+0,-0x1.1a1bfcae926a7p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.b6d6de59c8195p+4,0x0p+0,-0x1.3f0e501c326c7p+5,0x0p+0,0x1.7a8bf60d70b5ap+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.ed7cf3d57369cp+3,-0x1.28e05658c60a6p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.675d45cb97cf9p+4,-0x1.6c1696b6c3e54p+6,-0x1.2a8d3e9571221p+6,0x1.a0b2da8bffcdp+1,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.56461e65049a7p+3,-0x1.9b6862dc4a1bep+5,0x0p+0,0x0p+0,0x1.3bb55dfc4133dp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.05c4fded33fe1p+3,0x1.1b43f0fb8bd9dp+4,0x1.2b81115181745p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e8b2c5abde75ep+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.3e8f3fb0f6bc5p+4,0x0p+0,0x0p+0,-0x1.8d2aed2ebee58p+7,0x1.d5c8372b58c09p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.bff0d3b5ee58ap+5,0x0p+0,0x1.89b91d2219bdbp+8,-0x1.7fcffcbcdbf78p+5,0x0p+0,0x0p+0,0x1.6f6c74d9ecdddp+2,0x0p+0,0x0p+0,0x0p+0,0x1.2c54cf456b1a6p+3,0x0p+0,0x1.0df97df1c2c6bp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0d4b80ceb0299p+4,-0x1.6e58589ef504cp+6,-0x1.2a67e1c99dd11p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.edf4107447802p-7,0x0p+0,0x0p+0,-0x1.95b29f6d4920bp+5,0x1.4b36f0d8771f5p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0628d58847693p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1b43f0fb8bd9dp+4,0x1.2b80348b7603p+4,0x1.e31067ad1615fp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6cffed1c8ecep+2,0x0p+0,0x0p+0,-0x1.57295300e0aadp+3,-0x1.90dd7cafe67dcp+7,0x0p+0,0x0p+0,0x1.d05c28417c38fp+3,0x0p+0,0x1.3cc1d15e9b6f2p+2,0x0p+0,0x1.6c88c2dc2e07dp+2,0x0p+0,-0x1.7fcffcbcdbf78p+5,0x1.8923d9f69817bp+8,0x0p+0,0x0p+0,-0x1.8b8b7b5899805p+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.16363fb0b2c27p+3,-0x1.04b094aac0a65p+6,-0x1.8e456772f54c1p+6,0x1.e17e5c670322p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1ff2db8313e8cp+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2e7eeaed1249ap+3,0x0p+0,0x0p+0,0x1.7d8256eda5c73p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.ef6bfac8a5702p+4,0x1.1b1598abded16p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dbb9a12366905p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1bea199c721dap+4,0x0p+0,0x0p+0,-0x1.18e774793de29p+7,0x1.a89200b841d89p+2,0x0p+0,0x1.0062cadc4a81dp+5,0x1.adddc6373bdf7p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.761e8ef1c90bep+8,-0x1.f67bf7e9c712fp+4,0x0p+0,0x0p+0,-0x1.1e0f097f7c50ep+7,0x0p+0,0x0p+0,0x0p+0,0x1.929b5aa49bf0dp+2,-0x1.1266ae5f23fe3p+6,-0x1.19be37ebf4616p+6,0x1.27a70fc5e21e8p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.251e8ec7c5fdcp+6,0x1.9e514c8d468c8p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.2709c1a54f92fp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4f99ec924dc25p+4,0x1.719cc888aa1aap+4,0x1.00604309308cbp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6ae8d25ce7f9p+3,0x0p+0,0x0p+0,-0x1.084c758ef09ecp+6,-0x1.f5eadf59e80e4p+6,0x0p+0,0x0p+0,-0x1.7fd7e303ed878p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.f67bf7e9c712fp+4,0x1.3abbf1c412ba8p+8,0x1.182d9a11402cap+3,0x0p+0,0x1.66dfba9f657f8p+3,0x1.8bc9b5a086e2p+2,0x0p+0,0x0p+0,-0x1.8dd46ca643163p+6,0x1.1ae010713de85p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.0d9048d272de5p+4,-0x1.3fe59c45f93f1p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.784f9d7ead4p-4,-0x1.0a39dd05559p+6,0x0p+0,0x1.6c68fac61e9a8p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.04c748a515b99p+3,0x1.147c2143acba2p+4,0x1.15fd5af8819eep+4,0x0p+0,0x0p+0,0x0p+0,0x1.8987c340b9319p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6dcf117faf1dbp+2,-0x1.9affc4c012efdp+1,0x0p+0,0x1.a020da6688b3p-1,-0x1.25b2cd1d441dbp+7,0x0p+0,0x1.f9a346cf8536dp+2,0x0p+0,0x0p+0,0x0p+0,0x1.6f6c74d9ecdddp+2,-0x1.8b8b7b5899805p+5,0x0p+0,0x1.182d9a11402cap+3,0x1.647cb8115401fp+8,0x1.d16353ee3df43p+4,0x0p+0,-0x1.bb32573f16becp+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.a7631566c5b74p+4,0x1.34cfd20a8d5e2p+2,0x1.50bdbbb351494p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.8a6208a1d1cf4p+4,0x1.f6075245f1fap+3,-0x1.c071f49094b9dp+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.90181e24a5dap+4,0x0p+0,0x1.a25ae6da149cap+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.1733ea712ebf3p+3,-0x1.27612238327b6p+7,0x0p+0,0x0p+0,0x0p+0,-0x1.6a185ecb2c6ebp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.56dbf2a84be09p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.d16353ee3df43p+4,0x1.d0876b3f77475p+8,0x0p+0,0x1.00e2ca2ddac6ep+5,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.dee59750b101fp+2,0x1.bbe1d89d61de7p+4,0x1.c44eaaad31d5bp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e5b52cf64f397p+3,0x1.697062be35b05p+1,0x1.421ff3f9efec1p-1,0x1.2451ee3da4063p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.9f05c6ed79e51p+6,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.11d6a33035964p+4,-0x1.19fa7a8b857fep+7,-0x1.1ea42b0ddb3e8p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6818185edb00ep+3,0x0p+0,0x0p+0,0x1.7f5ef6fff5e34p+4,0x0p+0,0x0p+0,-0x1.5204c6944681dp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.1e0f097f7c50ep+7,0x1.66dfba9f657f8p+3,0x0p+0,0x0p+0,0x1.9e24ea12e003p+8,0x0p+0,0x0p+0,0x1.873cc5166ba41p+4,0x1.2924877dc5d46p+4,0x1.09a3f30851ca1p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.23d72b03b9af9p+3,0x1.67fa508fe64ep+3,0x1.af9de00cf4d9bp+1,0x1.1216ea4602118p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.7b654d0d2f613p+0,0x0p+0,-0x1.289ab4bcf5458p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.7699501dd8129p+3,0x0p+0,0x0p+0,0x1.4d8bd28607cf7p+1,-0x1.557df335b38ddp+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.53dfc319dc20cp+2,0x0p+0,0x1.75821c177cffep+2,-0x1.33dc8118d9621p+7,0x0p+0,-0x1.94b7dd0410fc2p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.8bc9b5a086e2p+2,-0x1.bb32573f16becp+5,0x1.00e2ca2ddac6ep+5,0x0p+0,0x1.d73893d3c8346p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.84626b7a6f86ep+3,-0x1.e32fd7fb044bap+3,0x1.79fb1c20b72b2p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6f81647fc2cc7p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.c3246fd6f6608p+7,0x1.6a598b37e98e7p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.dedc1bb71aafap+3,0x1.673c77a74b78fp+3,0x1.66ae694485f35p+4,0x1.65d1c781065cdp+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.3194fa15f71b7p+0,-0x1.518fcf93a07d5p+4,-0x1.50f3e722949a1p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.5b3aea946871ep+7,-0x1.2662b614c91d7p+7,0x1.ed7cf3d57369cp+3,0x1.2c54cf456b1a6p+3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.e4aa1fbdc6b98p+8,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.4fbff30cdcc37p+4,0x1.25deea0c2c1f9p+4,0x1.f0f50b9646f54p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.372a7c92eecb2p+2,0x1.953ae6e6b8987p+1,0x1.b7935228dacd9p+1,0x1.0861c84586355p+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.82eedf26dceabp+6,-0x1.7b15f0fd0b4cap-3,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.08c3865d569fp+3,-0x1.d9711b55b284bp+6,0x0p+0,0x1.f28bc361de6a9p+2,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.6f982149d385fp+4,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,-0x1.7302f114c6a59p+7,0x0p+0,0x1.66fed59ec062p+3,0x0p+0,-0x1.28e05658c60a6p+7,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x0p+0,0x1.873cc5166ba41p+4,0x0p+0,0x0p+0,0x1.8f694fd7f0c1ap+8};
  Eigen::MatrixXd Qexact = Eigen::Map<Eigen::MatrixXd>(&(initvec[0]), 99, 99);

  test_common::assert_near(Q,Qexact,1e-12);
}
