/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.metadata;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.UnaryOperator;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.Build;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.template.delete.DeleteIndexTemplateRequest;
import org.elasticsearch.action.admin.indices.template.put.PutIndexTemplateRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.support.master.MasterNodeRequest;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.metadata.IndexTemplateMetadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.indices.IndexTemplateMissingException;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContentType;

public class TemplateUpgradeService
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(TemplateUpgradeService.class);
    private final UnaryOperator<Map<String, IndexTemplateMetadata>> indexTemplateMetadataUpgraders;
    public final ClusterService clusterService;
    public final ThreadPool threadPool;
    public final Client client;
    final AtomicInteger upgradesInProgress = new AtomicInteger();
    private Map<String, IndexTemplateMetadata> lastTemplateMetadata;
    private static final ToXContent.Params PARAMS = new ToXContent.MapParams(Collections.singletonMap("reduce_mappings", "true"));

    public TemplateUpgradeService(Client client, ClusterService clusterService, ThreadPool threadPool, Collection<UnaryOperator<Map<String, IndexTemplateMetadata>>> indexTemplateMetadataUpgraders) {
        this.client = client;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.indexTemplateMetadataUpgraders = templates -> {
            Map upgradedTemplates = new HashMap(templates);
            for (UnaryOperator upgrader : indexTemplateMetadataUpgraders) {
                upgradedTemplates = (Map)upgrader.apply(upgradedTemplates);
            }
            return upgradedTemplates;
        };
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState state = event.state();
        if (!state.nodes().isLocalNodeElectedMaster()) {
            return;
        }
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        if (this.upgradesInProgress.get() > 0) {
            return;
        }
        Map<String, IndexTemplateMetadata> templates = state.getMetadata().getTemplates();
        if (templates == this.lastTemplateMetadata) {
            return;
        }
        this.lastTemplateMetadata = templates;
        Optional<Tuple<Map<String, BytesReference>, Set<String>>> changes = this.calculateTemplateChanges(templates);
        if (changes.isPresent() && this.upgradesInProgress.compareAndSet(0, ((Map)changes.get().v1()).size() + ((Set)changes.get().v2()).size() + 1)) {
            logger.info("Starting template upgrade to version {}, {} templates will be updated and {} will be removed", (Object)Build.current().version(), (Object)((Map)changes.get().v1()).size(), (Object)((Set)changes.get().v2()).size());
            assert (this.threadPool.getThreadContext().isSystemContext());
            this.threadPool.generic().execute(() -> this.upgradeTemplates((Map)((Tuple)changes.get()).v1(), (Set)((Tuple)changes.get()).v2()));
        }
    }

    void upgradeTemplates(Map<String, BytesReference> changes, Set<String> deletions) {
        MasterNodeRequest request;
        final AtomicBoolean anyUpgradeFailed = new AtomicBoolean(false);
        if (!this.threadPool.getThreadContext().isSystemContext()) {
            throw new IllegalStateException("template updates from the template upgrade service should always happen in a system context");
        }
        for (final Map.Entry<String, BytesReference> change : changes.entrySet()) {
            request = new PutIndexTemplateRequest(change.getKey()).source(change.getValue(), XContentType.JSON);
            request.masterNodeTimeout(TimeValue.timeValueMinutes((long)1L));
            this.client.admin().indices().putTemplate((PutIndexTemplateRequest)request, new ActionListener<AcknowledgedResponse>(){

                @Override
                public void onResponse(AcknowledgedResponse response) {
                    if (!response.isAcknowledged()) {
                        anyUpgradeFailed.set(true);
                        logger.warn("Error updating template [{}], request was not acknowledged", change.getKey());
                    }
                    TemplateUpgradeService.this.tryFinishUpgrade(anyUpgradeFailed);
                }

                @Override
                public void onFailure(Exception e) {
                    anyUpgradeFailed.set(true);
                    logger.warn(() -> "Error updating template [" + (String)change.getKey() + "]", (Throwable)e);
                    TemplateUpgradeService.this.tryFinishUpgrade(anyUpgradeFailed);
                }
            });
        }
        for (final String template : deletions) {
            request = new DeleteIndexTemplateRequest(template);
            request.masterNodeTimeout(TimeValue.timeValueMinutes((long)1L));
            this.client.admin().indices().deleteTemplate((DeleteIndexTemplateRequest)request, new ActionListener<AcknowledgedResponse>(){

                @Override
                public void onResponse(AcknowledgedResponse response) {
                    if (!response.isAcknowledged()) {
                        anyUpgradeFailed.set(true);
                        logger.warn("Error deleting template [{}], request was not acknowledged", (Object)template);
                    }
                    TemplateUpgradeService.this.tryFinishUpgrade(anyUpgradeFailed);
                }

                @Override
                public void onFailure(Exception e) {
                    anyUpgradeFailed.set(true);
                    if (!(e instanceof IndexTemplateMissingException)) {
                        logger.warn(() -> "Error deleting template [" + template + "]", (Throwable)e);
                    }
                    TemplateUpgradeService.this.tryFinishUpgrade(anyUpgradeFailed);
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void tryFinishUpgrade(AtomicBoolean anyUpgradeFailed) {
        assert (this.upgradesInProgress.get() > 0);
        if (this.upgradesInProgress.decrementAndGet() == 1) {
            try {
                if (anyUpgradeFailed.get()) {
                    logger.info("Templates were partially upgraded to version {}", (Object)Build.current().version());
                } else {
                    logger.info("Templates were upgraded successfully to version {}", (Object)Build.current().version());
                }
                Map<String, IndexTemplateMetadata> upgradedTemplates = this.clusterService.state().getMetadata().getTemplates();
                boolean changesRequired = this.calculateTemplateChanges(upgradedTemplates).isPresent();
                if (changesRequired) {
                    logger.warn("Templates are still reported as out of date after the upgrade. The template upgrade will be retried.");
                }
            }
            finally {
                int noMoreUpgrades = this.upgradesInProgress.decrementAndGet();
                assert (noMoreUpgrades == 0);
            }
        }
    }

    Optional<Tuple<Map<String, BytesReference>, Set<String>>> calculateTemplateChanges(Map<String, IndexTemplateMetadata> templates) {
        HashMap<String, IndexTemplateMetadata> existingMap = new HashMap<String, IndexTemplateMetadata>();
        for (Map.Entry<String, IndexTemplateMetadata> customCursor : templates.entrySet()) {
            existingMap.put(customCursor.getKey(), customCursor.getValue());
        }
        Map upgradedMap = (Map)this.indexTemplateMetadataUpgraders.apply(existingMap);
        if (!upgradedMap.equals(existingMap)) {
            HashSet deletes = new HashSet();
            HashMap changes = new HashMap();
            existingMap.keySet().forEach(s -> {
                if (!upgradedMap.containsKey(s)) {
                    deletes.add(s);
                }
            });
            upgradedMap.forEach((key, value) -> {
                if (!value.equals(existingMap.get(key))) {
                    changes.put(key, TemplateUpgradeService.toBytesReference(value));
                }
            });
            return Optional.of(new Tuple(changes, deletes));
        }
        return Optional.empty();
    }

    private static BytesReference toBytesReference(IndexTemplateMetadata templateMetadata) {
        try {
            return XContentHelper.toXContent((builder, params) -> {
                IndexTemplateMetadata.Builder.toInnerXContentWithTypes(templateMetadata, builder, params);
                return builder;
            }, XContentType.JSON, PARAMS, false);
        }
        catch (IOException ex) {
            throw new IllegalStateException("Cannot serialize template [" + templateMetadata.getName() + "]", ex);
        }
    }
}

