/*
 * crashtest
 * (c) 2006 by Bram Stolk, SARA for NWO
 * bram at gmail.com
 * LICENSED ACCORDING TO THE GPL
 */


#include <assert.h>
#include <libgen.h>	// for dirname(), Is this portable?

#ifdef __APPLE__
#include <OpenGL/glu.h>
#else
#include <GL/glu.h>
#endif

#include <FL/Fl.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Check_Button.H>
#include <FL/Fl_Chart.H>
#include <FL/Fl_Value_Slider.H>
#include <FL/Fl_Gl_Window.H>
#include <FL/fl_draw.H>

#include <plib/ssg.h>
#include <plib/ul.h>

#include "staticworldobject.h"
#include "cartobject.h"
#include "usercam.h"
#include "dynamicboxobject.h"
#include "dynamiccylobject.h"

#include "modelmap.h"
#include "crashworld.h"
#include "bipedobject.h"
#include "stereocontext.h"


static const int MENUH=144;
static const int MENUW=1024;
static const int WIDGETW=MENUW/4;
static const int WIDGETH=MENUH/4;


static float aspectratio=1.0;
static int winw, winh;
static ssgRoot *scene=0;
static ssgContext    *monocontext=0;
static StereoContext *stereocontext=0;
static float fps=60.0;
static float timescale=1.0;
static float groundgrip=10.0;
static float gametime=0.0;
static float dt_hist[10]={0.016,0.016,0.016,0.016,0.016,0.016,0.016,0.016,0.016,0.016};
static std::string dirprefix;
static std::string displaymode="monoscopic";

static dJointGroupID contactgroup;
static dSpaceID bigspace;
static dSpaceID staticspace;
static dWorldID world;

static CrashWorld *crashworld=0;

static CartObject *cart=0;
static BipedObject *biped=0;
static UserCam     *usercam=0;

static ModelMap *modelmap=0;

class GraphWidget;
class MainWindow;

MainWindow *mainwin = 0;

static Fl_Value_Slider *s_speed = 0;
static Fl_Value_Slider *s_react = 0;
static Fl_Value_Slider *s_tscal = 0;
static Fl_Value_Slider *s_bricm = 0;
static Fl_Value_Slider *s_cartm = 0;
static Fl_Value_Slider *s_ggrip = 0;

static Fl_Check_Button *c_seatb = 0;

static Fl_Button       *b_start = 0;
static Fl_Button       *b_reset = 0;
static GraphWidget     *c_force = 0;



static void stop_game(void);
static void start_game(void);
static void setup_plib(void);



static void OglErrorCheck(const std::string &context)
{
  GLenum error = glGetError();
  while (error != GL_NO_ERROR)
  {
    fprintf(stderr, "plode OpenGL error (%s): %s\n", context.c_str(),gluErrorString(error));
    error = glGetError();
  }
}


static void near_callback(void *data, dGeomID o1, dGeomID o2)
{
  assert(o1);
  assert(o2);

  if (dGeomIsSpace(o1) || dGeomIsSpace(o2))
  {
    // colliding a space with something
    dSpaceCollide2(o1,o2,data,&near_callback);
    // Note we do not want to test intersections within a space,
    // only between spaces.
    return;
  }

  // Two non space geoms

  WorldObject *wo1 = static_cast<WorldObject*>(dGeomGetData(o1));
  WorldObject *wo2 = static_cast<WorldObject*>(dGeomGetData(o2));

  const int N = 32;
  dContact contact[N];
  int n = dCollide (o1,o2,N,&(contact[0].geom),sizeof(dContact));

  if (n > 0)
  {
    assert(wo1);
    assert(wo2);

    // Create constraints for all the contacts between the geometries.
    for (int i=0; i<n; i++)
    {
#if 0
      contact[i].surface.slip1 = 0.14;
      contact[i].surface.slip2 = 0.14;
//      contact[i].surface.mode = dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactSlip1 | dContactSlip2;
#endif
      contact[i].surface.mode = dContactSoftERP | dContactSoftCFM;
      contact[i].surface.mu = 10.0;
      contact[i].surface.soft_erp = 0.95;
      contact[i].surface.soft_cfm = 0.05;

      if (wo1->name == "ground" || wo2->name == "ground")
      {
        contact[i].surface.mu = groundgrip;
//        contact[i].surface.slip1 = 0.55;
//        contact[i].surface.slip2 = 0.55;
      }

      dJointID c = dJointCreateContact (world,contactgroup,&contact[i]);
      dGeomID g1 = contact[i].geom.g1;
      dGeomID g2 = contact[i].geom.g2;
      dBodyID b1 = dGeomGetBody(g1);
      dBodyID b2 = dGeomGetBody(g2);
      assert(b1 || b2);
      dJointAttach (c, b1, b2);
    }
  }
}




static void reshape(int w, int h)
{
  assert(w && h);
  winw=w; winh=h;
  if (monocontext)
  {
    glViewport ( 0, 0, w, h ) ;
    aspectratio = w / (float) h;
    float fovy = 60.0 * M_PI / 180.0;
    float nearPlaneDistance = 0.4;
    float farPlaneDistance  = 2500.0;
    float y = tan(0.5 * fovy) * nearPlaneDistance;
    float x = aspectratio * y;
    assert(monocontext);
    monocontext->setFrustum(-x,x,-y,y,nearPlaneDistance,farPlaneDistance);
  }
  if (stereocontext)
  {
    stereocontext->SetWindowSize(winw/2, winh);
  }
  OglErrorCheck("reshape");
}


class GraphWidget : public Fl_Widget
{
  public:
    GraphWidget(int X, int Y, int W, int H) :
      Fl_Widget(X,Y,W,H,"graph"),
      sz(W)
    {
      values = new float[sz];
      pixels = new unsigned char[H*W*3];
      for (int i=0; i<sz; i++) values[i]=0.0;
      writepos = 0;
    }
    void draw(void)
    {
      memset(pixels, 192, 3*w()*h());
      for (int i=0; i<sz; i++)
      {
        float v = values[(writepos+i)%sz];
        int y = (int) (h() - 1 - v * h());
        if (y >= h()) y = h()-1;
        if (y <    0) y = 0;
        unsigned char *p = pixels + y *sz*3 + i*3;
        int b=4;
        while (y < h() && b--)
        {
          p[0]=p[1]=p[2]=0;
          p += 3*w();
          y++;
        }
      }
      fl_draw_image(pixels, x(), y(), w(), h());
    }
    void sample(float v)
    {
      values[writepos] = v;
      writepos = (writepos+1) % sz;
    }

  protected:
    int sz;
    float *values;
    unsigned char *pixels;
    int writepos;
};


class SimulationWindow : public Fl_Gl_Window 
{
  public:
    SimulationWindow(int X, int Y, int W, int H, const char *L=0) : Fl_Gl_Window(X,Y,W,H,L)
    {
      shape_dirty = false;
    }
    void draw()
    {
      if (!valid())
      {
        if (!scene)
        {
          setup_plib();
          start_game();
        }
        reshape(w(), h());
        return;
      }
      if (shape_dirty)
      {
        shape_dirty = false;
        reshape(neww, newh);
      }
      glClearColor(0,0,0.2,0);
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT ) ;
      glMatrixMode(GL_MODELVIEW);
      if (monocontext)
      {
        ssgCullAndDraw(scene);
      }
      if (stereocontext)
      {
        stereocontext->MakeCurrent("left");
        ssgCullAndDraw(scene);
        stereocontext->MakeCurrent("right");
        ssgCullAndDraw(scene);
      }
      OglErrorCheck("ssgCullAndDraw");
    }
    virtual int handle(int event)
    {
      static int lastx, lasty;
      if (event == FL_PUSH)
      {
        lastx = Fl::event_x();
        lasty = Fl::event_y();
        return 1;
      }
      if (event  == FL_DRAG)
      {
        int s = Fl::event_state();
        if (s & FL_BUTTON1)
        {
          usercam->ChangeHeading (0.5 * (Fl::event_x() - lastx));
          usercam->ChangePitch   (0.5 * (Fl::event_y() - lasty));
        }
        if (s & FL_BUTTON3)
        {
          usercam->ChangeDistance(0.1 * (Fl::event_y() - lasty));
        }
        lastx = Fl::event_x();
        lasty = Fl::event_y();
        return 1;
      }
      return Fl_Window::handle(event);
    }
    virtual void resize(int X, int Y, int W, int H)
    {
      if (W!=neww || H!=newh)
      {
        neww = W;
        newh = H;
        shape_dirty=true;
      }
      Fl_Gl_Window::resize(X, Y, W, H);
    }
  protected:
    bool shape_dirty;
    int neww, newh;
};



class MainWindow : public Fl_Window
{
  public:
    virtual void resize(int X, int Y, int W, int H)
    {
      child(0)->resize(0,0,W,H-MENUH);
      child(1)->resize(0,H-MENUH,W,MENUH);
      Fl_Window::resize(X,Y,W,H);
    }
    virtual int handle(int event)
    {
      if (event == FL_KEYBOARD)
      {
        switch (Fl::event_key())
        {
          case FL_Escape:
            stop_game();
            this->hide();
            break;
          default:
            break;
        }
        return 1;
      }
      return Fl_Window::handle(event);
    }
    MainWindow(int X, int Y, int W, int H, const char *L=0) : Fl_Window(X,Y,W,H,L)
    {
    }
};


static void setup_plib(void)
{
  ssgInit();

  glEnable(GL_DEPTH_TEST);
  glEnable(GL_CULL_FACE);

  float amb[4]={0.5, 0.5, 0.5, 1};
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, amb);

  if (getenv("PLODE_DISPLAYMODE"))
    displaymode = getenv("PLODE_DISPLAYMODE");
  assert(displaymode == "monoscopic" || displaymode == "quadbufferstereoscopic" || displaymode == "passivestereoscopic");

  if (displaymode == "monoscopic")
  {
    monocontext = new ssgContext();
    monocontext->makeCurrent();
  }
  else
  {
    stereocontext = new StereoContext(displaymode=="quadbufferstereoscopic");
  }

  if (displaymode=="passivestereoscopic")
  {
    mainwin->resize(0,0,2048,768);
  }

  ssgLight *light=ssgGetLight(0);
  light->setColour(GL_AMBIENT,0,0,0);
  light->setPosition(0,0,8);
  light->setSpotlight(true);
  light->setSpotDirection(0,0,-1);
  light->setSpotDiffusion(100,180);
  light->setSpotAttenuation(1, 0.1, 0.04);
  light->on();
}


static void idle(void *info)
{
  SimulationWindow *simwin = static_cast<SimulationWindow*>(info);
  assert(simwin);

  if (!scene)
    return;
  static ulClock clk;
  static int frameCounter = 0;
  clk.setMaxDelta(0.4);  // Don't accomodate for systems slower than 2.5 fps

  float elapsed = clk.getDeltaTime();
  clk.update();

  dt_hist[frameCounter] = elapsed;
  frameCounter = (frameCounter+1)%10;

  float dt=0;
  int i;
  for (i=0; i<10; i++)
    dt += dt_hist[i];
  dt = dt / 10;
  assert(dt>0.0);

  // do not update the fps every time, it is not readable when it is done each tick
  if (frameCounter == 0) 
    fps = 1.0f / dt;

  gametime += elapsed;

  float timestep=timescale*0.005; // Use 5ms timesteps
  static float remaining_sim_time=0;
  remaining_sim_time += timescale * dt;

  while (remaining_sim_time > timestep && timestep)
  {
    dSpaceCollide (bigspace, 0, &near_callback);
    if (crashworld) crashworld->Sustain(timestep);
    if (cart) cart->Sustain(timestep);
    dWorldQuickStep (world, timestep);
    dJointGroupEmpty (contactgroup);
    remaining_sim_time -= timestep;
  }

  if (biped) biped->Sustain(timescale*dt); // updates visual
  usercam->Update(timescale*dt);
  sgVec3 eye, coi, up;
  sgSetVec3(up,0,0,1);
  usercam->GetCameraPos(eye);
  usercam->GetTargetPos(coi);

  if (monocontext)   monocontext->setCameraLookAt(eye, coi, up);
  if (stereocontext) stereocontext->SetCameraLookAt(eye, coi, up);

  ssgLight *light = ssgGetLight(0);
  sgVec3 p;
  biped->GetPos(p);
  light->setPosition(p[0],p[1],p[2]+5.0);
  light->setColour(GL_DIFFUSE, 3,3,3);

  // stats
  if (biped)
  {
    static int pos=0;
    static float accum=0;
    static int cnt=0;
    static const int filtersize = 4;
    float f = biped->NeckForce();
    accum += f;
    cnt++;
    if (cnt==filtersize)
    {
      c_force->sample(0.3*accum/filtersize);
      c_force->redraw();
      pos = (pos+1)%MENUW;
      accum=0;
      cnt=0;
    }
  }

  simwin->redraw();
}



static void start_game(void)
{
  gametime = 0.0;

  // Create ODE world

  dInitODE();

  world = dWorldCreate();
  bigspace = dHashSpaceCreate(0);
  staticspace = dSimpleSpaceCreate(bigspace);
  contactgroup = dJointGroupCreate (0);
  dWorldSetGravity(world,0,0,-9.8);
  dWorldSetAutoDisableFlag(world, true);
  dWorldSetAutoDisableLinearThreshold(world, 0.04);
  dWorldSetAutoDisableAngularThreshold(world, 0.04);
  dWorldSetQuickStepNumIterations(world, 20);

  // Create PLIB world

  scene = new ssgRoot();

  crashworld = new CrashWorld(world, bigspace, staticspace, scene, modelmap, dirprefix);

  sgVec3 cartpos;
  sgSetVec3(cartpos, -29, 0, 0.5);
  cart = new CartObject(modelmap->Get("cart.3ds"), modelmap->Get("wheel.3ds"), modelmap->Get("nwoplate.ac"), world, bigspace, cartpos);
  scene->addKid(cart->GetEntity());

  biped = new BipedObject
  (
    world,
    bigspace,
    cartpos[0], cartpos[1], 1.55,
    0.2,
    modelmap->Get("biped_torso.3ds"),
    modelmap->Get("biped_head.3ds"),
    modelmap->Get("biped_upperarm.3ds"),
    modelmap->Get("biped_lowerarm.3ds"),
    modelmap->Get("biped_upperleg.3ds"),
    modelmap->Get("biped_lowerleg.3ds"),
    modelmap->Get("biped_foot.3ds")
  );
  scene->addKid(biped->GetEntity());
  biped->FixFeet(cart->GetBody());

  // Setup camera
  usercam = new UserCam(-27,0,0);
  usercam->AddTarget(biped->GetTransform());
}


static void stop_game(void)
{
  delete cart; cart=0;
  delete biped; biped=0;
  delete usercam; usercam=0;

  delete crashworld;
  crashworld=0;

  // Delete ODE stuff
  dJointGroupDestroy(contactgroup);
  dSpaceDestroy(staticspace);
  dSpaceDestroy(bigspace);
  dWorldDestroy(world);

  dCloseODE();
}


void start_cb(Fl_Widget *o)
{
  cart->SetDesiredLinearSpeed(s_speed->value());
  biped->ReleaseFeet();
  if (c_seatb->value())
    biped->FixTorso(cart->GetBody()); // seat belt
  crashworld->brickwall->SetMass(s_bricm->value());
  cart->SetReactionTime(s_react->value());
  cart->SetDensity(s_cartm->value());
}


void reset_cb(Fl_Widget *o)
{
  stop_game();
  start_game();
}


void play_cb(Fl_Widget *o)
{
  if (cart) cart->SetDesiredLinearSpeed(s_speed->value());
  if (biped) biped->ReleaseFeet();
}


void timescale_cb(Fl_Widget *o)
{
  timescale = s_tscal->value();
}


void brickmass_cb(Fl_Widget *o)
{
  if (crashworld) crashworld->brickwall->SetMass(s_bricm->value());
}


void cartmass_cb(Fl_Widget *o)
{
  if (cart) cart->SetDensity(s_cartm->value());
}


void groundgrip_cb(Fl_Widget *o)
{
  groundgrip = s_ggrip->value();
}


void reactiontime_cb(Fl_Widget *o)
{
  if (cart) cart->SetReactionTime(s_react->value());
}


void seatbelt_cb(Fl_Widget *o)
{
}


int main(int argc, char *argv[]) 
{
  char *bindirname  = dirname(argv[0]);
  if (!strcmp(bindirname,"."))
    dirprefix="/usr/share/games/crashtest";
  else
  {
    dirprefix = dirname(bindirname) + std::string("/share/games/crashtest");
  }
  if (getenv("PLODE_DATADIR"))
    dirprefix = getenv("PLODE_DATADIR");
  modelmap = new ModelMap(dirprefix);

  if (!Fl::gl_visual(FL_RGB | FL_MULTISAMPLE))
  {
    Fl::warning("Sorry, Your display does not do Multi Sample OpenGL");
    if (!Fl::gl_visual(FL_RGB))
      Fl::fatal("Sorry, Your display does not do OpenGL");
  }
  Fl::visual(FL_RGB);

  mainwin = new MainWindow(0,0, 1024, 576+MENUH, "crashtest simulator, developed for NWO by Bram Stolk, SARA");
  mainwin->size_range(352, 288, 1600, 1200, 16,16);
  SimulationWindow simwin(0, 0, mainwin->w(), mainwin->h()-MENUH, "simulation");
  Fl_Window guiwin(0,576, mainwin->w(), MENUH, "gui");

  int ypos=0;
  int xpos=0;
  int ww = WIDGETW-16;

  s_speed = new Fl_Value_Slider(xpos,ypos,ww,14,"cart speed (m/s)");
  s_speed->type(FL_HOR_NICE_SLIDER);
  s_speed->textsize(8);
  s_speed->precision(0);
  s_speed->range(-5,25);
  s_speed->value(20.0);
  s_speed->align(FL_ALIGN_BOTTOM);
  s_speed->labelsize(9);
  ypos += WIDGETH;

  s_react = new Fl_Value_Slider(xpos,ypos,ww,14,"reaction time (s)");
  s_react->type(FL_HOR_NICE_SLIDER);
  s_react->textsize(8);
  s_react->precision(1);
  s_react->range(0,1);
  s_react->value(0.5);
  s_react->callback(reactiontime_cb);
  s_react->align(FL_ALIGN_BOTTOM);
  s_react->labelsize(9);
  ypos += WIDGETH;

  s_bricm = new Fl_Value_Slider(xpos,ypos,ww,14,"wall density (kg/l)");
  s_bricm->type(FL_HOR_NICE_SLIDER);
  s_bricm->textsize(8);
  s_bricm->precision(0);
  s_bricm->range(1, 20.0);
  s_bricm->value(15.0);
  s_bricm->callback(brickmass_cb);
  s_bricm->align(FL_ALIGN_BOTTOM);
  s_bricm->labelsize(9);
  ypos += WIDGETH;

  s_cartm = new Fl_Value_Slider(xpos,ypos,ww,14,"cart density (kg/l)");
  s_cartm->type(FL_HOR_NICE_SLIDER);
  s_cartm->textsize(8);
  s_cartm->precision(1);
  s_cartm->range(0.5, 8.0);
  s_cartm->value(2.5);
  s_cartm->callback(cartmass_cb);
  s_cartm->align(FL_ALIGN_BOTTOM);
  s_cartm->labelsize(9);
  ypos += WIDGETH;


  ypos = 0;
  xpos += WIDGETW;


  s_ggrip = new Fl_Value_Slider(xpos,ypos,ww,14,"grip ground");
  s_ggrip->type(FL_HOR_NICE_SLIDER);
  s_ggrip->textsize(8);
  s_ggrip->precision(1);
  s_ggrip->range(0.1, 10.0);
  s_ggrip->value(5.0);
  s_ggrip->callback(groundgrip_cb);
  s_ggrip->align(FL_ALIGN_BOTTOM);
  s_ggrip->labelsize(9);
  ypos += WIDGETH;

  s_tscal = new Fl_Value_Slider(xpos,ypos,ww,14,"time scale (factor)");
  s_tscal->type(FL_HOR_NICE_SLIDER);
  s_tscal->textsize(8);
  s_tscal->precision(2);
  s_tscal->range(0.0, 1.0);
  s_tscal->value(1.0);
  s_tscal->callback(timescale_cb);
  s_tscal->align(FL_ALIGN_BOTTOM);
  s_tscal->labelsize(9);
  ypos += WIDGETH;

  c_seatb = new Fl_Check_Button(xpos,ypos,ww,14,"seat belt");
  c_seatb->labelsize(9);
  c_seatb->callback(seatbelt_cb);
  ypos += WIDGETH;

  ypos = 0;
  xpos += WIDGETW;


  b_start = new Fl_Button(xpos,ypos,ww,14,"start");
  b_start->callback(start_cb);
  ypos += WIDGETH;

  b_reset = new Fl_Button(xpos,ypos,ww,14,"reset experiment");
  b_reset->callback(reset_cb);
  ypos += WIDGETH;

  ypos = 0;
  xpos += WIDGETW;

  c_force = new GraphWidget(xpos,ypos,WIDGETW,MENUH);


  guiwin.end();
  guiwin.show();

  mainwin->end();
  mainwin->show();

  Fl::add_idle(idle, &simwin);
  Fl::run();
}

