/* This file is part of OpenBubbles.
 *
 * OpenBubbles is an SDL clone of Bubbles.
 * Copyright (C) 2004  Benny Sperisen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * highscores.cpp is the implementation file of highscores.h.
 *
 * Written by:
 *  Benny Sperisen
 *  lasindi@gmail.com
 *  www.freewebs.com/lasindi
 *****************************************************************************/

#include "highscores.h"

highscore::highscore(string name,int score)
:_name(name),_score(score)
{}

string highscore::name() const
{
  return _name;
}

int highscore::score() const
{
  return _score;
}

vector<highscore> high_scores;

void loadHighScores()
{
  string datadir,namebuffer;
  int scorebuffer,i;
  ifstream fromfile;
  ofstream tofile;
  datadir=getenv("HOME");
  datadir+="/.openbubbles";
  fromfile.open(datadir.c_str());
  if(fromfile.is_open())
  {
		getline(fromfile,namebuffer); // Skip the warning.
		getline(fromfile,namebuffer);
		if(namebuffer.substr(0,6)=="fading") // This is an OpenBubbles 1.1 file,
			getline(fromfile,namebuffer);      // so skip the next line too.
		else // This is (probably) an OpenBubbles 1.0 file,
		{
			fromfile.close(); // so restart ...
			fromfile.open(datadir.c_str());
			if(!fromfile.is_open())
			{
				cerr<<"ERROR: ~/.openbubbles suddenly couldn't be opened. Note: this i"
						<<"s an amazingly weird error, because it could be opened just a m"
						<<"oment ago\n";
				exit(1);
			}
			getline(fromfile,namebuffer); // and skip just the warning.
		}
    for(i=0;!fromfile.eof()&&(i<NUMBER_OF_SCORES);i++)
    {
      fromfile>>namebuffer;
      namebuffer.erase(namebuffer.length()-1,1); // Erase the last '|'.
      if(fromfile.eof())
      {
        cerr<<"ERROR: ~/.openbubbles was corrupted; try restarting OpenBubbles"
						<<".\n";
        fromfile.close();
        tofile.open(datadir.c_str());
        tofile<<"Don't mess with this file! If you do, and OpenBubbles crashes"
							<<", delete it and let OpenBubbles create a new one.\nfading 1"
							<<"\nsound 1\n";
        for(int i=0;i<NUMBER_OF_SCORES;i++)
          tofile<<"\nNobody| 0";
        tofile.close();
        exit(1);
      }
      fromfile>>scorebuffer;
      high_scores.push_back(highscore(namebuffer,scorebuffer));
    }
    if(i<NUMBER_OF_SCORES-1)
      for(;i<NUMBER_OF_SCORES;i++)
        high_scores.push_back(highscore("Nobody",0));
    fromfile.close();
  }
	else
	{
		cout<<"here\n";
		tofile<<"Don't mess with this file! If you do, and OpenBubbles crashes"
					<<", delete it and let OpenBubbles create a new one.\nfading 1"
					<<"\nsound 1\n";
		for(int i=0;i<NUMBER_OF_SCORES;i++)
			tofile<<"\nNobody 0";
		tofile.close();
	}
}

void displayHighScores(bool fadingOn)
{
  char *temp=new char[NAME_MAX_LENGTH+1];
  char *result;
  BFont_Info* font;
  SDL_Surface* highback;
  SDL_Surface* menuback;
  SDL_Surface* transfer;
  SDL_Surface* returnshot;
  SDL_Surface* clearshot;
  SDL_Event event;
  SDL_Rect src,dest;
  Uint32 colorkey,colorkey2,colorkey3,colorkey4;
  string datadir,datadir2,datadir3;
  int time1,time2,diff_time,pause_time; // Time loop
  datadir=DATA_PREFIX;
  datadir+="/highback.png";
  highback=IMG_Load(datadir.c_str());
  if(highback==NULL)
  {
    cerr<<"ERROR: unable to load highback.\n";
    exit(1);
  }
  colorkey=SDL_MapRGB(highback->format,15,15,15);
  SDL_SetColorKey(highback,SDL_SRCCOLORKEY,colorkey);
  datadir=DATA_PREFIX;
  datadir+="/returnshot.png";
  returnshot=IMG_Load(datadir.c_str());
  if(returnshot==NULL)
  {
    cerr<<"ERROR: unable to load returnshot.\n";
    exit(1);
  }
  colorkey2=SDL_MapRGB(returnshot->format,0,0,0);
  SDL_SetColorKey(returnshot,SDL_SRCCOLORKEY,colorkey2);
  datadir=DATA_PREFIX;
  datadir+="/clearshot.png";
  clearshot=IMG_Load(datadir.c_str());
  if(clearshot==NULL)
  {
    cerr<<"ERROR: unable to load clearshot.\n";
    exit(1);
  }
  colorkey3=SDL_MapRGB(clearshot->format,79,229,79);
  SDL_SetColorKey(clearshot,SDL_SRCCOLORKEY,colorkey3);
  datadir=DATA_PREFIX;
  datadir+="/blank.png";
  transfer=IMG_Load(datadir.c_str());
  if(transfer==NULL)
  {
    cerr<<"ERROR: unable to load transfer.\n";
    exit(1);
  }
  colorkey4=SDL_MapRGB(transfer->format,112,112,112);
  SDL_SetColorKey(transfer,SDL_SRCCOLORKEY,colorkey4);
  datadir=DATA_PREFIX;
  datadir+="/menushot.png";
  menuback=IMG_Load(datadir.c_str());
  if(menuback==NULL)
  {
    cerr<<"ERROR: unable to load menuback.\n";
    exit(1);
  }
  src.x=src.y=0;
  dest.x=dest.y=0;
  src.w=dest.w=screen->w;
  src.h=dest.h=screen->h;
  SDL_BlitSurface(screen,&src,menuback,&dest);
  datadir=DATA_PREFIX;
  datadir+="/font.png";
  font=BFont_LoadFont(datadir.c_str());
  if(font==NULL)
  {
    cerr<<"ERROR: unable to load font.\n";
    exit(1);
  }
  dest.x=dest.y=0;
  src.w=dest.w=highback->w;
  src.h=dest.h=highback->h;
  SDL_BlitSurface(highback,&src,transfer,&dest);
  for(unsigned int i=0;i<high_scores.size();i++)
  {
    for(unsigned int j=0;j<high_scores[i].name().length();j++)
      temp[j]=high_scores[i].name().c_str()[j];
    temp[high_scores[i].name().length()]='\0';
    BFont_LeftPrintStringFont(transfer,font,44+i*LINE_SPACING,"  %s",temp);
    BFont_RightPrintStringFont(transfer,font,44+i*LINE_SPACING,"%d  ",
                               high_scores[i].score());
  }
  // Clear the buttons.
  for(list<button*>::iterator i=buttons.begin();i!=buttons.end();i++)
    (*i)->freeSurfaces();
  buttons.clear();
  // Set up the new buttons.
  buttons.push_back(new button(SCREEN_WIDTH/2-139,420,
															 "return.png","return","return_hilight.png",
															 "return_pressed.png"));
  buttons.push_back(new button(SCREEN_WIDTH/2-75,
															 SCREEN_HEIGHT/2-transfer->h/2+45+
															 high_scores.size()*LINE_SPACING+3,"clear.png",
															 "clear","clear_hilight.png",
															 "clear_pressed.png"));
  // Fading transition.
	if(fadingOn)
	{
		for(int i=0;i<256;i+=TRANSITION_SPEED)
		{
			time1=clock()/CLOCKS_PER_SEC; //get time at start of loop
			SDL_SetAlpha(transfer,SDL_SRCALPHA,i);
			SDL_SetAlpha(returnshot,SDL_SRCALPHA,i);
			SDL_SetAlpha(clearshot,SDL_SRCALPHA,i);
			dest.x=dest.y=0;
			src.w=dest.w=menuback->w;
			src.h=dest.h=menuback->h;
			SDL_BlitSurface(menuback,&src,screen,&dest);
			boxRGBA(screen,0,0,SCREEN_WIDTH-1,SCREEN_HEIGHT-1,0,0,0,i/2);
			src.w=dest.w=transfer->w;
			src.h=dest.h=transfer->h;
			dest.x=SCREEN_WIDTH/2-src.w/2;
			dest.y=SCREEN_HEIGHT/2-src.h/2;
			SDL_BlitSurface(transfer,&src,screen,&dest);
			src.w=dest.w=returnshot->w;
			src.h=dest.h=returnshot->h;
			dest.x=SCREEN_WIDTH/2-139;
			dest.y=420;
			SDL_BlitSurface(returnshot,&src,screen,&dest);
			src.w=dest.w=clearshot->w;
			src.h=dest.h=clearshot->h;
			dest.x=SCREEN_WIDTH/2-75;
			dest.y=SCREEN_HEIGHT/2-transfer->h/2+45+high_scores.size()*LINE_SPACING
				+3;
			SDL_BlitSurface(clearshot,&src,screen,&dest);
			SDL_UpdateRect(screen,0,0,screen->w,screen->h);
			time2=clock()/CLOCKS_PER_SEC; //get time at end of loop
			diff_time=time2-time1;    //compare...
			pause_time=25-diff_time;
			SDL_Delay(pause_time);    //and pause for required amount of time
			if((i+TRANSITION_SPEED>=256)&&(i!=255)) // This is the last run
				i=255-TRANSITION_SPEED;               // through the loop.
		}
	}
  src.w=dest.w=transfer->w;
  src.h=dest.h=transfer->h;
  dest.x=0;
  dest.y=0;
  SDL_BlitSurface(transfer,&src,transfer,&dest);
  for(;;)
  {
    time1=clock()/CLOCKS_PER_SEC; //get time at start of loop
    dest.x=dest.y=0;
    src.w=dest.w=menuback->w;
    src.h=dest.h=menuback->h;
    SDL_BlitSurface(menuback,&src,screen,&dest);
    boxRGBA(screen,0,0,SCREEN_WIDTH-1,SCREEN_HEIGHT-1,0,0,0,128);
    src.w=dest.w=transfer->w;
    src.h=dest.h=transfer->h;
    dest.x=SCREEN_WIDTH/2-src.w/2;
    dest.y=SCREEN_HEIGHT/2-src.h/2;
    SDL_BlitSurface(transfer,&src,screen,&dest);
    while(SDL_PollEvent(&event))
      if(event.type==SDL_QUIT)
        goto leave;
    drawButtons();
    SDL_UpdateRect(screen,0,0,screen->w,screen->h);
    result=handleButtons();
    if(strcmp(result,"return")==0)
    {
			// Transition back
			if(fadingOn)
			{
				for(int i=0;i<256;i+=TRANSITION_SPEED)
				{
					time1=clock()/CLOCKS_PER_SEC; //get time at start of loop
					SDL_SetAlpha(transfer,SDL_SRCALPHA,255-i);
					SDL_SetAlpha(returnshot,SDL_SRCALPHA,255-i);
					SDL_SetAlpha(clearshot,SDL_SRCALPHA,255-i);
					dest.x=dest.y=0;
					src.w=dest.w=menuback->w;
					src.h=dest.h=menuback->h;
					SDL_BlitSurface(menuback,&src,screen,&dest);
					boxRGBA(screen,0,0,SCREEN_WIDTH-1,SCREEN_HEIGHT-1,0,0,0,128-i/2);
					src.w=dest.w=transfer->w;
					src.h=dest.h=transfer->h;
					dest.x=SCREEN_WIDTH/2-src.w/2;
					dest.y=SCREEN_HEIGHT/2-src.h/2;
					SDL_BlitSurface(transfer,&src,screen,&dest);
					src.w=dest.w=returnshot->w;
					src.h=dest.h=returnshot->h;
					dest.x=SCREEN_WIDTH/2-139;
					dest.y=420;
					SDL_BlitSurface(returnshot,&src,screen,&dest);
					src.w=dest.w=clearshot->w;
					src.h=dest.h=clearshot->h;
					dest.x=SCREEN_WIDTH/2-75;
					dest.y=SCREEN_HEIGHT/2-transfer->h/2+45+high_scores.size()*
						LINE_SPACING+3;
					SDL_BlitSurface(clearshot,&src,screen,&dest);
					SDL_UpdateRect(screen,0,0,screen->w,screen->h);
					time2=clock()/CLOCKS_PER_SEC; //get time at end of loop
					diff_time=time2-time1;    //compare...
					pause_time=25-diff_time;
					SDL_Delay(pause_time);    //and pause for required amount of time
					if((i+TRANSITION_SPEED>=256)&&(i!=255)) // This is the last run
						i=255-TRANSITION_SPEED;               // through the loop.
				}
			}
      break;
    }
    else if(strcmp(result,"clear")==0)
    {
      high_scores.clear();
      for(int i=0;i<NUMBER_OF_SCORES;i++)
        high_scores.push_back(highscore("Nobody",0));
      dest.x=dest.y=0;
      src.w=dest.w=highback->w;
      src.h=dest.h=highback->h;
      SDL_BlitSurface(highback,&src,transfer,&dest);
      for(unsigned int i=0;i<high_scores.size();i++)
      {
        for(unsigned int j=0;j<high_scores[i].name().length();j++)
          temp[j]=high_scores[i].name().c_str()[j];
        temp[high_scores[i].name().length()]='\0';
        BFont_LeftPrintStringFont(transfer,font,44+i*LINE_SPACING,"  %s",temp);
        BFont_RightPrintStringFont(transfer,font,44+i*LINE_SPACING,"%d  ",
                                   high_scores[i].score());
      }
      saveEverything();
    }
    else if(strcmp(result,"quit")==0)
      goto leave;
    time2=clock()/CLOCKS_PER_SEC; //get time at end of loop
    diff_time=time2-time1;    //compare...
    pause_time=30-diff_time;
    SDL_Delay(pause_time);    //and pause for required amount of time
  }
leave:
  SDL_FreeSurface(returnshot);
  SDL_FreeSurface(clearshot);
  SDL_FreeSurface(highback);
  SDL_FreeSurface(transfer);
  SDL_FreeSurface(menuback);
  BFont_FreeFont(font);
  delete[] temp;
}

void typestr(SDL_Event event,string & str,int & cursor,BFont_Info* font)
{
  int totalwidth=0;
  if(cursor<0)
  {
    cerr<<"ERROR: cursor in typestr() was <0.\n";
    exit(1);
  }
  else if((unsigned int)cursor>str.length())
  {
    cerr<<"ERROR: cursor in typestr() was >str.length().\n";
    exit(1);
  }
  if(event.type!=SDL_KEYDOWN)
    return;
  if((event.key.keysym.sym==SDLK_BACKSPACE)&&(cursor!=0))
  {
    cursor--;
    str.erase(cursor,1);
  }
  else if((event.key.keysym.sym==SDLK_LEFT)&&(cursor!=0))
    cursor--;
  else if((event.key.keysym.sym==SDLK_RIGHT)&&((unsigned)cursor!=str.length()))
    cursor++;
  else if((event.key.keysym.sym==SDLK_DELETE)&&
          ((unsigned)cursor!=str.length()))
    str.erase(cursor,1);
  else if(event.key.keysym.sym==SDLK_HOME)
    cursor=0;
  else if(event.key.keysym.sym==SDLK_END)
    cursor=str.length();
  else if((event.key.keysym.sym==SDLK_RIGHT)&&((unsigned)cursor!=str.length()))
    cursor++;
  else if(event.key.keysym.sym==SDLK_a)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="A";
      else
        str.insert(cursor,"A");
    else
      if((unsigned)cursor==str.length())
        str+="a";
      else
        str.insert(cursor,"a");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_b)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="B";
      else
        str.insert(cursor,"B");
    else
      if((unsigned)cursor==str.length())
        str+="b";
      else
        str.insert(cursor,"b");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_c)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="C";
      else
        str.insert(cursor,"C");
    else
      if((unsigned)cursor==str.length())
        str+="c";
      else
        str.insert(cursor,"c");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_d)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="D";
      else
        str.insert(cursor,"D");
    else
      if((unsigned)cursor==str.length())
        str+="d";
      else
        str.insert(cursor,"d");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_e)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="E";
      else
        str.insert(cursor,"E");
    else
      if((unsigned)cursor==str.length())
        str+="e";
      else
        str.insert(cursor,"e");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_f)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="F";
      else
        str.insert(cursor,"F");
    else
      if((unsigned)cursor==str.length())
        str+="f";
      else
        str.insert(cursor,"f");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_g)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="G";
      else
        str.insert(cursor,"G");
    else
      if((unsigned)cursor==str.length())
        str+="g";
      else
        str.insert(cursor,"g");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_h)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="H";
      else
        str.insert(cursor,"H");
    else
      if((unsigned)cursor==str.length())
        str+="h";
      else
        str.insert(cursor,"h");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_i)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="I";
      else
        str.insert(cursor,"I");
    else
      if((unsigned)cursor==str.length())
        str+="i";
      else
        str.insert(cursor,"i");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_j)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="J";
      else
        str.insert(cursor,"J");
    else
      if((unsigned)cursor==str.length())
        str+="j";
      else
        str.insert(cursor,"j");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_k)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="K";
      else
        str.insert(cursor,"K");
    else
      if((unsigned)cursor==str.length())
        str+="k";
      else
        str.insert(cursor,"k");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_l)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="L";
      else
        str.insert(cursor,"L");
    else
      if((unsigned)cursor==str.length())
        str+="l";
      else
        str.insert(cursor,"l");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_m)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="M";
      else
        str.insert(cursor,"M");
    else
      if((unsigned)cursor==str.length())
        str+="m";
      else
        str.insert(cursor,"m");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_n)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="N";
      else
        str.insert(cursor,"N");
    else
      if((unsigned)cursor==str.length())
        str+="n";
      else
        str.insert(cursor,"n");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_o)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="O";
      else
        str.insert(cursor,"O");
    else
      if((unsigned)cursor==str.length())
        str+="o";
      else
        str.insert(cursor,"o");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_p)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="P";
      else
        str.insert(cursor,"P");
    else
      if((unsigned)cursor==str.length())
        str+="p";
      else
        str.insert(cursor,"p");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_q)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="Q";
      else
        str.insert(cursor,"Q");
    else
      if((unsigned)cursor==str.length())
        str+="q";
      else
        str.insert(cursor,"q");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_r)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="R";
      else
        str.insert(cursor,"R");
    else
      if((unsigned)cursor==str.length())
        str+="r";
      else
        str.insert(cursor,"r");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_s)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="S";
      else
        str.insert(cursor,"S");
    else
      if((unsigned)cursor==str.length())
        str+="s";
      else
        str.insert(cursor,"s");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_t)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="T";
      else
        str.insert(cursor,"T");
    else
      if((unsigned)cursor==str.length())
        str+="t";
      else
        str.insert(cursor,"t");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_u)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="U";
      else
        str.insert(cursor,"U");
    else
      if((unsigned)cursor==str.length())
        str+="u";
      else
        str.insert(cursor,"u");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_v)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="V";
      else
        str.insert(cursor,"V");
    else
      if((unsigned)cursor==str.length())
        str+="v";
      else
        str.insert(cursor,"v");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_w)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="W";
      else
        str.insert(cursor,"W");
    else
      if((unsigned)cursor==str.length())
        str+="w";
      else
        str.insert(cursor,"w");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_x)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="X";
      else
        str.insert(cursor,"X");
    else
      if((unsigned)cursor==str.length())
        str+="x";
      else
        str.insert(cursor,"x");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_y)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="Y";
      else
        str.insert(cursor,"Y");
    else
      if((unsigned)cursor==str.length())
        str+="y";
      else
        str.insert(cursor,"y");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_z)
  {
    if(!((event.key.keysym.mod & KMOD_LSHIFT)||
          (event.key.keysym.mod & KMOD_RSHIFT))!=
          !(event.key.keysym.mod & KMOD_CAPS))
      if((unsigned)cursor==str.length())
        str+="Z";
      else
        str.insert(cursor,"Z");
    else
      if((unsigned)cursor==str.length())
        str+="z";
      else
        str.insert(cursor,"z");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_0)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+=")";
      else
        str.insert(cursor,")");
    else
      if((unsigned)cursor==str.length())
        str+="0";
      else
        str.insert(cursor,"0");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_1)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="!";
      else
        str.insert(cursor,"!");
    else
      if((unsigned)cursor==str.length())
        str+="1";
      else
        str.insert(cursor,"1");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_2)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="@";
      else
        str.insert(cursor,"@");
    else
      if((unsigned)cursor==str.length())
        str+="2";
      else
        str.insert(cursor,"2");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_3)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="#";
      else
        str.insert(cursor,"#");
    else
      if((unsigned)cursor==str.length())
        str+="3";
      else
        str.insert(cursor,"3");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_4)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="$";
      else
        str.insert(cursor,"$");
    else
      if((unsigned)cursor==str.length())
        str+="4";
      else
        str.insert(cursor,"4");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_5)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="%";
      else
        str.insert(cursor,"%");
    else
      if((unsigned)cursor==str.length())
        str+="5";
      else
        str.insert(cursor,"5");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_6)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="^";
      else
        str.insert(cursor,"^");
    else
      if((unsigned)cursor==str.length())
        str+="6";
      else
        str.insert(cursor,"6");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_7)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="&";
      else
        str.insert(cursor,"&");
    else
      if((unsigned)cursor==str.length())
        str+="7";
      else
        str.insert(cursor,"7");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_8)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="*";
      else
        str.insert(cursor,"*");
    else
      if((unsigned)cursor==str.length())
        str+="8";
      else
        str.insert(cursor,"8");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_9)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="(";
      else
        str.insert(cursor,"(");
    else
      if((unsigned)cursor==str.length())
        str+="9";
      else
        str.insert(cursor,"9");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_QUOTE)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="\"";
      else
        str.insert(cursor,"\"");
    else
      if((unsigned)cursor==str.length())
        str+="'";
      else
        str.insert(cursor,"'");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_SEMICOLON)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+=":";
      else
        str.insert(cursor,":");
    else
      if((unsigned)cursor==str.length())
        str+=";";
      else
        str.insert(cursor,";");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_LEFTBRACKET)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="{";
      else
        str.insert(cursor,"{");
    else
      if((unsigned)cursor==str.length())
        str+="[";
      else
        str.insert(cursor,"[");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_RIGHTBRACKET)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="}";
      else
        str.insert(cursor,"}");
    else
      if((unsigned)cursor==str.length())
        str+="]";
      else
        str.insert(cursor,"]");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_MINUS)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="_";
      else
        str.insert(cursor,"_");
    else
      if((unsigned)cursor==str.length())
        str+="-";
      else
        str.insert(cursor,"-");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_SLASH)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="?";
      else
        str.insert(cursor,"?");
    else
      if((unsigned)cursor==str.length())
        str+="/";
      else
        str.insert(cursor,"/");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_EQUALS)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="+";
      else
        str.insert(cursor,"+");
    else
      if((unsigned)cursor==str.length())
        str+="=";
      else
        str.insert(cursor,"=");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_COMMA)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="<";
      else
        str.insert(cursor,"<");
    else
      if((unsigned)cursor==str.length())
        str+=",";
      else
        str.insert(cursor,",");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_PERIOD)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+=">";
      else
        str.insert(cursor,">");
    else
      if((unsigned)cursor==str.length())
        str+=".";
      else
        str.insert(cursor,".");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_BACKSLASH)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="|";
      else
        str.insert(cursor,"|");
    else
      if((unsigned)cursor==str.length())
        str+="\\";
      else
        str.insert(cursor,"\\");
    cursor++;
  }
  else if(event.key.keysym.sym==SDLK_BACKQUOTE)
  {
    if((event.key.keysym.mod & KMOD_LSHIFT)||
        (event.key.keysym.mod & KMOD_RSHIFT))
      if((unsigned)cursor==str.length())
        str+="~";
      else
        str.insert(cursor,"~");
    else
      if((unsigned)cursor==str.length())
        str+="`";
      else
        str.insert(cursor,"`");
    cursor++;
  }
 else if(event.key.keysym.sym==SDLK_SPACE)
  {
    if((unsigned)cursor==str.length())
      str+=" ";
    else
      str.insert(cursor," ");
    cursor++;
  }
  for(int i=0;i<cursor;i++)
    totalwidth+=BFont_CharWidth(font,str[i]);
  if(totalwidth>MAX_STR_WIDTH)
  {
    cursor--;
    str.erase(cursor,1);
  }
}
