/*
    libzint - the open source barcode library
    Copyright (C) 2025 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* SPDX-License-Identifier: BSD-3-Clause */

#include "testcommon.h"

#include "../gs1.h"

/*
 * Check that GS1_128-based and DBAR_EXP-based symbologies reduce GS1 data
 */
static void test_gs1_reduce(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        const char *data;
        const char *composite;
        int ret;

        const char *comment;
        const char *expected;
    };
    static const struct item data[] = {
        /* 0*/ { BARCODE_GS1_128, -1, "12345678901234", "", ZINT_ERROR_INVALID_DATA, "GS1 data required", "" },
        /* 1*/ { BARCODE_GS1_128, -1, "[01]12345678901231", "", 0, "Input mode ignored; verified manually against TEC-IT",
                    "11010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110100001100101100011101011"
               },
        /* 2*/ { BARCODE_GS1_128, GS1PARENS_MODE, "(01)12345678901231", "", 0, "Input mode ignored (parentheses instead of square brackets)",
                    "11010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110100001100101100011101011"
               },
        /* 3*/ { BARCODE_GS1_128, GS1_MODE, "[01]12345678901231[21]()", "", 0, "Parentheses in AI data",
                    "1101001110011110101110110011011001011001110010001011000111000101101100001010011011110110101100111001101100011011011100100101111011101000110010011001001000110111000101100011101011"
               },
        /* 4*/ { BARCODE_GS1_128, ESCAPE_MODE | GS1PARENS_MODE, "(01)12345678901231(21)\\(\\)", "", 0, "Parentheses in AI data",
                    "1101001110011110101110110011011001011001110010001011000111000101101100001010011011110110101100111001101100011011011100100101111011101000110010011001001000110111000101100011101011"
               },
        /* 5*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901231", "[21]1234", 0, "Input mode ignored",
                    "0000000000000000000001101101110110100001000001101001100111011000010011101001100001010001100010010011011000000110110001010000000000000000000000000"
                    "0000000000000000000001101101100111110100010011001101011100100000010011001001001111001011110011101011001000000110010001010000000000000000000000000"
                    "0000000000000000000001101101000101111100110000101001111010000001010011001101011101110011110010011110110000110111010001010000000000000000000000000"
                    "0010110001100001010001001100100110100110001101110100111000111010010011110101100100001001010011000110010011100100010100001000101001110011100010100"
                    "1101001110011110101110110011011001011001110010001011000111000101101100001010011011110110101100111001101100011011101011110111010110001100011101011"
               },
        /* 6*/ { BARCODE_GS1_128_CC, GS1PARENS_MODE, "(01)12345678901231", "(21)1234", 0, "Input mode ignored (parentheses instead of square brackets)",
                    "0000000000000000000001101101110110100001000001101001100111011000010011101001100001010001100010010011011000000110110001010000000000000000000000000"
                    "0000000000000000000001101101100111110100010011001101011100100000010011001001001111001011110011101011001000000110010001010000000000000000000000000"
                    "0000000000000000000001101101000101111100110000101001111010000001010011001101011101110011110010011110110000110111010001010000000000000000000000000"
                    "0010110001100001010001001100100110100110001101110100111000111010010011110101100100001001010011000110010011100100010100001000101001110011100010100"
                    "1101001110011110101110110011011001011001110010001011000111000101101100001010011011110110101100111001101100011011101011110111010110001100011101011"
               },
        /* 7*/ { BARCODE_GS1_128_CC, GS1_MODE, "[01]12345678901231[10])", "[21](", 0, "Parentheses in AI data",
                    "0000000000000000000000000000000000000000000110110111010100001000001000110000110001010001001110100111010100011100001110001100101100011011000101000000000000000000000000000000000000"
                    "0000000000000000000000000000000000000000000110110110011100000101110110110000010011100101001100100100011110010111101101011100010000011001000101000000000000000000000000000000000000"
                    "0000000000000000000000000000000000000000000110110100010001000100011110111100101111001001001100110111111010011010001111100100011101011101000101000000000000000000000000000000000000"
                    "0010110001100001010001001100100110100110001101110100111000111010010011110101100100001001010011000110010011100100110111011010000100010011011011101000100001011101000010011100010100"
                    "1101001110011110101110110011011001011001110010001011000111000101101100001010011011110110101100111001101100011011001000100101111011101100100100010111011110100010111101100011101011"
               },
        /* 8*/ { BARCODE_GS1_128_CC, ESCAPE_MODE | GS1PARENS_MODE, "(01)12345678901231(10)\\)", "(21)\\(", 0, "Parentheses in AI data",
                    "0000000000000000000000000000000000000000000110110111010100001000001000110000110001010001001110100111010100011100001110001100101100011011000101000000000000000000000000000000000000"
                    "0000000000000000000000000000000000000000000110110110011100000101110110110000010011100101001100100100011110010111101101011100010000011001000101000000000000000000000000000000000000"
                    "0000000000000000000000000000000000000000000110110100010001000100011110111100101111001001001100110111111010011010001111100100011101011101000101000000000000000000000000000000000000"
                    "0010110001100001010001001100100110100110001101110100111000111010010011110101100100001001010011000110010011100100110111011010000100010011011011101000100001011101000010011100010100"
                    "1101001110011110101110110011011001011001110010001011000111000101101100001010011011110110101100111001101100011011001000100101111011101100100100010111011110100010111101100011101011"
               },
        /* 9*/ { BARCODE_EAN14, -1, "1234567890123", "", 0, "Input mode ignored; verified manually against TEC-IT",
                    "11010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110100001100101100011101011"
               },
        /*10*/ { BARCODE_EAN14, GS1PARENS_MODE, "1234567890123", "", 0, "Input mode ignored (parentheses instead of square brackets)",
                    "11010011100111101011101100110110010110011100100010110001110001011011000010100110111101101011001110011011000110100001100101100011101011"
               },
        /*11*/ { BARCODE_NVE18, -1, "12345678901234567", "", 0, "Input mode ignored; verified manually against TEC-IT",
                    "110100111001111010111011011001100101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010010101101110001100011101011"
               },
        /*12*/ { BARCODE_NVE18, GS1PARENS_MODE, "12345678901234567", "", 0, "Input mode ignored (parentheses instead of square brackets)",
                    "110100111001111010111011011001100101100111001000101100011100010110110000101001101111011010110011100100010110001110001011011000010010101101110001100011101011"
               },
        /*13*/ { BARCODE_DBAR_EXP, -1, "2012", "", ZINT_ERROR_INVALID_DATA, "GS1 data required", "" },
        /*14*/ { BARCODE_DBAR_EXP, -1, "[20]12[01]12345678901231", "", 0, "Input mode ignored",
                    "01001100000110011010111111110000101000001000011001010111110010001110101111110000111010001001111101110111111001110010100011111100001011110011101000010110111000011010001011111111001110000110111101011101"
               },
        /*15*/ { BARCODE_DBAR_EXP, GS1PARENS_MODE, "(20)12(01)12345678901231", "", 0, "Input mode ignored (parentheses instead of square brackets)",
                    "01001100000110011010111111110000101000001000011001010111110010001110101111110000111010001001111101110111111001110010100011111100001011110011101000010110111000011010001011111111001110000110111101011101"
               },
        /*16*/ { BARCODE_DBAR_EXP, UNICODE_MODE, "[20]12[01]12345678901231", "", 0, "Input mode ignored",
                    "01001100000110011010111111110000101000001000011001010111110010001110101111110000111010001001111101110111111001110010100011111100001011110011101000010110111000011010001011111111001110000110111101011101"
               },
        /*17*/ { BARCODE_DBAR_EXP, -1, "[21])[8006]123456789012310102", "", 0, "Parentheses in AI data",
                    "01011100110010001110111111110000101010100000001100010011011111101110101111100000011011011001110000010101100000010000100011111100001010001011100000110100011001011110001011111111001110000100101110000100001110001001101000110000000010100000001100101001110101111011000010111111111001101"
               },
        /*18*/ { BARCODE_DBAR_EXP, ESCAPE_MODE | GS1PARENS_MODE, "(21)\\)(8006)123456789012310102", "", 0, "Parentheses in AI data",
                    "01011100110010001110111111110000101010100000001100010011011111101110101111100000011011011001110000010101100000010000100011111100001010001011100000110100011001011110001011111111001110000100101110000100001110001001101000110000000010100000001100101001110101111011000010111111111001101"
               },
        /*19*/ { BARCODE_DBAR_EXP_CC, -1, "[20]12", "[21]1234[8006]123456789012310102", 0, "",
                    "001101001000110100001000001101001100111011000010010001101100001000001011011110111000101100111011001010"
                    "001101011000111110110100001001000111011111101010010000101101100001111001010100001111000100110011001010"
                    "001101011100100110011001111001001000001001111010110000101011011000011110010100110001111100110111001010"
                    "001101011110111100111001001101101001000011000010111000101111011100001011011010011101111000110111101010"
                    "000001111111010110010000000010100100111001100001011010000011010001110100001010101001010000011110100000"
                    "010010000000101001101111111100001011000110011110100101111100101110001011110000000010101111100001011101"
               },
        /*20*/ { BARCODE_DBAR_EXP_CC, UNICODE_MODE, "[20]12", "[21]1234[8006]123456789012310102", 0, "Input mode ignored",
                    "001101001000110100001000001101001100111011000010010001101100001000001011011110111000101100111011001010"
                    "001101011000111110110100001001000111011111101010010000101101100001111001010100001111000100110011001010"
                    "001101011100100110011001111001001000001001111010110000101011011000011110010100110001111100110111001010"
                    "001101011110111100111001001101101001000011000010111000101111011100001011011010011101111000110111101010"
                    "000001111111010110010000000010100100111001100001011010000011010001110100001010101001010000011110100000"
                    "010010000000101001101111111100001011000110011110100101111100101110001011110000000010101111100001011101"
               },
        /*21*/ { BARCODE_DBAR_EXP_CC, -1, "[10]()[8006]123456789012310202", "[21]()", 0, "Parentheses in AI data",
                    "00110110111010100001000001000110000110001010001001110100111010111001111101100111011001000011011000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "00110110110010001000111110110111101100001101001001100100111110111101110101111101000110111011001000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "00110110100010111110110100000100001111011011001001100110100111001100111001000001001001111011101000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "00000110111001100001000000001010010101110011000001101100100000010001010000010101000100011011110001101110101110111000010100000010100101111101110010111000100010111101110100000000100001100000110011010010001100011000100111001111111101010001101110011110001111011100100101000000000110000"
                    "01011001000110011110111111110000101010001100111110010011011111101110101111100000011011100100001110010001010001000111100011111100001010000010001101000111011101000010001011111111001110011111001100101101110011100111011000110000000010101110010001100001110000100011011010111111111001101"
               },
        /*22*/ { BARCODE_DBAR_EXP_CC, GS1PARENS_MODE, "(10)\\(\\)(8006)123456789012310202", "(21)\\(\\)", 0, "Parentheses in AI data",
                    "00110110111010100001000001000110000110001010001001110100111010111001111101100111011001000011011000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "00110110110010001000111110110111101100001101001001100100111110111101110101111101000110111011001000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "00110110100010111110110100000100001111011011001001100110100111001100111001000001001001111011101000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                    "00000110111001100001000000001010010101110011000001101100100000010001010000010101000100011011110001101110101110111000010100000010100101111101110010111000100010111101110100000000100001100000110011010010001100011000100111001111111101010001101110011110001111011100100101000000000110000"
                    "01011001000110011110111111110000101010001100111110010011011111101110101111100000011011100100001110010001010001000111100011111100001010000010001101000111011101000010001011111111001110011111001100101101110011100111011000110000000010101110010001100001110000100011011010111111111001101"
               },
        /*23*/ { BARCODE_DBAR_EXPSTK, -1, "12", "", ZINT_ERROR_INVALID_DATA, "GS1 data required", "" },
        /*24*/ { BARCODE_DBAR_EXPSTK, -1, "[01]12345678901231[20]12", "", 0, "Input mode ignored",
                    "010011010011100000101111111100001011000011101110010111000110110001001011111100001110001111000010101101"
                    "000000101100011111010000000010100100111100010001101000111001001110110100000010100001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000100000000101110010000110100011000111110110010100101000000101001000001111010000000000000000000000"
                    "101110011111111010001101111001011100111000001001101010000111111000110111110000101001000000000000000000"
               },
        /*25*/ { BARCODE_DBAR_EXPSTK, UNICODE_MODE, "[01]12345678901231[20]12", "", 0, "Input mode ignored",
                    "010011010011100000101111111100001011000011101110010111000110110001001011111100001110001111000010101101"
                    "000000101100011111010000000010100100111100010001101000111001001110110100000010100001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000100000000101110010000110100011000111110110010100101000000101001000001111010000000000000000000000"
                    "101110011111111010001101111001011100111000001001101010000111111000110111110000101001000000000000000000"
               },
        /*26*/ { BARCODE_DBAR_EXPSTK, -1, "[01]12345678901231[21](21)", "", 0, "Parentheses in AI data",
                    "010000100011011001101111111100001010010000111011000111000110110001001011111000000110001111000010101101"
                    "000011011100100110010000000010100101101111000100111000111001001110110100000101010001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011000011100010100001000000001011001000110110000111010011011110001001010000001010010000011110100000"
                    "101100111100011101011100111111110100110111001001111000101100100001110100001111110001101111100001010010"
                    "000011000011100010100001000000001011001000110110000111010011011110001001010000001010010000011110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010010011100000010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010101101100011111100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*27*/ { BARCODE_DBAR_EXPSTK, ESCAPE_MODE | GS1PARENS_MODE, "(01)12345678901231(21)\\(21\\)", "", 0, "Parentheses in AI data",
                    "010000100011011001101111111100001010010000111011000111000110110001001011111000000110001111000010101101"
                    "000011011100100110010000000010100101101111000100111000111001001110110100000101010001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011000011100010100001000000001011001000110110000111010011011110001001010000001010010000011110100000"
                    "101100111100011101011100111111110100110111001001111000101100100001110100001111110001101111100001010010"
                    "000011000011100010100001000000001011001000110110000111010011011110001001010000001010010000011110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010010011100000010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010101101100011111100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*28*/ { BARCODE_DBAR_EXPSTK_CC, -1, "12", "[21]1234", ZINT_ERROR_INVALID_DATA, "GS1 data required", "" },
        /*29*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[20]12", "[21]1234[8006]123456789012310102", 0, "Input mode ignored; (same as BARCODE_DBAR_EXP_CC above)",
                    "001101001000110100001000001101001100111011000010010001101100001000001011011110111000101100111011001010"
                    "001101011000111110110100001001000111011111101010010000101101100001111001010100001111000100110011001010"
                    "001101011100100110011001111001001000001001111010110000101011011000011110010100110001111100110111001010"
                    "001101011110111100111001001101101001000011000010111000101111011100001011011010011101111000110111101010"
                    "000001111111010110010000000010100100111001100001011010000011010001110100001010101001010000011110100000"
                    "010010000000101001101111111100001011000110011110100101111100101110001011110000000010101111100001011101"
               },
        /*30*/ { BARCODE_DBAR_EXPSTK_CC, GS1PARENS_MODE, "(20)12", "(21)1234(8006)123456789012310102", 0, "Input mode ignored (parentheses instead of square brackets)",
                    "001101001000110100001000001101001100111011000010010001101100001000001011011110111000101100111011001010"
                    "001101011000111110110100001001000111011111101010010000101101100001111001010100001111000100110011001010"
                    "001101011100100110011001111001001000001001111010110000101011011000011110010100110001111100110111001010"
                    "001101011110111100111001001101101001000011000010111000101111011100001011011010011101111000110111101010"
                    "000001111111010110010000000010100100111001100001011010000011010001110100001010101001010000011110100000"
                    "010010000000101001101111111100001011000110011110100101111100101110001011110000000010101111100001011101"
               },
        /*31*/ { BARCODE_DBAR_EXPSTK_CC, UNICODE_MODE, "[20]12", "[21]1234[8006]123456789012310102", 0, "Input mode ignored",
                    "001101001000110100001000001101001100111011000010010001101100001000001011011110111000101100111011001010"
                    "001101011000111110110100001001000111011111101010010000101101100001111001010100001111000100110011001010"
                    "001101011100100110011001111001001000001001111010110000101011011000011110010100110001111100110111001010"
                    "001101011110111100111001001101101001000011000010111000101111011100001011011010011101111000110111101010"
                    "000001111111010110010000000010100100111001100001011010000011010001110100001010101001010000011110100000"
                    "010010000000101001101111111100001011000110011110100101111100101110001011110000000010101111100001011101"
               },
        /*32*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231[10]1)", "[21]12)(", 0, "Parentheses in AI data",
                    "001101101110110100001000001101110111011100111010011101001000010010000001010010001000010000110110001010"
                    "001101101100111101000010001001110000110111010010011001001101100011110001011010111000100000110010001010"
                    "001101101000111001111101011001011100100000011010011001101001111001000001010000011110010100111010001010"
                    "000000111001111001010000000010100100001100110001001000111001001110110100000010100001110000111101010000"
                    "010011000110000110101111111100001011110011001110110111000110110001001011111100001110001111000010101101"
                    "000000111001111001010000000010100100001100110001001000111001001110110100000010100001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000011001001111100001000000001010000110111100110101111011011100001001010000001010010000011110100000"
                    "101011100110110000011100111111110101111001000011001010000100100011110100001111110001101111100001010010"
               },
        /*33*/ { BARCODE_DBAR_EXPSTK_CC, GS1PARENS_MODE, "(01)12345678901231(10)1\\)", "(21)12\\)\\(", 0, "Parentheses in AI data",
                    "001101101110110100001000001101110111011100111010011101001000010010000001010010001000010000110110001010"
                    "001101101100111101000010001001110000110111010010011001001101100011110001011010111000100000110010001010"
                    "001101101000111001111101011001011100100000011010011001101001111001000001010000011110010100111010001010"
                    "000000111001111001010000000010100100001100110001001000111001001110110100000010100001110000111101010000"
                    "010011000110000110101111111100001011110011001110110111000110110001001011111100001110001111000010101101"
                    "000000111001111001010000000010100100001100110001001000111001001110110100000010100001110000111101010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000011001001111100001000000001010000110111100110101111011011100001001010000001010010000011110100000"
                    "101011100110110000011100111111110101111001000011001010000100100011110100001111110001101111100001010010"
               },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol = NULL;

    const char *text;

    char escaped[1024];
    char escaped2[1024];
    char bwipp_buf[8196];
    char bwipp_msg[1024];

    /* Only do BWIPP test if asked, too slow otherwise */
    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript();

    testStartSymbol(p_ctx->func_name, &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        if (data[i].composite[0]) {
            text = data[i].composite;
            strcpy(symbol->primary, data[i].data);
        } else {
            text = data[i].data;
        }
        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/,
                                    -1 /*option_1*/, -1 /*option_2*/, -1 /*option_3*/, -1 /*output_options*/,
                                    text, -1, debug);

        symbol->input_mode |= GS1SYNTAXENGINE_MODE;

        ret = ZBarcode_Encode(symbol, TCU(text), length);

        if (p_ctx->generate) {
            const int data_len = (int) strlen(data[i].data);
            const int composite_len = (int) strlen(data[i].composite);
            if (data[i].ret == 0) {
                printf("        /*%2d*/ { %s, %s, \"%s\", \"%s\", %d, \"%s\",\n",
                        i, testUtilBarcodeName(data[i].symbology), testUtilInputModeName(data[i].input_mode),
                        testUtilEscape(data[i].data, data_len, escaped, sizeof(escaped)),
                        testUtilEscape(data[i].composite, composite_len, escaped2, sizeof(escaped2)),
                        data[i].ret, data[i].comment);
                testUtilModulesPrint(symbol, "                    ", "\n");
                printf("               },\n");
            } else {
                printf("        /*%2d*/ { %s, %s, \"%s\", \"%s\", %s, \"%s\", \"\" },\n",
                        i, testUtilBarcodeName(data[i].symbology), testUtilInputModeName(data[i].input_mode),
                        testUtilEscape(data[i].data, data_len, escaped, sizeof(escaped)),
                        testUtilEscape(data[i].composite, composite_len, escaped2, sizeof(escaped2)),
                        testUtilErrorName(data[i].ret), data[i].comment);
            }
        } else {
            assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n",
                        i, ret, data[i].ret, symbol->errtxt);

            if (ret < ZINT_ERROR) {
                int width, row;
                ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
                assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n",
                            i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

                if (do_bwipp && testUtilCanBwipp(i, symbol, -1, -1, -1, debug)) {
                    ret = testUtilBwipp(i, symbol, -1, -1, -1, text, length, symbol->primary, bwipp_buf,
                                sizeof(bwipp_buf), NULL);
                    assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n",
                                i, testUtilBarcodeName(symbol->symbology), ret);

                    ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                    assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                                   i, testUtilBarcodeName(symbol->symbology), ret, bwipp_msg, bwipp_buf,
                                   data[i].expected);
                }
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_hrt(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        int output_options;
        const char *data;
        const char *composite;

        int ret;
        const char *expected;
        const char *expected_content;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /*  0*/ { BARCODE_EAN8_CC, -1, -1, "1234567", "[20]12", 0, "12345670", "" },
        /*  1*/ { BARCODE_EANX_CC, -1, -1, "1234567", "[20]12", 0, "12345670", "" }, /* EAN-8 */
        /*  2*/ { BARCODE_EAN8_CC, -1, BARCODE_CONTENT_SEGS, "1234567", "[20]12", 0, "12345670", "12345670|2012" },
        /*  3*/ { BARCODE_EANX_CC, -1, BARCODE_CONTENT_SEGS, "1234567", "[20]12", 0, "12345670", "12345670|2012" },
        /*  4*/ { BARCODE_EAN13_CC, -1, -1, "123456789012", "[20]12", 0, "1234567890128", "" },
        /*  5*/ { BARCODE_EANX_CC, -1, -1, "123456789012", "[20]12", 0, "1234567890128", "" }, /* EAN-13 */
        /*  6*/ { BARCODE_EAN13_CC, -1, BARCODE_CONTENT_SEGS, "123456789012", "[20]12", 0, "1234567890128", "1234567890128|2012" },
        /*  7*/ { BARCODE_EANX_CC, -1, BARCODE_CONTENT_SEGS, "123456789012", "[20]12", 0, "1234567890128", "1234567890128|2012" }, /* EAN-13 */
        /*  8*/ { BARCODE_EAN13_CC, -1, -1, "123456789012", "[10]LOT123[20]12", 0, "1234567890128", "" },
        /*  9*/ { BARCODE_EANX_CC, -1, -1, "123456789012", "[10]LOT123[20]12", 0, "1234567890128", "" }, /* EAN-13 */
        /* 10*/ { BARCODE_EAN13_CC, -1, BARCODE_CONTENT_SEGS, "123456789012", "[10]LOT123[20]12", 0, "1234567890128", "1234567890128|10LOT123\0352012" },
        /* 11*/ { BARCODE_EANX_CC, -1, BARCODE_CONTENT_SEGS, "123456789012", "[10]LOT123[20]12", 0, "1234567890128", "1234567890128|10LOT123\0352012" }, /* EAN-13 */
        /* 12*/ { BARCODE_EAN13_CC, -1, -1, "1234567890128", "[20]12", 0, "1234567890128", "" },
        /* 13*/ { BARCODE_EANX_CC, -1, -1, "1234567890128", "[20]12", 0, "1234567890128", "" },
        /* 14*/ { BARCODE_EAN13_CC, -1, -1, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" },
        /* 15*/ { BARCODE_EANX_CC, -1, -1, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" },
        /* 16*/ { BARCODE_EAN13_CC, GS1NOCHECK_MODE, -1, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" }, /* Still checked */
        /* 17*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, "1234567890123", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" }, /* Still checked */
        /* 18*/ { BARCODE_EAN13_CC, -1, -1, "1234567890128", "[20]1A", ZINT_ERROR_INVALID_DATA, "", "" }, /* AI (20) should be 2 nos. */
        /* 19*/ { BARCODE_EANX_CC, -1, -1, "1234567890128", "[20]1A", ZINT_ERROR_INVALID_DATA, "", "" }, /* AI (20) should be 2 nos. */
        /* 20*/ { BARCODE_EAN13_CC, GS1NOCHECK_MODE, -1, "1234567890128", "[20]1A", 0, "1234567890128", "" },
        /* 21*/ { BARCODE_EANX_CC, GS1NOCHECK_MODE, -1, "1234567890128", "[20]1A", 0, "1234567890128", "" },
        /* 22*/ { BARCODE_EAN13_CC, -1, -1, "1234567890128+12", "[20]12", 0, "1234567890128+12", "" },
        /* 23*/ { BARCODE_EANX_CC, -1, -1, "1234567890128+12", "[20]12", 0, "1234567890128+12", "" },
        /* 24*/ { BARCODE_EAN13_CC, -1, BARCODE_CONTENT_SEGS, "1234567890128+12", "[20]12", 0, "1234567890128+12", "123456789012812|2012" },
        /* 25*/ { BARCODE_EANX_CC, -1, BARCODE_CONTENT_SEGS, "1234567890128+12", "[20]12", 0, "1234567890128+12", "123456789012812|2012" },
        /* 26*/ { BARCODE_DBAR_OMN_CC, -1, -1, "1234567890123", "[20]12", 0, "(01)12345678901231", "" },
        /* 27*/ { BARCODE_DBAR_OMN_CC, -1, BARCODE_CONTENT_SEGS, "1234567890123", "[20]12", 0, "(01)12345678901231", "0112345678901231|2012" },
        /* 28*/ { BARCODE_DBAR_OMN_CC, -1, -1, "12345678901231", "[20]12", 0, "(01)12345678901231", "" },
        /* 29*/ { BARCODE_DBAR_OMN_CC, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" },
        /* 30*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" }, /* Still checked */
        /* 31*/ { BARCODE_DBAR_OMN_CC, -1, -1, "12345678901231", "[20]1A", ZINT_ERROR_INVALID_DATA, "", "" }, /* AI (20) should be 2 nos. */
        /* 32*/ { BARCODE_DBAR_OMN_CC, GS1NOCHECK_MODE, -1, "12345678901231", "[20]1A", 0, "(01)12345678901231", "" },
        /* 33*/ { BARCODE_DBAR_LTD_CC, -1, -1, "1234567890123", "[20]12", 0, "(01)12345678901231", "" },
        /* 34*/ { BARCODE_DBAR_LTD_CC, -1, BARCODE_CONTENT_SEGS, "1234567890123", "[20]12", 0, "(01)12345678901231", "0112345678901231|2012" },
        /* 35*/ { BARCODE_DBAR_LTD_CC, -1, -1, "12345678901231", "[20]12", 0, "(01)12345678901231", "" },
        /* 36*/ { BARCODE_DBAR_LTD_CC, -1, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" },
        /* 37*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, -1, "12345678901232", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" }, /* Still checked */
        /* 38*/ { BARCODE_DBAR_LTD_CC, -1, -1, "12345678901231", "[20]1A", ZINT_ERROR_INVALID_DATA, "", "" }, /* AI (20) should be 2 nos. */
        /* 39*/ { BARCODE_DBAR_LTD_CC, GS1NOCHECK_MODE, -1, "12345678901231", "[20]1A", 0, "(01)12345678901231", "" },
        /* 40*/ { BARCODE_UPCA_CC, -1, -1, "12345678901", "[20]12", 0, "123456789012", "" },
        /* 41*/ { BARCODE_UPCA_CC, -1, BARCODE_CONTENT_SEGS, "12345678901", "[20]12", 0, "123456789012", "0123456789012|2012" },
        /* 42*/ { BARCODE_UPCA_CC, -1, -1, "123456789012", "[20]12", 0, "123456789012", "" },
        /* 43*/ { BARCODE_UPCA_CC, -1, -1, "123456789013", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" },
        /* 44*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, -1, "123456789013", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" }, /* Still checked */
        /* 45*/ { BARCODE_UPCA_CC, -1, -1, "123456789012", "[20]1A", ZINT_ERROR_INVALID_DATA, "", "" }, /* AI (20) should be 2 nos. */
        /* 46*/ { BARCODE_UPCA_CC, GS1NOCHECK_MODE, -1, "123456789012", "[20]1A", 0, "123456789012", "" },
        /* 47*/ { BARCODE_UPCA_CC, -1, -1, "123456789012+123", "[20]12", 0, "123456789012+00123", "" },
        /* 48*/ { BARCODE_UPCA_CC, -1, BARCODE_CONTENT_SEGS, "123456789012+123", "[20]12", 0, "123456789012+00123", "012345678901200123|2012" },
        /* 49*/ { BARCODE_UPCE_CC, -1, -1, "123456", "[20]12", 0, "01234565", "" },
        /* 50*/ { BARCODE_UPCE_CC, -1, BARCODE_CONTENT_SEGS, "123456", "[20]12", 0, "01234565", "0012345000065|2012" },
        /* 51*/ { BARCODE_UPCE_CC, -1, -1, "1234567", "[20]12", 0, "12345670", "" },
        /* 52*/ { BARCODE_UPCE_CC, -1, -1, "12345670", "[20]12", 0, "12345670", "" },
        /* 53*/ { BARCODE_UPCE_CC, -1, -1, "12345671", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" },
        /* 54*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, -1, "12345671", "[20]12", ZINT_ERROR_INVALID_CHECK, "", "" }, /* Still checked */
        /* 55*/ { BARCODE_UPCE_CC, -1, -1, "12345670", "[20]12", 0, "12345670", "" }, /* Check digit can now be given for UPCE_CC, like UPCA_CC */
        /* 56*/ { BARCODE_UPCE_CC, -1, -1, "1234567", "[20]1A", ZINT_ERROR_INVALID_DATA, "", "" }, /* AI (20) should be 2 nos. */
        /* 57*/ { BARCODE_UPCE_CC, GS1NOCHECK_MODE, -1, "1234567", "[20]1A", 0, "12345670", "" },
        /* 58*/ { BARCODE_UPCE_CC, -1, BARCODE_CONTENT_SEGS, "1234567+2", "[20]12", 0, "12345670+02", "012345600007002|2012" },
        /* 59*/ { BARCODE_DBAR_STK_CC, -1, -1, "12345678901231", "[20]12", 0, "", "" }, /* No HRT for stacked symbologies */
        /* 60*/ { BARCODE_DBAR_STK_CC, -1, BARCODE_CONTENT_SEGS, "12345678901231", "[20]12", 0, "", "0112345678901231|2012" }, /* But have content segs */
        /* 61*/ { BARCODE_DBAR_OMNSTK_CC, -1, -1, "12345678901231", "[20]12", 0, "", "" },
        /* 62*/ { BARCODE_DBAR_OMNSTK_CC, -1, BARCODE_CONTENT_SEGS, "12345678901231", "[20]12", 0, "", "0112345678901231|2012" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, composite_length, ret;
    struct zint_symbol *symbol = NULL;
    int expected_length, expected_content_length;

    testStartSymbol(p_ctx->func_name, &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/,
                    -1 /*option_1*/, -1 /*option_2*/, -1 /*option_3*/, data[i].output_options,
                    data[i].data, -1, debug);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        symbol->input_mode |= GS1SYNTAXENGINE_MODE;
        strcpy(symbol->primary, data[i].data);

        expected_length = (int) strlen(data[i].expected);
        expected_content_length = (int) strlen(data[i].expected_content);

        composite_length = (int) strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, TCU(data[i].composite), composite_length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, data[i].ret, ret, symbol->errtxt);

        assert_equal(symbol->text_length, expected_length, "i:%d text_length %d != expected_length %d\n",
                    i, symbol->text_length, expected_length);
        assert_zero(strcmp((const char *) symbol->text, data[i].expected), "i:%d strcmp(%s, %s) != 0\n",
                    i, symbol->text, data[i].expected);
        if (symbol->output_options & BARCODE_CONTENT_SEGS) {
            assert_nonnull(symbol->content_segs, "i:%d content_segs NULL\n", i);
            assert_nonnull(symbol->content_segs[0].source, "i:%d content_segs[0].source NULL\n", i);
            assert_equal(symbol->content_segs[0].length, expected_content_length,
                        "i:%d content_segs[0].length %d != expected_content_length %d\n",
                        i, symbol->content_segs[0].length, expected_content_length);
            assert_zero(memcmp(symbol->content_segs[0].source, data[i].expected_content, expected_content_length),
                        "i:%d memcmp(%.*s, %s, %d) != 0\n",
                        i, symbol->content_segs[0].length, symbol->content_segs[0].source, data[i].expected_content,
                        expected_content_length);
        } else {
            assert_null(symbol->content_segs, "i:%d content_segs not NULL\n", i);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_gs1_verify(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int input_mode;
        const char *data;
        int ret;
        const char *expected;
        const char *expected_errtxt;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /*  0*/ { -1, "", ZINT_ERROR_INVALID_DATA, "", "264: Data does not start with an AI" },
        /*  1*/ { -1, "A", ZINT_ERROR_INVALID_DATA, "", "264: Data does not start with an AI" },
        /*  2*/ { -1, "[", ZINT_ERROR_INVALID_DATA, "", "268: Failed to parse AI data" },
        /*  3*/ { -1, "[]12", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: " },
        /*  4*/ { -1, "[1]12", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: 1" },
        /*  5*/ { -1, "[242]123456[1]12", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: 1" },
        /*  6*/ { -1, "[12345]12", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: 12345" },
        /*  7*/ { -1, "[9999]1234", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: 9999" },
        /*  8*/ { -1, "[[01]]1234", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: (01" },
        /*  9*/ { GS1PARENS_MODE, "((01))1234", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: (01" },
        /* 10*/ { -1, "[1A]12", ZINT_ERROR_INVALID_DATA, "", "268: Unrecognised AI: 1A" },
        /* 11*/ { -1, "[10]", ZINT_ERROR_INVALID_DATA, "", "268: Failed to parse AI data" },
        /* 12*/ { -1, "[90]\012", ZINT_ERROR_INVALID_DATA, "", "267: AI (90): A non-CSET 82 character was found where a CSET 82 character is expected. (90)|\\x0A|" },
        /* 13*/ { -1, "[00]123456789012345678", ZINT_ERROR_INVALID_DATA, "", "267: AI (00): The numeric check digit is incorrect. (00)12345678901234567|8|" },
        /* 14*/ { -1, "[00]123456789012345675", 0, "00123456789012345675", "" },
        /* 15*/ { GS1PARENS_MODE, "(00)123456789012345675", 0, "00123456789012345675", "" },
        /* 16*/ { -1, "[00]12345678901234567", ZINT_ERROR_INVALID_DATA, "", "268: AI (00) value is too short" },
        /* 17*/ { -1, "[00]1234567890123456789", ZINT_ERROR_INVALID_DATA, "", "268: AI (00) value is too long" },
        /* 18*/ { -1, "[3910]123123456789012345", ZINT_ERROR_INVALID_DATA, "", "267: AI (3910): A valid ISO 4217 three-digit currency code is required. (3910)|123|" },
        /* 19*/ { -1, "[3910]997123456789012345", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (3910) are not satisfied: 8020" },
        /* 20*/ { -1, "[3910]997123456789012345[8020]REF123", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (8020) are not satisfied: 415" },
        /* 21*/ { -1, "[3910]997123456789012345[8020]REF123[415]1234567890123", ZINT_ERROR_INVALID_DATA, "", "267: AI (415): The numeric check digit is incorrect. (415)123456789012|3|" },
        /* 22*/ { -1, "[3910]997123456789012345[8020]REF123[415]1234567890128", 0, "3910997123456789012345\0358020REF123\0354151234567890128", "" },
        /* 23*/ { GS1PARENS_MODE, "(3910)997123456789012345(8020)REF123(415)1234567890128", 0, "3910997123456789012345\0358020REF123\0354151234567890128", "" },
        /* 24*/ { -1, "[402]13131313131313132", ZINT_ERROR_INVALID_DATA, "", "267: AI (402): The numeric check digit is incorrect. (402)1313131313131313|2|" },
        /* 25*/ { -1, "[402]13131313131313130", 0, "40213131313131313130", "" },
        /* 26*/ { -1, "[4309]1234567890123456789A", ZINT_ERROR_INVALID_DATA, "", "267: AI (4309): A non-digit character was found where a digit is expected. (4309)1234567890123456789|A|" },
        /* 27*/ { -1, "[7006]200132", ZINT_ERROR_INVALID_DATA, "", "267: AI (7006): The date contains an illegal day of the month. (7006)2001|32|" },
        /* 28*/ { -1, "[7006]200131", ZINT_ERROR_INVALID_DATA, "7006200131", "268: Required AIs for AI (7006) are not satisfied: 01,02" },
        /* 29*/ { -1, "[7006]200131[01]12345678901231", 0, "7006200131\0350112345678901231", "" },
        /* 30*/ { -1, "[8001]12345678901234", ZINT_ERROR_INVALID_DATA, "", "267: AI (8001): The winding direction must be either \"0\", \"1\" or \"9\". (8001)123456789012|3|" },
        /* 31*/ { -1, "[8001]12345678901294", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (8001) are not satisfied: 01" },
        /* 32*/ { -1, "[8001]12345678901294[01]12345678901231", 0, "800112345678901294\0350112345678901231", "" },
        /* 33*/ { -1, "[8004]abcdefghijklmnopqrstuvwxyz1234", ZINT_ERROR_INVALID_DATA, "", "267: AI (8004): The GS1 Company Prefix is invalid. (8004)|a|bcdefghijklmnopqrstuvwxyz1234" },
        /* 34*/ { -1, "[8004]123", ZINT_ERROR_INVALID_DATA, "", "267: AI (8004): The component is shorter than the minimum length GS1 Company Prefix. (8004)|123|" },
        /* 35*/ { -1, "[8004]1234efghijklmnopqrstuvwxyz1234", 0, "80041234efghijklmnopqrstuvwxyz1234", "" },
        /* 36*/ { -1, "[8007]abcdefghijklmnopqrstuvwxyz12345678", ZINT_ERROR_INVALID_DATA, "", "267: AI (8007): The IBAN must start with a valid ISO 3166 two-character country code. (8007)|ab|cdefghijklmnopqrstuvwxyz12345678" },
        /* 37*/ { -1, "[8007]AD95EFGHIJKLMNOPQRSTUVWXYZ12345678", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (8007) are not satisfied: 415" },
        /* 38*/ { -1, "[8007]AD95EFGHIJKLMNOPQRSTUVWXYZ12345678[415]1234567890128", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (415) are not satisfied: 8020" },
        /* 39*/ { -1, "[8007]AD95EFGHIJKLMNOPQRSTUVWXYZ12345678[415]1234567890128[8020]REF123", 0, "8007AD95EFGHIJKLMNOPQRSTUVWXYZ12345678\03541512345678901288020REF123", "" },
        /* 40*/ { -1, "[8030]-1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXY^", ZINT_ERROR_INVALID_DATA, "", "268: AI (8030) contains illegal ^ character" },
        /* 41*/ { -1, "[8030]-1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXYZ", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (8030) are not satisfied: 00,01+21,253,255,8003,8004,8006+21,8010+8011,8017,8018" },
        /* 42*/ { -1, "[8030]-1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXYZ[8018]123456789012345675", 0, "8030-1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXYZ\0358018123456789012345675", "" },
        /* 43*/ { -1, "[01]12345678901234[7006]200101", ZINT_ERROR_INVALID_DATA, "", "267: AI (01): The numeric check digit is incorrect. (01)1234567890123|4|" },
        /* 44*/ { -1, "[01]12345678901231[7006]200101", 0, "01123456789012317006200101", "" },
        /* 45*/ { -1, "[253]12345678901284[01]12345678901231[3901]12345678901234[20]12", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (3901) are not satisfied: 255,8020" },
        /* 46*/ { -1, "[253]12345678901284[01]12345678901231[3901]12345678901234[20]12[255]1234567890128", ZINT_ERROR_INVALID_DATA, "", "268: It is invalid to pair AI (01) with AI (255)" },
        /* 47*/ { -1, "[253]12345678901284[01]12345678901231[3901]12345678901234[20]12[8020]REF123", ZINT_ERROR_INVALID_DATA, "", "268: Required AIs for AI (8020) are not satisfied: 415" },
        /* 48*/ { -1, "[253]12345678901284[01]12345678901231[3901]12345678901234[20]12[8020]REF123[415]1234567890128", 0, "25312345678901284\0350112345678901231390112345678901234\03520128020REF123\0354151234567890128", "" },
        /* 49*/ { -1, "[253]12345678901284[01]12345678901231[3901]12345678901234[20]12[8020]REF123[415]1234567890128[90]123", 0, "25312345678901284\0350112345678901231390112345678901234\03520128020REF123\035415123456789012890123", "" },
        /* 50*/ { -1, "[8001]12345678901294[01]12345678901231|[8012]VER1", 0, "800112345678901294\0350112345678901231|8012VER1", "" }, /* Composite */
        /* 51*/ { -1, "[8001]12345678901294[01]12345678901231[415]1234567890128|[8020]ABCDEFGHIJKLMNOPQRSTUVXWY", 0, "800112345678901294\03501123456789012314151234567890128|8020ABCDEFGHIJKLMNOPQRSTUVXWY", "" },
        /* 52*/ { -1, "[8001]12345678901294[01]12345678901231|[415]1234567890128[8020]ABCDEFGHIJKLMNOPQRSTUVXWY", 0, "800112345678901294\0350112345678901231|41512345678901288020ABCDEFGHIJKLMNOPQRSTUVXWY", "" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol = NULL;
    int reduced_length = 0; /* Suppress clang -fsanitize=memory false positive */

    char reduced[1024] = {0}; /* Suppress clang -fsanitize=memory false positive */
    char escaped[1024];
    char escaped2[1024];

    testStartSymbol(p_ctx->func_name, &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, BARCODE_GS1_128, data[i].input_mode, -1 /*eci*/,
                                    -1 /*option_1*/, -1 /*option_2*/, -1 /*option_3*/, -1 /*output_options*/,
                                    data[i].data, -1, debug);
        symbol->input_mode |= GS1SYNTAXENGINE_MODE;

        ret = zint_gs1_verify(symbol, ZUCP(data[i].data), length, ZUCP(reduced), &reduced_length);

        if (p_ctx->generate) {
            printf("        /*%3d*/ { %s, \"%s\", %s, \"%s\", \"%s\" },\n",
                        i, testUtilInputModeName(data[i].input_mode),
                        testUtilEscape(data[i].data, length, escaped, sizeof(escaped)), testUtilErrorName(ret),
                        data[i].expected, symbol->errtxt);
        } else {
            assert_equal(ret, data[i].ret, "i:%d ret %d != %d (length %d \"%s\") (%s)\n",
                        i, ret, data[i].ret, length, data[i].data, symbol->errtxt);

            if (ret < ZINT_ERROR) {
                const int expected_length = (int) strlen(data[i].expected);
                assert_zero(strcmp(reduced, data[i].expected), "i:%d strcmp(%s, %s) != 0\n",
                            i, testUtilEscape(reduced, (int) strlen(reduced), escaped, (int) sizeof(escaped)),
                            testUtilEscape(data[i].expected, expected_length, escaped2, (int) sizeof(escaped2)));
                assert_equal(reduced_length, (int) strlen(reduced), "i:%d reduced_length %d != strlen %d\n",
                            i, reduced_length, (int) strlen(reduced));
            }
            assert_zero(strcmp(symbol->errtxt, data[i].expected_errtxt), "i:%d strcmp(%s, %s) != 0\n",
                        i, symbol->errtxt, data[i].expected_errtxt);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_gs1_digital_link(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int input_mode;
        const char *data;
        int ret;
        const char *expected;
        const char *expected_errtxt;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /* 53*/ { -1, "https://", ZINT_ERROR_INVALID_DATA, "", "268: URI must contain a domain and path info" },
        /* 54*/ { -1, "HTTPS://", ZINT_ERROR_INVALID_DATA, "", "268: URI must contain a domain and path info" },
        /* 55*/ { -1, "http://", ZINT_ERROR_INVALID_DATA, "", "268: URI must contain a domain and path info" },
        /* 56*/ { -1, "HTTP://", ZINT_ERROR_INVALID_DATA, "", "268: URI must contain a domain and path info" },
        /* 57*/ { -1, "Https://", ZINT_ERROR_INVALID_DATA, "", "265: Data does not start with an AI or a Digital Link URI" },
        /* 58*/ { -1, "http:/", ZINT_ERROR_INVALID_DATA, "", "265: Data does not start with an AI or a Digital Link URI" },
        /* 59*/ { -1, "https://example.com/123", ZINT_ERROR_INVALID_DATA, "", "268: No GS1 DL keys found in path info" },
        /* 60*/ { -1, "https://example.com/01/12345678901231", 0, "https://example.com/01/12345678901231", "" },
        /* 61*/ { -1, "https://example.com/gtin/12345678901231", ZINT_ERROR_INVALID_DATA, "", "268: No GS1 DL keys found in path info" },
        /* 62*/ { -1, "https://example.com/gtin/01/12345678901231", 0, "https://example.com/gtin/01/12345678901231", "" },
        /* 63*/ { -1, "https://example.com/01/12345678901231/90/123", ZINT_ERROR_INVALID_DATA, "", "268: The AIs in the path are not a valid key-qualifier sequence for the key" },
        /* 64*/ { -1, "https://example.com/01/12345678901231/10/123", 0, "https://example.com/01/12345678901231/10/123", "" },
        /* 65*/ { -1, "https://example.com/01/12345678901231/10/12+3", 0, "https://example.com/01/12345678901231/10/12+3", "" },
        /* 66*/ { -1, "https://example.com/01/12345678901231/10/12%2B3", 0, "https://example.com/01/12345678901231/10/12%2B3", "" },
        /* 67*/ { -1, "http://example.com/01/09520123456788/22/2A?linkType=gs1:traceability", 0, "http://example.com/01/09520123456788/22/2A?linkType=gs1:traceability", "" },
        /* 68*/ { -1, "https://example.com/01/12345678901231/10/12?3", 0, "https://example.com/01/12345678901231/10/12?3", "" },
        /* 69*/ { -1, "https://example.com/01/12345678901231/10/12=3", 0, "https://example.com/01/12345678901231/10/12=3", "" },
        /* 70*/ { -1, "https://example.com/01/12345678901231/10/1?2=3", ZINT_ERROR_INVALID_DATA, "", "268: Unknown AI (2) in query parameters" },
        /* 71*/ { -1, "https://example.com/01/12345678901231/10/1%3F2=3", 0, "https://example.com/01/12345678901231/10/1%3F2=3", "" },
        /* 72*/ { -1, "https://example.com/01/12345678901231?3103=000195", 0, "https://example.com/01/12345678901231?3103=000195", "" },
        /* 73*/ { -1, "https://example.com/01/12345678901231|(20)12", ZINT_ERROR_INVALID_DATA, "", "268: URI contains illegal characters" },
        /* 74*/ { -1, "https://example.com/01/12345678901231|", ZINT_ERROR_INVALID_DATA, "", "268: URI contains illegal characters" },
        /* 75*/ { -1, "https://example.com/01/12345678901231(", ZINT_ERROR_INVALID_DATA, "", "268: AI (01) value is too long" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol = NULL;
    int reduced_length = 0; /* Suppress clang -fsanitize=memory false positive */

    char reduced[1024] = {0}; /* Suppress clang -fsanitize=memory false positive */
    char escaped[1024];
    char escaped2[1024];

    testStartSymbol(p_ctx->func_name, &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, BARCODE_QRCODE, GS1_MODE | GS1SYNTAXENGINE_MODE, -1 /*eci*/,
                                    -1 /*option_1*/, -1 /*option_2*/, -1 /*option_3*/, -1 /*output_options*/,
                                    data[i].data, -1, debug);
        if (data[i].input_mode != -1) {
            symbol->input_mode |= data[i].input_mode;
        }

        ret = zint_gs1_verify(symbol, ZUCP(data[i].data), length, ZUCP(reduced), &reduced_length);

        if (p_ctx->generate) {
            printf("        /*%3d*/ { %s, \"%s\", %s, \"%s\", \"%s\" },\n",
                        i, testUtilInputModeName(data[i].input_mode),
                        testUtilEscape(data[i].data, length, escaped, sizeof(escaped)), testUtilErrorName(ret),
                        data[i].expected, symbol->errtxt);
        } else {
            assert_equal(ret, data[i].ret, "i:%d ret %d != %d (length %d \"%s\") (%s)\n",
                        i, ret, data[i].ret, length, data[i].data, symbol->errtxt);

            if (ret < ZINT_ERROR) {
                const int expected_length = (int) strlen(data[i].expected);
                assert_zero(strcmp(reduced, data[i].expected), "i:%d strcmp(%s, %s) != 0\n",
                            i, testUtilEscape(reduced, (int) strlen(reduced), escaped, (int) sizeof(escaped)),
                            testUtilEscape(data[i].expected, expected_length, escaped2, (int) sizeof(escaped2)));
                assert_equal(reduced_length, (int) strlen(reduced), "i:%d reduced_length %d != strlen %d\n",
                            i, reduced_length, (int) strlen(reduced));
            }
            assert_zero(strcmp(symbol->errtxt, data[i].expected_errtxt), "i:%d strcmp(%s, %s) != 0\n",
                        i, symbol->errtxt, data[i].expected_errtxt);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

/* Check non-forced GS1 compliant symbologies (see gs1_compliant() in library.c) */
static void test_non_forced(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        const char *data;
        int input_mode;
        int output_options;
        int ret;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /*  0*/ { BARCODE_AZTEC, "[01]12345678901231", GS1_MODE, -1, 0 },
        /*  1*/ { BARCODE_AZTEC, "(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /*  2*/ { BARCODE_AZTEC, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*  3*/ { BARCODE_AZTEC, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*  4*/ { BARCODE_AZTEC, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*  5*/ { BARCODE_AZTEC, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*  6*/ { BARCODE_AZTEC, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*  7*/ { BARCODE_AZTEC, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*  8*/ { BARCODE_AZTEC, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*  9*/ { BARCODE_AZTEC, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 10*/ { BARCODE_AZTEC, "[01]1234567890123", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 11*/ { BARCODE_AZTEC, "[01]1234567890123", GS1_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 12*/ { BARCODE_AZTEC, "[01]12345678901231", GS1_MODE, READER_INIT, ZINT_ERROR_INVALID_OPTION },
        /* 13*/ { BARCODE_AZTEC, "[01]12345678901231[10]()", GS1_MODE, -1, 0 },
        /* 14*/ { BARCODE_AZTEC, "(01)12345678901231(10)()", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 15*/ { BARCODE_AZTEC, "(01)12345678901231(10)\\(\\)", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 16*/ { BARCODE_AZTEC, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 17*/ { BARCODE_AZTEC, "1234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, ZINT_ERROR_INVALID_DATA }, /* Must still begin with AI */
        /* 18*/ { BARCODE_CODABLOCKF, "[01]12345678901231", GS1_MODE, -1, ZINT_ERROR_INVALID_OPTION }, /* Codablock-F does not support GS1 */
        /* 19*/ { BARCODE_CODABLOCKF, "[01]12345678901231", GS1_MODE | GS1NOCHECK_MODE, -1, ZINT_ERROR_INVALID_OPTION },
        /* 20*/ { BARCODE_CODABLOCKF, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_OPTION },
        /* 21*/ { BARCODE_CODEONE, "[01]12345678901231", GS1_MODE, -1, 0 },
        /* 22*/ { BARCODE_CODEONE, "(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 23*/ { BARCODE_CODEONE, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 24*/ { BARCODE_CODEONE, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 25*/ { BARCODE_CODEONE, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 26*/ { BARCODE_CODEONE, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 27*/ { BARCODE_CODEONE, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 28*/ { BARCODE_CODEONE, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 29*/ { BARCODE_CODEONE, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 30*/ { BARCODE_CODEONE, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 31*/ { BARCODE_CODEONE, "[01]1234567890123", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 32*/ { BARCODE_CODEONE, "[01]1234567890123", GS1_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 33*/ { BARCODE_CODEONE, "[10]()[01]12345678901231", GS1_MODE, -1, 0 },
        /* 34*/ { BARCODE_CODEONE, "(10)()(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 35*/ { BARCODE_CODEONE, "(10)\\(\\)(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 36*/ { BARCODE_CODEONE, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 37*/ { BARCODE_CODE16K, "[01]12345678901231", GS1_MODE, -1, 0 },
        /* 38*/ { BARCODE_CODE16K, "(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 39*/ { BARCODE_CODE16K, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 40*/ { BARCODE_CODE16K, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 41*/ { BARCODE_CODE16K, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 42*/ { BARCODE_CODE16K, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 43*/ { BARCODE_CODE16K, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 44*/ { BARCODE_CODE16K, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 45*/ { BARCODE_CODE16K, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 46*/ { BARCODE_CODE16K, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 47*/ { BARCODE_CODE16K, "[01]1234567890123", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 48*/ { BARCODE_CODE16K, "[01]1234567890123", GS1_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 49*/ { BARCODE_CODE16K, "[01]12345678901231", GS1_MODE, READER_INIT, ZINT_ERROR_INVALID_OPTION },
        /* 50*/ { BARCODE_CODE16K, "[01]12345678901231[10]()", GS1_MODE, -1, 0 },
        /* 51*/ { BARCODE_CODE16K, "(01)12345678901231(10)()", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 52*/ { BARCODE_CODE16K, "(01)12345678901231(10)\\(\\)", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 53*/ { BARCODE_CODE16K, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 54*/ { BARCODE_CODE49, "[01]12345678901231", GS1_MODE, -1, 0 },
        /* 55*/ { BARCODE_CODE49, "(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 56*/ { BARCODE_CODE49, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 57*/ { BARCODE_CODE49, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 58*/ { BARCODE_CODE49, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 59*/ { BARCODE_CODE49, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 60*/ { BARCODE_CODE49, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 61*/ { BARCODE_CODE49, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 62*/ { BARCODE_CODE49, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 63*/ { BARCODE_CODE49, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 64*/ { BARCODE_CODE49, "[01]1234567890123", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 65*/ { BARCODE_CODE49, "[01]1234567890123", GS1_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 66*/ { BARCODE_CODE49, "[01]12345678901231[10]()", GS1_MODE, -1, 0 },
        /* 67*/ { BARCODE_CODE49, "(01)12345678901231(10)()", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 68*/ { BARCODE_CODE49, "(01)12345678901231(10)\\(\\)", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 69*/ { BARCODE_CODE49, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 70*/ { BARCODE_DATAMATRIX, "[01]12345678901231", GS1_MODE, -1, 0 },
        /* 71*/ { BARCODE_DATAMATRIX, "(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 72*/ { BARCODE_DATAMATRIX, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 73*/ { BARCODE_DATAMATRIX, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 74*/ { BARCODE_DATAMATRIX, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 75*/ { BARCODE_DATAMATRIX, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 76*/ { BARCODE_DATAMATRIX, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 77*/ { BARCODE_DATAMATRIX, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 78*/ { BARCODE_DATAMATRIX, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 79*/ { BARCODE_DATAMATRIX, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 80*/ { BARCODE_DATAMATRIX, "[01]1234567890123", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 81*/ { BARCODE_DATAMATRIX, "[01]1234567890123", GS1_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 82*/ { BARCODE_DATAMATRIX, "[01]12345678901231", GS1_MODE, READER_INIT, ZINT_ERROR_INVALID_OPTION },
        /* 83*/ { BARCODE_DATAMATRIX, "[01]12345678901231[10]()", GS1_MODE, -1, 0 },
        /* 84*/ { BARCODE_DATAMATRIX, "(01)12345678901231(10)()", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 85*/ { BARCODE_DATAMATRIX, "(01)12345678901231(10)\\(\\)", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 86*/ { BARCODE_DATAMATRIX, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 87*/ { BARCODE_DOTCODE, "[01]12345678901231", GS1_MODE, -1, 0 },
        /* 88*/ { BARCODE_DOTCODE, "(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /* 89*/ { BARCODE_DOTCODE, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 90*/ { BARCODE_DOTCODE, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 91*/ { BARCODE_DOTCODE, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 92*/ { BARCODE_DOTCODE, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 93*/ { BARCODE_DOTCODE, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 94*/ { BARCODE_DOTCODE, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 95*/ { BARCODE_DOTCODE, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 96*/ { BARCODE_DOTCODE, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 97*/ { BARCODE_DOTCODE, "[01]1234567890123", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /* 98*/ { BARCODE_DOTCODE, "[01]1234567890123", GS1_MODE | GS1NOCHECK_MODE, -1, 0 },
        /* 99*/ { BARCODE_DOTCODE, "[01]12345678901231", GS1_MODE, READER_INIT, 0 }, /* Reader Init permissible with default GS1 mode */
        /*100*/ { BARCODE_DOTCODE, "[10]()[01]12345678901231", GS1_MODE, -1, 0 },
        /*101*/ { BARCODE_DOTCODE, "(10)()(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*102*/ { BARCODE_DOTCODE, "(10)\\(\\)(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /*103*/ { BARCODE_DOTCODE, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*104*/ { BARCODE_QRCODE, "[01]12345678901231", GS1_MODE, -1, 0 },
        /*105*/ { BARCODE_QRCODE, "(01)12345678901231", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /*106*/ { BARCODE_QRCODE, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*107*/ { BARCODE_QRCODE, "(01)12345678901234", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*108*/ { BARCODE_QRCODE, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*109*/ { BARCODE_QRCODE, "(01)123456789012345", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*110*/ { BARCODE_QRCODE, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*111*/ { BARCODE_QRCODE, "(01)12345678901234A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*112*/ { BARCODE_QRCODE, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*113*/ { BARCODE_QRCODE, "(01)1234567890123A", GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*114*/ { BARCODE_QRCODE, "[01]1234567890123", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*115*/ { BARCODE_QRCODE, "[01]1234567890123", GS1_MODE | GS1NOCHECK_MODE, -1, 0 },
        /*116*/ { BARCODE_QRCODE, "[02]12345678901231[10]()[37]12345678[00]123456789012345675", GS1_MODE, -1, 0 },
        /*117*/ { BARCODE_QRCODE, "(02)12345678901231(10)()(37)12345678(00)123456789012345675", GS1_MODE | GS1PARENS_MODE, -1, ZINT_ERROR_INVALID_DATA },
        /*118*/ { BARCODE_QRCODE, "(02)12345678901231(10)\\(\\)(37)12345678(00)123456789012345675", GS1_MODE | GS1PARENS_MODE, -1, 0 },
        /*119*/ { BARCODE_QRCODE, "1234", GS1_MODE, -1, ZINT_ERROR_INVALID_DATA },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol = NULL;

    testStartSymbol(p_ctx->func_name, &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/,
                                    -1 /*option_1*/, -1 /*option_2*/, -1 /*option_3*/, data[i].output_options,
                                    data[i].data, -1, debug);

        symbol->input_mode |= GS1SYNTAXENGINE_MODE;

        ret = ZBarcode_Encode(symbol, TCU(data[i].data), length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n",
                    i, ret, data[i].ret, symbol->errtxt);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

/* Check GS1_128-based and DBAR_EXP-based symbologies */
static void test_gs1_128_dbar_exp(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        const char *data;
        const char *composite;
        int ret;
        const char *expected_errtxt;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    static const struct item data[] = {
        /*  0*/ { BARCODE_GS1_128, -1, "[01]12345678901231", "", 0, "" },
        /*  1*/ { BARCODE_GS1_128, -1, "[01]12345678901234", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): The numeric check digit is incorrect. (01)1234567890123|4|" },
        /*  2*/ { BARCODE_GS1_128, -1, "[01]123456789012345", "", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too long" },
        /*  3*/ { BARCODE_GS1_128, -1, "[01]1234567890123", "", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /*  4*/ { BARCODE_GS1_128, -1, "[01]12345678901231[20]1", "", ZINT_ERROR_INVALID_DATA, "Error 268: AI (20) value is too short" },
        /*  5*/ { BARCODE_GS1_128, -1, "[04]1234[05]12345[06]123456", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: 04" },
        /*  6*/ { BARCODE_GS1_128, -1, "[01]1234567890123A", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): A non-digit character was found where a digit is expected. (01)1234567890123|A|" },
        /*  7*/ { BARCODE_GS1_128, -1, "[01]1234567890123.", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): A non-digit character was found where a digit is expected. (01)1234567890123|.|" },
        /*  8*/ { BARCODE_GS1_128, -1, "[01]1234567890123\177", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): A non-digit character was found where a digit is expected. (01)1234567890123|\\x7F|" },
        /*  9*/ { BARCODE_GS1_128, -1, "[01]1234567890123\200", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): A non-digit character was found where a digit is expected. (01)1234567890123|\\x80|" },
        /* 10*/ { BARCODE_GS1_128, -1, "0112345678901231", "", ZINT_ERROR_INVALID_DATA, "Error 264: Data does not start with an AI" },
        /* 11*/ { BARCODE_GS1_128, -1, "[01]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 12*/ { BARCODE_GS1_128, -1, "[01][20]12", "", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /* 13*/ { BARCODE_GS1_128, -1, "[0]123", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: 0" },
        /* 14*/ { BARCODE_GS1_128, -1, "[0]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: 0" },
        /* 15*/ { BARCODE_GS1_128, -1, "[]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: " },
        /* 16*/ { BARCODE_GS1_128, -1, "[01]12345678901231[]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: " },
        /* 17*/ { BARCODE_GS1_128, -1, "[123456789012345]123", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: 123456789012345" },
        /* 18*/ { BARCODE_GS1_128, GS1PARENS_MODE, "(91)AB[", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (91): A non-CSET 82 character was found where a CSET 82 character is expected. (91)AB|[|" },
        /* 19*/ { BARCODE_GS1_128, GS1PARENS_MODE, "(91)12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789[", "", ZINT_ERROR_INVALID_DATA, "Error 268: AI (91): A non-CSET 82 character was found where a CSET 82 character is expected." },
        /* 20*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901231", "[20]12", 0, "" },
        /* 21*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901234", "[20]12", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): The numeric check digit is incorrect. (01)1234567890123|4|" },
        /* 22*/ { BARCODE_GS1_128_CC, -1, "[01]123456789012345", "[20]12", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too long" },
        /* 23*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901231", "[20]123", ZINT_ERROR_INVALID_DATA, "Error 268: AI (20) value is too long" },
        /* 24*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901231", "[20]1A", ZINT_ERROR_INVALID_DATA, "Error 267: AI (20): A non-digit character was found where a digit is expected. (20)1|A|" },
        /* 25*/ { BARCODE_GS1_128_CC, -1, "[01]1234567890121", "[20]", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /* 26*/ { BARCODE_GS1_128_CC, -1, "[01]1234567890121", "[2]12", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /* 27*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901231", "[90]12]34", ZINT_ERROR_INVALID_DATA, "Error 267: AI (90): A non-CSET 82 character was found where a CSET 82 character is expected. (90)12|]|34" },
        /* 28*/ { BARCODE_GS1_128_CC, -1, "[01]12345678901231", "[90]12[34", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 29*/ { BARCODE_GS1_128_CC, -1, "https://example.com/01/12345678901231", "[20]12", ZINT_ERROR_INVALID_DATA, "Error 264: Data does not start with an AI" },
        /* 30*/ { BARCODE_DBAR_EXP, -1, "[01]12345678901231", "", 0, "" },
        /* 31*/ { BARCODE_DBAR_EXP, -1, "[01]12345678901231[10]123[11]12345", "", ZINT_ERROR_INVALID_DATA, "Error 268: AI (11) value is too short" },
        /* 32*/ { BARCODE_DBAR_EXP, -1, "[01]12345678901231[10]123[11]12345A", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (11): A non-digit character was found where a digit is expected. (11)12345|A|" },
        /* 33*/ { BARCODE_DBAR_EXP, -1, "011234567890121", "", ZINT_ERROR_INVALID_DATA, "Error 264: Data does not start with an AI" },
        /* 34*/ { BARCODE_DBAR_EXP, -1, "[10]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 35*/ { BARCODE_DBAR_EXP, -1, "[2]1", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: 2" },
        /* 36*/ { BARCODE_DBAR_EXP, -1, "[]1", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: " },
        /* 37*/ { BARCODE_DBAR_EXP, -1, "[]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: " },
        /* 38*/ { BARCODE_DBAR_EXP, -1, "[20]12[]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: " },
        /* 39*/ { BARCODE_DBAR_EXP, -1, "[90]12]34", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (90): A non-CSET 82 character was found where a CSET 82 character is expected. (90)12|]|34" },
        /* 40*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231[10]123[11]121212", "[21]ABC123[22]12345", 0, "" },
        /* 41*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231[10]123[11]1234", "[21]ABC123[22]12345", ZINT_ERROR_INVALID_DATA, "Error 268: AI (11) value is too short" },
        /* 42*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231[10]123[11]123456", "[21]ABC123[22]12345", ZINT_ERROR_INVALID_DATA, "Error 267: AI (11): The date contains an illegal month of the year. (11)12|34|56" },
        /* 43*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231[10]123[11]121212", "[21]ABC123[22]12345[30]123456789", ZINT_ERROR_INVALID_DATA, "Error 268: AI (30) value is too long" },
        /* 44*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231[10]123[11]121212", "[21]ABC123[22]12345[30]1234567A", ZINT_ERROR_INVALID_DATA, "Error 267: AI (30): A non-digit character was found where a digit is expected. (30)1234567|A|" },
        /* 45*/ { BARCODE_DBAR_EXP_CC, -1, "[01]1234567890121\177", "[20]12", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): A non-digit character was found where a digit is expected. (01)1234567890121|\\x7F|" },
        /* 46*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231", "[20]1\177", ZINT_ERROR_INVALID_DATA, "Error 267: AI (20): A non-digit character was found where a digit is expected. (20)1|\\x7F|" },
        /* 47*/ { BARCODE_DBAR_EXP_CC, -1, "[01]1234567890121", "2012", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /* 48*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231", "2012", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 49*/ { BARCODE_DBAR_EXP_CC, -1, "[01]1234567890121", "[10]", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /* 50*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231", "[10]", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 51*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231", "[2]1", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: 2" },
        /* 52*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231", "[]12", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: " },
        /* 53*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231", "[]", ZINT_ERROR_INVALID_DATA, "Error 268: Unrecognised AI: " },
        /* 54*/ { BARCODE_DBAR_EXP_CC, -1, "[01]12345678901231", "[90]12]34", ZINT_ERROR_INVALID_DATA, "Error 267: AI (90): A non-CSET 82 character was found where a CSET 82 character is expected. (90)12|]|34" },
        /* 55*/ { BARCODE_DBAR_EXPSTK, -1, "[01]12345678901231", "", 0, "" },
        /* 56*/ { BARCODE_DBAR_EXPSTK, -1, "[01]12345678901231[10]123[11]1234", "", ZINT_ERROR_INVALID_DATA, "Error 268: AI (11) value is too short" },
        /* 57*/ { BARCODE_DBAR_EXPSTK, -1, "[01]12345678901231[10]123[11]12345A", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (11): A non-digit character was found where a digit is expected. (11)12345|A|" },
        /* 58*/ { BARCODE_DBAR_EXPSTK, -1, "[01]1234567890121\177", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): A non-digit character was found where a digit is expected. (01)1234567890121|\\x7F|" },
        /* 59*/ { BARCODE_DBAR_EXPSTK, -1, "[01]1234567890121\200", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (01): A non-digit character was found where a digit is expected. (01)1234567890121|\\x80|" },
        /* 60*/ { BARCODE_DBAR_EXPSTK, -1, "011234567890121", "", ZINT_ERROR_INVALID_DATA, "Error 264: Data does not start with an AI" },
        /* 61*/ { BARCODE_DBAR_EXPSTK, -1, "[01]", "", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 62*/ { BARCODE_DBAR_EXPSTK, -1, "[90]12]34", "", ZINT_ERROR_INVALID_DATA, "Error 267: AI (90): A non-CSET 82 character was found where a CSET 82 character is expected. (90)12|]|34" },
        /* 63*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231[10]123[11]121212", "[21]ABC123[22]12345", 0, "" },
        /* 64*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231[10]123[11]1234", "[21]ABC123[22]12345", ZINT_ERROR_INVALID_DATA, "Error 268: AI (11) value is too short" },
        /* 65*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231[10]123[11]123456", "[21]ABC123[22]12345", ZINT_ERROR_INVALID_DATA, "Error 267: AI (11): The date contains an illegal month of the year. (11)12|34|56" },
        /* 66*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231[10]123[11]121212", "[21]ABC123[22]12345[30]123456789", ZINT_ERROR_INVALID_DATA, "Error 268: AI (30) value is too long" },
        /* 67*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231[10]123[11]121212", "[21]ABC123[22]12345[30]1234567A", ZINT_ERROR_INVALID_DATA, "Error 267: AI (30): A non-digit character was found where a digit is expected. (30)1234567|A|" },
        /* 68*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]1234567890121", "[20]1\177", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /* 69*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231", "[20]1\200", ZINT_ERROR_INVALID_DATA, "Error 267: AI (20): A non-digit character was found where a digit is expected. (20)1|\\x80|" },
        /* 70*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231", "2012", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 71*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]1234567890121", "[235]", ZINT_ERROR_INVALID_DATA, "Error 268: AI (01) value is too short" },
        /* 72*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231", "[235]", ZINT_ERROR_INVALID_DATA, "Error 268: Failed to parse AI data" },
        /* 73*/ { BARCODE_DBAR_EXPSTK_CC, -1, "[01]12345678901231", "[90]12]34", ZINT_ERROR_INVALID_DATA, "Error 267: AI (90): A non-CSET 82 character was found where a CSET 82 character is expected. (90)12|]|34" },
    };
    const int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol = NULL;

    const char *text;

    testStartSymbol(p_ctx->func_name, &symbol);

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        if (data[i].composite[0]) {
            text = data[i].composite;
            strcpy(symbol->primary, data[i].data);
        } else {
            text = data[i].data;
        }
        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/,
                                    -1 /*option_1*/, -1 /*option_2*/, -1 /*option_3*/, -1 /*output_options*/,
                                    text, -1, debug);
        symbol->input_mode |= GS1SYNTAXENGINE_MODE;

        ret = ZBarcode_Encode(symbol, TCU(text), length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n",
                    i, ret, data[i].ret, symbol->errtxt);
        assert_zero(strcmp(symbol->errtxt, data[i].expected_errtxt), "i:%d strcmp(%s, %s) != 0\n",
                    i, symbol->errtxt, data[i].expected_errtxt);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

int main(int argc, char *argv[]) {

    testFunction funcs[] = { /* name, func */
        { "test_gs1_reduce", test_gs1_reduce },
        { "test_hrt", test_hrt },
        { "test_gs1_verify", test_gs1_verify },
        { "test_gs1_digital_link", test_gs1_digital_link },
        { "test_non_forced", test_non_forced },
        { "test_gs1_128_dbar_exp", test_gs1_128_dbar_exp },
    };

    testRun(argc, argv, funcs, ARRAY_SIZE(funcs));

    testReport();

    return 0;
}

/* vim: set ts=4 sw=4 et : */
