package java_test

import (
	"context"
	"path"
	"testing"

	"github.com/quay/claircore"
	"github.com/quay/claircore/java"
	"github.com/quay/claircore/test"
)

// TestScan runs the java scanner over some layers known to have java
// packages installed.
func TestScan(t *testing.T) {
	ctx, done := context.WithCancel(context.Background())
	defer done()
	for _, tc := range scanTable {
		t.Run(path.Base(tc.Name), tc.Run(ctx))
	}
}

var scanTable = []test.ScannerTestcase{
	{
		Domain: "docker.io",
		Name:   "tinkerpop/gremlin-console",
		Hash:   "sha256:e6e10dd7da4509f51dbdaf50f9d786f4ca0096ba14bfa7443f19d032e35f73f7",
		Want: []*claircore.Package{
			{
				Name:           "org.apache.tinkerpop:gremlin-groovy",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/ext/gremlin-groovy/lib/gremlin-groovy-3.4.8.jar",
				RepositoryHint: "sha1:9ae4c997e7b38ef6f6bc72c53412434743705866",
			},
			{
				Name:           "org.apache.tinkerpop:gremlin-groovy",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/ext/gremlin-groovy/plugin/gremlin-groovy-3.4.8.jar",
				RepositoryHint: "sha1:9ae4c997e7b38ef6f6bc72c53412434743705866",
			},
			{
				Name:           "org.apache.tinkerpop:tinkergraph-gremlin",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/ext/tinkergraph-gremlin/lib/tinkergraph-gremlin-3.4.8.jar",
				RepositoryHint: "sha1:b438353c7514e468f983370a909328aa5957813a",
			},
			{
				Name:           "org.apache.tinkerpop:tinkergraph-gremlin",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/ext/tinkergraph-gremlin/plugin/tinkergraph-gremlin-3.4.8.jar",
				RepositoryHint: "sha1:b438353c7514e468f983370a909328aa5957813a",
			},
			{
				Name:           "com.github.ben-manes.caffeine",
				Version:        "2.3.1",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/caffeine-2.3.1.jar",
				RepositoryHint: "sha1:d6aec5cbd26313a341ee7c034bd56d604f68bebe",
			},
			{
				Name:           "commons-codec:commons-codec",
				Version:        "1.14",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/commons-codec-1.14.jar",
				RepositoryHint: "sha1:3cb1181b2141a7e752f5bdc998b7ef1849f726cf",
			},
			{
				Name:           "commons-collections:commons-collections",
				Version:        "3.2.2",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/commons-collections-3.2.2.jar",
				RepositoryHint: "sha1:8ad72fe39fa8c91eaaf12aadb21e0c3661fe26d5",
			},
			{
				Name:           "commons-configuration:commons-configuration",
				Version:        "1.10",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/commons-configuration-1.10.jar",
				RepositoryHint: "sha1:2b36e4adfb66d966c5aef2d73deb6be716389dc9",
			},
			{
				Name:           "commons-lang:commons-lang",
				Version:        "2.6",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/commons-lang-2.6.jar",
				RepositoryHint: "sha1:0ce1edb914c94ebc388f086c6827e8bdeec71ac2",
			},
			{
				Name:           "org.apache.commons:commons-lang3",
				Version:        "3.8.1",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/commons-lang3-3.8.1.jar",
				RepositoryHint: "sha1:6505a72a097d9270f7a9e7bf42c4238283247755",
			},
			{
				Name:           "commons-logging:commons-logging",
				Version:        "1.2",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/commons-logging-1.2.jar",
				RepositoryHint: "sha1:4bfc12adfe4842bf07b657f0369c4cb522955686",
			},
			{
				Name:           "net.objecthunter:exp4j",
				Version:        "0.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/exp4j-0.4.8.jar",
				RepositoryHint: "sha1:cf1cfc0f958077d86ac7452c7e36d944689b2ec4",
			},
			{
				Name:           "org.apache.tinkerpop:gremlin-console",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-console-3.4.8.jar",
				RepositoryHint: "sha1:f001a2644ca44cf60fdde8dbd271e919168ec208",
			},
			{
				Name:           "org.apache.tinkerpop:gremlin-core",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-core-3.4.8.jar",
				RepositoryHint: "sha1:7d6074aa75fc8e219fd7456fa94312ba52922dac",
			},
			{
				Name:           "org.apache.tinkerpop:gremlin-driver",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-driver-3.4.8.jar",
				RepositoryHint: "sha1:53a55a34441c49ad7b933a7ddb4276d3e81dbe72",
			},
			{
				Name:           "com.esotericsoftware:kryo-shaded",
				Version:        "3.0.3",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-shaded-3.4.8.jar",
				RepositoryHint: "sha1:eecca88aa8b7e6ca0d85821a0b7df9f9b873e95b",
			},
			{
				Name:           "com.esotericsoftware:minlog",
				Version:        "1.3.0",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-shaded-3.4.8.jar",
				RepositoryHint: "sha1:eecca88aa8b7e6ca0d85821a0b7df9f9b873e95b",
			},
			{
				Name:           "com.esotericsoftware:reflectasm",
				Version:        "1.10.1",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-shaded-3.4.8.jar",
				RepositoryHint: "sha1:eecca88aa8b7e6ca0d85821a0b7df9f9b873e95b",
			},
			{
				Name:           "com.fasterxml.jackson.core:jackson-annotations",
				Version:        "2.9.10",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-shaded-3.4.8.jar",
				RepositoryHint: "sha1:eecca88aa8b7e6ca0d85821a0b7df9f9b873e95b",
			},
			{
				Name:           "com.fasterxml.jackson.core:jackson-core",
				Version:        "2.9.10",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-shaded-3.4.8.jar",
				RepositoryHint: "sha1:eecca88aa8b7e6ca0d85821a0b7df9f9b873e95b",
			},
			{
				Name:           "com.fasterxml.jackson.core:jackson-databind",
				Version:        "2.9.10.5",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-shaded-3.4.8.jar",
				RepositoryHint: "sha1:eecca88aa8b7e6ca0d85821a0b7df9f9b873e95b",
			},
			{
				Name:           "org.apache.tinkerpop:gremlin-shaded",
				Version:        "3.4.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/gremlin-shaded-3.4.8.jar",
				RepositoryHint: "sha1:eecca88aa8b7e6ca0d85821a0b7df9f9b873e95b",
			},
			{
				Name:           "groovy",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-2.5.11-indy.jar",
				RepositoryHint: "sha1:1d90cbcff0947aaf43f31741b48839e5fe190f13",
			},
			{
				Name:           "groovy-cli-picocli",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-cli-picocli-2.5.11.jar",
				RepositoryHint: "sha1:d612d63d4ef1083bc05fcadc233b3d8f201d10f2",
			},
			{
				Name:           "groovy-console",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-console-2.5.11.jar",
				RepositoryHint: "sha1:3db61e9f5806dbf999bbeb44bf6c532540abc731",
			},
			{
				Name:           "groovy-groovysh",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-groovysh-2.5.11-indy.jar",
				RepositoryHint: "sha1:c4c372f662fdfb5f298aee7484553379ef207d1b",
			},
			{
				Name:           "groovy-json",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-json-2.5.11-indy.jar",
				RepositoryHint: "sha1:50233b0100cdb17a90a49a8aaaa9f0d020608600",
			},
			{
				Name:           "groovy-jsr223",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-jsr223-2.5.11-indy.jar",
				RepositoryHint: "sha1:c95ee910c2e74cfc37c73a6510b8476b146f3d10",
			},
			{
				Name:           "groovy-swing",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-swing-2.5.11.jar",
				RepositoryHint: "sha1:bcb2614685279e845f075cd1a22fe6950ce960b2",
			},
			{
				Name:           "groovy-templates",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-templates-2.5.11.jar",
				RepositoryHint: "sha1:941001acfda010320e2426a3b8fe056d6a1eb8f1",
			},
			{
				Name:           "groovy-xml",
				Version:        "2.5.11",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/groovy-xml-2.5.11.jar",
				RepositoryHint: "sha1:3b1e713e805d7ea354a83d1c9e17a0754ea74132",
			},
			{
				Name:           "com.carrotsearch:hppc",
				Version:        "0.7.1",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/hppc-0.7.1.jar",
				RepositoryHint: "sha1:8b5057f74ea378c0150a1860874a3ebdcb713767",
			},
			{
				Name:           "org.apache.httpcomponents:httpclient",
				Version:        "4.5.8",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/httpclient-4.5.8.jar",
				RepositoryHint: "sha1:c27c9d6f15435dc2b6947112027b418b0eef32b9",
			},
			{
				Name:           "org.apache.httpcomponents:httpcore",
				Version:        "4.4.11",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/httpcore-4.4.11.jar",
				RepositoryHint: "sha1:de748cf874e4e193b42eceea9fe5574fabb9d4df",
			},
			{
				Name:           "org.apache.ivy",
				Version:        "2.3.0.final_20130110142753",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/ivy-2.3.0.jar",
				RepositoryHint: "sha1:c5ebf1c253ad4959a29f4acfe696ee48cdd9f473",
			},
			{
				Name:           "com.squareup:javapoet",
				Version:        "1.8.0",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/javapoet-1.8.0.jar",
				RepositoryHint: "sha1:e858dc62ef484048540d27d36f3ec2177a3fa9b1",
			},
			{
				Name:           "org.javatuples:javatuples",
				Version:        "1.2",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/javatuples-1.2.jar",
				RepositoryHint: "sha1:507312ac4b601204a72a83380badbca82683dd36",
			},
			{
				Name:           "org.mindrot:jbcrypt",
				Version:        "0.4",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/jbcrypt-0.4.jar",
				RepositoryHint: "sha1:af7e61017f73abb18ac4e036954f9f28c6366c07",
			},
			{
				Name:           "com.jcabi:jcabi-log",
				Version:        "0.14",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/jcabi-log-0.14.jar",
				RepositoryHint: "sha1:819a57348f2448f01d74f8a317dab61d6a90cac2",
			},
			{
				Name:           "com.jcabi:jcabi-manifests",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/jcabi-manifests-1.1.jar",
				RepositoryHint: "sha1:e4f4488c0e3905c6fab287aca2569928fe1712df",
			},
			{
				Name:           "org.slf4j:jcl-over-slf4j",
				Version:        "1.7.25",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/jcl-over-slf4j-1.7.25.jar",
				RepositoryHint: "sha1:f8c32b13ff142a513eeb5b6330b1588dcb2c0461",
			},
			{
				Name:           "jline:jline",
				Version:        "2.14.6",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/jline-2.14.6.jar",
				RepositoryHint: "sha1:c3aeac59c022bdc497c8c48ed86fa50450e4896a",
			},
			{
				Name:           "log4j:log4j",
				Version:        "1.2.17",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/log4j-1.2.17.jar",
				RepositoryHint: "sha1:5af35056b4d257e4b64b9e8069c0746e8b08629f",
			},
			{
				Name:           "io.netty:netty-all",
				Version:        "4.1.49.Final",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/netty-all-4.1.49.Final.jar",
				RepositoryHint: "sha1:ffe903492be79f5bd8348b04c958de3734a22c6b",
			},
			{
				Name:           "picocli",
				Version:        "4.0.1",
				Kind:           "binary",
				PackageDB:      "jar:opt/gremlin-console/lib/picocli-4.0.1.jar",
				RepositoryHint: "sha1:282c164057d55e6b6af2de49e8930f3c760439da",
			},
			{
				Name:           "org.slf4j:slf4j-api",
				Version:        "1.7.25",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/slf4j-api-1.7.25.jar",
				RepositoryHint: "sha1:da76ca59f6a57ee3102f8f9bd9cee742973efa8a",
			},
			{
				Name:           "org.slf4j:slf4j-log4j12",
				Version:        "1.7.25",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/slf4j-log4j12-1.7.25.jar",
				RepositoryHint: "sha1:110cefe2df103412849d72ef7a67e4e91e4266b4",
			},
			{
				Name:           "org.yaml:snakeyaml",
				Version:        "1.15",
				Kind:           "binary",
				PackageDB:      "maven:opt/gremlin-console/lib/snakeyaml-1.15.jar",
				RepositoryHint: "sha1:3b132bea69e8ee099f416044970997bde80f4ea6",
			},
		},
		Scanner: &java.Scanner{},
	},

	{
		Domain: "docker.io",
		Name:   "jenkins/jenkins",
		Hash:   "sha256:6d33d0cf94a7c68a5da895916067699920f68b4fefa940725fef03cb4b59b2a6",
		Want: []*claircore.Package{
			{
				Name:           "org.jenkins-ci.main:jenkins-war",
				Version:        "2.316",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:f9f22bab538db3c075251ae693a524204efdc32b",
			},
			{
				Name:           "org.kohsuke:access-modifier-annotation",
				Version:        "1.25",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:fab291c4249c077b9bcc48db313a2255df7408b8",
			},
			{
				Name:           "org.jenkins-ci:annotation-indexer",
				Version:        "1.15",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:b5e9548356e53060e7844b2e7fd235f112a6fb8f",
			},
			{
				Name:           "ant",
				Version:        "1.10.11",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:b875cd48a0bc955ae9c5c477ad991e1f26fb24d2",
			},
			{
				Name:           "ant-launcher",
				Version:        "1.10.11",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ea0a0475fb6dfcdcf48b30410fd9d4f5c80df07e",
			},
			{
				Name:           "antlr",
				Version:        "2.7.7",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:83cd2cd674a217ade95a4bb83a8a14f351f48bd0",
			},
			{
				Name:           "args4j:args4j",
				Version:        "2.33",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:bd87a75374a6d6523de82fef51fc3cfe9baf9fc9",
			},
			{
				Name:           "org.objectweb.asm",
				Version:        "9.2.0",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:81a03f76019c67362299c40e0ba13405f5467bff",
			},
			{
				Name:           "org.objectweb.asm.tree.analysis",
				Version:        "9.2.0",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:7487dd756daf96cab9986e44b9d7bcb796a61c10",
			},
			{
				Name:           "org.objectweb.asm.commons",
				Version:        "9.2.0",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:f4d7f0fc9054386f2893b602454d48e07d4fbead",
			},
			{
				Name:           "org.objectweb.asm.tree",
				Version:        "9.2.0",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:d96c99a30f5e1a19b0e609dbb19a44d8518ac01e",
			},
			{
				Name:           "org.objectweb.asm.util",
				Version:        "9.2.0",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:fbc178fc5ba3dab50fd7e8a5317b8b647c8e8946",
			},
			{
				Name:           "asm5",
				Version:        "5.0.1",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:71ab0620a41ed37f626b96d80c2a7c58165550df",
			},
			{
				Name:           "com.infradna.tool:bridge-method-annotation",
				Version:        "1.21",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:f380b42653f21c503036a31ff87c6970eeaad080",
			},
			{
				Name:           "com.sun.activation:jakarta.activation",
				Version:        "2.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "commons-io:commons-io",
				Version:        "2.11.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "commons-lang:commons-lang",
				Version:        "2.6",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "jakarta.websocket:jakarta.websocket-api",
				Version:        "2.0.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "jakarta.websocket:jakarta.websocket-client-api",
				Version:        "2.0.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "jakarta.xml.bind:jakarta.xml.bind-api",
				Version:        "3.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "net.i2p.crypto:eddsa",
				Version:        "0.3.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.apache.sshd:sshd-common",
				Version:        "2.7.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.apache.sshd:sshd-core",
				Version:        "2.7.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-client",
				Version:        "2.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-container-jdk-client",
				Version:        "2.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-core",
				Version:        "2.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-spi",
				Version:        "2.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.glassfish.tyrus.bundles:tyrus-standalone-client-jdk",
				Version:        "2.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.jenkins-ci:annotation-indexer",
				Version:        "1.15",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.jenkins-ci.main:cli",
				Version:        "2.316",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.jvnet.localizer:localizer",
				Version:        "1.31",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.slf4j:jcl-over-slf4j",
				Version:        "1.7.32",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.slf4j:slf4j-api",
				Version:        "1.7.32",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "org.slf4j:slf4j-jdk14",
				Version:        "1.7.32",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ec1386735dc81ac7f42a4958e1e980f8f89dee11",
			},
			{
				Name:           "commons-beanutils:commons-beanutils",
				Version:        "1.9.4",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:d52b9abcd97f38c81342bb7e7ae1eee9b73cba51",
			},
			{
				Name:           "commons-codec:commons-codec",
				Version:        "1.15",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:49d94806b6e3dc933dacbd8acb0fdbab8ebd1e5d",
			},
			{
				Name:           "commons-collections:commons-collections",
				Version:        "3.2.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:8ad72fe39fa8c91eaaf12aadb21e0c3661fe26d5",
			},
			{
				Name:           "org.apache.commons:commons-compress",
				Version:        "1.21",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:4ec95b60d4e86b5c95a0e919cb172a0af98011ef",
			},
			{
				Name:           "commons-discovery:commons-discovery",
				Version:        "0.5",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:3a8ac816bbe02d2f88523ef22cbf2c4abd71d6a8",
			},
			{
				Name:           "commons-fileupload:commons-fileupload",
				Version:        "1.4",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:f95188e3d372e20e7328706c37ef366e5d7859b0",
			},
			{
				Name:           "commons-httpclient:commons-httpclient",
				Version:        "3.1-jenkins-3",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:56110dc7f655c56c4e2ae8ae2bb055bf2282ba99",
			},
			{
				Name:           "commons-io:commons-io",
				Version:        "2.11.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:a2503f302b11ebde7ebc3df41daebe0e4eea3689",
			},
			{
				Name:           "org.jenkins-ci:commons-jelly",
				Version:        "1.1-jenkins-20120928",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:2720a0d54b7f32479b08970d7738041362e1f410",
			},
			{
				Name:           "org.jvnet.hudson:commons-jelly-tags-define",
				Version:        "1.0.1-hudson-20071021",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:8b952d0e504ee505d234853119e5648441894234",
			},
			{
				Name:           "commons-jelly-tags-fmt",
				Version:        "1.0",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:2107da38fdd287ab78a4fa65c1300b5ad9999274",
			},
			{
				Name:           "commons-jelly-tags-xml",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:cc0efc2ae0ff81ef7737afc786a0ce16a8540efc",
			},
			{
				Name:           "org.jenkins-ci:commons-jexl",
				Version:        "1.1-jenkins-20111212",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:0a990a77bea8c5a400d58a6f5d98122236300f7d",
			},
			{
				Name:           "commons-lang:commons-lang",
				Version:        "2.6",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:0ce1edb914c94ebc388f086c6827e8bdeec71ac2",
			},
			{
				Name:           "org.jenkins-ci:crypto-util",
				Version:        "1.5",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:432ba8058e9a653fa191cdda3ed5fe4bb4fc4e53",
			},
			{
				Name:           "dom4j",
				Version:        "2.1.3",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:a75914155a9f5808963170ec20653668a2ffd2fd",
			},
			{
				Name:           "com.sun.solaris:embedded_su4j",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:9404130cc4e60670429f1ab8dbf94d669012725d",
			},
			{
				Name:           "net.sf.ezmorph:ezmorph",
				Version:        "1.0.6",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:01e55d2a0253ea37745d33062852fd2c90027432",
			},
			{
				Name:           "groovy-all",
				Version:        "2.4.12",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:760afc568cbd94c09d78f801ce51aed1326710af",
			},
			{
				Name:           "com.google.guava:guava",
				Version:        "11.0.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:57b40a943725d43610c898ac0169adf1b2d55742",
			},
			{
				Name:           "com.google.inject",
				Version:        "4.0.0",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:0f990a43d3725781b6db7cd0acf0a8b62dfd1649",
			},
			{
				Name:           "org.jenkins-ci.modules:instance-identity",
				Version:        "2.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:986e58370b11287dfa73a8fb2eac74e2dc4e2a23",
			},
			{
				Name:           "org.kohsuke.jinterop:j-interop",
				Version:        "2.0.8-kohsuke-1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:7bcd527550b8415ba14c33a8ff00658c43d0c78e",
			},
			{
				Name:           "org.kohsuke.jinterop:j-interopdeps",
				Version:        "2.0.8-kohsuke-1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:0c6ae17b2cbb5e07e62fddab1cebadaf351a97c0",
			},
			{
				Name:           "com.sun.activation:jakarta.activation",
				Version:        "1.2.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:8013606426a73d8ba6b568370877251e91a38b89",
			},
			{
				Name:           "com.sun.mail:jakarta.mail",
				Version:        "1.6.5",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:d08124137cf42397d00b71b5985fd1dc248ac07f",
			},
			{
				Name:           "jakarta.servlet.jsp.jstl:jakarta.servlet.jsp.jstl-api",
				Version:        "1.2.7",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:34a035507f0270f1c6b7722d728bd7b5a9bbac4c",
			},
			{
				Name:           "org.fusesource.hawtjni:hawtjni-runtime",
				Version:        "1.8",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:655c643309c2f45a56a747fda70e3fadf57e9f11",
			},
			{
				Name:           "org.fusesource.jansi:jansi",
				Version:        "1.11",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:655c643309c2f45a56a747fda70e3fadf57e9f11",
			},
			{
				Name:           "org.fusesource.jansi:jansi-native",
				Version:        "1.5",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:655c643309c2f45a56a747fda70e3fadf57e9f11",
			},
			{
				Name:           "javax.annotation:javax.annotation-api",
				Version:        "1.3.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:934c04d3cfef185a8008e7bf34331b79730a9d43",
			},
			{
				Name:           "javax.inject",
				Version:        "1",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:6975da39a7040257bd51d21a231b76c915872d38",
			},
			{
				Name:           "jaxen:jaxen",
				Version:        "1.2.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c10535a925bd35129a4329bc75065cc6b5293f2c",
			},
			{
				Name:           "jbcrypt",
				Version:        "1.0.0",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:f37bba2b8b78fcc8111bb932318b621dcc6c5194",
			},
			{
				Name:           "org.samba.jcifs:jcifs",
				Version:        "1.3.18-kohsuke-1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:35711c36411d962d7228b7e01ab2567ec213a9d3",
			},
			{
				Name:           "org.slf4j:jcl-over-slf4j",
				Version:        "1.7.32",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:32c060250bcc5282cdbc1fd7008c12eb4ebad00e",
			},
			{
				Name:           "org.jfree:jcommon",
				Version:        "1.0.23",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:a316f336ca996e0c6bec4e4fbd49be8f5e1c3968",
			},
			{
				Name:           "org.jenkins-ci.main:jenkins-core",
				Version:        "2.316",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:4b047f12b03840fb7bee61d84c7f2678afb9e86a",
			},
			{
				Name:           "io.jenkins.stapler:jenkins-stapler-support",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:95127e05b589288439684304a75474bb1261a11c",
			},
			{
				Name:           "com.github.jnr.jffi.native",
				Version:        "1.3.5",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:38602067b8cd1ed3dca6cbfcb3273fb24ab09a3f",
			},
			{
				Name:           "com.github.jnr:jffi",
				Version:        "1.3.5",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:1dadd62fc8434d4ba6e3c78ed42e4852d79d3a46",
			},
			{
				Name:           "org.jfree:jfreechart",
				Version:        "1.0.19",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ba9ee7dbb2e4c57a6901c79f614ed2dea9cc0e20",
			},
			{
				Name:           "jline:jline",
				Version:        "2.14.6",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c3aeac59c022bdc497c8c48ed86fa50450e4896a",
			},
			{
				Name:           "com.sun.jna",
				Version:        "5.9.0",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:8f503e6d9b500ceff299052d6be75b38c7257758",
			},
			{
				Name:           "com.github.jnr:jnr-a64asm",
				Version:        "1.0.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:0a1cb8dbe71b5a6a0288043c3ba3ca64545be165",
			},
			{
				Name:           "com.github.jnr:jnr-constants",
				Version:        "0.10.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:11a081b5482f415443d5e4b860b7a3cb62e319d3",
			},
			{
				Name:           "com.github.jnr:jnr-ffi",
				Version:        "2.2.7",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:73987a1153030a3f1fd6c304531235ad5a0abc74",
			},
			{
				Name:           "com.github.jnr:jnr-posix",
				Version:        "3.1.10",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:18f1ebd53c4a6d7d23487f8f73c3e6adc4cd6716",
			},
			{
				Name:           "com.github.jnr:jnr-x86asm",
				Version:        "1.0.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:006936bbd6c5b235665d87bd450f5e13b52d4b48",
			},
			{
				Name:           "org.kohsuke.stapler:json-lib",
				Version:        "2.4-jenkins-3",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:83f848493c09c735861a90e91591499fb88d7fbb",
			},
			{
				Name:           "com.jcraft:jzlib",
				Version:        "1.1.3-kohsuke-1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:af5d27e1de29df05db95da5d76b546d075bc1bc5",
			},
			{
				Name:           "kxml2",
				Version:        "2.3.0",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ccbc77a5fd907ef863c29f3596c6f54ffa4e9442",
			},
			{
				Name:           "org.jenkins-ci.modules:launchd-slave-installer",
				Version:        "1.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:8ef06fc2a3d9436aa214e318c66cb6dc274440b8",
			},
			{
				Name:           "org.kohsuke:libpam4j",
				Version:        "1.11",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:82cfca73e813c165c83b5d6b1d16808cb5291fe7",
			},
			{
				Name:           "org.jvnet.localizer:localizer",
				Version:        "1.31",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:aa7560693502015b7747697fa5a4fd1da9a95ccd",
			},
			{
				Name:           "org.slf4j:log4j-over-slf4j",
				Version:        "1.7.32",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:152d3b5e2470bb8e898bd82886fc783361e2c32b",
			},
			{
				Name:           "org.jenkins-ci:memory-monitor",
				Version:        "1.9",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:1935bfb46474e3043ee2310a9bb790d42dde2ed7",
			},
			{
				Name:           "io.github.x-stream:mxparser",
				Version:        "1.2.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:476fb3b3bb3716cad797cd054ce45f89445794e9",
			},
			{
				Name:           "relaxngDatatype",
				Version:        "20020414",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:de7952cecd05b65e0e4370cc93fc03035175eef5",
			},
			{
				Name:           "args4j:args4j",
				Version:        "2.33",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "javax.websocket:javax.websocket-api",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "javax.websocket:javax.websocket-client-api",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-client",
				Version:        "1.17",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-container-jdk-client",
				Version:        "1.17",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-core",
				Version:        "1.17",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.glassfish.tyrus:tyrus-spi",
				Version:        "1.17",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.glassfish.tyrus.bundles:tyrus-standalone-client-jdk",
				Version:        "1.17",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.jenkins-ci:constant-pool-scanner",
				Version:        "1.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.jenkins-ci.main:remoting",
				Version:        "4.10",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:c9016b2501e2a3849e8bd14aa7866b3a5385b330",
			},
			{
				Name:           "org.jvnet.robust-http-client:robust-http-client",
				Version:        "1.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:dee9fda92ad39a94a77ec6cf88300d4dd6db8a4d",
			},
			{
				Name:           "io.github.stephenc.crypto:self-signed-cert-generator",
				Version:        "1.0.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:45542f10d259405b42bf537afe30342575d37b18",
			},
			{
				Name:           "net.java.sezpoz:sezpoz",
				Version:        "1.13",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:822ac382900587d0a7d6aaec7b1859f1cf5de2fb",
			},
			{
				Name:           "org.jenkins-ci.modules:slave-installer",
				Version:        "1.7",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:921d195a7a3d2b8af9f0847099dbb9eb652c7bd8",
			},
			{
				Name:           "org.slf4j:slf4j-api",
				Version:        "1.7.32",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:cdcff33940d9f2de763bc41ea05a0be5941176c3",
			},
			{
				Name:           "org.slf4j:slf4j-jdk14",
				Version:        "1.7.32",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:4d5bdc7da49cd69c25f7eed5ae0b259358c7edeb",
			},
			{
				Name:           "spotbugs-annotations",
				Version:        "4.4.1",
				Kind:           "binary",
				PackageDB:      "jar:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:a409167be27a1b197c0432ebfc421f8f79f9bf41",
			},
			{
				Name:           "spring-aop",
				Version:        "5.3.9",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:9bcad31a74e60d205500dd67d2220bd0195c63f8",
			},
			{
				Name:           "spring-beans",
				Version:        "5.3.9",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:48600db2cb1abc0f7ef2b073f0c1abd78a83bcfc",
			},
			{
				Name:           "spring-context",
				Version:        "5.3.9",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:887f4579ade4f47cf0102856f4f4c88eda8ec9d7",
			},
			{
				Name:           "spring-core",
				Version:        "5.3.9",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:cfef19d1dfa41d56f8de66238dc015334997d573",
			},
			{
				Name:           "spring-expression",
				Version:        "5.3.9",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:f5ca763cfb9d62d196efd5d25e8daca7d555ed75",
			},
			{
				Name:           "spring-security-core",
				Version:        "5.5.2",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:cbb11a2fd6e86c45249562b980ba532476d9cae6",
			},
			{
				Name:           "spring-security-crypto",
				Version:        "5.5.2",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:83d2198c12e74dc78a6516563dcc3e989f134b54",
			},
			{
				Name:           "spring-security-web",
				Version:        "5.5.2",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:aa1b6fe3c48fbfbf47992633ff63b02208757d3a",
			},
			{
				Name:           "spring-web",
				Version:        "5.3.9",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:88c920ec1bda67fea04daa8e16165777440df473",
			},
			{
				Name:           "org.kohsuke.stapler:stapler",
				Version:        "1593.v0e838714faae",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:fc5431d95cd9602a9ef20dd8d0da9e6903b8b98d",
			},
			{
				Name:           "org.kohsuke.stapler:stapler-adjunct-codemirror",
				Version:        "1.3",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:fd1d45544400d2a4da6dfee9e60edd4ec3368806",
			},
			{
				Name:           "org.kohsuke.stapler:stapler-adjunct-timeline",
				Version:        "1.5",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:3fa806cbb94679ceab9c1ecaaf5fea8207390cb7",
			},
			{
				Name:           "org.kohsuke.stapler:stapler-groovy",
				Version:        "1593.v0e838714faae",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:700d2b6e99d7d8fb46746907105839dba51b1a20",
			},
			{
				Name:           "org.kohsuke.stapler:stapler-jelly",
				Version:        "1593.v0e838714faae",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:ce9a4f0f5c3eed64165c5915cf1fc2811448fc83",
			},
			{
				Name:           "org.jenkins-ci:symbol-annotation",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:14fe06e7287a8aff81434a2fe8226744183fe955",
			},
			{
				Name:           "org.jenkins-ci.modules:systemd-slave-installer",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:a9a4adb0480ec8942d031b0fff9005077b889875",
			},
			{
				Name:           "org.jenkins-ci:task-reactor",
				Version:        "1.5",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:62964c07bb64457bbbb5711482e9c79bab720f20",
			},
			{
				Name:           "org.jvnet:tiger-types",
				Version:        "2.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:7ddc6bbc8ca59be8879d3a943bf77517ec190f39",
			},
			{
				Name:           "com.sun.xml.txw2:txw2",
				Version:        "20110809",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:46afa3f3c468680875adb8f2a26086a126c89902",
			},
			{
				Name:           "org.jenkins-ci.modules:upstart-slave-installer",
				Version:        "1.1",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:a2ce1f49745d63c8520d50a0c6430afd377ece48",
			},
			{
				Name:           "org.jenkins-ci:version-number",
				Version:        "1.7",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:e2cb3f1d3e296adf74463b046bf79f498cc9b937",
			},
			{
				Name:           "org.kohsuke:windows-package-checker",
				Version:        "1.2",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:86b5d2f9023633808d65dbcfdfd50dc5ad3ca31f",
			},
			{
				Name:           "org.jenkins-ci.modules:windows-slave-installer",
				Version:        "2.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:7dec945c7b97096c6bcd7ad06f43767c544bb66e",
			},
			{
				Name:           "org.jvnet.winp:winp",
				Version:        "1.28",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:8dcfcc7214e73a906786b80704784d90fb593e75",
			},
			{
				Name:           "xpp3",
				Version:        "1.1.4c",
				Kind:           "binary",
				PackageDB:      "file:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:9b988ea84b9e4e9f1874e390ce099b8ac12cfff5",
			},
			{
				Name:           "com.thoughtworks.xstream:xstream",
				Version:        "1.4.18",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:12eb80b4c4b83b184b669866f510a0eae13f9475",
			},
			{
				Name:           "io.jenkins.lib:support-log-formatter",
				Version:        "1.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "javax.servlet:javax.servlet-api",
				Version:        "3.1.0",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-alpn-server",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-client",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-http",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-io",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-jmx",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-security",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-server",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-servlet",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-util",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-util-ajax",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-webapp",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty:jetty-xml",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.alpn:alpn-api",
				Version:        "1.1.3.v20160715",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.http2:http2-common",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.http2:http2-hpack",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.http2:http2-server",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.websocket:websocket-api",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.websocket:websocket-client",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.websocket:websocket-common",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.websocket:websocket-server",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.eclipse.jetty.websocket:websocket-servlet",
				Version:        "9.4.43.v20210629",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
			{
				Name:           "org.jenkins-ci:winstone",
				Version:        "5.21",
				Kind:           "binary",
				PackageDB:      "maven:usr/share/jenkins/jenkins.war",
				RepositoryHint: "sha1:40c26ecf0491aae65f2fe5c4d08c9352a846f412",
			},
		},
		Scanner: &java.Scanner{},
	},
}
