// Copyright The OpenTelemetry Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package env

import (
	"os"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	ottest "go.opentelemetry.io/otel/internal/internaltest"
)

func TestIntEnvOr(t *testing.T) {
	testCases := []struct {
		name          string
		envValue      string
		defaultValue  int
		expectedValue int
	}{
		{
			name:          "IntEnvOrTest - Basic",
			envValue:      "2500",
			defaultValue:  500,
			expectedValue: 2500,
		},
		{
			name:          "IntEnvOrTest - Invalid Number",
			envValue:      "localhost",
			defaultValue:  500,
			expectedValue: 500,
		},
	}

	envStore := ottest.NewEnvStore()
	envStore.Record(BatchSpanProcessorMaxQueueSizeKey)
	defer func() {
		require.NoError(t, envStore.Restore())
	}()
	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			require.NoError(t, os.Setenv(BatchSpanProcessorMaxQueueSizeKey, tc.envValue))
			actualValue := BatchSpanProcessorMaxQueueSize(tc.defaultValue)
			assert.Equal(t, tc.expectedValue, actualValue)
		})
	}
}
