/*
*         OpenPBS (Portable Batch System) v2.3 Software License
*
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
*
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
*
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
*
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
*
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
*
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
*
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
*
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
*
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
*
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
*
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
*
* 7. DISCLAIMER OF WARRANTY
*
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
*
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <ctype.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pbs_ifl.h"
#include "list_link.h"
#include "attribute.h"
#include "pbs_error.h"
#include "pbs_helper.h"

/*
 * This file contains functions for manipulating attributes of type
 * frequency, which can be an integer optionally followed by mhz.
 * It could also be a power state P0 - P15.
 * It could also be a linux power governor policy:
 *       Performance, PowerSave, OnDemand, Conservative, UserSpace
 *
 * The pbs_attribute has functions for:
 * Decoding the value string to the machine representation.
 * Encoding the internal pbs_attribute to external form
 *
 * Some or all of the functions for an pbs_attribute type may be shared with
 * other pbs_attribute types.
 *
 * The prototypes are declared in "attribute.h"
 *
 * --------------------------------------------------
 * The Set of pbs_Attribute Functions for attributes with
 * value type "size"
 * --------------------------------------------------
 */


int to_frequency(const char *, struct cpu_frequency_value *);
void from_frequency(struct cpu_frequency_value *, char *);

/*
 * decode_frequency - decode frequency into pbs_attribute structure
 *
 * Returns: 0 if ok
 *  >0 error number if error
 *  *patr elements set
 */

int decode_frequency(

  pbs_attribute *patr,
  const char * UNUSED(name), /* pbs_attribute name */
  const char * UNUSED(rescn), /* resource name, unused here */
  const char    *val, /* pbs_attribute value */
  int          UNUSED(perm))  /* only used for resources */

  {

  patr->at_val.at_frequency.frequency_type = Invalid;
  patr->at_val.at_frequency.mhz = 0;

  if ((val != NULL) && (val[0] != '\0'))
    {
    if (to_frequency(val, &patr->at_val.at_frequency) != 0)
      {
      return(PBSE_BADATVAL);
      }

    patr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
    }
  else
    {
    patr->at_flags = (patr->at_flags & ~ATR_VFLAG_SET) | ATR_VFLAG_MODIFY;
    }

  return(0);
  }



/*
 * encode_freqency - encode pbs_attribute of type frequency into attr_extern
 *
 * Returns: >0 if ok
 *   =0 if no value, no attrlist link added
 *   <0 if error
 */
/*ARGSUSED*/

#define ENCODE_FREQUENCY_SIZE 23

int encode_frequency(

  pbs_attribute  *attr,    /* ptr to pbs_attribute */
  tlist_head     *phead,   /* head of attrlist list */
  const char    *atname,  /* pbs_attribute name */
  const char    *rsname,  /* resource name (optional) */
  int            UNUSED(mode),    /* encode mode (not used) */
  int            UNUSED(perm))    /* only used for resources */

  {
  size_t   ct;
  char     cvnbuf[ENCODE_FREQUENCY_SIZE];
  svrattrl *pal;

  if (attr == NULL)
    {
    /* FAILURE */

    return(-1);
    }

  if (!(attr->at_flags & ATR_VFLAG_SET))
    {
    return(0);
    }

  from_frequency(&attr->at_val.at_frequency, cvnbuf);

  ct = strlen(cvnbuf);

  pal = attrlist_create(atname, rsname, ct + 1);

  if (pal == NULL)
    {
    /* FAILURE */

    return(-1);
    }

  memcpy(pal->al_value, cvnbuf, ct);

  pal->al_flags = attr->at_flags;

  append_link(phead, &pal->al_link, pal);

  /* SUCCESS */

  return(1);
  }





/*
 * set_frequency - set pbs_attribute A to pbs_attribute B,
 * A=B is only valid operation for frequency.
 *
 * Returns: 0 if ok
 *  >0 if error
 */

int set_frequency(
   
  pbs_attribute *attr,
  pbs_attribute *new_attr,
  enum batch_op  op)

  {
  assert(attr && new_attr && (new_attr->at_flags & ATR_VFLAG_SET));

  switch (op)
    {

    case SET:
      attr->at_val.at_frequency.frequency_type   = new_attr->at_val.at_frequency.frequency_type;
      attr->at_val.at_frequency.mhz = new_attr->at_val.at_frequency.mhz;
      break;

    case INCR:
    case DECR:
    default:
      return (PBSE_INTERNAL);
    }

  attr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;

  return (0);
  }




/*
 * comp_frequency - compare two attributes of type frequency
 *
 * Returns: +1 if 1st > 2nd
 *    0 if 1st == 2nd
 *   -1 if 1st < 2nd
 */

int comp_frequency(
   
  pbs_attribute *attr,
  pbs_attribute *with)

  {

  if(attr->at_val.at_frequency.frequency_type != with->at_val.at_frequency.frequency_type)
    {
    return 1;
    }
  if(attr->at_val.at_frequency.frequency_type != AbsoluteFrequency)
    {
    return 0;
    }
  if(attr->at_val.at_frequency.mhz == with->at_val.at_frequency.mhz)
    {
    return 0;
    }
  if(attr->at_val.at_frequency.mhz > with->at_val.at_frequency.mhz)
    {
    return 1;
    }
  return -1;
  }



/*
 * to_frequency - decode the value string TO a cpu_frequency_value structure
 *
 * This function decodes a value string into a cpu_frequency_value struct.
 */

int to_frequency(

  const char        *val,   /* I */
  struct cpu_frequency_value *pfreq) /* O */

  {
  if((*val >= '0')&&(*val <= '9'))
    {
    //Needs to be all digits.
    const char *pVal = val;
    unsigned long iVal = 0l;
    bool          hasMhz = FALSE;
    while(*pVal)
      {
      if((*pVal < '0')||(*pVal > '9'))
        {
        if((*pVal == 'M')||(*pVal == 'm'))
          {
          pVal++;
          if((*pVal == 'H')||(*pVal == 'h'))
            {
            pVal++;
            if((*pVal == 'Z')||(*pVal == 'z'))
              {
              pVal++;
              if(*pVal == '\0')
                {
                hasMhz = TRUE;
                break;
                }
              }
            }
          }
        return PBSE_BADATVAL;
        }
      iVal = ((iVal * 10) + (unsigned long)(*pVal - '0'));
      pVal++;
      }
    if(iVal <= 15)
      {
      if(hasMhz)
        {
	return PBSE_BADATVAL;
	}
      pfreq->frequency_type = iVal;
      pfreq->mhz = 0;
      return 0;
      }
    pfreq->frequency_type = AbsoluteFrequency;
    pfreq->mhz = iVal;
    return 0;
    }
  if(((*val == 'p')||(*val == 'P'))&&(*(val + 1) >= '0')&&(*(val + 1) <= '9'))
    {
    //P0 - P15
    const char *pVal = val + 1;
    unsigned long iVal = 0l;
    while(*pVal)
      {
      if((*pVal < '0')||(*pVal > '9'))
        {
        return PBSE_BADATVAL;
        }
      iVal = ((iVal * 10) + (unsigned long)(*pVal - '0'));
      pVal++;
      }
    if(iVal > 15)
      {
      return PBSE_BADATVAL;
      }
    pfreq->frequency_type = iVal;
    pfreq->mhz = 0;
    return 0;
    }
  pfreq->mhz = 0;
  if(!strcasecmp(val,"Performance"))
    {
    pfreq->frequency_type = Performance;
    return 0;
    }
  if(!strcasecmp(val,"PowerSave"))
    {
    pfreq->frequency_type = PowerSave;
    return 0;
    }
  if(!strcasecmp(val,"OnDemand"))
    {
    pfreq->frequency_type = OnDemand;
    return 0;
    }
  if(!strcasecmp(val,"Conservative"))
    {
    pfreq->frequency_type = Conservative;
    return 0;
    }

  return(PBSE_BADATVAL);
  }

/*
 * from_frequency - encode a string FROM a cpu_frequency_value structure
 */

void from_frequency(

  struct cpu_frequency_value *pfreq,  /* I */
  char                       *cvnbuf) /* O (minsize=ENCODE_SIZE_SIZE) */

  {
  switch(pfreq->frequency_type)
    {
    case P0:
    case P1:
    case P2:
    case P3:
    case P4:
    case P5:
    case P6:
    case P7:
    case P8:
    case P9:
    case P10:
    case P11:
    case P12:
    case P13:
    case P14:
    case P15:
      sprintf(cvnbuf,"P%u",pfreq->frequency_type);
      break;
    case Performance:
      strcpy(cvnbuf,"Performance");
      break;
    case PowerSave:
      strcpy(cvnbuf,"PowerSave");
      break;
    case OnDemand:
      strcpy(cvnbuf,"OnDemand");
      break;
    case Conservative:
      strcpy(cvnbuf,"Conservative");
      break;
    case UserSpace:
      strcpy(cvnbuf,"UserSpace");
      break;
    case AbsoluteFrequency:
      sprintf(cvnbuf,"%luMHz",pfreq->mhz);
      break;
    default:
      cvnbuf[0] = '\0';
      break;
    }
  }

