/*  job_finddiagramshifts.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_finddiagramshifts.h"
#include "functions.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "files.h"
#include "diagram.h"
#include "graph.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "int.h"

using namespace std;

namespace Reduze {

typedef std::pair<CanonicalLabel, CanonicalRelabeling> LabelPair;

// register job type at JobFactory
namespace {
JobProxy<FindDiagramShifts> dummy;
}

struct dia_is_less_by_name {
	bool operator()(const Diagram& d1, const Diagram& d2) {
		return d1.name() < d2.name();
	}
};

void FindDiagramShifts::run_serial() {
	using GiNaC::exmap;
	using GiNaC::lst;
	using GiNaC::subs_options;
	Files* files = Files::instance();

	bool write_info_file = !info_filename_form_.empty();
	string info_tmp_filename = info_filename_form_ + ".tmp";
	ofstream os_info;
	if (write_info_file)
		os_info.open(info_tmp_filename.c_str());

	// read in the graphs of the diagrams
	bool rename_dias = true;
	map<int, map<int, list<pair<Graph, Diagram> > > > diasbytbyloop;
	size_t num_dias = Reduze::find_graphs_of_diagrams(qgraf_filename_, names_,
			kinematics_, amputate_external_legs_, join_external_legs_,
			minimize_graphs_by_twists_, diasbytbyloop, rename_dias);
	int l_min = 0, l_max = 0;
	if (!diasbytbyloop.empty()) {
		l_min = diasbytbyloop.begin()->first;
		l_max = diasbytbyloop.rbegin()->first;
	}

	// map of new_name -> old_name of the diagrams
	map<string, string> old_by_new_names;
	map<int, map<int, list<pair<Graph, Diagram> > > >::iterator it1;
	for (it1 = diasbytbyloop.begin(); it1 != diasbytbyloop.end(); ++it1) {
		map<int, list<pair<Graph, Diagram> > >::iterator it2;
		for (it2 = it1->second.begin(); it2 != it1->second.end(); ++it2) {
			list<pair<Graph, Diagram> >::iterator it3;
			for (it3 = it2->second.begin(); it3 != it2->second.end(); ++it3)
				old_by_new_names[it3->second.name()] = it3->first.name();
		}
	}

	// read in the graphs of the sectors to which the diagrams can be matched
	LOG("Finding sectors and their graphs to match with diagrams:");
	map<int, map<int, list<SectorGL> > > secbytbyloop;
	bool construct_minimal_graphs = false;
	SectorMappings::find_shift_targets_with_graph(secbytbyloop, l_min, l_max,
			minimize_graphs_by_twists_, construct_minimal_graphs);

	// match diagrams, for-loop over loop and t
	size_t num_matched = 0, num_unmatched = 0;
	set<Sector> secs; // direct target sectors
	set<Sector> minsecs; // minimal target sectors (use crossings, no trivial zeros)
	int col = 0;

	LOGN("\nMatching diagrams:");
	list<Diagram> unmatched;
	//map<int, map<int, list<pair<Graph, Diagram> > > >::iterator it1;
	for (it1 = diasbytbyloop.begin(); it1 != diasbytbyloop.end(); ++it1) {
		int loop = it1->first;
		map<int, list<pair<Graph, Diagram> > >& diasbyt = it1->second;
		map<int, list<SectorGL> >& secbyt = secbytbyloop[loop];

		map<int, list<pair<Graph, Diagram> > >::iterator it2;
		for (it2 = diasbyt.begin(); it2 != diasbyt.end(); ++it2) {
			int t = it2->first;
			list<pair<Graph, Diagram> >& dias = it2->second;
			list<SectorGL>& sectors = secbyt[t];

			list<pair<Graph, Diagram> >::iterator g;
			for (g = dias.begin(); g != dias.end(); ++g) {
				Diagram& dia = g->second;
				const Graph& graph = g->first;
				LOGX("\nAnalyzing diagram " << dia.name());
				if ((col++) % 60 == 0)
					LOG("");
				LabelPair label = graph.find_canonical_label();
				list<SectorGL>::const_iterator s;
				for (s = sectors.begin(); s != sectors.end(); ++s) {
					if (label.first != s->label_pair().first)
						continue; // not isomorphic
					exmap shift;
					const Kinematics* kin =
							s->sector().integralfamily()->kinematics();
					Crossing crossing(kin);
					list<map<int, int> > node_perm;
					GiNaC::ex det;
					try {
						det  = Graph::find_shift(graph, label, s->sectorgraph(),
								s->label_pair(), shift, crossing, node_perm);
					} catch (exception& e) {
						// could fail for isomorphic label if not all crossing available
						// try again, but just for allowed crossings
						// TODO: this is a hack and might be improved
						LOGX(e.what());
						LOGX("trying again with restricted set of crossings");
						node_perm = s->sectorgraph().find_node_symmetry_group();
						try {
							det  = Graph::find_shift(graph, label, s->sectorgraph(),
										s->label_pair(), shift, crossing, node_perm);
						} catch (exception&) {
							continue;
						}
					}
					if (!det.is_equal(1))
						continue; // det not 1

					// graph has same propagators as sector once shift and crossing are applied to graph
					// if only the shift is applied to the graph then the graph matches to the sector
					// where the inverse crossing has been applied

					const IntegralFamily* f = s->sector().integralfamily();
					string name_inv_cf =
							crossing.inverse().name_for_crossed_family(f);
					const IntegralFamily* inv_cf = files->integralfamily(
							name_inv_cf);
					Sector matched(inv_cf, s->sector().id());

					if (!SectorMappings::is_zero(s->sector())) {
						LOGN("x");
						minsecs.insert(s->sector()); // uncrossed
					} else {
						LOGN("o");
					}

					LOGX("  => matched diagram " << dia.name() << " to sector " << matched);
					LOGX("     via shift: " << shift);

					// write to info file
					os_info << "id DiaMatch(" << dia.name() << ") = " //
							<< "Sector(" << name_inv_cf//
							<< ", " << matched.t()//
							<< ", " << matched.id() << ")"//
							<< " * Shift(";
					for (exmap::const_iterator i = shift.begin(); i
							!= shift.end();) {
						os_info << i->first << ", " << i->second << ", []";
						if (++i != shift.end())
							os_info << ", ";
					}
					os_info << ");\n";

					dia.substitute_momenta(shift, subs_options::no_pattern);
					dia.set_loop_momenta(s->sectorgraph().loop_momenta());
					dia.set_sector(matched);
					++num_matched;
					secs.insert(matched);
					break;
				} // sectors
				if (s == sectors.end()) {
					++num_unmatched;
					unmatched.push_back(dia);
					LOGN("n");
					LOGX("  no match for diagram " << dia.name());
				}
			} // graphs
		} // t
	} // loop
	LOG("");

	if (write_info_file) {
		os_info.close();
		rename(info_tmp_filename, info_filename_form_);
	}

	VERIFY(num_matched + num_unmatched == num_dias);

	list<Diagram> outdias;
	for (it1 = diasbytbyloop.begin(); it1 != diasbytbyloop.end(); ++it1) {
		map<int, list<pair<Graph, Diagram> > >::iterator it2;
		for (it2 = it1->second.begin(); it2 != it1->second.end(); ++it2) {
			list<pair<Graph, Diagram> >::iterator g;
			for (g = it2->second.begin(); g != it2->second.end(); ++g) {
				outdias.push_back(Diagram(g->second.kinematics()));
				outdias.back().swap(g->second);
			}
		}
	}
	outdias.sort(dia_is_less_by_name());
	for (list<Diagram>::iterator it = outdias.begin(); it != outdias.end(); ++it)
		it->set_name(old_by_new_names[it->name()]);
	write_diagrams(outdias, output_filename_);

	LOG("\nUnmatched diagrams: " << num_unmatched);
	for (list<Diagram>::const_iterator d = unmatched.begin(); d
			!= unmatched.end(); ++d) {
		map<string, string>::const_iterator on = old_by_new_names.find(
				d->name());
		VERIFY(on != old_by_new_names.end());
		LOGX("  " << d->name() << " (" << on->second << ")");
	}

	LOG("\nMatched diagrams: " << num_matched);

	map<int, list<string> >::const_iterator it;
	list<string>::const_iterator n;
	LOGN("\nSectors: " << secs.size());
	//int col = 0;
	int last_t = -1;
	const IntegralFamily* last_ic = 0;
	for (set<Sector>::const_iterator s = secs.begin(); s != secs.end(); ++s) {
		if (last_ic == 0 || *s->integralfamily() != *last_ic) {
			LOGN("\nfamily " << s->integralfamily()->name() << ":");
			last_ic = s->integralfamily();
			last_t = -1;
		}
		if (s->t() != last_t)
			LOGN("\nt = " << s->t() << ": ");
		last_t = s->t();
		LOGN(s->id() << " ");
	}
	LOG("");

	LOG("Determining compact recursive sector selection");
	SectorSelection sectree =
			SectorSelection::find_compact_recursive_selection(minsecs, true);
	YAML::Emitter ye;
	ye << YAML::BeginSeq << YAML::BeginSeq << YAML::BeginSeq;
	ye << YAML::BeginMap << YAML::Key << "sector_selection" << YAML::Value
			<< sectree << YAML::EndMap;
	ye << YAML::EndSeq << YAML::EndSeq << YAML::EndSeq;
	LOG("\nThis selection of sectors should be reduced for the matched diagrams:");
	LOG(ye.c_str());
}

bool FindDiagramShifts::find_dependencies(const set<string>& outothers, list<
		string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(qgraf_filename_);
	out.push_back(output_filename_);
	if (!info_filename_form_.empty())
		out.push_back(info_filename_form_);
	if (Files::instance()->integralfamilies().empty())
		ABORT("Can't match any diagrams since no integral families defined");
	return true;
}

std::string FindDiagramShifts::get_description() const {
	return "find diagram shifts for " + short_filename(qgraf_filename_);
}

}
