/*  job_finddiagramshifts.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_FINDDIAGRAMSHIFTS_H_
#define JOB_FINDDIAGRAMSHIFTS_H_

#include "job.h"

namespace Reduze {

class Graph;
class Diagram;
class SectorGL;

/// Job to match diagrams to sectors via graphs
class FindDiagramShifts: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("find_diagram_shifts");
		s.set_short_description("Matches diagrams to sectors via graphs.");
		s.set_long_description(""//
					"Job to match diagrams to sectors by comparing graphs"
					" and shifting loop momenta. The job reads in diagrams and"
					" tries to match their graphs with graphs generated from a"
					" minimal number of sector which can appear as shift targets.");
		s.add_option("qgraf_file", true, "string", ""//
					"Input file: QGRAF output (using style file for Reduze YAML"
					" format).");
		s.add_option("output_file", true, "string", "Output file name (YAML).");
		s.add_option("info_file_form", false, "string",
				"Output file for matching info (FORM). If file name is missing"
					" or empty no info file is produced.");
		/// \todo use positions instead of names to select (for consistency with compute_diagram_interferences)
		s.add_option("names", false, "sequence of strings", ""//
					"Input files: list of the names of the diagrams that are"
					" processed; empty means all.");
		s.add_option("minimize_graphs_by_twists", false, "boolean", ""//
					"With this option set to true the generated graphs are twisted"
					" to a unique equivalent of its matroid class.");
		s.add_option("amputate_external_legs", false, "boolean", ""//
					"Whether the legs of the generated graphs should be amputated"
					" before the diagrams are matched to sectors."
					" Sets the external momenta to zero.");
		s.add_option("join_external_legs", false, "boolean", ""//
					"Whether the legs of the generated graphs should be joined"
					" together (in a new vertex) before the diagrams are matched"
					" to sectors.");
		s.add_option("kinematics", false, "string", ""//
					"Name of the kinematics to be used to read in the diagrams."
					" If left empty the default kinematics is used.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	FindDiagramShifts() :
		minimize_graphs_by_twists_(true), amputate_external_legs_(false),
				join_external_legs_(false) {
		add_auto_options();
	}
	virtual ~FindDiagramShifts() {
	}

	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;

protected:
	virtual void add_auto_options() {
		add_auto_io("qgraf_file", qgraf_filename_);
		add_auto_io("output_file", output_filename_);
		add_auto_io("info_file_form", info_filename_form_);
		add_auto_io("names", names_);
		add_auto_io("minimize_graphs_by_twists", minimize_graphs_by_twists_);
		add_auto_io("amputate_external_legs", amputate_external_legs_);
		add_auto_io("join_external_legs", join_external_legs_);
		add_auto_io("kinematics", kinematics_);
	}

private:
	std::string qgraf_filename_; // input (YAML)
	std::string output_filename_; // output of diagrams (YAML)
	std::string info_filename_form_; // output of matching info (FORM)
	std::set<std::string> names_; // input: list of the names of the diagrams that are processed; empty means all
	bool minimize_graphs_by_twists_;
	bool amputate_external_legs_;
	bool join_external_legs_;
	std::string kinematics_;
};

}

#endif /* JOB_FINDDIAGRAMSHIFTS_H_ */
