/*  job_normalize.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_normalize.h"
#include "functions.h"
#include "ginacutils.h"
#include "files.h"
#include "filedata.h"
#include "sectormappings.h"
#include "streamutils.h"
#include "equation.h"
#include "identitygenerator.h"
#include "kinematics.h"
#include "globalsymbols.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<Normalize> dummy;
}

bool Normalize::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(input_filename_);
	out.push_back(output_filename_);
	return true;
}

void Normalize::init() {
	if (input_filename_.empty() || output_filename_.empty())
		throw runtime_error("input or output file name undefined.");
	if (input_filename_ == output_filename_)
		throw runtime_error("input and output file identical.");
}

LinearCombination shift_dimension(const LinearCombination& lc, int shift) {
	using namespace GiNaC;
	LinearCombination res;
	LinearCombination::const_iterator t;
	for (t = lc.begin(); t != lc.end(); ++t) {
		string oldn = t->first.integralfamily()->name();
		string newn = Kinematics::dimension_shifted_label(oldn, shift);
		INT newi(Files::instance()->integralfamily(newn), t->first.v());
		const symbol& d = Files::instance()->globalsymbols()->d();
		ex newc = t->second.subs(d == d + shift, subs_options::no_pattern);
		res.insert(newi, newc);
	}
	return res;
}

void Normalize::run_serial() {
	using namespace GiNaC;
	INT::load_preferred(preferred_masters_file_);

	LOG("Checking type of input file \"" << input_filename_ << "\"");
	bool have_eqs = false;
	if (InFileEquations::check_filetype(input_filename_))
		have_eqs = true;
	else if (!InFileLinearCombinations::check_filetype(input_filename_))
		throw runtime_error("unknown contents in file " + input_filename_);
	LOG("  => file contains " << (have_eqs ? "equations" : "linear combinations"));

	InFileLinearCombinations in(input_filename_);
	OutFileLinearCombinations out(output_filename_);
	LinearCombination lc;
	if (dimension_shift_ != 0) {
		LOG("note: disabling options because of dim-shifted integrals");
		apply_permutations_ = apply_shifts_ = normalize_coefficients_ =
				factor_coefficients_ = false;
	}
	while (in.get(lc)) {
		if (dimension_shift_ != 0) {
			lc = shift_dimension(lc, dimension_shift_);
		}
		if (apply_permutations_ || apply_shifts_) {
			set<INT> ints;
			lc.find_INT(ints);
			for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i) {
				Identity red;
				if (apply_shifts_)
					red = IdentityGenerator::get_shift_identity(*i, have_eqs,
							false);
				if (apply_permutations_ && red.empty()) {
					// no shift, apply permutations and known zeros
					INT equiv = SectorMappings::get_equivalent_or_unique_zero(
							*i);
					if (equiv != *i) {
						red.insert(*i, 1);
						red.insert(equiv, -1);
					}
				}
				if (!red.empty())
					lc.replace(red, false);
			}
		}
		if (!map_on_coeff_.empty()) {
			lc.apply(map_on_coeff_);
		}
		if ((normalize_coefficients_ || factor_coefficients_) && have_eqs) {
			Identity eq;
			eq.swap_terms(lc);
			eq.solve(factor_coefficients_);
			lc.swap_terms(eq);
		} else if ((normalize_coefficients_ || factor_coefficients_)
				&& !have_eqs) {
			LinearCombination raw;
			raw.swap_terms(lc);
			LinearCombination::const_iterator t;
			for (t = raw.begin(); t != raw.end(); ++t) {
				LOGN(".");
				GiNaC::exset cs;
				t->second.find(Color(GiNaC::wild()), cs);
				LOGX("\n  found color structs:");
				GiNaC::lst cl;
				for (GiNaC::exset::const_iterator c = cs.begin(); c != cs.end(); ++c)
					cl.append(*c);
				LOGX("  " << cl);
				ex n = t->second;
				n = factor_coefficients_ ? factor(n) : normal_form(n);
				n = n.collect(cl, true /* distributed */);
				ex nc;
				if (is_a<add> (n)) {
					nc = 0;
					for (size_t c = 0; c < n.nops(); ++c)
						nc += collect_common_factors(n.op(c));
				} else {
					nc = collect_common_factors(n);
				}
				lc.insert(t->first, nc, false /* no extra normal() */);
			}
		}
		lc.remove_zero_integral();
		out << lc;
		LOGN('.');
	}
	out.finalize();
	in.close();
	LOG("");
}

std::string Normalize::get_description() const {
	return string("normalize ") + short_filename(input_filename_);
}

void Normalize::print_manual_options(YAML::Emitter& os) const {
	using namespace YAML;
	os << Key << "map_on_coeff" << Value << Flow
	<< map_on_coeff_ << EndSeq;
}

void Normalize::read_manual_options(const YAML::Node& node) {
	Files* files = Files::instance();
	Kinematics* kin = files->kinematics();
	GiNaC::lst syms = kin->all_symbols();
	GiNaC::lst s = files->globalsymbols()->coupling_constants();
	syms = add_lst(syms, s);
	syms.append(files->globalsymbols()->Nc());
	
	if (node.FindValue("map_on_coeff"))
	Reduze::read(node["map_on_coeff"], map_on_coeff_, syms);
}
	
} // namespace Reduze
