/*  sector.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "sector.h"
#include <string>
#include <fstream>
#include <exception>
#include "kinematics.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "functions.h"
#include "files.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "graph.h"
#include "int.h"

using namespace std;

namespace Reduze {

// register type
namespace {
YAMLProxy<Sector> dummy1;
YAMLProxy<SectorSelection> dummy2;
}

bool AllowReadingCrossedSectors::value_ = true;

// private default constructor
Sector::Sector() :
		integralfamily_(0), id_(0), t_(0) {
}

Sector::Sector(const YAML::Node& node) {
	if (node.Type() != YAML::NodeType::Sequence || node.size() != 2)
		throw std::runtime_error("sector spec is not a two-element sequence "
				+ position_info(node));
	std::string family_name;
	node[0] >> family_name;
	Files* files = Files::instance();
	try {
		if (AllowReadingCrossedSectors::value())
			integralfamily_ = files->integralfamily(family_name);
		else
			integralfamily_ = files->uncrossed_integralfamily(family_name);
	} catch (std::exception& e) {
		throw std::runtime_error(
				"sector spec has unknown integral family "
						+ position_info(node[0]) + "\n" + e.what());
	}
	if (node[1].Type() == YAML::NodeType::Scalar) {
		node[1] >> id_;
	} else {
		throw std::runtime_error("unknown sector spec");
	}

	t_ = t_of_id(id_);
}

void Sector::read(const YAML::Node& node) {
	if (node.Type() == YAML::NodeType::Scalar) {
		int id;
		node >> id;
		*this = Sector(integralfamily_, id);
	} else {
		*this = Sector(node);
	}
}

Sector::Sector(const IntegralFamily* ic, int id) :
	integralfamily_(ic), id_(id), t_(Sector::t_of_id(id)) {
	ASSERT(integralfamily_ != 0);
}

Sector::Sector(const Sector& s1, const Sector& s2) {
	const IntegralFamily* ic1 = s1.integralfamily();
	const IntegralFamily* ic2 = s2.integralfamily();

	Files* f = Files::instance();
	const IntegralFamily* ic = f->integralfamily(ic1, ic2);

	int id1 = s1.id();
	int id2 = s2.id();
	int max = ic1->max_sector_id();

	integralfamily_ = ic;
	id_ = id1 + (max + 1) * id2;
	t_ = Sector::t_of_id(id_);
}

std::string Sector::get_safe_string_for_filename(bool fill_digits) const {
	ostringstream ss;
	if (!fill_digits) {
		ss << integralfamily_->name() << "_" << t_ << "_" << id_;
	} else {
		int maxid = integralfamily_->max_sector_id();
		ss << integralfamily_->name() << "_";
		ss << setfill('0') << setw(number_of_digits(t_of_id(maxid))) << t_;
		ss << "_";
		ss << setfill('0') << setw(number_of_digits(maxid)) << id_;
	}
	return safe_filename(ss.str());
}

bool Sector::is_obviously_zero() const {
	int cutmask = integralfamily_->cut_propagators_mask();
	return (static_cast<size_t> (t_) < integralfamily_->loop_momenta().nops())
			 || (cutmask & id_) != cutmask;
}

bool Sector::is_obviously_non_zero() const {
	return integralfamily_->loop_momenta().nops() == 0; // id = 0
}

bool Sector::operator<(const Sector& other) const {
	if (*integralfamily_->source_integralfamily()
			!= *other.integralfamily_->source_integralfamily())
		return (*integralfamily_->source_integralfamily()
				< *other.integralfamily_->source_integralfamily());
	if (t_ != other.t_)
		return (t_ < other.t_);
	if (id_ != other.id_) {
		if (smaller_sector_id_is_less_)
			return (id_ < other.id_);
		else
			return (id_ > other.id_);
	}
	return (*integralfamily_ < *other.integralfamily_);
}

std::ostream& operator<<(std::ostream& os, const Sector& s) {
	return os << "(" << s.integralfamily_->name() << ":" << s.t_ << ":"
			<< s.id_ << ")";
}

void Sector::print(YAML::Emitter& os) const {
	os << YAML::Flow << YAML::BeginSeq//
			<< (integralfamily_ ? integralfamily_->name() : "") //
			<< id_ //
			<< YAML::EndSeq;
}

void Sector::print_mma(std::ostream& os) const {
	os << "{\"" << integralfamily()->name() << "\"," << id_ << "}";
}

int Sector::t_of_id(int sector_id) {
	unsigned t = 0;
	for (unsigned i = sector_id; i != 0; i = i >> 1)
		if (i & 1)
			t++;
	return t;
}

Sector Sector::get_uncrossed() const {
	return Sector(integralfamily_->source_integralfamily(), id_);
}

Sector Sector::get_equivalent() const {
	return Sector(integralfamily_, integralfamily_->get_equivalent(id_));
}

namespace {
/*
 * A U-polynomial of a given graph is the sum over all spanning trees
 * where each term is a product of the lines x[i] not contained in the tree.
 *
 * A polynomial obtained by the determinant as in get_U_polynomial(...) is
 * a possible graph U-polynomial if:
 *
 * - it is not zero
 * - each term has exactly #loops different factors x[i]
 * - the number of terms is smaller or equal than binomial(#edges, #loops)
 * - every term has the same numerical coefficient
 *
 * NB:
 * Two possible U-polynomials, which differ by a global numerical factor,
 * can correspond to the same graph but with different momentum mappings
 * which can only be identified by a transformation of the loop momenta
 * with determinant not equal to one.
 *
 */

#ifdef UNUSED_CODE
void assert_possible_graph_U_polynomial(const GiNaC::ex& u, unsigned int loops,
		unsigned int edges) {
	using namespace GiNaC;
	string messsage("U-polynomial doesn't correspond to a graph: ");
	ex u_expanded = u.expand();
	if (u_expanded.is_zero())
		throw runtime_error(messsage + "is zero.");
	ASSERT(edges >= loops);
	ex terms = is_a<add> (u_expanded) ? u_expanded : lst(u_expanded);
	if (terms.nops() > binomial(edges, loops))
		throw runtime_error(messsage + "too many terms.");
	exset num_factors;
	for (const_iterator t = terms.begin(); t != terms.end(); ++t) {
		ex fac = is_a<mul> (*t) ? *t : lst(*t);
		exset symb_factors;
		ex num(1);
		for (const_iterator f = fac.begin(); f != fac.end(); ++f) {
			if (is_a<numeric> (*f))
				num *= *f;
			else
				symb_factors.insert(*f);
		}
		num_factors.insert(num);
		if (symb_factors.size() != loops)
			throw runtime_error(messsage
					+ "term with wrong number of lines detected.");
	}
	if (num_factors.size() != 1) // several numeric coefficients in terms
		throw runtime_error(messsage + "several numerical coefficients.");
}
#endif

// returns the edges numbers (a set for each connected component) of lines
// not contained in the spanning tree from the U-polynomial 'u'
// the line numbers are the position of the symbols in 'symbs'
// selects lexicographic simplest term from all the possible terms in the U-polynomial
#ifdef UNUSED_CODE
std::set<std::set<int> > get_fundamental_cycles(const GiNaC::ex u,
		const GiNaC::lst& symbs, unsigned loops) {
	using namespace GiNaC;
	set<set<int> > res;
	try {
		assert_possible_graph_U_polynomial(u, loops, symbs.nops()); // may throw
	} catch (exception& e) {
		throw runtime_error(e.what());
	}

	exmap symbs2int;
	int lauf(0);
	for (lst::const_iterator s = symbs.begin(); s != symbs.end(); ++s)
		symbs2int[*s] = lauf++;

	ex u_factor = factor(u);
	u_factor = is_a<mul> (u_factor) ? u_factor : lst(u_factor);
	for (const_iterator f = u_factor.begin(); f != u_factor.end(); ++f) {
		if (is_a<numeric> (*f))
			continue;
		ex f_expanded = f->expand();
		f_expanded = is_a<add> (f_expanded) ? f_expanded : lst(f_expanded);
		set<set<int> > sorted;
		for (const_iterator e = f_expanded.begin(); e != f_expanded.end(); ++e) {
			set<int> tmp;
			for (lst::const_iterator s = symbs.begin(); s != symbs.end(); ++s)
				if (e->has(*s))
					tmp.insert(ex_to<numeric> (symbs2int[*s]).to_int());
			sorted.insert(tmp);
		}
		ASSERT(!sorted.empty());
		res.insert(*sorted.begin());
	}

	unsigned count(0);
	for (set<set<int> >::const_iterator s = res.begin(); s != res.end(); ++s)
		count += s->size();
	ASSERT(count == loops);
	return res;
}
#endif

/// computes U and F polynomials from sum_i x_i (q_i^2-m_i^2)
/** k are the loop momenta, x are the Feynman parameters,
 ** f==0 means: don't compute f polynomial,
 ** p are the external momenta (needed only for f!=0) **/
void find_UF_polynomials(const GiNaC::ex& sum, const GiNaC::lst& k,
		const GiNaC::lst& x, GiNaC::ex& u, const GiNaC::lst& p, GiNaC::ex* f) {
	using namespace GiNaC;
	// we decompose sum = k M k - 2 k Q - J  where k is vector of loop momenta
	ex s = sum.subs(ScalarProduct(wild(1),wild(2))==wild(1)*wild(2)).expand();
    // compute U polynomial
	matrix M(k.nops(), k.nops());
	for (unsigned i = 0; i < k.nops(); ++i) {
		for (unsigned j = i; j < k.nops(); ++j) {
			if (i != j) {
				M(i, j) = numeric(1, 2) * s.coeff(k.op(i),1).coeff(k.op(j),1);
				M(j, i) = M(i, j);
			} else {
				M(i, j) = s.coeff(k.op(i), 2);
			}
		}
	}
	if (k.nops() == 0)
		u = 1;
	else // GiNaC's determinant computation is much faster than Fermat's
		u = M.determinant(determinant_algo::laplace);
	if (f == 0) // just U poly required
		return;
	// compute F polynomial
	matrix Q(k.nops(), 1);
	exmap k2zero;
	for (unsigned i = 0; i < k.nops(); ++i)
		k2zero[k.op(i)] = 0;
	for (unsigned i = 0; i < k.nops(); ++i) {
		Q(i, 0) = 0;
		Q(i, 0) = -numeric(1,2) * s.coeff(k.op(i),1).subs(k2zero);
	}
	ex J = -s.subs(k2zero);
    exmap tosp;
    for (unsigned i = 0 ; i < p.nops(); ++i)
        for (unsigned j = i ; j < p.nops(); ++j)
        	tosp[p.op(i)*p.op(j)*wild(1)] = ScalarProduct(p.op(i), p.op(j))*wild(1);
	ex QMinvQ = (Q.transpose().mul(M.inverse().mul(Q)))(0,0);
	*f = normal(u * (J + QMinvQ)).expand().subs(tosp);
}
} // unnamed namespace

GiNaC::ex Sector::find_U_polynomial(const GiNaC::lst& xi) const {
	using namespace GiNaC;
	int dummy;
	vector<Propagator> props = get_propagators(dummy);
	VERIFY(xi.nops() >= props.size());
	const lst& loopmom = integralfamily_->loop_momenta();
	const lst& extmom = integralfamily_->kinematics()->external_momenta();
	lst zeros;
	for (unsigned i = 0; i < extmom.nops(); ++i)
		zeros.append(0);
	exmap ext2zero = make_translation(extmom, zeros);
	// set masses and external moment< to zero
	for (unsigned i = 0; i < props.size(); ++i) {
		props[i].let_op(0) = props[i].let_op(0).subs(ext2zero,
				GiNaC::subs_options::no_pattern);
		props[i].let_op(1) = props[i].let_op(1).subs(ext2zero,
				GiNaC::subs_options::no_pattern);
		props[i].let_op(2) = 0;
	}
	ex u;
	ex sum = 0;
	for (unsigned i = 0; i < props.size(); ++i)
		sum += props[i].inverse_ex() * xi.op(i);
	Reduze::find_UF_polynomials(sum, loopmom, xi, u, lst(), 0);
	return u;
}

void Sector::find_UF_polynomials(const GiNaC::lst& xi, GiNaC::ex& u,
		GiNaC::ex& f) const {
	using namespace GiNaC;
	int dummy;
	vector<Propagator> props = get_propagators(dummy);
	VERIFY(xi.nops() >= props.size());
	const lst& loopmom = integralfamily_->loop_momenta();
	const lst& extmom = integralfamily_->kinematics()->independent_external_momenta();
	ex sum = 0;
	for (unsigned i = 0; i < props.size(); ++i)
		sum += props[i].inverse_ex() * xi.op(i);
	Reduze::find_UF_polynomials(sum, loopmom, xi, u, extmom, &f);
	const exmap& s2i = integralfamily_->kinematics()->rules_sp_to_invariants();
	f = f.subs(s2i);
}

// graph construction

void Sector::construct_graphs(std::list<SectorGraph>& graphs, bool& det_non_zero) const {
	using namespace GiNaC;
	graphs.clear();
	det_non_zero = true;

	const int num_props_fam = integralfamily_->propagators().size();
	const Kinematics* kin = integralfamily_->kinematics();
	const exmap& mom_cons = kin->rule_momentum_conservation();
	ex in_mom = kin->incoming_momenta().subs(mom_cons);
	ex out_mom = kin->outgoing_momenta().subs(mom_cons);
	VERIFY(is_a<lst>(in_mom) && is_a<lst>(out_mom));
	const exmap& sp2inv = kin->rules_sp_to_invariants();
	const unsigned num_in = in_mom.nops();
	const unsigned num_out = out_mom.nops();
	const unsigned num_ext = num_in + num_out;

	// numbering starting with 1: incoming, outgoing, internal edges
	const int edge_id_offset = num_ext;
	int dummy;
	map<int, Propagator> props_by_edge = get_propagators_by_pos(dummy,
			edge_id_offset + 1);
	// return if a propagator doesn't allow access to momentum
	map<int, Propagator>::const_iterator p;
	map<int, int> edge_by_prop;
	for (p = props_by_edge.begin(); p != props_by_edge.end(); ++p) {
		if (!p->second.has_squared_momentum())
			return;
		edge_by_prop[p->first - 1 - edge_id_offset] = p->first;
	}

	// get fundamental cycles and check whether they are independent wrt. the loop momenta
	const GiNaC::lst loopmom = integralfamily_->loop_momenta();
	pair<ex, set<int> > dc = independent_propagators(props_by_edge, loopmom);
	if (dc.first.is_zero()) {
		det_non_zero = false;
		return;
	}
	const set<int>& cycles = dc.second;
	map<int, GraphEdge> edges_tree, edges_cycle;
	ASSERT(loopmom.nops() <= props_by_edge.size());
	for (p = props_by_edge.begin(); p != props_by_edge.end(); ++p) {
		GraphEdge gedge(p->second.momentum(), p->second.squaredmass());
		if (cycles.find(p->first) != cycles.end())
			edges_cycle.insert(make_pair(p->first, gedge));
		else
			edges_tree.insert(make_pair(p->first, gedge));
	}
	ASSERT(edges_tree.size() == props_by_edge.size() - loopmom.nops());
	ASSERT(edges_cycle.size() == loopmom.nops());

	// set up the graph with the fundamental cycles and external legs
	SectorGraph graph;
	graph.set_loop_momenta(loopmom);
	graph.set_name(get_safe_string_for_filename());

	// attach the fundamental edges to the root_node and a dummy node of degree 2
	// the degree 2 nodes must be removed at the end
	const int root_node = 1;
	int dummy_id = num_props_fam + edge_id_offset + num_ext + 2; // id for dummy edge and node
	map<int, GraphEdge>::const_iterator ge;
	for (ge = edges_cycle.begin(); ge != edges_cycle.end(); ++ge, ++dummy_id) {
		int edge_id = ge->first;
		graph.insert_edge(Edge(root_node, dummy_id, edge_id));
		graph.insert_edge(Edge(dummy_id, root_node, dummy_id));
		graph.set_momentum(edge_id, ge->second.momentum_);
		graph.set_squaredmass(edge_id, ge->second.squaredmass_);
		graph.set_momentum(dummy_id, ge->second.momentum_);
		graph.set_squaredmass(dummy_id, ge->second.squaredmass_);
	}

	// attach external legs to the root_node with negative node and postive edge numbers
	vector<int> edges_in, edges_out;
	edges_in.reserve(num_in);
	edges_out.reserve(num_out);
	for (unsigned int i = 1; i <= num_in; ++i)
		edges_in.push_back(graph.insert_edge(Edge(-i, root_node, i)));
	for (unsigned int i = 1; i <= num_out; ++i)
		edges_out.push_back(
				graph.insert_edge(Edge(root_node, -i - num_in, i + num_in)));

	for (unsigned int i = 0; i < num_in; ++i) {
		graph.set_momentum(edges_in[i], in_mom.op(i));
		ex sqm = ScalarProduct(in_mom.op(i), in_mom.op(i));
		sqm = sqm.expand().subs(sp2inv).expand();
		graph.set_squaredmass(edges_in[i], sqm);
	}
	for (unsigned int i = 0; i < num_out; ++i) {
		graph.set_momentum(edges_out[i], out_mom.op(i));
		ex sqm = ScalarProduct(out_mom.op(i), out_mom.op(i));
		sqm = sqm.expand().subs(sp2inv).expand();
		graph.set_squaredmass(edges_out[i], sqm);
	}

	// add remaining edges from the spanning tree to the graph
	list<Graph> result;
	graph.insert_graph_edge(edges_tree, result);

	// transform Graph to SectorGraph
	list<SectorGraph> secgraphs;
	for (list<Graph>::iterator g = result.begin(); g != result.end(); ++g) {
		SectorGraph sg(*this, *g, edge_by_prop);
		secgraphs.push_back(SectorGraph());
		secgraphs.back().swap(sg);
	}
	secgraphs.swap(graphs);
}

void Sector::construct_graphs(std::list<SectorGraph>& graphs, int subsector,
		const std::list<SectorGraph>& subgraphs) const {

	graphs.clear();
	VERIFY(!subgraphs.empty());
	map<int, int> edge_by_prop = subgraphs.front().edge_id_by_prop_pos();

	Sector propsec(integralfamily_, id_ - subsector);
	const int edge_id_offset =
			integralfamily()->kinematics()->external_momenta().nops();
	int dummy;
	map<int, Propagator> prop = propsec.get_propagators_by_pos(dummy,
			edge_id_offset + 1);
	if (prop.size() != 1)
		throw runtime_error("Sector " + to_string(subsector) //
				+ " is not a direct subsector of " + to_string(*this));
	int edge_id = prop.begin()->first;
	bool b = edge_by_prop.insert(
			make_pair(edge_id - 1 - edge_id_offset, edge_id)).second;
	ASSERT(b);

	if (!prop.begin()->second.has_squared_momentum())
		return;
	GiNaC::ex momentum = prop.begin()->second.momentum();
	GiNaC::ex squaredmass = prop.begin()->second.squaredmass();

	GraphEdge graphedge(momentum, squaredmass);
	list<SectorGraph>::const_iterator g;
	list<Graph> tmp_graphs;
	for (g = subgraphs.begin(); g != subgraphs.end(); ++g)
		g->insert_graph_edge(edge_id, graphedge, tmp_graphs);

	// set name and construct SectorGraph
	for (list<Graph>::iterator l = tmp_graphs.begin(); l != tmp_graphs.end(); ++l) {
		l->set_name(get_safe_string_for_filename());
		SectorGraph sg(*this, *l, edge_by_prop);
		graphs.push_back(SectorGraph());
		graphs.back().swap(sg);
	}
}

SectorGraph Sector::cleanup_and_select_first_graph(
		const std::list<SectorGraph>& graphlist) {
	VERIFY(!graphlist.empty());
	SectorGraph graph = graphlist.front();
	graph.remove_degree_2_nodes();
	graph.disconnect_vacuum_components();
	return graph;
}

SectorGraph Sector::cleanup_and_select_minimal_graph(
		const std::list<SectorGraph>& graphlist) {
	VERIFY(!graphlist.empty());
	// clean graphs
	list<SectorGraph> cleaned_graphs;
	list<IFindCanonicalLabel*> ptrs;
	for (list<SectorGraph>::const_iterator g = graphlist.begin();
			g != graphlist.end(); ++g) {
		SectorGraph graph = *g;
		graph.remove_degree_2_nodes();
		graph.disconnect_vacuum_components();
		cleaned_graphs.push_back(SectorGraph());
		cleaned_graphs.back().swap(graph);
		ptrs.push_back(&(cleaned_graphs.back()));
	}
	IFindCanonicalLabel* min = select_minimal(ptrs);
	ASSERT(dynamic_cast<SectorGraph*>(min) != 0);
	return dynamic_cast<SectorGraph&>(*min);
}

bool Sector::find_graph(SectorGraph& graph, bool& det_non_zero,
		bool select_minimal_graph) const {
	list<SectorGraph> graphlist;
	construct_graphs(graphlist, det_non_zero);
	if (graphlist.empty())
		return false;
	if (select_minimal_graph)
		graph = Sector::cleanup_and_select_minimal_graph(graphlist);
	else
		graph = Sector::cleanup_and_select_first_graph(graphlist);
	return true;
}

std::vector<Propagator> Sector::get_propagators(int& cutmask) const {
	const vector<Propagator>& allprops = integralfamily_->propagators();
	vector<Propagator> p;
	int tt = t();
	p.reserve(tt);
	cutmask = 0;
	for (int j = 0, i = id_; i != 0; ++j, i = i >> 1)
		if (i & 1) {
			if (integralfamily_->is_cut_propagator(j))
				cutmask |= (1 << (int)p.size());
			p.push_back(allprops[j]);
		}
	return p;
}

std::map<int, Propagator> Sector::get_propagators_by_pos(int& cutmask,
		int offset) const {
	const vector<Propagator>& allprops = integralfamily_->propagators();
	map<int, Propagator> p;
	cutmask = 0;
	for (int j = 0, i = id_; i != 0; ++j, i = i >> 1)
		if (i & 1) {
			if (integralfamily_->is_cut_propagator(j))
				cutmask |= (1 << (int) p.size());
			p.insert(make_pair(j + offset, allprops[j]));
		}
	return p;
}

long int Sector::get_num_integrals_static(int n, int t, int r, int s) {
	long int res;
	if (n > t)
		res = binom_int(r - 1, r - t) * binom_int(n - 1 - t + s, s);
	else
		res = binom_int(r - 1, r - t);
	return res;
}

long int Sector::get_num_integrals(int r, int s) const {
	size_t nprops = integralfamily_->num_propagators();
	return get_num_integrals_static(nprops, t_, r, s);
}

long int Sector::get_num_integrals(int r_min, int r_max, int s_min, int s_max) const {
	long int res = 0;
	for (int i = r_min; i <= r_max; ++i)
		for (int j = s_min; j <= s_max; ++j)
			res += get_num_integrals(i, j);
	return res;
}

// SectorSelection

SectorSelection SectorSelection::find_compact_recursive_selection(
		const std::set<Sector>& secs, bool use_uncrossed) {
	typedef set<Sector, greater<Sector> > secset;
	secset needed, have;
	for (set<Sector>::const_iterator s = secs.begin(); s != secs.end(); ++s) {
		Sector snox = s->get_uncrossed();
		Sector smin = SectorMappings::get_shifted_sector(snox, true);
		if (!SectorMappings::is_zero(smin))
			needed.insert(smin);
	}
	for (secset::const_iterator s = needed.begin(); s != needed.end(); ++s) {
		set<Sector> sub = find_prerequisite_sectors(*s, true);
		for (set<Sector>::const_iterator h = sub.begin(); h != sub.end(); ++h)
			needed.erase(*h);
	}
	SectorSelection sectree;
	sectree.select_recursively_.insert(sectree.select_recursively_.end(),
			needed.begin(), needed.end());
	return sectree;
}

void SectorSelection::read(const YAML::Node& node) {
	verify_yaml_spec(node);
	Sector* pro = 0;
	if (node.FindValue("select"))
		read_proto_ctor(node["select"], select_, pro);
	if (node.FindValue("select_recursively"))
		read_proto_ctor(node["select_recursively"], select_recursively_, pro);
	//if (node.FindValue("select_dependents"))
	//	read_proto_ctor(node["select_dependents"], select_dependents_, pro);
	if (node.FindValue("select_all"))
		node["select_all"] >> select_all_;
	if (node.FindValue("deselect"))
		read_proto_ctor(node["deselect"], deselect_, pro);
	if (node.FindValue("deselect_recursively"))
		read_proto_ctor(node["deselect_recursively"], deselect_recursively_,
				pro);
	if (node.FindValue("deselect_independents"))
		read_proto_ctor(node["deselect_independents"], deselect_independents_,
				pro);
	if (node.FindValue("deselect_graphless"))
		node["deselect_graphless"] >> deselect_graphless_;
	delete pro;
	if (node.FindValue("t_restriction")) {
		node["t_restriction"][0] >> t_min_;
		node["t_restriction"][1] >> t_max_;
	}
}

void SectorSelection::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;
	os << Key << "select_all" << Value << select_all_;
	os << Key << "select" << Value << select_;
	os << Key << "select_recursively" << Value << select_recursively_;
	//os << Key << "select_dependents" << Value << select_dependents_;
	os << Key << "deselect" << Value << deselect_;
	os << Key << "deselect_recursively" << Value << deselect_recursively_;
	os << Key << "deselect_independents" << Value << deselect_independents_;
	os << Key << "deselect_graphless" << Value << deselect_graphless_;
	os << Key << "t_restriction" << Value << Flow << BeginSeq << t_min_
			<< t_max_ << EndSeq;
	os << EndMap;
}

/// adds a sector
void SectorSelection::add(const Sector& s) {
	select_.push_back(s);
}

/// adds a sector and prerequisite sectors (subsectors)
void SectorSelection::add_recursively(const Sector& s) {
	select_recursively_.push_back(s);
}

// apply shifts and ignore zero sectors
void filter_userinput(list<Sector>& l) {
	for (list<Sector>::iterator s = l.begin(); s != l.end(); ) {
		Sector min = SectorMappings::get_shifted_sector(*s);
		if (SectorMappings::is_zero(min)) {
			LOG("Ignoring zero sector " << *s);
			l.erase(s++);
		} else if (min != *s) {
			LOG("Considering " << min << " instead of " << *s);
			*(s++) = min;
		} else {
			++s;
		}
	}
}

std::set<Sector> SectorSelection::find_dependent_sectors(const std::list<Sector>& sub) {

	set<Sector> res;
	Files* files = Files::instance();
	list<IntegralFamily*> fams = files->uncrossed_integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	map<Sector, list<Sector> > sources; // source[uncrossed(target)]
	for (f = fams.begin(); f != fams.end(); ++f) {
		const SectorMappings* sm = files->sectormappings((*f)->name());
		map<int, Sector> rel = sm->sector_relations();
		map<int, Sector>::const_iterator r;
		for (r = rel.begin(); r != rel.end(); ++r)
			sources[r->second.get_uncrossed()].push_back(Sector(*f, r->first));
	}
	set<Sector> visited;
	stack<Sector> todo;
	for (list<Sector>::const_iterator s = sub.begin(); s != sub.end(); ++s)
		todo.push(*s);
	while (!todo.empty()) {
		Sector sec = todo.top();
		todo.pop();
		if (visited.find(sec) != visited.end())
			continue;
		visited.insert(sec);
		// generally minimize wrt permutations
		sec = sec.get_equivalent();
		// if this is a minimal sector, add it
		if (sec == SectorMappings::get_shifted_sector(sec))
			res.insert(sec);
		// if uncrossed sector => add crossed versions to res but don't follow
		if (sec == sec.get_uncrossed()) {
			set<Crossing> xs = files->crossings()->equivalent_crossing_classes();
			for (set<Crossing>::const_iterator x = xs.begin(); x != xs.end(); ++x)
				res.insert(x->transform(sec));
		}
		// dependents of crossed treated as for uncrossed
		sec = sec.get_uncrossed();
		// add immediate super sectors of uncrossed sector
		set<int> p = sec.integralfamily()->get_immediate_supersectors(sec.id());
		for (set<int>::const_iterator s = p.begin(); s != p.end(); ++s)
			todo.push(Sector(sec.integralfamily(), *s));
		// add shift sources
		const list<Sector> sl = sources[sec];
		for (list<Sector>::const_iterator s = sl.begin(); s != sl.end(); ++s)
			todo.push(*s);
	}
	return res;
}

std::set<Sector> SectorSelection::find_prerequisite_sectors(const Sector& sin,
		bool recurse) {
	set<Sector> res; // result
	set<Sector> visited; // needed for a->b->c shifts (see also rev 1410)
	stack<Sector> todo;
	Sector next = sin;
	do {
		// collect first level sub sectors (support a->b->c shift chains)
		// each shift can introduce sub sectors of (intermediate) shift target
		Sector n = next;
		do {
			n = next;
			visited.insert(n);
			const IntegralFamily* f = n.integralfamily();
			set<int> ids = f->get_subsector_equivalents(n.id(), false);
			for (set<int>::const_iterator i = ids.begin(); i != ids.end(); ++i) {
				Sector sub = Sector(f, *i);
				Sector s = SectorMappings::get_shifted_sector(sub, false);
				if (!s.is_obviously_zero() && visited.find(s) == visited.end()) {
					res.insert(SectorMappings::get_shifted_sector(s, true));
					todo.push(sub); // just first shift applied
				}
			}
			next = SectorMappings::get_shifted_sector(n, false);
		} while (next != n);
		if (n.integralfamily()->is_crossed()) {
			Sector nox = n.get_uncrossed();
			if (!nox.is_obviously_zero() && visited.find(nox) == visited.end()) {
				res.insert(nox);
				todo.push(nox);
			}
		}
	} while (recurse && !todo.empty() && (next = todo.top(), todo.pop(), true));
	return res;
}

set<Sector> SectorSelection::find_sectors() const {
	LOG("Determining sectors in selection");
	list<Sector> sel = select_;
	list<Sector> selrec = select_recursively_;
	list<Sector> desel = deselect_;
	list<Sector> deselrec = deselect_recursively_;
	LOGX("  normalizing user input");
    filter_userinput(sel);
    filter_userinput(selrec);
    filter_userinput(desel);
    filter_userinput(deselrec);
	if (select_all_) {
		LOGX("  selecting all sectors");
		Files* files = Files::instance();
		list<IntegralFamily*> fams = files->uncrossed_integralfamilies();
		list<IntegralFamily*>::const_iterator f;
		for	(f = fams.begin(); f != fams.end(); ++f) {
			Sector topsec(*f, (*f)->max_sector_id());
			LOGX("    considering top level sector " << topsec);
			selrec.push_back(topsec);
		}
	    filter_userinput(selrec);
	}
	LOGX("  selecting explicit sectors");
	set<Sector> sectors;
	sectors.insert(sel.begin(), sel.end());
	LOGX("  selecting recursively");
	list<Sector>::const_iterator s;
	for (s = selrec.begin(); s != selrec.end(); ++s) {
		sectors.insert(*s);
		set<Sector> sub = find_prerequisite_sectors(*s, true);
		sectors.insert(sub.begin(), sub.end());
	}
	/*
	if (!select_dependents_.empty()) {
		LOG("Selecting sectors which are dependent of specified sectors");
		list<Sector> dd(select_dependents_);
		filter_userinput(dd);
		set<Sector> dep = find_dependent_sectors(dd);
		sectors.insert(dep.begin(), dep.end());
	}
	*/
	LOGX("  deselecting explicit sectors");
	for (s = desel.begin(); s != desel.end(); ++s)
		sectors.erase(*s);
	LOGX("  deselecting recursively");
	for (s = deselrec.begin(); s != deselrec.end(); ++s) {
		sectors.erase(*s);
		set<Sector> sub = find_prerequisite_sectors(*s, true);
		for (set<Sector>::const_iterator d = sub.begin(); d != sub.end() ; ++d)
			sectors.erase(*d);
	}
	set<Sector> dep;
	if (!deselect_independents_.empty()) {
		LOG("  deselecting sectors which are independent of specified sectors");
		list<Sector> dd(deselect_independents_);
		filter_userinput(dd);
		dep = find_dependent_sectors(dd);
	}
    if (deselect_graphless_) {
    	LOGX("  deselecting graphless sectors");
    	for (set<Sector>::iterator d = sectors.begin(); d != sectors.end();) {
    		const string n = d->integralfamily()->name();
    		const SectorMappings* m = Files::instance()->sectormappings(n);
    		const set<int>& ng = m->sectors_without_graph();
    		if (ng.find(d->id()) != ng.end())
    			sectors.erase(d++);
    		else
    			++d;
    	}
    }
	LOGX("  deselecting sectors by t restriction");
	for (set<Sector>::iterator d = sectors.begin(); d != sectors.end();)
		if ((t_min_ >= 0 && d->t() < t_min_) ||
				(t_max_ >= 0 && d->t() > t_max_) ||
				(!deselect_independents_.empty() && dep.find(*d) == dep.end()))
			sectors.erase(d++);
		else
			++d;
	return sectors;
}

void UncrossedReadSectorSelection::read(const YAML::Node& node) {
	bool remember = AllowReadingCrossedSectors::value_;
	AllowReadingCrossedSectors::value_ = false;
	try {
		SectorSelection::read(node);
	} catch (exception& e) {
		ERROR("Failed reading uncrossed sectors: " << e.what());
	}
	AllowReadingCrossedSectors::value_ = remember;
}

// global functions


std::pair<GiNaC::ex, std::set<int> > independent_propagators(const std::vector<
		Propagator>& props, const GiNaC::lst& loopmoms) {

	if (loopmoms.nops() == 0)
		return make_pair(1, set<int> ());

	vector<set<int> > props_by_mom;
	props_by_mom.reserve(loopmoms.nops());
	for (GiNaC::lst::const_iterator l = loopmoms.begin(); l != loopmoms.end(); ++l) {
		ASSERT(GiNaC::is_a<GiNaC::symbol>(*l));
		props_by_mom.push_back(set<int> ());
		for (unsigned i = 0; i < props.size(); ++i) {
			if (!props[i].has_squared_momentum())
				continue;
			GiNaC::ex momentum = props[i].momentum();
			if (momentum.has(*l))
				props_by_mom.back().insert(i);
		}
	}

	GiNaC::ex det(0);

	try {
		combinations<int> combs(props_by_mom);
	} catch (exception&) {
		return make_pair(det, set<int> ());
	}

	combinations<int> combs(props_by_mom);
	vector<int> r;
	while (combs.get_next(r)) {
		set<int> res(r.begin(), r.end());
		if (res.size() != loopmoms.nops())
			continue;

		GiNaC::matrix mat(loopmoms.nops(), loopmoms.nops());
		for (unsigned i = 0; i < loopmoms.nops(); ++i)
			for (unsigned j = 0; j < loopmoms.nops(); ++j)
				mat(i, j) = props[r[i]].momentum().diff(
						GiNaC::ex_to<GiNaC::symbol>(loopmoms.op(j)));

		det = mat.determinant(GiNaC::determinant_algo::laplace);
		if (!det.is_zero())
			return make_pair(det, res);
	}

	return make_pair(det, set<int> ());
}

std::pair<GiNaC::ex, std::set<int> > independent_propagators(
		const std::map<int, Propagator>& props, const GiNaC::lst& loopmoms) {
	vector<Propagator> propvec;
	propvec.reserve(props.size());
	vector<int> pos;
	pos.reserve(props.size());
	map<int, Propagator>::const_iterator p;
	for (p = props.begin(); p != props.end(); ++p) {
		propvec.push_back(p->second);
		pos.push_back(p->first);
	}
	pair<GiNaC::ex, set<int> > dc = independent_propagators(propvec, loopmoms);
	set<int> fc;
	for (set<int>::iterator c = dc.second.begin(); c != dc.second.end(); ++c) {
		fc.insert(pos[*c]);
	}
	return make_pair(dc.first, fc);
}


}
// namespace Reduze

