/*
 *  Copyright CANON INC. 2018
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

 /*

Copyright (c) 2003-2004, AXE, Inc.  All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <errno.h>
#include <zlib.h>
#include "opvp_utils.h"


void crpcMsg(char *form, ...)
{
#ifndef DBG_USAKO
	va_list ap;

	va_start(ap, form);
	vfprintf(stderr, form, ap);
	va_end(ap);
#else
	FILE *fd = NULL;
	va_list ap;

	fd = fopen("/tmp/usako.log", "a+");
	va_start(ap, form);
	vfprintf(fd, form, ap);
	va_end(ap);
	fclose(fd);
#endif
}

static int z_Code2Val(
	char cVal
){
	int nVal = 0;

	if(('A' <= cVal)&&(cVal <= 'Z')){
		nVal = (cVal - 'A');
	}
	else if(('a' <= cVal)&&(cVal <= 'z')){
		nVal = (cVal - 'a' + 26);
	}
	else if(('0' <= cVal)&&(cVal <= '9')){
		nVal = (cVal - '0' + 52);
	}
	else if(cVal == '+'){
		nVal = 62;
	}
	else if(cVal == '/'){
		nVal = 63;
	}
	else{
		nVal = -1;
	}

	return nVal;
}

static void z_Decode(
	const char *pSrc,
	char *pDst,
	int *nDecodeByte
){
	unsigned long lDecode = 0;
	char c6bit = 0;
	int nDecodeBit = 0;
	int i = 0;

	if((pSrc == NULL)||(pDst == NULL)){
		return;
	}

	for(i = 0; i < 4; i++){
		c6bit = z_Code2Val(pSrc[i]);
		if(c6bit >= 0){
			lDecode |= c6bit;
			nDecodeBit += 6;
		}

		if(i != 3) lDecode <<= 6;
	}

	pDst[0] = (lDecode & 0x00FF0000) >> 16;
	pDst[1] = (lDecode & 0x0000FF00) >> 8;
	pDst[2] = (lDecode & 0x000000FF);

	if(nDecodeByte) *nDecodeByte = nDecodeBit / 8;
}

static int util_decodeBase64(
	const char *pSrc,
	char **pDst,
	long *lDstByte
){
	int err = DEF_NO_ERR;

	char *pDupSrc = NULL;
	long lSrcByte = 0;

	if((pSrc == NULL)||(pDst == NULL)||(lDstByte == NULL)){
		error_proc(__FILE__,__LINE__,err);
		return err;
	}

	*pDst = NULL;
	*lDstByte = 0;

	pDupSrc = strdup(pSrc);
	if(pDupSrc == NULL){
		error_proc(__FILE__,__LINE__,err);
	}
	else{
		long lTmp = 0;

		lSrcByte = strlen(pDupSrc);
		for(lTmp = 0; lTmp < lSrcByte; lTmp++){
			if(pDupSrc[lTmp] == 0x0A){
				memmove(&pDupSrc[lTmp], &pDupSrc[lTmp+1], lSrcByte - lTmp);
				lSrcByte--;
				lTmp--;
			}
		}
	}

	if(err == DEF_NO_ERR){
		lSrcByte = strlen(pDupSrc);
		if(lSrcByte % 4){
			error_proc(__FILE__,__LINE__,err);
		}
		else{
			lSrcByte /= 4;
		}
	}

	if(err == DEF_NO_ERR){
		*pDst = (char *)calloc(1, (lSrcByte * 3) + 1);
		if(*pDst == NULL){
			error_proc(__FILE__,__LINE__,err);
		}
	}

	if(err == DEF_NO_ERR){
		char *pTmpDest = *pDst, *pTmpSrc = pDupSrc;
		int nDecodeByte = 0;

		while(lSrcByte--){
			z_Decode(pTmpSrc, pTmpDest, &nDecodeByte);
			pTmpSrc+=4;
			pTmpDest+=nDecodeByte;
		}

		*lDstByte = pTmpDest - *pDst;
	}

	if(pDupSrc){
		free(pDupSrc);
	}

	if(err != DEF_NO_ERR){
		error_proc(__FILE__,__LINE__,err);
	}

	return err;
}

static char* getHeadStr(const char* jobInfo)
{
	char *headStr = NULL;
	char *tempStr = NULL;
	int size = 0;

	if (jobInfo == NULL) {
		return NULL;
	}

	tempStr = strstr(jobInfo, CN_JOB_INFO_HEADER);
	if (tempStr == NULL) {
		return NULL;
	}

	if (tempStr == jobInfo) {
		return NULL;
	}

	size = tempStr - jobInfo;
	headStr = (char*)malloc(size + 1);
	if (headStr == NULL) {
		return NULL;
	}

	memset(headStr, 0, size + 1);
	strncpy(headStr, jobInfo, size);

	return headStr;
}

static char* getTailStr(const char* jobInfo)
{
	char *tailStr = NULL;
	char *tempStr = NULL;

	if (jobInfo == NULL) {
		return NULL;
	}

	tempStr = strstr(jobInfo, CN_JOB_INFO_HEADER);
	if (tempStr == NULL) {
		return NULL;
	}

	while((*tempStr != OPTIONLIST_VALUE_DELIMITER) && (*tempStr != '\0')) {
		tempStr++;
	}

	if (*tempStr == '\0') {
		return NULL;
	}

	tailStr = (char*)malloc(strlen(tempStr) + 1);
	if (tailStr == NULL) {
		return NULL;
	}

	memset(tailStr, 0, strlen(tempStr)+1);
	strcpy(tailStr, tempStr);

	return tailStr;
}

static char* getBase64Str(const char* jobInfo)
{
	char *base64Str = NULL;
	char *pBase64Start = NULL;
	char *tempStr = NULL;
	int size = 0;

	if (jobInfo == NULL) {
		return NULL;
	}

	tempStr = strstr(jobInfo, CN_JOB_INFO_HEADER);
	if (tempStr == NULL) {
		return NULL;
	}

	tempStr += strlen(CN_JOB_INFO_HEADER);
	pBase64Start = tempStr;

	while((*tempStr != OPTIONLIST_VALUE_DELIMITER) && (*tempStr != '\0')) {
		tempStr++;
	}

	size = tempStr - pBase64Start;
	base64Str = (char*)malloc(size + 1);

	if (base64Str == NULL) {
		return NULL;
	}

	memset(base64Str, 0, size + 1);
	strncpy(base64Str, pBase64Start, size);
	return base64Str;
}

static unsigned long getUncompressStr(char **dstStr, char *sourceStr, long sourceSize)
{
	int errorNo = 0;
	unsigned long size = JOBINFO_INIT_MEM_SIZE;

	if ((dstStr == NULL) || (sourceStr == NULL) || (sourceSize == 0)) {
		return 0;
	}

	*dstStr = (char*)malloc(size + 1);
	if (*dstStr == NULL) {
		return 0;
	}

	memset(*dstStr, 0, size + 1);
	errorNo = uncompress((Bytef*)(*dstStr), &size, (Bytef*)sourceStr, sourceSize);

	while( errorNo == Z_BUF_ERROR)
	{
		size += JOBINFO_REALLOC_MEM_SIZE;
		*dstStr = (char*)realloc(*dstStr, size + 1);

		if (*dstStr != NULL)
		{
			memset(*dstStr, 0, size + 1);
			errorNo = uncompress((Bytef*)(*dstStr), &size, (Bytef*)sourceStr, sourceSize);
		} else {
			break;
		}
	}

	if (errorNo != Z_OK)
	{
		if (*dstStr != NULL)
		{
			free(*dstStr);
			*dstStr = NULL;
		}

		size = 0;
	}

	return size;
}

int get_orignalJobInfo(const char* jobInfo, char**originalJobInfo)
{
	char *headStr = NULL;
	char *tailStr = NULL;
	char *base64Src = NULL;
	char *base64Dst = NULL;
	char *uncompressStr = NULL;
	long base64DstLen = 0;
	unsigned long uncompressSize = 0;
	int headStrSize = 0;
	int tailStrSize = 0;
	int size = 0;
	int ret = RET_ERROR;

	if ((jobInfo == NULL) || (originalJobInfo == NULL)) {
		return RET_NOHANDLE;
	}

	if (strstr(jobInfo, CN_JOB_INFO_HEADER) == NULL) {
		return RET_NOHANDLE;
	}

	base64Src = getBase64Str(jobInfo);

	if (base64Src != NULL)
	{
		if (util_decodeBase64(base64Src, &base64Dst, &base64DstLen) == DEF_NO_ERR)
		{
			uncompressSize = getUncompressStr(&uncompressStr, base64Dst, base64DstLen);

			if ((uncompressStr != NULL) && (uncompressSize != 0))
			{
				headStr = getHeadStr(jobInfo);
				tailStr = getTailStr(jobInfo);
				headStrSize = ((headStr == NULL) ? 0 : strlen(headStr));
				tailStrSize = ((tailStr == NULL) ? 0 : strlen(tailStr));

				size = headStrSize + uncompressSize + tailStrSize;
				*originalJobInfo = (char *)malloc(size + 1);

				if (*originalJobInfo != NULL)
				{
					memset(*originalJobInfo, 0, size + 1);

					if (headStr != NULL) {
						memcpy(*originalJobInfo, headStr, headStrSize);
					}

					if (uncompressStr != NULL) {
						memcpy(*originalJobInfo+headStrSize, uncompressStr, uncompressSize);
					}

					if (tailStr != NULL) {
						memcpy(*originalJobInfo+headStrSize+uncompressSize, tailStr, tailStrSize);
					}

					ret = RET_SUCCESS;

				}else {
					crpcMsg("Cannot malloc the memory for originalJobInfo\n");
				}
			} else {
				crpcMsg("Cannot get the uncompress string\n");
			}
		} else {
			crpcMsg("Cannot get the base64's destination string\n");
		}
	} else  {
		crpcMsg("Cannot get the base64's source string\n");
	}


	if (headStr != NULL)
	{
		free(headStr);
		headStr = NULL;
	}

	if (tailStr != NULL)
	{
		free(tailStr);
		tailStr = NULL;
	}

	if (base64Src != NULL)
	{
		free(base64Src);
		base64Src = NULL;
	}

	if (base64Dst != NULL)
	{
		free(base64Dst);
		base64Dst = NULL;
	}

	if (uncompressStr != NULL)
	{
		free(uncompressStr);
		uncompressStr = NULL;
	}

	crpcMsg("originalJobInfo is %s\n", *originalJobInfo);

	return ret;
}
