/*
 *  Print Dialog for Canon LIPS/PS/LIPSLX/UFR2 Printer.
 *  Copyright CANON INC. 2004
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>

#include <cups/cups.h>

#include <iconv.h>

#define ACCOUNT_PATH "/etc/cngplp/account"
#define ACCOUNT_FILE_TYPE_STATUS_READ 1
#define ACCOUNT_FILE_TYPE_STATUS_WRITE 2
#define ACCOUNT_FILE_TYPE_CONF_READ 3
#define ACCOUNT_FILE_TYPE_CONF_WRITE 4

#define ACCOUNT_FILE_STATUS "status"
#define ACCOUNT_FILE_CONF ".conf"

#define ACCOUNT_FILE_STATUS_NEW "status.new"
#define ACCOUNT_FILE_CONF_NEW ".conf.new"

#define MAX_BUFSIZE     512

#define ACCOUNT_STATUS_FORMAT "<%s>%s</%s>\n"
#define ACCOUNT_CONF_FORMAT_START "<%s>\n"
#define ACCOUNT_CONF_FORMAT_ID "id=%s\n"
#define ACCOUNT_CONF_FORMAT_PSWD "password=%s\n"
#define ACCOUNT_CONF_FORMAT_END "</%s>\n"
#define PRINTER_NAME_FORMAT_START "<%s>"
#define PRINTER_NAME_FORMAT_END "</%s>"
#define PRINTER_NAME_FORMAT_STARTEND "<%s>%s</%s>"
#define ACCOUNT_ON "ON"
#define ACCOUNT_OFF "OFF"

#define ACCOUNT_CONF_ID "id="
#define ACCOUNT_CONF_PSWD "password="
#define ACCOUNT_CONF_USERAUTH_FORMAT_ID "u_id=%s\n"
#define ACCOUNT_CONF_USERAUTH_FORMAT_PSWD "u_password=%s\n"
#define ACCOUNT_CONF_SECUREDPRINT_FORMAT_ID "s_id=%s\n"
#define ACCOUNT_CONF_SECUREDPRINT_FORMAT_PSWD "s_password=%s\n"
#define ACCOUNT_CONF_USERAUTH_ID "u_id="
#define ACCOUNT_CONF_USERAUTH_PSWD "u_password="
#define ACCOUNT_CONF_SECUREDPRINT_ID "s_id="
#define ACCOUNT_CONF_SECUREDPRINT_PSWD "s_password="
#define ACCOUNT_U_ON "U_ON"

#define	MAX_INPUT_LENGTH_ID		7
#define MAX_INPUT_LENGTH_USER	32

static int check_printer_name(char * const printer_name, const char * const t_line, const char * const format);
static int check_account_printer_name(char * const printer_name, char * const t_line);
static int get_account_status(char * const printer_name, char * const t_line);

#include <stdbool.h>
#define	DEF_NO_ERR	0
#define	DEF_ERR		-1
#define	error_proc(src,line,err)	{\
	if(err == DEF_NO_ERR){err = DEF_ERR;}}


static char z_Val2Code(
	int nVal
){
	char cVal = 0x00;

	if((0x00 <= nVal)&&(nVal <= 0x19)){
		cVal = ('A' + nVal);
	}
	else if((0x1a <= nVal)&&(nVal <= 0x33)){
		cVal = ('a' + (nVal - 0x1a));
	}
	else if((0x34 <= nVal)&&(nVal <= 0x3d)){
		cVal = ('0' + (nVal - 0x34));
	}
	else if(nVal == 0x3e){
		cVal = '+';
	}
	else if(nVal == 0x3f){
		cVal = '/';
	}
	else{
		cVal = 0x00;
	}

	return cVal;
}

int z_GetVal(
	const char *pSrc,
	long lSrcByte,
	long nChar
){
	const unsigned char *pUniSrc = NULL;
	long lStartByte = 0;
	unsigned int nSrcData = 0;
	int nTmp = 0;
	bool bSingleByte = false;

	if(pSrc){
		pUniSrc = (const unsigned char *)pSrc;

		lStartByte = (nChar * 6) / 8;

		bSingleByte = ((nChar * 6) % 8) ? false : true;

		if(bSingleByte){
			nSrcData = pUniSrc[lStartByte];
		}
		else{
			nSrcData = pUniSrc[lStartByte] << 8;
			if(lStartByte + 1 < lSrcByte){
				nSrcData |= pUniSrc[lStartByte + 1];
			}
		}

		nTmp = (nChar * 6) - (lStartByte * 8);
		nSrcData <<= nTmp;
		nSrcData &= 0x0000FFFF;
		nSrcData >>= nTmp;

		if(bSingleByte){
			nTmp = ((lStartByte + 1) * 8) - ((nChar + 1) * 6);
		}
		else{
			nTmp = ((lStartByte + 2) * 8) - ((nChar + 1) * 6);
		}
		nSrcData >>= nTmp;
	}

	return nSrcData;
}

int util_encodeBase64(
	const char *pSrc,
	char **pDst,
	long *lDstByte
){
	int err = DEF_NO_ERR;

	unsigned char *pTmpSrc = NULL;
	long lSrcByte = 0;

	char *pTmpDst = NULL;
	long lTmpDstEnableByte = 0;
	long lTmpDstAddByte = 0;


	if((pSrc == NULL)||(pDst == NULL)){
		error_proc(__FILE__,__LINE__,err);
		return err;
	}

	*pDst = NULL;
	if(lDstByte) *lDstByte = 0;
	pTmpSrc = (unsigned char *)pSrc;

	lSrcByte = strlen(pSrc);

	if(err == DEF_NO_ERR){
		lTmpDstEnableByte = ((lSrcByte * 8) / 6);
		if((lSrcByte * 8) % 6){
			lTmpDstEnableByte++;
		}
		if(lTmpDstEnableByte % 4){
			lTmpDstAddByte = (4 - (lTmpDstEnableByte % 4));
		}

		*pDst = (char *)calloc(1, lTmpDstEnableByte + lTmpDstAddByte + 1);
		if(*pDst == NULL){
			error_proc(__FILE__,__LINE__,err);
		}
		else{
			pTmpDst = *pDst;
		}
	}

	if(err == DEF_NO_ERR){
		long nCharNo = 0;
		int nSrcData = 0;
		for(nCharNo = 0; nCharNo < lTmpDstEnableByte; nCharNo++){
			nSrcData = z_GetVal(pSrc, lSrcByte, nCharNo);
			*(pTmpDst++) = z_Val2Code(nSrcData);
		}
		for(nCharNo = 0; nCharNo < lTmpDstAddByte; nCharNo++){
			*(pTmpDst++) = '=';
		}
	}

	if(err != DEF_NO_ERR){
		if(*pDst){
			free(*pDst);
			*pDst = NULL;
		}
		error_proc(__FILE__,__LINE__,err);
	}
	else{
        if(lDstByte){
            *lDstByte = lTmpDstEnableByte + lTmpDstAddByte;
        }
	}

	return err;
}
#if 0
static int z_Code2Val(
	char cVal
){
	int nVal = 0;

	if(('A' <= cVal)&&(cVal <= 'Z')){
		nVal = (cVal - 'A');
	}
	else if(('a' <= cVal)&&(cVal <= 'z')){
		nVal = (cVal - 'a' + 26);
	}
	else if(('0' <= cVal)&&(cVal <= '9')){
		nVal = (cVal - '0' + 52);
	}
	else if(cVal == '+'){
		nVal = 62;
	}
	else if(cVal == '/'){
		nVal = 63;
	}
	else{
		nVal = -1;
	}

	return nVal;
}

static void z_Decode(
	const char *pSrc,
	char *pDst,
	int *nDecodeByte
){
	unsigned long lDecode = 0;
	char c6bit = 0;
	int nDecodeBit = 0;
	int i = 0;

	if((pSrc == NULL)||(pDst == NULL)){
		return;
	}

	for(i = 0; i < 4; i++){
		c6bit = z_Code2Val(pSrc[i]);
		if(c6bit >= 0){
			lDecode |= c6bit;
			nDecodeBit += 6;
		}

		if(i != 3) lDecode <<= 6;
	}

	pDst[0] = (lDecode & 0x00FF0000) >> 16;
	pDst[1] = (lDecode & 0x0000FF00) >> 8;
	pDst[2] = (lDecode & 0x000000FF);

	if(nDecodeByte) *nDecodeByte = nDecodeBit / 8;
}

int util_decodeBase64(
	const char *pSrc,
	char **pDst,
	long *lDstByte
){
	int err = DEF_NO_ERR;

	char *pDupSrc = NULL;
	long lSrcByte = 0;

	if((pSrc == NULL)||(pDst == NULL)||(lDstByte == NULL)){
		error_proc(__FILE__,__LINE__,err);
		return err;
	}

	*pDst = NULL;
	*lDstByte = 0;

	pDupSrc = strdup(pSrc);
	if(pDupSrc == NULL){
		error_proc(__FILE__,__LINE__,err);
	}
	else{
		long lTmp = 0;

		lSrcByte = strlen(pDupSrc);
		for(lTmp = 0; lTmp < lSrcByte; lTmp++){
			if(pDupSrc[lTmp] == 0x0A){
				memmove(&pDupSrc[lTmp], &pDupSrc[lTmp+1], lSrcByte - lTmp);
				lSrcByte--;
				lTmp--;
			}
		}
	}

	if(err == DEF_NO_ERR){
		lSrcByte = strlen(pDupSrc);
		if(lSrcByte % 4){
			error_proc(__FILE__,__LINE__,err);
		}
		else{
			lSrcByte /= 4;
		}
	}

	if(err == DEF_NO_ERR){
		*pDst = (char *)calloc(1, (lSrcByte * 3) + 1);
		if(*pDst == NULL){
			error_proc(__FILE__,__LINE__,err);
		}
	}

	if(err == DEF_NO_ERR){
		char *pTmpDest = *pDst, *pTmpSrc = pDupSrc;
		int nDecodeByte = 0;

		while(lSrcByte--){
			z_Decode(pTmpSrc, pTmpDest, &nDecodeByte);
			pTmpSrc+=4;
			pTmpDest+=nDecodeByte;
		}

		*lDstByte = pTmpDest - *pDst;
	}

	if(pDupSrc){
		free(pDupSrc);
	}

	if(err != DEF_NO_ERR){
		error_proc(__FILE__,__LINE__,err);
	}

	return err;
}
#endif

static int file_write(int fd, char *line, int line_size)
{
	int w_bytes = 0;
	int offset = 0;

	fsync(fd);
	do{
		w_bytes = write(fd, line + offset, line_size - offset);
		if(w_bytes + offset == line_size)
			break;
		offset += w_bytes;
	}while(w_bytes + offset < line_size);
	sync();

	return 0;
}

static int get_line_from_buffer(char **line, char *buff, int offset, int max)
{
	int i;
	char *tmp = buff + offset;
	char *l_tmp = NULL;

	if(*line == NULL){
		l_tmp = (char *)calloc(max + 1, 1);
		if(l_tmp == NULL)
			return -2;
	}else{
		int len = strlen(*line) + max + 1;
		l_tmp = (char *)calloc(len , 1);
		if(l_tmp == NULL)
			return -2;
		strncpy(l_tmp, *line, strlen(*line));
		free(*line);
	}

	for(i = 0; offset + i < max; i++){
		if(*tmp == '\n' || *tmp == '\r' || *tmp == EOF){
			strncat(l_tmp, buff + offset, i);
			*line = l_tmp;
			if( *(tmp + 1) == '\n' || *(tmp + 1) == '\r' || *tmp == EOF)
				i++;
			return (i + 1);
		}
		tmp++;
	}

	strncat(l_tmp, buff + offset, max);
	*line = l_tmp;

	return -1;
}

static int check_printer_name(char * const printer_name, const char * const t_line, const char * const format)
{
	int n_diff_size = 1;
	char str_printer_name[MAX_BUFSIZE];

	if( (printer_name == NULL) || (t_line == NULL) || (format == NULL) ) {
		return n_diff_size;
	}

	memset( str_printer_name, 0x00, MAX_BUFSIZE );

	snprintf( str_printer_name, MAX_BUFSIZE - 1, format, printer_name );
	n_diff_size = strcmp( t_line, str_printer_name );

	return n_diff_size;
}

static int check_account_printer_name(char * const printer_name, char * const t_line)
{
	int n_diff_size = 1;
	int status = 0;
	char *useAccountStr = NULL;
	char str_printer_name[MAX_BUFSIZE];

	if( (printer_name == NULL) || (t_line == NULL) ) {
		return n_diff_size;
	}

	memset( str_printer_name, 0x00, MAX_BUFSIZE );

	status = get_account_status( printer_name, t_line );
	if( status == 1 ) {
		useAccountStr = ACCOUNT_ON;
	}
	else if( status == 2 ) {
		useAccountStr = ACCOUNT_U_ON;
	}
	else {
		useAccountStr = ACCOUNT_OFF;
	}

	snprintf( str_printer_name, MAX_BUFSIZE - 1, PRINTER_NAME_FORMAT_STARTEND, printer_name, useAccountStr, printer_name );
	n_diff_size = strcmp( t_line, str_printer_name );

	return n_diff_size;
}

static int get_account_status(char * const printer_name, char * const t_line)
{
	int n_ret = 0;
	unsigned int un_print_name_size = 0;
	char *buf_str = NULL;
	char str_printer_name[MAX_BUFSIZE];

	if( (printer_name == NULL) || (t_line == NULL) ) {
		return n_ret;
	}

	memset( str_printer_name, 0x00, MAX_BUFSIZE );

	snprintf( str_printer_name, MAX_BUFSIZE - 1, PRINTER_NAME_FORMAT_START, printer_name );
	un_print_name_size = strlen( str_printer_name );
	buf_str = t_line;
	buf_str += un_print_name_size;

	if( strncasecmp( buf_str, ACCOUNT_ON, strlen(ACCOUNT_ON) ) == 0 ) {
		n_ret = 1;
	}
	else if( strncasecmp( buf_str, ACCOUNT_U_ON, strlen(ACCOUNT_U_ON) ) == 0 ) {
		n_ret = 2;
	}
	else {
		n_ret = 0;
	}
	return n_ret;
}

int exist_file(char *file_name)
{
	struct stat info;
	int ret = 0;

	if(file_name != NULL){
		ret = stat(file_name, &info);
	}
	return (ret == 0) ? 0 : 1;
}

char* make_file_path(int type, char *user, char *filename)
{
	char path[256];

	memset(path, 0, 256);
	switch(type){
	case ACCOUNT_FILE_TYPE_STATUS_READ:
		strncpy(path, ACCOUNT_PATH, 255);
		strncat(path, "/", 255 - strlen(path));
		strncat(path, filename, 255 - strlen(path));
		if(exist_file(path))
			return NULL;
		break;
	case ACCOUNT_FILE_TYPE_STATUS_WRITE:
		strncpy(path, ACCOUNT_PATH, 255);
		strncat(path, "/", 255 - strlen(path));
		strncat(path, filename, 255 - strlen(path));
		break;
	case ACCOUNT_FILE_TYPE_CONF_READ:
		strncpy(path, ACCOUNT_PATH, 255);
		strncat(path, "/", 255 - strlen(path));
		if(user == NULL)
		{
			strncat(path, "root", 255 - strlen(path));
		}else{
			strncat(path, user, 255);
		}
		strncat(path, filename, 255 - strlen(path));
		if(exist_file(path))
			return NULL;
		break;
	case ACCOUNT_FILE_TYPE_CONF_WRITE:
		strncpy(path, ACCOUNT_PATH, 255);
		strncat(path, "/", 255 - strlen(path));
		if(user == NULL)
		{
			strncat(path, "root", 255 - strlen(path));
		}else{
			strncat(path, user, 255);
		}
		strncat(path, filename, 255 - strlen(path));
		break;
	default:
		break;
	}
	return strdup(path);
}


int save_account_status(char *printer, int useAccount)
{
	int fd_new, fd_org;
	char *file_new = NULL, *file_org = NULL;
	char *useAccountStr = NULL;
	char *t_line = NULL;

	if(printer == NULL)
		return 1;

	if((file_org = make_file_path(ACCOUNT_FILE_TYPE_STATUS_WRITE, NULL,
				ACCOUNT_FILE_STATUS)) == NULL)
		return 1;

	if((file_new = make_file_path(ACCOUNT_FILE_TYPE_STATUS_WRITE, NULL,
				ACCOUNT_FILE_STATUS_NEW)) == NULL)
		goto err;

	if((fd_new = open(file_new, O_RDWR|O_CREAT|O_EXCL, 0644)) < 0)
		goto err;

	if( 1 == useAccount ){
		useAccountStr = ACCOUNT_ON;
	} else if( 2 == useAccount ){
		useAccountStr = ACCOUNT_U_ON;
	} else {
		useAccountStr = ACCOUNT_OFF;
	}

	if((fd_org = open(file_org, O_RDONLY)) < 0){
		t_line = (char *)calloc(MAX_BUFSIZE + 1, 1);
		if(t_line == NULL)
			goto err;
		snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_STATUS_FORMAT,
			printer, useAccountStr, printer);
		file_write(fd_new, t_line, strlen(t_line));
	}else{
		int is_curr = 0;
		int offset = 0, l_bytes = 0, r_bytes = 0;
		char t_buf[MAX_BUFSIZE + 1];
		memset(t_buf, 0, sizeof(t_buf));
		while((r_bytes = read(fd_org, t_buf, MAX_BUFSIZE))){
			if(r_bytes == -1){
				if(errno == EINTR){
					continue;
				}else{
					break;
				}
			}
			offset = 0;
			while( r_bytes > offset){
				char new_line[MAX_BUFSIZE];
				memset(new_line, 0, MAX_BUFSIZE);
				l_bytes = get_line_from_buffer(&t_line, t_buf, offset, MAX_BUFSIZE);
				if(l_bytes < 0){
					break;
				}


				if((t_line[0] == '<') &&
				   (check_account_printer_name(printer, &t_line[0]) == 0))
				{
					snprintf(new_line, MAX_BUFSIZE - 1,
						ACCOUNT_STATUS_FORMAT, printer,
						useAccountStr, printer);
					file_write(fd_new, new_line, strlen(new_line));
					is_curr = 1;
				}else{
					char *tmp = NULL;
					int len = strlen(t_line) + 1;
					tmp = (char *)calloc(len + 1, 1);
					if(tmp != NULL){
						snprintf(tmp, len + 1, "%s\n", t_line);
						file_write(fd_new, tmp, strlen(tmp));
						free(tmp);
					}
				}
				offset += l_bytes;
				if(t_line != NULL){
					free(t_line);
					t_line = NULL;
				}
			}
			memset(t_buf, 0, sizeof(t_buf));
		}
		if(is_curr == 0){
			if(t_line != NULL)
				free(t_line);
			t_line = (char *)calloc(MAX_BUFSIZE + 1, 1);
			if(t_line != NULL){
				snprintf(t_line, MAX_BUFSIZE - 1,
					ACCOUNT_STATUS_FORMAT, printer,
					 useAccountStr, printer);
				file_write(fd_new, t_line, strlen(t_line));
			}
		}
	}

	unlink(file_org);
	rename(file_new, file_org);

	if(t_line != NULL)
		free(t_line);
	if(file_new != NULL)
		free(file_new);
	if(file_org != NULL)
		free(file_org);
	if(fd_new > 0)
		close(fd_new);
	if(fd_org > 0)
		close(fd_org);

	return 0;

err:
	if(file_new != NULL)
		free(file_new);
	if(file_org != NULL)
		free(file_org);

	return 1;
}

static int check_account_conf(int infd, char *printer, int useAccount)
{
	int ret = 1;

	if( (0 <= infd) && (NULL != printer) ){
		if( 0 == useAccount ){
			ret = 1;
		} else if( (1 == useAccount) || (2 == useAccount) || (3 == useAccount) ){
			off_t iores = 0;
			iores = lseek( infd, 0, SEEK_SET );
			if( -1 != iores ){
				char *t_line = NULL;

				ret = 1;
				t_line = (char *)calloc(MAX_BUFSIZE + 1, 1);
				if( t_line != NULL ){
					int offset = 0;
					int l_bytes = 0;
					int r_bytes = 0;
					int is_curr = 0;
					char t_buf[MAX_BUFSIZE + 1];

					memset(t_buf, 0, sizeof(t_buf));
					while((r_bytes = read(infd, t_buf, MAX_BUFSIZE))){
						if(r_bytes == -1){
							if(errno == EINTR){
								continue;
							} else {
								break;
							}
						}
						offset = 0;

						while(r_bytes > offset){
							char new_line[MAX_BUFSIZE];

							memset(new_line, 0, sizeof(new_line));
							l_bytes = get_line_from_buffer(&t_line, t_buf, offset, MAX_BUFSIZE);

							if(l_bytes < 0){
								break;
							}

							if((t_line[0] == '<') && (check_printer_name(printer, &t_line[0], PRINTER_NAME_FORMAT_START) == 0)){
								is_curr = 1;
							}
							else if((t_line[0] == '<') && (t_line[1] == '/') && (check_printer_name(printer, &t_line[0], PRINTER_NAME_FORMAT_END) == 0)){
								is_curr = 2;
							}
							else if(is_curr == 1){
								if( 1 == useAccount ){
									if( (strncmp(t_line, ACCOUNT_CONF_USERAUTH_ID, strlen(ACCOUNT_CONF_USERAUTH_ID)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_USERAUTH_PSWD, strlen(ACCOUNT_CONF_USERAUTH_PSWD)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_SECUREDPRINT_ID, strlen(ACCOUNT_CONF_SECUREDPRINT_ID)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_SECUREDPRINT_PSWD, strlen(ACCOUNT_CONF_SECUREDPRINT_PSWD)) == 0) ){
										ret = 0;
									}
								} else if( 2 == useAccount ){
									if( (strncmp(t_line, ACCOUNT_CONF_ID, strlen(ACCOUNT_CONF_ID)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_PSWD, strlen(ACCOUNT_CONF_PSWD)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_SECUREDPRINT_ID, strlen(ACCOUNT_CONF_SECUREDPRINT_ID)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_SECUREDPRINT_PSWD, strlen(ACCOUNT_CONF_SECUREDPRINT_PSWD)) == 0) ){
										ret = 0;
									}
								} else{
									if( (strncmp(t_line, ACCOUNT_CONF_ID, strlen(ACCOUNT_CONF_ID)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_PSWD, strlen(ACCOUNT_CONF_PSWD)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_USERAUTH_ID, strlen(ACCOUNT_CONF_USERAUTH_ID)) == 0) ||
										(strncmp(t_line, ACCOUNT_CONF_USERAUTH_PSWD, strlen(ACCOUNT_CONF_USERAUTH_PSWD)) == 0) ){
										ret = 0;
									}
								}
							}

							offset += l_bytes;
							if(t_line != NULL){
								free(t_line);
								t_line = NULL;
							}
						}
					}
				} else {
					ret = 0;
				}
			} else {
				ret = 0;
			}
			iores = lseek( infd, 0, SEEK_SET );
		} else {
			ret = 0;
		}
	} else {
		ret = 0;
	}

	return ret;
}

int save_account_conf(char *printer, char *user, char *id, char *ps, int del, int useAccount)
{
	int fd_new, fd_org;
	char *file_new = NULL, *file_org = NULL;
	char *t_line = NULL;
	char* pEncodeID = NULL;
	char* pEncodePass = NULL;
	long lEncodeLen = 0;

	if(printer == NULL)
		return 1;

	if( (NULL != id) && (NULL != ps) && ((3 == useAccount) || (2 == useAccount)) ){
		int res = DEF_NO_ERR;
		res = util_encodeBase64( id, &pEncodeID, &lEncodeLen );
		if( DEF_NO_ERR == res ){
			res = util_encodeBase64( ps, &pEncodePass, &lEncodeLen );
			if( DEF_NO_ERR != res ){
				goto err;
			}
		}
	}

	if((file_org = make_file_path(ACCOUNT_FILE_TYPE_CONF_WRITE, user,
				ACCOUNT_FILE_CONF)) == NULL)
		return 1;

	if((file_new = make_file_path(ACCOUNT_FILE_TYPE_CONF_WRITE, user,
				ACCOUNT_FILE_CONF_NEW)) == NULL)
		goto err;

	if((fd_new = open(file_new, O_RDWR|O_CREAT|O_EXCL, 0600)) < 0)
		goto err;

	if((fd_org = open(file_org, O_RDONLY)) < 0){
		if(del){
			close(fd_new);
			unlink(file_new);
			goto err;
		}
		t_line = (char *)calloc(MAX_BUFSIZE + 1, 1);
		if(t_line == NULL)
			goto err;
		snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_START, printer);
		file_write(fd_new, t_line, strlen(t_line));
		if( 1 == useAccount ){
			memset(t_line, 0, MAX_BUFSIZE + 1);
			snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_ID, id);
			file_write(fd_new, t_line, strlen(t_line));
			memset(t_line, 0, MAX_BUFSIZE + 1);
			snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_PSWD, ps);
			file_write(fd_new, t_line, strlen(t_line));
		} else if( 2 == useAccount ){
			if( (NULL != pEncodeID) && (NULL != pEncodePass) ){
				memset(t_line, 0, MAX_BUFSIZE + 1);
				snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_ID, pEncodeID);
				file_write(fd_new, t_line, strlen(t_line));
				memset(t_line, 0, MAX_BUFSIZE + 1);
				snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_PSWD, pEncodePass);
				file_write(fd_new, t_line, strlen(t_line));
			}
		} else {
			if( (NULL != pEncodeID) && (NULL != pEncodePass) ){
				memset(t_line, 0, MAX_BUFSIZE + 1);
				snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_ID, pEncodeID);
				file_write(fd_new, t_line, strlen(t_line));
				memset(t_line, 0, MAX_BUFSIZE + 1);
				snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_PSWD, pEncodePass);
				file_write(fd_new, t_line, strlen(t_line));
			}
		}
		memset(t_line, 0, MAX_BUFSIZE + 1);
		snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_END, printer);
		file_write(fd_new, t_line, strlen(t_line));
	}else{
		int is_curr = 0;
		int offset = 0, l_bytes = 0, r_bytes = 0;
		char t_buf[MAX_BUFSIZE + 1];
		int is_mod = 0;
		int is_delete_queue = 0;

		is_delete_queue = check_account_conf( fd_org, printer, useAccount );

		memset(t_buf, 0, sizeof(t_buf));
		while((r_bytes = read(fd_org, t_buf, MAX_BUFSIZE))){
			if(r_bytes == -1){
				if(errno == EINTR)
					continue;
				else
					break;
			}
			offset = 0;

			while(r_bytes > offset){
				char new_line[MAX_BUFSIZE];

				memset(new_line, 0, sizeof(new_line));
				l_bytes = get_line_from_buffer(&t_line, t_buf, offset, MAX_BUFSIZE);

				if(l_bytes < 0){
					break;
				}

				if((t_line[0] == '<') &&
				   (check_printer_name(printer, &t_line[0], PRINTER_NAME_FORMAT_START) == 0))
				{
					is_curr = 1;
					if(del && (1 == is_delete_queue))
						goto skip;
					snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_START, printer);
					file_write(fd_new, new_line, strlen(new_line));
				}
				else if((t_line[0] == '<') && (t_line[1] == '/') &&
				        (check_printer_name(printer, &t_line[0], PRINTER_NAME_FORMAT_END) == 0))
				{
					is_curr = 2;
					if(del && (1 == is_delete_queue))
						goto skip;
					if( (0 == del) && (0 == is_mod) ){
						if( 3 == useAccount ){
							if( (NULL != pEncodeID) && (NULL != pEncodePass) ){
								snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_ID, pEncodeID);
								file_write(fd_new, new_line, strlen(new_line));
								snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_PSWD, pEncodePass);
								file_write(fd_new, new_line, strlen(new_line));
								is_mod = 1;
							}
						} else if( 2 == useAccount ){
							if( (NULL != pEncodeID) && (NULL != pEncodePass) ){
								snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_ID, pEncodeID);
								file_write(fd_new, new_line, strlen(new_line));
								snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_PSWD, pEncodePass);
								file_write(fd_new, new_line, strlen(new_line));
								is_mod = 1;
							}
						} else {
							snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_ID, id);
							file_write(fd_new, new_line, strlen(new_line));
							snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_PSWD, ps);
							file_write(fd_new, new_line, strlen(new_line));
							is_mod = 1;
						}
					}
					snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_END, printer);
					file_write(fd_new, new_line, strlen(new_line));
				}
				else if((is_curr == 1) && (strncmp(t_line, ACCOUNT_CONF_SECUREDPRINT_ID, strlen(ACCOUNT_CONF_SECUREDPRINT_ID)) == 0) && (3 == useAccount))
				{
					if(del){
						goto skip;
					}
					if( NULL != pEncodePass ){
						snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_ID, pEncodeID);
						file_write(fd_new, new_line, strlen(new_line));
						is_mod = 1;
					}
				}
				else if((is_curr == 1) && (strncmp(t_line, ACCOUNT_CONF_SECUREDPRINT_PSWD, strlen(ACCOUNT_CONF_SECUREDPRINT_PSWD)) == 0) && (3 == useAccount))
				{
					if(del){
						goto skip;
					}
					if( NULL != pEncodePass ){
						snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_PSWD, pEncodePass);
						file_write(fd_new, new_line, strlen(new_line));
						is_mod = 1;
					}
				}
				else if((is_curr == 1) && (strncmp(t_line, ACCOUNT_CONF_USERAUTH_ID, strlen(ACCOUNT_CONF_USERAUTH_ID)) == 0) && (2 == useAccount))
				{
					if(del){
						goto skip;
					}
					if( NULL != pEncodeID ){
						snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_ID, pEncodeID);
						file_write(fd_new, new_line, strlen(new_line));
						is_mod = 1;
					}
				}
				else if((is_curr == 1) &&
				        (strncmp(t_line, "id=", strlen("id=")) == 0) && (1 == useAccount))
				{
					if(del)
						goto skip;
					snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_ID, id);
					file_write(fd_new, new_line, strlen(new_line));
					is_mod = 1;
				}
				else if((is_curr == 1) && (strncmp(t_line, ACCOUNT_CONF_USERAUTH_PSWD, strlen(ACCOUNT_CONF_USERAUTH_PSWD)) == 0) && (2 == useAccount))
				{
					if(del){
						goto skip;
					}
					if( NULL != pEncodePass ){
						snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_PSWD, pEncodePass);
						file_write(fd_new, new_line, strlen(new_line));
						is_mod = 1;
					}
				}
				else if((is_curr == 1) &&
			     	    (strncmp(t_line, "password=", strlen("password=")) == 0) && (1 == useAccount))
				{
					if(del)
						goto skip;
					snprintf(new_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_PSWD, ps);
					file_write(fd_new, new_line, strlen(new_line));
					is_mod = 1;
				}else{
					char *tmp = NULL;
					int len = strlen(t_line) + 1;
					tmp = (char *)calloc(len + 1, 1);
					if(tmp != NULL){
						snprintf(tmp, len + 1, "%s\n", t_line);
						file_write(fd_new, tmp, strlen(tmp));
						free(tmp);
					}
				}
skip:
				offset += l_bytes;
				if(t_line != NULL){
					free(t_line);
					t_line = NULL;
				}
			}
			memset(t_buf, 0, sizeof(t_buf));
		}
		if(is_curr == 0 && del == 0){
			if(t_line != NULL)
				free(t_line);
			t_line = (char *)calloc(MAX_BUFSIZE + 1, 1);
			if(t_line != NULL){
				memset(t_line, 0, MAX_BUFSIZE);
				snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_START, printer);
				file_write(fd_new, t_line, strlen(t_line));
				if( 1 == useAccount ){
					memset(t_line, 0, MAX_BUFSIZE);
					snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_ID, id);
					file_write(fd_new, t_line, strlen(t_line));
					memset(t_line, 0, MAX_BUFSIZE);
					snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_PSWD, ps);
					file_write(fd_new, t_line, strlen(t_line));
				} else if( 2 == useAccount ){
					if( (NULL != pEncodeID) && (NULL != pEncodePass) ){
						memset(t_line, 0, MAX_BUFSIZE + 1);
						snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_ID, pEncodeID);
						file_write(fd_new, t_line, strlen(t_line));
						memset(t_line, 0, MAX_BUFSIZE + 1);
						snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_USERAUTH_FORMAT_PSWD, pEncodePass);
						file_write(fd_new, t_line, strlen(t_line));
					}
				} else {
					if( (NULL != pEncodeID) && (NULL != pEncodePass) ){
						memset(t_line, 0, MAX_BUFSIZE + 1);
						snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_ID, pEncodeID);
						file_write(fd_new, t_line, strlen(t_line));
						memset(t_line, 0, MAX_BUFSIZE + 1);
						snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_SECUREDPRINT_FORMAT_PSWD, pEncodePass);
						file_write(fd_new, t_line, strlen(t_line));
					}
				}
				memset(t_line, 0, MAX_BUFSIZE);
				snprintf(t_line, MAX_BUFSIZE - 1, ACCOUNT_CONF_FORMAT_END, printer);
				file_write(fd_new, t_line, strlen(t_line));
			}
		}
	}

	unlink(file_org);
	rename(file_new, file_org);

	if(t_line != NULL)
		free(t_line);
	if(file_new != NULL)
		free(file_new);
	if(file_org != NULL)
		free(file_org);
	if(fd_new > 0)
		close(fd_new);
	if(fd_org > 0)
		close(fd_new);
	if( NULL != pEncodeID ){
		free( pEncodeID );
	}
	if( NULL != pEncodePass ){
		free( pEncodePass );
	}

	return 0;
err:
	if(file_new != NULL)
		free(file_new);
	if(file_org != NULL)
		free(file_org);
	if( NULL != pEncodeID ){
		free( pEncodeID );
	}
	if( NULL != pEncodePass ){
		free( pEncodePass );
	}

	return 1;
}

int ChangedJobAccount(char *id, char *ps, char *org_id, char *org_ps)
{
	if(strlen(id) == strlen(org_id)){
		if(strcmp(id, org_id) == 0){
			if(strlen(ps) == strlen(org_ps)){
				if(strcmp(ps, org_ps) == 0)
					return 0;
			}
		}
	}
	return 1;
}

int SaveJobAccount(char *printer, char *user, char *id, char *ps, int del, int useAccount)
{
	if(printer == NULL)
		return -1;

#if 0
#else
	switch(del){
		case 0:
		case 10:
		case 20:
			if(id != NULL && ps != NULL){
				save_account_conf(printer, user, id, ps, 0, useAccount);
			}
			else{
				return -1;
			}
			break;
		case 1:
		case 11:
		case 21:
			save_account_conf(printer, user, NULL, NULL, 1, useAccount);
			break;
		case 3:
		case 13:
			save_account_status(printer, useAccount);
			break;
		case 4:
			save_account_status(printer,0);
			break;
		default:
			return -1;
	}
#endif

	return 0;
}

int CheckCUPSEntry(char *printer)
{
	cups_dest_t *dest;
	cups_dest_t *curr;
	int num = 0;
	int ret = 1;

	if((num = cupsGetDests(&dest)) != 0){
		if((curr = cupsGetDest(printer, NULL, num, dest)) != NULL)
			ret = 0;
		cupsFreeDests(num, dest);
	}

	return ret;
}

#define EUC_CODE	"EUC-JP"
#define UTF_CODE	"UTF-8"
#define SJIS_CODE	"SHIFT-JIS"
#define ASCII_CODE	"ASCII"
#define LATIN1_CODE	"LATIN1"
#define GB18030_CODE	"GB18030"
#define CP949_CODE		"CP949"
#define BIG5HKSCS_CODE	"BIG5-HKSCS"
#define EUC_KR_CODE		"EUC-KR"
#define EUC_CN_CODE		"EUC-CN"
#define EUC_TW_CODE		"EUC-TW"

typedef enum
{
	ENCODING_ASCII,
	ENCODING_EUC,
	ENCODING_UTF8,
	ENCODING_SJIS,
	ENCODING_LATIN1,
	ENCODING_EUC_KR,
	ENCODING_EUC_CN,
	ENCODING_EUC_TW,
	ENCODING_GB18030,
	ENCODING_CP949,
	ENCODING_BIG5HKSCS
} ENCODING;

typedef enum
{
	LANG_NONE,
	LANG_JA,
	LANG_EN,
	LANG_EN_US,
	LANG_FR,
	LANG_IT,
	LANG_DE,
	LANG_ES,
	LANG_CN,
	LANG_TW,
	LANG_KR
} LANG;


bool convert_encoding(
	const char *from_code,
	const char *to_code,
	const char *from_buf,
	char *to_buf,
	size_t out_bytes)
{
	iconv_t icon_d;
	bool	result	= false;

	if (! to_buf)
	{
		return false;
	}
	if (! from_code || ! to_code || ! from_buf)
	{
		to_buf[0] = '\0';
		return false;
	}

	icon_d = iconv_open(to_code, from_code);

	if ((int)(icon_d) != -1)
	{
		size_t ret;
		char *to_ptr = (char *)to_buf;
		char *from_ptr = (char *)from_buf;
		size_t from_len = strlen(from_buf);
		size_t to_len = out_bytes;

		ret = iconv(icon_d, &from_ptr, &from_len, &to_ptr, &to_len);
		if (ret == -1)
		{
			if (errno == E2BIG)
			{
				*to_ptr = '\0';
				result = true;
			}
			else
			{
				to_buf[0] = '\0';
				result = false;
			}
		}
		else
		{
			*to_ptr = '\0';
			result = true;
		}
		iconv_close(icon_d);
	}
	else
	{
		to_buf[0] = '\0';
	}
	return result;
}

LANG GetLang(void)
{
	char	*pLang	= NULL;
	LANG	lang	= LANG_NONE;

	pLang = getenv("LC_CTYPE");
	if (! pLang)
	{
		pLang = getenv("LANG");
	}
	if (pLang)
	{
		if (strncasecmp(pLang, "ja", 2) == 0)
		{
			lang = LANG_JA;
		}
		else if (strncasecmp(pLang, "en", 2) == 0)
		{
			if (strncasecmp(pLang, "en_us", 5) == 0)
			{
				lang = LANG_EN_US;
			}
			else
			{
				lang = LANG_EN;
			}
		}
		else if (strncasecmp(pLang, "fr", 2) == 0)
		{
			lang = LANG_FR;
		}
		else if (strncasecmp(pLang, "it", 2) == 0)
		{
			lang = LANG_IT;
		}
		else if (strncasecmp(pLang, "de", 2) == 0)
		{
			lang = LANG_DE;
		}
		else if (strncasecmp(pLang, "es", 2) == 0)
		{
			lang = LANG_ES;
		}
		else if (strncasecmp(pLang, "ko", 2) == 0)
		{
			lang = LANG_KR;
		}
		else if (strncasecmp(pLang, "zh_CN", 5) == 0)
		{
			lang = LANG_CN;
		}
		else if (strncasecmp(pLang, "zh_TW", 5) == 0)
		{
			lang = LANG_TW;
		}
		else
		{
			lang = LANG_NONE;
		}
	}
	return lang;
}
bool wrap_convert_encoding_utf8(
	const char	*from_buf,
	char		*to_buf,
	size_t		out_bytes,
	ENCODING	*encoding)
{
	LANG	lang	= GetLang();
	bool	result	= false;
	char	work[1024];

	if (lang == LANG_JA)
	{
		if (convert_encoding(UTF_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(EUC_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(SJIS_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
	}
	else if (lang == LANG_KR)
	{
		if (convert_encoding(UTF_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(CP949_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(EUC_KR_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
	}
	else if (lang == LANG_CN)
	{
		if (convert_encoding(UTF_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(EUC_CN_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(GB18030_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
	}
	else if (lang == LANG_TW)
	{
		if (convert_encoding(UTF_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(EUC_TW_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(BIG5HKSCS_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
	}
	else
	{
		if (convert_encoding(UTF_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(EUC_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(ASCII_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
		else if (convert_encoding(UTF_CODE, UTF_CODE, from_buf, work, sizeof(work)))
		{
			if (convert_encoding(UTF_CODE, UTF_CODE, work, to_buf, out_bytes))
			{
				*encoding = ENCODING_UTF8;
				result = true;
			}
			else
			{
				if (convert_encoding(LATIN1_CODE, UTF_CODE, work, to_buf, out_bytes))
				{
					*encoding = ENCODING_UTF8;
					result = true;
				}
			}
		}
		else if (convert_encoding(LATIN1_CODE, UTF_CODE, from_buf, to_buf, out_bytes))
		{
			*encoding = ENCODING_UTF8;
			result = true;
		}
	}
	return result;
}

#define ASCII_FLAG		0x80
#define TWO_BYTE_WORD	0x8F
#define TWO_BYTE		2
#define THREE_BYTE_WORD	0x40
#define THREE_BYTE		3

int CheckEnter(char *checkStr, int maxbyte)
{
	char* tmp = NULL;
	int i = 0;

	tmp = checkStr;
	if( NULL != tmp ){
		int num = 0, cnt = 0;
		num = strlen(tmp);
		for(i = 0; i < num; i++, cnt++){
			char ch = tmp[i];
			if(cnt >= maxbyte){
				break;
			}

			if((ch & ASCII_FLAG) != 0){
				int addcnt = 0;

				if((ch & THREE_BYTE_WORD) != 0){
					if((ch & (THREE_BYTE_WORD >> 1)) != 0){
						if((ch & (THREE_BYTE_WORD >> 2)) != 0){
							i += THREE_BYTE;
							addcnt = THREE_BYTE;
						}else{
							i += TWO_BYTE;
							addcnt = TWO_BYTE;
						}
					}else{
						i++;
					}
				}
				cnt+=addcnt;
				if(cnt >= maxbyte){
					i -= (addcnt+1);
				}
			}
		}

		tmp[i] = 0x00;
	}

	return i;
}

static int convertInputString(const char* src, char* dst, int dstSize )
{
	int length = 0;

	if( (NULL != src) && (NULL != dst) && (0 < dstSize) ){
		char* 		tmp		= NULL;

		tmp = strdup( src );
		if( NULL != tmp ){
			ENCODING	encode	= 0;
			bool		result	= false;
			result = wrap_convert_encoding_utf8( tmp, dst, dstSize, &encode );
			if( true == result ){
				length = CheckEnter( dst, dstSize );
				dst[length] = 0x00;
			} else {
				dst[0] = 0x00;
				length = 0;
			}
		} else {
			dst[0] = 0x00;
			length = 0;
		}

		if( NULL != tmp ){
			free( tmp );
		}
	}
	return length;
}

int GetInputID(char *id, int useAccount)
{
	int c;
	int cnt = 0;
	int maxlen = 0;
	char	input[512];

	memset( input, 0x00, sizeof( input ) );
	if( (3 == useAccount) || (2 == useAccount) ){
		maxlen = MAX_INPUT_LENGTH_USER;
		fprintf(stdout, "user name : ");
	} else {
		maxlen = MAX_INPUT_LENGTH_ID;
		fprintf(stdout, "id : ");
	}

	id[cnt] = 0;
	while((c = getchar()) != '\n'){
		if( (3 == useAccount) || (2 == useAccount) ){
			if(cnt < (sizeof(input)-1)){
				input[cnt] = c;
			}
		} else {
			if(cnt < maxlen){
				id[cnt] = c;
			}
		}
		cnt++;
	}
	if( (3 == useAccount) || (2 == useAccount) ){
		input[cnt] = 0x00;
		cnt = convertInputString( input, id, maxlen );
	}

	return cnt;
}

int GetInputPSWD(char *ps, int useAccount)
{
	struct termios new, org;
	int c;
	int cnt = 0;
	int maxlen = 0;
	char	input[512];

	memset( input, 0x00, sizeof( input ) );
	if( 2 == useAccount ){
		maxlen = MAX_INPUT_LENGTH_USER;
	} else {
		maxlen = MAX_INPUT_LENGTH_ID;
	}

	tcgetattr(STDIN_FILENO, &new);
	org = new;

	new.c_lflag &=~ECHO;
	new.c_lflag |= ECHONL;
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &new);

	fprintf(stdout, "pin : ");

	ps[cnt] = 0;
	while((c = getchar()) != '\n'){
		if( (3 == useAccount) || (2 == useAccount) ){
			if(cnt < (sizeof(input)-1)){
				input[cnt] = c;
			}
		} else {
			if(cnt < maxlen){
				ps[cnt] = c;
			}
		}
		cnt++;
	}

	tcsetattr(STDIN_FILENO, TCSANOW, &org);
	if( (3 == useAccount) || (2 == useAccount) ){
		input[cnt] = 0x00;
		cnt = convertInputString( input, ps, maxlen );
	}

	if( (0 == cnt) && (3 != useAccount) ){
		cnt = 1;
	}

	return cnt;
}

int GetArg(int ac, char *av[])
{
	int ret = 0;

	if(strncmp(av[1], "-ps", 3) == 0){
		ret = 20;
	}
	else if(strncmp(av[1], "-xs", 3) == 0){
		ret = 21;
	}
	else if(strncmp(av[1], "-pu", 3) == 0){
		ret = 10;
	}
	else if(strncmp(av[1], "-xu", 3) == 0){
		ret = 11;
	}
	else if(strncmp(av[1], "-eu", 3) == 0){
		ret = 13;
	} else
	if(strncmp(av[1], "-p", 2) == 0)
		ret = 0;
	else if(strncmp(av[1], "-x", 2) == 0)
		ret = 1;
	else if(strncmp(av[1], "-e", 2) == 0)
		ret = 3;
	else if(strncmp(av[1], "-d", 2) == 0)
		ret = 4;
	else
		ret = -1;

	if(CheckCUPSEntry(av[2]))
		return 2;

	return ret;
}

void Usage(void)
{
	fprintf(stderr, "cnjatool Usage: \n");
	fprintf(stderr, "Setting Job Accounting ID/Password \n");
	fprintf(stderr, "  cnjatool -p [Printer Name]\n");
	fprintf(stderr, "Remove ID/Password \n");
	fprintf(stderr, "  cnjatool -x [Printer Name]\n\n");
	fprintf(stderr, "Setting User Authentication Function User Name/Password \n");
	fprintf(stderr, "  cnjatool -pu [Printer Name]\n");
	fprintf(stderr, "Remove User Name/Password \n");
	fprintf(stderr, "  cnjatool -xu [Printer Name]\n\n");
	fprintf(stderr, "Setting Secured Print Function User Name/Password \n");
	fprintf(stderr, "  cnjatool -ps [Printer Name]\n");
	fprintf(stderr, "Remove User Name/Password \n");
	fprintf(stderr, "  cnjatool -xs [Printer Name]\n\n");
	fprintf(stderr, "Root privilege is required to perform following operations \n");
	fprintf(stderr, "Enable Job Accounting \n");
	fprintf(stderr, "  cnjatool -e [Printer Name]\n");
	fprintf(stderr, "Enable User Authentication Function \n");
	fprintf(stderr, "  cnjatool -eu [Printer Name]\n");
	fprintf(stderr, "Disable Job Accounting and User Authentication Function \n");
	fprintf(stderr, "  cnjatool -d [Printer Name]\n");
}

int main(int ac, char *av[])
{
	char *user = NULL;
	char id[36];
	char ps[36];
	int useAccount = 0;
	int ret = 0;

	if(ac != 3){
		Usage();
		return 0;
	}

	if(getuid() != 0){
		if((user = getenv("USER")) == NULL)
			return 1;
	}

	ret = GetArg(ac, av);
	if( (ret == 20) || (ret == 21) ){
		useAccount = 3;
	} else if( (10 <= ret) && (13 >= ret) ){
		useAccount = 2;
	} else if( (0 <= ret) && (3 >= ret) ){
		useAccount = 1;
	} else {
		useAccount = 0;
	}

	if((ret == 0) || (ret == 10) || (ret == 20)){
		memset(id, 0, sizeof(id));
		memset(ps, 0, sizeof(ps));
		while(GetInputID(id, useAccount) == 0);
		while(GetInputPSWD(ps, useAccount) == 0);
		SaveJobAccount(av[2], user, id, ps, 0, useAccount);
	}else if((ret == 1) || (ret == 11) || (ret == 21)){
		SaveJobAccount(av[2], user, NULL, NULL, 1, useAccount);
	}else if(ret == 2){
		fprintf(stderr, "No Printer Queue Entry\n");
	}else if((ret == 3) || (ret == 4) || (ret == 13)){
		if(getuid() == 0 ){
			SaveJobAccount(av[2], user, NULL, NULL, ret, useAccount);
		}
		else{
			Usage();
		}
	}else{
		Usage();
	}

	return 0;
}

