/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <string.h>

#include <glib.h>
 
#include <libraw1394/raw1394.h>
#include <libraw1394/csr.h>

#include <librbc/rbcdisk.h>
#include <librbc/rbccommand.h>

#include "sbp2byteswap.h"
#include "sbp2constants.h"
#include "sbp2csr.h"
#include "sbp2main.h"
#include "sbp2pagetable.h"
#include "sbp2raw1394.h"
#include "sbp2scsicommand.h"
#include "sbp2status.h"
#include "sbp2worker.h"


enum
{
  CMD_AGENT_STATE               = 0x00,
  CMD_AGENT_RESET               = 0x04,
  CMD_ORB_POINTER               = 0x08,
  CMD_DOORBELL                  = 0x10,
  CMD_UNSOLICITED_STATUS_ENABLE = 0x14
};

typedef enum
{
  STATE_RESET     = 0x0,
  STATE_ACTIVE    = 0x1,
  STATE_SUSPENDED = 0x2,
  STATE_DEAD      = 0x3
} State;


typedef struct raw1394_arm_request ARMRequest;
typedef struct raw1394_arm_request_response ARMReqResp;

typedef struct _ORBData ORBData;

struct _ORBData
{
  SBP2Worker         *worker;

  gsize               chunk_size;

  nodeid_t            node;
  SBP2Pointer         pointer;

  SBP2ScsiCommandORB  orb;
  SBP2Status         *status;

  RBCBuffer          *buffer;

  SBP2Pointer        *page_pointers;
};

struct _SBP2Worker
{
  raw1394handle_t               handle;
  guint                         id;

  RBCDisk                      *disk;

  gboolean                      unattached;

  State                         state;
  gboolean                      doorbell;

  guint                         login_ID;

  SBP2Pointer                  *pointer;

  nodeaddr_t                    csr_address;
  SBP2Pointer                   status_FIFO;

  struct raw1394_arm_reqhandle  arm_agent_state;
  struct raw1394_arm_reqhandle  arm_agent_reset;
  struct raw1394_arm_reqhandle  arm_orb_pointer;
  struct raw1394_arm_reqhandle  arm_doorbell;
  struct raw1394_arm_reqhandle  arm_unsolicited_status_enable;
};


static gint     agent_reset               (raw1394handle_t  handle,
                                           ARMReqResp      *arm_req_resp,
					   guint            requested_length,
					   gpointer         pcontext,
					   guint8           request_type);
static gint     orb_pointer               (raw1394handle_t  handle,
                                           ARMReqResp      *arm_req_resp,
					   guint            requested_length,
					   gpointer         pcontext,
					   guint8           request_type);
static gint     doorbell                  (raw1394handle_t  handle,
                                           ARMReqResp      *arm_req_resp,
					   guint            requested_length,
					   gpointer         pcontext,
					   guint8           request_type);
static gint     unsolicited_status_enable (raw1394handle_t  handle,
                                           ARMReqResp      *arm_req_resp,
					   guint            requested_length,
					   gpointer         pcontext,
					   guint8           request_type);

static void     fetch_orb                 (SBP2Worker      *worker,
                                           nodeid_t         node,
					   SBP2Pointer     *pointer);
static gboolean process_orb               (gpointer         data);

static gboolean fetch_data                (gpointer         data);
static gboolean process_data              (gpointer         data);

static gboolean fetch_page_table          (gpointer         data);
static gboolean process_page_table        (gpointer         data);

static gboolean fetch_page_table_data     (gpointer         data);

static void     complete_orb              (ORBData         *orb_data);

static void     write_orb_data_buffer     (ORBData         *orb_data);

static void     free_orb_data             (gpointer         data);

static gint     bus_reset_handler         (raw1394handle_t  handle,
                                           guint            generation);


SBP2Worker *
sbp2_worker_new (RBCDisk      *disk,
                 GMainContext *context,
                 gint          port,
                 guint         login_ID)
{
  SBP2Worker      *worker = NULL;
  raw1394handle_t  handle;
  nodeaddr_t       csr_address;
  gint             ret;

  handle = sbp2_raw1394_handle ();

  if (!handle)
    return NULL;

  if (raw1394_set_port (handle, port) != 0)
    goto fail;

  csr_address = CSR_REGISTER_BASE + SBP2_CSR_COMMAND_BLOCK_AGENT_BASE +
                (SBP2_CSR_COMMAND_BLOCK_AGENT_SIZE * login_ID);

  worker = g_new (SBP2Worker, 1);

  /* TODO: Way to update the register */
  ret = raw1394_arm_register (handle,
                              csr_address + CMD_AGENT_STATE,
			      4, NULL,
                              0,
			      RAW1394_ARM_READ,
			      0,
			      0);
  if (ret < 0)
    goto fail;

  worker->arm_agent_reset.arm_callback = agent_reset;
  worker->arm_agent_reset.pcontext = worker;

  ret = raw1394_arm_register (handle,
                              csr_address + CMD_AGENT_RESET,
			      4, NULL,
                              (gulong) &worker->arm_agent_reset,
			      RAW1394_ARM_WRITE,
			      RAW1394_ARM_WRITE,
			      0);
  if (ret < 0)
    goto fail;

  worker->arm_orb_pointer.arm_callback = orb_pointer;
  worker->arm_orb_pointer.pcontext = worker;

  ret = raw1394_arm_register (handle,
                              csr_address + CMD_ORB_POINTER,
			      8, NULL,
                              (gulong) &worker->arm_orb_pointer,
			      RAW1394_ARM_READ | RAW1394_ARM_WRITE,
			      RAW1394_ARM_WRITE,
			      0);
  if (ret < 0)
    goto fail;

  worker->arm_doorbell.arm_callback = doorbell;
  worker->arm_doorbell.pcontext = worker;

  ret = raw1394_arm_register (handle,
                              csr_address + CMD_DOORBELL,
			      4, NULL,
                              (gulong) &worker->arm_doorbell,
			      RAW1394_ARM_WRITE,
			      RAW1394_ARM_WRITE,
			      0);
  if (ret < 0)
    goto fail;

  worker->arm_unsolicited_status_enable.arm_callback = unsolicited_status_enable;
  worker->arm_unsolicited_status_enable.pcontext = worker;

  ret = raw1394_arm_register (handle,
                              csr_address + CMD_UNSOLICITED_STATUS_ENABLE,
			      4, NULL,
                              (gulong) &worker->arm_unsolicited_status_enable,
			      RAW1394_ARM_WRITE,
			      RAW1394_ARM_WRITE,
			      0);
  if (ret < 0)
    goto fail;

  raw1394_set_userdata (handle, worker);
  raw1394_set_bus_reset_handler (handle, bus_reset_handler);

  worker->disk = disk;

  worker->unattached = TRUE;

  worker->state = STATE_RESET;

  worker->csr_address = csr_address;

  worker->handle = handle;
  worker->id = sbp2_watch_handle (handle, context);

  return worker;

fail:
  g_free (worker);
  raw1394_destroy_handle (handle);

 return NULL; 
}

void
sbp2_worker_destroy (SBP2Worker *worker)
{
  g_source_remove (worker->id);

  raw1394_destroy_handle (worker->handle);

  g_free (worker);
}

gboolean
sbp2_worker_login (SBP2Worker   *worker,
                   nodeid_t      node,
		   SBP2Pointer  *pointer,
                   SBP2LoginORB *orb)
{
  SBP2LoginResponse *resp;
  SBP2Status        *status;
  guint              length;

  memcpy (&worker->status_FIFO, &orb->status_FIFO,
          sizeof (worker->status_FIFO));

  resp = g_new (SBP2LoginResponse, 1);

  if (orb->resp_length >= 16)
    length = 16;
  else
    length = 12;

  resp->length = length;

  resp->login_ID = worker->login_ID;

  resp->reconnect_hold = SBP2_RECONNECT_HOLD;

  resp->command_block_agent.node = raw1394_get_local_id (worker->handle);

  sbp2_pointer_from_1394_address (&resp->command_block_agent,
                                  worker->csr_address);

  sbp2_cpu_to_be32_data (resp, length);
  sbp2_write_data (worker->handle, 0, node, &orb->login_resp, resp, length);

  status = g_new0 (SBP2Status, 1);

  sbp2_status_fill_pointer (status, pointer);

  status->len = 1;
  status->src = SBP2_STATUS_ORIGIN_NULL;
  status->resp = SBP2_STATUS_RESP_REQUEST_COMPLETE;

  sbp2_cpu_to_be32_data (status, 8);
  sbp2_write_data (worker->handle, 0, node, &orb->status_FIFO, status, 8);

  worker->unattached = FALSE;

  return TRUE;
}

gboolean
sbp2_worker_reconnect (SBP2Worker *worker,
                       nodeid_t    node)
{
  g_return_val_if_fail (worker != NULL, FALSE);

  worker->unattached = FALSE;

  return TRUE;
}

static gint
agent_reset (raw1394handle_t  handle,
             ARMReqResp      *arm_req_resp,
	     guint            requested_length,
	     gpointer         pcontext,
	     guint8           request_type)
{
  SBP2Worker *worker = pcontext;

  g_return_val_if_fail (worker != NULL, -1);

  if (worker->unattached)
    return 0;
 
  worker->state = STATE_RESET;
  worker->doorbell = 0;

  return 0;
}

static gint
orb_pointer (raw1394handle_t  handle,
             ARMReqResp      *arm_req_resp,
	     guint            requested_length,
	     gpointer         pcontext,
	     guint8           request_type)
{
  ARMRequest  *request;
  SBP2Worker  *worker = pcontext;
  SBP2Pointer *pointer;
 
  g_return_val_if_fail (worker != NULL, -1);

  if (worker->state == STATE_DEAD)
    return 0;

  request = arm_req_resp->request;

  worker->state = STATE_ACTIVE;
  worker->doorbell = 0;
 
  pointer = (SBP2Pointer *) request->buffer;
  sbp2_be32_to_cpu_data (pointer, sizeof (*pointer));

  fetch_orb (worker, request->source_nodeid, pointer);

  return 0;
}

static gint
doorbell (raw1394handle_t  handle,
          ARMReqResp      *arm_req_resp,
	  guint            requested_length,
	  gpointer         pcontext,
	  guint8           request_type)
{
  ARMRequest *request;
  SBP2Worker *worker = pcontext;

  g_return_val_if_fail (worker != NULL, -1);
 
  if (worker->state != STATE_SUSPENDED)
    return 0;

  request = arm_req_resp->request;

  worker->state = STATE_ACTIVE;

  fetch_orb (worker, request->source_nodeid, worker->pointer);

  return 0;
}


static gint
unsolicited_status_enable (raw1394handle_t  handle,
			   ARMReqResp      *arm_req_resp,
			   guint            requested_length,
			   gpointer         pcontext,
			   guint8           request_type)
{
  /* TODO */
  return 0;
}

static void
fetch_orb (SBP2Worker  *worker,
           nodeid_t     node,
	   SBP2Pointer *pointer)
{
  ORBData *orb_data;
  SBP2Tag *tag;

  g_return_if_fail (worker != NULL);
  g_return_if_fail (pointer != NULL);

  orb_data = g_new0 (ORBData, 1);

  orb_data->worker = worker;

  orb_data->node = node;
  memcpy (&orb_data->pointer, pointer, sizeof (orb_data->pointer));

  tag = sbp2_tag_new (process_orb, free_orb_data, orb_data);

  sbp2_read_data (worker->handle, 0, node, pointer,
                  &orb_data->orb, sizeof (orb_data->orb),
		  tag);
}

static gboolean
process_orb (gpointer data)
{
  ORBData            *orb_data = data;
  SBP2Worker         *worker;
  nodeid_t            node;
  SBP2Pointer        *pointer;
  SBP2ScsiCommandORB *orb;
  SBP2Status         *status;
  RBCBuffer          *buffer;

  g_return_val_if_fail (orb_data != NULL, FALSE);

  worker = orb_data->worker;

  node = orb_data->node;
  pointer = &orb_data->pointer;
  orb = &orb_data->orb;

  sbp2_be32_to_cpu_data (orb, 20);

  orb_data->chunk_size = 1 << (orb->max_payload + 2);

  orb_data->status = status = g_new0 (SBP2Status, 1);

  sbp2_status_fill_pointer (status, pointer);

  status->len = 1;
  status->resp = SBP2_STATUS_RESP_REQUEST_COMPLETE;

  if (orb->req_fmt == 0x0)
    {
      if (orb->has_page_table)
	{
	  if (orb->page_size == 0)
	    return fetch_page_table (orb_data);
	  else
	    status->sbp_status = SBP2_STATUS_REQUEST_PAGE_SIZE_NOT_SUPPORTED;
	}
      else
	{
	  if (orb->data_size != 0)
	    {
	      if (!orb->direction)
		return fetch_data (orb_data);
	      else
		buffer = rbc_buffer_simple_new (orb->data_size);
	    }
	  else
	    buffer = rbc_buffer_empty_new ();

	  orb_data->buffer = buffer;

	  if (rbc_disk_command (worker->disk, orb->cdb, buffer, status->sense))
	    write_orb_data_buffer (orb_data);
	  else
	    status->len = 7;
	}
    }
  else
    status->sbp_status = SBP2_STATUS_REQUEST_DUMMY_ORB_COMPLETED;

  complete_orb (orb_data);

  return TRUE;
}

static gboolean
fetch_data (gpointer data)
{
  ORBData            *orb_data = data;
  SBP2Worker         *worker;
  SBP2ScsiCommandORB *orb;
  RBCBuffer          *buffer;
  gsize               chunk_size;
  SBP2Tag            *tag;

  g_return_val_if_fail (orb_data != NULL, FALSE);

  worker = orb_data->worker;

  orb = &orb_data->orb;

  chunk_size = orb_data->chunk_size;

  orb_data->buffer = buffer = rbc_buffer_simple_new (orb->data_size);

  tag = sbp2_tag_new (process_data, free_orb_data, data);

  sbp2_read_data (worker->handle, chunk_size, orb_data->node, &orb->data_desc,
                  buffer->data, buffer->length,
		  tag);

  return FALSE;
}

static gboolean
process_data (gpointer data)
{
  ORBData            *orb_data = data;
  SBP2Worker         *worker;
  SBP2ScsiCommandORB *orb;
  RBCBuffer          *buffer;
  SBP2Status         *status;

  g_return_val_if_fail (orb_data != NULL, FALSE);
  g_return_val_if_fail (orb_data->status != NULL, FALSE);

  worker = orb_data->worker;

  orb = &orb_data->orb;
  status = orb_data->status;

  buffer = orb_data->buffer;

  if (!rbc_disk_command (worker->disk, orb->cdb, buffer, status->sense))
    status->len = 7;

  complete_orb (orb_data);

  return TRUE;
}

static gboolean
fetch_page_table (gpointer data)
{
  ORBData            *orb_data = data;
  SBP2Worker         *worker;
  SBP2ScsiCommandORB *orb;
  gsize               chunk_size;
  gsize               length;
  RBCBuffer          *buffer;
  SBP2Tag            *tag;

  g_return_val_if_fail (orb_data != NULL, FALSE);

  worker = orb_data->worker;

  orb = &orb_data->orb;

  chunk_size = orb_data->chunk_size;

  length = sizeof (SBP2Page) * orb->data_size;

  orb_data->buffer = buffer = rbc_buffer_simple_new (length);

  tag = sbp2_tag_new (process_page_table, free_orb_data, data);

  sbp2_read_data (worker->handle, chunk_size, orb_data->node, &orb->data_desc,
                  buffer->data, length,
		  tag);

  return FALSE;
}

static gboolean
process_page_table (gpointer data)
{
  ORBData            *orb_data = data;
  SBP2Worker         *worker;
  SBP2ScsiCommandORB *orb;
  gsize               data_size;
  RBCBuffer          *buffer;
  SBP2Page           *ext_page;
  RBCPage            *page;
  SBP2Pointer        *page_pointer;
  gsize               i;
  SBP2Status         *status;

  g_return_val_if_fail (orb_data != NULL, TRUE);

  worker = orb_data->worker;

  orb = &orb_data->orb;
  data_size = orb->data_size;

  ext_page = orb_data->buffer->data;
  sbp2_be32_to_cpu_data (ext_page, orb_data->buffer->length);

  buffer = rbc_buffer_page_table_new (data_size);
  page = buffer->data;

  orb_data->page_pointers = page_pointer = g_new (SBP2Pointer, data_size);

  for (i = data_size; i > 0; i--, page++, page_pointer++, ext_page++)
    {
      page->length = ext_page->segment_length;
      page->data = g_malloc (page->length);

      sbp2_pointer_from_page (page_pointer, ext_page);
    }

  rbc_buffer_destroy (orb_data->buffer);

  orb_data->buffer = buffer;

  if (orb->direction)
    {
      status = orb_data->status;

      if (rbc_disk_command (worker->disk, orb->cdb, buffer, status->sense))
	write_orb_data_buffer (orb_data);
      else
	status->len = 7;

      complete_orb (orb_data);

      return TRUE;
    }

  return fetch_page_table_data (orb_data);
}

static gboolean
fetch_page_table_data (gpointer data)
{
  ORBData            *orb_data = data;
  SBP2Worker         *worker;
  raw1394handle_t     handle;
  nodeid_t            node;
  SBP2ScsiCommandORB *orb;
  gsize               chunk_size;
  RBCBuffer          *buffer;
  RBCPage            *page;
  SBP2Pointer        *page_pointer;
  gsize               i;
  SBP2Tag            *tag;

  g_return_val_if_fail (orb_data != NULL, FALSE);

  worker = orb_data->worker;
  handle = worker->handle;

  node = orb_data->node;
  orb = &orb_data->orb;

  chunk_size = orb_data->chunk_size;

  buffer = orb_data->buffer;

  page = buffer->data;
  page_pointer = orb_data->page_pointers;

  tag = sbp2_tag_new (process_data, free_orb_data, data);

  for (i = buffer->length; i > 0; i--, page++, page_pointer++)
    sbp2_read_data (handle, chunk_size, node, page_pointer,
		    page->data, page->length,
		    tag);

  return FALSE;
}

static void
complete_orb (ORBData *orb_data)
{
  SBP2Worker         *worker;
  nodeid_t            node;
  SBP2ScsiCommandORB *orb;
  SBP2Status         *status;
  gsize               length;

  g_return_if_fail (orb_data != NULL);
  g_return_if_fail (orb_data->status != NULL);

  worker = orb_data->worker;

  node = orb_data->node;
  orb = &orb_data->orb;
  status = orb_data->status;

  length = (status->len + 1) * 4;

  if (!sbp2_pointer_is_null_orb (&orb->next_ORB))
    {
      status->src = SBP2_STATUS_ORIGIN_NON_NULL;
      fetch_orb (worker, node, &orb->next_ORB);
    }
  else
    status->src = SBP2_STATUS_ORIGIN_NULL;

  sbp2_cpu_to_be32_data (status, length);
  sbp2_write_data (worker->handle, 0, node, &worker->status_FIFO,
                   status, length);

  orb_data->status = NULL;
}

static void
write_orb_data_buffer (ORBData *orb_data)
{
  SBP2Worker         *worker;
  raw1394handle_t     handle;
  nodeid_t            node;
  SBP2ScsiCommandORB *orb;
  gsize               chunk_size;
  RBCBuffer          *buffer;
  SBP2Pointer        *page_pointer;
  RBCPage            *page;
  gsize               i;

  g_return_if_fail (orb_data != NULL);
  g_return_if_fail (orb_data->buffer != NULL);

  worker = orb_data->worker;
  handle = worker->handle;

  node = orb_data->node;
  orb = &orb_data->orb;

  chunk_size = orb_data->chunk_size;

  buffer = orb_data->buffer;
  page_pointer = orb_data->page_pointers;

  switch (buffer->type)
    {
    case RBC_BUFFER_PAGE_TABLE:
      page = buffer->data;

      for (i = buffer->length; i > 0; i--, page++, page_pointer++)
	{
	  sbp2_write_data (handle, chunk_size, node, page_pointer,
			   page->data, page->length);

	  page->data = NULL;
	}

      break;

    case RBC_BUFFER_SIMPLE:
      sbp2_write_data (handle, chunk_size, node, &orb->data_desc,
		       buffer->data, buffer->length);

      buffer->data = NULL;
      break;

    case RBC_BUFFER_EMPTY:
      /* do nothing */
      break;

    default:
      g_assert_not_reached ();
      break;
    }
}

static void
free_orb_data (gpointer data)
{
  ORBData *orb_data = data;

  g_return_if_fail (orb_data != NULL);

  if (orb_data->status)
    g_free (orb_data->status);

  if (orb_data->buffer)
    rbc_buffer_destroy (orb_data->buffer);

  if (orb_data->page_pointers)
    g_free (orb_data->page_pointers);

  g_free (orb_data);
}

static gint
bus_reset_handler (raw1394handle_t handle,
                   guint           generation)
{
  SBP2Worker *worker;

  g_return_val_if_fail (handle != NULL, -1);

  raw1394_update_generation (handle, generation);

  worker = raw1394_get_userdata (handle);

  g_return_val_if_fail (worker != NULL, -1);

  worker->state = STATE_RESET;

  worker->unattached = TRUE;

  return 0;
}
