#!/usr/bin/env python3

# Copyright 2021 Stanford University
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

from __future__ import print_function
import argparse, hashlib, multiprocessing, os, platform, re, subprocess, sys, tempfile, traceback

def discover_llvm_version():
    if os.environ.get('LMOD_SYSTEM_NAME') == 'summit': # Summit doesn't set hostname
        return '60'
    elif os.environ.get('NERSC_HOST') == 'perlmutter':
        return '110'
    else:
        return '60'

def discover_skip_certificate_check():
    if platform.node().startswith('titan'):
        return True
    else:
        return False

def discover_conduit():
    if 'CONDUIT' in os.environ:
        return os.environ['CONDUIT']
    elif platform.node().startswith('cori'):
        return 'aries'
    elif platform.node().startswith('daint'):
        return 'aries'
    elif platform.node().startswith('excalibur'):
        return 'aries'
    elif platform.node().startswith('quartz'):
        return 'psm'
    elif os.environ.get('LMOD_SYSTEM_NAME') == 'summit': # Summit doesn't set hostname
        return 'ibv'
    elif os.environ.get('NERSC_HOST') == 'perlmutter':
        return 'ucx'
    else:
        raise Exception('Please set CONDUIT in your environment')

def gasnet_enabled():
    if 'USE_GASNET' in os.environ:
        return os.environ['USE_GASNET'] == '1'
    return platform.system() != 'Darwin'

def hdf_enabled():
    return 'USE_HDF' in os.environ and os.environ['USE_HDF'] == '1'

def check_sha256(file_path, sha256):
    with open(file_path, 'rb') as f:
        assert hashlib.sha256(f.read()).hexdigest() == sha256

def download(dest_path, url, sha256, insecure=False):
    dest_dir = os.path.dirname(dest_path)
    dest_file = os.path.basename(dest_path)
    insecure_flag = []
    if insecure:
        insecure_flag = ['--insecure']

    if os.path.exists(dest_path):
        check_sha256(dest_path, sha256)
        return

    subprocess.check_call(['curl', '-L'] + insecure_flag + ['-o', dest_path, url])
    check_sha256(dest_path, sha256)

def extract(dest_dir, archive_path, format):
    if format == 'gz':
        subprocess.check_call(['tar', 'xfz', archive_path], cwd=dest_dir)
    elif format == 'xz':
        subprocess.check_call(['tar', 'xfJ', archive_path], cwd=dest_dir)
    else:
        raise Exception('Unknown format %s' % format)

def apply_patch(dest_dir, diff_path, strip_levels=1):
    subprocess.check_call(['patch', '-p%d' % strip_levels, '-i', diff_path], cwd=dest_dir)

def git_clone(repo_dir, url, branch=None):
    if branch is not None:
        subprocess.check_call(['git', 'clone', '-b', branch, url, repo_dir])
    else:
        subprocess.check_call(['git', 'clone', url, repo_dir])

def git_update(repo_dir):
    subprocess.check_call(
        ['git', 'pull', '--ff-only'],
        cwd=repo_dir)

def build_gasnet(gasnet_dir, conduit):
    subprocess.check_call(['make', 'CONDUIT=%s' % conduit], cwd=gasnet_dir)

def build_llvm(source_dir, build_dir, install_dir, use_cmake, cmake_exe, thread_count, is_cray):
    env = None
    if is_cray:
        env = dict(list(os.environ.items()) + [
            ('CC', os.environ['HOST_CC']),
            ('CXX', os.environ['HOST_CXX']),
        ])
    if use_cmake:
        subprocess.check_call(
            [cmake_exe,
             '-DCMAKE_INSTALL_PREFIX=%s' % install_dir,
             '-DCMAKE_BUILD_TYPE=Release',
             '-DLLVM_ENABLE_ASSERTIONS=OFF',
             '-DLLVM_ENABLE_ZLIB=OFF',
             '-DLLVM_ENABLE_TERMINFO=OFF',
             '-DLLVM_ENABLE_LIBEDIT=OFF',
             source_dir],
            cwd=build_dir,
            env=env)
    else:
        subprocess.check_call(
            [os.path.join(source_dir, 'configure'),
             '--prefix=%s' % install_dir,
             '--enable-optimized',
             '--disable-assertions',
             '--disable-zlib',
             '--disable-terminfo'],
            cwd=build_dir,
            env=env)
    subprocess.check_call(['make', '-j', str(thread_count)], cwd=build_dir)
    subprocess.check_call(['make', 'install'], cwd=build_dir)

def build_terra(terra_dir, terra_branch, terra_lua, use_cmake, cmake_exe, llvm_dir, cache, is_cray, thread_count):
    if cache:
        assert not use_cmake
        subprocess.check_call(['make', 'download'], cwd=terra_dir)
        return

    env = dict(list(os.environ.items()))
    if is_cray:
        env.update(dict([
            ('CC', os.environ['HOST_CC']),
            ('CXX', os.environ['HOST_CXX']),
        ]))

    if terra_lua is not None:
        assert use_cmake

    if use_cmake:
        flags = [
            '-DCMAKE_PREFIX_PATH=%s' % llvm_dir,
            '-DCMAKE_INSTALL_PREFIX=%s' % os.path.join(terra_dir, 'release'),
            '-DTERRA_LUA=%s' % (terra_lua or 'moonjit'),
        ]
        subprocess.check_call(
            [cmake_exe] + flags + [terra_dir],
            cwd=os.path.join(terra_dir, 'build'),
            env=env)
        subprocess.check_call(
            ['make', 'install', '-j', str(thread_count)],
            cwd=os.path.join(terra_dir, 'build'),
            env=env)
    else:
        flags = [
            'LLVM_CONFIG=%s' % os.path.join(llvm_dir, 'bin', 'llvm-config'),
            'CLANG=%s' % os.path.join(llvm_dir, 'bin', 'clang'),
        ]
        if platform.system() != 'Darwin':
            flags.append('REEXPORT_LLVM_COMPONENTS=irreader mcjit x86')
        flags.extend(['-j', str(thread_count)])

        subprocess.check_call(
            ['make'] + flags,
            cwd=terra_dir,
            env=env)

def build_hdf(source_dir, install_dir, thread_count, is_cray):
    env = None
    if is_cray:
        env = dict(list(os.environ.items()) + [
            ('CC', os.environ['HOST_CC']),
            ('CXX', os.environ['HOST_CXX']),
        ])
    subprocess.check_call(
        ['./configure',
         '--prefix=%s' % install_dir,
         '--enable-threadsafe',
         '--disable-hl'],
        cwd=source_dir,
        env=env)
    subprocess.check_call(['make', '-j', str(thread_count)], cwd=source_dir)
    subprocess.check_call(['make', 'install'], cwd=source_dir)

def build_regent(root_dir, use_cmake, cmake_exe, extra_flags,
                 gasnet_dir, llvm_dir, terra_dir, hdf_dir, conduit, thread_count):
    env = dict(list(os.environ.items()) +
        ([('CONDUIT', conduit),
          ('GASNET', gasnet_dir),
          ('USE_GASNET', '1')]
         if gasnet_enabled() else []) +
        ([('HDF_ROOT', hdf_dir),
          ('USE_HDF', '1')]
         if hdf_enabled() else []) +
        [('LLVM_CONFIG', os.path.join(llvm_dir, 'bin', 'llvm-config')),
         ('CMAKE_PREFIX_PATH', llvm_dir)]
    )

    subprocess.check_call(
        [os.path.join(root_dir, 'install.py'),
         '--with-terra', terra_dir,
         '--rdir', 'auto',
         '-j', str(thread_count),
        ] + (['--cmake', '--with-cmake', cmake_exe]
             if use_cmake else ['--no-cmake']) +
        ['--extra=%s' % flag for flag in extra_flags],
        env=env)

def install_llvm(llvm_dir, llvm_install_dir, scratch_dir, llvm_version, llvm_use_cmake, cmake_exe, thread_count, cache, is_cray, insecure):
    try:
        os.mkdir(llvm_dir)
    except OSError:
        pass # Hope this means it already exists
    assert(os.path.isdir(llvm_dir))

    # mirror = 'http://sapling.stanford.edu/~eslaught/llvm'
    mirror = 'https://releases.llvm.org'
    if llvm_version == '35':
        llvm_tarball = os.path.join(llvm_dir, 'llvm-3.5.2.src.tar.xz')
        llvm_source_dir = os.path.join(llvm_dir, 'llvm-3.5.2.src')
        clang_tarball = os.path.join(llvm_dir, 'cfe-3.5.2.src.tar.xz')
        clang_source_dir = os.path.join(llvm_dir, 'cfe-3.5.2.src')
        download(llvm_tarball, '%s/3.5.2/llvm-3.5.2.src.tar.xz' % mirror, '44196156d5749eb4b4224fe471a29cc3984df92570a4a89fa859f7394fc0c575', insecure=insecure)
        download(clang_tarball, '%s/3.5.2/cfe-3.5.2.src.tar.xz' % mirror, '4feb575f74fb3a74b6245400460230141bf610f235ef3a25008cfe6137828620', insecure=insecure)
    elif llvm_version == '38':
        llvm_tarball = os.path.join(llvm_dir, 'llvm-3.8.1.src.tar.xz')
        llvm_source_dir = os.path.join(llvm_dir, 'llvm-3.8.1.src')
        clang_tarball = os.path.join(llvm_dir, 'cfe-3.8.1.src.tar.xz')
        clang_source_dir = os.path.join(llvm_dir, 'cfe-3.8.1.src')
        download(llvm_tarball, '%s/3.8.1/llvm-3.8.1.src.tar.xz' % mirror, '6e82ce4adb54ff3afc18053d6981b6aed1406751b8742582ed50f04b5ab475f9', insecure=insecure)
        download(clang_tarball, '%s/3.8.1/cfe-3.8.1.src.tar.xz' % mirror, '4cd3836dfb4b88b597e075341cae86d61c63ce3963e45c7fe6a8bf59bb382cdf', insecure=insecure)
    elif llvm_version == '39':
        llvm_tarball = os.path.join(llvm_dir, 'llvm-3.9.1.src.tar.xz')
        llvm_source_dir = os.path.join(llvm_dir, 'llvm-3.9.1.src')
        clang_tarball = os.path.join(llvm_dir, 'cfe-3.9.1.src.tar.xz')
        clang_source_dir = os.path.join(llvm_dir, 'cfe-3.9.1.src')
        download(llvm_tarball, '%s/3.9.1/llvm-3.9.1.src.tar.xz' % mirror, '1fd90354b9cf19232e8f168faf2220e79be555df3aa743242700879e8fd329ee', insecure=insecure)
        download(clang_tarball, '%s/3.9.1/cfe-3.9.1.src.tar.xz' % mirror, 'e6c4cebb96dee827fa0470af313dff265af391cb6da8d429842ef208c8f25e63', insecure=insecure)
    elif llvm_version == '60':
        llvm_tarball = os.path.join(llvm_dir, 'llvm-6.0.1.src.tar.xz')
        llvm_source_dir = os.path.join(llvm_dir, 'llvm-6.0.1.src')
        clang_tarball = os.path.join(llvm_dir, 'cfe-6.0.1.src.tar.xz')
        clang_source_dir = os.path.join(llvm_dir, 'cfe-6.0.1.src')
        download(llvm_tarball, '%s/6.0.1/llvm-6.0.1.src.tar.xz' % mirror, 'b6d6c324f9c71494c0ccaf3dac1f16236d970002b42bb24a6c9e1634f7d0f4e2', insecure=insecure)
        download(clang_tarball, '%s/6.0.1/cfe-6.0.1.src.tar.xz' % mirror, '7c243f1485bddfdfedada3cd402ff4792ea82362ff91fbdac2dae67c6026b667', insecure=insecure)
    elif llvm_version == '90':
        llvm_tarball = os.path.join(llvm_dir, 'llvm-9.0.0.src.tar.xz')
        llvm_source_dir = os.path.join(llvm_dir, 'llvm-9.0.0.src')
        clang_tarball = os.path.join(llvm_dir, 'cfe-9.0.0.src.tar.xz')
        clang_source_dir = os.path.join(llvm_dir, 'cfe-9.0.0.src')
        download(llvm_tarball, '%s/9.0.0/llvm-9.0.0.src.tar.xz' % mirror, 'd6a0565cf21f22e9b4353b2eb92622e8365000a9e90a16b09b56f8157eabfe84', insecure=insecure)
        download(clang_tarball, '%s/9.0.0/cfe-9.0.0.src.tar.xz' % mirror, '7ba81eef7c22ca5da688fdf9d88c20934d2d6b40bfe150ffd338900890aa4610', insecure=insecure)
    elif llvm_version == '110':
        mirror = 'https://github.com/llvm/llvm-project/releases/download'
        llvm_tarball = os.path.join(llvm_dir, 'llvm-11.1.0.src.tar.xz')
        llvm_source_dir = os.path.join(llvm_dir, 'llvm-11.1.0.src')
        clang_tarball = os.path.join(llvm_dir, 'clang-11.1.0.src.tar.xz')
        clang_source_dir = os.path.join(llvm_dir, 'clang-11.1.0.src')
        download(llvm_tarball, '%s/llvmorg-11.1.0/llvm-11.1.0.src.tar.xz' % mirror, 'ce8508e318a01a63d4e8b3090ab2ded3c598a50258cc49e2625b9120d4c03ea5', insecure=insecure)
        download(clang_tarball, '%s/llvmorg-11.1.0/clang-11.1.0.src.tar.xz' % mirror, '0a8288f065d1f57cb6d96da4d2965cbea32edc572aa972e466e954d17148558b', insecure=insecure)
    else:
        assert False

    if not cache:
        extract(llvm_dir, llvm_tarball, 'xz')
        extract(llvm_dir, clang_tarball, 'xz')
        if llvm_version == '35':
            apply_patch(llvm_source_dir, os.path.join(os.path.dirname(os.path.realpath(__file__)), 'llvm-3.5-gcc.patch'))
        elif llvm_version == '38':
            apply_patch(llvm_source_dir, os.path.join(os.path.dirname(os.path.realpath(__file__)), 'llvm-3.8-gcc.patch'))
        os.rename(clang_source_dir, os.path.join(llvm_source_dir, 'tools', 'clang'))

        llvm_build_dir = tempfile.mkdtemp(prefix='setup_env_llvm_build', dir=scratch_dir or llvm_dir)
        os.mkdir(llvm_install_dir)
        build_llvm(llvm_source_dir, llvm_build_dir, llvm_install_dir, llvm_use_cmake, cmake_exe, thread_count, is_cray)

def install_hdf(hdf_dir, hdf_install_dir, thread_count, cache, is_cray, insecure):
    try:
        os.mkdir(hdf_dir)
    except OSError:
        pass # Hope this means it already exists
    assert(os.path.isdir(hdf_dir))
    hdf_tarball = os.path.join(hdf_dir, 'hdf5-1.10.1.tar.gz')
    hdf_source_dir = os.path.join(hdf_dir, 'hdf5-1.10.1')
    download(hdf_tarball, 'http://sapling.stanford.edu/~manolis/hdf/hdf5-1.10.1.tar.gz', '048a9d149fb99aaa1680a712963f5a78e9c43b588d0e79d55e06760ec377c172', insecure=insecure)
    if not cache:
        extract(hdf_dir, hdf_tarball, 'gz')
        build_hdf(hdf_source_dir, hdf_install_dir, thread_count, is_cray)

def print_advice(component_dir):
    print('Given the number of things that could potentially have gone')
    print('wrong, this script is not designed to handle this situation.')
    print('You will need to fix the problem on your own (or ask for help')
    print('fixing it).')
    print()
    print('The files are located here:')
    print()
    print(component_dir)
    print()
    print('Once you have fixed the problem, you have two options:')
    print()
    print(' 1. Go to the directory and rebuild it yourself. This')
    print('    script will not touch the directory again.')
    print()
    print(' 2. Remove the directory. Then rerun this script.')
    print()
    print('Good luck and please ask for help if you get stuck!')

def report_build_failure(name, component_dir, exception):
    print()
    print('#' * 68)
    print('## Build Failed')
    print('#' * 68)
    print()
    print('It appears that %s has failed to build. The failure was:' % name)
    print()
    traceback.print_exc()
    print()
    print_advice(component_dir)
    sys.exit(1)

def check_dirty_build(name, build_result, component_dir):
    if not os.path.exists(build_result):
        print()
        print('#' * 68)
        print('## Dirty Previous Build Detected')
        print('#' * 68)
        print()
        print('It appears that %s was not built successfully on a' % name)
        print('previous invocation of this script.')
        print()
        print_advice(component_dir)
        sys.exit(1)

def driver(prefix_dir=None, scratch_dir=None, cache=False,
           legion_use_cmake=False, extra_flags=[], llvm_version=None,
           terra_url=None, terra_branch=None, terra_lua=None, terra_use_cmake=None,
           thread_count=None, insecure=False):
    if not cache:
        if 'CC' not in os.environ:
            raise Exception('Please set CC in your environment')
        if 'CXX' not in os.environ:
            raise Exception('Please set CXX in your environment')
        if 'LG_RT_DIR' in os.environ:
            raise Exception('Please unset LG_RT_DIR in your environment')

    is_cray = 'CRAYPE_VERSION' in os.environ

    if not cache and is_cray:
        print('This system has been detected as a Cray system.')
        print()
        print('Note: The Cray wrappers are broken for various purposes')
        print('(particularly, dynamically linked libraries). For this')
        print('reason this script requires that HOST_CC and HOST_CXX')
        print('be set to the underlying compilers (GCC and G++, etc.).')
        print()
        if 'HOST_CC' not in os.environ:
            raise Exception('Please set HOST_CC in your environment')
        if 'HOST_CXX' not in os.environ:
            raise Exception('Please set HOST_CXX in your environment')

    if llvm_version in ('35', '38'):
        llvm_use_cmake = False
    elif llvm_version in ('39', '60', '90', '110'):
        llvm_use_cmake = True
    else:
        raise Exception('Unrecognized LLVM version %s' % llvm_version)

    if terra_use_cmake is None:
        terra_use_cmake = llvm_use_cmake

    root_dir = os.path.dirname(os.path.dirname(os.path.realpath(__file__)))
    legion_dir = os.path.dirname(root_dir)

    if prefix_dir is None:
        prefix_dir = root_dir
    else:
        prefix_dir = os.path.abspath(prefix_dir)

    if thread_count is None:
        try:
            # this correctly considers the current affinity mask
            thread_count = len(os.sched_getaffinity(0))
        except AttributeError:
            # this works on macos
            thread_count = multiprocessing.cpu_count()

    gasnet_release_dir = None
    conduit = None
    if gasnet_enabled():
        gasnet_dir = os.path.realpath(os.path.join(prefix_dir, 'gasnet'))
        if not os.path.exists(gasnet_dir):
            git_clone(gasnet_dir, 'https://github.com/StanfordLegion/gasnet.git')
        if not cache:
            conduit = discover_conduit()
            gasnet_release_dir = os.path.join(gasnet_dir, 'release')
            gasnet_build_result = os.path.join(
                gasnet_release_dir, '%s-conduit' % conduit,
                'libgasnet-%s-par.a' % conduit)
            if not os.path.exists(gasnet_release_dir):
                try:
                    build_gasnet(gasnet_dir, conduit)
                except Exception as e:
                    report_build_failure('gasnet', gasnet_dir, e)
            else:
                check_dirty_build('gasnet', gasnet_build_result, gasnet_dir)
            assert os.path.exists(gasnet_build_result)

    cmake_exe = None
    try:
        cmake_version = subprocess.check_output(['cmake', '--version']).decode('utf-8')
    except:
        pass # Can't find CMake, continue to download
    else:
        m = re.match(r'cmake version (\d+)[.](\d+)', cmake_version)
        if m is not None and (int(m.group(1)) < 3 or int(m.group(2)) < 1):
            pass # CMake is too old, continue to download
        elif m is None:
            raise Exception('Cannot parse CMake version:\n\n%s' % cmake_version)
        else:
            cmake_exe = 'cmake' # CMake is ok, use it
    if cache or ((legion_use_cmake or llvm_use_cmake or terra_use_cmake) and cmake_exe is None):
        cmake_stem = 'cmake-3.7.2-%s-x86_64' % platform.system()
        cmake_basename = '%s.tar.gz' % cmake_stem
        cmake_url = 'https://cmake.org/files/v3.7/%s' % cmake_basename
        if cmake_stem == 'cmake-3.7.2-Linux-x86_64':
            cmake_shasum = '0e6ec35d4fa9bf79800118916b51928b6471d5725ff36f1d0de5ebb34dcd5406'
        elif cmake_stem == 'cmake-3.7.2-Darwin-x86_64':
            cmake_shasum = '0175e97748052dfc15ebd3c0aa65286e5ec20ca22ed606ce88940e699496b03c'

        cmake_dir = os.path.realpath(os.path.join(prefix_dir, 'cmake'))
        cmake_install_dir = os.path.join(cmake_dir, cmake_stem)
        if not os.path.exists(cmake_dir):
            os.mkdir(cmake_dir)

            proc_type = subprocess.check_output(['uname', '-p']).decode('utf-8').strip()
            if proc_type != 'x86_64' and proc_type != 'i386':
                raise Exception("Don't know how to download CMake binary for %s" % proc_type)

            cmake_tarball = os.path.join(cmake_dir, cmake_basename)
            download(cmake_tarball, cmake_url, cmake_shasum, insecure=insecure)
            extract(cmake_dir, cmake_tarball, 'gz')
        assert os.path.exists(cmake_install_dir)
        cmake_exe = os.path.join(cmake_install_dir, 'bin', 'cmake')

    llvm_dir = os.path.realpath(os.path.join(prefix_dir, 'llvm'))
    llvm_install_dir = os.path.join(llvm_dir, 'install')
    llvm_build_result = os.path.join(llvm_install_dir, 'bin', 'llvm-config')
    if not os.path.exists(llvm_install_dir):
        try:
            install_llvm(llvm_dir, llvm_install_dir, scratch_dir, llvm_version, llvm_use_cmake, cmake_exe, thread_count, cache, is_cray, insecure)
        except Exception as e:
            report_build_failure('llvm', llvm_dir, e)
    else:
        check_dirty_build('llvm', llvm_build_result, llvm_dir)
    if not cache:
        assert os.path.exists(llvm_build_result)

    terra_dir = os.path.join(prefix_dir, 'terra.build')
    terra_build_dir = os.path.join(terra_dir, 'build', 'bin')
    terra_build_result = os.path.join(terra_dir, 'release', 'bin', 'terra')
    if not os.path.exists(terra_dir):
        git_clone(terra_dir, terra_url, terra_branch)
    if not os.path.exists(terra_build_dir):
        try:
            build_terra(terra_dir, terra_branch, terra_lua, terra_use_cmake, cmake_exe, llvm_install_dir, cache, is_cray, thread_count)
        except Exception as e:
            report_build_failure('terra', terra_dir, e)
    else:
        check_dirty_build('terra', terra_build_result, terra_dir)
    if not cache:
        assert os.path.exists(terra_build_result)

    hdf_install_dir = None
    if hdf_enabled():
        hdf_dir = os.path.join(prefix_dir, 'hdf')
        hdf_install_dir = os.path.join(hdf_dir, 'install')
        hdf_build_result = os.path.join(hdf_install_dir, 'lib', 'libhdf5.so')
        if not os.path.exists(hdf_install_dir):
            try:
                install_hdf(hdf_dir, hdf_install_dir, thread_count, cache, is_cray, insecure)
            except Exception as e:
                report_build_failure('hdf', hdf_dir, e)
        else:
            check_dirty_build('hdf', hdf_build_result, hdf_dir)
        if not cache:
            assert os.path.exists(hdf_build_result)

    if not cache:
        build_regent(root_dir, legion_use_cmake, cmake_exe, extra_flags,
                     gasnet_release_dir, llvm_install_dir, terra_dir, hdf_install_dir,
                     conduit, thread_count)

if __name__ == '__main__':
    parser = argparse.ArgumentParser(
        description='Setup tool for Regent.')
    parser.add_argument(
        '--prefix', dest='prefix_dir', required=False,
        help='Directory in which to install dependencies.')
    parser.add_argument(
        '--scratch', dest='scratch_dir', required=False,
        help='Directory in which to store temporary build files.')
    parser.add_argument(
        '--cache-only', dest='cache', action='store_true',
        help='Only cache downloads (do not install).')
    parser.add_argument(
        '--skip-certificate-check', dest='insecure', action='store_true',
        default=discover_skip_certificate_check(),
        help='Skip certificate checks on downloads.')
    parser.add_argument(
        '--cmake', dest='legion_use_cmake', action='store_true',
        default=os.environ.get('USE_CMAKE') == '1',
        help='Use CMake to build Legion.')
    parser.add_argument(
        '--extra', dest='extra_flags', action='append', required=False,
        default=[],
        help='Extra flags for Make/CMake command.')
    parser.add_argument(
        '--llvm-version', dest='llvm_version', required=False, choices=('35', '38', '39', '60', '90', '110'),
        default=discover_llvm_version(),
        help='Select LLVM version.')
    parser.add_argument(
        '--terra-url', dest='terra_url', required=False,
        default='https://github.com/StanfordLegion/terra.git',
        help='URL of Terra repository to clone from.')
    parser.add_argument(
        '--terra-branch', dest='terra_branch', required=False,
        default='luajit2.1',
        help='Branch of Terra repository to checkout.')
    parser.add_argument(
        '--terra-lua', dest='terra_lua', required=False,
        default=None,
        help='Lua implementation to use for Terra (luajit or moonjit).')
    parser.add_argument(
        '--terra-cmake', dest='terra_use_cmake', action='store_true', default=None,
        help='Use CMake to build Terra.')
    parser.add_argument(
        '--no-terra-cmake', dest='terra_use_cmake', action='store_false', default=None,
        help='Use CMake to build Terra.')
    parser.add_argument(
        '-j', dest='thread_count', nargs='?', type=int,
        help='Number threads used to compile.')
    args = parser.parse_args()
    driver(**vars(args))
