/*
 * intf.h - interface monitor
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 *
 * Description:
 *
 *     This module keeps track of statistics about each configured interface
 *     by reading /proc/net/dev. Bytes and packet rates per seconds are
 *     calculated frequently and can be displayed in a table showing the
 *     current rate or in form of a diagram showing the last 60
 *     seconds/minutes/hours graphical like mrtg or rrd tool.
 *
 *
 * Howto Use:
 *
 *     1) Initialize a curses screen
 *     2) Call intf_readfile("/proc/net/dev") every second in your mainloop
 *     3) Call intf_handle_input(key) for every input key received _after_
 *        you've handled it
 *     4) Call intf_draw() if needed, this is normally if intf_handle_input()
 *        returned 1, if the window was resized and at least every second.
 */

#ifndef INTF_H_
#define INTF_H_

#define MAX_IF 255                              /* max configured interfaces */
#define IFNAME_MAX 12                           /* max interface name length */

typedef struct intf_s {

    char name[IFNAME_MAX];                                 /* interface name */

    unsigned long rx_bytes;                              /* total recv bytes */
    unsigned long rx_bytes_old;           /* total recv bytes at last update */
    unsigned int  rx_bytes_rate;                /* current recv bytes rate/s */

    unsigned long tx_bytes;                              /* total sent bytes */
    unsigned long tx_bytes_old;           /* total sent bytes at last update */
    unsigned int  tx_bytes_rate;                /* current sent bytes rate/s */

    unsigned long rx_packets;                          /* total recv packets */
    unsigned long rx_packets_old;       /* total recv packets at last update */
    unsigned int  rx_packets_rate;            /* current recv packets rate/s */
   
    unsigned long tx_packets;                          /* total sent packets */
    unsigned long tx_packets_old;       /* total sent packets at last update */
    unsigned int  tx_packets_rate;            /* current sent packets rate/s */

    unsigned long rx_errors;
    unsigned long rx_drop;
    unsigned long rx_fifo;
    unsigned long rx_frame;
    unsigned long rx_compressed;      
    unsigned long rx_multicast;       

    unsigned long tx_errors;
    unsigned long tx_drop;
    unsigned long tx_fifo;
    unsigned long tx_frame;
    unsigned long tx_compressed;
    unsigned long tx_multicast;

    time_t        time;                 /* last time the rate was calculated */


    /*
     * table to display the bandwidth graphically
     * each table consists of 60 slices.
     */
    struct intf_g_s {

        unsigned long rx_bytes[60];
        unsigned long tx_bytes[60];

        unsigned long rx_bytes_old;
        unsigned long tx_bytes_old;

        char rx_g[6][61];               /* actual table like it is displayed */
        char tx_g[6][61];            
        int i;                                    /* points to current slice */


        float rx_scale[6];                              /* values for y axis */
        float tx_scale[6];

    } s, m, h;

    time_t g_time;             /* last time the seconds table was calculated */

    int to_remove:1;                 /* flag to mark interfaces as to remove */

} intf_t;


/*
 * reads /proc/net/dev and updates each interface entry. must be called
 * frequently like every second
 */
void intf_readfile(const char *path);


/*
 * handles user input:
 *  - change selected interface
 *  - change modes (overview, graphical, detailed)
 *  - change unit (seconds, minutes, hours)
 *
 *  Keys:
 *   'o'          overview mode
 *   'd'          detailed mode
 *   'g' or <CR>  graphical mode
 *   'S'          unit -> seconds
 *   'M'          unit -> minutes
 *   'H'          unit -> hours
 *   Up           previous interface
 *   Down         next interface
 *   '0'-'9'      jump to interface entry n
 *
 * Returns 1 if you should redraw the screen (i.e. call intf_draw())
 * otherwise 0.
 */
int intf_handle_input(int ch);


/*
 * draws the current selected mode to the current curses screen
 */
void intf_draw(void);

#endif
