const std = @import("std");
const log = std.log.scoped(.main);
const mem = std.mem;
const process = std.process;

const vaxis = @import("vaxis");
const View = vaxis.widgets.View;
const Cell = vaxis.Cell;
const border = vaxis.widgets.border;

const Event = union(enum) {
    key_press: vaxis.Key,
    winsize: vaxis.Winsize,
};

pub fn main() !void {
    var gpa = std.heap.GeneralPurposeAllocator(.{}){};
    defer {
        const deinit_status = gpa.deinit();
        if (deinit_status == .leak) {
            log.err("memory leak", .{});
        }
    }
    const alloc = gpa.allocator();

    var world_map: []const u8 = lg_world_map;
    var map_width = lg_map_width;
    var map_height = lg_map_height;
    var use_sm_map = false;
    var use_mini_view = false;

    var x: u16 = 0;
    var y: u16 = 0;
    var h: u16 = 0;
    var w: u16 = 0;
    defer log.info(
        \\Results:
        \\ - Map Size:    {d}x{d}
        \\ - Screen Size: {d}x{d}
        \\ - Position:    {d}, {d}
    , .{
        map_width, map_height,
        w,         h,
        x,         y,
    });

    var tty = try vaxis.Tty.init();
    defer tty.deinit();

    var buffered_writer = tty.bufferedWriter();
    const writer = buffered_writer.writer().any();

    // Initialize Vaxis
    var vx = try vaxis.init(alloc, .{
        .kitty_keyboard_flags = .{ .report_events = true },
    });
    defer vx.deinit(alloc, tty.anyWriter());
    var loop: vaxis.Loop(Event) = .{
        .vaxis = &vx,
        .tty = &tty,
    };
    try loop.init();
    try loop.start();
    defer loop.stop();
    try vx.enterAltScreen(writer);
    try buffered_writer.flush();
    try vx.queryTerminal(tty.anyWriter(), 20 * std.time.ns_per_s);

    // Initialize Views
    // - Large Map
    var lg_map_view = try View.init(alloc, &vx.unicode, .{ .width = lg_map_width, .height = lg_map_height });
    defer lg_map_view.deinit();
    //w = lg_map_view.screen.width;
    //h = lg_map_view.screen.height;
    var lg_map_buf: [lg_map_width * lg_map_height]u8 = undefined;
    _ = mem.replace(u8, lg_world_map, "\n", "", lg_map_buf[0..]);
    _ = lg_map_view.printSegment(.{ .text = lg_map_buf[0..] }, .{ .wrap = .grapheme });
    // - Small Map
    var sm_map_view = try View.init(alloc, &vx.unicode, .{ .width = sm_map_width, .height = sm_map_height });
    defer sm_map_view.deinit();
    w = sm_map_view.screen.width;
    h = sm_map_view.screen.height;
    var sm_map_buf: [sm_map_width * sm_map_height]u8 = undefined;
    _ = mem.replace(u8, sm_world_map, "\n", "", sm_map_buf[0..]);
    _ = sm_map_view.printSegment(.{ .text = sm_map_buf[0..] }, .{ .wrap = .grapheme });
    // - Active Map
    var map_view = lg_map_view;

    // TUI Loop
    while (true) {
        const event = loop.nextEvent();
        switch (event) {
            .key_press => |key| {
                // Close Demo
                if (key.matches('c', .{ .ctrl = true })) break;
                // Scroll
                if (key.matches(vaxis.Key.left, .{})) x -|= 1;
                if (key.matches(vaxis.Key.right, .{})) x +|= 1;
                if (key.matches(vaxis.Key.up, .{})) y -|= 1;
                if (key.matches(vaxis.Key.down, .{})) y +|= 1;
                // Quick Scroll
                if (key.matches(vaxis.Key.left, .{ .ctrl = true })) x -|= 30;
                if (key.matches(vaxis.Key.right, .{ .ctrl = true })) x +|= 30;
                if (key.matches(vaxis.Key.up, .{ .ctrl = true })) y -|= 10;
                if (key.matches(vaxis.Key.down, .{ .ctrl = true })) y +|= 10;
                // Goto Side
                if (key.matches(vaxis.Key.left, .{ .shift = true })) x -|= map_width;
                if (key.matches(vaxis.Key.right, .{ .shift = true })) x +|= map_width;
                if (key.matches(vaxis.Key.up, .{ .shift = true })) y -|= map_height;
                if (key.matches(vaxis.Key.down, .{ .shift = true })) y +|= map_height;
                // Change Zoom (Swap Map Views)
                if (key.matches('z', .{})) {
                    if (use_sm_map) {
                        world_map = lg_world_map;
                        map_width = lg_map_width;
                        map_height = lg_map_height;
                        map_view = lg_map_view;
                    } else {
                        world_map = sm_world_map;
                        map_width = sm_map_width;
                        map_height = sm_map_height;
                        map_view = sm_map_view;
                    }
                    use_sm_map = !use_sm_map;
                    w = map_width;
                    h = map_height;
                }
                // Mini View (Forced Width & Height Limits)
                if (key.matches('m', .{})) use_mini_view = !use_mini_view;
            },
            .winsize => |ws| try vx.resize(alloc, tty.anyWriter(), ws),
        }

        const win = vx.window();
        win.clear();

        const controls_win = win.child(.{
            .height = 1,
        });
        _ = controls_win.print(
            if (win.width >= 112) &.{
                .{ .text = "Controls:", .style = .{ .bold = true, .ul_style = .single } },
                .{ .text = " Exit: ctrl + c | Scroll: dpad | Quick Scroll: ctrl + dpad | Goto Side: shift + dpad | Zoom: z | Mini: m" },
            } else if (win.width >= 25) &.{
                .{ .text = "Controls:", .style = .{ .bold = true, .ul_style = .single } },
                .{ .text = " Win too small!" },
            } else &.{
                .{ .text = "" },
            },
            .{ .wrap = .none },
        );

        // Views require a Window to render to.
        const map_win = if (use_mini_view)
            win.child(.{
                .y_off = controls_win.height,
                .border = .{ .where = .top },
                .width = 45,
                .height = 15,
            })
        else
            win.child(.{
                .y_off = controls_win.height,
                .border = .{ .where = .top },
            });

        // Clamp x and y
        x = @min(x, map_width -| map_win.width);
        y = @min(y, map_height -| map_win.height);

        map_view.draw(map_win, .{
            .x_off = x,
            .y_off = y,
        });
        if (use_mini_view) {
            _ = win.printSegment(
                .{ .text = "This is a mini portion of the View." },
                .{ .row_offset = 16, .col_offset = 5, .wrap = .word },
            );
        }

        // Render the screen
        try vx.render(writer);
        try buffered_writer.flush();
    }
}

const sm_world_map =
    \\ +NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN+
    \\ W................................................................................................E
    \\ W................................................................................................E
    \\ W............................ @:::::::::@  .....................    .............................E
    \\ W.....      .   .....   :::  ...::::::: ........   ......  %@%...........:*%@    . ..............E
    \\ W..@::::::::::::::::::::  :::  ::::@  @@ ..... %#%%%%%%%%%%%%...........................* .......E
    \\ W ::::@::::::::::::: .. ::  ... * .......  . %%% %%%%%%%%%%%%%..................... ..* .........E
    \\ W  .....@::::::::::::.=::::: ........... @%  @. %%      %%%-...................:%%  @. ..........E
    \\ W...... :::%@@:@:::::::::@ :-...........  @%%%%%%%%%%%%%%%%........................ .. ..........E
    \\ W......@::::@@@.=-:::::=:  ............. %@%%@# %%%%%%%..%%%.........@............ @=............E
    \\ W..... :::::::::::::: ................. %%%    @ % ...........................  . . .............E
    \\ W..... #:::::::::::@ .................. @=====@       .........................@  .= ............E
    \\ W...... :::::@   @ ................... ============== .... ..................... .. .............E
    \\ W....... .::=..  :# ................. ================ ......   ............... .................E
    \\ W......... ::::...  @-............... ================- ....@ ...-... . ....@ ..  ...............E
    \\ W............. :=.. %  ...............==================+   ......@. .... ... .. @ ..............E
    \\ W...............  ######@ ............ ========@=@@@======@........ @.... .  ..  ................E
    \\ W................ ######### ................. ===========@............... .@  ..@ =..............E
    \\ W............... #############= ............. ========== ................. .  ..*. . ---  .......E
    \\ W................ ###@@@#@#####%.............. ======== .................... @.@.... .*-=- ......E
    \\ W................. ##@##@@#@### .............. ========-. -......................   -- + ........E
    \\ W................... ##########................=======% *= .................... @-------- .......E
    \\ W................... #######*  ................-====== .-+....................#-@--@-@#--- ......E
    \\ W................... ####### .................. ====- ........................@----------- ......E
    \\ W................... ###### ....................@-* .......................... =    ----:.....  .E
    \\ W....................####@ ......................................................... -  ....  -@ E
    \\ W.................... ###.................................................................%-@ ...E
    \\ W..................... #@........................................................................E
    \\ W......................    ......................................................................E
    \\ +SSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSS+
    \\
;
const sm_map_width = mapWidth: {
    @setEvalBranchQuota(100_000);
    break :mapWidth mem.indexOfScalar(u8, sm_world_map, '\n').?;
};
const sm_map_height = mapHeight: {
    @setEvalBranchQuota(100_000);
    break :mapHeight mem.count(u8, sm_world_map, "\n");
};

const lg_world_map =
    \\  +NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN+ 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W........................................................................................................................................................       ............................................................................................................................................................................................................................................E 
    \\  W................................................................................................................................                                           ................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................            @@@@@@@@%-::::---::::::=#@@@@....   ................................................................................................................................................................................................................................E 
    \\  W.....................................................................................................................       .@@@@:::::::::::::::::::::::::::::::::::::-@@   ...............................................................................................................................................................................................................................E 
    \\  W....................................................................................................................   %@@@#:::::::::::::::::::::::::::::::::::::@@-       ......................................................................................                 .........................................................................................................................E 
    \\  W...................................................................................................................   @:::::::::::::::::::::::::::::::::::::::::@     ....................................................................................         .@....@@:        .......................................................................................................................E 
    \\  W...............................................................................................                ....           +:::::::::::::::::::::::::::::::::-  ...........................................................................    ..         #@-.................@                   ......................................................................................................E 
    \\  W.........................................................................................         +@@@@@@@@@@    ....         @:::::::::::::::::::::::::::::::@:   ........................................................................            @@...........................%%%%*....%%@@=                       ..................................................................................E 
    \\  W........................                        ...               ....................          @:::::::::::::@     ........  @::::::::::::::::::::::::::::+%    ...............................               .......................       @%%%%@@@%%%............................................-@@   %@@*******@@                ..       ............................................................E 
    \\  W...................       *@@#+++++++++@@%===          ========                 ...     @@::@  +::::::::::::::::-@   .......   *:::::::::::::::::::::::::::%@   .............................        %@@@@@=                ..            :@%%%%%%%%%%#.........................................................................+@=                        ................................................E 
    \\  W................    :@%=:::::::::::::::::::::::%%%-.:::::::::::::+%@@                   =:::::--:::::@##%%::::::::@   .....    @:::::::::::::::::::::::%@+      ............................   @@%#%%%%%%%%%%#@@*                +***@@%%%%%%%%%%%%%##..........................................................................................-%@@@@@+          .........................................E 
    \\  W.............      .*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::%@@@=:::::::::::::::@@     @::::::::@   ....  @:::::::::::::::::::@@.                ......................    @%%%%%%%%%%%%%%%#%%%%%%%@@%%%%@@@%%%%%%%%%#%%%%%%%%%%%%=.....................................................................................................@@@%.       ....................................E 
    \\  W............  =#-:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::=@*     @+:::::=:::+@  ...   @::::::::::::::::@.        @@@@@%%@@@   ...................    @@%#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%#%%%%%%%%%%%%%%............................................................................................................@@@   ...................................E 
    \\  W.........      *::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::@  *%::::::::::@ @@    ...  *::::::::::@@@        ...    %%%%%%%%@   ................     @@%%%%%##%* -%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%#%%......................................................................................................*   @...@  ...................................E 
    \\  W.......    =@@=::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::@ =@@@@          @:::::=@     .....  *:::::::::@        .......   :=@%%@     .............      %@%%%%%%%%%@: @%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%#=.......................................................................................................@        ...................................E 
    \\  W......   @:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::@:            @:::-@    @@*    ........   @::::::@-   ..............           ...............  @@@%%%%%%%%%%@   @%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%.....................................................................................................+@@    .......................................E 
    \\  W.....  :@::::::::::::::::#%*::::::::::::::::::::::::::::::::::::::::::::::::::::@      ......   @:::::::@        ..........   ..@:@:    ..............................       ....  *@%%%%%%%%%%%@*  +@%#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%....................................................................................@ .@.=@.........@     .........................................E 
    \\  W....    @::::::::::=%@@         #%:::::::::::::::::::::::::::::::::::::::::::::-    ........   @::::::::@:@=::@   ...........        ................................   -@@    ..   @%%%@@%%%%%%%=#   @%%%%%%%%%%%%%%%%%#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%.........................................................................%@@@@@####@#     +....#%      ............................................E 
    \\  W.      -@-:::*@-           ..     @::::::::::::::::::::::::::::::::::::::::::::@+     .....   @:::::::::::::::.-  ...................................................  @%%%@%  ...         @%%%@#    #@%%%   : : -   %=  ::   :  +%%%%%%%%%%%%%%%%%%%%%%.......................................................................-.                @.....@    ...............................................E 
    \\  W   @:::::%@        .............  ::::::::::::::::::::::::::::::::::::::::::::::::%+        +@:::::::::::::::::@   ...............................................     %@%%%=   ...    %@  @%%%@    @%%%%%  %. . -        : . . +%%%%%%%%%%%%%%%#*%##%-.......................................................................:*         .....   @......@:   ..............................................E 
    \\  W@::::+@       ..................  ::::::::::::::::::::::::::::::::::::::::::::::::::.+@   @*::::::::::::::::::::@    ...........................................    :@@  @%%@#   .    #%%-       @@@%%%%%%   . . -   #    :   .  *#%%%%%%%%%%%#...............................................................................-@%%         ....   @........  ..............................................E 
    \\  W          ......................  :::::::::::::::::::::::::::::::::::::::::::::::::::-.  @::::::::::::::::::::::::*  ...........................................   @%%%@  @%%%@     @@@%%%%%%%%%%%%%%%%%%%  @. . -        : *#. *%%%%%%%%%%%%%#.........................................................................................@   ....    @.....%  ..............................................E 
    \\  W..     .........................  ::::::::::@:@:@@@:%@@%%@@@@@=@:::::::::::::::::::::=  #=:::::::::::::::::::::::@   ...........................................  :@%%%@ #%%%%%@  @@%%%%%%%%%%%%%%%%%%%%%%   %   % . .*  += *%:  -%%%%%%%%%#%............................................................................................@     ...    +@..#  ..............................................E 
    \\  W...............................  .@.::::::::@#@.@+@@%@*@:%@:@@#@:::::::::::::::::::::::::::::::::::::::::::::::::@   ...........................................   @@@  @@%@@@@ @@%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%...............................................................................................@-   ....     @   ..............................................E 
    \\  W...............................  @::::::::::@@@:@+@%%@%@:%@:@@@@:::::::::::::::::::::::::::::::::::::::%@@  @:::::@   ...........................................          @@@@@%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%.................................................................................................@    .....     ...............................................E 
    \\  W...............................   @:::::::::@=@:@*@#%@+@:%@:@@=@::::::::::::::::::::::::::::::::::::::*@   @::::::*   ................................................  .@%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%##%##%%%%%%%%%%%%%.............................................................................................@@=.-:  .........................................................E 
    \\  W...............................   @:::::::::::::::::::::::::::.::::::::::::::::::::::::::::::::::::::.     @@@:@-:+=  ................................................    @%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%#%#.......*%##%%%%%%%#%...........................................................................................-   @.@  .........................................................E 
    \\  W..............................   @::::::::.:@@::@@:@@+@@.@@@:@*:@@*:@@*:::::::::::::::::::::::::::::::::@             .............................................  .     @%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%#%%#............%%%%%%%%%%..............................@@.+@-@.@.@@@................................................@            .....................................................E 
    \\  W............................   -=:::::::::::@@*:@@*@@+@::@:@:@*@@=@:@@@::::::::::::::::::::::::::::::@@@    .....................................................          @%%%%%%%%%#%%%%%%@-@@%%%#%%%%%%%%%%%%%%%%%%%%##%#.........%#%%%%%%%%%=...........................@@=.@@..@.@.@...............................................@      *@@     ....................................................E 
    \\  W...........................   @:::::::::::::@=@:@@@%@+@@.@@@:@*@@:::@.@:::::::::::::::::::::@@#  :%@      ......................................................   @%%%%%%%%%%%%%@@@@@- *@%#@- .@%%%%%%%%%%%%%%%%%%%%%%%%%%%%%........#%%%%%%%%%%..........................*@@@.:*@.@.@+@..............................................@      =....#   ....................................................E 
    \\  W..........................   @:::::::::::::+@@@:@=@%@+@::@.@:@*#@=@:@@@:::::::::::::::::::.@           ........................................................   @%%%%%%%%%%%%%%@        #@%%@@  .@%%%%%%%%%%%%%%%#%%%%%%%%%%..........%%%%%%%#%=.........................@@=@.@@@.@.@.@............................................@    .   =..++   .....................................................E 
    \\  W.........................   @::::::::::::::-=:=:=:--=:==:=:=:=::*@::=:=::::::::::::::::@#     .................................................................   @%%%%%%%%%%%@@-   ....    .@@%%@* @%%%%@@@***%%%#......................%%%%%%%%%..................................................................................%   ...           .....................................................E 
    \\  W.........................  @.:::::::::::::::::::::::::::::::::::::::::::::::::::::::-%      ...................................................................  @%%%%%%%%%%%@     .......     @%@  @%%#@  -...............................#%%%%#.........................................................................% #*@@....+   .....    ..@  .....................................................E 
    \\  W.........................   @::::::::::::::::::::::::::::::::::::::::::::::::::::::@    .......................................................................   @%%%%%%%%%@:              @%%%%@   :%%%  .@.............................................................................................................@     @....@    ...   ...@  .....................................................E 
    \\  W..........................  @:::::::::::::::::::::::::::::::::::::::::::::::::::::-   ..........................................................................   ==-@@@@@  .@@+--*#####@#   @%@     @@@     @................................................................................................................=   @...        +...@  .....................................................E 
    \\  W..........................  @::::::::::::::::::::::::::::::::::::::::::::::::::::.@   ...........................................................................    %@@@@@@=-============@        .                @+.=.....................................................................................................@      #..@    @@%....@  .....................................................E 
    \\  W..........................  =@::::::::::::::::::::::::::::::::::::::::::::::::::@    ...........................................................................    @=====================@       ..       ......      @......................................................................................................@      +-   @.....@     .....................................................E 
    \\  W...........................   *::::::::::::::::::::::::::::::::::::::::::::::@=    ............................................................................   @-======================-@+..       ...              @.......................................................................................................@        .@...%*+    .......................................................E 
    \\  W............................   *:::::::::::::::::::::::::::::::::::::::::::@     .............................................................................  :*-============================-@:.  +====-@@@@@@@@*  %.........................................................................................................@   ..  @..-@ @@  .........................................................E 
    \\  W............................   @:::@@::::::::::::::::::::::::::::-*=:::::=.   ...............................................................................   @==================================--===============-@#..........................................................................................................-  ..   *@  @.@  .........................................................E 
    \\  W.............................   @:@ @:::::::::::::::::::.@......     #@:::   .............................................................................    ##=====================================================@ *................- #.....................................................................................=   ...  :..      .........................................................E 
    \\  W..............................  @:@ @::::::::::::::::@                :-::   ............................................................................   @@-======================================================%  -...............@   @..................................................................................+.   ...        ............................................................E 
    \\  W..............................  @::@ %::::::::::::::+    ...........   @.:   ..........................................................................    @=========================================================-@ %.................@#  @.:@@:...........................................................................@   ......    ..............................................................E 
    \\  W..............................   @:.- *-:::::::::::.@   ............    .@   .........................................................................   @============================================================-= *+.................@    .  ==@@......................................................................:.  .........................................................................E 
    \\  W...............................  =@:+% @::::::::::.@   ...........           ........................................................................   @-============================================================-@.  @................:%%:..:@           :+-...........................................................%    .........................................................................E 
    \\  W................................    =.@ -@::::::::.%  .....        -@::::*@@     ....................................................................  #-===============================================================:@  @:........................+   ..      %........................................................@%   ...........................................................................E 
    \\  W.................................        =::::::::.@   ...    #@   @:  -@:::@@:        .............................................................   @=================================================================%   @........................@   ......   @ .....................-@@-@.........................**     ............................................................................E 
    \\  W.......................................  @:::::::::.@       #::%           *:::=:        ...........................................................  *-=================================================================-   @.......................@   ........     ...................@     @..................@ -@@    .......     ....................................................................E 
    \\  W.......................................  +@.:::::::::%@@@@@@:::@   .....         @::::@   ..........................................................   @-=================================================================@.   @....................@   ............  :@...............@    .   :-...............-       ........        ..................................................................E 
    \\  W........................................    -@::::::::::::::::@    .......      .-+@@-:   ...........................................................   :==================================================================-@   @.................@    .............  @..............@    ....   *................+@   .........   @..@  ..................................................................E 
    \\  W..........................................     @@**+:.::::::::@       .........          ...........................................................   @====================================================================-+   @.............@+    ...............   :*..........*@   .......   +.................@   ........  +:..-   .................................................................E 
    \\  W.............................................          *%::::::::::@   .............................................................................  +-=====================================================================*@   #........-#@     ..................   #........**    ........       @..............-   .......  @:..@#   ................................................................E 
    \\  W....................................................     -@::::::::%   .........      ..............................................................  +-========================================================================%  @...:@@      ......................   @.......@   .............     =.............+   .......    @@..#   ...............................................................E 
    \\  W........................................................     @+::::@  ......     :+:         .......................................................   @-========================================================================-@ =%-         .......................   @......*   ...............    ..............@  .........    *@@   ...............................................................E 
    \\  W..........................................................     @:::@   ...    @@%####@.           ...................................................   %+===================================@@=-@@=@@@-@@-@@@-@@@=================.  .#@+-=@   ........................   @.....:   ................   ..=  @.......-=  ...........       ................................................................E 
    \\  W.............................................................   -::+@       ###################@   ...................................................    #-================================-@%%-@--@=@+@@=@-@-@+@-=========================@  ..........................   @...@-   ................   ..@    @...@@    ..................................................................................E 
    \\  W.............................................................    -@@:+@@@#######################@    ...................................................   *-================================@=@-@=-@=@+@@=@-=-@:@=========================@   ...........................   @.%     ................   ...@    @@:    ........    ........................................................................E 
    \\  W...............................................................         @%########################@       ...............................................   --==============================*@*@-@--@=@+@@-@-@-@+@========================-*  .............................       @+   .............    @...@       .........   :    ......................................................................E 
    \\  W...................................................................      =###########################@@*    ..............................................   @-================---========================================================@   ...............................    *.-   ...........       %%...@   .........    #.@     ....................................................................E 
    \\  W.......................................................................  .##############################+@    .............................................    @%=---=-*+--%@     @-====================================================-@   ..................................        ...........   @.@   @...@  .......    @#....-:  ....................................................................E 
    \\  W......................................................................   @################################=@  ...............................................                          @-==============================================-@   ....................................     ..............   @..@  @..@   ...     =@.....@    ..    ..............................................................E 
    \\  W.....................................................................   @##################################@   ...................................................................      +============================================--*   .........................................................   *...@  @.@        @........@           .............................................................E 
    \\  W....................................................................   @####################################@      ..................................................................   *===========================================%#    ............................................................   @..#@@       @............@ .++#@.@  ....     ....................................................E 
    \\  W...................................................................   @######################################@+         .............................................................  @==========================================-@    ...............................................................   +....@      @..........@# *.....%   ..               ............................................E 
    \\  W...................................................................  @#############################################%@+      .........................................................  @-========================================-@   ..................................................................   @-...@-    @.........+:  @...*@   ..   @---+:            .......................................E 
    \\  W...................................................................  @##################################################@@     ......................................................   =-=====================================-@    ....................................................................    @....%    @........@  @....#   ....   #+--+##---=@%      .....................................E 
    \\  W...................................................................   %##############%@@@#@@@#@%@%@@@#@%@@#################@.    .....................................................   @-===================================-@   ........................................................................   %@..@       @@@@#@   .....+   .....   @+=-----------+@     ..................................E 
    \\  W...................................................................  @###############@@###@#@#@%@%#@%#@@@@####################@=   ....................................................   @-==================================@   ..........................................................................    @..@               .:@@@@   ......      @------------#@   .................................E 
    \\  W...................................................................  +%################@@#@#@#@%@%#@%#@@@@#####################*   .....................................................   :-=================================@  .............................................................................   *...:@@@@#                ...........     #-----------@   ................................E 
    \\  W....................................................................   @#############%@%@#@#@#@%@##@%#@%@@#####################@   ......................................................   @=================================@  ..............................................................................     @........@#   ......................   @------------@   ...............................E 
    \\  W.....................................................................   #######################################################.  .......................................................   @=================================@    ..............................................................................        @@%.++@  ...............     ..   :+:@---@  @--*%   ..............................E 
    \\  W.....................................................................    @############@@#%@%%@%@@@#@@@#@#%@@%#@@##############%   ........................................................  .*===================================  ..................................................................................             ..............                      @%%@@   .............................E 
    \\  W.......................................................................   @###########@@#%@@@@%@@##@#@#@#@%@@#@@@#############*   ........................................................   %=================================-#  ....       .................................................................................................   %=@@@@=  .   :@             .............................E 
    \\  W........................................................................   @#########%@@@%@@@@%@@@%@@@#@#@%###@%@###########@    ........................................................   @-=================================%   ...   :*@   ........................................................................................          @------+  .  -=-@   ....   ...............................E 
    \\  W.........................................................................   @########@@@@%@@@@%@@#%@#@#@#@%@@%@@@###########   .........................................................   @===================================%        @-=+   .......................................................................................   #@@@@%@--------:     @--+    .....................................E 
    \\  W..........................................................................   @############################%#################+  .........................................................   ===================================-@     =@@===+   ....................................................................................     %=--------------=    .=---=+   ....................................E 
    \\  W...........................................................................     @@#########################################**  .........................................................   ==================================-@    @-======@   ...................................................................................   %+------------------*@= @-----#   ....................................E 
    \\  W..............................................................................     @#######################################@   .........................................................   +===============================-@      =======%   .................................................................................     .@-----------------------------#   ....................................E 
    \\  W.................................................................................  =#######################################@   .........................................................   -+============================-@    .   @=====+   ................................................................................    :@--------------------------------%*    ..................................E 
    \\  W.................................................................................   @#####################################@   ...........................................................   --==========================@    ..    @====-@   ..............................................................................    @--------------------------------------@   .................................E 
    \\  W..................................................................................  @#####################################@  .............................................................   %==========================@   ...  @-=====@   .............................................................................    @-----------------------------------------%   ................................E 
    \\  W..................................................................................  @*#################################%@    ..............................................................   %=========================@   ...  +-====*   .............................................................................   @-----+@@-@*@@-@@@*@@@*@@#-@@=-@--@-*@@-----=@   ...............................E 
    \\  W..................................................................................  :*############################+@*      ................................................................   %========================-@   ...   @-==-@  ..............................................................................   *-----@+@-@*@@+@---#@-*@*@-@*#-@--@-@+@-------*   ..............................E 
    \\  W..................................................................................  :*###########################@     .....................................................................  @-=======================*   .....   @@-@   ..............................................................................   *-----@-@-@*@@--@@-#@-*@%@-@=@-@--@-@-@-------@   ..............................E 
    \\  W..................................................................................  :*###########################   ........................................................................  =+=====================@    .......        ...............................................................................   @-----@#@+@#@@+@-@-#@-*@+@*@%@-@--@-@#@+------@   ..............................E 
    \\  W..................................................................................  :############################@  ........................................................................    #===================*.  ................................................................................................   @---------------------------------------------=  ...............................E 
    \\  W..................................................................................   %##########################@   .........................................................................    +================-@    ................................................................................................   =--------------------------------------------#   ...............................E 
    \\  W..................................................................................   @#########################@   ...........................................................................   @================@   ..................................................................................................   @-------------------------------------------@   ................................E 
    \\  W...................................................................................  @#######################%*   .............................................................................   @-===========-@:   ....................................................................................................   *----------------------------------------+@   .................................E 
    \\  W...................................................................................  @#######################*   ...............................................................................  *:=========-@*   ......................................................................................................   *----%            @--------------------@    ..................       ..........E 
    \\  W...................................................................................  @#####################%=   ................................................................................  .@-=-#%%%%@    ........................................................................................................  :*...               @------------------@    ..................   @@   ..........E 
    \\  W...................................................................................  @####################@    .................................................................................               ...........................................................................................................        ...........    @--------------@    ....................  @--@   .........E 
    \\  W...................................................................................   %####################   .....................................................................................     ......................................................................................................................................    @-----------=.   .....................   @---@:   .......E 
    \\  W...................................................................................   @##################%   ...................................................................................................................................................................................................................................  .%=---------@   ....................    @#-----@  .......E 
    \\  W....................................................................................   @##############%@-    ....................................................................................................................................................................................................................................      @@@       ..................     ==----=@    .......E 
    \\  W....................................................................................   @#############@     .........................................................................................................................................................................................................................................         ...................     @+----@      .........E 
    \\  W.....................................................................................   @############   ....................................................................................................................................................................................................................................................................      %@----*@:   .............E 
    \\  W......................................................................................   %##########=   .................................................................................................................................................................................................................................................................       @-----@     ...............E 
    \\  W......................................................................................   @#########@   .................................................................................................................................................................................................................................................................   =@-----#@#    ..................E 
    \\  W.......................................................................................   %#######@   .................................................................................................................................................................................................................................................................   %--+@#%      ....................E 
    \\  W.......................................................................................   @#######@@.  .................................................................................................................................................................................................................................................................           ........................E 
    \\  W........................................................................................   @########@  ..................................................................................................................................................................................................................................................................     .............................E 
    \\  W.........................................................................................   @#####%.   ....................................................................................................................................................................................................................................................................................................E 
    \\  W..........................................................................................   @#####+   ....................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................   @#####@     .................................................................................................................................................................................................................................................................................................E 
    \\  W............................................................................................   +@@@##%@#    ...............................................................................................................................................................................................................................................................................................E 
    \\  W.............................................................................................      +@@@@@   ...............................................................................................................................................................................................................................................................................................E 
    \\  W.................................................................................................          ................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  W...........................................................................................................................................................................................................................................................................................................................................................................................................E 
    \\  +SSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSS+ 
    \\
;
const lg_map_width: u16 = mapWidth: {
    @setEvalBranchQuota(100_000);
    break :mapWidth @intCast(mem.indexOfScalar(u8, lg_world_map, '\n').?);
};
const lg_map_height: u16 = mapHeight: {
    @setEvalBranchQuota(100_000);
    break :mapHeight @intCast(mem.count(u8, lg_world_map, "\n"));
};
