// Copyright 2017 VMware, Inc. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"fmt"
	"log"
	"net/http"
	_ "net/http/pprof"
	"os"

	loads "github.com/go-openapi/loads"
	flags "github.com/jessevdk/go-flags"

	"github.com/vmware/vic/lib/apiservers/service/restapi"
	"github.com/vmware/vic/lib/apiservers/service/restapi/operations"
	"github.com/vmware/vic/pkg/trace"
)

// This file was generated by the swagger tool.

func main() {

	swaggerSpec, err := loads.Analyzed(restapi.SwaggerJSON, "")
	if err != nil {
		log.Fatalln(err)
	}

	api := operations.NewVicMachineAPI(swaggerSpec)
	server := restapi.NewServer(api)
	defer server.Shutdown()

	parser := flags.NewParser(server, flags.Default)
	parser.ShortDescription = "vic-machine API"
	parser.LongDescription = "An API for interacting with vic-machine as a RESTful web service."

	server.ConfigureFlags()
	for _, optsGroup := range api.CommandLineOptionsGroups {
		_, err := parser.AddGroup(optsGroup.ShortDescription, optsGroup.LongDescription, optsGroup.Options)
		if err != nil {
			log.Fatalln(err)
		}
	}

	var pprofFlags = struct {
		Host string `long:"pprof-host" description:"the IP to listen on for pprof connections" default:"localhost" env:"PPROF_HOST"`
		Port int    `long:"pprof-port" description:"the port to listen on for pprof connections" default:"6060" env:"PPROF_PORT"`
	}{}
	_, err = parser.AddGroup("Profiling Options", "", &pprofFlags)
	if err != nil {
		log.Fatalln(err)
	}

	if _, err := parser.Parse(); err != nil {
		code := 1
		if fe, ok := err.(*flags.Error); ok {
			if fe.Type == flags.ErrHelp {
				code = 0
			}
		}
		os.Exit(code)
	}

	server.ConfigureAPI()

	go func() {
		addr := fmt.Sprintf("%s:%d", pprofFlags.Host, pprofFlags.Port)
		trace.Logger.Infof("Serving pprof at http://%s", addr)
		trace.Logger.Info(http.ListenAndServe(addr, nil))
		trace.Logger.Infof("Stopped serving pprof at http://%s", addr)
	}()

	if err := server.Serve(); err != nil {
		trace.Logger.Error(err)
		log.Fatalln(err)
	}
}
