/*
 * Common code for cracking DiskCryptor.
 *
 * This file takes replicated but common code, shared between the CPU
 * and the GPU formats, and places it into one common location.
 */

#include "arch.h"
#include "memory.h"
#include "jumbo.h"
#include "johnswap.h"
#include "xts.h"
#include "diskcryptor_common.h"

// #define DEBUG 1

struct fmt_tests diskcryptor_tests[] = {
	/* AES XTS test vectors */
	{"$diskcryptor$0*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", "openwall"},
#ifndef DEBUG
	{"$diskcryptor$0*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", "openwall"},
	{"$diskcryptor$0*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", "openwall123"},
	/* Twofish XTS test vectors */
	{"$diskcryptor$0*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", "password"},
	/* Serpent XTS test vectors */
	{"$diskcryptor$0*64702e0ca0c2e3e3a201fc3668ae074b0a6379b2fd705b230ba8aa490869723e0976758b81017b472eb758e568ba8583108ed521d31bd4c097197dc01bf75cb0e36fde97368546cfda4dad24e8c02f84b14dfe740555162993341cccac77a11a8753cfc297057c5dd30f5a8362ef905a74a9958dd4ec1803d2e68cc31973de426a52978607e8f794162e609422b80889721a0be94a1f3911fe761d3c8fe64049c8f985541d6d15c09258275a08f326cf526ef3f49b9892578dee7b01ef26351c39f40cb28f63d25e0d67ebfed4b44aaac693aff8f58f87e16a1f8fb27f5eeeed20a299107bb945095cc91dfa8bc5425951985858440f3ae3e526e5bbcbcb194a25f3e19bff8c85a9f608a405e5910eddbfcf11c78080f27bca631ffc80df0fa786951687764a9e3286c18e241f82cc7e3bab843532000fdae0ae5dca16c1e57dce1282d822fc3093324fe976006e3b5b6b3879c4eb43f8f15f1f8905375ce7aef4113904d944584374ae3ab1aae2467e70a45518aa90934096703b48ac22555e2fe5ae6ea185a579d72550c90f449f5bdf51931d1974a2fc4be8b600019766d5e27e0d60d12a717a50e42a2ce1b2a8eb0823e7600f8e79b2fe33d3a3fb4f7fe13fff91f6d7089a43eee02c663f59b69d987b05f64160347c2624e7ecc037ed4e7e141b405257e4cb0cfcabe55b550f097102eb1d5b355ac9c809873c604d6ca376a66fc10909ec319639ecb5457ff0fd5147f821141f818f7f1776addfd4b04dbad911765dacc610103e02ca1c914c21ae8897fdc19c7f506bb121cbd2f49740818d6da5f60e67c628a9011ea5550312945bfba3adf0eda7160f1d85b9a369965a683a1f0b50c4b6956439a481c87b5c45769976bbc783f8619aba5062db4206a068113273c2865853fe1c700b097d7d7f18f8cafc1e9c81996ec60c4d6e8000d5b3be24cd5b20e9150509d399fa5c17205a8500312d9adccbd92518aa17ca51fb93c939d7259165c87ba0f8e1031dc29db652e98781f8ef735b2fac055eae76c30be29a0b3af9b2cdf2cb9a2520eb92308aba2b6605d177a06bd94265eae92d2ec115ba54221fb385185147cdfa9e8ea4e2516fcf1dd0f0dd271f477cad5f0e6ec4ca1582f2f461ee712109c45b2f9ab78c8d5b2500827256f70eed8df4ce9b0043b646a4c02dacd4c457891e045130be6cb383e5ff12a6e69b1418f687492e7bd2d44d51059b0ab36cb42cc93a47c4881ca77e1f95f346bbfa91c7ca8ab060f1486d2f256159c387691213acb8b41417c3b9bf004881a9912de8a89868adfdd7e9cea05c670653f2ff9f8e7bb19409ae7e6ca026e5cd89f74bfae1087b66158339f43b743e6f57e1bce923380f6b119ad4eacf34f49aa2c4e111c2c47335df812200292b7774b498be8bff64bba8d9f8e21f5ffcf35b2563ca6a59e5002c92c577d0f5db37ddccd3ae94c27e9b5456f03bd5b1132c919ee88fcba2d62137ce1e1e10bc90c94980432736bffcecb10ae29f912d5d6b0bcbdc811cea8e5434075c3c8f1962b7573824d030fd8e90006a8f5d32ff53e03fd0c81b09057e4ed214533bae4c82a79e108e8c63d96bdec98ec891df9cf6434469c4411b98d4fcd71a6a85d2d87a8f266b6605d191c243ae7c2390eb279069f96125362ef3311e71cfdc8095808511fa98abc8040ad7a94510940b4877754fafbb2970efdd87403b727f525b9617a163139a48c03927320e043097bbe940b2ae35c06cf5849c8153660b4697bffa73c3b40a94a95563ad04c8b98a47789de5f2155c0bacf52c0caef6d1be9c6a67cdb3c32e7690144b4a2b9503a47b0c3ae48046929b70aeb1505374fef98c3db84932e873192e695cef171537fac417f7bd6990f052e5ab6d4288ed925f624bbf5b4538ccc5e6157ccb9049d70b8e5921be1591661d00bb59e84f16b9d5518f27e0f14bfb9bc8313df7eebee9e2edba2ab342795ff55df36f28cf3f6d77b6627eee2609eab70b144dd356fc277643f605bb0c99a54fbd80d462bd85e8b9cdf38355bd4cdeb528df7a33f0a2c9a159d5d9f7f5126b90f095596bfcaee81f9bdb63d430c59d29855c60a78d8c5851deddadac87bc6bb49bf4563a36cad0529a870b0df53d61d371f00a2b2635540ca131455bbd612d45543d2ea138146914d2193a0f8cb62e4f7cdd1a8b173e813e7b827aa59138ba8a95d9ea65d2c794e7e5a130d669eda896ef549828d5d556ab196a809ff835a892202674e47134a6922f219f3fc7b3b8cdbfa1b576413486f3e195f835bfaccf7a558ba29d1a07d7db35193c7f53ce0ffa829b18b377aecf7e4a998827a83f51d9f8dcc19983aa80623d06857d582f1103e3334fd87421c06f2d4dd768f9b425c0868b8c42e4a939a1683e83f0376e5f2171624910d223d90475fd83a6237a9515fd2f635a31fbd999d43477a1cf1a94bb5132f3ce45022b992e314f8f01a20abb6e28eedee58d8b0da3e401ca363d92a61f1fee9ef879da5443fc3119562004b07fcd875e393689b926f4c39932b67bd731537dbdb9969ac30a165eb4d4f3c3f06e79c257872e33254e031568c24d88eb73cfd8a517bf83baf55d4b9586a8ad336a9165dfca847d510e78cdaddd1dd76387cc75d576303c8a59edfdb01f47b47b0835913bbcd9838638ad44f0e197df30b1d9abc7d47b512c096988ab527ea9f3feb4e4adc1b0365934d5cbb3010369de4ba150986921d82bfdf73548872372915499b4a22337f78c90a59d13bf5c8fb3c3edf171a2c627d5059a954c9e81efeb72da188dfb0279e22f13d78aeea8bffa7142a33e5c45aa4d5d8bb0a9d1676cf4b8e4d9be9620e8b9e166de06484466fd3f24e91ddd7b9d89c9920968b45de549b61f6d8b3619", "serpent"},
#endif
	{NULL}
};

int diskcryptor_valid(char *ciphertext, struct fmt_main *self)
{
	char *ctcopy, *keeptr, *p;
	int extra;

	if (strncmp(ciphertext, FORMAT_TAG, TAG_LENGTH) != 0)
		return 0;

	ctcopy = xstrdup(ciphertext);
	keeptr = ctcopy;

	ctcopy += TAG_LENGTH;
	if ((p = strtokm(ctcopy, "*")) == NULL) // type
		goto err;
	if (strcmp(p, "0"))
		goto err;
	if ((p = strtokm(NULL, "*")) == NULL) // header
		goto err;
	if (hexlenl(p, &extra) != 2048 * 2 || extra)
		goto err;

	MEM_FREE(keeptr);
	return 1;

err:
	MEM_FREE(keeptr);
	return 0;
}

void *diskcryptor_get_salt(char *ciphertext)
{
	static struct custom_salt cs;
	char *ctcopy = xstrdup(ciphertext);
	char *keeptr = ctcopy;
	char *p;
	int i;

	memset(&cs, 0, sizeof(struct custom_salt));
	ctcopy += TAG_LENGTH;
	p = strtokm(ctcopy, "*");
	cs.type = atoi(p);
	p = strtokm(NULL, "*");
	for (i = 0; i < 2048; i++)
		cs.header[i] = (atoi16[ARCH_INDEX(p[2 * i])] << 4) | atoi16[ARCH_INDEX(p[2 * i + 1])];

	memcpy(cs.salt, cs.header, 64);
	cs.iterations = 1000; // fixed as of version 1.1.846.118 (09.07.2014)
	cs.saltlen = 64;

	MEM_FREE(keeptr);
	return &cs;
}

unsigned int diskcryptor_iteration_count(void *salt)
{
	struct custom_salt *cs = (struct custom_salt*)salt;

	return cs->iterations;
}

int diskcryptor_decrypt_data(unsigned char *key, struct custom_salt *cur_salt)
{
	const int len = 96; /* up to 2048 */
	union {
		unsigned char uc[len];
		struct dc_header header;
	} out;
	int algorithm;

	for (algorithm = 0; algorithm < 3; algorithm++) {
		unsigned char tweak[16] = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

		// Note: header.reserved is actually all zeroes but isn't for us because our XTS code is wonky?
		XTS_decrypt_custom_tweak(key, tweak, out.uc, cur_salt->header, len, 256, algorithm);
#if ARCH_LITTLE_ENDIAN==0
		out.header.alg_1 = JOHNSWAP(out.header.alg_1);
		out.header.version = (out.header.version >> 8) | (out.header.version << 8);
#endif
		if (!memcmp(out.uc + 64, "DCRP", 4) &&
		    (out.header.version == 2 || out.header.version == 1) &&
		    (out.header.alg_1 >= 0 && out.header.alg_1 <= 7))
			return 1;
	}

	return 0;
}
