/*
 * This software is Copyright (c) 2017 Dhiru Kholia and it is hereby released
 * to the general public under the following terms:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted.
 *
 * Based on opencl_pbkdf2_hmac_sha512_fmt_plug.c file.
 */

#if AC_BUILT
#include "autoconfig.h"
#endif

#include "arch.h"
#if !AC_BUILT
#define HAVE_LIBZ 1
#endif
#if HAVE_LIBZ

#if HAVE_OPENCL && HAVE_LIBCRYPTO

#if FMT_EXTERNS_H
extern struct fmt_main fmt_opencl_electrum_modern;
#elif FMT_REGISTERS_H
john_register_one(&fmt_opencl_electrum_modern);
#else

#include <stdint.h>
#include <string.h>
#include <zlib.h>
#include <openssl/bn.h>
#ifdef _OPENMP
#include <omp.h>
#endif

#include "misc.h"
#include "common.h"
#include "formats.h"
#include "options.h"
#include "opencl_common.h"
#include "johnswap.h"
#include "secp256k1.h"
#include "aes.h"
#include "sha2.h"
#include "hmac_sha.h"
#include "pbkdf2_hmac_common.h"

#define FORMAT_NAME             "Electrum Wallet 2.8+"
#define FORMAT_LABEL            "electrum-modern-opencl"
#define FORMAT_TAG              "$electrum$"
#define TAG_LENGTH              (sizeof(FORMAT_TAG) - 1)
#define ALGORITHM_NAME          "PBKDF2-SHA512 OpenCL"
#define BINARY_SIZE             16
#define BINARY_ALIGN            sizeof(uint32_t)
#define SALT_SIZE               sizeof(struct custom_salt)
#define SALT_ALIGN              sizeof(uint32_t)
#define PLAINTEXT_LENGTH        110
#define MIN_KEYS_PER_CRYPT      1
#define MAX_KEYS_PER_CRYPT      1
#define KERNEL_NAME             "pbkdf2_sha512_kernel"
#define SPLIT_KERNEL_NAME       "pbkdf2_sha512_loop"

#define HASH_LOOPS              250
#define ITERATIONS              10000

static struct fmt_tests electrum_tests[] = {
	// Electrum 2.8.0+ encrypted wallets
	{"$electrum$4*03c2a94eb01e9453c24c9bf49102356788673cc26fbe27b9bf54b0f150758c7864*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*40c7a179573d57c54d0da0a1c4d71e306e1eea823f637f29c3e43b9792469d15", "openwall123"},
	{"$electrum$4*0328e536dd1fbbb85d78de1a8c21215d4646cd87d6b6545afcfb203e5bb32e0de4*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*ec90c1ff54632e7c8cfb812eeb14d7ec49ddaf576dca10bfb16f965e6106ce48", "btcr-test-password"},
	// Electrum 2.8.0+ encrypted wallet with truncated hash, "electrum28-wallet" from btcrecover project
	{"$electrum$5*0328e536dd1fbbb85d78de1a8c21215d4646cd87d6b6545afcfb203e5bb32e0de4*61b1e287a5acff4b40e4abd73ff62dc233c1c7a6a54b3270949281b9d44bc6e746743733360500718826e50bb28ea99a6378dc0b0c578e9d0bf09c667671c82a1bd71c8121edbb4c9cbca93ab0e17e218558ead81755e62b0d4ad547aa1b3beb0b9ee43b11270261c9b38502f00e7f6f096811b7fdae6f3dce85c278d3751fec044027054218ccf20d404bab24380b303f094704e626348a218f44ab88ce2ac5fa7d450069fca3bb53f9359dbbaad0ea1b3859129b19c93ed7888130f8a534f84a629c67edc150a1c5882a83cb0add4615bb569e8dc471de4d38fc8b1e0b9b28040b5ea86093fcdeceaedb6b8f073f6f0ee5541f473a4b1c2bfae4fc91e4bbb40fa2185ecfa4c72010bcf8df05b1a7db45f64307dbc439f8389f0e368e38960b6d61ac88c07ce95a4b03d6d8b13f4c7dc7d7c447097865235ab621aeef38dc4172bf2dc52e701132480127be375fe98834f16d9895dce7f6cdfe900a2ce57eaa6c3036c1b9a661c3c9adbf84f4adfe6d4d9fa9f829f2957cfb353917dc77fd8dd4872b7d90cb71b7d3a29c9bfe3440e02449220acba410fa0af030f51aa2438f7478dbb277d62613112e4eebc66d5d7bdba793fb2073d449954f563284819189ffb5dbcdeb6c95c64bc24e0ef986bce07bafe96ab449ae2b6edaf4f98ffbd392a57bd93c2359444ec4046ae65b440adb96b6e4eef9d06bb04d2f3fa2e4175165bcadbf7e13cc3b6e65e67df901f96a2f154bc763b56b3736a335e1d1bc16e99736f757a4ae56c099645c917360b1ecf8dcefc7281541c6ff65d87cadab4a48f1f6b7b73a3e5a67e2e032abb56b499e73a9f3b69ce065e43b0174639785ae30635d105ebcc827dcf9b19bdd1a92879a5d4bc4e12b5630c188b1b96e3c586e19901b8f96084bcd59b2f4b201a3a8b6e633a5c194901d4609add9671b0bcc12b2b94ae873d201258b36315484e4b9c5f5d6289656baa93eec9e92aec88e2d73d86b9e3d1f24294e3d8ebe9a9f2f6edfbf28f530670c5b086fc4f74df89b4e4cbe06ee7e45cbd238b599d19c2d5da5523b12b1e7050ea0a9b47a5d22c6c3fc476f814f9705dc7ed3aeb1b44fc6b4d69f02a74963dce5057c3c049f92e595a4da5035cffc303a4cb162803aa3f816527a7e466b8424789a0d77e26819615662420c370457e29fcc1938fd754f3acfd21416ce3ab27e9febbc0e24fc7055eddc31e48faa014f9f3695c2e956f0e6c94c507a8d2f8c3aeb4b98b69b6340b6a3acb1acdde9581279f78ee10687616360c018e9f67d6c8bb5950e8fdabd3d0d5808824975aa4a50f88581472212f24ad58a700fe4787642b973924575fe71d1ecd7b2b6acd363f48c40bdd55f35f60a06dee544c266e608fd5a6d263f745e8b11d1160638eb301adfd1a88eddf6d0ccb9e1021e0bde9cf5163583a202b3dc95c255c8cc24*ec90c1ff54632e7c8cfb812eeb14d7ec49ddaf576dca10bfb16f965e6106ce48", "btcr-test-password"},
	// This triggers the problem of not using BN_bn2binpad()
	{"$electrum$4*02324ebc7dbebd38017fd6f20961d07d50386dd2f813cb5949bb9c905881e79487*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*2fb4fb957e5f7ad154250785d908838b06c823b46185f260db41fa88413c1ce6", "hashcat2056201445940"},
	{NULL}
};

static struct custom_salt {
	uint32_t type;
	unsigned char salt[8]; // fake salt
	uint32_t saltlen;
	unsigned char ephemeral_pubkey[128];
	unsigned char data[16384]; // is 16 KiB enough?
	uint32_t datalen;
	unsigned char mac[32];
	secp256k1_pubkey pubkey;
} *cur_salt;

typedef struct {
	// for plaintext, we must make sure it is a full uint64_t width.
	uint64_t v[(PLAINTEXT_LENGTH + 7) / 8]; // v must be kept aligned(8)
	uint64_t length; // keep 64 bit aligned, length is overkill, but easiest way to stay aligned.
} pass_t;

typedef struct {
	uint64_t hash[8];
} crack_t;

typedef struct {
	// for salt, we append \x00\x00\x00\x01\x80 and must make sure it is a full uint64 width
	uint64_t salt[(PBKDF2_64_MAX_SALT_SIZE + 1 + 4 + 7) / 8]; // salt must be kept aligned(8)
	uint32_t length;
	uint32_t rounds;
} salt_t;

typedef struct {
	uint64_t ipad[8];
	uint64_t opad[8];
	uint64_t hash[8];
	uint64_t W[8];
	cl_uint rounds;
} state_t;

static int new_keys;
static pass_t *host_pass;			      /** plain ciphertexts **/
static salt_t *host_salt;			      /** salt **/
static crack_t *host_crack;			      /** cracked or no **/
static cl_mem mem_in, mem_out, mem_salt, mem_state;
static cl_kernel split_kernel;
static cl_int cl_error;
static struct fmt_main *self;
static uint32_t (*crypt_out)[BINARY_SIZE * 2 / sizeof(uint32_t)];

#define STEP			0
#define SEED			256

static const char *warn[] = {
        "xfer: ",  ", init: " , ", crypt: ", ", res xfer: "
};

static int split_events[] = { 2, -1, -1 };

//This file contains auto-tuning routine(s). Has to be included after formats definitions.
#include "opencl_autotune.h"

/* ------- Helper functions ------- */
static size_t get_task_max_work_group_size()
{
	size_t min_lws =
		autotune_get_task_max_work_group_size(FALSE, 0, crypt_kernel);
	return MIN(min_lws, autotune_get_task_max_work_group_size(FALSE, 0,
	                                                          split_kernel));
}

static void release_clobj(void);

static void create_clobj(size_t kpc, struct fmt_main *self)
{
	release_clobj();

	host_pass = mem_calloc(kpc, sizeof(pass_t));
	host_crack = mem_calloc(kpc, sizeof(crack_t));
	host_salt = mem_calloc(1, sizeof(salt_t));
	crypt_out = mem_calloc(kpc, sizeof(*crypt_out));
#define CL_RO CL_MEM_READ_ONLY
#define CL_WO CL_MEM_WRITE_ONLY
#define CL_RW CL_MEM_READ_WRITE

#define CLCREATEBUFFER(_flags, _size, _string)	  \
	clCreateBuffer(context[gpu_id], _flags, _size, NULL, &cl_error); \
	HANDLE_CLERROR(cl_error, _string);

#define CLKERNELARG(kernel, id, arg, msg)	  \
	HANDLE_CLERROR(clSetKernelArg(kernel, id, sizeof(arg), &arg), msg);

	mem_in = CLCREATEBUFFER(CL_RO, kpc * sizeof(pass_t),
			"Cannot allocate mem in");
	mem_salt = CLCREATEBUFFER(CL_RO, sizeof(salt_t),
			"Cannot allocate mem salt");
	mem_out = CLCREATEBUFFER(CL_WO, kpc * sizeof(crack_t),
			"Cannot allocate mem out");
	mem_state = CLCREATEBUFFER(CL_RW, kpc * sizeof(state_t),
			"Cannot allocate mem state");

	CLKERNELARG(crypt_kernel, 0, mem_in, "Error while setting mem_in");
	CLKERNELARG(crypt_kernel, 1, mem_salt, "Error while setting mem_salt");
	CLKERNELARG(crypt_kernel, 2, mem_state, "Error while setting mem_state");

	CLKERNELARG(split_kernel, 0, mem_state, "Error while setting mem_state");
	CLKERNELARG(split_kernel, 1, mem_out, "Error while setting mem_out");
}

static void init(struct fmt_main *_self)
{
	self = _self;
	opencl_prepare_dev(gpu_id);
}

static void reset(struct db_main *db)
{
	if (!program[gpu_id]) {
		char build_opts[128];

		snprintf(build_opts, sizeof(build_opts),
		         "-DHASH_LOOPS=%u -DPLAINTEXT_LENGTH=%d -DPBKDF2_64_MAX_SALT_SIZE=%d",
		         HASH_LOOPS, PLAINTEXT_LENGTH, PBKDF2_64_MAX_SALT_SIZE);

		opencl_init("$JOHN/opencl/pbkdf2_hmac_sha512_kernel.cl",
		            gpu_id, build_opts);

		crypt_kernel = clCreateKernel(program[gpu_id], KERNEL_NAME, &cl_error);
		HANDLE_CLERROR(cl_error, "Error creating kernel");

		split_kernel =
			clCreateKernel(program[gpu_id], SPLIT_KERNEL_NAME, &cl_error);
		HANDLE_CLERROR(cl_error, "Error creating split kernel");
	}

	//Initialize openCL tuning (library) for this format.
	opencl_init_auto_setup(SEED, HASH_LOOPS, split_events, warn,
	                       2, self, create_clobj, release_clobj,
	                       sizeof(state_t), 0, db);

	//Auto tune execution from shared/included code.
	autotune_run(self, ITERATIONS, 0, 200);
}

static void release_clobj(void)
{
	if (host_pass) {
		MEM_FREE(host_pass);
		MEM_FREE(host_salt);
		MEM_FREE(host_crack);

		HANDLE_CLERROR(clReleaseMemObject(mem_in), "Release mem in");
		HANDLE_CLERROR(clReleaseMemObject(mem_salt), "Release mem salt");
		HANDLE_CLERROR(clReleaseMemObject(mem_out), "Release mem out");
		HANDLE_CLERROR(clReleaseMemObject(mem_state), "Release mem state");
	}
}

static void done(void)
{
	if (program[gpu_id]) {
		release_clobj();
		HANDLE_CLERROR(clReleaseKernel(crypt_kernel), "Release kernel");
		HANDLE_CLERROR(clReleaseKernel(split_kernel), "Release kernel");
		HANDLE_CLERROR(clReleaseProgram(program[gpu_id]), "Release Program");

		program[gpu_id] = NULL;
	}
}

static int valid(char *ciphertext, struct fmt_main *self)
{
	char *ctcopy, *keeptr, *p;
	int value, extra;

	if (strncmp(ciphertext, FORMAT_TAG, TAG_LENGTH) != 0)
		return 0;

	ctcopy = xstrdup(ciphertext);
	keeptr = ctcopy;

	ctcopy += TAG_LENGTH;
	if ((p = strtokm(ctcopy, "*")) == NULL) // type
		goto err;
	if (!isdec(p))
		goto err;
	value = atoi(p);
	if (value != 4 && value != 5)
		goto err;

	if ((p = strtokm(NULL, "*")) == NULL) // ephemeral_pubkey
		goto err;
	if (hexlenl(p, &extra) > 128 * 2 || extra)
		goto err;
	if ((p = strtokm(NULL, "*")) == NULL) // data
		goto err;
	if (hexlenl(p, &extra) > 16384 * 2 || extra)
		goto err;
	if ((p = strtokm(NULL, "*")) == NULL) // data
		goto err;
	if (hexlenl(p, &extra) != 32 * 2 || extra)
		goto err;
	if (strtokm(NULL, "*")) // no more fields
		goto err;

	MEM_FREE(keeptr);
	return 1;

err:
	MEM_FREE(keeptr);
	return 0;
}

static void *get_salt(char *ciphertext)
{
	static struct custom_salt cs;
	secp256k1_context *ctx;
	char *ctcopy = xstrdup(ciphertext);
	char *keeptr = ctcopy;
	char *p;
	int i, length;

	memset(&cs, 0, SALT_SIZE);
	ctcopy += TAG_LENGTH;
	p = strtokm(ctcopy, "*");
	cs.type = atoi(p);
	p = strtokm(NULL, "*");
	length = strlen(p) / 2;

	for (i = 0; i < length; i++)
		cs.ephemeral_pubkey[i] = (atoi16[ARCH_INDEX(p[2 * i])] << 4) | atoi16[ARCH_INDEX(p[2 * i + 1])];

	p = strtokm(NULL, "*");
	cs.datalen = strlen(p) / 2;
	for (i = 0; i < cs.datalen; i++)
		cs.data[i] = (atoi16[ARCH_INDEX(p[2 * i])] << 4) | atoi16[ARCH_INDEX(p[2 * i + 1])];
	p = strtokm(NULL, "*");
	for (i = 0; i < 32; i++)
		cs.mac[i] = (atoi16[ARCH_INDEX(p[2 * i])] << 4) | atoi16[ARCH_INDEX(p[2 * i + 1])];
	ctx = secp256k1_context_create(SECP256K1_CONTEXT_NONE);
	secp256k1_ec_pubkey_parse(ctx, &cs.pubkey, cs.ephemeral_pubkey, length);
	secp256k1_context_destroy(ctx);

	// we append the count and EOM here, one time.
	memcpy(cs.salt, "\x0\x0\x0\x1\x80", 5);
	cs.saltlen = 5; // we include the x80 byte in our saltlen, but the .cl kernel knows to reduce saltlen by 1

	MEM_FREE(keeptr);
	return &cs;
}

static void set_salt(void *salt)
{
	cur_salt = (struct custom_salt*)salt;

	memcpy(host_salt->salt, cur_salt->salt, cur_salt->saltlen);
	host_salt->length = cur_salt->saltlen;
	host_salt->rounds = 1024; // fixed

	HANDLE_CLERROR(clEnqueueWriteBuffer(queue[gpu_id], mem_salt,
		CL_FALSE, 0, sizeof(salt_t), host_salt, 0, NULL, NULL),
		"Salt transfer");
	HANDLE_CLERROR(clFlush(queue[gpu_id]), "clFlush failed in set_salt()");
}

void *electrum_get_binary(char *ciphertext)
{
	static union {
		unsigned char c[BINARY_SIZE];
		uint32_t dummy;
	} buf;
	unsigned char *out = buf.c;
	char *p;
	int i;
	p = strrchr(ciphertext, '*') + 1;
	for (i = 0; i < BINARY_SIZE; i++) {
		out[i] = (atoi16[ARCH_INDEX(*p)] << 4) | atoi16[ARCH_INDEX(p[1])];
		p += 2;
	}

	return out;
}

static const char *group_order = "fffffffffffffffffffffffffffffffebaaedce6af48a03bbfd25e8cd0364141";

// The decypted and decompressed wallet should start with one of these two, // Christopher Gurnee
#define EXPECTED_BYTES_1 "{\n    \""
#define EXPECTED_BYTES_2 "{\r\n    \""

/* Just for good measure */
#undef BN_bn2binpad32
#define BN_bn2binpad32 JTR_BN_bn2binpad32

static int BN_bn2binpad32(const BIGNUM *a, unsigned char *to)
{
	memset(to, 0, 32);

	return BN_bn2bin(a, to + 32 - BN_num_bytes(a));
}

static int crypt_all(int *pcount, struct db_salt *salt)
{
	int i;
	const int count = *pcount;
	int index;
	int loops = (host_salt->rounds + HASH_LOOPS - 1) / HASH_LOOPS;
	size_t *lws = local_work_size ? &local_work_size : NULL;

	global_work_size = GET_NEXT_MULTIPLE(count, local_work_size);

	// Copy data to gpu
	if (new_keys) {
		BENCH_CLERROR(clEnqueueWriteBuffer(queue[gpu_id], mem_in, CL_FALSE, 0,
			global_work_size * sizeof(pass_t), host_pass, 0, NULL,
			multi_profilingEvent[0]), "Copy data to gpu");

		new_keys = 0;
	}

	// Run kernel
	BENCH_CLERROR(clEnqueueNDRangeKernel(queue[gpu_id], crypt_kernel, 1,
				NULL, &global_work_size, lws, 0, NULL,
				multi_profilingEvent[1]), "Run kernel");

	for (i = 0; i < (ocl_autotune_running ? 1 : loops); i++) {
		BENCH_CLERROR(clEnqueueNDRangeKernel(queue[gpu_id],
					split_kernel, 1, NULL,
					&global_work_size, lws, 0, NULL,
					multi_profilingEvent[2]), "Run split kernel");
		BENCH_CLERROR(clFinish(queue[gpu_id]), "clFinish");
		opencl_process_event();
	}

	// Read the result back
	BENCH_CLERROR(clEnqueueReadBuffer(queue[gpu_id], mem_out, CL_TRUE, 0,
				global_work_size * sizeof(crack_t), host_crack,
				0, NULL, multi_profilingEvent[3]), "Copy result back");

	if (!ocl_autotune_running) {
#ifdef _OPENMP
#pragma omp parallel for
#endif
		for (index = 0; index < count; index++) {
			BIGNUM *p, *q, *r;
			BN_CTX *ctx;
			uint64_t u[8];
			unsigned char static_privkey[64];
			unsigned char shared_pubkey[33];
			unsigned char keys[128];
			unsigned char cmac[32];
			secp256k1_context *sctx;
			SHA512_CTX md_ctx;
			int shared_pubkeylen= 33;
			int j;

			memcpy(u, host_crack[index].hash, 64);
			for (j = 0; j < 8; j++)
				u[j] = JOHNSWAP64(u[j]);
			memcpy(static_privkey, u, 64);
			// do static_privkey % GROUP_ORDER
			p = BN_bin2bn(static_privkey, 64, NULL);
			q = BN_new();
			r = BN_new();
			BN_hex2bn(&q, group_order);
			ctx = BN_CTX_new();
			BN_mod(r, p, q, ctx);
			BN_CTX_free(ctx);
			BN_free(p);
			BN_free(q);
			BN_bn2binpad32(r, static_privkey);
			BN_free(r);
			sctx = secp256k1_context_create(SECP256K1_CONTEXT_NONE);
			// multiply point with a scaler, shared_pubkey is compressed representation
			secp256k1_mul(sctx, shared_pubkey, &cur_salt->pubkey, static_privkey);
			secp256k1_context_destroy(sctx);
			SHA512_Init(&md_ctx);
			SHA512_Update(&md_ctx, shared_pubkey, shared_pubkeylen);
			SHA512_Final(keys, &md_ctx);
			if (cur_salt->type == 4) {
				// calculate mac of data
				hmac_sha256(keys + 32, 32, cur_salt->data, cur_salt->datalen, cmac, 32);
				memcpy(crypt_out[index], cmac, BINARY_SIZE);
			} else if (cur_salt->type == 5) {
				z_stream z;
				unsigned char iv[16];
				unsigned char out[512] = { 0 };
				unsigned char fout[512] = { 0 };
				AES_KEY aes_decrypt_key;

				// common zlib settings
				z.zalloc = Z_NULL;
				z.zfree = Z_NULL;
				z.opaque = Z_NULL;
				z.avail_in = 512;
				z.avail_out = 512;
				z.next_out = fout;

				memcpy(iv, keys, 16);
				memset(crypt_out[index], 0, BINARY_SIZE);
				// fast zlib based rejection test, is this totally safe?
				AES_set_decrypt_key(keys + 16, 128, &aes_decrypt_key);
				AES_cbc_encrypt(cur_salt->data, out, 16, &aes_decrypt_key, iv, AES_DECRYPT);
				if ((memcmp(out, "\x78\x9c", 2) != 0) || (((out[2] & 0x7) != 0x4) && ((out[2] & 0x7) != 0x5))) {
				} else {
					AES_set_decrypt_key(keys + 16, 128, &aes_decrypt_key);
					AES_cbc_encrypt(cur_salt->data + 16, out + 16, 512 - 16, &aes_decrypt_key, iv, AES_DECRYPT);
					z.next_in = out;
					inflateInit2(&z, 15);
					inflate(&z, Z_NO_FLUSH);
					inflateEnd(&z);
					if ((memcmp(fout, EXPECTED_BYTES_1, 7) == 0) || (memcmp(fout, EXPECTED_BYTES_2, 8) == 0))
						memcpy(crypt_out[index], cur_salt->mac, BINARY_SIZE); // dirty hack!
				}
			}
		}
	}

	return count;
}

static int cmp_all(void *binary, int count)
{
	int index;

	for (index = 0; index < count; index++)
		if (!memcmp(binary, crypt_out[index], ARCH_SIZE))
			return 1;
	return 0;
}

static int cmp_one(void *binary, int index)
{
	return !memcmp(binary, crypt_out[index], BINARY_SIZE);
}

static int cmp_exact(char *source, int index)
{
	return 1;
}

static void set_key(char *key, int index)
{
	int saved_len = MIN(strlen(key), PLAINTEXT_LENGTH);
	// make sure LAST uint64 that has any key in it gets null, since we simply
	// ^= the whole uint64 with the ipad/opad mask
	strncpy((char*)host_pass[index].v, key, PLAINTEXT_LENGTH);
	host_pass[index].length = saved_len;

	new_keys = 1;
}

static char *get_key(int index)
{
	static char ret[PLAINTEXT_LENGTH + 1];
	memcpy(ret, host_pass[index].v, PLAINTEXT_LENGTH);
	ret[host_pass[index].length] = 0;
	return ret;
}

struct fmt_main fmt_opencl_electrum_modern = {
	{
		FORMAT_LABEL,
		FORMAT_NAME,
		ALGORITHM_NAME,
		BENCHMARK_COMMENT,
		BENCHMARK_LENGTH,
		0,
		PLAINTEXT_LENGTH,
		BINARY_SIZE,
		BINARY_ALIGN,
		SALT_SIZE,
		SALT_ALIGN,
		MIN_KEYS_PER_CRYPT,
		MAX_KEYS_PER_CRYPT,
		FMT_CASE | FMT_8_BIT | FMT_OMP | FMT_HUGE_INPUT,
		{ NULL },
		{ FORMAT_TAG },
		electrum_tests
	}, {
		init,
		done,
		reset,
		fmt_default_prepare,
		valid,
		fmt_default_split,
		electrum_get_binary,
		get_salt,
		{ NULL },
		fmt_default_source,
		{
			fmt_default_binary_hash
		},
		fmt_default_salt_hash,
		NULL,
		set_salt,
		set_key,
		get_key,
		fmt_default_clear_keys,
		crypt_all,
		{
			fmt_default_get_hash
		},
		cmp_all,
		cmp_one,
		cmp_exact
	}
};

#endif /* plugin stanza */

#endif /* HAVE_OPENCL && HAVE_LIBCRYPTO */

#endif /* HAVE_LIBZ */
