/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo, Marko Mattila

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"
#include <stdlib.h>

#ifdef _WIN32
#define DIRSEP "\\"
#else
#define DIRSEP "/"
#endif

#define CONFIG_FILE     ( ".kkrc" )

const char *config::DEFAULT_NAMES[config::MAX_LOCAL_PLAYERS] = {
  "player1", "player2"
};
const char *config::DEFAULT_SERVER = "localhost";
const char *config::DEFAULT_PORT = "6789";

config::config() {
  keymap[PLAYER_1][_KEY_UP] = SDLK_UP;
  keymap[PLAYER_1][_KEY_DOWN] = SDLK_DOWN;
  keymap[PLAYER_1][_KEY_LEFT] = SDLK_LEFT;
  keymap[PLAYER_1][_KEY_RIGHT] = SDLK_RIGHT;
  keymap[PLAYER_1][_KEY_ACTION1] = SDLK_o;
  keymap[PLAYER_1][_KEY_ACTION2] = SDLK_p;
  keymap[PLAYER_1][_KEY_TALK] = SDLK_1;


  keymap[PLAYER_2][_KEY_UP] = SDLK_w;
  keymap[PLAYER_2][_KEY_DOWN] = SDLK_s;
  keymap[PLAYER_2][_KEY_LEFT] = SDLK_a;
  keymap[PLAYER_2][_KEY_RIGHT] = SDLK_d;
  keymap[PLAYER_2][_KEY_ACTION1] = SDLK_v;
  keymap[PLAYER_2][_KEY_ACTION2] = SDLK_b;
  keymap[PLAYER_2][_KEY_TALK] = SDLK_2;

  player_names[PLAYER_1] = DEFAULT_NAMES[PLAYER_1];
  player_names[PLAYER_2] = DEFAULT_NAMES[PLAYER_2];
  server = DEFAULT_SERVER;
  port = DEFAULT_PORT;
}

config::~config() {
  /* There's not much to clean up in here... */
}

int config::get_key( int player, key_id key ) {
  return ( ( ( player > -1 && player < MAX_LOCAL_PLAYERS ) && ( key >= _KEY_UP && key < _NUM_KEYS ) )
      ? keymap[player][key]
      : -1 );
}

void config::set_key( int player, key_id key, int value ) {
  if ( ( player > -1 && player < MAX_LOCAL_PLAYERS ) && ( key >= _KEY_UP && key < _NUM_KEYS ) )
  {
    keymap[player][key] = value;
  }
}


file_config::file_config( config *cfg )
  : config()
{
  configfile = get_config_fn();

  if ( cfg == NULL ) {
    /* Read config from file */
    if ( configfile.length() ) {
#ifdef DEBUG
      fprintf( stderr, "Trying to read configuration from file %s... ", configfile.c_str() );
#endif
      try {
        read_config( configfile );
#ifdef DEBUG
        fprintf( stderr, "Success!\n" );
#endif
      }
      catch ( string s ) {
#ifdef DEBUG
        fprintf( stderr, "Failed!\n" );
        fprintf( stderr, "%s\n", s.c_str() );
#endif
      }
    }
    else {
      fprintf( stderr, "Error determining configuration file location. No settings will be loaded\n" );
    }
  }
  else {
    for ( int i = 0; i < MAX_LOCAL_PLAYERS; i++ ) {
      player_names[i] = cfg->get_name( i );

      for ( int j = 0; j < _NUM_KEYS; j++ ) {
        keymap[i][j] = cfg->get_key( i, key_id( j ) );
      }
    }

    server = cfg->get_server();
    port = cfg->get_port();
  }
}

file_config::~file_config() {
  /* Write config to file??? */
  if ( configfile.length() ) {
#ifdef DEBUG
    fprintf( stderr, "Trying to save configuration in file %s... ", configfile.c_str() );
#endif
    try {
      write_config( configfile );
#ifdef DEBUG
      fprintf( stderr, "Success!\n" );
#endif
    }
    catch ( string s ) {
#ifdef DEBUG
      fprintf( stderr, "Error\n" );
      fprintf( stderr, "%s\n", s.c_str() );
#endif
    }
  }
}

void file_config::read_config( string& fn ) throw ( string )
{
  char      buf[1024];
  size_t    linenum = 0;
  FILE      *fp = 0;
  char      *k = 0, *v = 0;

  fp = fopen( fn.c_str(), "r" );
  if ( !fp ) {
    string    s = "Failed to open configuration file " + fn + "\n";
    throw( s );
  }

  while ( !feof( fp ) ) {
    linenum++;

    if ( fgets( buf, sizeof( buf ), fp ) == (char *) NULL ) {
      if ( ferror( fp ) ) {
        fclose( fp );
        throw( "Error reading configuration file " + fn + "\n" );
      }
      else
        continue;
    }

    if ( ( k = strchr( buf, '#' ) ) ) {
      *k = 0;
    }

    if ( ( ( k = strtok( buf, "= \t\n\r" ) ) != (char *) NULL ) &&
        ( ( v = strtok( NULL, "= \t\n\r" ) ) != (char *) NULL ) ) {
      string    key = k;

      if ( !key.compare( "key_player1_up" ) ) {
        keymap[PLAYER_1][_KEY_UP] = str_to_key_default( v, keymap[PLAYER_1][_KEY_UP] );
      }
      else if ( !key.compare( "key_player1_down" ) ) {
        keymap[PLAYER_1][_KEY_DOWN] = str_to_key_default( v, keymap[PLAYER_1][_KEY_DOWN] );
      }
      else if ( !key.compare( "key_player1_left" ) ) {
        keymap[PLAYER_1][_KEY_LEFT] = str_to_key_default( v, keymap[PLAYER_1][_KEY_LEFT] );
      }
      else if ( !key.compare( "key_player1_right" ) ) {
        keymap[PLAYER_1][_KEY_RIGHT] = str_to_key_default( v, keymap[PLAYER_1][_KEY_RIGHT] );
      }
      else if ( !key.compare( "key_player1_action1" ) ) {
        keymap[PLAYER_1][_KEY_ACTION1] = str_to_key_default( v, keymap[PLAYER_1][_KEY_ACTION1] );
      }
      else if ( !key.compare( "key_player1_action2" ) ) {
        keymap[PLAYER_1][_KEY_ACTION2] = str_to_key_default( v, keymap[PLAYER_1][_KEY_ACTION2] );
      }
      else if ( !key.compare( "key_player1_talk" ) ) {
        keymap[PLAYER_1][_KEY_TALK] = str_to_key_default( v, keymap[PLAYER_1][_KEY_TALK] );
      }
      else if ( !key.compare( "key_player2_up" ) ) {
        keymap[PLAYER_2][_KEY_UP] = str_to_key_default( v, keymap[PLAYER_2][_KEY_UP] );
      }
      else if ( !key.compare( "key_player2_down" ) ) {
        keymap[PLAYER_2][_KEY_DOWN] = str_to_key_default( v, keymap[PLAYER_2][_KEY_DOWN] );
      }
      else if ( !key.compare( "key_player2_left" ) ) {
        keymap[PLAYER_2][_KEY_LEFT] = str_to_key_default( v, keymap[PLAYER_2][_KEY_LEFT] );
      }
      else if ( !key.compare( "key_player2_right" ) ) {
        keymap[PLAYER_2][_KEY_RIGHT] = str_to_key_default( v, keymap[PLAYER_2][_KEY_RIGHT] );
      }
      else if ( !key.compare( "key_player2_action1" ) ) {
        keymap[PLAYER_2][_KEY_ACTION1] = str_to_key_default( v, keymap[PLAYER_2][_KEY_ACTION1] );
      }
      else if ( !key.compare( "key_player2_action2" ) ) {
        keymap[PLAYER_2][_KEY_ACTION2] = str_to_key_default( v, keymap[PLAYER_2][_KEY_ACTION2] );
      }
      else if ( !key.compare( "key_player2_talk" ) ) {
        keymap[PLAYER_2][_KEY_TALK] = str_to_key_default( v, keymap[PLAYER_2][_KEY_TALK] );
      }
      else if ( !key.compare( "player1_name" ) ) {
        player_names[PLAYER_1] = ( strlen( v ) ? v : DEFAULT_NAMES[PLAYER_1] );
      }
      else if ( !key.compare( "player2_name" ) ) {
        player_names[PLAYER_2] = ( strlen( v ) ? v : DEFAULT_NAMES[PLAYER_2] );
      }
      else if ( !key.compare( "server" ) ) {
        server = ( strlen( v ) ? v : DEFAULT_SERVER );
      }
      else if ( !key.compare( "server_port" ) ) {
        port = ( strlen( v ) ? v : DEFAULT_PORT );
      }
      else {
        fprintf( stderr," Unknown key/value pair (%s/%s) on line %u\n", key.c_str(), v, linenum );
      }
    }
    /*
    else {
      if ( k != NULL && v == NULL ) {
        fprintf( stderr, "Malformed configuration file (line %d): No value for key %s\n", linenum, k );
      }
    }
    */
  }
  fclose( fp );
}

void file_config::write_config( string& fn ) throw ( string ) {
  FILE      *fp;
  
  fp = fopen( fn.c_str(), "w" );
  if ( !fp )
    throw ( "Failed to open configuration file " + fn + " for writing\n" );

  try {
    write_string( fp, "key_player1_up", key_to_str( keymap[PLAYER_1][_KEY_UP] ) );
    write_string( fp, "key_player1_down", key_to_str( keymap[PLAYER_1][_KEY_DOWN] ) );
    write_string( fp, "key_player1_left", key_to_str( keymap[PLAYER_1][_KEY_LEFT] ) );
    write_string( fp, "key_player1_right", key_to_str( keymap[PLAYER_1][_KEY_RIGHT] ) );
    write_string( fp, "key_player1_action1", key_to_str( keymap[PLAYER_1][_KEY_ACTION1] ) );
    write_string( fp, "key_player1_action2", key_to_str( keymap[PLAYER_1][_KEY_ACTION2] ) );
    write_string( fp, "key_player1_talk", key_to_str( keymap[PLAYER_1][_KEY_TALK] ) );

    write_string( fp, "key_player2_up", key_to_str( keymap[PLAYER_2][_KEY_UP] ) );
    write_string( fp, "key_player2_down", key_to_str( keymap[PLAYER_2][_KEY_DOWN] ) );
    write_string( fp, "key_player2_left", key_to_str( keymap[PLAYER_2][_KEY_LEFT] ) );
    write_string( fp, "key_player2_right", key_to_str( keymap[PLAYER_2][_KEY_RIGHT] ) );
    write_string( fp, "key_player2_action1", key_to_str( keymap[PLAYER_2][_KEY_ACTION1] ) );
    write_string( fp, "key_player2_action2", key_to_str( keymap[PLAYER_2][_KEY_ACTION2] ) );
    write_string( fp, "key_player2_talk", key_to_str( keymap[PLAYER_2][_KEY_TALK] ) );

    write_string( fp, "player1_name", player_names[PLAYER_1].c_str() );
    write_string( fp, "player2_name", player_names[PLAYER_2].c_str() );
    write_string( fp, "server", server.c_str() );
    write_string( fp, "server_port", port.c_str() );
  }
  catch ( int natas ) {
    fclose( fp );
    throw ( "Failed to write configuration file due to Satanic intervention!\n" );
  }
  
  fclose( fp );
}

void file_config::write_string( FILE *fp, const char *key, const char *value ) throw ( string ) {
  string		s = string( key ) + "=" + string( value ) + string( "\n" );

  if ( fputs( s.c_str(), fp ) == EOF )
    throw ( "Error writing configuration file" );
}

void file_config::write_int( FILE *fp, const char *key, int value ) throw ( string ) {
  string		s = string( key ) + "=";
  char			buf[16];

  sprintf( buf, "%d\n", value );

  s += buf;

  if ( fputs( s.c_str(), fp ) == EOF )
    throw ( "Error writing configuration file" );
}

string file_config::get_config_fn() {
  char      *p = 0;
  string    fn;

  p = getenv( "HOME" );
  if ( p ) {
    fn = p;
    fn += DIRSEP;
    fn += CONFIG_FILE;
  }
  else {
    /* Windows */
    p = getenv( "APPDATA" );
    if ( p ) {
      fn = p;
      fn += DIRSEP;
      fn += CONFIG_FILE;
    }

    /* Should we test some other as well? */
  }
    
  return ( fn );
}
  

const char *config::keystrings[] = {
  0, 0, 0, 0, 0, 0, 0, 0,
  "backspace",    /* = 8 SDLK_BACKSPACE */
  "tab",          /* = 9 SDLK_TAB */
  0, 0, 
  "clear",        /* = 12 SDLK_CLEAR */
  "return",       /* = 13 SDLK_RETURN */
  0, 0, 0, 0, 0,
  "pause",        /* = 19 SDLK_PAUSE */
  0, 0, 0, 0, 0, 0, 0,
  "escape",       /* = 27 SDLK_ESCAPE */
  0, 0, 0, 0,
  "space",        /* = 32  SDLK_SPACE */
  "!",            /* = 33 SDLK_EXCLAIM */
  "\"",           /* = 34 SDLK_QUOTEDBL */
  "#",            /* = 35 SDLK_HASH */
  "$",            /* = 36 SDLK_DOLLAR */
  0,
  "&",            /* = 38 SDLK_AMPERSAND */
  "'",            /* = 39 SDLK_QUOTE */
  "(",            /* = 40 SDLK_LEFTPAREN */
  ")",            /* = 41 SDLK_RIGHTPAREN */
  "*",            /* = 42 SDLK_ASTERISK */
  "+",            /* = 43 SDLK_PLUS */
  ",",            /* = 44 SDLK_COMMA */
  "-",            /* = 45 SDLK_MINUS */
  ".",            /* = 46 SDLK_PERIOD */
  "/",            /* = 47 SDLK_SLASH */
  "0",            /* = 48 SDLK_0 */
  "1",            /* = 49 SDLK_1 */
  "2",            /* = 50 SDLK_2 */
  "3",            /* = 51 SDLK_3 */
  "4",            /* = 52 SDLK_4 */
  "5",            /* = 53 SDLK_5 */
  "6",            /* = 54 SDLK_6 */
  "7",            /* = 55 SDLK_7 */
  "8",            /* = 56 SDLK_8 */
  "9",            /* = 57 SDLK_9 */
  ":",            /* = 58 SDLK_COLON */
  ";",            /* = 59 SDLK_SEMICOLON */
  "<",            /* = 60 SDLK_LESS */
  "=",            /* = 61 SDLK_EQUALS */
  ">",            /* = 62 SDLK_GREATER */
  "?",            /* = 63 SDLK_QUESTION */
  "@",            /* = 64 SDLK_AT */
  0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  "[",            /* = 91 SDLK_LEFTBRACKET */
  "\\",           /* = 92 SDLK_BACKSLASH */
  "]",            /* = 93 SDLK_RIGHTBRACKET */
  "^",            /* = 94 SDLK_CARET */
  "_",            /* = 95 SDLK_UNDERSCORE */
  "`",            /* = 96 SDLK_BACKQUOTE */
  "a",            /* = 97 SDLK_a */
  "b",            /* = 98 SDLK_b */
  "c",            /* = 99 SDLK_c */
  "d",            /* = 100 SDLK_d */
  "e",            /* = 101 SDLK_e */
  "f",            /* = 102 SDLK_f */
  "g",            /* = 103 SDLK_g */
  "h",            /* = 104 SDLK_h */
  "i",            /* = 105 SDLK_i */
  "j",            /* = 106 SDLK_j */
  "k",            /* = 107 SDLK_k */
  "l",            /* = 108 SDLK_l */
  "m",            /* = 109 SDLK_m */
  "n",            /* = 110 SDLK_n */
  "o",            /* = 111 SDLK_o */
  "p",            /* = 112 SDLK_p */
  "q",            /* = 113 SDLK_q */
  "r",            /* = 114 SDLK_r */
  "s",            /* = 115 SDLK_s */
  "t",            /* = 116 SDLK_t */
  "u",            /* = 117 SDLK_u */
  "v",            /* = 118 SDLK_v */
  "w",            /* = 119 SDLK_w */
  "x",            /* = 120 SDLK_x */
  "y",            /* = 121 SDLK_y */
  "z",            /* = 122 SDLK_z */
  0, 0, 0, 0,
  "delete",       /* = 127 SDLK_DELETE */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  "world0",       /* = 160 SDLK_WORLD_0 */
  "world1",       /* = 161 SDLK_WORLD_1 */
  "world2",       /* = 162 SDLK_WORLD_2 */
  "world3",       /* = 163 SDLK_WORLD_3 */
  "world4",       /* = 164 SDLK_WORLD_4 */
  "world5",       /* = 165 SDLK_WORLD_5 */
  "world6",       /* = 166 SDLK_WORLD_6 */
  "world7",       /* = 167 SDLK_WORLD_7 */
  "world8",       /* = 168 SDLK_WORLD_8 */
  "world9",       /* = 169 SDLK_WORLD_9 */
  "world10",      /* = 170 SDLK_WORLD_10 */
  "world11",      /* = 171 SDLK_WORLD_11 */
  "world12",      /* = 172 SDLK_WORLD_12 */
  "world13",      /* = 173 SDLK_WORLD_13 */
  "world14",      /* = 174 SDLK_WORLD_14 */
  "world15",      /* = 175 SDLK_WORLD_15 */
  "world16",      /* = 176 SDLK_WORLD_16 */
  "world17",      /* = 177 SDLK_WORLD_17 */
  "world18",      /* = 178 SDLK_WORLD_18 */
  "world19",      /* = 179 SDLK_WORLD_19 */
  "world20",      /* = 180 SDLK_WORLD_20 */
  "world21",      /* = 181 SDLK_WORLD_21 */
  "world22",      /* = 182 SDLK_WORLD_22 */
  "world23",      /* = 183 SDLK_WORLD_23 */
  "world24",      /* = 184 SDLK_WORLD_24 */
  "world25",      /* = 185 SDLK_WORLD_25 */
  "world26",      /* = 186 SDLK_WORLD_26 */
  "world27",      /* = 187 SDLK_WORLD_27 */
  "world28",      /* = 188 SDLK_WORLD_28 */
  "world29",      /* = 189 SDLK_WORLD_29 */
  "world30",      /* = 190 SDLK_WORLD_30 */
  "world31",      /* = 191 SDLK_WORLD_31 */
  "world32",      /* = 192 SDLK_WORLD_32 */
  "world33",      /* = 193 SDLK_WORLD_33 */
  "world34",      /* = 194 SDLK_WORLD_34 */
  "world35",      /* = 195 SDLK_WORLD_35 */
  "world36",      /* = 196 SDLK_WORLD_36 */
  "world37",      /* = 197 SDLK_WORLD_37 */
  "world38",      /* = 198 SDLK_WORLD_38 */
  "world39",      /* = 199 SDLK_WORLD_39 */
  "world40",      /* = 200 SDLK_WORLD_40 */
  "world41",      /* = 201 SDLK_WORLD_41 */
  "world42",      /* = 202 SDLK_WORLD_42 */
  "world43",      /* = 203 SDLK_WORLD_43 */
  "world44",      /* = 204 SDLK_WORLD_44 */
  "world45",      /* = 205 SDLK_WORLD_45 */
  "world46",      /* = 206 SDLK_WORLD_46 */
  "world47",      /* = 207 SDLK_WORLD_47 */
  "world48",      /* = 208 SDLK_WORLD_48 */
  "world49",      /* = 209 SDLK_WORLD_49 */
  "world50",      /* = 210 SDLK_WORLD_50 */
  "world51",      /* = 211 SDLK_WORLD_51 */
  "world52",      /* = 212 SDLK_WORLD_52 */
  "world53",      /* = 213 SDLK_WORLD_53 */
  "world54",      /* = 214 SDLK_WORLD_54 */
  "world55",      /* = 215 SDLK_WORLD_55 */
  "world56",      /* = 216 SDLK_WORLD_56 */
  "world57",      /* = 217 SDLK_WORLD_57 */
  "world58",      /* = 218 SDLK_WORLD_58 */
  "world59",      /* = 219 SDLK_WORLD_59 */
  "world60",      /* = 220 SDLK_WORLD_60 */
  "world61",      /* = 221 SDLK_WORLD_61 */
  "world62",      /* = 222 SDLK_WORLD_62 */
  "world63",      /* = 223 SDLK_WORLD_63 */
  "world64",      /* = 224 SDLK_WORLD_64 */
  "world65",      /* = 225 SDLK_WORLD_65 */
  "world66",      /* = 226 SDLK_WORLD_66 */
  "world67",      /* = 227 SDLK_WORLD_67 */
  "world68",      /* = 228 SDLK_WORLD_68 */
  "world69",      /* = 229 SDLK_WORLD_69 */
  "world70",      /* = 230 SDLK_WORLD_70 */
  "world71",      /* = 231 SDLK_WORLD_71 */
  "world72",      /* = 232 SDLK_WORLD_72 */
  "world73",      /* = 233 SDLK_WORLD_73 */
  "world74",      /* = 234 SDLK_WORLD_74 */
  "world75",      /* = 235 SDLK_WORLD_75 */
  "world76",      /* = 236 SDLK_WORLD_76 */
  "world77",      /* = 237 SDLK_WORLD_77 */
  "world78",      /* = 238 SDLK_WORLD_78 */
  "world79",      /* = 239 SDLK_WORLD_79 */
  "world80",      /* = 240 SDLK_WORLD_80 */
  "world81",      /* = 241 SDLK_WORLD_81 */
  "world82",      /* = 242 SDLK_WORLD_82 */
  "world83",      /* = 243 SDLK_WORLD_83 */
  "world84",      /* = 244 SDLK_WORLD_84 */
  "world85",      /* = 245 SDLK_WORLD_85 */
  "world86",      /* = 246 SDLK_WORLD_86 */
  "world87",      /* = 247 SDLK_WORLD_87 */
  "world88",      /* = 248 SDLK_WORLD_88 */
  "world89",      /* = 249 SDLK_WORLD_89 */
  "world90",      /* = 250 SDLK_WORLD_90 */
  "world91",      /* = 251 SDLK_WORLD_91 */
  "world92",      /* = 252 SDLK_WORLD_92 */
  "world93",      /* = 253 SDLK_WORLD_93 */
  "world94",      /* = 254 SDLK_WORLD_94 */
  "world95",      /* = 255 SDLK_WORLD_95 */
  "keypad0",      /* = 256 SDLK_KP0 */
  "keypad1",      /* = 257 SDLK_KP1 */
  "keypad2",      /* = 258 SDLK_KP2 */
  "keypad3",      /* = 259 SDLK_KP3 */
  "keypad4",      /* = 260 SDLK_KP4 */
  "keypad5",      /* = 261 SDLK_KP5 */
  "keypad6",      /* = 262 SDLK_KP6 */
  "keypad7",      /* = 263 SDLK_KP7 */
  "keypad8",      /* = 264 SDLK_KP8 */
  "keypad9",      /* = 265 SDLK_KP9 */
  "keypad.",      /* = 266 SDLK_KP_PERIOD */
  "keypad/",      /* = 267 SDLK_KP_DIVIDE */
  "keypad*",      /* = 268 SDLK_KP_MULTIPLY */
  "keypad-",      /* = 269 SDLK_KP_MINUS */
  "keypad+",      /* = 270 SDLK_KP_PLUS */
  "keypad_enter", /* = 271 SDLK_KP_ENTER */
  "keypad=",      /* = 272 SDLK_KP_EQUALS */
  "up",           /* = 273 SDLK_UP */
  "down",         /* = 274 SDLK_DOWN */
  "right",        /* = 275 SDLK_RIGHT */
  "left",         /* = 276 SDLK_LEFT */
  "insert",       /* = 277 SDLK_INSERT */
  "home",         /* = 278 SDLK_HOME */
  "end",          /* = 279 SDLK_END */
  "pageup",       /* = 280 SDLK_PAGEUP */
  "pagedown",     /* = 281 SDLK_PAGEDOWN */
  "f1",           /* = 282 SDLK_F1 */
  "f2",           /* = 283 SDLK_F2 */
  "f3",           /* = 284 SDLK_F3 */
  "f4",           /* = 285 SDLK_F4 */
  "f5",           /* = 286 SDLK_F5 */
  "f6",           /* = 287 SDLK_F6 */
  "f7",           /* = 288 SDLK_F7 */
  "f8",           /* = 289 SDLK_F8 */
  "f9",           /* = 290 SDLK_F9 */
  "f10",          /* = 291 SDLK_F10 */
  "f11",          /* = 292 SDLK_F11 */
  "f12",          /* = 293 SDLK_F12 */
  "f13",          /* = 294 SDLK_F13 */
  "f14",          /* = 295 SDLK_F14 */
  "f15",          /* = 296 SDLK_F15 */
  0, 0, 0,
  "numlock",      /* = 300 SDLK_NUMLOCK */
  "capslock",     /* = 301 SDLK_CAPSLOCK */
  "scrollock",    /* = 302 SDLK_SCROLLOCK */
  "rshift",       /* = 303 SDLK_RSHIFT */
  "lshift",       /* = 304 SDLK_LSHIFT */
  "rctrl",        /* = 305 SDLK_RCTRL */
  "lctrl",        /* = 306 SDLK_LCTRL */
  "ralt",         /* = 307 SDLK_RALT */
  "lalt",         /* = 308 SDLK_LALT */
  "rmeta",        /* = 309 SDLK_RMETA */
  "lmeta",        /* = 310 SDLK_LMETA */
  "lsuper",       /* = 311 SDLK_LSUPER */
  "rsuper",       /* = 312 SDLK_RSUPER */
  "altgr",        /* = 313 SDLK_MODE */
  "compose",      /* = 314 SDLK_COMPOSE */
  "help",         /* = 315 SDLK_HELP */
  "print-screen", /* = 316 SDLK_PRINT */
  "sysrq",        /* = 317 SDLK_SYSREQ */
  "break",        /* = 318 SDLK_BREAK */
  "menu",         /* = 319 SDLK_MENU */
  "power",        /* = 320 SDLK_POWER */
  "euro",         /* = 321 SDLK_EURO */
  "undo",         /* = 322 SDLK_UNDO */
  (char *) -1
};

const char *config::key_to_str( int key ) throw ( int ) {
  if ( key >= 0 && key < (int) ( sizeof( keystrings ) / (int) sizeof( keystrings[0] ) ) ) {
    return ( keystrings[key] );
  }
  throw ( 0x29A );
}

int config::str_to_key( const char *str ) {
  if ( !str ) return ( -1 );
  
  for ( int i = 0; keystrings[i] != (char *) -1; i++ ) {
    if ( ( keystrings[i] )
        && ( strcmp( keystrings[i], str ) == 0 ) ) {
      return ( i );
    }
  }
  return ( 0 );
}


