/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MSG_H
#define __MSG_H

#include <cassert>
#include "terrain.h"
#include "SDL.h"
#include "SDL_net.h"
#include "globals.h"
#include "coord.h"
#include "block.h"

/**
 * Base class for all messages.
 */
class msg
{ 
 public:
  enum id { SERVERHELLO = 0, CLIENTHELLO, PL_DISCONNECT, START_CSELECT, START_GAME, PLAYERINFO, TALKMESSAGE,
	    CASTLE_LOCATION, MOVE_CANNONPLACER, PLACE_CANNON, PLACE_BIG_CANNON, CHANGE_CANNON, MOVE_CHARACTERCHOOSER, CHOOSE_CHARACTER, START_SHOOT, START_REPAIR, SHOOT,
	    MOVE_CURSOR, MOVE_BLOCKPLACER, PLACE_BLOCK, ROTATE_BL, START_PLACECAN, START_ENDGAME,
	    GAME_OVER, MOVE_CHOOSER, CHOOSER_LOCK, MOVE_CHOOSER_REQ, END, START_ENDROUND, DISCONNECT
  , MSG_IDNOTSET = 9999};

  // copy constructor
  msg (const msg &m);
  msg() { msg_id = MSG_IDNOTSET; }

  id get_id() { return msg_id; }

  void send (const TCPsocket sock) const throw (string);

  // receive msg. NOTE! THE RETURNED MSG HAS TO BE DELETED!
  static msg* recv (TCPsocket sock) throw (string);

  virtual msg *deep_copy() = 0;

 protected:
  id msg_id; // set in constructor to correct id

  // abstract funcs
  virtual void write_msgdata(TCPsocket sock) const throw (string) = 0;
  virtual void read_msgdata(TCPsocket sock) throw (string) = 0;

  // funcs to read or write data
  static void write_uint(TCPsocket sock, Uint32 d) throw (string);
  static Uint32 read_uint(TCPsocket sock) throw (string);

  static void write_string(TCPsocket sock, const string &s) throw (string);
  static string read_string(TCPsocket sock) throw (string);

  static void write_coord(TCPsocket sock, const coord &c) throw (string);
  static coord read_coord(TCPsocket sock) throw (string);

};

class msg_serverhello : public msg
{
 public:
  msg_serverhello(const vector<int> &pindices);
  msg_serverhello();
  virtual ~msg_serverhello() { }
  const vector<int> & get_player_indices();
  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

  vector<int> pindices;
};

class msg_clienthello : public msg
{
 public:
  msg_clienthello(const vector<string> &names);
  msg_clienthello();
  virtual ~msg_clienthello() { }
  const string & get_name(int index) { return names[index]; }
  int get_num_players() const { return names.size(); }
  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string) ;
  void read_msgdata(TCPsocket sock) throw (string);
 private:
  vector<string> names;
};

class msg_player_disconnected : public msg
{
 public:
  msg_player_disconnected();
  msg_player_disconnected(int _pnum);
  virtual ~msg_player_disconnected() { }

  int get_playerid()
    { return pnum; }

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { write_uint(sock, pnum); }
  void read_msgdata(TCPsocket sock) throw (string)
    { pnum = read_uint(sock); }
 private:
  int pnum;
};

class msg_start_castleselect : public msg
{
 public:
  msg_start_castleselect();
  virtual ~msg_start_castleselect() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};


class msg_start_shoot : public msg
{
 public:
  msg_start_shoot()
    { msg_id = START_SHOOT; }
  virtual ~msg_start_shoot() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};

class msg_game_over : public msg
{
 public:
  msg_game_over()
    { msg_id = GAME_OVER; }
  virtual ~msg_game_over() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};

class msg_end : public msg
{
 public:
  msg_end()
    { msg_id = END; }
  virtual ~msg_end() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};

class msg_start_placecan : public msg
{
 public:
  msg_start_placecan()
    { msg_id = START_PLACECAN; }
  virtual ~msg_start_placecan() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};

class msg_start_endgame : public msg
{
 public:
  msg_start_endgame()
    { msg_id = START_ENDGAME; }
  virtual ~msg_start_endgame() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};

class msg_start_endround : public msg
{
 public:
  msg_start_endround()
    { msg_id = START_ENDROUND; }
  virtual ~msg_start_endround() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};

class msg_start_repair : public msg
{
 public:
  msg_start_repair()
    { msg_id = START_REPAIR; }
  virtual ~msg_start_repair() {}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string)
    { }
  void read_msgdata(TCPsocket sock) throw (string)
    { }
};

class msg_start_game : public msg
{
 public:
  msg_start_game();
  msg_start_game(int wins);
  virtual ~msg_start_game() {}
  int get_wins()
    {return wins;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int wins;
};

class msg_playerinfo : public msg
{
 public:
  msg_playerinfo();
  msg_playerinfo(int index, const string &name);
  virtual ~msg_playerinfo(){}
  const string & get_name() { return name; }
  int get_index() { return index; }

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  string name;
  int index;
};

class msg_talkmessage : public msg
{
 public:
  msg_talkmessage();
  msg_talkmessage(int index, const string &txt);
  virtual ~msg_talkmessage(){}
  const string & get_txt() { return txt; }
  int get_owner() { return owner; }

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  string txt;
  int owner;
};

class msg_castle_location : public msg
{
 public:
  msg_castle_location();
  msg_castle_location(const coord &pos, int owner_id);
  virtual ~msg_castle_location() {}
  const coord &get_position()
    {return pos;}
  int get_owner()
    {return owner;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  coord pos;
  int owner;
};

class msg_move_cannonplacer : public msg
{
 public:
  msg_move_cannonplacer();
  msg_move_cannonplacer(const coord &new_pos, int owner_id);
  virtual ~msg_move_cannonplacer() {}
  const coord &get_position()
    {return pos;}
  int get_owner()
    {return owner;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  coord pos;
  int owner;
};

class msg_move_characterchooser : public msg
{
 public:
  msg_move_characterchooser();
  msg_move_characterchooser(const coord &new_pos, int owner_id);
  virtual ~msg_move_characterchooser() {}
  const coord &get_position()
    {return pos;}
  int get_owner()
    {return owner;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  coord pos;
  int owner;
};

class msg_move_blockplacer : public msg
{
 public:
  msg_move_blockplacer();
  msg_move_blockplacer(const coord &new_pos, int owner_id);
  virtual ~msg_move_blockplacer() {}
  const coord &get_position()
    {return pos;}
  int get_owner()
    {return owner;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  coord pos;
  int owner;
};

class msg_place_cannon : public msg
{
 public:
  msg_place_cannon();
  msg_place_cannon(int owner_id, const coord &pos);
  virtual ~msg_place_cannon() {}
  int get_owner()
    {return owner;}
  const coord & get_position()
    {return pos;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
  coord pos;
};

class msg_change_cannon : public msg
{
 public:
  msg_change_cannon();
  msg_change_cannon(int owner_id);
  virtual ~msg_change_cannon() {}
  int get_owner()
    {return owner;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
};

class msg_place_big_cannon : public msg
{
 public:
  msg_place_big_cannon();
  msg_place_big_cannon(int owner_id, const coord &pos);
  virtual ~msg_place_big_cannon() {}
  int get_owner()
    {return owner;}
  const coord & get_position()
    {return pos;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
  coord pos;
};


class msg_choose_character : public msg
{
 public:
  msg_choose_character();
  msg_choose_character(int owner, const coord &_pos, int ch);
  virtual ~msg_choose_character() {}
  int get_character();
  int get_owner()
    {return owner;}
  const coord & get_position()
    {return pos;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner, character;
  coord pos;
};

class msg_chooser_lock : public msg
{
 public:
  msg_chooser_lock();
  msg_chooser_lock(int owner_id);
  virtual ~msg_chooser_lock() {}
  int get_owner()
    {return owner;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
};


class msg_move_chooser : public msg
{
 public:
  msg_move_chooser();
  msg_move_chooser(int owner_id, const coord &new_pos);
  virtual ~msg_move_chooser() {}
  int get_owner()
    {return owner;}
  const coord & get_position()
    {return pos;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
  coord pos;
};

class msg_move_chooser_req : public msg
{
 public:
  msg_move_chooser_req();
  msg_move_chooser_req(int owner_id, bool down);
  virtual ~msg_move_chooser_req() {}
  int get_owner()
    {return owner;}
  bool is_down() // chooser move down or up
    {return down;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
  bool down;
};

class msg_rotate_block : public msg
{
 public:
  msg_rotate_block();
  msg_rotate_block(int owner_id);
  virtual ~msg_rotate_block() {}
  int get_owner()
    {return owner;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
};

class msg_place_block : public msg
{
 public:
  msg_place_block();
  msg_place_block(int owner_id, int next_type, int next_orientation);
  virtual ~msg_place_block() {}
  int get_owner()
    {return owner;}
  int get_next_type()
    {return next_type;}
  int get_next_orientation()
    {return orientation;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  int owner;
  int next_type;
  int orientation;
};

class msg_shoot : public msg
{
 public:
  msg_shoot();
  msg_shoot(const coord &from, const coord &to, int shooter_id, int ammosize);
  virtual ~msg_shoot() {}
  const coord &get_from()
    {return from;}
  const coord &get_to()
    {return to;}
  int get_shooter()
    {return shooter;}
  int get_ammosize()
    {return ammosize;}

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  coord from;
  coord to;
  int shooter;
  int ammosize;
};

class msg_move_cursor : public msg
{
 public:
  msg_move_cursor();
  msg_move_cursor(const coord &new_pos, int owner_id, bool lock);
  virtual ~msg_move_cursor() {}
  const coord &get_position()
    {return pos;}
  int get_owner()
    {return owner;}
  bool get_lock()
    { return lock; }

  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);

 private:
  coord pos;
  int owner;
  bool lock;
};

class msg_disconnect : public msg
{
 public:
  msg_disconnect();
  virtual ~msg_disconnect() {}
  virtual msg * deep_copy();

 protected:
  void write_msgdata(TCPsocket sock) const throw (string);
  void read_msgdata(TCPsocket sock) throw (string);
};
#endif

