#define TRUE  1
#define FALSE 0

#include "automaton.h"
#include <assert.h>

#define DETERMINISTIC(s,a) (this->_states[s].next[a].size() == 1)
#define MATCHES_AB(a,b)    (matchingmatrix[(a)][(b)])
#define SCORES_AB(a,b)     (scoringmatrix[(a)][(b)])

/*
 * output method for the current automaton
 */

ostream& operator<<(ostream& os, automaton& automaton) { 
  os << automaton._states.size()  << endl;  
  /* display each state */
  for(int i=0; i<(int)automaton._states.size(); i++){
    os << dec <<  i << "\t" << automaton._states[i].final << endl;
    /* display for each state forward transitions */
    for(int a=0; a < gv_align_alphabet_size ; a++){
      os << "\t" << dec << a << "\t" << dec << automaton._states[i].next[a].size()  << endl;
      slist<transition> & list_tran = automaton._states[i].next[a];
      for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
	os << "\t\t"  << dec <<  (list_iter->state) << "\t" << (list_iter->prob)<< endl;
      }
    }
  }
  return os;
}



istream& operator>>(istream& is, automaton& automaton)  {

  /* clear previous automaton */
  for(int i = 0 ; i < (int)automaton._states.size() ; i++){
    automaton._states[i].clear();
  }
  automaton._states.clear();
  
  /* read automaton size */
  int size = 0;
  is >> size;
#ifdef DEBUGREADING
  cerr << size << endl; 
#endif
  if (size <= 0) {
    cerr << "> when reading automaton size" << endl;
      ERROR("operator>>","incorrect size "<< size);    
  }
  /* add states first */
  for(int i=0;i<size;i++){
    automaton.addNewState();
  }

  /* display each state */
  for(int statefrom=0; statefrom<size; statefrom++){
    int state = 0, final = 0;
    is >>  state >>  final;
#ifdef DEBUGREADING
    cerr << state << "\t" << final << endl;
#endif

    if (state < 0 || state >= size) {
      cerr << "> when reading automaton state " << state << endl;
      ERROR("operator>>","incorrect state " << state  << " (not in [0.."<<(size-1)<<"])");    
    }
    if (final != 0 && final != 1) {
      cerr << "> when reading automaton state " << state << endl;
      ERROR("operator>>","incorrect state final flag " << final  << " (not in [0..1])");    
    }
    
    automaton._states[statefrom].final = final;
    
    /* display for each state forward transitions */
    double probability_sum = 0.0;
    for(int a=0; a < gv_align_alphabet_size ; a++){
      int x = 0, alistsize = 0;
      is >> x >> alistsize;      
#ifdef DEBUGREADING
      cerr << "\t" << x << "\t" <<  alistsize << endl;
#endif

      if (x != a) {
	cerr << "> when reading automaton state " << state << endl;
	ERROR("operator>>","incorrect alphabet letter " << x << " (should be ordered and thus be " << a << ")");    
      }
      if (alistsize < 0 ) {
	cerr << "> when reading automaton state " << state << endl;
	ERROR("operator>>","incorrect transition list size "<< alistsize << " for letter " << x << " (should be >= 0)");    	
      }

      for(int j=0; j<alistsize; j++){
	int    stateto = 0;
	double probability  = 0.0;       
	is >>  stateto >> probability;	
#ifdef DEBUGREADING
	cerr << "\t\t" << stateto << "\t" << probability << endl;
#endif
	probability_sum += probability;

	if (stateto < 0 || stateto >= size) {
	  cerr << "> when reading automaton state " << state << endl;
	  ERROR("operator>>","incorrect transition to state "<< stateto << " for letter " << x << " (this state does not exist)");    	
	}	
	automaton.addNewTransition(a,statefrom,stateto,probability);
      }
    }
    if ( probability_sum > 1.0001 || probability_sum < 0.9999 ){
      cerr << "> when reading state \"" << state << "\"  letters " << endl;
      ERROR("operator>>","incorrect probability sum "<< probability_sum);
    }
  }
  return is;
}


/*
 * "dot" format output (use "graphviz\dotty" or "graphviz\dot file.dot -T ps -o file.eps" to visualise it)
 */


void automaton::dot(ostream& os) { 

  os << "digraph A {"  << endl;
  os << "\t fontsize =\"8\"" << endl;
      
  /* display each state */
  for(int i = 0;i<(int)this->_states.size();i++) {
    state &s = this->_states[i];
    os << "\t \"node" << i << "\" [ label = \"" << i << "\",shape = " << (s.final?"doublecircle":"circle") << " ];"<<endl;
  }
  
  /* display forward transitions */
  for(int i = 0;i<(int)this->_states.size();i++) { 
    state &s = this->_states[i];
    for(int a=0; a < gv_align_alphabet_size ; a++) {
      slist<transition> & list_tran = s.next[a];
      for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
	os << "\t \"node" << i <<"\"  -> \"node" << (list_iter->state) << "\" [label = \"" << a << //" (" << (list_iter->prob) << ")" <<
	  "\"] ;" << endl;
      }  
    }
  }
  os << "}"  << endl;
}



/*
 * This method create a new state 
 * and return number of states.
 */

int automaton::addNewState(int final) {
  _states.push_back(state(final));
  return _states.size() - 1;
}


/*
 * This method add a transition between two states
 * on letter "a"
 */
    
int automaton::addNewTransition(int a, int startingState, int endingState,double prob) {
    
#ifdef ASSERTB
    if (a < 0 || a >= gv_align_alphabet_size) {
      ERROR("addNewTransition","transition letter out of range");
    }
    
    if (startingState >= (int)_states.size() || endingState >= (int)_states.size()) {
      ERROR("addNewTransition","state required does not exist");
    }
#endif
 
    /* forward linking */
#ifdef ASSERTB
    // do not add twice the same state in the "a" backward list
    slist<transition>   & list_tran      = _states[startingState].next[a];
    for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
      if ( list_iter->state == endingState) {
	cerr << "> when linking state q2:" << dec << endingState << " with state q1:" << dec << startingState << " on letter a:" << dec << a << endl;
	ERROR("addNewTransition"," state q2 has already a transition on \"a\" ");
      }
    }
#endif    
    _states[startingState].next[a].push_front(transition(endingState,prob));

    /* backward linking */  
#ifdef ASSERTB
    // do not add twice the same state in the "a" backward list
    slist<transition>   & list_tran_back      = _states[endingState].prev[a];
    for(slist<transition>::iterator list_iter = list_tran_back.begin() ; list_iter != list_tran_back.end() ; list_iter++) {
      if ( list_iter->state == startingState) {
	cerr << "> when backlinking state q2:" << dec << endingState << " with state q1:" << dec << startingState << " on letter a:" << dec << a << endl;
	ERROR("addNewTransition"," state q2 has already a backtransition on \"a\" ");
      }
    }
#endif
    _states[endingState].prev[a].push_front(transition(startingState,prob));
    return 0;
}


/*
 * This method modifies the transition probability (a,startingState,endingState)
 */


int automaton::changeTransitionProb( int a, int startingState, int endingState, double prob) {

#ifdef ASSERTB
    if (a < 0 || a >= gv_align_alphabet_size) {
      ERROR(":changeTransitionProb","transition letter out of range");
    }
    
    if (startingState >= (int)_states.size() || endingState >= (int)_states.size()) {
      ERROR(":changeTransitionProb","state required does not exist");
    }
#endif


    /* forward link pr */
    slist<transition>   & list_tran      = _states[startingState].next[a];
    for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
      if ( list_iter->state == endingState) {
	list_iter->prob = prob;
	break;
      }
    }
   
    /* backward link pr */  
    slist<transition>   & list_tran_back      = _states[endingState].prev[a];
    for(slist<transition>::iterator list_iter = list_tran_back.begin() ; list_iter != list_tran_back.end() ; list_iter++) {
      if ( list_iter->state == startingState) {
	list_iter->prob = prob;
	break;
      }
    }
    return 0;
}



/*
 * Give the current probability of transition
 */

double automaton::Pr( int a, int startingState, int endingState) {
  
  /* forward link pr */
  slist<transition>   & list_tran      = _states[startingState].next[a];
  for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
    if ( list_iter->state == endingState) {
      return list_iter->prob;
    }
  }
  return 0.0;
}



/*
 * This method selfloops the given state on all letters
 */

void automaton::selfLoop(int stateNb) {
  for(int a=0; a<gv_align_alphabet_size; a++)
    addNewTransition(a,stateNb,stateNb,(+1e+0)/gv_align_alphabet_size);
}




/*
 * checks if there is at least on transition startingState --(a)--> ...
 */

bool automaton::hasTransition( int a, int startingState) {
#ifdef ASSERTB
  assert(startingState >= 0);
  assert(startingState < (int)_states.size());
  assert(a >= 0);
  assert(a < gv_align_alphabet_size);
#endif
  return _states[startingState].next[a].size() > 0;
}




#define USEQUEUEPRODUCT

/*
 * This method does the product of two automatons :
 * you can specify
 *     - "crossFinalStates"   if you need final states to be the product of BOTH automata final states : 
 *       PRODUCT_UNION        : automata "union" of final states, 
 *       PRODUCT_INTERSECTION : automata "intersection" of final states
 *       PRODUCT_BUTNOT       : automata "this".final BUT NOT "other".final
 *          with
 *       LOOP / NO_LOOP       : indicates if the final state is a single one that self-loops (force it, otherwise keep it as in the product)

 *       PRODUCT_UNION_NO_FINAL_LOOP_ADD
 *     - "otherIsProbabilist" if the second automaton is a probabilistic one (affect probabilities to new transitions)
 */


automaton * automaton::product(automaton & other, int crossFinalStates, bool otherIsProbabilist, int depth) {

#ifdef ASSERTB
  if ((this->size() * other.size()) > (1<<28)) {
    ERROR("product"," size of product automaton will \"certainly\" explode : better stop here ...");
  }
#endif

  automaton * result = new automaton();
  result->addNewState(TRUE);
  result->selfLoop(0);

#ifdef USEMAP
  typedef less< pair<int,int> > lessp;
  typedef map< pair<int,int> , int, lessp > maptype;
  maptype statesNbIndex;
  #define PRODINDEX(i) (i)
#else
  vector<int> statesNbIndex( this->size() * other.size(), 0);
  #define PRODINDEX(i) ((i).first * other.size() + (i).second)
#endif

#ifdef USEQUEUEPRODUCT
  queue< pair<int,int> >  statesNbRemaining;
#else	
  stack< pair<int,int> >  statesNbRemaining;
#endif
  
  /* (1) start the product init state */
  pair<int,int> indexN             = pair<int,int>(1,1);
  int stateNumber                  = result->addNewState();

  statesNbIndex[PRODINDEX(indexN)] = stateNumber;
  statesNbRemaining.push(indexN);


#ifdef USEQUEUEPRODUCT
  /* depth of the states being built */
  int level_i                    = 0;
  int stateN_of_level_i          = stateNumber;
#endif

  /* (2) take all the non-considered interesting cases pairs on both automatons */
  while(!statesNbRemaining.empty()) {
    
    // current state remaining
#ifdef USEQUEUEPRODUCT
    pair<int,int> indexN  =  statesNbRemaining.front();
#else
    pair<int,int> indexN  =  statesNbRemaining.top();
#endif

    statesNbRemaining.pop();
    
    int stateNA = indexN.first;
    int stateNB = indexN.second;
    int stateN =  statesNbIndex[PRODINDEX(indexN)];

    for(int a=0;a<gv_align_alphabet_size;a++) {
      
      slist<transition> & listA =  this->_states[stateNA].next[a];
      slist<transition> & listB =  other._states[stateNB].next[a];
      
      for(slist<transition>::iterator list_iterA = listA.begin() ; list_iterA != listA.end() ; list_iterA++) {
	for(slist<transition>::iterator list_iterB = listB.begin() ; list_iterB != listB.end() ; list_iterB++) {
	  
	  int stateAnext = list_iterA->state;
	  int stateBnext = list_iterB->state;
	  pair<int,int> indexNx  = pair<int,int>(stateAnext, stateBnext); 
	
	  int stateNx    = 0;

	  /* final state / final loop computation */
	  int final_state = 0;
	  int final_loop  = 0;

	  switch (crossFinalStates) {

	  case  PRODUCT_UNION_FINAL_LOOP:
	    final_loop = TRUE;
	  case  PRODUCT_UNION_NO_FINAL_LOOP:
	    final_state =  ((this->_states[stateAnext].final) || (other._states[stateBnext].final)) ? TRUE : FALSE;
	    break;
	  case  PRODUCT_UNION_NO_FINAL_LOOP_ADD:
	    final_state =  ((this->_states[stateAnext].final) + (other._states[stateBnext].final));
	    break;

	  case  PRODUCT_INTERSECTION_FINAL_LOOP:
	    final_loop = TRUE;
	  case  PRODUCT_INTERSECTION_NO_FINAL_LOOP:
	    final_state =  ((this->_states[stateAnext].final) && (other._states[stateBnext].final)) ? TRUE : FALSE;
	    break;

	  case  PRODUCT_BUTNOT_FINAL_LOOP:
	    final_loop = TRUE;
	  case  PRODUCT_BUTNOT_NO_FINAL_LOOP:
	    final_state =  ((this->_states[stateAnext].final) && (!(other._states[stateBnext].final))) ? TRUE : FALSE;
	    break;

	  }

	  /* add the "new" state, considering booleans "final_state" and "final_state" */
	  if (final_state && final_loop) {
	    stateNx = 0;
	  } else {    
	    if  (!statesNbIndex[PRODINDEX(indexNx)]) {
	     
#ifdef USEQUEUEPRODUCT
	      /* compute level */
	      if (stateN > stateN_of_level_i) {
		stateN_of_level_i = (result->size() - 1);
		level_i++;
	      }

	      if (level_i <= depth ){
#endif
		/* create a new state */
		stateNx = result->addNewState(final_state);
		statesNbIndex[PRODINDEX(indexNx)] = stateNx;
		statesNbRemaining.push(indexNx);
#ifdef BUILD		
		cout << "$push state:" << dec << stateNx << endl;
#endif
#ifdef USEQUEUEPRODUCT
	      } else {
		/* max level reached : goes to a "non final" loop state */
		stateNx = result->addNewState();
		result->selfLoop(stateNx);
	      }
#endif
	    } else {
	      stateNx = statesNbIndex[PRODINDEX(indexNx)];
	    }
	  }


#ifdef BUILD
	  cout << "> add transition ( a:" << dec << a << ", q1:" << dec << stateN << " , q2:" << stateNx << " ) " << endl;
#endif		
	  // add a transition on from stateN --a--> stateNx.
	  if(otherIsProbabilist)
	    result->addNewTransition(a,stateN,stateNx,list_iterB->prob);
	  else
	    result->addNewTransition(a,stateN,stateNx);
	}// for (listB)
      }// for (listA)
    }// for (a)
  }//while stack nonempty
  
  /* Free unused data needed to build the automaton */
  statesNbIndex.clear();    
  return result;
}




#define USEQUEUEMHIT

/*
 * This method computes the m hits automaton of the previous one
 * (see product, it is quite similar)
 */
automaton * automaton::mhit(unsigned int m, int depth) {
  automaton * result = new automaton();
  result->addNewState(TRUE);
  result->selfLoop(0);
  
  /* pair <int,int> 
   *   - first int is the state on the original automaton "this"
   *   - second is the number of finals "added to go to this state" 
   */

#ifdef USEMAP
  typedef map< pair<int,int> , int> maptype;
  maptype statesNbIndex;
  #define MHITINDEX(i) (i)
#else
  vector<int> statesNbIndex( this->size() * m, 0);
  #define MHITINDEX(i) ((i).first * m + (i).second)
#endif

#ifdef USEQUEUEMHIT
  queue< pair<int,unsigned int> >  statesNbRemaining;
#else	
  stack< pair<int,unsigned int> >  statesNbRemaining;
#endif

  /* (1) start the mhits init state */
  pair<int,unsigned int> indexN    = pair<int,unsigned int>(1,0);
  int stateNumber                  = result->addNewState();
  statesNbIndex[MHITINDEX(indexN)] = stateNumber;
  statesNbRemaining.push(indexN);

#ifdef USEQUEUEMHIT
  /* depth of the states beeing built */
  int level_i                    = 0;
  int stateN_of_level_i          = stateNumber;
#endif
  
  /* (2) take the non considered pair */
  while(!statesNbRemaining.empty()) {
    
    // current state remaining
#ifdef USEQUEUEMHIT
    pair<int,unsigned int> indexN  =  statesNbRemaining.front();
#else
    pair<int,unsigned int> indexN  =  statesNbRemaining.top();
#endif

    statesNbRemaining.pop();
    
    int thisStateN = indexN.first;
    int     mhitN  = indexN.second;
    int     stateN = statesNbIndex[MHITINDEX(indexN)];

    for(int a=0; a<gv_align_alphabet_size; a++) {
      
      slist<transition> & thisList =  this->_states[thisStateN].next[a];
      
      for(slist<transition>::iterator list_iter = thisList.begin() ; list_iter != thisList.end() ; list_iter++) {
	  
	  int thisStateNextN  = list_iter->state;
	  unsigned int mhitNx = mhitN + this->_states[thisStateNextN].final;
	
	  int stateNx       = 0;

	  /* add the "new" state, considering booleans "final_state" and "final_state" */
	  if (mhitNx >= m) {
	    stateNx = 0;
	  } else {    
	    pair<int, unsigned int> indexNx = pair<int, unsigned int>(thisStateNextN, mhitNx);

	    if  (!statesNbIndex[MHITINDEX(indexNx)]) {	     

#ifdef USEQUEUEMHIT
	      /* compute level */
	      if (stateN > stateN_of_level_i) {
		stateN_of_level_i = (result->size() - 1);
		level_i++;
	      }

	      if (level_i <= depth ){
#endif
		/* create a new state */
		stateNx = result->addNewState();
		statesNbIndex[MHITINDEX(indexNx)] = stateNx;
		statesNbRemaining.push(indexNx);
#ifdef BUILD		
		cout << "$push state:" << dec << stateNx << endl;
#endif	
#ifdef USEQUEUEMHIT
	      } else {
		/* max level reached : goes to a "non final" loop state */
		stateNx = result->addNewState();
		result->selfLoop(stateNx);
	      }
#endif   
	    } else {
	      stateNx = statesNbIndex[MHITINDEX(indexNx)];
	    }
	  }

#ifdef BUILD
	  cout << "> add transition ( a:" << dec << a << ", q1:" << dec << stateN << " , q2:" << stateNx << " ) " << endl;
#endif	
	  // add a transition on from stateN --a--> stateNx.
	  result->addNewTransition(a,stateN,stateNx,list_iter->prob);
      }// for (list)
    }// for (a)
  }//while stack nonempty
  
  /* Free unused data needed to build the automaton */
  statesNbIndex.clear();    
  return result;
}




/*
 * This method builds a simple Bernoulli
 * probabilistic automaton given a probability
 * vector for each alignment alphabet letter.
 */


int automaton::Automaton_Bernoulli(vector <double> & p /*[gv_align_alphabet_size]*/) { 
#ifdef ASSERTB
  assert((int)p.size() == gv_align_alphabet_size);
#endif
  addNewState();
  int InitState_I = addNewState();
  for(int a=0; a<gv_align_alphabet_size; a++) {
    addNewTransition(a, InitState_I,  InitState_I,p[a]);
  }
  return 0;
}


/*
 * This methodes builds a Markov 
 * probabilistic automaton given a probability
 * vector for each word of size k on the alignment alphabet.
 */

int automaton::Automaton_Markov(vector<double> & p, int k ){
  
  int ApowK     = 1;
  int ApowKplus = gv_align_alphabet_size;


  /* [A] build the trie */
  int nextbase  = 1;

  /* empty ending state */
  addNewState(TRUE);
  selfLoop(0);

  int InitState_I = addNewState();    /* starting at state 1 */


  for(int d = 1 ; d <= k ; d++, ApowK *= gv_align_alphabet_size, ApowKplus *= gv_align_alphabet_size) {

    nextbase += ApowK;

    /* [a] compute weights */
    /* (a.1) denumerators */
    vector<double> probasum  = vector<double>(ApowK,    0.0);
    int i = 0; 
    for(int buklet = 0 ; buklet < ApowK ; buklet++) {
      for(int ibuklet = 0 ; ibuklet < (int)p.size() ; ibuklet += ApowK) {
	probasum[buklet] += p[i++];
      }
    }
    /* (a.2) numerators */
    vector<double> proba     = vector<double>(ApowKplus,0.0);
    int j = 0;
    for(int buklet = 0 ; buklet < ApowKplus ; buklet++) {
      for(int ibuklet = 0 ; ibuklet < (int)p.size() ; ibuklet += ApowKplus) {
	proba[buklet]    += p[j++];
      }
    }
    
    /* [b] set states an transitions */
    for(int buklet = 0 ; buklet < ApowKplus ; buklet++) {
      int state = addNewState();
      int base  = state -  InitState_I - 1;
      addNewTransition(base%gv_align_alphabet_size,
		       base/gv_align_alphabet_size +  InitState_I,
		       state,
		       proba[buklet]/probasum[buklet/gv_align_alphabet_size]);
    } /* buklet */
  } /* depth */
  
  
  /* [B] add final transitions */
  for(int b = 0 ; b < (int)(ApowK) ; b ++) {
    int bplus = b * gv_align_alphabet_size;
    double psum = 0.0;
    for (int a = 0 ; a < gv_align_alphabet_size ; a++) {
      psum += p[bplus + a];
    }
    
    for (int a = 0 ; a < gv_align_alphabet_size ; a++) {
      addNewTransition(a , 
		       nextbase +  (b        )%ApowK /* previous word */,
		       nextbase +  (bplus + a)%ApowK /* next word */,
		       p[bplus + a]/psum);
    }
  }
  
  return 0;
}



/*
 * This method builds a (Brejova model) M3 
 * probabilistic automaton given a probability
 * matrix
 *
 *   --> [state 1] --> [state 2] --> [state 3] -
 *  |               A             A             |
 *  |                                           |
 *   -------------------------------------------
 *                       A
 */

int automaton::Automaton_BrejovaM3(vector< vector<double> > & p /* [3][gv_align_alphabet_size] */) { 
  addNewState(TRUE);
  selfLoop(0);

  addNewState();    /* starting at state 1 */
  for(int nbstates = 0 ; nbstates < 3 ; nbstates ++) {
    addNewState();
    for(int a=0; a<gv_align_alphabet_size; a++) {
      addNewTransition(a, (nbstates)%3+1,(nbstates+1)%3+1,p[nbstates][a]);
    }
  }
  return 0;
}


/*
 * this method build a (Brejova equivalent M8) M14
 * probabilistic automaton given a probability
 * of each word of size 3 (ALPHABET^3)
 */

#define M13LETTERWORDSIZE  (gv_align_alphabet_size*gv_align_alphabet_size*gv_align_alphabet_size)
#define M13SIZE            ((M13LETTERWORDSIZE-1)/2)

int automaton::Automaton_BrejovaM13(vector<double> & p /*[M13LETTERWORDSIZE]*/) {
  addNewState(TRUE);
  selfLoop(0);
  
#ifdef ASSERTB
  double pr = 0;
  for (int i=0; i<M13LETTERWORDSIZE ; i++){
    pr += p[i];
  }
  cout << "M13 : pr sum = " << pr << endl;
#endif	

  int  InitState_I = addNewState();
  int currentState =  InitState_I;
  int wpivot       = M13LETTERWORDSIZE;
 
  for(int i = 0 ; i < 3 ; i++) { 
    
    double num = 0, denom = 0; 
    int    a   = 0;
    wpivot    /= gv_align_alphabet_size;

    for(int w=0; w<M13LETTERWORDSIZE; w++) { 
      
      /* compute denominator */
      if (!(w%(wpivot*gv_align_alphabet_size))) {
	a     = 0;
	denom = 0;
	for (int wp=w; wp < w + wpivot*gv_align_alphabet_size; wp++)
	  denom += p[wp];
      }
      
      /* compute numerator */
      num   += p[w];
     
      /* update result */
      if (!((w+1)%wpivot)) {	
	
	(i == 2) ?  currentState++ : currentState = addNewState();
	
	addNewTransition(a,
			 (currentState+2-InitState_I)/gv_align_alphabet_size+InitState_I-1,
			 (i == 2) ? InitState_I : currentState,
			 num/denom
			 );
	num = 0;
	a++;
      } 
    }
  }
  return 0;
}

/*
 * this method buid a Brejova equivalent HMM automaton 
 * ptable gives several ("nbq" more exactly) M13 models.
 * qtable gives HMM state transition matrix (size "nbq" x "nbq").
 * nbq    indicates the number of hidden state. 	
 *
 */


#define QTABLE(i,j) (qtable[(i)*nbq+(j)])
int automaton::Automaton_HMM(vector< vector<double> > & ptable /*[M13LETTERWORDSIZE][nbq]*/ , vector< vector<double> > & qtable /* [nbq][nbq] */, int nbq) {
  addNewState(TRUE);
  selfLoop(0);
  
#ifdef ASSERTB	
  // check qtable consistency
  for(int i=0;i<nbq;i++){
    double pr = 0;
    for(int j=0;j<nbq;j++){
      pr += qtable[i][j];
    }
    cout << "H(" << i <<") : pr sum = "<< pr << endl;
  }
#endif
  
  // Create the M13 automatons
  for(int i=0;i<nbq;i++){
    Automaton_BrejovaM13(ptable[i]);
  }

  // Add non deterministic transitions on the first M13 states

  // memorise old probabilities
  double * p = new double[gv_align_alphabet_size * nbq];
  for(int a=0; a<gv_align_alphabet_size; a++){
     for(int i=0; i<nbq; i++){  
       p[a  + i* gv_align_alphabet_size] = Pr(a, 1+M13SIZE*i,1+M13SIZE*i+(a+1));
     }
  }
  
  for(int a=0; a<gv_align_alphabet_size; a++){
    for(int i=0; i<nbq; i++){  
      for(int j=0; j<nbq; j++){
	if (i == j)
	  changeTransitionProb( a, 1+M13SIZE*i,  1+M13SIZE*j+(a+1), qtable[i][j] * p[a + j* gv_align_alphabet_size]);
	else
	  addNewTransition( a, 1+M13SIZE*i,  1+M13SIZE*j+(a+1), qtable[i][j] *  p[a + j* gv_align_alphabet_size]);
      }
    }
  }
  delete[] p;
  


#ifdef ASSERTB	
  for(int i=1;i<=M13SIZE*nbq;i++){
    double pr = 0;
    for (int a=0; a<gv_align_alphabet_size ; a++) {
      slist<transition> & list_tr = _states[i].next[a];
      for( slist<transition>::iterator list_iter = list_tr.begin() ;  list_iter != list_tr.end() ; list_iter++){
	pr += list_iter->prob;
      }
    }
    cout << pr << endl;
  }
#endif	


  return 0;  
}







/*
 * this method builds an Homogeneous alignment model automaton given 
 * 
 * state 0 : accepting state
 * state 1 : starting state
 * state 2 : rejecting state
 *
 *
 *   /\/\...
 *  /\/\/...
 * 
 */

int automaton::Automaton_Homogeneous(vector<int> & scores,
				     int length) {  
  bool AcceptingStateReachable = false;
  int  Maxscore = 0;

  for(int i=0;i<gv_align_alphabet_size;i++) {
    Maxscore = MAX(scores[i],Maxscore);
  }
  
  if (Maxscore == 0) {
    ERROR("Automaton_Homogeneous","not positive score is provided");
  }
  
  // set the vector statesOfScores[2][currentscore][lastmaxscore]
  vector< vector< vector<int> > > statesOfScore = vector< vector< vector <int> > > ( 2 , vector< vector <int>  > (Maxscore*(length-1)+1,vector<int>(0)));
  for (int currentscore=0 ; currentscore < Maxscore*(length-1)+1; currentscore++) {
    statesOfScore[0][currentscore] = vector<int>(Maxscore*(length-1)+1-currentscore,0);
    statesOfScore[1][currentscore] = vector<int>(Maxscore*(length-1)+1-currentscore,0);
  }

  int stateFinalAccepting        = addNewState(TRUE);
  selfLoop(stateFinalAccepting);
  int stateInit                  = addNewState();
  int stateReject                = addNewState();
  selfLoop(stateReject);
  
  // insert the first state in the table
  statesOfScore[0][0][0] = stateInit;
  
  for (int l = 0 ; l < length ; l++ ) {
    vector< vector<int> > & statesFrom =  statesOfScore[l%2];
    vector< vector<int> > & statesTo   =  statesOfScore[(l+1)%2];
    for (int currentscore = 0 ; currentscore <= Maxscore*l ; currentscore++ ) {
      for (int lastreachedscore = 0 ; lastreachedscore <= Maxscore*l - currentscore; lastreachedscore++ ) {	
	int stateFrom = statesFrom[currentscore][lastreachedscore];
	if ( stateFrom > 0 ) {
	  for(int a = 0 ; a < gv_align_alphabet_size ; a++) {
	    int scoreTo = currentscore+scores[a];
	    if (scoreTo < 0) {
	      addNewTransition(a,stateFrom,stateReject);
	    } else {
	      if (l == (length - 1)) {
		if (scores[a] >= lastreachedscore) {
		  addNewTransition(a,stateFrom,stateFinalAccepting);
		  AcceptingStateReachable = true;
		} else {
		  addNewTransition(a,stateFrom,stateReject);
		}
	      } else {
		int lastreachedscoreTo = MAX(lastreachedscore-scores[a],0);
		int stateTo = statesTo[scoreTo][lastreachedscoreTo];
		if (stateTo == 0) {
		  stateTo = statesTo[scoreTo][lastreachedscoreTo] = addNewState();
		}
		addNewTransition(a,stateFrom,stateTo);
	      }
	    }
	  }
	}
	statesFrom[currentscore][lastreachedscore] = 0;
      }
    }
  }
  statesOfScore.clear();
  if (!AcceptingStateReachable) {
    ERROR("Automaton_Homogeneous","cannot reach the given score according to -u and -x parameters");
  }
  return 0;
}



/*
 * This method builds a cyclic automaton
 *
 */


int automaton::Automaton_Cycle(int cycle, int * final_list, int final_nb) {
    
  int final_i = 0;
  
  int Finalstate_I = addNewState(TRUE);
  selfLoop(Finalstate_I);
  
  // create a sorted list of final states (do not modify the current one used by "seed")
  int * final_list_sorted = new int[final_nb];
  for(int i = 0; i < final_nb; i++)
    final_list_sorted[i] = final_list[i];
  sort(final_list_sorted, final_list_sorted + final_nb);

  // create states
  for(int i = 0; i < cycle ; i++) {
    if  (final_i < (int)final_nb && final_list_sorted[final_i] == i) {
      addNewState(TRUE); /* final */
      final_i++;
    } else {
      addNewState(FALSE); /* non final */
    }
  }

  // link states
  for(int i = 0; i < cycle ; i++) {
    for(int a = 0 ; a < gv_align_alphabet_size ; a++) {
      addNewTransition( 
		       a,
		       1 + (i)%cycle,
		       1 + (i+1)%cycle
			);
    }
  }

  delete[] final_list_sorted;

  return 0;
}



/*
 * This method generates a new minimized automaton 
 * using Hopcroft minimization
 *
 */

typedef pair<int,int> ABpair;
#define REVERSESTATE(c,a,b) (((c) == (a))?(b):(((c) == (b))?(a):(c)))

automaton * automaton::Hopcroft() {

  int nbStates = this->size();

  vector< int >                  stclass(nbStates);  // class for a given state 
  vector< list<int> >            block(nbStates);    // list of states for a given class
  vector< list<int>::iterator >  location(nbStates); // pointer to each stat inside the given list
  vector< int >                  card(nbStates,0);   // cardinality of a given class
  vector< vector<int> >          SET(nbStates,vector<int>(gv_align_alphabet_size,0)); // stack membership 

  int nbClasses = 0;
  
  // (0) some preprocessing on final labels to get "final_to_class" function
  int max_final = 0;
  int min_final = INT_INFINITY;
  for ( int i=0 ; i<nbStates ; i++) {
    int f = _states[i].final;
    if (f) {
      min_final = MIN(min_final,f);
      max_final = MAX(max_final,f);
    }
  }
  vector< int > final_to_class(max_final+1,-1);
  {
    vector< int >  count(max_final+1,0); // cardinality of a given final label
    for ( int i=0; i<nbStates; i++)
      count[_states[i].final]++;        
    for (int f=0; f <= max_final; f++)
      if (count[f])
	final_to_class[f] = nbClasses++;
    // invert "0" and "min_final" 
    // (to keep some states where "final = 0" in the class 1 -> 
    // -> see why after when rebuiding the minimized automaton [*] )
    final_to_class[0] = min_final;
    final_to_class[1] = 0;
  }


  // (1) initialization
  for ( int i=0 ; i<nbStates ; i++) {
    int _class_ = final_to_class[_states[i].final];
    if (!card[_class_])
      block[_class_] = list<int>();
    stclass[i] = _class_;
    block[_class_].push_front(i);
    location[i] = block[_class_].begin();
    card[_class_]++;
  }
  
  // initial stack 
  stack<ABpair> Set;
  // find the max set and put the other(s) in the refining Set
  {
    int max = 0;
    int max_class = 0;
    for (int _class = 0 ; _class < nbClasses ; _class++) {
      if (max < card[_class]) {
	max       = card[_class];
	max_class = _class;
      } 
    }
    for (int _class = 0 ; _class < nbClasses ; _class++) {
      if (_class != max_class) {
	for(int a = 0; a < gv_align_alphabet_size ; a++) {
	  Set.push(ABpair(a,_class));
	  SET[_class][a] = 1; 
	}
      }
    }
  }

  /* used to count (P_a_inv)Intersert(B), states membership to P_a_inv, and B membership to intersection */
  vector<int> card_P_a_inv_Intersect_B(nbStates,0);
  vector<int> Bsplit(nbStates,0);

  // (2) main hopcroft loop  
  while(!Set.empty()) {
    //(P,a) <- First(S)
    ABpair aP = Set.top(); Set.pop();
    int a = aP.first;
    int P = aP.second;

    SET[P][a] = 0;

    /* (2.a) IsRefined precomputation step */   
    list<int> & Plist               = block[P];
    list<int>   Plist_a_inv         = list<int>();
    // for each state in (P)
    for(list<int>::iterator iter_Plist  = Plist.begin(); 
	                    iter_Plist != Plist.end(); 
                            iter_Plist ++ ) {

      int                Pstate            = *iter_Plist;
      slist<transition>  Pstate_list_a_inv = _states[Pstate].prev[a];

      // for each predecessor *--a-->(P)
      for(slist<transition>::iterator iter_Pstate_list_a_inv  = Pstate_list_a_inv.begin(); 
                                      iter_Pstate_list_a_inv != Pstate_list_a_inv.end(); 
                                      iter_Pstate_list_a_inv ++ ) {
	
	int Pstate_a_inv = iter_Pstate_list_a_inv->state;
	int b = stclass[Pstate_a_inv];
	Plist_a_inv.push_back(Pstate_a_inv);
	card_P_a_inv_Intersect_B[b]++;
      }// transitions
    }// iter_Plist



    int oldNbClasses  = nbClasses;
    vector<int> Bprev = vector<int>();

    /* (2.b) foreach a-1(P) state */
    for(list<int>::iterator i  = Plist_a_inv.begin();
	                    i != Plist_a_inv.end();
         	            i++){
      int Pstate_a_inv = *i;
      int b            = stclass[Pstate_a_inv];
      /* if b is refined by a-1(P)*/

      if ( card_P_a_inv_Intersect_B[b] > 0 && card[b] - card_P_a_inv_Intersect_B[b] > 0) {
	
	
	int bp;
	if ( Bsplit[b] != 0 ) {
	  /* a previous state has been already moved from b to b'=Bsplit[b] */
	  bp = Bsplit[b];
	} else {
	  /* first state to be splited from b */
	  block[nbClasses]  = list<int>();
	  bp = Bsplit[b]    = nbClasses;
	  Bprev.push_back(b);
	  nbClasses++;
	}
	
	block[b].erase(location[Pstate_a_inv]);
	block[bp].push_front(Pstate_a_inv);
	location[Pstate_a_inv] = block[bp].begin();
	card[bp]++;
	stclass[Pstate_a_inv] = bp;
	
      } else {
	
	card_P_a_inv_Intersect_B[b] = 0;
	
      }
    } // for each a-1(P)
    Plist_a_inv.clear();

    /* (2.c) foreach Bp state */
    // Update(S,B,B',B'')
    int i = 0;
    for(int bp = oldNbClasses ; bp < nbClasses; bp ++) {
      int b = Bprev[i++];
      /* reset tables */
      Bsplit[b] = 0;
      card[b]  -= card[bp];
      card_P_a_inv_Intersect_B[b] = 0;
      
      for(int a=0; a < gv_align_alphabet_size ; a++) {
	
	if (SET[b][a]) {
	  SET[bp][a] = 1;
	  Set.push(ABpair(a,bp));
	} else {
	  if (card[b] > card[bp]) {
	    SET[bp][a] = 1;
	    Set.push(ABpair(a,bp));
	  } else { 
	    SET[b][a] = 1;
	    Set.push(ABpair(a,b));
	  }
	}
      }
    }
    Bprev.clear();
    
  }//while (!empty)


  /* Clear Some Hopcroft data */
  Bsplit.clear();
  card_P_a_inv_Intersect_B.clear();
  location.clear();
  card.clear();
  SET.clear();


  /* create the minimized automaton */

  // [*] see why class "1" is "non-final" before
  automaton * result = new automaton();
  for(int c=0; c < nbClasses; c++) {
      result->addNewState(_states[*(block[c].begin())].final);
  }
  
  // put the transitions with the first state (class "stclass[1]") on pos "1"
  for(int c=0; c < nbClasses; c++) {
    for(int a=0 ; a<gv_align_alphabet_size; a++) {
#ifdef ASSERTB	
      assert(DETERMINISTIC(*(block[c].begin()),a));
#endif	
      result->addNewTransition(a,  
			       REVERSESTATE(c,stclass[1],1),
			       REVERSESTATE(stclass[(_states[*(block[c].begin())].next[a].begin()->state)],stclass[1],1)
			       );
    }
  }
  
  stclass.clear();
  block.clear();
  return result;
}


/* check isomorphism */

bool automaton::isIsomorphTo(automaton & other) {


  /* differents sizes */
  if (this->size() != other.size()) {
    return false;
  }
  
  
  /* check the mapping */
  vector<int> map_this_to_other = vector<int>(this->size(),-1);
  stack< pair<int,int> >  statesNbRemaining;

  statesNbRemaining.push(pair<int,int>(1,1));

  while( !statesNbRemaining.empty() ) {
     
    pair<int,int> states = statesNbRemaining.top();
    statesNbRemaining.pop();
    
    if (map_this_to_other[states.first]<0) { 
      
      /* not already found before */
      map_this_to_other[states.first] = states.second;
      
      for(int a=0 ; a<gv_align_alphabet_size; a++) {
	/* push neighboors */
	pair<int,int> states_next = pair<int,int>(this->_states[states.first].next[a].begin()->state,other._states[states.second].next[a].begin()->state);
	  statesNbRemaining.push(states_next);
      }
      
    } else {
      /* check otherwise the consistency */
      if ( map_this_to_other[states.first] != states.second ) {
	return false;
      }
    }
  } // while not empty
  return true;
}





/** @class SeedPrefixesMatchingSet
 *  @brief simple "inner class" to keep states @f$ <X,t> @f$ and @f$ k @f$ during SeedPrefixMatching build
 *  @see Automaton_SeedPrefixesMatching
 */

class SeedPrefixesMatchingSet {

 public:
  /// set of prefixes matching
  int X;
  /// lenght of the last run of '1'
  short t;
  /// length of the maximal prefix in the set @f$ X @f$ : @f$ k = max(X) @f$
  short k;
  /// gives the state @f$< X' = X / max(X), t > @f$ (this state does exists and is created before @f$ <X,t> @f$)
  int Xp;
  /** @brief gives the last state @f$ <X'',t> @f$ created that verifies :
   *   @li @f$ X''/max(X'') = X @f$
   *   @li @f$ max(X'') @f$ is greatest among all created @f$ X'' @f$ verifying the previous condition
   */
  int RevMaxXp;
  /** @brief build a SeedPrefixesMatchingSet
   */
 SeedPrefixesMatchingSet(int X=0,short t=0,short k=0,int Xp=0,int RevMaxXp=0) : X(X), t(t), k(k), Xp(Xp), RevMaxXp(RevMaxXp) {};
};

#define SEEDPREFIXESMATCHING_X(s)          (statesSeedPrefixesMatchingSet[(s)].X)
#define SEEDPREFIXESMATCHING_T(s)          (statesSeedPrefixesMatchingSet[(s)].t)
#define SEEDPREFIXESMATCHING_K(s)          (statesSeedPrefixesMatchingSet[(s)].k)
#define SEEDPREFIXESMATCHING_XP(s)         (statesSeedPrefixesMatchingSet[(s)].Xp)
#define SEEDPREFIXESMATCHING_REVMAXXP(s)   (statesSeedPrefixesMatchingSet[(s)].RevMaxXp)



int automaton::Automaton_SeedPrefixesMatching_old(seed& s, vector< vector <int> > & matchingmatrix, bool nomerge) {
  
  /* Get the seed span */
  int   motif_span = s.span();
  int * motif      = s.table();
    
#ifdef ASSERTB  
  if (motif_span <= 0) {
    ERROR("SeedPrefixesMatching","null or negative span");
  }
  
  for (int i = 0 ; i < motif_span ; i ++)
    if ( motif[i] < 0 || motif[i] >= gv_seed_alphabet_size)
      ERROR("SeedPrefixesMatching","incorrect seed element");
#endif
  
#ifdef BUILD
  cout << " motif_span :" << motif_span << " , motif:" ;
  for (int i = 0 ; i < motif_span ; i++ )
    cout << motif[i] << " " ;
  cout << endl;
#endif


  /* Compute xset_bitsize and X */
  vector<int> L(motif_span+1,0);      
  int xset_bitsize  = 0; // bitsize of the set 
  int xset_size     = 0; // size of the set  
  L[0] = -1;
  for(int i=0; i<motif_span; i++) {
    if (motif[i] < (gv_seed_alphabet_size - (gv_matching_symbol_flag?1:0))) {
      xset_bitsize++;
      L[xset_bitsize]=i;
#ifdef BUILD
      cout << " L[" << xset_bitsize << "] : " <<  L[xset_bitsize] << endl;
#endif
    }
  }
  xset_size = 1<<xset_bitsize;
#ifdef BUILD
      cout << " xset_size :"     << xset_size << ",\t" <<
	      " xset_bitsize : " <<  xset_bitsize << endl;
#endif  
  
  /* Compute TCODE[t] to get an index by use of IndexNb = TCODE[t] + X coding system */ 
  vector<int> TCODE(motif_span+1+(nomerge?1:0),0);
  TCODE[0] = 0;
  for(int maxxsetsize = xset_size , l=xset_bitsize , t = 1 ; t <= motif_span+(nomerge?1:0); t++) {
    if (L[l] == motif_span+(nomerge?1:0)-t) {
      maxxsetsize>>=1;
      l--;
    }
    TCODE[t] = TCODE[t-1] + maxxsetsize;
#ifdef BUILD
    cout << " TCODE["<<t<<"] : " <<  TCODE[t] << endl;
#endif	
  }
  
  
  /* 
   * 1) Precompute E,F functions
   */
  vector< vector<int> > FX(motif_span+(nomerge?1:0),vector<int>(gv_align_alphabet_size,0));
  vector< vector<int> > Fk(motif_span+(nomerge?1:0),vector<int>(gv_align_alphabet_size,0));
  vector< vector< vector<int> > > EX(motif_span+(nomerge?1:0), vector< vector<int> >((xset_bitsize+1), vector<int>(gv_align_alphabet_size,0)));
  vector< vector< vector<int> > > Ek(motif_span+(nomerge?1:0), vector< vector<int> >((xset_bitsize+1), vector<int>(gv_align_alphabet_size,0)));
  
  // F[t][a]
  for (int t=0; t<motif_span+(nomerge?1:0); t++) {
    for (int i = 1 ; i<= xset_bitsize && L[i] <= t ; i++) {
      int r = 1<<(i-1);
      int b = motif[L[i]];
      
      for (int a = 0 ; a<gv_align_alphabet_size ; a++){
	if (MATCHES_AB(a,b)){
	  FX[t][a] = (t>0) ? (FX[t-1][a]) | r : r; 
	  Fk[t][a] = i;
	}
      }
    }
  }
  
  
  // E[t][a]
  for (int i=1; i<=xset_bitsize; i++) { 
    int L_i = L[i]; 
    int b   = motif[L_i];
  
    for (int a=0; a<gv_align_alphabet_size; a++) {
      // does b match the mismatch a 
      if (MATCHES_AB(a,b)) {
	// if X[ib]-t-1 is a joker position, do a mask to have its bit position.
	for (int k=1; k < i; k++) {
	  int t = L[i] - L[k] - 1;
#ifdef ASSERTB
	  assert(t>=0);
	  assert(t<motif_span);
	  assert(k>0);
	  assert(k<=xset_bitsize);
#endif
	  EX[t][k][a] = 1<<(i-1);
	  Ek[t][k][a] = i;
	  //
	}
      }
    }
  }
  
    
  /* 2) automaton build */
  
  // fast state index (designed to retrieve a state given <X,t> code
  vector<int> statesNbIndex(TCODE[motif_span+(nomerge?1:0)]+1 , 0);
  // queue/stack used to store non preprocessed states <X,t> code
#ifdef USEQUEUE
  queue<int>  statesNbRemaining;
#else
  stack<int>  statesNbRemaining;
#endif
  // keep each state information (X,t,k) inside this table
  vector<SeedPrefixesMatchingSet> statesSeedPrefixesMatchingSet(0);
  
  /* create a first state [0] and put it as the final one */
  int Finalstate_I = addNewState(TRUE);
  statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(0,0,0));
  selfLoop(Finalstate_I);

  /* create a second state [1]  : it will be the initial one */
  int Initstate_I   = addNewState();
  statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(0,0,0));
  statesNbIndex[0]  = Initstate_I;
  statesNbRemaining.push(Initstate_I);

  
  while( !statesNbRemaining.empty() ) {
     
    // current state remaining
#ifdef USEQUEUE
     int Xstate_I = statesNbRemaining.front();
#else
     int Xstate_I = statesNbRemaining.top();
#endif
     statesNbRemaining.pop();
 
#ifdef ASSERTB	
     assert(Xstate_I >= 0);
     assert(Xstate_I <  (int)_states.size());
#endif
     int Xstate_X = SEEDPREFIXESMATCHING_X(Xstate_I);
     int Xstate_T = SEEDPREFIXESMATCHING_T(Xstate_I);
     int Xstate_K = SEEDPREFIXESMATCHING_K(Xstate_I);

#ifdef BUILD
    cout << "$pop  I:" << Xstate_I << " < X:" << Xstate_X << ", t:" << Xstate_T <<", k:"<< Xstate_K <<" > " << endl;
#endif	

#ifdef ASSERTB	
    assert(Xstate_X >= 0);
    assert(Xstate_X < xset_size);
    assert(Xstate_T >= 0);
    assert(Xstate_T < motif_span + (nomerge?1:0));
    assert(Xstate_K >= 0);
    assert(Xstate_K <= xset_bitsize);
    assert(TCODE[Xstate_T] + Xstate_X < TCODE[motif_span+(nomerge?1:0)]);
#endif
    

   
    int XPstate_Ix=0, XPstate_I=0, XPstate_X=0;
    
    if (Xstate_K > 0) {
      XPstate_X   = Xstate_X ^ (1 << (Xstate_K-1));    // get X'
      XPstate_Ix  = TCODE[Xstate_T] + XPstate_X;   // get (X',t) index
      XPstate_I   = statesNbIndex[XPstate_Ix];
    }
    
    for(int a=0;a<gv_align_alphabet_size;a++) {
      
	// next state according to letter "a"
	int Ystate_X = Xstate_X;
	int Ystate_T = Xstate_T;
	int Ystate_K = Xstate_K;
      
	if ( a == (gv_align_alphabet_size - (gv_matching_symbol_flag?1:0))) {
	  Ystate_T++;
        } else {

	  if ( Xstate_K > 0 ) {
#ifdef ASSERTB	
	    assert(DETERMINISTIC(XPstate_I,a));
#endif
	    int YPstate_I = _states[XPstate_I].next[a].begin()->state;
	    // YP result
	    int YPstate_X = SEEDPREFIXESMATCHING_X(YPstate_I);
	    int YPstate_K = SEEDPREFIXESMATCHING_K(YPstate_I);
	    //  Y result
#ifdef BUILD
	    assert(Xstate_T >= 0);
	    assert(Xstate_T < motif_span+(nomerge?1:0));
	    assert(Xstate_K > 0);
	    assert(Xstate_K <= xset_bitsize);
#endif

	    Ystate_X  =      YPstate_X | EX[Xstate_T][Xstate_K][a] ; // 
	    Ystate_K  = MAX( YPstate_K , Ek[Xstate_T][Xstate_K][a] );
#ifdef BUILD 
	    cout << "**V "<< endl;
	    cout << "YPstate_X:" <<  YPstate_X << ",  YPstate_K:" << YPstate_K << endl;
	    cout << "Ystate_X:"  <<  Ystate_X << ",  Ystate_K:" << Ystate_K << endl;
	  
#endif
	  } else {
	    Ystate_X  = FX[Xstate_T][a];
	    Ystate_K  = Fk[Xstate_T][a];

#ifdef BUILD 
	    cout << "*U "<< endl;
	    cout << "Ystate_X:"  <<  Ystate_X << ",  Ystate_K:" << Ystate_K<< endl;
	  
#endif

	  } // if ( Xstate_K > 0 )
	  Ystate_T  = 0;
	  
	}// if (a == gv_alphabet_size - 1)

	// Y result Index
	int Ystate_Ix = TCODE[Ystate_T] + Ystate_X;
	int Ystate_I  = statesNbIndex[Ystate_Ix];
	int final     = ((Ystate_T + L[Ystate_K]) >= (motif_span - 1)) ? TRUE : FALSE;


	//Y : either final state (->0) or "after final state" thus a previous one  
	if (Ystate_T + L[Ystate_K] >= motif_span - 1 + (nomerge?1:0)) {
	  if (nomerge){ // final states are not merged and  their transitions are consided as "normal states" :
	                // the only difference is their maximal prefix matching that cannot be extended more that "span"
	    if (Ystate_K > 0) 
	      Ystate_X ^= 1<<(Ystate_K-1);
	    else
	      Ystate_T--;
	    Ystate_Ix = TCODE[Ystate_T] + Ystate_X;
	    Ystate_I  = statesNbIndex[Ystate_Ix];
	  }else{ // final states merged and absorbant (->0)
	    Ystate_I = 0;
	  }
	} else {
	  // else check if the state has not been already created
	  if (!Ystate_I) {
	    // create it and push it inside the list of states which need update.
	    Ystate_I = addNewState(final);
	    statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(Ystate_X,Ystate_T,Ystate_K));
	    statesNbIndex[Ystate_Ix] =  Ystate_I;
	    statesNbRemaining.push(Ystate_I);

#ifdef BUILD
	    cout << "$push I:"<< Ystate_I << " < X:" << hex << Ystate_X << " , t:" << dec << Ystate_T << " , k:" << Ystate_K << " > " << endl;
#endif
	  }
	}
#ifdef BUILD
	cout << "> add transition ( a:" << dec << a << ", q1:" << dec << Xstate_I << " , q2:" << Ystate_I << " ) " << endl;
#endif			
	addNewTransition(a,Xstate_I,Ystate_I);
#ifdef ASSERTB	
	assert(DETERMINISTIC(Xstate_I,a));
#endif
    }// for all letter in ALPHABET.
  }// while states remain. 
  
  /* Free unused data needed to build the automaton */
  L.clear();
  FX.clear();
  Fk.clear();
  EX.clear();
  Ek.clear();
  statesNbIndex.clear();
  statesSeedPrefixesMatchingSet.clear();
  return 0;
}







int automaton::Automaton_SeedPrefixesMatching(seed& s, vector< vector <int> > & matchingmatrix, bool nomerge) {

  /* Get the seed span */
  int   motif_span = s.span();
  int * motif      = s.table();
    
#ifdef ASSERTB  
  if (motif_span <= 0) {
    ERROR("SeedPrefixesMatching","null or negative span");
  }
  
  for (int i = 0 ; i < motif_span ; i ++)
    if ( motif[i] < 0 || motif[i] >= gv_seed_alphabet_size)
      ERROR("SeedPrefixesMatching","incorrect seed element");
#endif
  
#ifdef BUILD
  cout << " motif_span :" << motif_span << " , motif:" ;
  for (int i = 0 ; i < motif_span ; i++ )
    cout << motif[i] << " " ;
  cout << endl;
#endif


  /* Compute xset_bitsize and X */
  vector<int> L(motif_span+1,0);      
  int xset_bitsize  = 0; // bitsize of the set 
  int xset_size     = 0; // size of the set  
  L[0] = -1;
  for(int i=0; i<motif_span; i++) {
    if (motif[i] < (gv_seed_alphabet_size - (gv_matching_symbol_flag?1:0))) {
      xset_bitsize++;
      L[xset_bitsize]=i;
#ifdef BUILD
      cout << " L[" << xset_bitsize << "] : " <<  L[xset_bitsize] << endl;
#endif
    }
  }
  xset_size = 1<<xset_bitsize;
#ifdef BUILD
      cout << " xset_size :"     << xset_size << ",\t" <<
	      " xset_bitsize : " <<  xset_bitsize << endl;
#endif  
  
  
  /* 
   * 1) Precompute E,F functions
   */
  vector< vector<int> > FX(motif_span+(nomerge?1:0),vector<int>(gv_align_alphabet_size,0));
  vector< vector<int> > Fk(motif_span+(nomerge?1:0),vector<int>(gv_align_alphabet_size,0));
  vector< vector< vector<int> > > EX(motif_span+(nomerge?1:0), vector< vector<int> >((xset_bitsize+1), vector<int>(gv_align_alphabet_size,0)));
  vector< vector< vector<int> > > Ek(motif_span+(nomerge?1:0), vector< vector<int> >((xset_bitsize+1), vector<int>(gv_align_alphabet_size,0)));
  
  // F[t][a]
  for (int t=0; t<motif_span+(nomerge?1:0); t++) {
    for (int i = 1 ; i<= xset_bitsize && L[i] <= t ; i++) {
      int r = 1<<(i-1);
      int b = motif[L[i]];
      
      for (int a = 0 ; a<gv_align_alphabet_size ; a++){
	if (MATCHES_AB(a,b)){
	  FX[t][a] = (t>0) ? (FX[t-1][a]) | r : r; 
	  Fk[t][a] = i;
	}
      }
    }
  }
  
  
  // E[t][a]
  for (int i=1; i<=xset_bitsize; i++) { 
    int L_i = L[i]; 
    int b   = motif[L_i];
  
    for (int a=0; a<gv_align_alphabet_size; a++) {
      // does b match the mismatch a 
      if (MATCHES_AB(a,b)) {
	// if X[ib]-t-1 is a joker position, do a mask to have its bit position.
	for (int k=1; k < i; k++) {
	  int t = L[i] - L[k] - 1;
#ifdef ASSERTB
	  assert(t>=0);
	  assert(t<motif_span);
	  assert(k>0);
	  assert(k<=xset_bitsize);
#endif
	  EX[t][k][a] = 1<<(i-1);
	  Ek[t][k][a] = i;
	  //
	}
      }
    }
  }
  
    
  /* 2) automaton build */  
  // queue/stack used to store non preprocessed states <X,t> code
#ifdef USEQUEUE
  queue<int>  statesNbRemaining;
#else
  stack<int>  statesNbRemaining;
#endif
  // keep each state information (X,t,k) inside this table
  vector<SeedPrefixesMatchingSet> statesSeedPrefixesMatchingSet(0);
  
  /* create a first state [0] and put it as the final one */  
  int Finalstate_I = addNewState(TRUE);
  statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(0,0,0,0,0));
  selfLoop(Finalstate_I);

  /* create a second state [1]  : it will be the initial one */
  int Initstate_I   = addNewState();
  statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(0,0,0,1,1));
  

  /* create first level states and push them in the queue */
  int State_R = 0;
  for(int a=0;a<gv_align_alphabet_size;a++) {
    bool matches = MATCHES_AB(a,motif[0]);
    int  final   = ((motif_span == 1) && matches) ? TRUE : FALSE;
    
    if ( a == (gv_align_alphabet_size - (gv_matching_symbol_flag?1:0))) {
      if (final && !nomerge) {
	addNewTransition(a,Initstate_I,Finalstate_I);
      } else {
	int State_I = addNewState(final);
	statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(0,1,0,Initstate_I,-1));
	statesNbRemaining.push(State_I);
	addNewTransition(a,Initstate_I,State_I);
      }
    } else {
      if (matches) {
	if (final && !nomerge) {
	  State_R = Finalstate_I;
	} else {
	  if (!State_R) {
	    State_R = addNewState(final);
	    statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(1,0,1,Initstate_I,-1));
	    statesNbRemaining.push(State_R);	  
	  }
	}
	addNewTransition(a,Initstate_I,State_R);
      } else {
	addNewTransition(a,Initstate_I,Initstate_I);
      }
    }
  }




  

  /* main loop */
  while( !statesNbRemaining.empty() ) {
     
    // current state Xstate
#ifdef USEQUEUE
     int Xstate_I = statesNbRemaining.front();
#else
     int Xstate_I = statesNbRemaining.top();
#endif
     statesNbRemaining.pop();
 
#ifdef ASSERTB	
     assert(Xstate_I >= 0);
     assert(Xstate_I <  (int)_states.size());
#endif

     int Xstate_X    = SEEDPREFIXESMATCHING_X (Xstate_I);
     int Xstate_T    = SEEDPREFIXESMATCHING_T (Xstate_I);
     int Xstate_K    = SEEDPREFIXESMATCHING_K (Xstate_I);

#ifdef ASSERTB	
     assert(Xstate_X >= 0);
     assert(Xstate_X < (1<<xset_bitsize));
     assert(Xstate_T >= 0);
     assert(Xstate_T < motif_span + (nomerge?1:0));
     assert(Xstate_K >= 0);
     assert(Xstate_K <= xset_bitsize);
#endif

     // XPstate : defined as "Xstate" where the maximal matching prefix fails ...
     int XPstate_I   = SEEDPREFIXESMATCHING_XP(Xstate_I);

#ifdef ASSERTB
      assert(XPstate_I >= 0);
      assert(XPstate_I < (int)_states.size());
#endif

#ifdef BUILD
    cout << "$pop  I:" << Xstate_I << " < X:" << Xstate_X << ", t:" << Xstate_T <<", k:"<< Xstate_K <<" > " << endl;
#endif	
    
    /* for each letter (a) : compute the transition (Xstate)--a-->Ystate [and determine Ystate] */
    for(int a=0; a<gv_align_alphabet_size; a++) {

#ifdef BUILD
      cout << "a = " << a << endl;
#endif
      // set at true if Ystate has not been already created before
      bool Y_not_there = false;

      // YPstate : defined as (XPstate)--a-->
      int YPstate_I   = _states[XPstate_I].next[a].begin()->state;
   
#ifdef ASSERTB
      assert(YPstate_I >= 0);
      assert(YPstate_I < size());
#endif

      int YPstate_T   = SEEDPREFIXESMATCHING_T(YPstate_I);
      int YPstate_X   = SEEDPREFIXESMATCHING_X(YPstate_I);
      int YPstate_K   = SEEDPREFIXESMATCHING_K(YPstate_I);
      
#ifdef ASSERTB
      assert(YPstate_X >= 0);
      assert(YPstate_X < (1<<xset_bitsize));
      assert(YPstate_T >= 0);
      assert(YPstate_T < motif_span + (nomerge?1:0));
      assert(YPstate_K >= 0);
      assert(YPstate_K <= xset_bitsize);
#endif

      // YPRstate ...value is either -1 if undefined, 
      // otherwise >= 0 to indicate the last state created that
      // verifies YPRstate / maximalprefixmatching{YPRstate} = YPstate.
      int YPRstate_I  = SEEDPREFIXESMATCHING_REVMAXXP(YPstate_I);

#ifdef ASSERTB
      assert(YPRstate_I >= (-1));
      assert(YPRstate_I < size());
#endif

      // Ystate_I : next state according to letter "a"
      // by default values are given here
      // and will be processed 
      int Ystate_I    = Xstate_I;
      int Ystate_X    = Xstate_X;
      int Ystate_T    = Xstate_T;
      int Ystate_K    = Xstate_K;
 
      // (a == match)
      if ( a == (gv_align_alphabet_size - (gv_matching_symbol_flag?1:0))) {
	// UPDATE STATE VALUES
	Ystate_T++;         // (a) add a 1 to the run of 1 and 
	// NOMERGE : CHECK FINAL STATE
	if (nomerge && L[Ystate_K] + Ystate_T >= motif_span) {
	  if (Ystate_K > 0) {
	    Ystate_I = YPstate_I;
	    Ystate_T = YPstate_T;
	    Ystate_X = YPstate_X;
	    Ystate_K = YPstate_K;
	  } else {	    
	    Ystate_I = Xstate_I;
	    Ystate_T = Xstate_T;
	    Ystate_X = Xstate_X;
	    Ystate_K = Xstate_K;
	  }
	} else {
	  //UPDATE STATE INDEX 
	  Y_not_there = true;
	}
      } else {
	// (a != match)
	// either (K > 0) or (K == 0) 
	if ( Xstate_K > 0 ) {

#ifdef ASSERTB	
	  assert(DETERMINISTIC(XPstate_I,a));
#endif
	  // (b) add a possible prefix when compared with XP ??
	  //   - either EX[Xstate_T][Xstate_K][a] == 0 
	  //     and  Y = YP : thus  Y already exists 
	  //   - otherwise EX = 1<<Ek and thus the possible Y may exist ( it can be YPR )

	  if (EX[Xstate_T][Xstate_K][a]) {

#ifdef BUILD
	    cout << "EX != 0 : maximal prefix does match " << endl;
#endif
	    // UPDATE STATE VALUES
	    Ystate_X  =      YPstate_X | EX[Xstate_T][Xstate_K][a]; 
	    Ystate_K  = MAX( YPstate_K , Ek[Xstate_T][Xstate_K][a]);

	    // NOMERGE : CHECK FINAL STATE  
	    if (nomerge && L[Ystate_K] >= motif_span) {
#ifdef ASSERTB
	      assert(YPstate_X ==  Ystate_X^(1 << (Ystate_K-1)));
#endif
	      goto Ex0;
	    }

	    // UPDATE STATE INDEX
	    if (YPRstate_I >=0 && SEEDPREFIXESMATCHING_X(YPRstate_I) == Ystate_X /* **COMMENT:NOT NEEDED** && SEEDPREFIXESMATCHING_T(YPRstate_I) == 0 */) {
#ifdef ASSERTB
	      assert(SEEDPREFIXESMATCHING_T(YPRstate_I) == 0);
#endif
	      Ystate_I = YPRstate_I; // state already exists (has been processed before)	     	      
	    } else {
	      Y_not_there = true;     // state must be created for the first time
	    }
	  } else {
	  Ex0:
#ifdef BUILD
	    cout << "Ex == 0 : maximal prefix doesnt match : Y is thus YP" << endl;
#endif

	    // UPDATE STATE VALUES
	    Ystate_X  =      YPstate_X ;
	    Ystate_K  =      YPstate_K ;
	    // NOMERGE : CHECK FINAL STATE  
	    // ** not needed **
	    // UPDATE STATE INDEX 
	    Ystate_I  =      YPstate_I ;
	  }
	  
#ifdef BUILD
	  cout << "**V " << endl;
	  cout << "YPstate_X:" <<  YPstate_X << ",  YPstate_K:" << YPstate_K << endl;
	  cout << "Ystate_X:"  <<  Ystate_X  << ",  Ystate_K:"  << Ystate_K  << endl;
#endif
	} else { // Xstate is empty ... and we have read 1^t.a  since ... 

	  // (c) initial case when reading 1^t . a
	  //     three subcases can be deduced here :
	  //     - either there is one prefix matching on the full 1^t.a span ? in this case does such state exists ?
	  //     - either the maximal prefix does not 
	  //
#ifdef BUILD
	  cout << " FX on 1^"<< Xstate_T << "." << a << endl;
#endif

	  // UPDATE STATE VALUES
	  Ystate_X   = FX[Xstate_T][a];
	  Ystate_K   = Fk[Xstate_T][a];
	  
	  // NOMERGE : CHECK FINAL STATE
	  if (nomerge && L[Ystate_K] >= motif_span) {
#ifdef ASSERTB
	    assert(YPstate_X ==  Ystate_X^(1 << (Ystate_K-1)));
#endif
	    Ystate_I = YPstate_I;
	  } else {

	    // UPDATE INDEX
	    if (YPRstate_I >= 0 && SEEDPREFIXESMATCHING_X(YPRstate_I) == Ystate_X && SEEDPREFIXESMATCHING_T(YPRstate_I) == 0) { // if FX does gives a new prefix when compared with YP
	      Ystate_I  = YPRstate_I;
	      //cerr << "*";
	    } else {
	      if (SEEDPREFIXESMATCHING_X(YPstate_I) == Ystate_X && SEEDPREFIXESMATCHING_T(YPstate_I) == 0) {
		Ystate_I  = YPstate_I;
		//cerr << "+";
	      } else {
		Y_not_there = true;
	      }
	    }
	  }
#ifdef BUILD
	  cout << "*U "<< endl;
	  cout << "Ystate_X:"  <<  Ystate_X << ",  Ystate_K:" << Ystate_K<< endl;
#endif
	}// if ( Xstate_K > 0 )

	Ystate_T  = 0;

      }// if (a == gv_alphabet_size - 1)
      
      
      int final = ((Ystate_T + L[Ystate_K]) >= (motif_span - 1)) ? TRUE : FALSE;

      if (!nomerge  && final) {

	Ystate_I = 0;

      } else {

	// else check if the state has not been already created
	if (Y_not_there) {
	  // create it and push it inside the list of states which need update.
	  Ystate_I = addNewState(final);
	  SEEDPREFIXESMATCHING_REVMAXXP(YPstate_I) = Ystate_I;
	  statesSeedPrefixesMatchingSet.push_back(SeedPrefixesMatchingSet(Ystate_X,Ystate_T,Ystate_K,YPstate_I,0));
	  statesNbRemaining.push(Ystate_I);
	  
#ifdef BUILD
	  cout << "$push I:"<< Ystate_I << " < X:" << hex << Ystate_X << " , t:" << dec << Ystate_T << " , k:" << Ystate_K << " > " << endl;
#endif
	}
      }
#ifdef BUILD
      cout << "> add transition ( a:" << dec << a << ", q1:" << dec << Xstate_I << " , q2:" << Ystate_I << " ) " << endl;
#endif
      addNewTransition(a,Xstate_I,Ystate_I);
#ifdef ASSERTB
      assert(DETERMINISTIC(Xstate_I,a));
#endif
    }// for all letter in ALPHABET.
  }// while states remain. 
  
  /* Free unused data needed to build the automaton */
  L.clear();
  FX.clear();
  Fk.clear();
  EX.clear();
  Ek.clear();
  
  statesSeedPrefixesMatchingSet.clear();
  return 0;
}




/** @class SeedPrefixesMatchingCostSet
 *  @brief simple "inner class" to keep states @f$ <X,t> @f$ and  @f$ k + Cost @f$
 *         during SeedPrefixMatching build
 *  @see Automaton_SeedPrefixesMatchingCost
 */

class SeedPrefixesMatchingCostSet {

 public:
  /// set of prefixes matching
  int X;
  /// lenght of the last run of '1'
  short t;
  /// length of the maximal prefix in the set @f$ X @f$ : @f$ k = max(X) @f$
  short k;
  /// gives the state @f$< X' = X / max(X), t > @f$ (this state does exists and is created before @f$ <X,t> @f$)
  int Xp;
  /** @brief gives the last state @f$ <X'',t> @f$ created that verifies :
   *   @li @f$ X''/max(X'') = X @f$
   *   @li @f$ max(X'') @f$ is greatest among all created @f$ X'' @f$ verifying the previous condition
   */
  int RevMaxXp;
  /// gives the Cost at this state
  int Cost;
  /** @brief build a SeedPrefixesMatchingCostSet
   */
 SeedPrefixesMatchingCostSet(int X=0, short t=0, short k=0, int Xp=0, int RevMaxXp=0, int Cost = 0) : X(X), t(t), k(k), Xp(Xp), RevMaxXp(RevMaxXp), Cost(Cost) {};
};

#define SEEDPREFIXESMATCHINGCOST_X(s)          (statesSeedPrefixesMatchingCostSet[(s)].X)
#define SEEDPREFIXESMATCHINGCOST_T(s)          (statesSeedPrefixesMatchingCostSet[(s)].t)
#define SEEDPREFIXESMATCHINGCOST_K(s)          (statesSeedPrefixesMatchingCostSet[(s)].k)
#define SEEDPREFIXESMATCHINGCOST_XP(s)         (statesSeedPrefixesMatchingCostSet[(s)].Xp)
#define SEEDPREFIXESMATCHINGCOST_REVMAXXP(s)   (statesSeedPrefixesMatchingCostSet[(s)].RevMaxXp)
#define SEEDPREFIXESMATCHINGCOST_COST(s)       (statesSeedPrefixesMatchingCostSet[(s)].Cost)



int automaton::Automaton_SeedPrefixesMatchingCost(seed& s, vector< vector <int> > & matchingmatrix, bool nomerge, vector<int> & costs, int cost_threshold) {

  /* Get the seed span */
  int   motif_span = s.span();
  int * motif      = s.table();
    
#ifdef ASSERTB  
  if (motif_span <= 0) {
    ERROR("SeedPrefixesMatchingCost","null or negative span");
  }
  
  for (int i = 0 ; i < motif_span ; i ++)
    if ( motif[i] < 0 || motif[i] >= gv_seed_alphabet_size)
      ERROR("SeedPrefixesMatchingCost","incorrect seed element");
#endif
  
#ifdef BUILD
  cout << " motif_span :" << motif_span << " , motif:" ;
  for (int i = 0 ; i < motif_span ; i++ )
    cout << motif[i] << " " ;
  cout << endl;
#endif


  /* Compute xset_bitsize and X */
  vector<int> L(motif_span+1,0);      
  int xset_bitsize  = 0; // bitsize of the set 
  int xset_size     = 0; // size of the set  
  L[0] = -1;
  for(int i=0; i<motif_span; i++) {
    if (motif[i] < (gv_seed_alphabet_size - (gv_matching_symbol_flag?1:0))) {
      xset_bitsize++;
      L[xset_bitsize]=i;
#ifdef BUILD
      cout << " L[" << xset_bitsize << "] : " <<  L[xset_bitsize] << endl;
#endif
    }
  }
  xset_size = 1<<xset_bitsize;
#ifdef BUILD
      cout << " xset_size :"     << xset_size << ",\t" <<
	      " xset_bitsize : " <<  xset_bitsize << endl;
#endif  
  
  
  /* 
   * 1) Precompute E,F functions
   */
  vector< vector<int> > FX(motif_span+(nomerge?1:0),vector<int>(gv_align_alphabet_size,0));
  vector< vector<int> > Fk(motif_span+(nomerge?1:0),vector<int>(gv_align_alphabet_size,0));
  vector< vector< vector<int> > > EX(motif_span+(nomerge?1:0), vector< vector<int> >((xset_bitsize+1), vector<int>(gv_align_alphabet_size,0)));
  vector< vector< vector<int> > > Ek(motif_span+(nomerge?1:0), vector< vector<int> >((xset_bitsize+1), vector<int>(gv_align_alphabet_size,0)));
  
  // F[t][a]
  for (int t=0; t<motif_span+(nomerge?1:0); t++) {
    for (int i = 1 ; i<= xset_bitsize && L[i] <= t ; i++) {
      int r = 1<<(i-1);
      int b = motif[L[i]];
      
      for (int a = 0 ; a<gv_align_alphabet_size ; a++){
	if (MATCHES_AB(a,b)){
	  FX[t][a] = (t>0) ? (FX[t-1][a]) | r : r; 
	  Fk[t][a] = i;
	}
      }
    }
  }
  
  
  // E[t][a]
  for (int i=1; i<=xset_bitsize; i++) { 
    int L_i = L[i]; 
    int b   = motif[L_i];
  
    for (int a=0; a<gv_align_alphabet_size; a++) {
      // does b match the mismatch a 
      if (MATCHES_AB(a,b)) {
	// if X[ib]-t-1 is a joker position, do a mask to have its bit position.
	for (int k=1; k < i; k++) {
	  int t = L[i] - L[k] - 1;
#ifdef ASSERTB
	  assert(t>=0);
	  assert(t<motif_span);
	  assert(k>0);
	  assert(k<=xset_bitsize);
#endif
	  EX[t][k][a] = 1<<(i-1);
	  Ek[t][k][a] = i;
	  //
	}
      }
    }
  }
  
    
  /* 2) automaton build */  
  // queue/stack used to store non preprocessed states <X,t> code
#ifdef USEQUEUE
  queue<int>  statesNbRemaining;
#else
  stack<int>  statesNbRemaining;
#endif
  // keep each state information (X,t,k) inside this table
  vector<SeedPrefixesMatchingCostSet> statesSeedPrefixesMatchingCostSet(0);
  
  /* create a first state [0] and put it as the final one */  
  int Finalstate_I = addNewState(TRUE);
  statesSeedPrefixesMatchingCostSet.push_back(SeedPrefixesMatchingCostSet(0,0,0,0,0,INT_INFINITY));
  selfLoop(Finalstate_I);

  /* create a second state [1]  : it will be the initial one */
  int Initstate_I = addNewState();
  statesSeedPrefixesMatchingCostSet.push_back(SeedPrefixesMatchingCostSet(0,0,0,1,1,0));
  
  /* bag reject set [2] */
  int RejectBagstate_I  = addNewState();
  statesSeedPrefixesMatchingCostSet.push_back(SeedPrefixesMatchingCostSet(0,0,0,0,0,INT_INFINITY));
  selfLoop(RejectBagstate_I);

  /* create first level states and push them in the queue */
  int State_R = 0;
  for(int a=0;a<gv_align_alphabet_size;a++) {
    bool matches = MATCHES_AB(a,motif[0]);
    int  final   =  ((motif_span == 1) && matches) ? TRUE : FALSE;
    int  cost    = SEEDPREFIXESMATCHINGCOST_COST(Initstate_I) + costs[a];
    if (cost > cost_threshold) {
      addNewTransition(a,Initstate_I,RejectBagstate_I);
    } else {      
      if ( a == (gv_align_alphabet_size - (gv_matching_symbol_flag?1:0))) {
	if (final && !nomerge) {
	  addNewTransition(a,Initstate_I,Finalstate_I);
	} else {
	  int State_I = addNewState(final);
	  statesSeedPrefixesMatchingCostSet.push_back(SeedPrefixesMatchingCostSet(0,1,0,Initstate_I,-1,cost));
	  statesNbRemaining.push(State_I);
	  addNewTransition(a,Initstate_I,State_I);
	}      
      } else {
	if (matches) {
	  if (final && !nomerge) {
	    State_R = Finalstate_I;
	  } else {
	    if (!State_R) {
	      State_R = addNewState(final);
	      statesSeedPrefixesMatchingCostSet.push_back(SeedPrefixesMatchingCostSet(1,0,1,Initstate_I,-1,cost));
	      statesNbRemaining.push(State_R);
	    }
	  }
	  addNewTransition(a,Initstate_I,State_R);
	} else {
	  addNewTransition(a,Initstate_I,Initstate_I);
	}
      }
    }
  }



  /* main loop */
  while( !statesNbRemaining.empty() ) {
     
    // current state Xstate
#ifdef USEQUEUE
     int Xstate_I = statesNbRemaining.front();
#else
     int Xstate_I = statesNbRemaining.top();
#endif
     statesNbRemaining.pop();
 
#ifdef ASSERTB	
     assert(Xstate_I >= 0);
     assert(Xstate_I <  (int)_states.size());
#endif

     int Xstate_X      = SEEDPREFIXESMATCHINGCOST_X (Xstate_I);
     int Xstate_T      = SEEDPREFIXESMATCHINGCOST_T (Xstate_I);
     int Xstate_K      = SEEDPREFIXESMATCHINGCOST_K (Xstate_I);
     int Xstate_cost   = SEEDPREFIXESMATCHINGCOST_COST (Xstate_I);

#ifdef ASSERTB	
     assert(Xstate_X >= 0);
     assert(Xstate_X < (1<<xset_bitsize));
     assert(Xstate_T >= 0);
     assert(Xstate_T < motif_span + (nomerge?1:0));
     assert(Xstate_K >= 0);
     assert(Xstate_K <= xset_bitsize);
     assert(Xstate_cost <= cost_threshold);
#endif

     // XPstate : defined as "Xstate" where the maximal matching prefix fails ...
     int XPstate_I   = SEEDPREFIXESMATCHINGCOST_XP(Xstate_I);

#ifdef ASSERTB
      assert(XPstate_I >= 0);
      assert(XPstate_I < (int)_states.size());
#endif

#ifdef BUILD
    cout << "$pop  I:" << Xstate_I << " < X:" << Xstate_X << ", t:" << Xstate_T <<", k:"<< Xstate_K <<" > " << endl;
#endif	
    
    /* for each letter (a) : compute the transition (Xstate)--a-->Ystate [and determine Ystate] */
    for(int a=0; a<gv_align_alphabet_size; a++) {
      int Ystate_cost   = Xstate_cost + costs[a];

#ifdef BUILD
      cout << "a = " << a << endl;
#endif
      if (Ystate_cost > cost_threshold) {
	addNewTransition(a,Xstate_I,RejectBagstate_I);	  
      } else {
	// set at true if Ystate has not been already created before
	bool Y_not_there = false;
	
	// YPstate : defined as (XPstate)--a-->
	int YPstate_I   = _states[XPstate_I].next[a].begin()->state;
	
#ifdef ASSERTB
	assert(YPstate_I >= 0);
	assert(YPstate_I < size());
#endif
	
	int YPstate_T   = SEEDPREFIXESMATCHINGCOST_T(YPstate_I);
	int YPstate_X   = SEEDPREFIXESMATCHINGCOST_X(YPstate_I);
	int YPstate_K   = SEEDPREFIXESMATCHINGCOST_K(YPstate_I);
	
#ifdef ASSERTB
	assert(YPstate_X >= 0);
	assert(YPstate_X < (1<<xset_bitsize));
	assert(YPstate_T >= 0);
	assert(YPstate_T < motif_span + (nomerge?1:0));
	assert(YPstate_K >= 0);
	assert(YPstate_K <= xset_bitsize);
#endif
	
	// YPRstate ...value is either -1 if undefined, 
	// otherwise >= 0 to indicate the last state created that
	// verifies YPRstate / maximalprefixmatching{YPRstate} = YPstate.
	int YPRstate_I  = SEEDPREFIXESMATCHINGCOST_REVMAXXP(YPstate_I);
	
#ifdef ASSERTB
	assert(YPRstate_I >= (-1));
	assert(YPRstate_I < size());
#endif
	
	// Ystate_I : next state according to letter "a"
	// by default values are given here
	// and will be processed 
	int Ystate_I    = Xstate_I;
	int Ystate_X    = Xstate_X;
	int Ystate_T    = Xstate_T;
	int Ystate_K    = Xstate_K;
	
	// (a == match)
	if ( a == (gv_align_alphabet_size - (gv_matching_symbol_flag?1:0))) {
	  // UPDATE STATE VALUES
	  Ystate_T++;         // (a) add a 1 to the run of 1 and 
	  // NOMERGE : CHECK FINAL STATE
	  if (nomerge && L[Ystate_K] + Ystate_T >= motif_span) {
	    if (Ystate_K > 0) {
	      Ystate_I = YPstate_I;
	      Ystate_T = YPstate_T;
	      Ystate_X = YPstate_X;
	      Ystate_K = YPstate_K;
	    } else {	    
	      Ystate_I = Xstate_I;
	      Ystate_T = Xstate_T;
	      Ystate_X = Xstate_X;
	      Ystate_K = Xstate_K;
	    }
	  } else {
	    //UPDATE STATE INDEX 
	    Y_not_there = true;
	  }
	} else {
	  // (a != match)
	  // either (K > 0) or (K == 0) 
	  if ( Xstate_K > 0 ) {
	    
#ifdef ASSERTB	
	    assert(DETERMINISTIC(XPstate_I,a));
#endif
	    // (b) add a possible prefix when compared with XP ??
	    //   - either EX[Xstate_T][Xstate_K][a] == 0 
	    //     and  Y = YP : thus  Y already exists 
	    //   - otherwise EX = 1<<Ek and thus the possible Y may exist ( it can be YPR )
	    
	    if (EX[Xstate_T][Xstate_K][a]) {
	      
#ifdef BUILD
	      cout << "EX != 0 : maximal prefix does match " << endl;
#endif
	      // UPDATE STATE VALUES
	      Ystate_X  =      YPstate_X | EX[Xstate_T][Xstate_K][a]; 
	      Ystate_K  = MAX( YPstate_K , Ek[Xstate_T][Xstate_K][a]);
	      
	      // NOMERGE : CHECK FINAL STATE  
	      if (nomerge && L[Ystate_K] >= motif_span) {
#ifdef ASSERTB
		assert(YPstate_X ==  Ystate_X^(1 << (Ystate_K-1)));
#endif
		goto Ex0;
	      }
	      
	      // UPDATE STATE INDEX
	      if (YPRstate_I >=0 && SEEDPREFIXESMATCHINGCOST_X(YPRstate_I) == Ystate_X /* **COMMENT:NOT NEEDED** && SEEDPREFIXESMATCHING_T(YPRstate_I) == 0 */) {
#ifdef ASSERTB
		assert(SEEDPREFIXESMATCHINGCOST_T(YPRstate_I) == 0);
#endif
		Ystate_I = YPRstate_I; // state already exists (has been processed before)	     	      
	      } else {
		Y_not_there = true;     // state must be created for the first time
	      }
	    } else {
	    Ex0:
#ifdef BUILD
	      cout << "Ex == 0 : maximal prefix doesnt match : Y is thus YP" << endl;
#endif
	      
	      // UPDATE STATE VALUES
	      Ystate_X  =      YPstate_X ;
	      Ystate_K  =      YPstate_K ;
	      // NOMERGE : CHECK FINAL STATE  
	      // ** not needed **
	      // UPDATE STATE INDEX 
	      Ystate_I  =      YPstate_I ;
	    }
	    
#ifdef BUILD
	    cout << "**V " << endl;
	    cout << "YPstate_X:" <<  YPstate_X << ",  YPstate_K:" << YPstate_K << endl;
	    cout << "Ystate_X:"  <<  Ystate_X  << ",  Ystate_K:"  << Ystate_K  << endl;
#endif
	  } else { // Xstate is empty ... and we have read 1^t.a  since ... 
	    
	    // (c) initial case when reading 1^t . a
	    //     three subcases can be deduced here :
	    //     - either there is one prefix matching on the full 1^t.a span ? in this case does such state exists ?
	    //     - either the maximal prefix does not 
	    //
#ifdef BUILD
	    cout << " FX on 1^"<< Xstate_T << "." << a << endl;
#endif
	    
	    // UPDATE STATE VALUES
	    Ystate_X   = FX[Xstate_T][a];
	    Ystate_K   = Fk[Xstate_T][a];
	    
	    // NOMERGE : CHECK FINAL STATE
	    if (nomerge && L[Ystate_K] >= motif_span) {
#ifdef ASSERTB
	      assert(YPstate_X ==  Ystate_X^(1 << (Ystate_K-1)));
#endif
	      Ystate_I = YPstate_I;
	    } else {
	      
	      // UPDATE INDEX
	      if (YPRstate_I >= 0 && SEEDPREFIXESMATCHINGCOST_X(YPRstate_I) == Ystate_X && SEEDPREFIXESMATCHINGCOST_T(YPRstate_I) == 0) { // if FX does gives a new prefix when compared with YP
		Ystate_I  = YPRstate_I;
		//cerr << "*";
	      } else {
		if (SEEDPREFIXESMATCHINGCOST_X(YPstate_I) == Ystate_X && SEEDPREFIXESMATCHINGCOST_T(YPstate_I) == 0) {
		  Ystate_I  = YPstate_I;
		  //cerr << "+";
		} else {
		  Y_not_there = true;
		}
	      }
	    }
#ifdef BUILD
	    cout << "*U "<< endl;
	    cout << "Ystate_X:"  <<  Ystate_X << ",  Ystate_K:" << Ystate_K << endl;
#endif
	  }// if ( Xstate_K > 0 )
	  
	  Ystate_T  = 0;
	  
	}// if (a == gv_alphabet_size - 1)
	
	int final = (Ystate_T + L[Ystate_K] >= motif_span - 1) ? TRUE : FALSE;
	
	if (!nomerge  && final) {
	  
	  Ystate_I = 0;
	  
	} else {
	  
	  // else check if the state has not been already created
	  if (Y_not_there) {
	    // create it and push it inside the list of states which need update.
	    Ystate_I = addNewState(final);
	    SEEDPREFIXESMATCHINGCOST_REVMAXXP(YPstate_I) = Ystate_I;
	    statesSeedPrefixesMatchingCostSet.push_back(SeedPrefixesMatchingCostSet(Ystate_X,Ystate_T,Ystate_K,YPstate_I,0,Ystate_cost));
	    statesNbRemaining.push(Ystate_I);
	    
#ifdef BUILD
	    cout << "$push I:"<< Ystate_I << " < X:" << hex << Ystate_X << " , t:" << dec << Ystate_T << " , k:" << Ystate_K << " > " << endl;
#endif
	  }
	}
#ifdef BUILD
	cout << "> add transition ( a:" << dec << a << ", q1:" << dec << Xstate_I << " , q2:" << Ystate_I << " ) " << endl;
#endif
	addNewTransition(a,Xstate_I,Ystate_I);

	statesSeedPrefixesMatchingCostSet[Ystate_I].Cost = MIN(Ystate_cost,statesSeedPrefixesMatchingCostSet[Ystate_I].Cost);


#ifdef ASSERTB
	assert(DETERMINISTIC(Xstate_I,a));
#endif
      }// YstateCost <= cost_threshold 
    }// for all letter in ALPHABET.
  }// while states remain. 
  
  /* Free unused data needed to build the automaton */
  L.clear();
  FX.clear();
  Fk.clear();
  EX.clear();
  Ek.clear();
  
  statesSeedPrefixesMatchingCostSet.clear();
  return 0;
}













/** @class SeedBuhlerSet
 *  @brief simple "inner class" to keep states properties 
 *         during SeedBuhler (Aho Corasick) method
 *  @see Automaton_SeedBuhler
 */

class SeedBuhlerSet {

 public:
  /// Aho-Corasick Fail function for this state
  int   Fail;
  /// Number of transition that need to be completed for this state
  short Rest;
  /// Level (distance from the root) in the AC-tree for this state
  short Level;
  /** @brief build a SeedBuhlerSet
   */
 SeedBuhlerSet(int Fail=0, short Rest=0, short Level=0) : Fail(Fail), Rest(Rest), Level(Level) {};
};

#define SEEDBUHLER_FAIL(s)  (statesSeedBuhlerSet[(s)].Fail)
#define SEEDBUHLER_REST(s)  (statesSeedBuhlerSet[(s)].Rest)
#define SEEDBUHLER_LEVEL(s) (statesSeedBuhlerSet[(s)].Level)




int automaton::Automaton_SeedBuhler(seed& s, vector< vector <int> > & matchingmatrix, bool nomerge) {
 
  /* Get the seed span */
  int   motif_span  = s.span();
  int * motif       = s.table();
  
#ifdef ASSERTB
  if (motif_span <= 0) {
    ERROR("Automaton_SeedBuhler","negative span");
  }
#endif
  
  /* (1) build the trie */

#ifdef USEQUEUE
  queue<int>  statesNbRemaining;
#else
  stack<int>  statesNbRemaining;
#endif
  // keep each state information (Fail,Rest,Level) inside this table
  vector<SeedBuhlerSet> statesSeedBuhlerSet(0);
  
  /* create a first state [0] and put it as the final one */
  int Finalstate_I = addNewState(TRUE);
  statesSeedBuhlerSet.push_back(SeedBuhlerSet(0,0,0));
  selfLoop(Finalstate_I);

  /* create a second state [1] : it will be the initial one */
  int Initstate_I = addNewState();
  statesSeedBuhlerSet.push_back(SeedBuhlerSet(0,0,0));
  statesNbRemaining.push(Initstate_I);
  
  while( !statesNbRemaining.empty() ) {
    
    // current state remaining
#ifdef USEQUEUE
    int stateN =  statesNbRemaining.front();
#else
    int stateN =  statesNbRemaining.top();
#endif

    statesNbRemaining.pop();
    int i        =  SEEDBUHLER_LEVEL(stateN);
    SEEDBUHLER_REST(stateN) = gv_align_alphabet_size; 
    int final    = (i >= motif_span - 1) ? TRUE : FALSE;
    int b = motif[i];
    
    for( int a=0 ; a<gv_align_alphabet_size ;a++) {
      if (MATCHES_AB(a,b)){
	int StateX   = 0; 
	if (i < motif_span - 1 || nomerge){
	  StateX = addNewState(final); // add new state
	  statesSeedBuhlerSet.push_back(SeedBuhlerSet(0,0,i+1));
	  statesNbRemaining.push(StateX);
	}
	addNewTransition(a,stateN,StateX);
	SEEDBUHLER_REST(stateN)--;
      }//matches
    }//for a
  }//while states remaining
  

  /* 
   * (2) add failure links 
   */
  
    
  /* init first states */ 
  int  r  = 1;
  SEEDBUHLER_FAIL(r) = r;
  for(int a=0; a<SEEDBUHLER_REST(r); a++){
    addNewTransition(a,r,SEEDBUHLER_FAIL(r));
#ifdef ASSERTB		
    assert(DETERMINISTIC(r,a));
#endif
  }
  
  for(int a=SEEDBUHLER_REST(r) ; a < gv_align_alphabet_size ; a++) {
#ifdef ASSERTB		
    assert(DETERMINISTIC(r,a));
#endif
    int p = _states[r].next[a].begin()->state;
    SEEDBUHLER_FAIL(p) = r;
  }
 
  /* init last state */
  SEEDBUHLER_FAIL(0) = 0;
  SEEDBUHLER_REST(0) = 0;

 
  /* intermediate states:  BORDER function from Aho-Corasick*/
  for(int r = 2 ; r < size(); r++){
    for(int a=0; a<SEEDBUHLER_REST(r); a++){
#ifdef ASSERTB		
      assert(DETERMINISTIC(SEEDBUHLER_FAIL(r),a));
#endif
      //if (!(_states[r].final)) /*FIXME || nomerge ??*/
      addNewTransition(a,r,_states[SEEDBUHLER_FAIL(r)].next[a].begin()->state);
#ifdef ASSERTB		
      assert(DETERMINISTIC(r,a));
#endif
    }
    
    for(int a=SEEDBUHLER_REST(r) ; a < gv_align_alphabet_size ; a++){ 
#ifdef ASSERTB			
      assert(DETERMINISTIC(r,a));
#endif
      int p = _states[r].next[a].begin()->state;
      int s = SEEDBUHLER_FAIL(r);
#ifdef ASSERTB			
      assert(DETERMINISTIC(s,a));
#endif
     SEEDBUHLER_FAIL(p) = _states[s].next[a].begin()->state;
    }
  }
  statesSeedBuhlerSet.clear();
  return 0;
}




/** @class SeedScoreSet
 *  @brief simple "inner class" to keep states properties
 *         during SeedScore method
 *  @see Automaton_SeedScore
 */

class SeedScoreSet {

 public:
  /// Aho-Corasick Fail function for this state
  int   Fail;
  /// Score reached at this state
  short Score;
  /// Level (distance from the root) in the AC-tree for this state
  short Level;
  /** @brief build a SeedScoreSet
   */
 SeedScoreSet(int Fail=0, short Score=0, short Level=0) : Fail(Fail), Score(Score), Level(Level) {};
};

#define SEEDSCORE_FAIL(s)  (statesSeedScoreSet[(s)].Fail)
#define SEEDSCORE_SCORE(s) (statesSeedScoreSet[(s)].Score)
#define SEEDSCORE_LEVEL(s) (statesSeedScoreSet[(s)].Level)


int automaton::Automaton_SeedScore(seed & s, vector< vector <int> > & matchingmatrix, vector< vector<int> > & scoringmatrix, int scoringthreehold, bool nomerge) {
 
  /* Get the seed span */
  int   motif_span  = s.span();
  int * motif       = s.table();

#ifdef ASSERTB
  if (motif_span <= 0) {
    ERROR("Automaton_SeedScore","negative span");
  }
#endif
  
  /* (0) compute the suffix best score (sbs) :
   *  it gives, for a suffix of length i the best suffix score
   *  reched : thus is prefix_score[i] + bbs[i] < scoringthreehold 
   * then it is not possible to reach the require scoringthreehold.
   */
  
  vector<int> sbs(motif_span,0);
  sbs[0] = 0;
  
  /* for all suffixes length */
  for(int i = 1 ; i < motif_span ; i++){
    int b  = motif[motif_span - i];
    int bs = -INT_INFINITY;
    /* FIXME : this part can be precomputed out of the function */
    for(int a = 0 ; a < gv_align_alphabet_size ; a++ ) {
      if (MATCHES_AB(a,b)){
	bs = MAX(bs,SCORES_AB(a,b));
      }
    }
    sbs[i] = sbs[i-1] + bs;
#ifdef BUILD
    fprintf(stderr," sbs[%d]=%d\n",i,sbs[i]);
#endif
  }
  


  /* (1) build the trie */

#ifdef USEQUEUE
  queue<int>  statesNbRemaining;
#else
  stack<int>  statesNbRemaining;
#endif
  
  // keep each state information (Fail,Rest,Level) inside this table
  vector<SeedScoreSet> statesSeedScoreSet(0);
  
  /* create a first state [0] and put it as the final one */
  int Finalstate_I = addNewState(TRUE);
  statesSeedScoreSet.push_back(SeedScoreSet(0,0,0));
  selfLoop(Finalstate_I);

  /* create a second state [1]  : it will be the initial one */
  int Initstate_I = addNewState();
  statesSeedScoreSet.push_back(SeedScoreSet(0,0,0));
  statesNbRemaining.push(Initstate_I);
  
  while( !statesNbRemaining.empty() ) {
    
    // current state remaining
#ifdef USEQUEUE
    int stateN =  statesNbRemaining.front();
#else
    int stateN =  statesNbRemaining.top();
#endif
    statesNbRemaining.pop();
    int i        = SEEDSCORE_LEVEL(stateN);
    int final    = (i >= motif_span - 1) ? TRUE : FALSE;

    int b = motif[i];
    
    for( int a=0 ; a<gv_align_alphabet_size; a++) {
      if (MATCHES_AB(a,b)){
	if (SEEDSCORE_SCORE(stateN) + SCORES_AB(a,b) + sbs[motif_span - (i+1)] >= scoringthreehold){
	  int StateX = 0;
	  if ((!nomerge && (i+1) < motif_span) || (nomerge && i < motif_span)){
	    StateX = addNewState(final);
	    statesSeedScoreSet.push_back(SeedScoreSet(0,SEEDSCORE_SCORE(stateN) + SCORES_AB(a,b),i+1));
	    statesNbRemaining.push(StateX);
	  }
	  addNewTransition(a,stateN,StateX);
#ifdef ASSERTB		
	  assert(DETERMINISTIC(stateN,a));
#endif
	}
      }//prefix penalty
    }//for a
  }//while states remaining
  
  sbs.clear();

  /* 
   * (2) add failure links 
   */


    
  /* init first states */ 
  int  r  = 1;
  SEEDSCORE_FAIL(r) = r;
  for(int a = 0; a < gv_align_alphabet_size; a++ ) {
    if (!hasTransition(a,r)) {
      addNewTransition(a,r,SEEDSCORE_FAIL(r));
    } else {
      int p = _states[r].next[a].begin()->state;
      SEEDSCORE_FAIL(p) = r;
    }
#ifdef ASSERTB		
    assert(DETERMINISTIC(r,a));
#endif
  }
  
  /* init last state */
  SEEDSCORE_FAIL(0) = 0;


  /* intermediate states:  BORDER function from Aho-Corasick*/
  for(int r = 2 ; r < size(); r++){
    for(int a = 0 ; a < gv_align_alphabet_size; a++){
      if (!hasTransition(a,r)) {
#ifdef ASSERTB
	assert(DETERMINISTIC(SEEDSCORE_FAIL(r),a));
#endif	
	//if (!(_states[r].final)) //FIXME /*FIXME || nomerge ??*/
	  addNewTransition(a,r,_states[SEEDSCORE_FAIL(r)].next[a].begin()->state);
#ifdef ASSERTB		
	assert(DETERMINISTIC(r,a));
#endif
     
      } else {
	
#ifdef ASSERTB			
	assert(DETERMINISTIC(r,a));
#endif
	int p = _states[r].next[a].begin()->state;
	int s = SEEDSCORE_FAIL(r);
#ifdef ASSERTB			
	assert(DETERMINISTIC(s,a));
#endif
	SEEDSCORE_FAIL(p) = _states[s].next[a].begin()->state;
      }
    }
  }
  statesSeedScoreSet.clear();
  return 0;
}










/** @class SeedScoreCostSet
 *  @brief simple "inner class" to keep states properties
 *         during SeedScoreCost method
 *  @see Automaton_SeedScore
 */

class SeedScoreCostSet {

 public:
  /// Aho-Corasick Fail function for this state
  int   Fail;
  /// Score reached at this state
  short Score;
  /// Level (distance from the root) in the AC-tree for this state
  short Level;
  /// gives the Cost at this state
  int   Cost;
  /** @brief build a SeedScoreCostSet
   */
 SeedScoreCostSet(int Fail=0, short Score=0, short Level=0,int Cost=0) : Fail(Fail), Score(Score), Level(Level), Cost(Cost) {};
};

#define SEEDSCORECOST_FAIL(s)  (statesSeedScoreCostSet[(s)].Fail)
#define SEEDSCORECOST_SCORE(s) (statesSeedScoreCostSet[(s)].Score)
#define SEEDSCORECOST_LEVEL(s) (statesSeedScoreCostSet[(s)].Level)
#define SEEDSCORECOST_COST(s)  (statesSeedScoreCostSet[(s)].Cost)

int automaton::Automaton_SeedScoreCost(seed & s, vector< vector <int> > & matchingmatrix, vector< vector<int> > & scoringmatrix, int scoringthreehold, bool nomerge, vector<int> & costs, int cost_threshold) {

  /* Get the seed span */
  int   motif_span  = s.span();
  int * motif       = s.table();

#ifdef ASSERTB
  if (motif_span <= 0) {
    ERROR("Automaton_SeedScoreCost","negative span");
  }
#endif
  
  /* (0) compute the suffix best score (sbs) :
   *  it gives, for a suffix of length i the best suffix score
   *  reched : thus is prefix_score[i] + bbs[i] < scoringthreehold 
   * then it is not possible to reach the require scoringthreehold.
   */
  
  vector<int> sbs(motif_span,0);
  sbs[0] = 0;
  
  /* for all suffixes length */
  for(int i = 1 ; i < motif_span ; i++){
    int b  = motif[motif_span - i];
    int bs = -INT_INFINITY;
    /* FIXME : this part can be precomputed out of the function */
    for(int a = 0 ; a < gv_align_alphabet_size ; a++ ) {
      if (MATCHES_AB(a,b)){
	bs = MAX(bs,SCORES_AB(a,b));
      }
    }
    sbs[i] = sbs[i-1] + bs;
#ifdef BUILD
    fprintf(stderr," sbs[%d]=%d\n",i,sbs[i]);
#endif
  }
  


  /* (1) build the trie */

#ifdef USEQUEUE
  queue<int>  statesNbRemaining;
#else
  stack<int>  statesNbRemaining;
#endif
  
  // keep each state information (Fail,Rest,Level) inside this table
  vector<SeedScoreCostSet> statesSeedScoreCostSet(0);
  
  /* create a first state [0] and put it as the final one */
  int Finalstate_I = addNewState(TRUE);
  statesSeedScoreCostSet.push_back(SeedScoreCostSet(0,0,0,0));
  selfLoop(Finalstate_I);

  /* create a second state [1]  : it will be the initial one */
  int Initstate_I = addNewState();
  statesSeedScoreCostSet.push_back(SeedScoreCostSet(0,0,0,0));
  statesNbRemaining.push(Initstate_I);

  /* bag reject set [2] */
  int RejectBagstate_I   = addNewState();
  statesSeedScoreCostSet.push_back(SeedScoreCostSet(0,0,0,INT_INFINITY));
  selfLoop(RejectBagstate_I);

  
  while( !statesNbRemaining.empty() ) {
    
    // current state remaining
#ifdef USEQUEUE
    int stateN =  statesNbRemaining.front();
#else
    int stateN =  statesNbRemaining.top();
#endif
    statesNbRemaining.pop();
    int i        = SEEDSCORECOST_LEVEL(stateN);
    int final    = (i >= motif_span - 1) ? TRUE : FALSE;

    int b = motif[i];
    
    for( int a=0 ; a<gv_align_alphabet_size; a++) {
      int cost  = SEEDSCORECOST_COST(stateN) + costs[a];
      if (MATCHES_AB(a,b)){
	if (cost > cost_threshold) {
	  addNewTransition(a,stateN,RejectBagstate_I);
	} else {
	  if (SEEDSCORECOST_SCORE(stateN) + SCORES_AB(a,b) + sbs[motif_span - (i+1)] >= scoringthreehold){
	    int StateX = Finalstate_I; //FIXME
	    if ((!nomerge && (i+1) < motif_span) || (nomerge && i < motif_span)){
	      StateX = addNewState(final);
	      statesSeedScoreCostSet.push_back(SeedScoreCostSet(0,SEEDSCORECOST_SCORE(stateN) + SCORES_AB(a,b),i+1,cost));
	      statesNbRemaining.push(StateX);
	    }
	    addNewTransition(a,stateN,StateX);
#ifdef ASSERTB		
	    assert(DETERMINISTIC(stateN,a));
#endif
	  }
	}// costs[a] <= cost_threshold
      }//prefix penalty
    }//for a
  }//while states remaining
  
  sbs.clear();

  /* 
   * (2) add failure links 
   */

    
  /* init first states */ 
  int  r  = 1;
  SEEDSCORECOST_FAIL(r) = r;
  for(int a = 0; a < gv_align_alphabet_size; a++ ) {
    if (!hasTransition(a,r)) {
      addNewTransition(a,r,SEEDSCORECOST_FAIL(r));
    } else {
      int p = _states[r].next[a].begin()->state;
      SEEDSCORECOST_FAIL(p) = r;
    }
#ifdef ASSERTB		
    assert(DETERMINISTIC(r,a));
#endif
  }
  
  /* init last state */
  SEEDSCORECOST_FAIL(0) = 0;


  /* intermediate states:  BORDER function from Aho-Corasick*/
  for(int r = 2 ; r < size(); r++){
    for(int a = 0 ; a < gv_align_alphabet_size; a++){
      if (!hasTransition(a,r)) {
#ifdef ASSERTB
	assert(DETERMINISTIC(SEEDSCORECOST_FAIL(r),a));
#endif	
	//if (!(_states[r].final)) //FIXME /*FIXME || nomerge ??*/	
	int cost = SEEDSCORECOST_COST(r) + costs[a];
	if (cost > cost_threshold)
	  addNewTransition(a,r,RejectBagstate_I);
	else
	  addNewTransition(a,r,_states[SEEDSCORECOST_FAIL(r)].next[a].begin()->state);
#ifdef ASSERTB		
	assert(DETERMINISTIC(r,a));
#endif
     
      } else {
	
#ifdef ASSERTB			
	assert(DETERMINISTIC(r,a));
#endif
	int p = _states[r].next[a].begin()->state;
	int s = SEEDSCORECOST_FAIL(r);
#ifdef ASSERTB			
	assert(DETERMINISTIC(s,a));
#endif
	SEEDSCORECOST_FAIL(p) = _states[s].next[a].begin()->state;
      }
    }
  }
  statesSeedScoreCostSet.clear();
  return 0;
}


















/*
 * Gives the probability to reach a final state
 * after nbSteps 
 * (64 is the common value used by everyone, i'm a sheep so i use 64)
 *
 *  ^   ^
 *   + +
 *  =(o)=
 *
 */


double automaton::PrFinal(int nbSteps) {

  vector<double> v0(size(),0);
  vector<double> v1(size(),0);

  /* set initial state probability to 1.0 */
  v1[1] = 1.00;

  /* compute probability at step i provided probabilities at step i-1 */
  for(int i= 0 ; i < nbSteps ; i++) {
    if (i&1){
      for(int nbstate=0 ; nbstate < (int)_states.size() ; nbstate++) {

	double val =  0.00;
	for(int a=0 ; a<gv_align_alphabet_size ; a++) {
	  slist<transition>   & list_tran      = _states[nbstate].prev[a];
	  for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
	    val += (list_iter->prob) * v0[(list_iter->state)];
	  }
	}
	v1[nbstate] = val;
      }

    } else {
      
      for(int nbstate=0 ; nbstate < (int)_states.size() ; nbstate++) {
	double val =  0.00;
	for(int a=0 ; a<gv_align_alphabet_size ; a++) {
	  slist<transition>   & list_tran      = _states[nbstate].prev[a];
	  for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
	    val += (list_iter->prob) * v1[(list_iter->state)];
	  }
	}
	v0[nbstate] = val;
      }
    }
  }

  // sum final states probabilities
  double result = 0;
  if (nbSteps & 1) {
    for (int i=0;i<(int)_states.size();i++)
      if (_states[i].final)
	result += v0[i];
  } else {
    for (int i=0;i<(int)_states.size();i++)
      if (_states[i].final)
	result += v1[i];
  }
  v0.clear();
  v1.clear();

  return result;
}


bool  automaton::Lossless(int nbSteps, vector<int> costs, int cost_threshold, int Nocc) {
  vector<int> mincost0(size(),INT_INFINITY/2);
  vector<int> finalcount0(size(),INT_INFINITY/2);

  vector<int> mincost1(size(),INT_INFINITY/2);
  vector<int> finalcount1(size(),INT_INFINITY/2);

  /* set initial counts */
  mincost1[1] = 0;
  finalcount1[1] = 0;

  for(int i= 0 ; i < nbSteps ; i++) {
    if (i&1){

      for(int nbstate=0 ; nbstate < (int)_states.size() ; nbstate++) {
	int mincost    =  INT_INFINITY/2;
	int finalcount =  INT_INFINITY/2;
	for(int a=0 ; a<gv_align_alphabet_size ; a++) {
	  slist<transition>   & list_tran      = _states[nbstate].prev[a];
	  for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
	    int cost       = costs[a] + mincost0[(list_iter->state)];
	    if (cost <= cost_threshold) {
	      mincost    = MIN(mincost,cost);
	      finalcount = MIN(finalcount, _states[nbstate].final + finalcount0[(list_iter->state)]);
	    }
	  }
	}
	mincost1[nbstate] = mincost;
	finalcount1[nbstate] = finalcount;
      }
    } else {
      
      for(int nbstate=0 ; nbstate < (int)_states.size() ; nbstate++) {
	int mincost    =  INT_INFINITY/2;
	int finalcount =  INT_INFINITY/2;
	for(int a=0 ; a<gv_align_alphabet_size ; a++) {
	  slist<transition>   & list_tran      = _states[nbstate].prev[a];
	  for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
	     int cost       = costs[a] + mincost1[(list_iter->state)];
	    if (cost <= cost_threshold) {
	      mincost    = MIN(mincost,cost);
	      finalcount = MIN(finalcount, _states[nbstate].final + finalcount1[(list_iter->state)]);
	    }
	  }
	}
	mincost0[nbstate] = mincost;
	finalcount0[nbstate] = finalcount;
      }
    }
  }

  // sum final states probabilities
  bool    bNocc  = true;
  if (nbSteps & 1) {
    for (int i=0; bNocc && (i<(int)_states.size());i++)
      if (mincost0[i] <= cost_threshold) {
	bNocc = (finalcount0[i] >= Nocc);
      }
  } else {
    for (int i=0; bNocc && (i<(int)_states.size());i++)
      if (mincost1[i] <= cost_threshold) {
	bNocc = (finalcount1[i] >= Nocc);
      }
  }
  
  mincost0.clear();
  mincost1.clear();
  finalcount0.clear();
  finalcount1.clear();

#ifdef BUILD
  cerr << "Lossless : " << bNocc << endl;
#endif

  return bNocc;
}



double automaton::PrLossless(int nbSteps, vector<int> costs, int cost_threshold) {

  vector< vector<double> > v0 = vector< vector<double> > (size(), vector<double>(cost_threshold+2,0.0));
  vector< vector<double> > v1 = vector< vector<double> > (size(), vector<double>(cost_threshold+2,0.0));
  
  /* set initial state probability to 1.0 */
  v1[1][0] = 1.0;
  
  /* compute probability at step i provided probabilities at step i-1 */
  for(int i= 0 ; i < nbSteps ; i++) {
    if (i&1){
      for(int nbstate=0 ; nbstate < (int)_states.size() ; nbstate++) {
	for (int k=0;k<=cost_threshold;k++)
	  v1[nbstate][k] = 0.0;
	for (int kp=0;kp<=cost_threshold+1;kp++) {
	  for(int a=0 ; a<gv_align_alphabet_size ; a++) {
	    int k  = kp + costs[a];
	    slist<transition>   & list_tran      = _states[nbstate].prev[a];
	    for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++)
	      v1[nbstate][MIN(k,cost_threshold+1)] +=  (list_iter->prob) * v0[(list_iter->state)][kp];
	  }
	}
      }
    } else {
      for(int nbstate=0 ; nbstate < (int)_states.size() ; nbstate++) {
	for (int k=0;k<=cost_threshold;k++)
	  v0[nbstate][k] = 0.0;
	for (int kp=0;kp<=cost_threshold+1;kp++) {
	  for(int a=0 ; a<gv_align_alphabet_size ; a++) {
	    int k  = kp + costs[a];
	    slist<transition>   & list_tran      = _states[nbstate].prev[a];
	    for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++)
	      v0[nbstate][MIN(k,cost_threshold+1)] +=  (list_iter->prob) * v1[(list_iter->state)][kp];
	  }
	}
      }
    }
  }

  // sum final states probabilities
  double  PrFinalLessThanThreshold = 0.0;
  double  PrNonFinalLessThanThreshold = 0.0;
  double  PrFullLessThanThreshold = 0.0;
  if (nbSteps & 1) {
    for (int i=0;i<(int)size();i++) {
      for(int k = 0; k <= cost_threshold; k++)
	PrFullLessThanThreshold += v0[i][k];      
      if(_states[i].final)
	for(int k = 0; k <= cost_threshold; k++)	    
	  PrFinalLessThanThreshold += v0[i][k];
      else
	for(int k = 0; k <= cost_threshold; k++)	    
	  PrNonFinalLessThanThreshold += v0[i][k];
    }
  } else {
    for (int i=0;i<(int)size();i++) {
      for(int k = 0; k <= cost_threshold; k++)	    
	PrFullLessThanThreshold += v1[i][k];
      if(_states[i].final)
	for(int k = 0; k <= cost_threshold; k++)	    
	  PrFinalLessThanThreshold += v1[i][k];
      else
	for(int k = 0; k <= cost_threshold; k++)	    
	  PrNonFinalLessThanThreshold += v1[i][k];
    }
  }
  v0.clear();
  v1.clear();

#ifdef BUILD
  cerr << "PrLossless : " <<  PrFinalLessThanThreshold << "/" << PrFullLessThanThreshold << "=" << (PrFinalLessThanThreshold / PrFullLessThanThreshold) <<endl;
#endif
  
  return PrFinalLessThanThreshold / PrFullLessThanThreshold;
}





void automaton::GenSeq(int len) {
  int q = 1;
  for (int l=0;l<len;l++) {
    double p = (double)rand()/(double)RAND_MAX;
    int q_new=0,a=0;
    double psum = 0;
    for(a=0; a < gv_align_alphabet_size ; a++){
      slist<transition> & list_tran = this->_states[q].next[a];
      for(slist<transition>::iterator list_iter = list_tran.begin() ; list_iter != list_tran.end() ; list_iter++) {
	double pr = list_iter->prob;
	q_new = list_iter->state;
	psum +=  pr;
	if (psum >= p)
	  goto eol;
      }
    }
  eol:
    q = q_new;
    cout << a;
  }
  cout << endl;
}

