#!/bin/sh
#
# Test the nef -d example in Section 6.4.24 of the PALP manual
#
. tests/lib/run-test.sh

: ${DIM:=6}

# Page 52
COMMAND="./nef-${DIM}d.x -N -d2 tests/input/6.4.24-nef-d.txt"
DESCRIPTION="nef-${DIM}d.x -N -d2 example on page 52"
EXPECTED=$(cat<<-EOF
M:300 18 N:9 8  codim=2 #part=15
7 63 Points of dual PG: (nv=27)
   1   0   1   0   0   1   1   1   0   0   0   0   0   1   1   0   0   0   0   0   1   0   0   0   0   1   0   1   1   1   1   1   1   1   1   1   1   0   1   0   0   0   0   1   1   1   1   1   1   0   0   0   1   1   1   1   1   1   1   1   1   0   1
   0   1   0   1   1   0   0   0   1   1   1   1   1   0   0   1   1   1   1   1   0   1   1   1   1   0   1   0   0   0   0   0   0   0   0   0   0   1   0   1   1   1   1   0   0   0   0   0   0   1   1   1   0   0   0   0   0   0   0   0   0   1   0
  -1   0   1   1   0  -1  -1  -1   0   0   0   1   1   1   1   1   1   0   0   0  -1   0   0   0   0  -1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0  -1   1  -1   1   1  -1   1   1   1  -1  -1  -1  -1  -1  -1  -1  -1  -1   0   0
  -1   1  -1   0   1   1   1  -1   1   1   0   0   0  -1  -1   0   0   1   1   0  -1   0   0   0   0   1   0  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   1   1   1   0  -1   1  -1  -1  -1   0   0   0   1  -1  -1   1   0   0   0   0   0   0   0
   0   1   0   1  -1   0   0   0   1   1   1  -1  -1   0   0   1   1  -1  -1  -1   0   1   1  -1  -1   0  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0
  -1   0  -1   0   0   1  -1  -1   1   0   0   1   0   1  -1   1   0   1   0   0   1   1   0   1   0  -1   0  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1   1   0   0   1   1  -1   0   0   0   0   0   0   1  -1  -1   0   0   0  -1  -1   0  -1   0   0
  -1   1   1   1   1  -1  -1   1   0   0   1   0   0  -1  -1   0   0   0   0   1  -1   0   0   0   0   1   1  -1   0   1  -1   0  -1  -1   0   1  -1   1  -1   0   0   1   0  -1   0   0  -1   0   0   0   1   0   0   0  -1  -1   0   1   0  -1  -1   0   0
7 72 Points of dual PG: (nv=36)
   0   0   0   1   0   0   0   1   0   1   1   0   0   0   0   0   1   1   0   0   0   0   0   1   1   1   1   1   0   1   1   1   1   1   1   1   1   0   0   1   0   1   1   1   0   1   0   0   1   1   1   1   0   1   0   0   0   1   0   1   0   0   1   1   0   1   0   0   1   0   0   1
   1   1   1   0   1   1   1   0   1   0   0   1   1   1   1   1   0   0   1   1   1   1   1   0   0   0   0   0   1   0   0   0   0   0   0   0   0   1   1   0   1   0   0   0   1   0   1   1   0   0   0   0   1   0   1   1   1   0   1   0   1   1   0   0   1   0   1   1   0   1   1   0
   0   0   1  -1   0   0   0  -1   0   1   1   1   1   1   1   1  -1  -1   0   0   0   0   0  -1  -1  -1  -1  -1   0   1  -1  -1  -1   1   1   1  -1   0   0  -1   0   0  -1   0   1   0   1   1   0  -1   0   0   0   0   0   0   0   0   0  -1   0   0   0   0   1   0   1   1  -1   0   0   0
   0   1   0   1   1   1   1  -1   0  -1  -1   0   0   0   0   0   1   1   1   1   0   0   0  -1  -1  -1  -1  -1   0  -1   1   1   1  -1  -1  -1   0   1   1   0   1  -1   0   0   0   0   0   0   0   0   0  -1   0   0   0   0   0  -1   1   0   1   1  -1  -1   0  -1   0   0   0   0   0   0
   0   1   1   0   0   1   1   0   1  -1  -1   0   0   0   1   1  -1  -1   0   0   0   1   1  -1   0   0  -1  -1   0   0  -1   0   0  -1   0   0  -1   0   0   0   0   0   0   0   0  -1   0   0  -1  -1   0   0   1  -1   1   1   0  -1   1   0   1   1  -1   0   1  -1   1   1  -1   0   0   0
   0   0   0  -1   0   2   0  -1   0   0  -1   2   0   0   2   0   0  -1   2   0   0   2   0  -1   0  -1   0  -1   2  -1  -1   0  -1  -1   0  -1   0   1   1  -1   0  -1  -1  -1   0  -1   1   1   0  -1  -1   0   0  -1   1   1   1  -1   0   0   1   1  -1  -1   0   0   1   1  -1   1   0   0
  -1   1   1   1   1  -1  -1   1   1   0   0  -1  -1   1  -1  -1   0   0  -1  -1   1  -1  -1   1   0   0   0   0  -1   1   1   0   0   1   0   0   0  -1   0   0   0   0   1   1   0   1  -1   0   0   1   0   0   0   0  -1   0   0   1   0   0  -1   0   0   1   0   0  -1   0   0  -1   0   0
7 90 Points of dual PG: (nv=35)
   1   0   1   0   1   1   1   0   0   0   0   1   1   1   1   1   1   0   1   0   1   0   1   0   0   0   0   1   1   1   1   0   0   0   0   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1
   0   1   0   1   0   0   0   1   1   1   1   0   0   0   0   0   0   1   0   1   0   1   0   1   1   1   1   0   0   0   0   1   1   1   1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0
  -1   0   1   1  -1  -1  -1   0   0   0   1   1   1  -1   1  -1  -1   0  -1   0  -1   0  -1   1   0   1   0   1   1  -1  -1   0   0   0   1   1  -1  -1  -1  -1  -1  -1   0  -1   0  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   1  -1  -1   0   0   1  -1   1   1   0   1   0   0  -1  -1  -1  -1   0  -1  -1  -1   1  -1   0   0  -1  -1   0  -1   0
  -1   1  -1   0  -1   1   1   1   0   0   0  -1  -1   1  -1  -1  -1   0  -1   1   1   1   1   0   0   0   0  -1  -1  -1   1   0   1   1   0   0   0   0   0   0   0   0   0   1   0   1   1   1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0  -1   0  -1   0  -1  -1   1  -1  -1  -1  -1  -1  -1   0   0  -1  -1  -1  -1   0  -1  -1   1   0  -1  -1   0   0   0   0
  -1   1   0   1   0   0   0   0   1   1   0   0   0  -1  -1   0   0   0  -1   0  -1   0  -1   0   0   0   0  -1  -1  -1   0   1   1   1   1   1   0   0   0   0   0   0  -1   0  -1   0   0  -1   0   0   0   0   0   0   0   0   0   0  -1   0  -1  -1  -1  -1  -1   0  -1   0   0  -1  -1  -1  -1  -1  -1  -1  -1  -1   0  -1   0  -1  -1  -1  -1   0  -1  -1  -1   0
  -1   0  -1   0  -1   1  -1   0   1   0   0   1  -1  -1  -1   1  -1   0  -1   1   1   0  -1   1   1   0   0   1  -1   1  -1   0   1   0   1   0  -1  -1  -1   0   0   1   1  -1  -1   0   0   0  -1  -1  -1   0   0   1  -1  -1  -1   0  -1   1   0  -1   0   0   0  -1  -1   0   0  -1   0  -1   1   0  -1  -1   0  -1  -1   0   0  -1   0  -1   0   0  -1   0   1   0
  -1   1   1   1   1  -1  -1   1   0   0   1  -1  -1   1   1  -1  -1   1   1   0  -1   0  -1   0   0   0   0  -1  -1  -1   1   1   0   0   0   0  -1   0   1  -1   0  -1  -1   0   0  -1   0   0  -1   0   1  -1   0  -1  -1   0   1  -1   1  -1  -1   0   0   0  -1   0   0  -1   0   1   0   0  -1  -1   1   0  -1  -1   0   0  -1   0  -1  -1   0   0  -1  -1  -1   0
7 72 Points of dual PG: (nv=27)
   0   0   0   1   1   0   0   1   1   0   1   1   1   1   0   0   1   1   1   1   0   1   1   1   1   1   1   1   1   1   1   0   0   1   1   0   1   0   0   1   1   1   1   1   0   1   0   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   1
   1   1   1   0   0   1   1   0   0   1   0   0   0   0   1   1   0   0   0   0   1   0   0   0   0   0   0   0   0   0   0   1   1   0   0   1   0   1   1   0   0   0   0   0   1   0   1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   0
   0   0   1  -1  -1   0   0  -1  -1   0   1   1   1   1   1   1  -1  -1  -1  -1   0  -1  -1  -1  -1   1   1   0   0  -1   0   0   0  -1   0   0  -1   0   0   0   1   1   1  -1   1  -1   1   1  -1  -1   0  -1  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0
   0   1   0   1   1   1   1  -1  -1   0  -1  -1  -1  -1   0   0   1   1   1   1   0  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0  -1   1   0   1   1  -1  -1  -1  -1  -1   0   0   0   0   0   1  -1   1   0   0   0   0  -1   1  -1  -1  -1  -1  -1   0   0   0  -1  -1   0   0   0   0
   0   0   0   1  -1   0   0   1  -1   0   1   1  -1  -1   0   0   1   1  -1  -1   0   1   1  -1  -1   1  -1   0   0   0  -1   0   0   0  -1   0  -1   0   0  -1   0   0   0   0   0   0   0   0  -1   0  -1   0  -1   1   1   1   0   0   0   0   1   1   1   1   1   1   0   0  -1  -1   0   0
   0   0   0  -1  -1   2   0  -1  -1   0   0  -1   0  -1   2   0   0  -1   0  -1   2   0  -1   0  -1  -1  -1  -1  -1  -1  -1   1   1   0  -1   0   0   1   1  -1  -1  -1   0  -1   0  -1   1   1  -1  -1   0  -1  -1  -1  -1   0   0   0  -1  -1  -1  -1   0  -1  -1   0   0  -1   0  -1   0   0
  -1   1   1   1   1  -1  -1   1   1   1   0   0   0   0  -1  -1   0   0   0   0  -1   0   0   0   0   1   1   0   1   1   1  -1   0   0   1   0   0  -1   0   0   0   1   0   0   0   0  -1   0   0   1   0   0   1   0   1   0   0   0   1   0   0   1   0   0   1   0   0   1   0   0   0   0
7 117 Points of dual PG: (nv=26)
   1   0   1   0   1   1   1   1   0   0   1   1   1   1   1   1   1   1   1   1   1   1   0   0   0   0   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1
   0   1   0   1   0   0   0   0   1   1   0   0   0   0   0   0   0   0   0   0   0   0   1   1   1   1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0
  -1   0   1   1  -1  -1  -1  -1   0   0  -1  -1  -1   1   1   1   1   1  -1  -1  -1  -1   0   0   0   1   1   0  -1  -1   0  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0  -1   0   0   1   1   1   1   1   1   1   1  -1   1  -1  -1   0  -1  -1   0  -1  -1  -1  -1  -1  -1  -1  -1  -1   0  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   1   0   1   1   0  -1   0  -1   0
  -1   1  -1   0  -1   1   1   1   0   0  -1   1   1  -1  -1  -1  -1  -1  -1  -1  -1   1   0   1   1   0   0   0   0   0   0  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   1   1   1   1   1   1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0  -1  -1   0  -1  -1  -1  -1  -1  -1  -1  -1   1  -1   0  -1  -1  -1  -1   0   0   0   0   0   0   0   0   1   1   0   1   1   1  -1  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0  -1  -1  -1  -1  -1  -1   0  -1   0   0
  -1   0   1   0   1   1   1  -1   0   0  -1  -1  -1   1   1  -1  -1  -1   1   1  -1   1   0   0   0   0   0  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0  -1   0  -1  -1  -1  -1   0   0   0   0   0   0  -1  -1  -1  -1   1  -1   1   1  -1  -1   1   1   1   1   1   1  -1   1  -1   1   1  -1  -1  -1   1   1   1   1   1   1   1   1   1   1   1   1  -1   1  -1   1   1  -1  -1  -1  -1   0
  -1   0  -1   0  -1   1  -1  -1   1   0  -1   1  -1   1  -1  -1   1  -1   1  -1   1  -1   0   1   0   1   0  -1   1  -1   0  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0  -1   1   0   1   0  -1  -1  -1  -1   0   0   1   0  -1   0   0  -1  -1   0   0  -1  -1  -1  -1  -1   0   0   1   0  -1   1   0   0   0  -1   0  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1   0  -1  -1   0   0   0  -1  -1   0   0
  -1   1   1   1   1  -1  -1   1   0   0   1  -1  -1  -1  -1   1  -1  -1  -1  -1  -1   1   1   0   0   0   0  -1  -1   0   0  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1   0  -1  -1   0  -1   0   1  -1   0  -1   0   0  -1  -1   0   0  -1   0   1   1  -1   0   1  -1   0  -1   0   0  -1  -1   0  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1   0   0   1  -1   0  -1  -1  -1   0   0
7 48 Points of dual PG: (nv=15)
   0   0   1   1   1   0   0   0   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   0   0   0   0   1   1   1   1   1   1   1   1   1   1   0   1   0   0   0   0   0   1   0   1
   1   1   0   0   0   1   1   1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   1   1   1   1   0   0   0   0   0   0   0   0   0   0   1   0   1   1   1   1   1   0   1   0
   0   0  -1   2  -1   0   0   0   0  -1  -1   2   2  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   1   1   1   1   1   1  -1   0  -1   0   0   0   0   0  -1   0   0
   0   0   2  -1  -1   0   0   0   0   2   2  -1  -1  -1  -1   0   1   0  -1  -1  -1   0   0   0   0   0   0   0   1   1   1  -1  -1  -1   0   0   0   1   0   0   0   0   0   0   0   1   0   0
  -1   1   0   0   0   1  -1   1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0  -1   0  -1   1  -1   1   1   0   0   0
   0   0  -1  -1  -1   2   0   0   2   0  -1   0  -1   0  -1  -1   0  -1  -1  -1   0  -1  -1   0   0   1   1   2  -1  -1   0  -1  -1   0  -1  -1   0  -1   1   0   1   0   0   1   1  -1   0   0
  -1   1   1   1   1  -1   1  -1  -1   0   0   0   0   0   0   1   0   0   0   1   0   0   1  -1   1  -1   0  -1   0   1   0   0   1   0   0   1   0   0  -1   0   0   0   0  -1   0   1   0   0
7 69 Points of dual PG: (nv=15)
   1   0   1   1   0   1   1   1   1   1   1   0   0   0   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   1   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   1
   0   1   0   0   1   0   0   0   0   0   0   1   1   1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   0   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   0
  -1   0   2  -1   0  -1  -1  -1   2   2  -1   0   0   0   0  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   1   1   1   1   1   1   1   1   1   1   1  -1   2  -1   2   2  -1  -1  -1  -1   0   0
  -1   0  -1   2   0   2  -1   2  -1  -1  -1   0   0   0   0   1   1   1   1   1   1  -1   2   0   2   2  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   1   1   1   1   1   1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0  -1   0  -1  -1   0   0  -1   0   0   0
   0   1   0   0  -1   0   0   0   0   0   0   1   1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0
  -1   0  -1  -1   0   1  -1  -1   1  -1   1   1   0   1   0  -1  -1  -1   0   0   1   0  -1   1   0   0   0  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1   0   0   0  -1   0  -1  -1   0   0
  -1   1   1   1   1  -1   1  -1  -1  -1  -1   0   0   0   0  -1   0   1  -1   0  -1   0   0  -1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   1  -1   0  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   0  -1   0   0  -1   0   1   0   0
7 40 Points of dual PG: (nv=12)
   0   0   1   1   1   0   0   0   1   1   1   0   1   1   1   1   0   0   1   0   0   0   0   1   1   1   1   1   1   0   0   1   0   0   0   0   0   1   0   1
   1   1   0   0   0   1   1   1   0   0   0   1   0   0   0   0   1   1   0   1   1   1   1   0   0   0   0   0   0   1   1   0   1   1   1   1   1   0   1   0
   0   0  -1   2  -1   0   0   0  -1   2  -1   0  -1   1   0  -1   0   0   1   0   0   0   0  -1   0   1   1  -1   0   0   0   0   0   0   0   0   0   0   0   0
   0   0   2  -1  -1   0   0   0   2  -1  -1   0   1   0  -1   1   0   0  -1   0   0   0   0   0   1  -1   0   0   1   0   0  -1   0   0   0   0   0   0   0   0
   0   1   0   0   0   1   1   0  -1  -1  -1   0   0  -1   0  -1   0   0  -1   0   0   0   0   0   0   0   0  -1  -1   1   1  -1   1   1   1   1   1  -1   0   0
  -1  -1   0   0   0   2  -1  -1   0   0   0   2   0   0   0   0  -1  -1   0   0   0   1   1   0   0   0   0   0   0  -1  -1   0   0   0   0   1   1   0   0   0
  -1   2   0   0   0  -1  -1   2   0   0   0  -1   0   0   0   0   0   1   0  -1   1  -1   0   0   0   0   0   0   0   0   1   0  -1   0   1  -1   0   0   0   0
7 72 Points of dual PG: (nv=24)
   0   0   1   1   1   0   0   0   1   1   1   1   1   1   1   1   1   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   1   0   0   1   1   0   1
   1   1   0   0   0   1   1   1   0   0   0   0   0   0   0   0   0   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   0   1   1   0   0   1   0
   0   0  -1   2  -1   0   0   0  -1  -1  -1   2   2   2  -1  -1  -1   0  -1   2  -1  -1   2  -1   0  -1   0   0   0   1  -1  -1  -1  -1  -1  -1   0  -1   1   0   0   0   0   0  -1   0   0   0   1   0   0   0   1   1   1   1   1   1   0   1   1  -1   1  -1   0  -1   0   0   0   0   0   0
   0   0   2  -1  -1   0   0   0   2   2   2  -1  -1  -1  -1  -1  -1   0   2  -1  -1   2  -1  -1   0   0   0   1  -1  -1   0   0   0   1   1   1  -1   1   0  -1  -1  -1   0   0   1  -1   0   0  -1   1   1   1  -1  -1  -1   0   0   0   0  -1   0   0   0   0   0   1   0   0   1   1   0   0
   0   1   0   0   0   1   0   1  -1  -1   0  -1  -1   0  -1  -1   0   0  -1  -1  -1   0   0   0  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0  -1  -1  -1   0   0   0   0   0  -1  -1   0   0  -1   0   0   0   0   0   0   0   0   0  -1  -1  -1  -1  -1  -1   1  -1   1   1  -1  -1   0   0
   0   0  -1  -1  -1   2   0   0   0  -1  -1   0  -1  -1   0  -1  -1   2  -1  -1  -1   0   0   0  -1  -1  -1   0  -1   0  -1  -1   0  -1  -1   0   0   0  -1  -1  -1   0  -1  -1  -1  -1   1   1  -1  -1  -1   0  -1  -1   0  -1  -1   0   0  -1   0  -1  -1   0   0  -1   1   1  -1  -1   0   0
  -1   1   1   1   1  -1   1  -1   0   1   0   0   1   0   0   1   0  -1   0   0   0   0   0   0   1   0   0   0   1   0   0   1   0   0   1   0   0   0   1   0   1   0   0   1   0   0  -1   0   0   0   1   0   0   1   0   0   1   0   0   1   0   1   0   0   0   1  -1   0   0   1   0   0
7 126 Points of dual PG: (nv=23)
   1   0   1   1   1   1   1   0   1   1   1   1   0   0   1   1   1   0   1   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1
   0   1   0   0   0   0   0   1   0   0   0   0   1   1   0   0   0   1   0   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0
  -1   0   2  -1  -1  -1  -1   0   2   2  -1  -1   0   0  -1   2  -1   0  -1   0  -1   2   2  -1   0   0   0   0   0   0   0   0   0   0   1   1   1   1   1   1   1   1   1   1   1   1  -1   2  -1  -1   2   2   0   0  -1  -1  -1  -1  -1   0  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0  -1   0   0   0   0   0   0   0   0   1   1   1   1   1   1   1   1   1   1   1   1  -1   2   0   2   2  -1  -1  -1  -1  -1  -1   0  -1  -1   0   0   0
  -1   0  -1   2  -1   2   2   0  -1  -1  -1  -1   0   0   2  -1  -1   0   2   0   2  -1  -1  -1   0   0   0   0   1   1   1   1   1   1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0  -1   0   0  -1  -1  -1   0  -1  -1   1  -1  -1  -1   0   0   0   0   0   0   1   1   1   1   1   1   1   2   1   2   2   1  -1  -1  -1  -1  -1  -1   0   0   0   0   1   0  -1   1   1   1   1   1   1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   2  -1   0  -1  -1   1   0   0  -1  -1   2  -1   2   0  -1  -1   0
  -1   1   0   0   0   0   0   0   0   0   0   0   1   1  -1  -1  -1   0  -1   0  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1   0  -1   0   0  -1   0   0   0   0   0   0   0   0   0   0   0   0  -1   0  -1   0   0  -1   0   0   0   0   0   0   0   0   0   0  -1   0  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0  -1   0  -1   0   0  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1   0
  -1   0  -1  -1  -1   1  -1   0   1  -1   1  -1   1   0  -1  -1  -1   1   1   0  -1   1  -1   1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1   1  -1   0  -1   0   0   1  -1   0  -1  -1   0   0  -1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1   0  -1  -1   0   0  -1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1  -1  -1  -1   0   0   1   0  -1  -1   0   0   0   0  -1  -1   0   0
  -1   1   1   1   1  -1  -1   1  -1  -1  -1  -1   0   0   1   1   1   0  -1   0  -1  -1  -1  -1   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   0   0  -1   0  -1   0   0   0   0  -1   0   1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0  -1  -1   0   1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1   0  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1  -1   0   1  -1   0  -1   0   0  -1  -1   0  -1  -1   1   0  -1   0   0  -1  -1  -1  -1   0
7 96 Points of dual PG: (nv=21)
   0   0   1   1   1   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   0   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   0   1
   1   1   0   0   0   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   1   0
   0   0  -1   2  -1   0  -1   2  -1  -1  -1   2   2  -1  -1  -1   2  -1  -1   2  -1   1   1   1  -1  -1   0  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0  -1  -1   0   0   1   0   0   0   1   1   1   1   1   1   2   2   2   1   0   0  -1  -1  -1  -1  -1  -1  -1   0  -1   1   0   0   0   0   0   0   0   0   0   0   1   1   1   1   1   1   0   0   0   0   0   0
   0   0   2  -1   2   0  -1  -1  -1   2   2  -1  -1   2   2  -1  -1  -1  -1  -1  -1   0   0  -1   0   0   1  -1  -1  -1   0   0   0   1   1   1   2   2   2  -1  -1  -1   0   0   1   1   0   0  -1   1   1   1  -1  -1  -1   0   0   0  -1  -1  -1  -1   0   1   0   0   0   0   1   1   1  -1   1   0  -1  -1  -1  -1   0   0   0   1   1   1  -1  -1  -1   0   0   0   0   1  -1   0   0   0
   0   0   1   1  -1   0   1  -1  -1   1   1   1   1  -1  -1   1  -1  -1   1  -1  -1  -1  -1  -1  -1  -1  -1   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0   0  -1  -1   0   0  -1   0   0   0   0   0   0   0   0   0   0   0   0  -1  -1  -1  -1   1   1   1   1   1   1  -1  -1  -1  -1   1   1   1   1   1   1   1   1   1   1   1   1   1   1   1  -1  -1  -1  -1   0   0
   0   0  -1  -1  -1   2  -1  -1  -1   0  -1   0  -1   0  -1   0   0   0  -1  -1  -1  -1   0   0   0  -1   0  -1  -1   0  -1  -1   0  -1  -1   0  -1  -1   0  -1  -1   0  -1  -1  -1  -1   1   1  -1  -1  -1   0  -1  -1   0  -1  -1   0  -1  -1   0  -1   0  -1  -1  -1  -1   0  -1  -1   0   0   0  -1  -1  -1  -1   0  -1  -1   0  -1  -1   0  -1  -1   0  -1  -1   0  -1  -1  -1  -1   0   0
  -1   1   1   1   1  -1   1   1   1   0   0   0   0   0   0   0   0   0   0   0   0   1   0   0   0   0   0   0   1   0   0   1   0   0   1   0   0   1   0   0   1   0   0   1   0   1  -1   0   0   0   1   0   0   1   0   0   1   0   0   1   0   1   0   1   1   0   1   0   0   1   0   0   0   0   1   0   1   0   0   1   0   0   1   0   0   1   0   0   1   0   1   0   0   0   0   0
EOF
)
if [ $DIM -lt 6 ]; then
    EXPECTED="Please increase POLY_Dmax to at least 6 = 5 + 2 - 1
(POLY_Dmax >= dim N + codim - 1 is required)"
fi
run_test


# Pages 53-54
# The "sed" command strips the unpredictable timing information
COMMAND="./nef-${DIM}d.x -N -Lv tests/input/6.4.24-nef-N-Lv.txt | sed 's/ *[0-9]*sec.*//g'"
DESCRIPTION="nef-${DIM}d.x -N -Lv example on pages 53-54"
EXPECTED=$(cat<<-EOF
M:24 15 N:39 12  codim=2 #part=2
5 12 Vertices in N-lattice:
    0    0   -1    2   -1    0    0    0   -1    2   -1    0
    0    0    2   -1   -1    0    0    0    2   -1   -1    0
    0    1    0    0    0    1    1    0   -1   -1   -1    0
   -1   -1    0    0    0    2   -1   -1    0    0    0    2
   -1    2    0    0    0   -1   -1    2    0    0    0   -1
------------------------------------------------------------
    1    1    2    2    0    1    0    0    0    0    2    0  d=9  codim=0
    1    1    1    1    0    0    0    0    0    0    1    1  d=6  codim=0
    1    1    2    0    2    1    0    0    0    2    0    0  d=9  codim=0
    1    1    1    0    1    0    0    0    0    1    0    1  d=6  codim=0
    1    1    1    0    0    1    0    0    0    1    1    0  d=6  codim=0
    2    2    1    0    0    0    0    0    0    1    1    2  d=9  codim=0
    1    1    0    2    2    1    0    0    2    0    0    0  d=9  codim=0
    1    1    0    1    1    0    0    0    1    0    0    1  d=6  codim=0
    1    1    0    1    0    1    0    0    1    0    1    0  d=6  codim=0
    2    2    0    1    0    0    0    0    1    0    1    2  d=9  codim=0
    1    1    0    0    1    1    0    0    1    1    0    0  d=6  codim=0
    2    2    0    0    1    0    0    0    1    1    0    2  d=9  codim=0
    3    3    0    0    0    3    0    0    2    2    2    0  d=15  codim=0
    3    3    0    0    0    0    0    0    1    1    1    3  d=12  codim=0
    1    0    1    1    0    1    0    1    0    0    1    0  d=6  codim=0
    1    0    1    0    1    1    0    1    0    1    0    0  d=6  codim=0
    2    0    1    0    0    2    0    2    0    1    1    0  d=9  codim=0
    1    0    0    1    1    1    0    1    1    0    0    0  d=6  codim=0
    2    0    0    1    0    2    0    2    1    0    1    0  d=9  codim=0
    2    0    0    0    1    2    0    2    1    1    0    0  d=9  codim=0
    3    0    0    0    0    3    0    3    1    1    1    0  d=12  codim=0
    1    0    0    0    0    0    0    1    0    0    0    1  d=3  codim=3
    0    1    3    3    0    1    1    0    0    0    3    0  d=12  codim=0
    0    1    2    2    0    0    1    0    0    0    2    1  d=9  codim=0
    0    1    3    0    3    1    1    0    0    3    0    0  d=12  codim=0
    0    1    2    0    2    0    1    0    0    2    0    1  d=9  codim=0
    0    2    3    0    0    2    2    0    0    3    3    0  d=15  codim=0
    0    1    1    0    0    0    1    0    0    1    1    1  d=6  codim=0
    0    1    0    3    3    1    1    0    3    0    0    0  d=12  codim=0
    0    1    0    2    2    0    1    0    2    0    0    1  d=9  codim=0
    0    2    0    3    0    2    2    0    3    0    3    0  d=15  codim=0
    0    1    0    1    0    0    1    0    1    0    1    1  d=6  codim=0
    0    2    0    0    3    2    2    0    3    3    0    0  d=15  codim=0
    0    1    0    0    1    0    1    0    1    1    0    1  d=6  codim=0
    0    1    0    0    0    1    1    0    1    1    1    0  d=6  codim=0
    0    3    0    0    0    0    3    0    2    2    2    3  d=15  codim=0
    0    0    1    1    1    0    0    0    0    0    0    0  d=3  codim=3
    0    0    2    2    0    1    1    1    0    0    2    0  d=9  codim=0
    0    0    1    1    0    0    1    1    0    0    1    1  d=6  codim=0
    0    0    2    0    2    1    1    1    0    2    0    0  d=9  codim=0
    0    0    1    0    1    0    1    1    0    1    0    1  d=6  codim=0
    0    0    1    0    0    1    1    1    0    1    1    0  d=6  codim=0
    0    0    1    0    0    0    2    2    0    1    1    2  d=9  codim=0
    0    0    0    2    2    1    1    1    2    0    0    0  d=9  codim=0
    0    0    0    1    1    0    1    1    1    0    0    1  d=6  codim=0
    0    0    0    1    0    1    1    1    1    0    1    0  d=6  codim=0
    0    0    0    1    0    0    2    2    1    0    1    2  d=9  codim=0
    0    0    0    0    1    1    1    1    1    1    0    0  d=6  codim=0
    0    0    0    0    1    0    2    2    1    1    0    2  d=9  codim=0
    0    0    0    0    0    3    3    3    2    2    2    0  d=15  codim=0
    0    0    0    0    0    0    3    3    1    1    1    3  d=12  codim=0
H:19 19 [0] P:0 V:1 2 3 4 5 6  13 15 17 19 21 23 25 26 28 30 32 34 36   (6 3) (3 3) (6 3) (3 3) (3 3) (3 6) (6 3) (3 3) (3 3) (3 6) (3 3) (3 6) (6 9) (3 9) (3 3) (3 3) (3 6) (3 3) (3 6) (3 6) (3 9) (0 3) (9 3) (6 3) (9 3) (6 3) (9 6) (3 3) (9 3) (6 3) (9 6) (3 3) (9 6) (3 3) (3 3) (6 9) (3 0) (6 3) (3 3) (6 3) (3 3) (3 3) (3 6) (6 3) (3 3) (3 3) (3 6) (3 3) (3 6) (6 9) (3 9)
H:19 19 [0] P:1 V:2 3 4 8 9 10  16 17 18 19 20 21 24 29 30 31 32 33 34   (6 3) (3 3) (6 3) (3 3) (3 3) (3 6) (6 3) (3 3) (3 3) (3 6) (3 3) (3 6) (6 9) (3 9) (3 3) (3 3) (3 6) (3 3) (3 6) (3 6) (3 9) (0 3) (9 3) (6 3) (9 3) (6 3) (9 6) (3 3) (9 3) (6 3) (9 6) (3 3) (9 6) (3 3) (3 3) (6 9) (3 0) (6 3) (3 3) (6 3) (3 3) (3 3) (3 6) (6 3) (3 3) (3 3) (3 6) (3 3) (3 6) (6 9) (3 9)
np=2 d:0 p:0
EOF
)
if [ $DIM -lt 6 ]; then
    EXPECTED="Please increase POLY_Dmax to at least 6 = 5 + 2 - 1
(POLY_Dmax >= dim N + codim - 1 is required)"
fi
run_test
