/*-------------------------------------------------------------------------- 
  Filename        : sgdefects.c
  Abstract        : SCSI Generic Disk Defects tool
  Operation System: Linux 2.2 or greater
  Author  	  : Andy Cress   <arcress@users.sourceforge.net>
  Copyright (c) 2001-2008 Intel Corporation 
  Copyright (c) 2009, Kontron America, Inc.
  ----------- Change History -----------------------------------------------
  05/17/00 v0.70 ARCress  created 
  09/05/01 v0.84 ARCress  cleanup
  10/30/01 v1.0  ARCress  cleanup includes & indent -kr
  11/09/01 v1.1  ARCress  enlarged devstat2 from 80 to 120 for overrun
  08/15/02 v1.2  ARCress  moved common subroutines to sgcommon
			  added more usage messages 
  09/03/02 v1.3  ARCress  streamline display for errors in get_scsi_info 
  10/09/03 v1.4  ARCress  relinked with new sgcommon.c
  05/11/05 v1.5  ARCress  included patch for larger devlist from Nate Dailey
  12/27/05 v1.6  ARCress  skip if emulated USB device
  ----------- Description --------------------------------------------------
  sgdefects
Sequence of events for this utility:
  * List each device on the system with firmware versions.
  * User selects a device for examination (or automatic if using -a)
  * Verify that the disk is present and ready
  * Get the defect count for the primary and grown defect lists.
  * Get the entire list if (if -l is specified)
sgdefects    [-e -a -l -x] 
Options:
  -e  	Do not write to any files. Usually a log file is created
    	and written to.
  -m  	Automatically get defects for all drives.
  -l    Show the defect list values as well as the counts.
  -x    Outputs extra debug messages
  -n    Use numeric device names (e.g. /dev/sg0)
----------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------- 
Copyright (c) 2002, Intel Corporation
Copyright (c) 2009, Kontron America, Inc.
All rights reserved.
Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:
  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  --------------------------------------------------------------------------*/
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <time.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/utsname.h>
#include "sgsub.h"
#include "sgcommon.h"

/* External Functions prototypes */
extern int uname(struct utsname *buf);
extern void setbuf(FILE * stream, char *buf);
extern int errno;
extern char fdebugsub;		/* used in sgsub.c */
extern FILE *dbgout;		/* used in sgsub.c */
extern char HeaderStr[1];       /* from sgcommon.c */
#define  EBUFF_LEN   80
#define  MAX_ERRORS   4

/* Global data definitions */
char *progver  = "1.66";		/* program version */
char *progname = "sgdefects";	/* program name */
char logfile[] = "/var/log/sgdefects.log";	/* log filename */
char model[20] = "";		/* default model string */
char fsetfile = 0;
char device[80] = "/dev/sda";	/* UNIX device name (w path) */
int sdelay = 10;		/* num sec to delay before TUR */
FILE *fdlog = NULL;		/* log file descriptor */
FILE *fdmsg;			/* file descriptor for messages  */
char fauto = 0;			/* =1 to automatically do all disks */
char filesok = 1;		/* =1 if ok to write to files */
char fdebug = 0;		/* =1 for debug messages */
char finteractive = 0;		/* =1 for interactive mode */
char flogopen = 0;		/* =1 if log file is open */
char fgetall = 0;		/* =1 get PDL always, =0 check kern ver */
int do_numeric = 1;		/* NUMERIC_SCAN_DEF; */
char fonedev = 0;		/* =1 to autorun w one device */
int ionedev = 0;		/* index of one device from user -i param */
uchar sense_buffer[80];		/* buffer for sense data */
char output[132];		/* message output buffer */
char output2[80];		/* extra message output buffer */
uchar devstat[80];		/* used for inquiry & device status */
uchar devstat2[120];		/* used for Seagate inquiry2, etc.  */
uchar disk_buffer[0x200];	/* 512 bytes, used for misc scsi data */
uchar modebuf[300];		/* ~240 bytes, used for mode pages */
uchar defectlist[0xFFFF];	/* big buffer for defects */
int idev = 0;
int ndev = 0;
DEVLIST_TYPE devlist[MAX_DEVLIST_DEVS];

/* Local subroutines */
int do_getdefects(int idx, char fvalues);
int beforegd(int idx, int *pnumrdy);

void dumpdlist(FILE * fdout, uchar * bufp, int mlen)
{
    int i, j;
    char abuf[80];
    fprintf(fdout, "\nDefect List: \n%04x: %02x %02x %02x %02x ",
	    0, bufp[0], bufp[1], bufp[2], bufp[3]);
    for (i = 4; i < mlen; i += 8) {
	if ((i - 4) % 16 == 0)	/* start a new line */
	    fprintf(fdout, "\n%04x: ", i);
	for (j = 0; j < 8; j++) {
	    itoh(&bufp[i + j], 1, &abuf[j * 2]);
	}
	fprintf(fdout, "%s ", abuf);
    }
    fprintf(fdout, "\n");
    return;
}				/*end dumpdlist() */

int main(int argc, char **argv)
{
    int i, c, idx;
    char response = 0;
    short sts, rc;
    ulong ltime1;
    char fname[64];
    char ebuff[EBUFF_LEN];
    int sg_fd;
    int flags;
    int res, k;
    int iautodev;
    int eacces_err = 0;
    int num_errors = 0;
    char fgetdefects = 0;
    char fgetvalues = 0;	/* =1 to get defect list values */
    DEVFS_LIST devfs_components = {0,0,0,0,0};
    fdmsg = stdout;		/* set default fdmsg */
    // progname = argv[0];
    while ((c = getopt(argc, argv, "aei:lnvxI?")) != EOF)
	switch (c) {
	case 'a':
	    fauto = 1;		/* automatically do all disks */
	    break;
	case 'e':
	    filesok = 0;
	    break;
	case 'l':
	    fgetall = 1;
	    break;
	case 'n':
	    do_numeric = 0;
	    break;
	case 'i':
	    fonedev = 1;
            ionedev = atoi(optarg);
	    break;
	case 'v':
	    fgetvalues = 1;
	    break;
	case 'x':
	    fdebug = 1;
	    break;
	case 'I':
	    finteractive = 1;
	    break;
	case '?':
	default:
	    printf("Usage: %s [-aeilnvxI]\n", progname);
	    printf(" -a  Automatically get defect counts for all disks\n");
	    printf(" -e  Do not write to any files, even the log file\n");
	    printf(" -iN Autorun for one disk, at index N\n");
	    printf(" -l  Show the defect list values as well as the counts\n");
	    printf(" -n  Don't use numeric device names, use alpha instead (/dev/sga)\n");
	    printf(" -v  Get all defect Values, not just counts\n");
	    printf(" -x  Outputs extra debug messages\n");
	    printf(" -I  Interactive mode, prompts for input\n");
	    exit(1);
	}
    /* only run this as superuser */
    i = geteuid();
    if (i > 1) {
	printf("Not superuser (%d)\n", i);
	exit(1);
    }
    /* open log file */
    if (filesok)
	fdlog = fopen(logfile, "a+");
    if (fdlog == NULL) {
	flogopen = 0;
	filesok = 0;
	fdlog = stderr;
    } else {			/* logfile open successful */
	flogopen = 1;
	time((time_t *) & ltime1);
	fprintf(fdlog, "\n--- %s v%s log started at %s", progname, progver,
		ctime((long *) &ltime1));	// ctime outputs a '\n' at end
    }
    fdebugsub = fdebug;
    dbgout = fdlog;
    iautodev = 0;
    /* loop to display disk choices and get defects */
    while (1) {
	/* display header */
	printf("\n");
	printf("                %s utility v%s for SCSI disks\n", progname,
	       progver);
	printf
	    ("              ******************************************\n");
	if (flogopen)
	    printf("Log file %s is open\n", logfile);
	printf(HeaderStr);
	/* get SCSI Device Info */
	idev = 0;
	flags = O_RDWR;		/* could use OPEN_FLAG if read-only. */
	num_errors = 0;
        for (k = 0; (k < MAX_SG_DEVS) && (idev < MAX_DEVLIST_DEVS) &&
                    (num_errors < MAX_ERRORS); ++k) {
	    make_dev_name(fname, k, do_numeric, &devfs_components);
	    if (devfs_components.all_leaves_searched != 0) {
		devfs_components.all_leaves_searched=0;
		break;
	    }
	    sg_fd = open(fname, flags | O_NONBLOCK);
	    if (sg_fd < 0) {
		if (EBUSY == errno) {
		    printf("%s: device busy (O_EXCL lock), skipping\n",
			   fname);
		    continue;
		}
		    else if ((ENODEV == errno) || (ENOENT == errno) ||
			     (ENXIO == errno)) {
		    ++num_errors;
		    continue;
		} else {
		    if (EACCES == errno)
			eacces_err = 1;
		    sprintf(ebuff, "Error opening %s ", fname);
		    perror(ebuff);
		    ++num_errors;
		    continue;
		}
	    }
            /* do scsi inquiry and fill in devlist[idev] */
	    sts = get_scsi_info(sg_fd, idev, fname,0);
	    if (sts) {
		showlog( "device %s failed with sts = %d, skip\n",
			fname, sts);
	    } 
	    {
		if (sg_fd > 0) {
		    res = close(sg_fd);
		    if (res < 0)
			fprintf(fdmsg, "Error closing %s, errno = %d\n",
				fname, errno);
		    else
			devlist[idev].sgfd = 0;	/* set in get_scsi_info() */
		}
		idev++;
	    }
	}			/*end loop */
	ndev = idev;
	if (ndev == 0 && errno == ENODEV) {	/* CONFIG_CHR_DEV_SG != y ? */
	    sprintf(output,
		    "Cant open any SCSI devices.\nMake sure CONFIG_CHR_DEV_SG is set in /usr/src/linux/.config\n");
	    showit(output);
	    quit(1);
	}
        else if (ndev == 0) {
            sprintf(output,"Cant open any sg SCSI devices, errno = %d\n",errno);
	    if (errno == ENOENT) 
	       strcat(output,"Try option -n to change /dev/sg* device names\n");
            showit(output);
        }
	if (fauto || fonedev) {
	    response = 'g';
	} else if (finteractive) {
	    printf("\nEnter Selection ('g' to get defects, 'v' to get defect"
		   " values, 'q' to quit) : ");
	    response = get_func();
	} else {  /*default*/
	    response = 'g';
	}
	if (response == 'q' || response == 'Q') {
	    printf("\n");
	    quit(0);
	} else if (response == 'g' || response == 'G') {
	    fgetdefects = 1;	/* get defect counts only */
	    fgetvalues = 0;
	    printf("Get Defects ...\n");
	} else if (response == 'v' || response == 'V') {
	    fgetdefects = 1;
	    fgetvalues = 1;
	    printf("Get Defects and Values ...\n");
	} else {		/* response wrong */
	    printf("Invalid input [%c], enter 'g' or 'q'.\n", response);
	    continue;
	}
	if (fauto)
	    i = iautodev;	//get_ndev(iautodev); // get first matching dev
	else if (fonedev)
	    i = ionedev;
	else if (finteractive)
	    i = get_idev();	/* get the device index that the user picks */
	else
	    i = iautodev;
	if (i == 0xff) {
	    if (finteractive) {
		printf("Index out of range, try again.\n");
            } else {     // if (fauto || fonedev) 
                printf("Device not found.\n");
                quit(1);        /* exit */
	    }
	} else {		/*index ok */
	    int numrdy;
	    idx = i;		/* do first disk */
	    if (fauto)
		iautodev = idx;	/* save disk index */
	    {
		rc = beforegd(idx, &numrdy);
		if (numrdy == 0)
		    idx = ndev;
		/* Get defects for each matching disk */
		for (i = idx; i < ndev; i++) {
		    if (devlist[i].fdoit) {
			rc = do_getdefects(i, fgetvalues);
			if (fdebug && rc != 0)
			    quit(rc);
		    }
		    if (devlist[i].sgfd > 0) {
			close(devlist[i].sgfd);	/*opened in beforegd() */
			devlist[i].sgfd = 0;
		    }
		    if (finteractive || fonedev)
			i = ndev;	/* stop after one if interactive */
		}		/*endfor each disk */
	    }			/*endif doit */
	    if (finteractive) {	/* interactive, go to menu again */
		do_pause();
	    } else {	// if (fauto || fonedev) * then done 
		quit(0);	/* normal exit, successful */
	    }
	}			/*index ok */
    }				/*end while(1) */
}				/*end main() */

char isver24(void)
{
    struct utsname name;
    char fis24;
    fis24 = 0;
    if (uname(&name) != -1) {
	if (fdebug)
	    printf("release: %s\n", name.release);
	if (strncmp(name.release, "2.4", 3) == 0)
	    fis24 = 1;
    }
    return (fis24);
}				/*end isver24() */

int do_getdefects(int idx, char fvalues)
{
    int sts;
    int devfd;
    int isave;
    int rcdl;
    int dlen;
    uchar ch, dv;
    uchar *bufp;
    ulong len_glist, num_glist;
    ulong len_plist, num_plist;
    char fis24;
    isave = idx;
    rcdl = 0;
    sts = 0;
    /* Get Defects for each disk */
    {
	rcdl = 0;
	if (finteractive || fonedev) 
	    idx = isave;	/* only do selected disk */
	devfd = devlist[idx].sgfd;
	ch = devlist[idx].chn;
	dv = devlist[idx].tgt;
	if (devlist[idx].fdoit == 1) {
	    if (!finteractive) {	/*++++ if (fauto) */
		/* Sometimes next device may have a unit attention/reset check 
		 * condition pending, so clear that before the mode command.
		 */
		sts = test_unit_ready(devfd, 0);
		if (fdebug)
		    printf("Extra test_unit_ready sts = %x\n", sts);
	    }
	    /* start the get defects command */
	    sprintf(output, "Getting defects from disk %d\n", idx);
	    showit(output);
	    /* do it once for plist, then again for glist */
	    if (fvalues)
		dlen = sizeof(defectlist);	/* get defect values */
	    else
		dlen = 80;	/* only want counts, avoid very large buffers */
	    bufp = defectlist;
	    fis24 = isver24();
	    if (fis24 || fgetall) {   /* get primary defects if 2.4.x kernel */
		sts = get_defects(devfd, bufp, dlen, 1);
		if (sts) {
		    rcdl = sts;
		    sprintf(output,
			    "[%d] Error %d in get_defects(primary)\n", idx,
			    sts);
		    showit(output);
		} else {	/* good results */
		    /* note that there is a 4-byte header, not included in length */
		    len_plist = ((bufp[2] << 8) | bufp[3]);
		    num_plist = len_plist / 8;
		    if (fdebug) {
			sprintf(output,
				"len_plist=%lx num_plist=%ld, buf: %x %x\n",
				len_plist, num_plist, bufp[2], bufp[3]);
			showit(output);
		    }
		    sprintf(output, "[%d] Disk has %ld primary defects\n",
			    idx, num_plist);
		    showit(output);
		    if (fvalues) {
			printf("Complete Defect List written to %s\n",
			       logfile);
			if (len_plist + 4 > dlen)
			    len_plist = dlen - 4;
			dumpdlist(fdlog, bufp, len_plist + 4);
		    }
		}		/*endif good */
	    } /*endif fis24 */
	    else {
		sprintf(output, "[%d] Skipping primary defects\n", idx);
		showit(output);
	    }
	    /* dlen & bufp are already set above */
	    /* get grown defect list */
	    sts = get_defects(devfd, bufp, dlen, 0);
	    if (sts) {
		rcdl = sts;
		sprintf(output, "[%d] Error %d in get_defects(grown)\n",
			idx, sts);
		showit(output);
		if (fdebug || !fauto)
		    return (rcdl);
	    } else {		/* good results */
		/* note that there is a 4-byte header, not included in length */
		len_glist = ((bufp[2] << 8) | bufp[3]);
		num_glist = len_glist / 8;
		sprintf(output, "[%d] Disk has %ld grown defects\n", idx,
			num_glist);
		showit(output);
		if (fvalues) {
		    printf("Complete Defect List written to %s\n",
			   logfile);
		    if (len_glist + 4 > dlen)
			len_glist = dlen - 4;
		    dumpdlist(fdlog, bufp, len_glist + 4);
		}
	    }			/*endif good */
	    if (fauto && rcdl != 0)
		return (rcdl);	/*skip to next one if error */
	    sprintf(output,
		    "\n[%d] get_defects command complete, status = 0x%x\n",
		    idx, rcdl);
	    showit(output);
	    if (fdebug && rcdl != 0) {
		return (rcdl);	/* return (only if debug on) */
	    }
	    if (finteractive || fonedev) 
		idx = ndev;	/* end loop */
	}			/*endif fdoit */
    }				/*endfor each disk */
    return (rcdl);
}				/* end do_getdefects() */

int beforegd(int idx, int *pnumrdy)
{
    struct SCSI_INQUIRY *scsi_inq;
    int sts;
    int sgfd;
    int isave, nrdy;
    char *pl;
    int i, rc, rcdl;
    int k, a, q;
    int openflags;
    char *fname;
    int devtype;

    isave = idx;
    nrdy = 0;
    sts = 0;
    rcdl = 0;
    openflags = O_RDWR;		/* | O_NONBLOCK; */
    for (idx = 0; idx < ndev; idx++) {
	if (finteractive || fonedev)
	    idx = isave;	/* only do selected disk */
	fname = devlist[idx].fname;
	sgfd = open(fname, openflags);	/* open blocking */
	if (sgfd < 0) {
	    printf("%s: open error, errno = %d\n", fname, errno);
	    return (errno);
	}
	devlist[idx].sgfd = sgfd;
	scsi_inq = (struct SCSI_INQUIRY *) devlist[idx].inq;
        devtype = scsi_inq->dev_type & 0x1F;
        /* if emul USB, dcap==0 from get_scsi_info() */
	if ((devtype == 0) && (devlist[idx].dcap != 0)) {  /* if type = disk */
	    if (fdebug) /*DEBUG*/ 
		printf("Doing Test Unit Ready on idx = %d\n", idx);
	    sts = test_unit_ready(sgfd, 0);	/* dont show errors */
	    if (sts == SCHECK_CND) {
		/* not ready initially, try to make it ready */
		i = get_sense(sts, sense_buffer);
		k = sense_buffer[2] & 0x0f;	/*Sense Key */
		a = sense_buffer[12];
		/*ASC*/ q = sense_buffer[13];
		/*ASCQ*/ if (k == 2 && a == 4 && q == 2) {
		    /* 02-04-02 means it needs a start command, so issue it. */
		    sts = start_unit(sgfd);
		    if (fdebug) {
			/*DEBUG*/ printf("\nStart Unit: sts=%d, ", sts);
			if (sts == SCHECK_CND) {
			    rc = get_sense(sts, sense_buffer);
			    printf("sense data: %x %x %x\n",
				   sense_buffer[2] & 0x0f,
				   sense_buffer[12], sense_buffer[13]);
			}
		    }
		}		/*endif 02-04-02 */
		sts = test_unit_ready(sgfd, 1);	/* try again & show errors */
	    }
	    if (sts) {
		sprintf(output, "[%d] Error %d from Test Unit Ready\n",
			idx, sts);
		showit(output);
		devlist[idx].fdoit = 0;
	    } else {
		devlist[idx].fdoit = 1;
		sprintf(output, "Device [%d] is ready for get_defects\n",
			idx);
		showit(output);
		nrdy++;		/* number ready to get_defects */
	    }
	} else
	    devlist[idx].fdoit = 0;	/* not ok  */
	if (finteractive || fonedev)
	    idx = ndev;		/* end loop */
    }				/* end for devlist TUR loop */
    *pnumrdy = nrdy;		/* set return value */
    if (nrdy == 0) {
	sprintf(output, "There are no ready devices.\n");
        strcat(output,"Try 'modprobe sg' if not already loaded\n"); 
	showit(output);
	return (sts);
    } else {
	if (nrdy > 1)
	    pl = "s";
	else
	    pl = "";
	sprintf(output, "Starting get_defects process for %d disk%s.\n",
		nrdy, pl);
	showit(output);
    }
    setbuf(fdmsg, NULL);	/* set for unbuffered writes */
    return (rcdl);
}				/*end beforegd() */

/* end sgdefects.c */
