/*--------------------------------------------------------------------------
  Filename:	    sgsafte.c
  Abstract: 	    SCSI Generic SAF-TE Monitor
		    Poll the hot-swap backplane with SAF-TE commands 
		    to check for insertion events.
		    Allow the user to set the drive slot status, so 
		    that the HSBP and drive LEDs can be tested.

  Author:  	    Andy Cress   <arcress@users.sourceforge.net>
  Operating System: Linux 2.2 or greater
  Copyright (c) Intel Corporation 2001-2008

  ----------- Change History -----------------------------------------------
  06/03/03 v1.20 ARCress  copied & cut from sgraidmon.c v1.19
  03/14/07 v1.21 ARCress  changes for sgsubmon.c and enabling >1 HSBP
----------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002-2003, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  -------------------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>
#include <time.h>
#include <sys/ioctl.h>
#include <scsi/sg.h>

#include "sgsubmon.h"

//////////////////////////////////////////////////////////////////////
// Global Data 
//////////////////////////////////////////////////////////////////////
char *progver  = "1.66";        /* program version */
char *progname    = "sgsafte";   /* program name */
char logfile[]    = "/var/log/sgsafte.log";     /* log filename */   
FILE *fdlog = NULL;             /* log file descriptor */
FILE *fdmsg = NULL;             /* message file descriptor (stdout) */
char output[132];		/* output buffer for text messages */
char output2[80];		/* 2nd output buffer for messages */
int  drivenum = -1;             /* drive number to set */
char setval = 0;                /* value to set slot state  */
char fdebug = 0;                /* =1 for debug messages */
char frunonce = 1;              /* =1 to run only one pass */
char filesok = 1;               /* =1 if ok to write to files */
char flogopen = 0;              /* =1 if log file is open */
char finsert = 1;               /* =1 to insert in devactive */
char fskipreset = 0;            /* =1 to skip SCSI resets */
char do_numeric = 1;            /* =1 to use numeric sg devnames (sg0) */ 
char background = 0;            /* =1 if background, =0 if foreground app */
char moresn = 0;                /* =1 if 12-byte sernum, =0 if 8-byte sernum */
char fdevfs = 0;		/* =1 if DevFS is detected */
int interval = 5;               /* sleep 5 sec between scans */ 
int sernum_len = 8;
int idev = 0;
int ndev = 0;
int nprocdev = 0;
int npart = 0;
int nrdy = 0;
int nchgs = 0;
int lastnrdy = MAX_DEVS;
int iremove  = MAX_DEVS;
int iinsert  = MAX_DEVS;
SLOT_LIST slot[MAX_DEVS];
SLOT_LIST *pslot;
int nslots = MAX_DEVS;	/* SAF-TE/SES: number of slots for this HSC/HSBP */
int islot  = 0;		/* SAF-TE/SES: next index in slot table */
uchar  slotmode; 	/* SAF-TE: 3=#insertions, 4=status */
int slotoff = 0xff;	/* SAF-TE: offset into data for slot info */
char fsafte = 0; 	/* SAF-TE: is there a safte HSC/HSBP? */
char fses = 0;    	/* SES: is there a SES HSC? */
int max_try = 12;
char blank4[5]   = "    ";
char blank8[9]   = "        ";
char blank16[17] = "                ";
uchar inqcdb[6]     = { 0x12, 0, 0, 0, SGDATALEN, 0 };
uchar turcdb[6]     = { 0x00, 0, 0, 0, 0,    0 };
uchar startcdb[6]   = { 0x1B, 0, 0, 0, 0x01, 0 };
uchar defectcdb[10] = { 0x37, 0, 0x0d, 0, 0, 0, 0, 0x00, SGDATALEN, 0 };

/* Device Statuses for devlist.dstatus: */
char dstat_str[5][8] = { 
    "dead  ",  "failed", "notrdy", "ready ", "online"
};
DEVLISTX_TYPE devlist[MAX_DEVS];
struct {
    uchar vendor[9];
    uchar model[17];
    uchar fwrev[5];
    uchar sernum[13];
    uchar dstatus;      /* device status: DSTAT_* */
} devmatch[MAX_DEVS];
#define NDEVTYP  15
char *devtype_array[NDEVTYP] = {
    "Disk", /*0*/
    "Tape", /*1*/
    "Prtr", /*2*/
    "Proc", /*3*/
    "WORM", /*4*/
    "CDRM", /*5*/
    "Scan", /*6*/
    "Opti", /*7*/
    "MChg", /*8*/
    "Comm", /*9*/
    "Emul", /*10*/
    "Un11", /*11*/
    "RAID", /*12*/
    "Encl", /*13*/
    "Unkn"  /*14*/
};

/* List of vendors who use 12-byte serial numbers, instead of 8-byte. */
#define NVEND12  3
char *vendor12[NVEND12] = {"FUJITSU", "MAXTOR", "HP 73.4G" };

//////////////////////////////////////////////////////////////////////
// Local Subroutines
//////////////////////////////////////////////////////////////////////
void quit(int rc)
{
    if (flogopen) fclose(fdlog);     /* close the log file */
    flogopen = 0;
    fdlog = stderr;
    exit(rc);
} 

/*
 * showit
 * Outputs the message from buf to fdmsg, if not background,
 * and to fdlog, if opened ok.
 * Note that logit() takes var args and calls showit().
 *
 * Note that some of the fdebug output assumes foreground and
 * uses printf directly.  So some debug output would go to
 * /dev/null if background option is also set.
 */
void showit(char *buf)
{
    if (filesok && !flogopen) { /* log is not open */
        fdlog = fopen(logfile, "a+");
        if (fdlog != NULL)
            flogopen = 1;
        else {
            printf("Cannot open %s, errno = %d\n", logfile, errno);
            fdlog = stderr;
        }
    }
    if (flogopen)
        fprintf(fdlog, buf);    /*write to log */
    if (!background)
        fprintf(fdmsg, buf);    /*message display defaults to stdout */
}
 
/*
 * getmd
 * This function scans the raidtab file for a pattern
 * and returns a matching /dev/md* device name and a
 * corresponding /dev/sd* partition name.
 * Input parameters:  devpattn = pattern to scan for
 *                    nth      = find nth matching partition
 * Output paramters:  mddev    = /dev/md* device of match
 *                    mdpart   = /dev/sd* partition of match
 * returns  0 if this is a match in raidtab
 *         >0 if errno in opening raidtab
 *         -1 if no match in raidtab
 */
int 
getmd(char *devname, char *mddev, char *mdpart, int nth)
{
    char buf[80];
    char rdev[75];
    int rlen, i, j;
    FILE *fd;
    char whspace[] = " \t\n";
    char raidfile[] = "/etc/raidtab";
    char raidstr[] = "raiddev";
    char devpattn[75];
    int  sdevpattn; 
    int  sraidstr;
    int  foundit;

    sraidstr = strlen(raidstr);
    strcpy(devpattn,devname);
    if (fdevfs) {
	char *tail;
	/* 
	 * Caller passes sdname (/dev/scsi/.../disc), but 
	 * pattern should not include "disc" or "generic".
	 */
	tail = strrchr(devpattn,'/');
	if (strcmp(tail,"/disc") == 0) *tail = 0;  /* truncate */
	else if (strcmp(tail,"/generic") == 0) *tail = 0; /* truncate */
	sdevpattn = strlen(devpattn);
    } else {
	sdevpattn = strlen(devpattn);
	if (sdevpattn >= 7)  /* swap sg -> sd if needed */
	   if (devpattn[6] == 'g') devpattn[6] = 'd';
    }
    if (fdebug) showlog("getmd: devpattn=%s, fdevfs=%d, nth=%d\n",
			devpattn,fdevfs,nth);
    fd = fopen(raidfile, "r");
    if (fd == NULL) return (errno);
    rdev[0] = 0;	/* default rdev (mddev) */
    strcpy(mdpart, "/dev/sdz1");	/* default mdpart */
    foundit = 0;
    while ((rlen = get_line(fd, buf, 80)) > 0) {	/* process each line */
	buf[rlen] = 0;		/* stringify */
	/* look for raiddev string */
	i = findmatch(buf, rlen, raidstr, sraidstr, 0);
	/* if so, get value for rdev */
	if (i >= 0) {
	    i += sraidstr;
	    j = strspn(&buf[i], whspace);
	    i += j;
	    j = strcspn(&buf[i], whspace);
	    strncpy(rdev, &buf[i], j);
	    rdev[j] = 0;
	}
	/* look for devpattn */
	i = findmatch(buf, rlen, devpattn, sdevpattn, 0);
	if (i >= 0) {		/* if so, stop and show mdpart */
	    foundit++;
	    j = strcspn(&buf[i], whspace);
	    strncpy(mdpart, &buf[i], j);
	    mdpart[j] = 0;
	    // strcat(rpart,mdpart);
	    // strcat(rpart," ");
	    if (foundit == nth) break;
	}
    }
    fclose(fd);
    if (foundit == nth) {
	strcpy(mddev, rdev);
	// strcpy(mdpart, rpart);
	return (0);
    } else {
	strcpy(mddev, rdev);
	return (-1);
    }
}				/*end getmd() */

/*
 * getnpart
 * Use raidtab to get the number of raid partitions.
 */
int getnpart(void)
{
   FILE *fd;
   char raidfile[] = "/etc/raidtab";
   char raidstr[] = "raiddev";
   int sraidstr;
   char buf[80];
   int i, rlen, np;

   np = 0;
   fd = fopen(raidfile, "r");
   if (fd == NULL) return (np);
   sraidstr = strlen(raidstr);
   while ((rlen = get_line(fd, buf, 80)) > 0) {	/* process each line */
	buf[rlen] = 0;		/* stringify */
	/* look for raiddev string */
	i = findmatch(buf, rlen, raidstr, sraidstr, 0);
	if (i >= 0) np++;
   }
   fclose(fd);
   return(np);
}

/*
 * mdstat
 * Scan /proc/mdstat to see if this device is active in
 * the raid or not.
 * returns 0 if devname is active, 
 *         1 if failed, 
 *        -1 if missing
 */
int 
mdstat(char *devname)
{
    char mdfile[] = "/proc/mdstat";
    char buff[LINESZ+1];
    char sdev[64];
    FILE *fp;
    char *pdev, *pb;
    int sz, i, j, dlen, szf;
    char *failstr;
    // char *mdstr="md";
    int failed, foundmd;
    
    failed = 0;  /*returns 0 if sdev is active in the raid*/
    foundmd = 0;
    fp = fopen(mdfile,"r");
    if (fp == NULL) return(errno);
    /* Input devname is of the form /dev/sda1, need to strip off path. */
    /* Skip the "/dev/" at the beginning of the sdev. */
    strcpy(sdev,devname);
    pdev = sdev;
    if (strncmp(sdev,"/dev/",5) == 0) pdev += 5;
    dlen = strlen(pdev);
    if (fdevfs) {  /* drop the trailing descriptor */
	if (strncmp(&pdev[dlen-4],"disc",4) == 0) pdev[dlen-4] = 0;
        dlen -= 4;
	failstr = "(faulty)";
    }
    else failstr = "(F)";
    szf = strlen(failstr);
    while (fgets(buff,LINESZ,fp)) {
	sz = strlen(buff);
	/* look on the line starting with "md" */
	if (buff[0] == 'm' && buff[1] == 'd') {
	   foundmd = 1;
	   if (fdebug) showlog("mdstat: sdev=%s, mdbuf=%s",pdev,buff);
	   /*
	    * If it had failed, the mdevt script should have removed it
	    * already, so if the string is there, it should be active.
	    * Let's check for "(F)" anyway, though.
	    */
	   i = findmatch(buff,sz,pdev,dlen,0);
	   if (i < 0) {   /* devname not found */
		if (fdebug) showlog("mdstat: %s missing\n",pdev);
		failed = -1;  /*missing*/
		break;
	   } else {  /* devname string is there */
		for (j = i; j < sz; j++) 
			if (buff[j] == ' ' || buff[j] == '\n') 
				{ buff[j] = 0; break; }
		pb = &buff[i];
		if (fdebug) 
			showlog("mdstat: devbuf=/%s/ len=%d\n",pb,j-i);
	   	i = findmatch(pb,j-i,failstr,szf,0);
		if (i >= 0) {
			if (fdebug) showlog("mdstat: %s failed\n",pdev);
			failed = 1;   /*shows failed*/
			break;
		} else {
			if (fdebug) showlog("mdstat: %s active\n",pdev);
			/* active, keep looking */
		}
	   }  /*endif devname is there*/
	}  /*endif md line match*/
    } /*end while*/
    if (foundmd == 0) failed = -1;  /*missing*/
    fclose(fp);
    return(failed);
}  /*end mdstat()*/

int 
add_scsi_dev(int bus, int ch, int id, int lun)
{
	char cmd[40];
	FILE *fp;

	/* 
	 * echo "add-single-device 0 0 3 0" >/proc/scsi/scsi
	 */
	fp = fopen("/proc/scsi/scsi","r+");
	if (fp == NULL) return(errno); 
	sprintf(cmd,"add-single-device %d %d %d %d",bus,ch,id,lun);
	fprintf(fp,cmd);
	fclose(fp);
	strcat(cmd,"\n");
	showit(cmd);
	return(0);
}

char device_there(int idev)
{
   int sgfd;
   int ret;
   uchar sdbuf[SGDATALEN];
   int j;

   /* if >= ndev, it didn't exist before. */
   if (idev >= ndev) return(0);  /* not there */

   /* assume device node is not already open */
   sgfd = open(devlist[idev].fname, O_RDWR | O_NONBLOCK);
   if (fdebug) printf("devthere [%d] %s open sgfd = %d\n",
			idev, devlist[idev].fname, sgfd);
   if (sgfd == -1) {
	return(0);  /* device node is not there */
   }

   /* See if device is ready */
   ret = sg_cmd(sgfd,turcdb,sizeof(turcdb),sdbuf,sizeof(sdbuf));
   if (fdebug) printf("devthere [%d] test_unit_ready ret = %d\n",idev,ret);
   if (ret < 0) {
	ret = sense_err(sgfd,sdbuf,sizeof(sdbuf),idev,ret,&j);
	if (fdebug) printf("devthere [%d] tur sense_err ret = %d\n",idev,ret);
   }
   close(sgfd);

   // if (ret >= 0) devlist[idev].dstatus = DSTAT_READY;
   // else devlist[idev].dstatus = DSTAT_NOTRDY;
   return(1);  /* device is there */
}  /* end device_there */

int
safte_inserted(int sgfd, int i, int hdev)
{
	char output[132];
	uchar sdbuf[SGDATALEN];
	int bus, ch, lun;
	int idev, j;
	int ret;
        
	if (!fsafte) return(0);
	/* sgfd and hdev pertain to the HSBP device, not the disk. */
	/* Translate slot# (i) into idev of the disk */
	idev = ndev;
	for (j = 0; j < ndev; j++) {
	  if ((slot[i].bus == devlist[j].bus) &&
	      (slot[i].id == devlist[j].tgt)) { idev = j; break; }
	}
	if (fdebug) showlog("slot %d slotid=%d, idev=%d devid=%d\n",
				i,slot[i].id,idev,devlist[idev].tgt);

	/* bus, chn, lun are same as HSBP */
	bus = devlist[hdev].bus;
	ch  = devlist[hdev].chn;
	lun = devlist[hdev].lun;
	sprintf(output,"*** NEW SAF-TE INSERTION [%d] in slot %d, id %d\n",
	                        idev,i,slot[i].id);
	showit(output);
	/* if orig scsi/aic driver, get data overrun here. */

	ret = sg_cmd(sgfd,turcdb,sizeof(turcdb),sdbuf,sizeof(sdbuf));
	if (fdebug) printf("hsbp test_unit_ready ret = %d\n",ret);

	if (!device_there(idev)) {  /*if device node isn't already there*/
		/* Do add-single-device now.  */
		ret = add_scsi_dev(bus,ch,slot[i].id,lun);
	}

	/* Do the actual mdevt insertion at the bottom */
	iinsert = idev;
	devlist[idev].dstatus = DSTAT_NOTRDY; /*not yet ready*/
        if (ret >= 0) ret = 0;
	return(ret);
}

int check_disk(int idev, int sgfd)
{
   int ret, j;
   int gdl;
   int fretry;
   uchar gdbuf[SGDATALEN];

	   /* 
	    * If the device is a DISK:
	    * Do a get_defects command for sanity, since 
	    * sometimes an inquiry will return ok, but other
	    * commands will not.
	    */
	      ret = sg_cmd(sgfd,defectcdb,sizeof(defectcdb),
			  gdbuf,sizeof(gdbuf));
	      if (ret < 0) {  
		  j = sense_err(sgfd,gdbuf,sizeof(gdbuf),idev,ret,&fretry);
		  if (fretry) 
	               ret = sg_cmd(sgfd,defectcdb,sizeof(defectcdb),
			            gdbuf,sizeof(gdbuf));
	      }
	      if (ret < 0) {
		if (ret == SCHECK_CND) {
		    sprintf(output,"get_defects: sense err %02x:%02x:%02x\n",
			    gdbuf[2],gdbuf[12],gdbuf[13]);
		    showit(output);
		} else if (fdebug) {
		    sprintf(output,"get_defects ret = %d\n",ret);
		    showit(output);
		}
	      }
	      else { 
		 gdl = ((gdbuf[2] << 8) | gdbuf[3]) / 8;
	         if (fdebug) printf("grown defects=%d\n",gdl);
	      }
    if (ret >= 0) ret = 0;
    return(ret);
}

int check_proc(int idev, int sgfd)
{
	int ret;
	int i,j, fretry;
	uchar sdbuf[SGDATALEN];

	/* 
	 * If the device is a PROCESSOR (Hot-swap backplane),
	 * issue a SAF-TE command to get the insertion counts.
	 */
        if (fses) return(-1);
        /* new proc device if dcap/nslots == 0 */
        if (devlist[idev].dcap == 0) slotoff = 0xff;

	      if (slotoff == 0xff) {  /* initialize safte stuff */
		/* 0 - Read Enclosure Configuration */
	        ret = read_safte(sgfd, 0, sdbuf, sizeof(sdbuf),idev);
		if (fdebug) showlog("read_safte/0 ret = %d\n",ret);
	        if (ret < 0) {
		  j = sense_err(sgfd,sdbuf,sizeof(sdbuf),idev,ret,&fretry);
		  if (fretry) 
	        	ret = read_safte(sgfd, 0, sdbuf, sizeof(sdbuf),idev);
		}
	        if (ret != 0) fsafte = 0;
	        else {   /* ret == 0 */
	         slotoff = sdbuf[0] + sdbuf[1];
	         nslots = sdbuf[2];
		 if (nslots > (MAX_DEVS - islot)) nslots = 6; /*sane nslots*/
		 pslot = &slot[islot];
                 devlist[idev].dcap = nslots; /*save nslots for this proc*/
		 fsafte = 1;
	         if (fdebug) {
	            sprintf(output,"read_safte/0 ok, sgfd=%d off=%d nslots=%d\n",
				sgfd,slotoff,nslots);
		    showit(output);
		 }
	         for (i=0; i<nslots; i++) {
	             pslot[i].num = i;
	             pslot[i].bus = devlist[idev].bus;
	             pslot[i].ins = 0xFFFF; /* init to max, fill in below*/
		 }
		 /* 1 - Read Enclosure Status */
	         ret = read_safte(sgfd, 1, sdbuf, sizeof(sdbuf),idev);
		 if (ret != 0) {
		    j = sense_err(sgfd,sdbuf,sizeof(sdbuf),idev,ret,&fretry);
		    if (fretry) 
	         	ret = read_safte(sgfd, 1, sdbuf, sizeof(sdbuf),idev);
		 } 
	         if (fdebug) showlog("read_safte/1 ret = %d\n",ret);
		 if (ret == 0) {
	            for (i=0; i<nslots; i++) 
	               pslot[i].id = sdbuf[slotoff + i];
	            if (fdebug) { 
			sprintf(output,
			       "read_safte/1 ok, offset=%d slotid[0,1]=%d,%d\n",
				slotoff,pslot[0].id, pslot[1].id);
			showit(output);
		    }
		    if (pslot[0].id >= pslot[1].id) {
			/*bogus HSBP correction*/
			pslot[0].id = 0;
			pslot[1].id = 1;
		    }
		    /* HSBP may not support Read_Device_Insertions, so try */
		    /* 4 - Read_Slot_Status.  */
	            ret = read_safte(sgfd, 4, sdbuf, sizeof(sdbuf),idev);
	            if (fdebug) 
	                printf("read_safte/4 Read_Slot_Status ret = %d\n",ret);
		    if (ret == 0) {
			uchar *p;
			slotmode = 4;
			for (i=0; i<nslots; i++) {
				p = &sdbuf[4*i];
				pslot[i].stat = p[3];
			        if (fdebug) {
				   sprintf(output,
				       "slot %d, id %d: %02x %02x %02x %02x\n",
					i,pslot[i].id,p[0],p[1],p[2],p[3]);
				   showit(output);
				}
			}
		    } else {  /* error in Read_Slot_Status */
		        j = sense_err(sgfd,sdbuf,sizeof(sdbuf),idev,ret,&fretry);
			/* 3- Read_Device_Insertions */
			ret = read_safte(sgfd, 3, sdbuf, sizeof(sdbuf),idev);
			if (fdebug) printf("read_safte/3 ret = %d\n",ret);
			if (ret == 0) {
			   slotmode = 3;
			   slotoff = 0; 
	                   for (i=0; i<nslots; i++) {
	             	      j = 2 * i;
			      pslot[i].ins = sdbuf[j] + (sdbuf[j+1] << 8);
			   }
			}
		    }
	          }  /* safte/1 is ok */
                  islot += nslots;
	        } /*safte/0 is ok */
	      }  /*endif initial pass*/

	      else {  /* already initialized, so just get safte slot status */
		ret = read_safte(sgfd, slotmode, sdbuf, sizeof(sdbuf),idev); 
	        if (ret < 0) {
		  j = sense_err(sgfd,sdbuf,sizeof(sdbuf),idev,ret,&fretry);
		  if (fretry) 
	              ret = read_safte(sgfd,slotmode,sdbuf,sizeof(sdbuf),idev);
		}
	      } /*end-else initialized*/

	      /* 
	       * Already have the slot status info, so
	       * fill in the slot array and analyze it. 
	       */
	      if (ret == 0) {
	         ushort tmpins;
	         uchar tmpstat;
	         if (fdebug) {
	           showlog("read_safte/%d ret ok\n",slotmode);
	           dump_log(sdbuf,sizeof(sdbuf),"safte_buf",1);
	           }
		 /* find pslot and nslots for this proc */
		 for (i=0; i<islot; i++) 
		   if (slot[i].bus == devlist[idev].bus) break;
		 pslot = &slot[i];
                 nslots = devlist[idev].dcap;
	         /* check for an insertion event in each slot */
		 if (slotmode == 4) {
	           for (i=0; i<nslots; i++) {
	             j = 4 * i;
	             tmpstat = sdbuf[j+3];
		     if (fdebug) {
			sprintf(output,
				"slot=%d, id=%d, %02x %02x %02x %02x stat=%x\n",
	                        i, pslot[i].id, sdbuf[j],sdbuf[j+1],
				sdbuf[j+2], tmpstat, pslot[i].stat);
			showit(output);
			}
		     if (tmpstat != pslot[i].stat) {  /* a change happened */
			/* if it was removed, and is now inserted, act. */
			if ((pslot[i].stat & 0x01) == 0 && (tmpstat & 0x01)) 
				ret = safte_inserted(sgfd,i,idev);
			pslot[i].stat = tmpstat;
		     }
		   }
		 } else {  /* slotmode == 3 */
	           for (i=0; i<nslots; i++) {
	             j = 2 * i;
	             tmpins = sdbuf[j] + (sdbuf[j+1] << 8);
		     if (fdebug) {
	               if (tmpins > 0) {
	                 sprintf(output,"slot %d, id %d had %d insertions\n",
	                        i,pslot[i].id,tmpins);
            		 showit(output);
			}
		     }
	             if (tmpins > pslot[i].ins) 
			ret = safte_inserted(sgfd,i,idev);
	             pslot[i].ins = tmpins;
	           }
	         }
	      } else {  /* read_safte error */
	         if (ret == SCHECK_CND) { 
	              sprintf(output,
				"check_proc: read_safte sense = %x/%x/%x\n",
				sdbuf[2], sdbuf[12], sdbuf[13]);
	         } else
	              sprintf(output,"check_proc ret = %d\n", ret);
	         showit(output);
	      }
	return(ret);
}  /*end check_proc*/

/* 
 * scandev
 * This function sends inquiries to each possible device and
 * collects the information needed to display the active devices.
 */
static int
scandev(int bmode)
{
   int i, j, idev;
   char fname[64];
   char sdname[64];
   char tmpname[64];
   uchar dbuf[SGDATALEN];
   int sgfd;
   int num_err;
   int ret = 0;
   int iret;
   DEVFS_LIST devfs_components = {0,0,0,0,0};

   /*
    * Iterate through the list of possible devices by number.
    */
   ndev = 0;
   nprocdev = 0;
   num_err = 0;
   for (idev = 0; (idev < MAX_DEVS) && (num_err < MAX_ERRS); idev++)
   {
	/* Open each corresponding sg device name */
	make_dev_name(idev,fname,sdname,do_numeric, &devfs_components);
	if (devfs_components.all_leaves_searched != 0) {
		devfs_components.all_leaves_searched = 0;
		break;
	}
	if (nprocdev > 0) {
	   /* Proc devices don't use an sdname, so get 
	    * the previous block device name that will match. */
	   make_dev_name((idev-nprocdev),tmpname,sdname,do_numeric, 
			 &devfs_components);
	}
	devlist[idev].dstatus = DSTAT_DEAD;  /*initial value*/
	sgfd = open(fname, O_RDWR | O_NONBLOCK);
	if (sgfd < 0) {
	   if (fdebug) {
		if (errno == ENXIO) showlog("%s no device\n",fname);
		else showlog("%s open errno: %d\n",fname,errno);
	   }
	   if (errno == EBUSY) continue;
	   else if (errno == ENODEV || errno == ENOENT 
		    || errno == ENXIO) {
                        ++num_err;
                        continue;
	   }
	   else {  /* EACCES or other, needs attention */
		if (fdebug)
			showlog("Error %d opening %s\n",errno,fname);
                        ++num_err;
                        continue;           
	   }
	} else {  /* opened ok, the device exists (or did exist) */
	   struct { 
	       int dev_id;
	       int host_unique_id; 
	       } idlun;
	   int bus,ch,id,lun;
	   int devtype = 0;
	   int fretry;
	   int f, ilen, slen;
	   int emul;

	   strcpy(devlist[idev].fname,fname);
	   strcpy(devlist[idev].sdname,sdname);
	   devlist[idev].sernum[0] = 0;
	   devlist[idev].fwrev[0] = 0;
	   ret = ioctl(sgfd, SCSI_IOCTL_GET_BUS_NUMBER, &bus);
	   if (ret < 0) ret = SIOCTL_ERR;
	   if (ret >= 0) 
	 	ret = ioctl(sgfd, SCSI_IOCTL_GET_IDLUN, &idlun);
	   if (ret >= 0) 
	 	ret = ioctl(sgfd, SG_EMULATED_HOST, &emul);
	   if (ret < 0) { 
		ret = SIOCTL_ERR; 
		goto abortsg; 
	   } 
	   ch  = (idlun.dev_id >> 16) & 0xff;
	   id  = idlun.dev_id & 0xff;
	   lun = (idlun.dev_id >> 8) & 0xff;
	   if (bmode == 1) {
	      f = fcntl(sgfd, F_GETFL);
	      f = f & (~O_NONBLOCK);  /* set to blocking mode */
	      ret = fcntl(sgfd, F_SETFL, f);      
	   }
	   devlist[idev].bus = bus;
	   devlist[idev].chn = ch;
	   devlist[idev].tgt = id;
	   devlist[idev].lun = lun;

	   /* 
	    * Do a SCSI inquiry and save the info. 
	    */
	   ilen = sizeof(dbuf)-1;
           memset(dbuf,0,sizeof(dbuf));
	   ret = sg_cmd(sgfd,inqcdb,sizeof(inqcdb),dbuf,sizeof(dbuf));
	   if (fdebug) showlog("inquiry ret=%d, errno=%d\n", ret,errno);
	   if (ret < 0) {   
		f = sense_err(sgfd,dbuf,sizeof(dbuf),idev,ret,&fretry);
		if (fretry) {
                   memset(dbuf,0,sizeof(dbuf));
	           ret = sg_cmd(sgfd,inqcdb,sizeof(inqcdb),
			        dbuf,sizeof(dbuf));
                }
	   }  /*endif sense err*/
           if (ret >= 0) iret = 0;
           else iret = ret;
	   devtype = dbuf[0] & 0x1f;
	   if (iret == 0) {
	      int fbad;
	      uchar ci;
              int ftrunc = 1;    /*ARC*/
	      ilen = dbuf[4];
	      fbad = 0;
              if (ilen > 36 && ilen < 44) ilen = 46; /*Fix Fujitsu fw bug*/
              if (ilen < 36 && ilen < ret) ilen = ret;
	      if (ilen >= sizeof(dbuf)) ilen = sizeof(dbuf)-1;
              if ((ilen < 24) || (devtype > DEVT_MAX)) { 
                 j = 32; fbad = 1; 
              } else for (j = 8; j < ilen; j++) {
		 ci = dbuf[j];
		 if (ci > 0x7f) { 
		    if (fdebug) 
		       showlog(
			      "[%d] bad inquiry char 0x%02x (%c), offset=%d\n",
			      idev, ci, ci, j);
		    if (ftrunc && j > 24) {   /* truncate it with no error */
		       dbuf[j] = 0;
		       ilen = j;
		       break;
		    } else {        /* have bad inquiry data */
		       fbad = 1; 
		       break;
		    }
		}
	      } 
	      if (fbad) {
		 /* gets here if standby (not-active) disk is removed */
                 if (fdebug)
                     showlog("Inquiry error: len=%d type=%d badc=%02x\n",
                                        ilen,devtype,dbuf[j]);
                 iret = SIZE_ERR;
                 ilen = j; /*truncate, was 24*/
	      }
	      if (ilen > 36) dbuf[ilen] = 0; /*stringify*/
              else dbuf[36] = 0;
	      if (fdebug)
		dump_log(dbuf,ilen,"inq buffer",1);
	   } else if (fdebug && iret == SCHECK_CND) {
	 	sprintf(output,"inq: scsi sense err %02x:%02x:%02x\n",
			   dbuf[2],dbuf[12],dbuf[13]);
		showit(output);
	   }

         ret = iret;
         if (iret == 0) {  /* only check more if inquiry was ok */
             if (emul) { devtype = DEV_EMUL; } /*Emul = 10., usu usb-storage*/
             if (devtype >= NDEVTYP) devtype = NDEVTYP - 1;  /*Unkn*/
             if (devtype == DEV_PROC || devtype == DEV_ENCL) {
                 /* proc, HotSwap backplane */
                 devlist[idev].devtype = devtype;
                 fillstr(devlist[idev].vendor,&dbuf[8],8);
                 ret = check_proc(idev,sgfd);
                 if (ret != 0) {   /* fsafte==0 */
                    if ((dbuf[6] & ENCSERV) || (devtype == DEV_ENCL))
                       ret = check_ses(idev,sgfd);
                 }
             } else if (devtype == DEV_DISK) {          /* disk */
                 ret = check_disk(idev,sgfd);
             }
         }
	 if (fdebug) {
		/* display the device */
		sprintf(output,"dev[%d] iret=%d, ret=%d, len=%d %d:%d:%d:%d %s %s %s\n",
		       idev,iret,ret,ilen,bus,ch,id,lun,fname,sdname,&dbuf[8]);
		showit(output);
	 }
	 /* 
	  * Now update the device list info, if valid 
	  */
	 if (iret < 0) {
		devlist[idev].ret = iret;
		devlist[idev].dstatus = DSTAT_FAIL;  /*it is failed*/
		devlist[idev].sernum[0] = 0;
	 } else {    // If here, valid, so update the devlist
		// dbuf[ilen] = 0;  /* end of buffer */
		if (devtype >= NDEVTYP) devtype = NDEVTYP - 1;  /*Unkn*/
		devlist[idev].devtype = devtype;
		devlist[idev].ret = ret;
		if (ret == 0) devlist[idev].dstatus = DSTAT_READY;  /*ready*/
                else if (ret == SCHECK_CND) 
			devlist[idev].dstatus = DSTAT_NOTRDY; /*not yet ready*/
                else 	devlist[idev].dstatus = DSTAT_FAIL; 
		if (iret == 0) {
		  fillstr(devlist[idev].vendor,&dbuf[8],8);
		  fillstr(devlist[idev].model,&dbuf[16],16);
		  fillstr(devlist[idev].fwrev,&dbuf[32],4);
		  slen = sernum_len;  /*set default from global value*/
		  for (i=0; i < NVEND12; i++) {
		    if (strncmp(devlist[idev].vendor,vendor12[i],
				strlen(vendor12[i])) == 0)
			 slen = 12;
		  }
		  if (dbuf[44] <= ' ') slen = 8;
		  if ((ilen > 44) && (dbuf[36+8] > ' ') && 
		      (devtype == DEV_DISK)) 
			 slen = 12; /*apparently a long sernum disk*/
		  strncpy(devlist[idev].sernum,&dbuf[36],slen);
		  devlist[idev].sernum[slen] = 0;
		  /* could validate sernum here */
		  if (ilen >= 54) {
		    strncpy(devlist[idev].other,&dbuf[44],10);
		    devlist[idev].other[10] = 0;
	          }
		} /*endif iret==0*/
	 } 
	 /* check the status of this disk in the raid */
	 if (devtype == DEV_DISK && iret == 0) {  /*if it is a live disk*/
		char mdev[75];
		char mpart[80];
		/* is it configured in the raidtab? */
		devlist[idev].mdtab = getmd(devlist[idev].sdname,mdev,mpart,1);
		/* is it active in the raid? */
		devlist[idev].sdactive = mdstat(devlist[idev].sdname);
		if (fdebug) 
		   showlog("dev[%d] getmd = %d mdstat = %d\n",idev,
			   devlist[idev].mdtab, devlist[idev].sdactive);
	 } else {  /* either inquiry error, or not a disk */
		/* TODO: should we call md routines here sometimes 
		 * if sdname is valid?  */
		devlist[idev].mdtab = -1;
		devlist[idev].sdactive = -1;
	 }
         abortsg: 
	    ndev++;
	    if ((devtype == DEV_PROC) || (devtype == DEV_ENCL)) nprocdev++;
	    close(sgfd);
	    if (ret != 0) {
		showlog("scandev: device %d error=%d, errno=%d\n", 
			idev,ret,errno);
	    }
	} /*endif good open*/
   }  /*end for each device*/

   /* set up the next fname */
   strcpy(devlist[ndev].fname,fname);
   return ret; 
}  /* end scandev */

/* 
 * mkdaemon
 * Called when sgraidmon needs to become a daemon/background process.
 */
int 
mkdaemon(int fchdir, int fclose)
{
    int fdlimit = sysconf(_SC_OPEN_MAX);
    int fd = 0;
 
    switch (fork()) {
        case 0:  break;
        case -1: return -1;
        default: _exit(0);          /* exit the original process */
    }
    if (setsid() < 0) return -1;    /* shouldn't fail */
    switch (fork()) {
        case 0:  break;
        case -1: return -1;
        default: _exit(0);
    }
    if (fchdir) chdir("/"); 
    if (fclose) {
	/* Close stdin,stdout,stderr and replace them with /dev/null */
	while (fd < fdlimit) close(fd++);
	open("/dev/null",O_RDWR);
	dup(0); dup(0);
    }
    return 0;
}

/* 
 * do_insert
 * Handle hot-inserted disks.
 */
int do_insert(int iinsert)
{
	int ret = 0;
	int i, j;
	// int jslot = MAX_DEVS;
/*	 int c;
	char mddev[75];
	char mdpart[80]; */

	ret = 0;
	if (iinsert != MAX_DEVS) {
	   if (fdebug)
	      showlog("do_insert/pre: i=%d dstat=%d ret=%d type=%d\n",
			iinsert,devlist[iinsert].dstatus,devlist[iinsert].ret,
			devlist[iinsert].devtype);
	   if ((devlist[iinsert].dstatus <= DSTAT_FAIL) || 
	       (devlist[iinsert].devtype == DEV_PROC)  ||
	       (devlist[iinsert].devtype == DEV_ENCL)) iinsert = MAX_DEVS; 
	}
	if (iinsert == MAX_DEVS) {  /*iinsert not set*/
	     /* find which one was inserted */
             for (i = 0; i < ndev; i++) {  
		/* See if this disk position came ready again */
		if (devlist[i].dstatus > devmatch[i].dstatus) {
			/* This one was dead and is back alive again */
		   	if (fdebug) 
			  showlog("do_insert/st: i=%d dstat=%d ret=%d\n",
				i,devlist[i].dstatus,devlist[i].ret);
			iinsert = i; 
			break;  /* found it, done */
		     }
		/* See if this is a new serial number in devlist */
		if (devlist[i].sernum[0] != 0) {
		   for (j = 0; j < ndev; j++) {
		      if (strcmp(devlist[i].sernum,devmatch[j].sernum)==0) 
		   	 break;  /*matches, so not new*/
		   }
	           if (j >= ndev) {  /*wasn't in the list before*/
		      if (fdebug) 
			showlog("do_insert/sn: i=%d dstat=%d ret=%d\n",
				i,devlist[i].dstatus,devlist[i].ret);
		      iinsert = i; 
		      break; 
		   }
		}
	     } /*end for i*/
	}  /*endif iinsert not set */
	/* failsafe check for wrong iinsert */
	if ((devlist[iinsert].dstatus <= DSTAT_FAIL) || 
	       (devlist[iinsert].devtype == DEV_PROC)  ||
	       (devlist[iinsert].devtype == DEV_ENCL)) iinsert = MAX_DEVS; 

	if (iinsert != MAX_DEVS) {
              sprintf(output,"insert: disk %d was inserted.\n",iinsert);
              showit(output);
	      if (fsafte) {
	        /* if SAFTE HSC, write slot status */
	        ret = write_slots(iinsert,0x01);
	        ret = 0;
	      } else if (fses) {
	        ret = ses_write_slots(iinsert,0x01);
	        ret = 0;
              }
	}
	return(ret);
} /*end do_insert*/

/*
 * do_remove
 * Handle hot-removed disks.
 */
int do_remove(int iremove)
{
	int ret = 0;
	int i; //, j;
/*	int c;
	char mddev[75];
	char mdpart[80]; */

	if (iremove == MAX_DEVS) 
              for (i = 0; i < ndev; i++) {
		if (devlist[i].devtype == DEV_DISK &&   /*disk*/
		    (devlist[i].dstatus <= DSTAT_FAIL))
			{ iremove = i; break; } 
	      }
	if (iremove >= ndev) {
               showit("remove: no disk to remove\n");
	} else {    /* (iremove < ndev) */
	       // int jslot = MAX_DEVS;
	       i = iremove;
               sprintf(output,"remove: sdname = %s, dstat = %d, idev = %d\n",
	  	       devlist[i].sdname,devlist[i].dstatus,i);
	       showit(output);
	       if (fsafte) {
	         /* if SAFTE HSC, write slot status */
	         ret = write_slots(i,0x02);
	         ret = 0;  // continue even if write_slots error 
	       } else if (fses) {
	         ret = ses_write_slots(i,0x02);
	         ret = 0;
	       }
	}
	return(ret);
} /*end do_remove*/

/* 
 * check_devstatus
 * This routine checks each device to see what status it really has.
 * It tracks the previous set of information in devmatch and uses
 * the serial number for uniqueness.
 */
void check_devstatus(void)
{
	int i, j, k, c;
        nrdy = 0;
	for (i = 0; i < ndev; i++) {
	   if (lastnrdy != MAX_DEVS) {  /*not first time*/
	     if (strcmp(devlist[i].sernum,devmatch[i].sernum)!=0) {
		/* 
		 * Serial number doesn't match what it was before.
		 * This mismatch could either be an impostor or a real 
		 * replacement.  Assume that it is a real replacement 
		 * unless the serial number is missing.
		 * I saw one impostor return all blanks for sernum, 
		 * but some drives may have leading blanks on a valid
		 * sernum of 4 digits.  If all blanks, invalid.
		 */
		if (devlist[i].sernum[0] == 0) devlist[i].dstatus = DSTAT_FAIL;
		for (j=0;j<sernum_len;j++)  /* sernum missing, all spaces? */
			if (devlist[i].sernum[j] != 0x20) break;
		if (j == sernum_len) devlist[i].dstatus = DSTAT_FAIL;
		if (fdebug) {
	          sprintf(output,"disk %d sernum changed, sn0=%x oldsn0=%x dstat=%d iremove=%d\n",
			  i,devlist[i].sernum[0],devmatch[i].sernum[0],
			  devlist[i].dstatus,iremove);
	          showit(output);
		}
		if (devlist[i].dstatus == DSTAT_FAIL) {
			/* found an impostor, failed */
			devlist[i].devtype = DEV_DISK; /* assume failed disk */
			iremove = i;
		}
	     } /*endif mismatch*/
	   } /*endif not first time*/
	   /* 
	    * Check if a disk is shown with a different serial number than 
	    * before.  If so, it is probably removed, and one of the other
	    * disks is answering in its place (duplicate).  Mark it as dead. 
	    * Note that this assumes that you won't remove an active disk
	    * and then re-insert the same one with a different SCSI ID.
	    */
	   for (j = 0; j < ndev; j++) {
		/* 
		 * Check if there is another device listed with same sernum.
		 * Non-disks can also appear here as impostors, and 
		 * those serial numbers are not always globally unique, 
		 * so also match by model. 
		 */
		k = strcmp(devlist[i].model,devlist[j].model);
		if (k==0 && strcmp(devlist[i].sernum,devlist[j].sernum)==0) {
		   if ((j != i) &&     /*two different devices match*/
		       (devlist[i].devtype != DEV_PROC) && 
		       (devlist[i].devtype != DEV_ENCL) )
		   {      /* mismatch, and not a proc */
		        int idup;
		      /*
		       * If same serial number appears twice, mark the one
		       * that is out of place as dead.  Use mdstat to find 
		       * out which one is the impostor.
		       */
			idup = MAX_DEVS;
			if (fdebug) showlog(
				"dup_chk: [%d] ret=%d mdstat=%d [%d] ret=%d mdstat=%d\n", 
				i,devlist[i].ret,devlist[i].sdactive, 
				j,devlist[j].ret,devlist[j].sdactive);
	        	if (strcmp(devlist[j].sernum,devmatch[j].sernum)!=0)
			      idup = j;   /*j sernum is different*/
	        	else if(strcmp(devlist[i].sernum,devmatch[i].sernum)!=0)
			      idup = i;   /*i sernum is different*/
			else if (devlist[j].ret != 0)   /*bad inquiry*/
			      idup = j; 
			else if (devlist[i].ret != 0)  /*bad inquiry*/
			      idup = i;
			else if ((devlist[j].mdtab == 0) && 
				 (devlist[j].sdactive != 0)) idup = j;
			else if ((devlist[i].mdtab == 0) &&
				 (devlist[i].sdactive != 0)) idup = i;
			if (idup != MAX_DEVS) {
			   /* idup is the impostor, mark it dead */
			   devlist[idup].dstatus = DSTAT_FAIL;
	        	   devlist[idup].vendor[0] = 0;
	        	   devlist[idup].model[0] = 0;
	        	   devlist[idup].fwrev[0] = 0;
	        	   devlist[idup].sernum[0] = 0;
			   devlist[idup].devtype = DEV_DISK; /* assume failed disk */
			   iremove = idup;
			   if (fdebug) {
			      sprintf(output,"device %d is duplicate\n",idup);
			      showit(output);
			   }
			}  /*endif found impostor*/
		   }  /*endif j!=i*/
		}  /*endif sernums match*/
	   } /*end for j*/
	   /*
	    * Check this disk's status in the raid.
	    */
	   if (devlist[i].devtype == DEV_DISK) {   /* is a disk */
		k = devlist[i].sdactive; 
		if (k == 0 && 			/* active in raid */
		    devlist[i].dstatus >= DSTAT_READY) /* and ready */
		    devlist[i].dstatus = DSTAT_ONL;  /* mark it online */
		else if (k == 1) { 			/* failed in raid */
			/* decided above if it is failed, or newly inserted */
			if (fdebug) {
			   sprintf(output,
				"dev=%d %s shows faulty status=%x ret=%d\n",
				i,devlist[i].sdname,devlist[i].dstatus,
				devlist[i].ret);
			   showit(output);
			}
		}
	     
	      if (lastnrdy == MAX_DEVS) { /*first time*/
		/* 
		 * Check if this disk has become ready since the last
		 * reboot and should be inserted as part of the raid.  
		 * This is important if the raid booted up degraded.
		 */
		c = devlist[i].mdtab;
		if (fdebug) showlog("1st: dev %d dstat=%d mdtab=%d sdactive=%d\n",i,devlist[i].dstatus,c,k);
	       if (devlist[i].dstatus >= DSTAT_READY) {  /* disk is ready*/
		   /* mdtab=0 if it is configured in the raid */
		   if (c == 0 && k != 0) {
		     /* configured, working, but not active in raid */
		     nchgs++;
		     iinsert = i;
		   }
		
		 }
	      } /*endif first time*/
	   }  /*endif is a disk*/

	   if (lastnrdy != MAX_DEVS && 
	       devlist[i].dstatus != devmatch[i].dstatus) nchgs++;
           /* ndev may include some that are offline or dead, so count nrdy. */
           if (devlist[i].dstatus > DSTAT_NOTRDY) nrdy++;
	}  /*end check devices loop*/
	return;
}  /*end check_devstatus*/

int
main(int argc, char **argv)
{
   int ret;
   int i, c;
   ulong ltime1;
   int bmode = 1;
   char *pfwrev;
   char *psernum;
   char *pvendor;
   char *pmodel;
   char *psdname;

   fdmsg = stdout;
   sprintf(output2,"\n     %s utility v%s for SCSI SAF-TE testing\n",
		progname, progver);

   while ((c = getopt(argc, argv, "erfmnd:x?")) != EOF) {
        switch (c) {
        case 'x':
            fdebug = 1;
            break;
        case 'm':
            moresn = 1;
	    sernum_len = 12;
            break;
        case 'e':
            filesok = 0;
            break;
        case 'r':
            setval = 0x01;  /*inserted, ready*/
            break;
        case 'f':
	    setval = 0x02;  /*inserted, faulty*/
            break;
        case 'd':
            drivenum = atoi(optarg);
            break;
        case 'n':
            do_numeric = 0;
            break; 
	default:
            printf(output2);    /*prog header*/
            printf("Usage: %s [-d num -rf -emnx]\n",progname);
	    printf(" -d  drive number to set (0, 1, ...)\n");
	    printf(" -r  set SAF-TE status to ready\n");
	    printf(" -f  set SAF-TE status to faulty\n");
	    printf(" -e  Do not write to any files.\n");
	    printf(" -m  Use More than 8-byte serial numbers (12 bytes)\n");
	    printf(" -n  Turn off numeric device names, use alpha instead\n");
	    printf(" -x  Show eXtra debug messages\n");
            exit(1); 
	}
   }
   if (!background) printf(output2);  /*prog header*/

   if (setval != 0 && drivenum == -1) drivenum = 0;  /*default to first drive*/

   /* only run this as superuser */
   i = geteuid();
   if (i > 1) {
        printf("%s: Not superuser (%d)\n", progname,i);
        exit(1);
   } 

   /* open log file */
   if (filesok)
        fdlog = fopen(logfile, "a+");
   if (fdlog == NULL) {
        flogopen = 0;
        filesok = 0;
        fdlog = stderr;
   } else {                    /* logfile open successful */
        flogopen = 1;
        time((time_t *) & ltime1);
        showlog( "\n--- %s v%s log started at %s", progname, progver,
                ctime((long *)&ltime1));   // ctime outputs a '\n' at end
	showlog(output2);  /* prog header */
   }
   npart = getnpart();         /* get number of raid partitions */
   if (fdebug) printf("main: npart=%d\n",npart);
   memset(devlist,0,sizeof(devlist));
   while (1) 
   {
        nchgs = 0;
	/* 
	 * scandev gets the scsi data from the devices, 
	 * sets ndev, and fills devlist.
	 */
	ret = scandev(bmode);
	if (fdebug || ndev == 0) printf("scan ret = %d\n",ret);
	bmode = 0;

	/*
	 * check_devstatus scans the devlist for unique serial
	 * numbers and matches against the previous snapshot.
	 */
	check_devstatus();

	/*
	 * Now, if there have been any changes since the previous snapshot,
	 * display the new list of devices.
	 */
	if (fdebug) {
           sprintf(output,"nchgs=%d nrdy=%d lastnrdy=%d ndev=%d\n",
			nchgs,nrdy,lastnrdy,ndev);
	   showit(output);
	}
	if (lastnrdy == MAX_DEVS || nchgs > 0 || lastnrdy != nrdy) {
           /* Something changed.  Display header */
	   sprintf(output,
	    "Num Name      bus:ch:id:lun Type Vendor   Device_Model     FRev Serial#  Status\n");
	   showit(output);
	   for (i = 0; i < ndev; i++) {
	      char *tmpl;
	      if (devlist[i].vendor[0] == 0) pvendor = blank8;
	      else pvendor = devlist[i].vendor;
	      if (devlist[i].model[0] == 0) pmodel = blank16;
	      else pmodel = devlist[i].model;
	      if (devlist[i].fwrev[0] == 0) pfwrev = blank4;
	      else pfwrev = devlist[i].fwrev;
	      if (devlist[i].sernum[0] == 0) psernum = blank8;
	      else psernum = devlist[i].sernum;
	      if ((devlist[i].devtype == DEV_PROC) ||
	         (devlist[i].devtype == DEV_ENCL)) psdname = blank8;
	      else psdname = devlist[i].sdname;
	      /* Display each device */
	      if (fdevfs) 
		 tmpl = "%d %s\n  %s\n\t\t    %d:%d:%d:%d %s %s %s %s %s %s\n";
	      else tmpl = "%d %s %s %d:%d:%d:%d %s %s %s %s %s %s\n";
	      /* TODO: if (devlist[i].fses) show something */
	      sprintf(output,tmpl, i,
	              devlist[i].fname, psdname, devlist[i].bus,
		      devlist[i].chn, devlist[i].tgt, devlist[i].lun,
	       	      devtype_array[devlist[i].devtype],
		      pvendor, pmodel, pfwrev, psernum, 
		      dstat_str[devlist[i].dstatus]);
	      showit(output);
	   }
	}

	/* 
	 * Decide if there is any action to take.
	 * If first time, no action unless insert needed.
	 * If hot-removal, invoke mdevt Fail.
	 * If hot-insert, invoke mdevt Insert.
	 */
	if ((lastnrdy == MAX_DEVS && iinsert != MAX_DEVS) &&
	    (devlist[iinsert].ret == 0)) {
		/* iinsert is set, and that device is alive. */
		if (fdebug) {
	          sprintf(output,"Need to insert %s\n",devlist[iinsert].sdname);
	          showit(output);
		}
		lastnrdy = nrdy - 1; 
	}
        if (lastnrdy == MAX_DEVS) {     /* first time */
            sprintf(output,"monitoring %d scsi devices ...\n", nrdy);
	    showit(output);
	    if (ndev == 0) {
		sprintf(output,"Cant open any sg scsi devices, errno = %d\n",
		        errno);
		if (errno == ENODEV) strcat(output,"Make sure CONFIG_CHR_DEV_SG is set in /usr/src/linux/.config\n");
		else if (errno == ENOENT) strcat(output,"Check /dev/sg* device names\n");
                strcat(output,"Try 'modprobe sg' if not already loaded\n"); 
	        showit(output);
		quit(1);
	    }
        } 

	/* 
	 * Set the safte slot state, if user specified it.
	 */
        if (setval != 0) {
	   // slot[drivenum].wstat = setval;
	   if (fsafte) {  
	      ret = write_slots(drivenum,setval);
	      sprintf(output,"Set SAF-TE state to %02x for drive %d, rc = %d\n",
		   setval,drivenum,ret);
	      showit(output);
           } else if (fses) {
	      ret = ses_write_slots(drivenum,setval);
	      sprintf(output,"Set SES state to %02x for drive %d, rc = %d\n",
		   setval,drivenum,ret);
	      showit(output);
           }
	}

	/* 
	 * Save the current good snapshot for the next pass.
	 */
	lastnrdy = nrdy;
        for (i = 0; i < ndev; i++) {  
	   if (devlist[i].dstatus > DSTAT_FAIL) {
	      if (fdebug && devlist[i].dstatus != DSTAT_ONL) 
		 printf("dev %d stat=%d, saving\n",i,devlist[i].dstatus);
	      strcpy(devmatch[i].vendor,devlist[i].vendor);
	      strcpy(devmatch[i].model, devlist[i].model);
	      strcpy(devmatch[i].fwrev, devlist[i].fwrev);
	      strcpy(devmatch[i].sernum,devlist[i].sernum);
	   }
	   /* Copy the dstatus, even if 0 */
	   devmatch[i].dstatus = devlist[i].dstatus;
	}
        fflush(fdlog);

        quit(0);
   }                           /*end while(1) */
}  /* end main */

/* end sgsafte.c */
