/*
 * Copyright 2016-2017 Józef Kucia for CodeWeavers
 * Copyright 2020-2021 Philip Rebohle for Valve Corporation
 * Copyright 2020-2021 Joshua Ashton for Valve Corporation
 * Copyright 2020-2021 Hans-Kristian Arntzen for Valve Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#define VKD3D_DBG_CHANNEL VKD3D_DBG_CHANNEL_API
#include "d3d12_crosstest.h"

static void run_64bit_atomics_test(struct test_context *context,
        D3D12_SHADER_BYTECODE cs,
        bool use_heap, bool use_typed)
{
    static const uint64_t inputs[] = { 1ull << 40, 3ull << 31, 3ull << 29, 1ull << 63 };
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_DESCRIPTOR_RANGE descriptor_range[2];
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D12_ROOT_PARAMETER root_parameters[3];
    ID3D12RootSignature *root_signature;
    ID3D12Resource *output_texture;
    ID3D12Resource *output_buffer;
    D3D12_CPU_DESCRIPTOR_HANDLE h;
    ID3D12Resource *input_buffer;
    uint64_t expected_values[11];
    struct resource_readback rb;
    ID3D12DescriptorHeap *heap;
    ID3D12PipelineState *pso;
    const uint64_t *values;
    unsigned int i, j;

    input_buffer = create_upload_buffer(context->device, sizeof(inputs), inputs);
    output_buffer = create_default_buffer(context->device, sizeof(expected_values),
            D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    output_texture = create_default_texture2d(context->device,
            ARRAY_SIZE(expected_values), ARRAY_SIZE(expected_values),
            1, 1, DXGI_FORMAT_R32G32_UINT,
            D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    expected_values[0] = inputs[0] + inputs[1] + inputs[2] + inputs[3];
    expected_values[1] = ~inputs[0] & ~inputs[1] & ~inputs[2] & ~inputs[3];
    expected_values[2] = inputs[0] | inputs[1] | inputs[2] | inputs[3];
    expected_values[3] = inputs[3];
    expected_values[4] = inputs[2];
    expected_values[5] = inputs[0];
    expected_values[6] = inputs[3];
    expected_values[7] = ~0ull ^ inputs[0] ^ inputs[1] ^ inputs[2] ^ inputs[3];

    heap = NULL;

    if (use_heap)
    {
        heap = create_gpu_descriptor_heap(context->device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 3);
        memset(&root_signature_desc, 0, sizeof(root_signature_desc));
        memset(root_parameters, 0, sizeof(root_parameters));
        memset(descriptor_range, 0, sizeof(descriptor_range));
        root_signature_desc.NumParameters = 1;
        root_signature_desc.pParameters = &root_parameters[0];
        root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
        root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
        root_parameters[0].DescriptorTable.NumDescriptorRanges = ARRAY_SIZE(descriptor_range);
        root_parameters[0].DescriptorTable.pDescriptorRanges = descriptor_range;
        descriptor_range[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
        descriptor_range[0].NumDescriptors = 1;
        descriptor_range[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;
        descriptor_range[1].NumDescriptors = 2;
        descriptor_range[1].BaseShaderRegister = 0;
        descriptor_range[1].OffsetInDescriptorsFromTableStart = 1;

        memset(&srv_desc, 0, sizeof(srv_desc));
        srv_desc.Format = DXGI_FORMAT_UNKNOWN;
        srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        srv_desc.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
        srv_desc.Buffer.FirstElement = 0;
        srv_desc.Buffer.NumElements = 4;
        srv_desc.Buffer.StructureByteStride = 8;
        h = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
        ID3D12Device_CreateShaderResourceView(context->device, input_buffer, &srv_desc, h);

        memset(&uav_desc, 0, sizeof(uav_desc));
        h = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);

        if (use_typed)
        {
            uav_desc.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;
            uav_desc.Format = DXGI_FORMAT_R32G32_UINT;
            h.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(context->device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
            ID3D12Device_CreateUnorderedAccessView(context->device, output_texture, NULL, &uav_desc, h);

            uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
            memset(&uav_desc.Buffer, 0, sizeof(uav_desc.Buffer));
            uav_desc.Buffer.NumElements = ARRAY_SIZE(expected_values);
            h.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(context->device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
            ID3D12Device_CreateUnorderedAccessView(context->device, output_buffer, NULL, &uav_desc, h);
        }
        else
        {
            uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
            uav_desc.Buffer.FirstElement = 0;
            uav_desc.Buffer.NumElements = ARRAY_SIZE(expected_values);
            uav_desc.Format = DXGI_FORMAT_UNKNOWN;
            uav_desc.Buffer.StructureByteStride = 8;

            h.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(context->device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
            ID3D12Device_CreateUnorderedAccessView(context->device, output_buffer, NULL, &uav_desc, h);
            h.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(context->device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
            ID3D12Device_CreateUnorderedAccessView(context->device, output_buffer, NULL, &uav_desc, h);
        }
    }
    else
    {
        memset(&root_signature_desc, 0, sizeof(root_signature_desc));
        memset(root_parameters, 0, sizeof(root_parameters));
        root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
        root_signature_desc.pParameters = root_parameters;
        root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
        root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
        root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
        root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
        root_parameters[2].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
        root_parameters[2].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
        root_parameters[2].Descriptor.ShaderRegister = 1;
    }

    create_root_signature(context->device, &root_signature_desc, &root_signature);
    pso = create_compute_pipeline_state(context->device, root_signature, cs);
    if (heap)
        ID3D12GraphicsCommandList_SetDescriptorHeaps(context->list, 1, &heap);
    ID3D12GraphicsCommandList_SetComputeRootSignature(context->list, root_signature);
    if (heap)
    {
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context->list, 0,
                ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap));
    }
    else
    {
        ID3D12GraphicsCommandList_SetComputeRootShaderResourceView(context->list, 0,
                ID3D12Resource_GetGPUVirtualAddress(input_buffer));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context->list, 1,
                ID3D12Resource_GetGPUVirtualAddress(output_buffer));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context->list, 2,
                ID3D12Resource_GetGPUVirtualAddress(output_buffer));
    }
    ID3D12GraphicsCommandList_SetPipelineState(context->list, pso);
    ID3D12GraphicsCommandList_Dispatch(context->list, 1, 1, 1);
    transition_resource_state(context->list, output_buffer,
              D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(output_buffer, 0, &rb, context->queue, context->list);
    reset_command_list(context->list, context->allocator);

    values = get_readback_data(&rb, 0, 0, 0, 1);
    for (i = 0; i < 8; i++)
    {
        if (use_typed && i != 5 && i != 6)
            continue;

        ok(values[i] == expected_values[i], "Value %u = 0x%"PRIx64", expected 0x%"PRIx64"\n",
                i, values[i], expected_values[i]);
    }

    /* We're spamming exchanges or compare exchanges. There is only one winner. */
    if (!use_typed)
    {
        for (i = 8; i < ARRAY_SIZE(expected_values); i++)
        {
            for (j = 0; j < ARRAY_SIZE(inputs); j++)
                if (values[i] == inputs[j])
                    break;

            ok(j < ARRAY_SIZE(inputs), "Got value 0x%"PRIx64", but it does not exist in inputs.\n", values[i]);
        }
    }

    release_resource_readback(&rb);

    if (use_typed)
    {
        transition_resource_state(context->list, output_texture,
                D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
        get_texture_readback_with_command_list(output_texture, 0, &rb, context->queue, context->list);
        reset_command_list(context->list, context->allocator);

        for (i = 0; i < 8; i++)
        {
            if (i == 5 || i == 6)
                continue;
            values = get_readback_data(&rb, i, i, 0, sizeof(uint64_t));
            ok(*values == expected_values[i], "Value %u = 0x%"PRIx64", expected 0x%"PRIx64"\n",
                    i, *values, expected_values[i]);
        }

        for (i = 8; i < ARRAY_SIZE(expected_values); i++)
        {
            values = get_readback_data(&rb, i, i, 0, sizeof(uint64_t));
            for (j = 0; j < ARRAY_SIZE(inputs); j++)
                if (*values == inputs[j])
                    break;

            ok(j < ARRAY_SIZE(inputs), "Got value 0x%"PRIx64", but it does not exist in inputs.\n", *values);
        }
        release_resource_readback(&rb);
    }

    if (heap)
        ID3D12DescriptorHeap_Release(heap);
    ID3D12Resource_Release(output_texture);
    ID3D12Resource_Release(input_buffer);
    ID3D12Resource_Release(output_buffer);
    ID3D12PipelineState_Release(pso);
    ID3D12RootSignature_Release(root_signature);
}

void test_shader_sm66_64bit_atomics(void)
{
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_FEATURE_DATA_D3D12_OPTIONS11 options11;
    D3D12_FEATURE_DATA_D3D12_OPTIONS1 options1;
    D3D12_FEATURE_DATA_D3D12_OPTIONS9 options9;
    struct test_context context;
    ID3D12Device *device;
    HRESULT hr;

    static const BYTE cs_code_64bit_atomic[] =
    {
#if 0
        StructuredBuffer<uint64_t> RO : register(t0);
        RWStructuredBuffer<uint64_t> RW : register(u0);
        RWStructuredBuffer<int64_t> RWSigned : register(u1);

        [numthreads(4, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                if (thr == 0)
                {
                        RW[1] = uint64_t(-1);
                        RW[4] = uint64_t(-1);
                        RW[7] = uint64_t(-1);
                }
                AllMemoryBarrierWithGroupSync();

                uint64_t v = RO[thr];
                InterlockedAdd(RW[0], v);
                InterlockedAnd(RW[1], ~v);
                InterlockedOr(RW[2], v);
                InterlockedMax(RW[3], v);
                InterlockedMin(RW[4], v);
                InterlockedMax(RWSigned[5], v);
                InterlockedMin(RWSigned[6], v);
                InterlockedXor(RW[7], v);
                uint64_t old_value;
                InterlockedExchange(RW[8], v, old_value);
                InterlockedCompareStore(RW[9], 0, v);
                InterlockedCompareStore(RW[9], 0, v + 1);

                InterlockedCompareExchange(RW[10], 0, v, old_value);
                InterlockedCompareExchange(RW[10], 0, v + 1, old_value);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x0e, 0x0f, 0x0b, 0x8b, 0x37, 0xac, 0x1d, 0xe5, 0x09, 0x72, 0x31, 0x81, 0x63, 0xd3, 0x82, 0xea, 0x01, 0x00, 0x00, 0x00, 0x68, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x1c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65, 0x50, 0x51, 0x66, 0xc3, 0x15, 0x1f, 0x0f, 0x1f, 0xb8, 0xd7, 0x53, 0x61, 0x8b, 0xe3, 0xe7, 0x44, 0x58, 0x49, 0x4c,
        0x44, 0x09, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x51, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x2c, 0x09, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0x48, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff,
        0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x36, 0x18, 0xc4, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x03, 0x48, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x86, 0x09, 0x01, 0x01, 0x00, 0x89, 0x20, 0x00, 0x00, 0x62, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x90, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x47, 0x00, 0x0a,
        0x73, 0x04, 0x08, 0x1d, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x02, 0xa4, 0x20, 0xc6, 0x90, 0x0c, 0xc3, 0x30, 0x18, 0xa4, 0xdc, 0x34, 0x5c, 0xfe, 0x84,
        0x3d, 0x84, 0xe4, 0xaf, 0x84, 0xb4, 0x12, 0x93, 0x8f, 0xe8, 0x38, 0x2a, 0x0c, 0xc3, 0x30, 0x86, 0x72, 0x30, 0x43, 0x32, 0x0c, 0xc7, 0x40, 0x4d, 0x59, 0x80, 0x21, 0x19, 0x06, 0xc3, 0x30, 0x8c,
        0x63, 0xa0, 0xa7, 0x20, 0xc6, 0x90, 0x0c, 0xc3, 0x60, 0x18, 0x14, 0x95, 0x01, 0x18, 0x06, 0x9a, 0x6e, 0x1b, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0x72, 0xa8, 0x48, 0x20, 0xd2, 0xc8,
        0x79, 0x88, 0x68, 0x42, 0x08, 0x09, 0x09, 0xc3, 0x50, 0x88, 0x64, 0x48, 0x2a, 0xb2, 0x0e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0xda, 0x90, 0x66, 0x40, 0xc4, 0x30, 0x0c, 0xc7,
        0x1c, 0x41, 0x50, 0x8a, 0x64, 0xc0, 0x86, 0x8c, 0xb4, 0x81, 0x80, 0x99, 0xd4, 0x60, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x60, 0x16, 0xe8, 0x41, 0x1e, 0xea, 0x61, 0x1c, 0xe8, 0xa1,
        0x1e, 0xe4, 0xa1, 0x1c, 0xc8, 0x41, 0x14, 0xea, 0xc1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xea, 0xc1, 0x1d, 0xe6, 0x21, 0x1d, 0xce, 0xc1, 0x1d, 0xca, 0x81, 0x1c, 0xc0, 0x80, 0x1d, 0xde,
        0xc1, 0x1d, 0xce, 0x01, 0x0c, 0xd8, 0xe1, 0x1d, 0xdc, 0xe1, 0x1c, 0xfc, 0x00, 0x05, 0x06, 0x75, 0x33, 0xb1, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x81, 0x2c, 0xdc, 0xc2, 0x2c,
        0xd0, 0x83, 0x3c, 0xd4, 0xc3, 0x38, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0x90, 0x83, 0x28, 0xd4, 0x83, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x03, 0x1f, 0xd4, 0x83, 0x3b, 0xcc, 0x43, 0x3a, 0x9c, 0x83,
        0x3b, 0x94, 0x03, 0x39, 0x80, 0x01, 0x3b, 0xbc, 0x83, 0x3b, 0x9c, 0x03, 0x18, 0xb0, 0xc3, 0x3b, 0xb8, 0xc3, 0x39, 0xf8, 0x01, 0x0a, 0x0c, 0xfa, 0x66, 0x1a, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98,
        0x87, 0x79, 0x70, 0x03, 0x59, 0xb8, 0x85, 0x59, 0xa0, 0x07, 0x79, 0xa8, 0x87, 0x71, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x20, 0x07, 0x51, 0xa8, 0x07, 0x73, 0x30, 0x87, 0x72, 0x90, 0x07, 0x3e,
        0x60, 0x87, 0x77, 0x70, 0x87, 0x73, 0x00, 0x03, 0x76, 0x78, 0x07, 0x77, 0x38, 0x07, 0x3f, 0x40, 0x81, 0x41, 0xe1, 0x14, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87,
        0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76,
        0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x60, 0xc8, 0x13, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f,
        0x05, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x0d, 0x08, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x40, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x0a, 0xa3, 0x20, 0xca, 0xa0, 0x10, 0xc8, 0x28, 0x50, 0x40, 0xc0,
        0x00, 0xca, 0x46, 0x00, 0xc8, 0x9b, 0x01, 0x20, 0x70, 0x06, 0x80, 0xc4, 0x19, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44,
        0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a,
        0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x21, 0x99, 0x20, 0x0c, 0xca, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98,
        0x20, 0x0c, 0xcb, 0x04, 0xc1, 0x93, 0x08, 0x4c, 0x10, 0x06, 0x66, 0x82, 0x30, 0x34, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0x00, 0x06,
        0xd3, 0x04, 0x21, 0x8b, 0x26, 0x08, 0x83, 0xb3, 0x81, 0x18, 0x18, 0x69, 0xd8, 0xb0, 0x20, 0x90, 0x82, 0x20, 0xc3, 0x12, 0x45, 0xd1, 0x34, 0x41, 0x10, 0x03, 0x6a, 0xc3, 0x32, 0x54, 0x0a, 0x32,
        0x0c, 0x4b, 0x14, 0x45, 0xd3, 0x06, 0x81, 0xb2, 0x36, 0x10, 0xcf, 0x05, 0x00, 0x13, 0x04, 0x01, 0x20, 0xd1, 0x16, 0x96, 0xe6, 0x36, 0x41, 0x30, 0xa0, 0x09, 0xc2, 0xf0, 0x6c, 0x18, 0xb8, 0x61,
        0xd8, 0x40, 0x20, 0x1b, 0xd7, 0x6d, 0x28, 0x32, 0x0d, 0xc0, 0xbc, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc,
        0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x20, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91,
        0x95, 0xb1, 0x4d, 0x09, 0x8c, 0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x82, 0xab, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b, 0x9d, 0x5c, 0x1e, 0xd4, 0x5b,
        0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0xc0, 0x03, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3,
        0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30,
        0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07,
        0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d,
        0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76,
        0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87,
        0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c,
        0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8,
        0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87,
        0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x30, 0x83, 0x81,
        0xc8, 0x01, 0x1f, 0xdc, 0x40, 0x1c, 0xe4, 0xa1, 0x1c, 0xc2, 0x61, 0x1d, 0xdc, 0x40, 0x1c, 0xe4, 0x01, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x76, 0x40, 0x0d, 0x97,
        0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xc0, 0xe4, 0x30, 0x88, 0xb0, 0x21, 0x0d, 0xfa, 0xf8, 0x88, 0x8e, 0x5b,
        0xc1, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0xc0, 0xe4, 0x30, 0x88, 0xc0, 0x39, 0xcc, 0x03, 0x44, 0x84, 0x77, 0x09, 0x07, 0xd0, 0x18, 0x84, 0x8f, 0xe8, 0xb8, 0x19, 0x34, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f,
        0x00, 0x51, 0x84, 0x10, 0x91, 0x21, 0x74, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0xf1, 0x45, 0x0e, 0xb3, 0x21, 0xcd, 0x80, 0x34, 0x86, 0x0d, 0x5c, 0xc3,
        0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0x88, 0x8e, 0x1b, 0x01, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39,
        0x11, 0xe1, 0x23, 0x3a, 0x6e, 0x01, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0x47, 0x44, 0x00, 0x83, 0x38, 0xf8, 0xc8, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00,
        0xa8, 0x00, 0x00, 0x00, 0x13, 0x04, 0x43, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x34, 0x14, 0xec, 0xc0, 0x0c, 0x40, 0x71, 0x06, 0x94, 0x67, 0x40, 0xc9, 0x0e, 0x14, 0xa6, 0x40,
        0x39, 0x14, 0x41, 0x29, 0x94, 0x44, 0x51, 0x94, 0x6e, 0x40, 0x61, 0x02, 0x15, 0x68, 0x40, 0x59, 0x94, 0xa5, 0x00, 0x19, 0x33, 0x00, 0x35, 0x30, 0x02, 0x50, 0x02, 0x74, 0x94, 0x00, 0x55, 0x73,
        0x08, 0x61, 0xd0, 0xcc, 0x21, 0x30, 0x0d, 0x61, 0x73, 0x10, 0x49, 0xb2, 0xa4, 0xc1, 0x1c, 0xc4, 0xb2, 0x2c, 0x69, 0x30, 0x02, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xa0, 0x9d,
        0x81, 0x35, 0x98, 0x81, 0x37, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x1a, 0x1a, 0x5c, 0x43, 0x19, 0x7c, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xa0, 0xa5, 0x01, 0x36, 0x98, 0x01, 0x18, 0x8c, 0x18,
        0x18, 0x00, 0x08, 0x82, 0x01, 0x11, 0x07, 0xd2, 0x19, 0x0c, 0x37, 0x04, 0x68, 0x00, 0x06, 0xb3, 0x0c, 0x81, 0x10, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0xd5, 0x06, 0x1f, 0xa1, 0x8c, 0x18,
        0x38, 0x00, 0x08, 0x82, 0xc1, 0x13, 0x07, 0x5b, 0xc0, 0x06, 0x6a, 0xf0, 0x40, 0x10, 0xc4, 0xa0, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x58, 0x6e, 0x00, 0x06, 0xc5, 0x32, 0x62, 0xe0, 0x00,
        0x20, 0x08, 0x06, 0x8f, 0x1c, 0x70, 0xc1, 0x19, 0xac, 0x01, 0x14, 0x45, 0x51, 0x93, 0x06, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0xbd, 0x41, 0x18, 0x18, 0xcc, 0x88, 0x81, 0x03, 0x80, 0x20,
        0x18, 0x3c, 0x73, 0xd0, 0x05, 0x1c, 0x1b, 0x44, 0x92, 0x24, 0x39, 0x6a, 0x30, 0x4b, 0x20, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0xc1, 0x14, 0x07, 0x15, 0x35, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06,
        0x16, 0x1c, 0x88, 0x81, 0xc1, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0x41, 0x53, 0x07, 0x54, 0x60, 0xb4, 0xc1, 0xb3, 0x06, 0xa3, 0x09, 0x01, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0xd6, 0x1c,
        0x94, 0x81, 0x02, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x81, 0x92, 0x07, 0x64, 0x10, 0xc0, 0x01, 0x1c, 0xc0, 0x41, 0x19, 0x08, 0x35, 0x58, 0x3c, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x1d,
        0xa0, 0x41, 0x33, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x81, 0xc2, 0x07, 0x67, 0x10, 0xd4, 0x41, 0x1d, 0xcc, 0x01, 0x1a, 0x08, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0xe1, 0x81, 0x1a, 0x3c,
        0xd5, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x28, 0x7e, 0x90, 0x06, 0x01, 0x19, 0x90, 0x41, 0x1d, 0xa8, 0xc1, 0x31, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x1e, 0xb0, 0x41, 0x74, 0x8d, 0x18,
        0x2c, 0x00, 0x08, 0x82, 0x81, 0x02, 0x0a, 0x6b, 0x10, 0x9c, 0x01, 0x1d, 0xdc, 0x01, 0x1b, 0x24, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0xf1, 0x81, 0x1b, 0x4c, 0xd9, 0x88, 0xc1, 0x02, 0x80,
        0x20, 0x18, 0x28, 0xa2, 0xd0, 0x06, 0x81, 0x1e, 0xd4, 0x41, 0x1e, 0xb8, 0xc1, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x1f, 0xc0, 0x81, 0xb5, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x81,
        0x42, 0x0a, 0x6f, 0x10, 0xa4, 0x41, 0x1a, 0xec, 0x01, 0x1c, 0x34, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0x81, 0x82, 0x1c, 0x60, 0xdd, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x28, 0xa6, 0x10,
        0x07, 0x41, 0x1e, 0xf8, 0x41, 0x1f, 0xc8, 0xc1, 0x33, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x28, 0xd0, 0x41, 0xf6, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x81, 0x82, 0x0a, 0x73, 0x10, 0xf0,
        0xc1, 0x1b, 0xfc, 0x01, 0x1d, 0x44, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0x91, 0x82, 0x1d, 0x6c, 0x61, 0x30, 0x62, 0xb0, 0x00, 0x20, 0x08, 0x06, 0x8a, 0x2a, 0xd4, 0x41, 0xf0, 0x07, 0x7f,
        0x10, 0x0a, 0x76, 0x30, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0x65, 0x0a, 0x78, 0xd0, 0x8d, 0xc1, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x44, 0xa9, 0x60, 0x07, 0x01, 0x1c, 0x8c, 0x02, 0x1e,
        0x9c, 0x41, 0x55, 0xd6, 0x19, 0xc0, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x58, 0xa9, 0xb0, 0x07, 0x60, 0x60, 0x06, 0x23, 0x06, 0x0b, 0x00, 0x82, 0x60, 0x10, 0xb1, 0x42, 0x1e, 0x04, 0x73, 0x60,
        0x0a, 0x7b, 0xa0, 0x06, 0xc2, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x58, 0xab, 0xd0, 0x07, 0x62, 0x80, 0x06, 0x23, 0x06, 0x0b, 0x00, 0x82, 0x60, 0x10, 0xb9, 0xc2, 0x1e, 0x04, 0x74, 0x80, 0x0a,
        0x7d, 0xc0, 0x06, 0xda, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x58, 0xad, 0xf0, 0x07, 0x64, 0xa0, 0x06, 0x23, 0x06, 0x0b, 0x00, 0x82, 0x60, 0x10, 0xc1, 0x42, 0x1f, 0x04, 0x76, 0xa0, 0x0a, 0x7f,
        0xe0, 0x06, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_code_64bit_atomic_typed[] =
    {
#if 0
        StructuredBuffer<uint64_t> RO : register(t0);
        RWTexture2D<uint64_t> RW : register(u0);
        RWBuffer<int64_t> RWSigned : register(u1);

        [numthreads(4, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                if (thr == 0)
                {
                        RW[int2(1, 1)] = uint64_t(-1);
                        RW[int2(4, 4)] = uint64_t(-1);
                        RW[int2(7, 7)] = uint64_t(-1);
                }
                AllMemoryBarrierWithGroupSync();

                uint64_t v = RO[thr];
                InterlockedAdd(RW[int2(0, 0)], v);
                InterlockedAnd(RW[int2(1, 1)], ~v);
                InterlockedOr(RW[int2(2, 2)], v);
                InterlockedMax(RW[int2(3, 3)], v);
                InterlockedMin(RW[int2(4, 4)], v);
                InterlockedMax(RWSigned[5], v);
                InterlockedMin(RWSigned[6], v);
                InterlockedXor(RW[int2(7, 7)], v);
                uint64_t old_value;
                InterlockedExchange(RW[int2(8, 8)], v, old_value);
                InterlockedCompareStore(RW[int2(9, 9)], 0, v);
                InterlockedCompareStore(RW[int2(9, 9)], 0, v + 1);

                InterlockedCompareExchange(RW[int2(10, 10)], 0, v, old_value);
                InterlockedCompareExchange(RW[int2(10, 10)], 0, v + 1, old_value);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x3c, 0xc1, 0xda, 0x34, 0xf2, 0x59, 0x4d, 0x8b, 0x20, 0x3f, 0xab, 0x14, 0xcb, 0x67, 0x8f, 0x61, 0x01, 0x00, 0x00, 0x00, 0x6c, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x1c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xef, 0x51, 0xa0, 0x91, 0x3c, 0x45, 0x25, 0x13, 0xcc, 0x4e, 0x79, 0x1d, 0x96, 0x1e, 0x4b, 0xb8, 0x44, 0x58, 0x49, 0x4c,
        0x48, 0x09, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x52, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x30, 0x09, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0x49, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff,
        0xff, 0xff, 0xff, 0x1f, 0x00, 0x06, 0x90, 0x36, 0x18, 0xc4, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x80, 0x04, 0x54, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x86, 0x09, 0x01, 0x01, 0x00, 0x89, 0x20, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x90, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x47, 0x80, 0x90,
        0x71, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x38, 0xca, 0x02, 0x0c, 0xc8, 0x30, 0x0c, 0xc3, 0x30, 0x0c, 0x06, 0x25, 0x73, 0x04, 0xa0, 0x50, 0x90, 0x65,
        0x40, 0x86, 0x61, 0x18, 0x16, 0x62, 0x0a, 0xb2, 0x0c, 0xc8, 0x30, 0x0c, 0xcb, 0x42, 0x4e, 0x19, 0x80, 0x61, 0x20, 0xe8, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98,
        0x7c, 0x44, 0xc7, 0x51, 0xb1, 0x2c, 0xcb, 0x32, 0x94, 0x43, 0x1a, 0x90, 0x61, 0x30, 0x06, 0x9a, 0x6e, 0x1b, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0x72, 0xa8, 0x48, 0x20, 0xd2, 0xc8,
        0x79, 0x88, 0x68, 0x42, 0x08, 0x09, 0x09, 0xc3, 0x50, 0x08, 0x64, 0x40, 0x2a, 0xb2, 0x0e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0xda, 0x90, 0x66, 0x40, 0xc4, 0x30, 0x0c, 0xc6,
        0x1c, 0x41, 0x50, 0x0a, 0x64, 0xc0, 0x86, 0x8c, 0xb4, 0x81, 0x80, 0x99, 0xd4, 0x60, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x60, 0x16, 0xe8, 0x41, 0x1e, 0xea, 0x61, 0x1c, 0xe8, 0xa1,
        0x1e, 0xe4, 0xa1, 0x1c, 0xc8, 0x41, 0x14, 0xea, 0xc1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xea, 0xc1, 0x1d, 0xe6, 0x21, 0x1d, 0xce, 0xc1, 0x1d, 0xca, 0x81, 0x1c, 0xc0, 0x80, 0x1d, 0xde,
        0xc1, 0x1d, 0xce, 0x01, 0x0c, 0xd8, 0xe1, 0x1d, 0xdc, 0xe1, 0x1c, 0xfc, 0x00, 0x05, 0x0b, 0x75, 0x33, 0x95, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x81, 0x2c, 0xdc, 0x02, 0x2d,
        0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03, 0x18, 0xb0, 0xc3, 0x3b, 0xb8, 0xc3,
        0x39, 0x80, 0x01, 0x3b, 0xbc, 0x83, 0x3b, 0x9c, 0x83, 0x1f, 0xa0, 0x60, 0xa1, 0x6f, 0xa6, 0x6c, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x40, 0x16, 0x6e, 0x41, 0x14, 0xea, 0xc1, 0x1c,
        0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xd8, 0xe1, 0x1d, 0xdc, 0xe1, 0x1c, 0xc0, 0x80, 0x1d, 0xde, 0xc1, 0x1d, 0xce, 0xc1, 0x0f, 0x50, 0xb0, 0x50, 0x38, 0x05, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0,
        0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0,
        0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x79, 0x12, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x03, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x21, 0x8f, 0x05, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x0d, 0x08, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x40, 0x00, 0x00,
        0x0d, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x0a, 0xa3, 0x20, 0x8a, 0xa0, 0x14, 0xca,
        0xa0, 0x28, 0x0a, 0x81, 0x96, 0x02, 0x05, 0x04, 0x0c, 0x0c, 0xa0, 0x6c, 0x04, 0x80, 0xbc, 0x19, 0x00, 0x02, 0x67, 0x00, 0x48, 0x9c, 0x01, 0x00, 0x79, 0x18, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x21, 0x99, 0x20, 0x0c, 0xca, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xcb, 0x04, 0xc1, 0xab, 0x08, 0x4c, 0x10, 0x06, 0x66, 0x82, 0x30, 0x34, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0x00, 0x06, 0xd6, 0x04, 0x61, 0x70, 0x26, 0x08, 0x19, 0x35, 0x41, 0x18, 0x9e, 0x09, 0xc2, 0x00, 0x6d, 0x20, 0x90, 0x89, 0x1a, 0x36, 0x2c, 0x08, 0xa4, 0x20,
        0xc8, 0x10, 0x49, 0x92, 0x54, 0x4d, 0x10, 0xc4, 0xe0, 0x9a, 0x20, 0x0c, 0xd1, 0x86, 0x65, 0xb8, 0x14, 0x64, 0x18, 0x30, 0x49, 0x92, 0xaa, 0x0d, 0x82, 0x95, 0x6d, 0x20, 0x1e, 0x0d, 0x00, 0x26,
        0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x6d, 0x82, 0xb0, 0x4c, 0x13, 0x84, 0x41, 0xda, 0x30, 0x7c, 0xc3, 0xb0, 0x81, 0x40, 0xbc, 0x0f, 0x0c, 0x36, 0x14, 0x5c, 0x07, 0x6c, 0x61, 0x50, 0x85,
        0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2,
        0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x44, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x81, 0x51, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee,
        0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0xa0, 0xd5, 0x21, 0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb, 0x83, 0x7a, 0x4b, 0x73, 0xa3, 0x9b, 0x9b, 0x12, 0x84, 0x01, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20,
        0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x30, 0x83, 0x81, 0xc8, 0x01, 0x1f, 0xdc, 0x40, 0x1c, 0xe4, 0xa1, 0x1c, 0xc2, 0x61, 0x1d,
        0xdc, 0x40, 0x1c, 0xe4, 0x01, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x76, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3,
        0x42, 0xd8, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xc0, 0xe4, 0x30, 0x88, 0xb0, 0x21, 0x0d, 0xfa, 0xf8, 0x88, 0x8e, 0x1b, 0xc1, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0xc0, 0xe4, 0x30, 0x88, 0xc0, 0x39,
        0xcc, 0x03, 0x44, 0x84, 0x77, 0x09, 0x07, 0xd0, 0x18, 0x84, 0x8f, 0xe8, 0xb8, 0x15, 0x34, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x00, 0x51, 0x84, 0x10, 0x91, 0x21, 0x74, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f,
        0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0xf1, 0x45, 0x0e, 0xb3, 0x21, 0xcd, 0x80, 0x34, 0x86, 0x19, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0x95, 0x0e,
        0x30, 0xf8, 0x88, 0x8e, 0x9b, 0x80, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x13, 0x71, 0x4d, 0x54, 0x44, 0xb0, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44,
        0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x13, 0x04, 0x43, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00, 0x34, 0xcc, 0x00, 0x14, 0xec, 0x40, 0x0d, 0x14, 0x67, 0x40, 0x39, 0x94, 0x67, 0x40, 0x49, 0x94, 0xec, 0x40, 0x19, 0x06, 0x94, 0x22, 0x41, 0xe9, 0x06, 0x14, 0x21, 0x50,
        0x81, 0x06, 0x94, 0x45, 0x59, 0x0a, 0x14, 0x25, 0x10, 0x19, 0x25, 0x50, 0x1e, 0xb4, 0x8c, 0x00, 0xd4, 0x40, 0x09, 0x50, 0x35, 0x87, 0x30, 0x06, 0x61, 0x30, 0x87, 0x70, 0x06, 0x61, 0x30, 0x87,
        0xc0, 0x34, 0x84, 0xcd, 0x41, 0x24, 0xc9, 0x82, 0x06, 0x73, 0x10, 0xcb, 0xb2, 0xa0, 0xc1, 0x08, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xa0, 0xad, 0x01, 0x35, 0xa8, 0xc1,
        0x37, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x1a, 0x1b, 0x54, 0x43, 0x1a, 0x80, 0xc1, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x68, 0x6d, 0x60, 0x0d, 0x6a, 0x10, 0x06, 0x23, 0x06, 0x06, 0x00, 0x82,
        0x60, 0x40, 0xd4, 0x01, 0xb5, 0x06, 0xc3, 0x0d, 0x01, 0x1b, 0x80, 0xc1, 0x2c, 0x43, 0x20, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0xc5, 0xc1, 0x47, 0x2c, 0x23, 0x06, 0x0e, 0x00, 0x82,
        0x60, 0x90, 0xdc, 0x41, 0x16, 0xc0, 0x01, 0x1c, 0x84, 0xc1, 0xf7, 0x7d, 0x1f, 0x34, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x1c, 0x80, 0x41, 0xc1, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0x41,
        0x82, 0x07, 0x5a, 0x80, 0x06, 0x68, 0x20, 0x06, 0x60, 0x00, 0x06, 0x60, 0x00, 0x06, 0xd1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x58, 0x73, 0x10, 0x06, 0x46, 0x33, 0x62, 0xe0, 0x00, 0x20, 0x08,
        0x06, 0x49, 0x1e, 0x6c, 0xc1, 0xf7, 0x8d, 0x41, 0x18, 0x84, 0x41, 0x18, 0x84, 0x81, 0x34, 0x4b, 0x20, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0x01, 0x64, 0x07, 0x97, 0x35, 0x62, 0x70, 0x00, 0x20,
        0x08, 0x06, 0x16, 0x1d, 0x88, 0x81, 0xc1, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0xc1, 0x64, 0x07, 0x56, 0x60, 0xc4, 0x01, 0xf5, 0x06, 0xa3, 0x09, 0x01, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06,
        0xd6, 0x1d, 0x94, 0x81, 0x12, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x01, 0xc3, 0x07, 0x64, 0x10, 0xd0, 0x01, 0x1d, 0xd0, 0xc1, 0x19, 0x08, 0x35, 0x54, 0x3c, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06,
        0x96, 0x1e, 0xa0, 0x41, 0x43, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x01, 0xf3, 0x07, 0x67, 0x10, 0xe4, 0x41, 0x1e, 0xe4, 0x81, 0x1a, 0x08, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0xf1, 0x81,
        0x1a, 0x3c, 0xd6, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x30, 0xa1, 0x90, 0x06, 0x81, 0x1d, 0xd8, 0x81, 0x1d, 0xb0, 0xc1, 0x31, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x1f, 0xb0, 0x41, 0x84,
        0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x01, 0x33, 0x0a, 0x6b, 0x10, 0xa8, 0x81, 0x1d, 0xd8, 0x81, 0x1b, 0x24, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0x81, 0x82, 0x1b, 0x4c, 0xda, 0x88, 0xc1,
        0x02, 0x80, 0x20, 0x18, 0x30, 0xa5, 0xd0, 0x06, 0x81, 0x1f, 0xd8, 0x81, 0x1d, 0xc0, 0xc1, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x28, 0xc0, 0x81, 0xb5, 0x8d, 0x18, 0x2c, 0x00, 0x08,
        0x82, 0x01, 0x73, 0x0a, 0x6f, 0x10, 0xec, 0xc1, 0x1e, 0xc8, 0x81, 0x1c, 0x34, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0x91, 0x82, 0x1c, 0x60, 0xdd, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x30,
        0xa9, 0x10, 0x07, 0x81, 0x1e, 0x88, 0x02, 0x1d, 0xd0, 0xc1, 0x33, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x96, 0x29, 0xd0, 0x41, 0x06, 0x06, 0x23, 0x06, 0x0b, 0x00, 0x82, 0x60, 0xc0, 0xac, 0xc2,
        0x1c, 0x04, 0x7e, 0x20, 0x07, 0x72, 0x60, 0x07, 0xd1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x58, 0xa8, 0x60, 0x07, 0x9b, 0x18, 0x8c, 0x18, 0x2c, 0x00, 0x08, 0x82, 0x01, 0xd3, 0x0a, 0x75, 0x10,
        0x8c, 0xc2, 0x28, 0x8c, 0x02, 0x1e, 0x4c, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x60, 0xa9, 0x02, 0x1e, 0x74, 0x64, 0x30, 0x62, 0xb0, 0x00, 0x20, 0x08, 0x06, 0x8e, 0x2b, 0xd4, 0x41, 0x40, 0x07,
        0x74, 0xa0, 0x07, 0x68, 0x50, 0x95, 0x75, 0x06, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x56, 0x2b, 0xec, 0x01, 0x18, 0x9c, 0xc1, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x38, 0xb1, 0x80, 0x07,
        0xc1, 0x1d, 0xdc, 0x41, 0x1f, 0xac, 0x81, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0xd6, 0x2b, 0xf4, 0x81, 0x18, 0xa4, 0xc1, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x38, 0xb3, 0xa0, 0x07, 0x81,
        0x29, 0x98, 0xc2, 0x1f, 0xb4, 0x81, 0x36, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x56, 0x2c, 0xfc, 0x01, 0x19, 0xac, 0xc1, 0x88, 0xc1, 0x02, 0x80, 0x20, 0x18, 0x38, 0xb5, 0xc0, 0x07, 0x01, 0x2a,
        0xa0, 0x42, 0x28, 0xbc, 0x81, 0x81, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_code_64bit_atomic_shared[] =
    {
#if 0
        StructuredBuffer<uint64_t> RO : register(t0);
        RWStructuredBuffer<uint64_t> RW : register(u0);

        groupshared uint64_t uv[11];
        groupshared int64_t sv[11];

        [numthreads(11, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                uv[thr] = thr == 1 || thr == 4 || thr == 7 ? uint64_t(-1) : uint64_t(0);
                sv[thr] = 0;
                GroupMemoryBarrierWithGroupSync();

                if (thr < 4)
                {
                        uint64_t v = RO[thr];
                        InterlockedAdd(uv[0], v);
                        InterlockedAnd(uv[1], ~v);
                        InterlockedOr(uv[2], v);
                        InterlockedMax(uv[3], v);
                        InterlockedMin(uv[4], v);

                        InterlockedMax(sv[5], v);
                        InterlockedMin(sv[6], v);

                        InterlockedXor(uv[7], v);
                        uint64_t old_value;
                        InterlockedExchange(uv[8], v, old_value);
                        InterlockedCompareStore(uv[9], 0, v);
                        InterlockedCompareStore(uv[9], 0, v + 1);
                        InterlockedCompareExchange(uv[10], 0, v, old_value);
                        InterlockedCompareExchange(uv[10], 0, v + 1, old_value);
                }

                GroupMemoryBarrierWithGroupSync();

                RW[thr] = thr == 5 || thr == 6 ? uint64_t(sv[thr]) : uv[thr];
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0xa5, 0xe0, 0x11, 0x7e, 0x39, 0x74, 0x8e, 0x8a, 0xb4, 0x4d, 0x47, 0x6e, 0x2a, 0xed, 0x7a, 0xab, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x09, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3d, 0x46, 0x8b, 0x93, 0xd2, 0x3a, 0x9a, 0xb6, 0xe1, 0x07, 0x33, 0xc8,
        0x4a, 0xc3, 0x46, 0xb4, 0x44, 0x58, 0x49, 0x4c, 0x70, 0x08, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x1c, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x58, 0x08, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x13, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0xc0, 0x00, 0xd2, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x36, 0x18, 0xc4, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x80, 0x04, 0x54, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x86, 0x09, 0x01, 0x01, 0x00, 0x89, 0x20, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04,
        0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x88, 0xc1, 0x1c, 0x01, 0x28, 0xe4, 0x02, 0x30, 0x88, 0x10, 0x0c,
        0x23, 0x00, 0x25, 0x18, 0x88, 0x98, 0x23, 0x00, 0x83, 0x32, 0x18, 0x86, 0x41, 0x47, 0x19, 0x06, 0xc3, 0xa0, 0x64, 0x8e, 0x00, 0xa1, 0xe5, 0x9e, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x3f, 0x04,
        0x9a, 0x61, 0x21, 0x50, 0xc0, 0x94, 0x65, 0x30, 0x1a, 0xc3, 0x00, 0x00, 0x00, 0x58, 0x0c, 0x72, 0x6e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0x5a, 0x89, 0xc9, 0x47, 0x74, 0x1c,
        0x15, 0x00, 0x00, 0x00, 0x46, 0x39, 0x20, 0xa3, 0x31, 0x8c, 0xc5, 0xa0, 0xe8, 0xb6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0x24, 0x87, 0x8a, 0x04, 0x22, 0x8d, 0x9c, 0x87, 0x88, 0x26,
        0x84, 0x90, 0x90, 0x60, 0x18, 0x85, 0x68, 0x8c, 0x66, 0x22, 0xea, 0xa0, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x0d, 0x69, 0x06, 0x44, 0x18, 0x86, 0xb1, 0xcc, 0x11, 0x04, 0xa5,
        0x68, 0x0c, 0xcb, 0xb8, 0x08, 0x1b, 0x08, 0x98, 0x49, 0x0d, 0xc6, 0x81, 0x1d, 0xc2, 0x61, 0x1e, 0xe6, 0xc1, 0x0d, 0x66, 0x81, 0x1e, 0xe4, 0xa1, 0x1e, 0xc6, 0x81, 0x1e, 0xea, 0x41, 0x1e, 0xca,
        0x81, 0x1c, 0x44, 0xa1, 0x1e, 0xcc, 0xc1, 0x1c, 0xca, 0x41, 0x1e, 0xf8, 0xa0, 0x1e, 0xdc, 0x61, 0x1e, 0xd2, 0xe1, 0x1c, 0xdc, 0xa1, 0x1c, 0xc8, 0x01, 0x0c, 0xd8, 0xe1, 0x1d, 0xdc, 0xe1, 0x1c,
        0xc0, 0x80, 0x1d, 0xde, 0xc1, 0x1d, 0xce, 0xc1, 0x0f, 0x50, 0x00, 0xd0, 0x36, 0x13, 0x1b, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43,
        0x3d, 0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90,
        0x03, 0x18, 0xb0, 0xc3, 0x3b, 0xb8, 0xc3, 0x39, 0x80, 0x01, 0x3b, 0xbc, 0x83, 0x3b, 0x9c, 0x83, 0x1f, 0xa0, 0x00, 0xa0, 0x6e, 0x10, 0x01, 0x18, 0xa6, 0x00, 0x8c, 0x00, 0x5c, 0x00, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x24, 0x90, 0x21, 0x23, 0x45, 0x46, 0x00, 0x76, 0x00, 0x40, 0x76, 0x00, 0xc0, 0x21, 0x0f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x03, 0x08, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x09, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x79, 0x1c, 0x20, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x44, 0x40, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0xa1, 0x80, 0x00, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x83, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14,
        0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x0a, 0x14, 0x10, 0x30, 0x50, 0x60, 0x04, 0x80, 0x8c, 0x12, 0x28, 0x86, 0x11, 0x80, 0xc2, 0x28, 0x88, 0x42, 0x28, 0x0b,
        0xba, 0x46, 0x00, 0x88, 0x9b, 0x01, 0x20, 0x6f, 0x06, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b,
        0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81,
        0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x63, 0x99, 0x20, 0x18, 0xcc, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x18, 0xcd, 0x04,
        0x81, 0x9b, 0x08, 0x4c, 0x10, 0x0c, 0x67, 0x82, 0x60, 0x3c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xe0, 0x51, 0x13, 0x84, 0x4b, 0xda,
        0xb0, 0x20, 0x90, 0x82, 0x20, 0xc3, 0x12, 0x45, 0x51, 0xb3, 0x21, 0x90, 0x36, 0x10, 0xcf, 0x04, 0x00, 0x13, 0x84, 0x42, 0x20, 0xd1, 0x16, 0x96, 0xe6, 0x36, 0x41, 0x00, 0x92, 0x09, 0x82, 0x01,
        0x4d, 0x10, 0x8c, 0x68, 0xc3, 0x90, 0x0d, 0xc3, 0x06, 0x02, 0xb9, 0x30, 0x6d, 0x43, 0x51, 0x59, 0x00, 0xb5, 0x31, 0x61, 0x7b, 0x73, 0x3b, 0x03, 0x61, 0x7b, 0x73, 0x3b, 0xe3, 0xf3, 0xd6, 0xe6,
        0x96, 0x06, 0xf7, 0x46, 0x57, 0xe6, 0x46, 0x07, 0x32, 0x86, 0x16, 0x26, 0xc7, 0x68, 0x2a, 0xad, 0x0d, 0x8e, 0xad, 0x0c, 0x64, 0xe8, 0x65, 0x68, 0x65, 0x05, 0x84, 0x4a, 0x28, 0x28, 0x68, 0x43,
        0xf0, 0x4d, 0x10, 0x00, 0x65, 0xc3, 0xe0, 0x81, 0x41, 0x18, 0x6c, 0x18, 0x3a, 0x31, 0x08, 0x83, 0x0d, 0xc3, 0x18, 0x8c, 0x41, 0x18, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73,
        0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87,
        0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x4c,
        0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c,
        0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e,
        0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c,
        0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e,
        0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4,
        0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07,
        0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5,
        0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b,
        0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2,
        0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x30, 0x83, 0x81, 0xc8, 0x01, 0x1f, 0xdc, 0x40, 0x1c, 0xe4, 0xa1, 0x1c, 0xc2, 0x61, 0x1d, 0xdc, 0x40, 0x1c, 0xe4, 0x01, 0x00, 0x00, 0x00,
        0x71, 0x20, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x15, 0xc0, 0x04, 0x7e, 0x73, 0x3b, 0x10, 0x38, 0x83, 0x06, 0xbf, 0xca, 0x60, 0x01, 0x30, 0x81, 0x5f, 0xdd, 0x0e, 0x04, 0xce, 0xa0, 0xc1, 0xef,
        0x32, 0x68, 0x07, 0xd4, 0x70, 0xf9, 0xce, 0xe3, 0x03, 0x4d, 0xe3, 0x4c, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x11, 0x34, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x00, 0x51, 0x84, 0x10, 0x91, 0x21,
        0x74, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0xf1, 0x45, 0x0e, 0xb3, 0x21, 0xcd, 0x80, 0x34, 0x86, 0x19, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6,
        0x46, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0x88, 0x8e, 0x5b, 0x01, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39, 0x11, 0xe1, 0x23, 0x3a, 0x6e, 0x03,
        0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0x47, 0x44, 0x00, 0x83, 0x38, 0xf8, 0xc8, 0x6d, 0x9b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x61, 0x20, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00, 0x13, 0x04, 0x43, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x04, 0xcc, 0x00, 0xd4, 0x40, 0x09, 0x90, 0x51, 0x12, 0x25, 0x3b, 0x50, 0x0e, 0x05, 0x1a, 0x50, 0xb0, 0x03, 0x45, 0x50, 0xba, 0x01, 0x65, 0x29,
        0x50, 0x06, 0xa5, 0x50, 0x14, 0xb5, 0x50, 0x98, 0x40, 0x85, 0x29, 0x40, 0x4b, 0x09, 0xd0, 0x34, 0x87, 0xe0, 0x3c, 0x73, 0x08, 0x64, 0xf0, 0x90, 0x35, 0x07, 0xd1, 0x34, 0x8d, 0x19, 0x8c, 0x00,
        0xd0, 0x37, 0xd4, 0x11, 0x08, 0x80, 0xd1, 0x18, 0xd8, 0x50, 0x47, 0x20, 0x00, 0x46, 0x63, 0x88, 0xc1, 0x50, 0x47, 0x20, 0x00, 0x46, 0x63, 0x34, 0x43, 0x1d, 0x81, 0x00, 0x18, 0x8d, 0xb1, 0x0c,
        0x75, 0x04, 0x02, 0x60, 0x34, 0x46, 0x37, 0xd4, 0x11, 0x08, 0x80, 0xd1, 0x18, 0x60, 0x30, 0xd4, 0x11, 0x08, 0x80, 0xd1, 0x18, 0xd0, 0x50, 0x47, 0x20, 0x04, 0x46, 0x63, 0x84, 0xc1, 0x50, 0x47,
        0x20, 0x04, 0x46, 0x63, 0x30, 0x43, 0x1d, 0x81, 0x00, 0x18, 0x8d, 0xa1, 0x0d, 0x75, 0x04, 0x02, 0x60, 0x34, 0xc6, 0x03, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x80, 0xb9, 0x41, 0xd7, 0xa4, 0x01,
        0x19, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0x01, 0xf6, 0x06, 0x5e, 0xa3, 0x06, 0x65, 0x30, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x47, 0x1d, 0x68, 0x6b, 0x30, 0xdc, 0x10, 0xb8, 0x01, 0x18, 0x0c,
        0x37, 0x08, 0x6a, 0x00, 0x06, 0x25, 0x04, 0x3b, 0xdc, 0x40, 0x88, 0x01, 0x18, 0x54, 0x20, 0xec, 0x05, 0x80, 0xac, 0x60, 0xf8, 0x03, 0x39, 0x38, 0x86, 0x0d, 0x88, 0x40, 0x20, 0x80, 0x15, 0x0c,
        0x7f, 0x30, 0x07, 0xc8, 0xb0, 0x01, 0x11, 0xdc, 0x01, 0x01, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0x41, 0xd2, 0x07, 0x66, 0x90, 0x06, 0xc3, 0x0d, 0x49, 0x1c, 0x90, 0xc1, 0x2c, 0x43, 0x20, 0x04,
        0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x40, 0xed, 0x81, 0x19, 0x2c, 0xdb, 0x88, 0x81, 0x02, 0x80, 0x20, 0x18, 0x44, 0x7d, 0x40, 0x06, 0xc1, 0x62, 0x07, 0x1d, 0x1d, 0x8c, 0x26, 0x04, 0xc0, 0x98,
        0x81, 0x81, 0x05, 0x01, 0x60, 0x04, 0x25, 0xc0, 0x01, 0x8f, 0x19, 0x18, 0x59, 0x30, 0x00, 0x46, 0x30, 0x66, 0x60, 0x64, 0x44, 0x01, 0x18, 0xc1, 0x98, 0x81, 0x91, 0x15, 0x09, 0x60, 0x04, 0x63,
        0x06, 0x46, 0x66, 0x28, 0x80, 0x11, 0x8c, 0x19, 0x18, 0xd9, 0x71, 0x00, 0x46, 0x30, 0x66, 0x60, 0x64, 0x08, 0x02, 0x18, 0xc1, 0x98, 0x81, 0x91, 0x25, 0x06, 0x60, 0x04, 0x63, 0x06, 0x46, 0xa6,
        0x00, 0x80, 0x11, 0x8c, 0x1b, 0x20, 0x64, 0x50, 0x0a, 0x0b, 0x60, 0x04, 0x06, 0x50, 0x4c, 0x1e, 0xc0, 0xb8, 0x01, 0x62, 0x06, 0xa7, 0x10, 0x00, 0x46, 0x60, 0x00, 0xe3, 0x06, 0x88, 0x19, 0xa0,
        0x82, 0x03, 0x18, 0x81, 0x01, 0x8c, 0x1b, 0x20, 0x67, 0x90, 0x0a, 0x03, 0x60, 0x04, 0x06, 0x30, 0x4b, 0x20, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0x41, 0x02, 0x0b, 0x79, 0xc0, 0x07, 0xc5, 0xc5,
        0x01, 0x0c, 0x37, 0x04, 0x78, 0x40, 0x06, 0x64, 0x01, 0x84, 0x30, 0x80, 0x4c, 0x37, 0x08, 0xc1, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x94, 0x2c, 0xf4, 0xc1, 0x18, 0xc4, 0xc1, 0x88, 0x81,
        0x03, 0x80, 0x20, 0x18, 0x38, 0xb5, 0x50, 0x07, 0x81, 0x18, 0xb4, 0x82, 0x50, 0x0a, 0xa5, 0x50, 0x0a, 0x74, 0xb0, 0x0a, 0x18, 0x10, 0x03, 0x00, 0x06, 0x00, 0x00, 0x00, 0x5b, 0x06, 0x25, 0x18,
        0x83, 0x2d, 0x03, 0x13, 0x8c, 0xc1, 0x96, 0xc1, 0x0c, 0x82, 0x31, 0xd8, 0x32, 0x9c, 0x41, 0x30, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const D3D12_SHADER_BYTECODE cs_64bit_atomic = SHADER_BYTECODE(cs_code_64bit_atomic);
    static const D3D12_SHADER_BYTECODE cs_64bit_atomic_typed = SHADER_BYTECODE(cs_code_64bit_atomic_typed);
    static const D3D12_SHADER_BYTECODE cs_64bit_atomic_shared = SHADER_BYTECODE(cs_code_64bit_atomic_shared);

    if (!init_compute_test_context(&context))
        return;

    device = context.device;

    memset(&options9, 0, sizeof(options9));
    memset(&options11, 0, sizeof(options11));
    hr = ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_D3D12_OPTIONS9, &options9, sizeof(options9));
    ok(SUCCEEDED(hr), "OPTIONS9 is not supported by runtime.\n");
    hr = ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_D3D12_OPTIONS11, &options11, sizeof(options11));
    ok(SUCCEEDED(hr), "OPTIONS11 is not supported by runtime.\n");
    /* For later, when we test more exotic 64-bit atomic scenarios ... */

    hr = ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_D3D12_OPTIONS1, &options1, sizeof(options1));
    if (FAILED(hr) || !options1.Int64ShaderOps)
    {
        skip("64-bit shader operations not supported, skipping ...\n");
        destroy_test_context(&context);
        return;
    }

    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    hr = ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model));
    if (FAILED(hr) || shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_6)
    {
        skip("Device does not support SM 6.6.\n");
        destroy_test_context(&context);
        return;
    }

    vkd3d_test_set_context("64-bit atomic root descriptor");
    run_64bit_atomics_test(&context, cs_64bit_atomic, false, false);
    vkd3d_test_set_context(NULL);

    if (options11.AtomicInt64OnDescriptorHeapResourceSupported)
    {
        vkd3d_test_set_context("64-bit atomic table (raw)");
        run_64bit_atomics_test(&context, cs_64bit_atomic, true, false);
        vkd3d_test_set_context(NULL);
    }
    else
        skip("AtomicInt64OnDescriptorHeapResourceSupported not set, skipping.\n");

    if (options11.AtomicInt64OnDescriptorHeapResourceSupported &&
        options9.AtomicInt64OnTypedResourceSupported)
    {
        vkd3d_test_set_context("64-bit atomic table (typed)");
        run_64bit_atomics_test(&context, cs_64bit_atomic_typed, true, true);
        vkd3d_test_set_context(NULL);
    }
    else
        skip("AtomicInt64OnTypedResourceSupported is not set, skipping.\n");

    if (options9.AtomicInt64OnGroupSharedSupported)
    {
        vkd3d_test_set_context("64-bit atomic groupshared");
        run_64bit_atomics_test(&context, cs_64bit_atomic_shared, false, false);
        vkd3d_test_set_context(NULL);
    }
    else
        skip("AtomicInt64OnGroupSharedSupported is not set, skipping.\n");

    destroy_test_context(&context);
}

void test_shader_sm66_compute_derivatives(void)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_STATIC_SAMPLER_DESC static_samplers[1];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[2];
    D3D12_ROOT_PARAMETER root_parameters[1];
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    ID3D12RootSignature *root_signature;
    ID3D12Resource *result_texture;
    ID3D12Resource *input_texture;
    struct test_context context;
    struct resource_readback rb;
    ID3D12DescriptorHeap *heap;
    ID3D12PipelineState *pso;
    ID3D12Device *device;
    unsigned int i, x, y;
    HRESULT hr;

    static const BYTE cs_derivative_2d_aligned_dxil[] =
    {
#if 0
        Texture2D<float> Tex : register(t0);
        RWTexture2D<float4> SOut : register(u0);
        SamplerState Samp : register(s0);

        [numthreads(8, 8, 1)]
        void main(uint2 thr : SV_DispatchThreadID)
        {
                float2 uv = float2(thr) / 32.0; // We should sample LOD 2 here.
                float v = round(Tex.Sample(Samp, uv) * 255.0);
                float dx = ddx_fine(uv.x);
                float dy = ddy_fine(uv.y);
                float w = Tex.CalculateLevelOfDetailUnclamped(Samp, uv);
                SOut[thr] = float4(v, dx, dy, w);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x27, 0x99, 0x4e, 0x1d, 0x9a, 0xfb, 0x55, 0xc9, 0x42, 0x68, 0x3d, 0xa9, 0x30, 0x7b, 0xfa, 0xd9, 0x01, 0x00, 0x00, 0x00, 0x94, 0x08, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x1c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf9, 0xe2, 0x05, 0x60, 0x9f, 0xb9, 0xa0, 0x65, 0x06, 0xe1, 0xce, 0x99, 0x2d, 0x89, 0x6f, 0xc7, 0x44, 0x58, 0x49, 0x4c,
        0x70, 0x07, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xdc, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x58, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0xd3, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62,
        0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00,
        0x5a, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c,
        0x10, 0x8c, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x03, 0x50, 0x06, 0x63, 0x30, 0xe8, 0x98, 0x23, 0x40, 0x28, 0xb9, 0x67, 0xb8, 0xfc, 0x09,
        0x7b, 0x08, 0xc9, 0x0f, 0x81, 0x66, 0x58, 0x08, 0x14, 0x28, 0x73, 0x04, 0x41, 0x41, 0x8c, 0x61, 0x59, 0x0c, 0xc3, 0x60, 0xa8, 0xb9, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69,
        0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x54, 0x18, 0x86, 0x61, 0x0c, 0x85, 0x79, 0x86, 0x65, 0x31, 0x0c, 0xc3, 0x18, 0x86, 0xc1, 0x20, 0xa8, 0x2c, 0xc0, 0xb0, 0x0c, 0xc3, 0x60, 0x18, 0x86, 0x91, 0x90,
        0x74, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43, 0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13, 0x42, 0x48, 0x48, 0x18, 0x86, 0x42, 0x2c, 0xc3, 0x42, 0x51, 0x75, 0xd0,
        0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x86, 0x34, 0x03, 0x22, 0x86, 0x61, 0x48, 0x4a, 0xb1, 0x0c, 0xd7, 0xc0, 0x10, 0x36, 0x10, 0x30, 0x53, 0x18, 0x8c, 0x03, 0x3b, 0x84, 0xc3,
        0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x83, 0x1f, 0xe8, 0x81, 0x1e, 0xb4,
        0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0x86, 0x99, 0xac, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87,
        0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07, 0x7a, 0xf0, 0x03, 0x24, 0x18, 0x1b, 0x71, 0xc3, 0x08, 0x84, 0x31, 0x13, 0x19, 0x8c, 0x03, 0x3b, 0x84,
        0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c,
        0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xa0, 0xc0, 0xa3, 0xef, 0x26, 0x69, 0x8a, 0x28, 0x61, 0xf2, 0x59, 0x80, 0x79, 0x16,
        0x22, 0x62, 0x27, 0x60, 0x22, 0x50, 0x30, 0x50, 0x08, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50,
        0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07,
        0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0,
        0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34,
        0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59,
        0x20, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x8a, 0xa0, 0x24,
        0x0a, 0xa1, 0x20, 0x88, 0x19, 0x01, 0xa0, 0x6e, 0x06, 0x80, 0xc0, 0x19, 0x00, 0x12, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62,
        0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x01, 0x99, 0x20, 0x0c, 0xc9, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d,
        0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xca, 0x04, 0xa1, 0x83, 0x08, 0x4c, 0x10, 0x86, 0x65, 0x82, 0x30, 0x30, 0x1b, 0x04, 0x84, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02,
        0x67, 0x82, 0x00, 0x06, 0xd1, 0x04, 0x81, 0x79, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0xd4, 0x6c, 0x08, 0xa4, 0x09, 0x82, 0x18, 0x48, 0x1b, 0x10, 0x84, 0x52, 0x10, 0x64, 0x40,
        0x80, 0x0d, 0x41, 0xb5, 0x81, 0x78, 0x26, 0xc0, 0x9a, 0x20, 0x08, 0x00, 0x89, 0xb6, 0xb0, 0x34, 0xb7, 0x09, 0xc2, 0xd0, 0x4c, 0x10, 0x06, 0x67, 0xc3, 0xb0, 0x6d, 0xc3, 0x06, 0x41, 0xe3, 0x36,
        0x14, 0x58, 0x06, 0x5c, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40,
        0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x10, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x46, 0x19,
        0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0x81, 0x55, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0xd0,
        0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19,
        0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x20, 0x00, 0x00, 0x00, 0x66, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0x80, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x02, 0xb0, 0x08, 0xd4,
        0x02, 0x4c, 0x44, 0x89, 0x76, 0x7e, 0x71, 0xdb, 0x76, 0xd0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0xc4, 0x17, 0x39, 0xcc, 0x86, 0x34, 0x03, 0xd2, 0x18,
        0x46, 0x00, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d, 0x5b, 0x81, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x13, 0x71, 0x4d, 0x54, 0x44, 0xb0, 0x93, 0x13, 0x11, 0x7e, 0x71,
        0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x3c, 0xc3, 0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x40, 0x84, 0xf9, 0xc5, 0x6d, 0x1b, 0x00,
        0xc1, 0x00, 0x48, 0x03, 0x61, 0x20, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x34, 0x14, 0xec, 0x40, 0xc9, 0x0e, 0x94, 0x6e,
        0xc0, 0x0c, 0x40, 0x49, 0x12, 0x14, 0x47, 0xe1, 0x15, 0x4d, 0xa9, 0x06, 0x14, 0x6b, 0x40, 0x89, 0x06, 0x14, 0x21, 0x50, 0x49, 0x42, 0x94, 0x61, 0x00, 0x25, 0xe5, 0x51, 0x06, 0x25, 0x40, 0xc6,
        0x18, 0x01, 0x08, 0x82, 0xfe, 0x1d, 0x06, 0x63, 0x04, 0x20, 0x08, 0x82, 0xb0, 0x37, 0x03, 0x40, 0xd4, 0x1c, 0xc2, 0x72, 0xcd, 0x21, 0x60, 0xca, 0x1c, 0x82, 0xf7, 0xd1, 0x35, 0x07, 0xa1, 0x28,
        0xca, 0x18, 0x8c, 0x00, 0xcc, 0x41, 0x28, 0x8a, 0x22, 0x06, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x80, 0x99, 0x41, 0x36, 0x8c, 0x81, 0x37, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0xd8,
        0x19, 0x68, 0x03, 0x19, 0x7c, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x80, 0xa1, 0xc1, 0x36, 0x94, 0x01, 0x18, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0x01, 0xf1, 0x06, 0x9b, 0x19, 0x8c, 0x18, 0x18,
        0x00, 0x08, 0x82, 0x01, 0x01, 0x07, 0x5c, 0x1a, 0x9c, 0x60, 0xd4, 0x09, 0x46, 0x99, 0xf0, 0xc8, 0xc7, 0x04, 0x48, 0x3e, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x50, 0xc1, 0xc1, 0x18, 0x20, 0xcf,
        0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x54, 0x71, 0x40, 0x06, 0xc8, 0x33, 0x62, 0xf0, 0x00, 0x20, 0x08, 0x06, 0x10, 0x1d, 0x7c, 0x42, 0x40, 0x0c, 0x92, 0xe4, 0x06, 0x6e, 0x20, 0x06, 0xd2, 0x68,
        0x42, 0x00, 0x58, 0x60, 0xc9, 0x67, 0xc4, 0xc0, 0x00, 0x40, 0x10, 0x0c, 0x8e, 0x3c, 0x08, 0x83, 0x60, 0xc4, 0xc0, 0x00, 0x40, 0x10, 0x0c, 0x0e, 0x3d, 0x08, 0x03, 0x64, 0xc4, 0xc0, 0x00, 0x40,
        0x10, 0x0c, 0x8e, 0x3d, 0x08, 0x03, 0x64, 0xc4, 0xe0, 0x00, 0x40, 0x10, 0x0c, 0x2a, 0x3c, 0x58, 0x03, 0xe8, 0x1a, 0x31, 0x38, 0x00, 0x10, 0x04, 0x83, 0x2a, 0x0f, 0xd8, 0x00, 0xba, 0x46, 0x0c,
        0x16, 0x00, 0x04, 0xc1, 0xa0, 0xe9, 0x83, 0x31, 0x10, 0x02, 0x66, 0xd1, 0xe2, 0x60, 0xc4, 0xe0, 0x00, 0x40, 0x10, 0x0c, 0xaa, 0x3d, 0x70, 0x03, 0x0a, 0x1b, 0x31, 0x70, 0x00, 0x10, 0x04, 0x03,
        0xa9, 0x0f, 0xc4, 0x20, 0x90, 0x22, 0x36, 0x38, 0x8c, 0x42, 0x08, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_derivative_2d_unaligned_dxil[] =
    {
#if 0
        Texture2D<float> Tex : register(t0);
        RWTexture2D<float4> SOut : register(u0);
        SamplerState Samp : register(s0);

        // Pretend we're 2D. Since groupsize is not aligned, we get 1D derivatives.
        uint2 rewrite_id(uint2 orig)
        {
                uint sub_quad_x = orig.x & 1u;
                uint sub_quad_y = (orig.x >> 1u) & 1u;

                uint quad_x = orig.x >> 2u;
                uint quad_y = orig.y;

                // Ensure we maintain square-ish dispatch.
                quad_x = 2 * quad_x + (orig.y & 1u);
                quad_y >>= 1u;

                uint2 coord = uint2(quad_x, quad_y) * 2 + uint2(sub_quad_x, sub_quad_y);
                return coord;
        }

        [numthreads(8, 9, 1)]
        void main(uint2 thr : SV_DispatchThreadID)
        {
                thr = rewrite_id(thr);
                float2 uv = float2(thr) / 32.0; // We should sample LOD 2 here.
                float v = round(Tex.Sample(Samp, uv) * 255.0);
                float dx = ddx_fine(uv.x);
                float dy = ddy_fine(uv.y);
                float w = Tex.CalculateLevelOfDetailUnclamped(Samp, uv);
                SOut[thr] = float4(v, dx, dy, w);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0xac, 0x9d, 0x49, 0x0a, 0xf0, 0x39, 0x8f, 0xd6, 0xc6, 0x8a, 0xdf, 0x3b, 0x7d, 0x13, 0x1f, 0xaf, 0x01, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x1c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x99, 0x80, 0xc0, 0x11, 0x48, 0x90, 0xda, 0x4b, 0xc0, 0x02, 0x04, 0x48, 0xf1, 0x5e, 0x5b, 0x34, 0x44, 0x58, 0x49, 0x4c,
        0x9c, 0x07, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xe7, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x84, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0xde, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62,
        0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00,
        0x5a, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c,
        0x10, 0x8c, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x03, 0x50, 0x06, 0x63, 0x30, 0xe8, 0x98, 0x23, 0x40, 0x28, 0xb9, 0x67, 0xb8, 0xfc, 0x09,
        0x7b, 0x08, 0xc9, 0x0f, 0x81, 0x66, 0x58, 0x08, 0x14, 0x28, 0x73, 0x04, 0x41, 0x41, 0x8c, 0x61, 0x59, 0x0c, 0xc3, 0x60, 0xa8, 0xb9, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69,
        0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x54, 0x18, 0x86, 0x61, 0x0c, 0x85, 0x79, 0x86, 0x65, 0x31, 0x0c, 0xc3, 0x18, 0x86, 0xc1, 0x20, 0xa8, 0x2c, 0xc0, 0xb0, 0x0c, 0xc3, 0x60, 0x18, 0x86, 0x91, 0x90,
        0x74, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43, 0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13, 0x42, 0x48, 0x48, 0x18, 0x86, 0x42, 0x2c, 0xc3, 0x42, 0x51, 0x75, 0xd0,
        0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x86, 0x34, 0x03, 0x22, 0x86, 0x61, 0x48, 0x4a, 0xb1, 0x0c, 0xd7, 0xc0, 0x10, 0x36, 0x10, 0x30, 0x53, 0x18, 0x8c, 0x03, 0x3b, 0x84, 0xc3,
        0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x83, 0x1f, 0xe8, 0x81, 0x1e, 0xb4,
        0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0x86, 0x99, 0xac, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87,
        0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07, 0x7a, 0xf0, 0x03, 0x24, 0x18, 0x1b, 0x71, 0xc3, 0x08, 0x84, 0x31, 0x13, 0x19, 0x8c, 0x03, 0x3b, 0x84,
        0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c,
        0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xa0, 0xc0, 0xa3, 0xef, 0x26, 0x69, 0x8a, 0x28, 0x61, 0xf2, 0x59, 0x80, 0x79, 0x16,
        0x22, 0x62, 0x27, 0x60, 0x22, 0x50, 0x30, 0x50, 0x08, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50,
        0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07,
        0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0,
        0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34,
        0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59,
        0x20, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x8a, 0xa0, 0x24,
        0x0a, 0xa1, 0x20, 0x88, 0x19, 0x01, 0xa0, 0x6e, 0x06, 0x80, 0xc0, 0x19, 0x00, 0x12, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62,
        0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x01, 0x99, 0x20, 0x0c, 0xc9, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d,
        0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xca, 0x04, 0xa1, 0x83, 0x08, 0x4c, 0x10, 0x86, 0x65, 0x82, 0x30, 0x30, 0x1b, 0x04, 0x84, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02,
        0x67, 0x82, 0x00, 0x06, 0xd1, 0x04, 0x81, 0x79, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0xd4, 0x6c, 0x08, 0xa4, 0x09, 0x82, 0x18, 0x48, 0x1b, 0x10, 0x84, 0x52, 0x10, 0x64, 0x40,
        0x80, 0x0d, 0x41, 0xb5, 0x81, 0x78, 0x26, 0xc0, 0x9a, 0x20, 0x08, 0x00, 0x89, 0xb6, 0xb0, 0x34, 0xb7, 0x09, 0xc2, 0xd0, 0x4c, 0x10, 0x06, 0x67, 0xc3, 0xb0, 0x31, 0xc3, 0x06, 0x41, 0xe3, 0x36,
        0x14, 0x58, 0x06, 0x5c, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40,
        0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x10, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x46, 0x19,
        0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0x81, 0x55, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0xd0,
        0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19,
        0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x20, 0x00, 0x00, 0x00, 0x66, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0x80, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x02, 0xb0, 0x08, 0xd4,
        0x02, 0x4c, 0x44, 0x89, 0x76, 0x7e, 0x71, 0xdb, 0x76, 0xd0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0xc4, 0x17, 0x39, 0xcc, 0x86, 0x34, 0x03, 0xd2, 0x18,
        0x46, 0x00, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d, 0x5b, 0x81, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x13, 0x71, 0x4d, 0x54, 0x44, 0xb0, 0x93, 0x13, 0x11, 0x7e, 0x71,
        0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x3c, 0xc3, 0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x40, 0x84, 0xf9, 0xc5, 0x6d, 0x1b, 0x00,
        0xc1, 0x00, 0x48, 0x03, 0x61, 0x20, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x34, 0x14, 0xec, 0x40, 0xc9, 0x0e, 0x94, 0x6e,
        0xc0, 0x0c, 0x40, 0x15, 0x94, 0x24, 0x41, 0x71, 0x14, 0x5e, 0xd1, 0x94, 0x6a, 0x40, 0xb1, 0x06, 0x94, 0x68, 0x40, 0x11, 0x02, 0x95, 0x24, 0x44, 0x19, 0x06, 0x50, 0x52, 0x1e, 0x65, 0x50, 0x02,
        0x64, 0x8c, 0x11, 0x80, 0x20, 0xe8, 0xdf, 0x61, 0x30, 0x46, 0x00, 0x82, 0x20, 0x08, 0x7b, 0x33, 0x00, 0x44, 0xcd, 0x21, 0x2c, 0xd8, 0x1c, 0x42, 0xa6, 0xcc, 0x21, 0x7c, 0x60, 0x40, 0xd7, 0x1c,
        0x84, 0xa2, 0x28, 0x64, 0x30, 0x02, 0x30, 0x07, 0xa1, 0x28, 0xca, 0x18, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x80, 0x9d, 0x81, 0x36, 0x90, 0xc1, 0x37, 0x62, 0x80, 0x00,
        0x20, 0x08, 0x06, 0x18, 0x1a, 0x6c, 0x43, 0x19, 0x80, 0xc1, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x60, 0x69, 0xc0, 0x0d, 0x66, 0x10, 0x06, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0xc0, 0x01,
        0x77, 0x06, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0xc4, 0x41, 0xa7, 0x06, 0x25, 0xac, 0x81, 0xd6, 0xc0, 0x06, 0x58, 0x41, 0x1b, 0x68, 0x15, 0x6b, 0x80, 0x16, 0xbc, 0xc1, 0x0d, 0xca, 0x80,
        0x03, 0x2d, 0x21, 0xd8, 0x42, 0xc6, 0x40, 0x4d, 0x98, 0x83, 0x0b, 0x54, 0x90, 0x6c, 0x0d, 0xc8, 0x9e, 0x60, 0xd4, 0x09, 0x46, 0x99, 0xa0, 0xc9, 0xc7, 0x84, 0x4d, 0x3e, 0x23, 0x06, 0x07, 0x00,
        0x82, 0x60, 0x50, 0xf1, 0xc1, 0x1b, 0x4c, 0xda, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x54, 0x7d, 0x00, 0x07, 0x93, 0x36, 0x62, 0xf0, 0x00, 0x20, 0x08, 0x06, 0x10, 0x28, 0xa8, 0x81, 0x10, 0x10,
        0x43, 0xd7, 0xe9, 0x81, 0x1e, 0xb8, 0x41, 0x37, 0x9a, 0x10, 0x00, 0x16, 0x84, 0x81, 0x7c, 0x46, 0x0c, 0x0c, 0x00, 0x04, 0xc1, 0xe0, 0x28, 0x05, 0x36, 0x08, 0x46, 0x0c, 0x0c, 0x00, 0x04, 0xc1,
        0xe0, 0x30, 0x05, 0x36, 0x40, 0x46, 0x0c, 0x0c, 0x00, 0x04, 0xc1, 0xe0, 0x38, 0x05, 0x36, 0x40, 0x46, 0x0c, 0x0e, 0x00, 0x04, 0xc1, 0xa0, 0x22, 0x85, 0x3b, 0xd8, 0xc4, 0x60, 0xc4, 0xe0, 0x00,
        0x40, 0x10, 0x0c, 0xaa, 0x52, 0xc0, 0x83, 0x4d, 0x0c, 0x46, 0x0c, 0x16, 0x00, 0x04, 0xc1, 0xa0, 0x49, 0x05, 0x37, 0x10, 0x02, 0x66, 0x29, 0x83, 0x3e, 0x18, 0x31, 0x38, 0x00, 0x10, 0x04, 0x83,
        0xea, 0x14, 0xf4, 0xe0, 0x1b, 0x83, 0x11, 0x03, 0x07, 0x00, 0x41, 0x30, 0x90, 0x52, 0xa1, 0x0d, 0x02, 0x29, 0xc2, 0x83, 0xc3, 0x28, 0x04, 0x36, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_derivative_1d_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x2a, 0x3a, 0xdd, 0x2c, 0x46, 0x17, 0x61, 0x57, 0x0c, 0x9d, 0x32, 0x5c, 0x8e, 0x6f, 0x45, 0x0e, 0x01, 0x00, 0x00, 0x00, 0xb0, 0x08, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x1c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xdf, 0x2e, 0x92, 0x0f, 0x45, 0x8e, 0x73, 0x49, 0x84, 0xe0, 0xba, 0xf7, 0xca, 0xdb, 0x48, 0x44, 0x58, 0x49, 0x4c,
        0x8c, 0x07, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xe3, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x74, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0xda, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62,
        0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00,
        0x5a, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c,
        0x10, 0x8c, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x03, 0x50, 0x06, 0x63, 0x30, 0xe8, 0x98, 0x23, 0x40, 0x28, 0xb9, 0x67, 0xb8, 0xfc, 0x09,
        0x7b, 0x08, 0xc9, 0x0f, 0x81, 0x66, 0x58, 0x08, 0x14, 0x28, 0x73, 0x04, 0x41, 0x41, 0x8c, 0x61, 0x59, 0x0c, 0xc3, 0x60, 0xa8, 0xb9, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69,
        0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x54, 0x18, 0x86, 0x61, 0x0c, 0x85, 0x79, 0x86, 0x65, 0x31, 0x0c, 0xc3, 0x18, 0x86, 0xc1, 0x20, 0xa8, 0x2c, 0xc0, 0xb0, 0x0c, 0xc3, 0x60, 0x18, 0x86, 0x91, 0x90,
        0x74, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43, 0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13, 0x42, 0x48, 0x48, 0x18, 0x86, 0x42, 0x2c, 0xc3, 0x42, 0x51, 0x75, 0xd0,
        0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x86, 0x34, 0x03, 0x22, 0x86, 0x61, 0x48, 0x4a, 0xb1, 0x0c, 0xd7, 0xc0, 0x10, 0x36, 0x10, 0x30, 0x53, 0x18, 0x8c, 0x03, 0x3b, 0x84, 0xc3,
        0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x83, 0x1f, 0xe8, 0x81, 0x1e, 0xb4,
        0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0x86, 0x99, 0xac, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87,
        0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07, 0x7a, 0xf0, 0x03, 0x24, 0x18, 0x1b, 0x71, 0xc3, 0x08, 0x84, 0x31, 0x13, 0x19, 0x8c, 0x03, 0x3b, 0x84,
        0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c,
        0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xa0, 0xc0, 0xa3, 0xef, 0x26, 0x69, 0x8a, 0x28, 0x61, 0xf2, 0x59, 0x80, 0x79, 0x16,
        0x22, 0x62, 0x27, 0x60, 0x22, 0x50, 0x30, 0x50, 0x08, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50,
        0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07,
        0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0,
        0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34,
        0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59,
        0x20, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x8a, 0xa0, 0x24,
        0x0a, 0xa1, 0x00, 0x03, 0x88, 0x19, 0x01, 0xa0, 0x6e, 0x06, 0x80, 0xc0, 0x19, 0x00, 0x12, 0x67, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62,
        0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x01, 0x99, 0x20, 0x0c, 0xc9, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d,
        0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xca, 0x04, 0xa1, 0x83, 0x08, 0x4c, 0x10, 0x86, 0x65, 0x82, 0x30, 0x30, 0x1b, 0x04, 0x84, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02,
        0x67, 0x82, 0x00, 0x06, 0xd1, 0x04, 0x81, 0x79, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0xd4, 0x6c, 0x08, 0xa4, 0x09, 0x82, 0x18, 0x48, 0x1b, 0x10, 0x84, 0x52, 0x10, 0x64, 0x40,
        0x80, 0x0d, 0x41, 0xb5, 0x81, 0x78, 0x26, 0xc0, 0x9a, 0x20, 0x08, 0x00, 0x89, 0xb6, 0xb0, 0x34, 0xb7, 0x09, 0xc2, 0xd0, 0x4c, 0x10, 0x06, 0x67, 0xc3, 0xb0, 0x0d, 0xc3, 0x06, 0x41, 0xe3, 0x36,
        0x14, 0x58, 0x06, 0x5c, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40,
        0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x10, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x46, 0x19,
        0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0x81, 0x55, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0xd0,
        0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19,
        0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x20, 0x00, 0x00, 0x00, 0x66, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0x80, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x02, 0xb0, 0x08, 0xd4,
        0x02, 0x4c, 0x44, 0x89, 0x76, 0x7e, 0x71, 0xdb, 0x76, 0xd0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0xc4, 0x17, 0x39, 0xcc, 0x86, 0x34, 0x03, 0xd2, 0x18,
        0x46, 0x00, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d, 0x5b, 0x81, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x13, 0x71, 0x4d, 0x54, 0x44, 0xb0, 0x93, 0x13, 0x11, 0x7e, 0x71,
        0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x3c, 0xc3, 0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x40, 0x84, 0xf9, 0xc5, 0x6d, 0x1b, 0x00,
        0xc1, 0x00, 0x48, 0x03, 0x61, 0x20, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x34, 0x14, 0xec, 0x40, 0xc9, 0x0e, 0xcc, 0x00,
        0x94, 0x6e, 0x40, 0x29, 0x94, 0x41, 0x69, 0x94, 0x24, 0x41, 0x71, 0x14, 0x5e, 0xd1, 0x94, 0x6a, 0x40, 0xb1, 0x06, 0x94, 0x68, 0x40, 0x11, 0x02, 0x95, 0x24, 0x44, 0x19, 0x06, 0x50, 0x52, 0x1e,
        0x65, 0x50, 0x02, 0x64, 0x8c, 0x11, 0x80, 0x20, 0xe8, 0xdf, 0x61, 0x30, 0x46, 0x00, 0x82, 0x20, 0x08, 0x7b, 0x33, 0x00, 0x44, 0xcd, 0x21, 0x2c, 0xda, 0x1c, 0xc2, 0xa6, 0xcc, 0x21, 0x84, 0x81,
        0x18, 0xd0, 0x35, 0x07, 0xa1, 0x28, 0x8a, 0x19, 0x8c, 0x00, 0xcc, 0x41, 0x28, 0x8a, 0x52, 0x06, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x80, 0xa5, 0x01, 0x37, 0x98, 0x41,
        0x18, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0x01, 0xa6, 0x06, 0xdd, 0x70, 0x06, 0x62, 0x30, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0xd8, 0x1a, 0x78, 0x03, 0x1a, 0x8c, 0xc1, 0x88, 0x81, 0x01, 0x80,
        0x20, 0x18, 0x10, 0x72, 0xd0, 0xa5, 0x41, 0x05, 0x6c, 0x80, 0x15, 0xb4, 0x81, 0xd6, 0xf0, 0x61, 0x11, 0x1f, 0x56, 0xd0, 0x06, 0x6a, 0x43, 0x1c, 0xdc, 0xa0, 0x8e, 0x30, 0xd0, 0x0a, 0x86, 0xad,
        0xe1, 0xd8, 0x13, 0x8c, 0x3a, 0xc1, 0x28, 0x13, 0x2e, 0xf9, 0x98, 0x80, 0xc9, 0x67, 0xc4, 0xe0, 0x00, 0x40, 0x10, 0x0c, 0xaa, 0x3d, 0x70, 0x03, 0xe8, 0x1a, 0x31, 0x38, 0x00, 0x10, 0x04, 0x83,
        0x8a, 0x0f, 0xde, 0x00, 0xba, 0x46, 0x0c, 0x1e, 0x00, 0x04, 0xc1, 0x00, 0xfa, 0x83, 0x33, 0x10, 0x02, 0x62, 0xd0, 0xb4, 0x3c, 0xc8, 0x03, 0x37, 0xd0, 0x46, 0x13, 0x02, 0xc0, 0x02, 0x4f, 0x3e,
        0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x70, 0x90, 0x42, 0x1a, 0x04, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x70, 0x94, 0x42, 0x1a, 0x20, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x70, 0x98, 0x42, 0x1a,
        0x20, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x50, 0x8d, 0x82, 0x1d, 0x60, 0xdf, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x54, 0xa4, 0x70, 0x07, 0xd8, 0x37, 0x62, 0xb0, 0x00, 0x20, 0x08, 0x06, 0x0d,
        0x2a, 0xac, 0x81, 0x10, 0x30, 0x8b, 0x18, 0xf0, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x54, 0xa6, 0x90, 0x07, 0x1c, 0x18, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0x81, 0x84, 0x0a, 0x6a, 0x10,
        0x48, 0x11, 0x1e, 0x1c, 0x46, 0x21, 0xa4, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const D3D12_SHADER_BYTECODE cs_derivative_2d_aligned = SHADER_BYTECODE(cs_derivative_2d_aligned_dxil);
    static const D3D12_SHADER_BYTECODE cs_derivative_2d_unaligned = SHADER_BYTECODE(cs_derivative_2d_unaligned_dxil);
    static const D3D12_SHADER_BYTECODE cs_derivative_1d = SHADER_BYTECODE(cs_derivative_1d_dxil);

    struct test
    {
        const D3D12_SHADER_BYTECODE *cs;
        unsigned int dispatch_x;
        unsigned int dispatch_y;
    };

    static const struct test tests[] =
    {
        { &cs_derivative_1d, 4, 1 },
        { &cs_derivative_2d_aligned, 2, 2 },
        { &cs_derivative_2d_unaligned, 2, 2 },
    };

    if (!init_compute_test_context(&context))
        return;

    device = context.device;

    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    hr = ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model));
    if (FAILED(hr) || shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_6)
    {
        skip("Device does not support SM 6.6.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&root_signature_desc, 0, sizeof(root_signature_desc));
    memset(root_parameters, 0, sizeof(root_parameters));
    memset(descriptor_ranges, 0, sizeof(descriptor_ranges));
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    root_signature_desc.pParameters = root_parameters;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = ARRAY_SIZE(descriptor_ranges);
    root_parameters[0].DescriptorTable.pDescriptorRanges = descriptor_ranges;
    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 0;
    descriptor_ranges[0].NumDescriptors = 1;
    descriptor_ranges[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;
    descriptor_ranges[1].OffsetInDescriptorsFromTableStart = 1;
    descriptor_ranges[1].NumDescriptors = 1;
    root_signature_desc.NumStaticSamplers = ARRAY_SIZE(static_samplers);
    root_signature_desc.pStaticSamplers = static_samplers;
    memset(static_samplers, 0, sizeof(static_samplers));
    static_samplers[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    static_samplers[0].Filter = D3D12_FILTER_MIN_MAG_MIP_POINT;
    static_samplers[0].AddressU = D3D12_TEXTURE_ADDRESS_MODE_CLAMP;
    static_samplers[0].AddressV = D3D12_TEXTURE_ADDRESS_MODE_CLAMP;
    static_samplers[0].AddressW = D3D12_TEXTURE_ADDRESS_MODE_CLAMP;
    static_samplers[0].MaxLOD = 1000.0f;
    static_samplers[0].RegisterSpace = 0;
    static_samplers[0].ShaderRegister = 0;

    create_root_signature(device, &root_signature_desc, &root_signature);
    heap = create_gpu_descriptor_heap(device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 2);

    input_texture = create_default_texture2d(device, 128, 128, 1, 8,
            DXGI_FORMAT_R8_UNORM, D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);
    result_texture = create_default_texture2d(device, 16, 16, 1, 1,
            DXGI_FORMAT_R32G32B32A32_FLOAT, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    {
        D3D12_SUBRESOURCE_DATA data[8];
        uint8_t buffer_data[128 * 128 * 2];
        uint8_t *pbuffer = buffer_data;
        unsigned int width;

        for (i = 0; i < ARRAY_SIZE(data); i++)
        {
            width = 128 >> i;
            memset(pbuffer, i + 1, width * width);
            data[i].RowPitch = width;
            data[i].SlicePitch = data[i].RowPitch * width;
            data[i].pData = pbuffer;
            pbuffer += width * width;
        }

        upload_texture_data(input_texture, data, 8, context.queue, context.list);
        reset_command_list(context.list, context.allocator);
        transition_resource_state(context.list, input_texture,
                   D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    }

    {
        D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
        memset(&srv_desc, 0, sizeof(srv_desc));
        srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        srv_desc.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE2D;
        srv_desc.Format = DXGI_FORMAT_R8_UNORM;
        srv_desc.Texture2D.MipLevels = 8;
        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
        ID3D12Device_CreateShaderResourceView(device, input_texture, &srv_desc, cpu_handle);
    }

    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
        memset(&uav_desc, 0, sizeof(uav_desc));
        uav_desc.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;
        uav_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;

        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
        cpu_handle.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
        ID3D12Device_CreateUnorderedAccessView(device, result_texture, NULL, &uav_desc, cpu_handle);
    }

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);

        pso = create_compute_pipeline_state(device, root_signature, *tests[i].cs);
        ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &heap);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0,
                ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap));
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
        ID3D12GraphicsCommandList_Dispatch(context.list, tests[i].dispatch_x, tests[i].dispatch_y, 1);
        transition_resource_state(context.list, result_texture,
                D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

        get_texture_readback_with_command_list(result_texture, 0, &rb, context.queue, context.list);

        for (y = 0; y < 16; y++)
        {
            for (x = 0; x < 16; x++)
            {
                const struct vec4 *values;
                struct vec4 expected;

                values = get_readback_vec4(&rb, x, y);

                expected.x = 1.0f + 2.0f; /* Expect to sample LOD 2. */
                expected.y = 1.0f / 32.0f; /* ddx_fine of UV.x is 1 / 32. */
                expected.z = 1.0f / 32.0f; /* ddy_fine of UV.y is also 1 / 32. */
                expected.w = 2.0f; /* queried LOD */

                ok(compare_vec4(values, &expected, 0),
                        "Mismatch at %u, %u: Expected (%f, %f, %f, %f), got (%f %f %f %f).\n",
                        x, y, expected.x, expected.y, expected.z, expected.w,
                        values->x, values->y, values->z, values->w);
            }
        }
        reset_command_list(context.list, context.allocator);
        transition_resource_state(context.list, result_texture,
                                  D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12PipelineState_Release(pso);
        release_resource_readback(&rb);
    }
    vkd3d_test_set_context(NULL);
    ID3D12Resource_Release(input_texture);
    ID3D12Resource_Release(result_texture);
    ID3D12DescriptorHeap_Release(heap);
    ID3D12RootSignature_Release(root_signature);
    destroy_test_context(&context);
}

void test_shader_sm66_wave_size(void)
{
    D3D12_COMPUTE_PIPELINE_STATE_DESC compute_desc;
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_FEATURE_DATA_D3D12_OPTIONS1 options1;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    struct test_context context;
    struct resource_readback rb;
    uint32_t expected_data[128];
    ID3D12PipelineState *pso;
    uint32_t input_data[128];
    bool supported_wave_size;
    ID3D12Resource *src;
    ID3D12Resource *dst;
    unsigned int i, j;
    uint32_t value;
    HRESULT hr;

#if 0
    StructuredBuffer<uint> RO : register(t0);
    RWStructuredBuffer<uint> RW : register(u0);

    void run(uint thr)
    {
        RW[thr] = WavePrefixSum(RO[thr]);
        RW[thr + 128] = WaveGetLaneCount();
    }

    [WaveSize(16)]
    [numthreads(128, 1, 1)]
    void main16(uint thr : SV_DispatchThreadID)
    {
        run(thr);
    }

    [WaveSize(32)]
    [numthreads(128, 1, 1)]
    void main32(uint thr : SV_DispatchThreadID)
    {
        run(thr);
    }

    [WaveSize(64)]
    [numthreads(128, 1, 1)]
    void main64(uint thr : SV_DispatchThreadID)
    {
        run(thr);
    }
#endif

    static const BYTE cs_code_wave16[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xf7, 0x33, 0x8b, 0x40, 0x13, 0x56, 0x9a, 0x59, 0x8b, 0x39, 0x62, 0x93, 0xa8, 0xa6, 0x69, 0x8d, 0x01, 0x00, 0x00, 0x00, 0xcc, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4c, 0x8c, 0x6c, 0x5b, 0x7b, 0xd5, 0xe2, 0x2d, 0x5e, 0x24, 0x39, 0xd8,
        0x0e, 0x86, 0xa2, 0xee, 0x44, 0x58, 0x49, 0x4c, 0xc0, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xb0, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0xa8, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xa7, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x48, 0x29, 0xc6, 0x18, 0xc6, 0xd0, 0x21,
        0x73, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x48, 0x65, 0x01, 0x43, 0x8d, 0x31, 0xc6, 0x18, 0x66, 0xd0, 0xba, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9,
        0x5f, 0x09, 0x69, 0x25, 0x26, 0x1f, 0xb9, 0x6d, 0x54, 0x8c, 0x31, 0xc6, 0x28, 0x47, 0x1b, 0x6a, 0x0c, 0x33, 0xc8, 0x15, 0x61, 0x0c, 0x82, 0xb7, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x2b,
        0x21, 0x39, 0x54, 0x24, 0x10, 0x69, 0xe4, 0x3c, 0x44, 0x34, 0x21, 0x84, 0x84, 0x84, 0x31, 0x0a, 0xa1, 0x86, 0x92, 0x34, 0x0f, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0xda, 0x90,
        0x66, 0x40, 0xc4, 0x18, 0xc3, 0xcc, 0x11, 0x04, 0xa5, 0x50, 0xa3, 0x0e, 0x4b, 0x77, 0x20, 0x60, 0x26, 0x32, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x98, 0x05, 0x7a, 0x90, 0x87,
        0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x83, 0x7a, 0x70, 0x87, 0x79, 0x48, 0x87, 0x73, 0x70, 0x87, 0x72, 0x20,
        0x07, 0x30, 0x48, 0x07, 0x77, 0xa0, 0x07, 0x3f, 0x40, 0xc1, 0x20, 0x3d, 0x93, 0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d,
        0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03,
        0x18, 0xa4, 0x83, 0x3b, 0xd0, 0x83, 0x1f, 0xa0, 0x60, 0x10, 0x9f, 0x23, 0x00, 0x05, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0,
        0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6,
        0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0xf2, 0x2c, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x03, 0x01, 0x01,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf, 0x05, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23,
        0x00, 0x85, 0x51, 0x08, 0x05, 0x28, 0x50, 0x16, 0x05, 0x42, 0x76, 0x04, 0x80, 0x7c, 0x81, 0x02, 0x02, 0x51, 0x9f, 0x01, 0xa0, 0x3d, 0x03, 0x00, 0x79, 0x18, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x01, 0x99, 0x20, 0x0c, 0xc9, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xca, 0x04, 0x61, 0x9b, 0x08, 0x4c, 0x10, 0x86, 0x65, 0x82, 0x30, 0x30, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xd0, 0x49, 0x13, 0x04, 0x0b, 0xda, 0xb0, 0x20, 0x90, 0x82, 0x20, 0xc3, 0x12, 0x45, 0x51, 0xb3, 0x21, 0x90, 0x36, 0x10, 0xcf, 0x04, 0x00, 0x13, 0x04, 0x01,
        0xa0, 0xd1, 0x16, 0x96, 0xe6, 0x16, 0x63, 0x33, 0x41, 0xf0, 0xa2, 0x09, 0xc2, 0xd0, 0x6c, 0x18, 0xb0, 0x61, 0x98, 0x20, 0x0c, 0xce, 0x04, 0x61, 0x78, 0x36, 0x04, 0xdb, 0x06, 0x03, 0xb9, 0x98,
        0x4c, 0xe3, 0x36, 0x14, 0x95, 0x05, 0x50, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd,
        0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x10, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6,
        0x04, 0x46, 0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0xc1, 0x54, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e,
        0x6e, 0x4a, 0xd0, 0x01, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19,
        0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x66, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13,
        0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x18, 0xc1, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc,
        0xb6, 0x0d, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30,
        0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x15, 0x58, 0xc3, 0xe5, 0x3b, 0x8f, 0x6f, 0x01, 0x15, 0x01, 0x12, 0x53, 0x09, 0x34, 0x04, 0xe7, 0x50, 0xcd, 0x64, 0x02, 0xd6, 0x70, 0xf9, 0xce, 0xe3, 0x5b, 0x40,
        0x45, 0xa4, 0x11, 0x51, 0x20, 0x17, 0xfa, 0xf8, 0xc8, 0x6d, 0x1b, 0x80, 0xc1, 0x00, 0x48, 0x53, 0xeb, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x0a, 0x76, 0xa0, 0x64, 0x07, 0x0a, 0x53, 0xa0, 0x74, 0x03, 0xca, 0x52, 0xa0, 0xe4, 0x03, 0x0a, 0x13, 0xa8, 0xc0, 0x03, 0xc8,
        0x94, 0xc0, 0x08, 0x00, 0xc9, 0x39, 0x84, 0x8d, 0x99, 0x43, 0x58, 0x18, 0xaa, 0x73, 0x10, 0x8a, 0xa2, 0x74, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x71,
        0x8d, 0x90, 0x4d, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x75, 0x8e, 0xa0, 0x51, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x90, 0x41, 0xb3, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0x04,
        0x06, 0x91, 0x60, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0x81, 0x33, 0x06, 0x4e, 0x20, 0x74, 0xca, 0x36, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x9c, 0xc1, 0x13, 0x2c, 0xcc,
        0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x64, 0x50, 0x1d, 0xcb, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x2c, 0x68, 0x40, 0x05, 0x46, 0x18, 0x08, 0xd7, 0x75, 0x39, 0xdf, 0x88, 0x41, 0x01, 0x80,
        0x20, 0x18, 0x40, 0x66, 0xf0, 0xd4, 0xf1, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x67, 0x80, 0x29, 0xce, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x2c, 0x6b, 0x70, 0x05, 0x02, 0x19, 0x0c,
        0x9a, 0xa6, 0x45, 0x62, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_wave32[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xd5, 0xc0, 0x67, 0xf6, 0x60, 0xf8, 0xe3, 0xa7, 0xef, 0x8b, 0xf9, 0x46, 0xc5, 0xd6, 0x05, 0xfe, 0x01, 0x00, 0x00, 0x00, 0xcc, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x31, 0xdb, 0xbd, 0x71, 0xea, 0xcb, 0x8c, 0x1e, 0x06, 0x1d, 0x3b, 0xf8,
        0x15, 0x9f, 0x34, 0xc0, 0x44, 0x58, 0x49, 0x4c, 0xc0, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xb0, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0xa8, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xa7, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x48, 0x29, 0xc6, 0x18, 0xc6, 0xd0, 0x21,
        0x73, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x48, 0x65, 0x01, 0x43, 0x8d, 0x31, 0xc6, 0x18, 0x66, 0xd0, 0xba, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9,
        0x5f, 0x09, 0x69, 0x25, 0x26, 0x1f, 0xb9, 0x6d, 0x54, 0x8c, 0x31, 0xc6, 0x28, 0x47, 0x1b, 0x6a, 0x0c, 0x33, 0xc8, 0x15, 0x61, 0x0c, 0x82, 0xb7, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x2b,
        0x21, 0x39, 0x54, 0x24, 0x10, 0x69, 0xe4, 0x3c, 0x44, 0x34, 0x21, 0x84, 0x84, 0x84, 0x31, 0x0a, 0xa1, 0x86, 0x92, 0x34, 0x0f, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0xda, 0x90,
        0x66, 0x40, 0xc4, 0x18, 0xc3, 0xcc, 0x11, 0x04, 0xa5, 0x50, 0xa3, 0x0e, 0x4b, 0x77, 0x20, 0x60, 0x26, 0x32, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x98, 0x05, 0x7a, 0x90, 0x87,
        0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x83, 0x7a, 0x70, 0x87, 0x79, 0x48, 0x87, 0x73, 0x70, 0x87, 0x72, 0x20,
        0x07, 0x30, 0x48, 0x07, 0x77, 0xa0, 0x07, 0x3f, 0x40, 0xc1, 0x20, 0x3d, 0x93, 0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d,
        0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03,
        0x18, 0xa4, 0x83, 0x3b, 0xd0, 0x83, 0x1f, 0xa0, 0x60, 0x10, 0x9f, 0x23, 0x00, 0x05, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0,
        0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6,
        0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0xf2, 0x2c, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x03, 0x01, 0x01,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf, 0x05, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23,
        0x00, 0x85, 0x51, 0x08, 0x05, 0x28, 0x50, 0x16, 0x05, 0x44, 0x76, 0x04, 0x80, 0x7c, 0x81, 0x02, 0x02, 0x51, 0x9f, 0x01, 0xa0, 0x3d, 0x03, 0x00, 0x79, 0x18, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x01, 0x99, 0x20, 0x0c, 0xc9, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xca, 0x04, 0x61, 0x9b, 0x08, 0x4c, 0x10, 0x86, 0x65, 0x82, 0x30, 0x30, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xd0, 0x49, 0x13, 0x04, 0x0b, 0xda, 0xb0, 0x20, 0x90, 0x82, 0x20, 0xc3, 0x12, 0x45, 0x51, 0xb3, 0x21, 0x90, 0x36, 0x10, 0xcf, 0x04, 0x00, 0x13, 0x04, 0x01,
        0xa0, 0xd1, 0x16, 0x96, 0xe6, 0x36, 0x23, 0x33, 0x41, 0xf0, 0xa2, 0x09, 0xc2, 0xd0, 0x6c, 0x18, 0xb0, 0x61, 0x98, 0x20, 0x0c, 0xce, 0x04, 0x61, 0x78, 0x36, 0x04, 0xdb, 0x06, 0x03, 0xb9, 0x98,
        0x4c, 0xe3, 0x36, 0x14, 0x95, 0x05, 0x50, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd,
        0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x10, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6,
        0x04, 0x46, 0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0xc1, 0x54, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e,
        0x6e, 0x4a, 0xd0, 0x01, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19,
        0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x66, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13,
        0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x18, 0xc1, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc,
        0xb6, 0x0d, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30,
        0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x15, 0x58, 0xc3, 0xe5, 0x3b, 0x8f, 0x6f, 0x01, 0x15, 0x01, 0x12, 0x53, 0x09, 0x34, 0x04, 0xe7, 0x50, 0xcd, 0x64, 0x02, 0xd6, 0x70, 0xf9, 0xce, 0xe3, 0x5b, 0x40,
        0x45, 0xa4, 0x11, 0x51, 0x20, 0x17, 0xfa, 0xf8, 0xc8, 0x6d, 0x1b, 0x80, 0xc1, 0x00, 0x48, 0x73, 0xdb, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x0a, 0x76, 0xa0, 0x64, 0x07, 0x0a, 0x53, 0xa0, 0x74, 0x03, 0xca, 0x52, 0xa0, 0xe4, 0x03, 0x0a, 0x13, 0xa8, 0xc0, 0x03, 0xc8,
        0x94, 0xc0, 0x08, 0x00, 0xc9, 0x39, 0x84, 0x8d, 0x99, 0x43, 0x58, 0x18, 0xaa, 0x73, 0x10, 0x8a, 0xa2, 0x74, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x71,
        0x8d, 0x90, 0x4d, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x75, 0x8e, 0xa0, 0x51, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x90, 0x41, 0xb3, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0x04,
        0x06, 0x91, 0x60, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0x81, 0x33, 0x06, 0x4e, 0x20, 0x74, 0xca, 0x36, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x9c, 0xc1, 0x13, 0x2c, 0xcc,
        0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x64, 0x50, 0x1d, 0xcb, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x2c, 0x68, 0x40, 0x05, 0x46, 0x18, 0x08, 0xd7, 0x75, 0x39, 0xdf, 0x88, 0x41, 0x01, 0x80,
        0x20, 0x18, 0x40, 0x66, 0xf0, 0xd4, 0xf1, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x67, 0x80, 0x29, 0xce, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x2c, 0x6b, 0x70, 0x05, 0x02, 0x19, 0x0c,
        0x9a, 0xa6, 0x45, 0x62, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_wave64[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x00, 0x4b, 0x2d, 0xb6, 0x10, 0x01, 0xd5, 0xf8, 0x09, 0x51, 0x77, 0x33, 0xb9, 0x4b, 0x99, 0xf2, 0x01, 0x00, 0x00, 0x00, 0xd0, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x2d, 0xc8, 0x0b, 0x8a, 0x13, 0xfa, 0x7f, 0x86, 0xec, 0xdd, 0xf2,
        0xec, 0x64, 0xb6, 0xd0, 0x44, 0x58, 0x49, 0x4c, 0xc4, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xb1, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0xac, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xa8, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x48, 0x29, 0xc6, 0x18, 0xc6, 0xd0, 0x21,
        0x73, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x48, 0x65, 0x01, 0x43, 0x8d, 0x31, 0xc6, 0x18, 0x66, 0xd0, 0xba, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9,
        0x5f, 0x09, 0x69, 0x25, 0x26, 0x1f, 0xb9, 0x6d, 0x54, 0x8c, 0x31, 0xc6, 0x28, 0x47, 0x1b, 0x6a, 0x0c, 0x33, 0xc8, 0x15, 0x61, 0x0c, 0x82, 0xb7, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x2b,
        0x21, 0x39, 0x54, 0x24, 0x10, 0x69, 0xe4, 0x3c, 0x44, 0x34, 0x21, 0x84, 0x84, 0x84, 0x31, 0x0a, 0xa1, 0x86, 0x92, 0x34, 0x0f, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0xda, 0x90,
        0x66, 0x40, 0xc4, 0x18, 0xc3, 0xcc, 0x11, 0x04, 0xa5, 0x50, 0xa3, 0x0e, 0x4b, 0x77, 0x20, 0x60, 0x26, 0x32, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x98, 0x05, 0x7a, 0x90, 0x87,
        0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x83, 0x7a, 0x70, 0x87, 0x79, 0x48, 0x87, 0x73, 0x70, 0x87, 0x72, 0x20,
        0x07, 0x30, 0x48, 0x07, 0x77, 0xa0, 0x07, 0x3f, 0x40, 0xc1, 0x20, 0x3d, 0x93, 0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d,
        0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03,
        0x18, 0xa4, 0x83, 0x3b, 0xd0, 0x83, 0x1f, 0xa0, 0x60, 0x10, 0x9f, 0x23, 0x00, 0x05, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0,
        0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6,
        0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0xf2, 0x2c, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x03, 0x01, 0x01,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf, 0x05, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23,
        0x00, 0x85, 0x51, 0x08, 0x05, 0x28, 0x50, 0x16, 0x05, 0x18, 0x40, 0x76, 0x04, 0x80, 0x7c, 0x81, 0x02, 0x02, 0x51, 0x9f, 0x01, 0xa0, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x41, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41,
        0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x01, 0x99, 0x20, 0x0c,
        0xc9, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xca, 0x04, 0x61, 0x9b, 0x08, 0x4c, 0x10, 0x86, 0x65, 0x82, 0x30, 0x30, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20,
        0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xd0, 0x49, 0x13, 0x04, 0x0b, 0xda, 0xb0, 0x20, 0x90, 0x82, 0x20, 0xc3, 0x12, 0x45, 0x51, 0xb3, 0x21, 0x90, 0x36, 0x10, 0xcf, 0x04,
        0x00, 0x13, 0x04, 0x01, 0xa0, 0xd1, 0x16, 0x96, 0xe6, 0x66, 0x43, 0x33, 0x41, 0xf0, 0xa2, 0x09, 0xc2, 0xd0, 0x6c, 0x18, 0xb0, 0x61, 0x98, 0x20, 0x0c, 0xce, 0x04, 0x61, 0x78, 0x36, 0x04, 0xdb,
        0x06, 0x03, 0xb9, 0x98, 0x4c, 0xe3, 0x36, 0x14, 0x95, 0x05, 0x50, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c,
        0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x10, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde,
        0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x46, 0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0xc1, 0x54, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea,
        0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0xd0, 0x01, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3,
        0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30,
        0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07,
        0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d,
        0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76,
        0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87,
        0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c,
        0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8,
        0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87,
        0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00,
        0x71, 0x20, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x66, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8,
        0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x18, 0xc1, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9,
        0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x0d, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe,
        0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x15, 0x58, 0xc3, 0xe5, 0x3b, 0x8f, 0x6f, 0x01, 0x15, 0x01, 0x12, 0x53, 0x09, 0x34, 0x04, 0xe7, 0x50, 0xcd, 0x64, 0x02, 0xd6, 0x70, 0xf9,
        0xce, 0xe3, 0x5b, 0x40, 0x45, 0xa4, 0x11, 0x51, 0x20, 0x17, 0xfa, 0xf8, 0xc8, 0x6d, 0x1b, 0x80, 0xc1, 0x00, 0x48, 0xa3, 0xe3, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00,
        0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x0a, 0x76, 0xa0, 0x64, 0x07, 0x0a, 0x53, 0xa0, 0x74, 0x03, 0xca, 0x52, 0xa0, 0xe4, 0x03, 0x0a, 0x13,
        0xa8, 0xc0, 0x03, 0xc8, 0x94, 0xc0, 0x08, 0x00, 0xc9, 0x39, 0x84, 0x8d, 0x99, 0x43, 0x58, 0x18, 0xaa, 0x73, 0x10, 0x8a, 0xa2, 0x74, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00,
        0x82, 0x60, 0x70, 0x71, 0x8d, 0x90, 0x4d, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x75, 0x8e, 0xa0, 0x51, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x90, 0x41, 0xb3, 0x8d, 0x18, 0x1c, 0x00,
        0x08, 0x82, 0xc1, 0x04, 0x06, 0x91, 0x60, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0x81, 0x33, 0x06, 0x4e, 0x20, 0x74, 0xca, 0x36, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x9c,
        0xc1, 0x13, 0x2c, 0xcc, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x64, 0x50, 0x1d, 0xcb, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x2c, 0x68, 0x40, 0x05, 0x46, 0x18, 0x08, 0xd7, 0x75, 0x39, 0xdf,
        0x88, 0x41, 0x01, 0x80, 0x20, 0x18, 0x40, 0x66, 0xf0, 0xd4, 0xf1, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x67, 0x80, 0x29, 0xce, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x2c, 0x6b, 0x70,
        0x05, 0x02, 0x19, 0x0c, 0x9a, 0xa6, 0x45, 0x62, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const struct D3D12_SHADER_BYTECODE cs_wave16 = SHADER_BYTECODE(cs_code_wave16);
    static const struct D3D12_SHADER_BYTECODE cs_wave32 = SHADER_BYTECODE(cs_code_wave32);
    static const struct D3D12_SHADER_BYTECODE cs_wave64 = SHADER_BYTECODE(cs_code_wave64);

    struct test
    {
        const struct D3D12_SHADER_BYTECODE *cs;
        unsigned int wave_size;
    };

    static const struct test tests[] =
    {
        { &cs_wave16, 16 },
        { &cs_wave32, 32 },
        { &cs_wave64, 64 },
    };

    if (!init_compute_test_context(&context))
        return;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&shader_model, 0, sizeof(shader_model));
    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    if (FAILED(ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model))) ||
            shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_6)
    {
        skip("Device does not support SM 6.6.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&rs_desc, 0, sizeof(rs_desc));
    rs_desc.NumParameters = ARRAY_SIZE(root_parameters);
    rs_desc.pParameters = root_parameters;
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    create_root_signature(context.device, &rs_desc, &context.root_signature);

    memset(&options1, 0, sizeof(options1));
    ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS1, &options1, sizeof(options1));

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);

        /* AMD Windows driver trips device lost if we give it wave16, so it's clear runtime doesn't validate.
         * If wave size is in range however, it must work. */
        supported_wave_size = tests[i].wave_size >= options1.WaveLaneCountMin && tests[i].wave_size <= options1.WaveLaneCountMax;
        if (!supported_wave_size)
        {
            skip("WaveSize %u not supported, skipping. Trips undefined behavior in driver to create PSO.\n", tests[i].wave_size);
            continue;
        }

        pso = NULL;
        memset(&compute_desc, 0, sizeof(compute_desc));
        compute_desc.CS = *tests[i].cs;
        compute_desc.pRootSignature = context.root_signature;

        hr = ID3D12Device_CreateComputePipelineState(context.device, &compute_desc, &IID_ID3D12PipelineState, (void**)&pso);
        ok(SUCCEEDED(hr), "Failed to create PSO, hr #%x.\n", hr);

        if (SUCCEEDED(hr))
        {
            for (j = 0; j < ARRAY_SIZE(input_data); j++)
            {
                input_data[j] = 100;
                expected_data[j] = 100 * (j & (tests[i].wave_size - 1));
            }

            src = create_upload_buffer(context.device, sizeof(input_data), input_data);
            dst = create_default_buffer(context.device, sizeof(input_data) * 2,
                    D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
            ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, context.root_signature);
            ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
            ID3D12GraphicsCommandList_SetComputeRootShaderResourceView(context.list, 0, ID3D12Resource_GetGPUVirtualAddress(src));
            ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, ID3D12Resource_GetGPUVirtualAddress(dst));
            ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);

            transition_resource_state(context.list, dst, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
            get_buffer_readback_with_command_list(dst, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

            for (j = 0; j < ARRAY_SIZE(input_data); j++)
            {
                value = get_readback_uint(&rb, j, 0, 0);
                ok(value == expected_data[j], "Prefix sum mismatch, value %u: %u != %u\n", j, value, expected_data[j]);
                value = get_readback_uint(&rb, j + 128, 0, 0);
                ok(value == tests[i].wave_size, "Expected wave size: %u, got %u\n", tests[i].wave_size, value);
            }

            ID3D12Resource_Release(src);
            ID3D12Resource_Release(dst);
            ID3D12PipelineState_Release(pso);
            release_resource_readback(&rb);
            reset_command_list(context.list, context.allocator);
        }
    }

    vkd3d_test_set_context(NULL);
    destroy_test_context(&context);
}

void test_shader_sm66_quad_op_semantics(void)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[2];
    D3D12_ROOT_PARAMETER root_parameters[1];
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    ID3D12RootSignature *root_signature;
    ID3D12Resource *result_texture;
    ID3D12Resource *input_texture;
    struct test_context context;
    struct resource_readback rb;
    ID3D12DescriptorHeap *heap;
    ID3D12PipelineState *pso;
    ID3D12Device *device;
    unsigned int i, x, y;

    static const BYTE cs_quad_swap_2d_sm66_dxil[] =
    {
#if 0
        Texture2D<float> Tex : register(t0);
        RWTexture2D<float4> SOut : register(u0);

        [numthreads(8, 8, 1)]
        void main(uint2 thr : SV_DispatchThreadID)
        {
                float v = Tex.Load(uint3(thr, 0));
                float horiz = QuadReadAcrossX(v);
                float vert = QuadReadAcrossY(v);
                float diag = QuadReadAcrossDiagonal(v);
                SOut[thr] = float4(v, horiz, vert, diag);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x46, 0x0c, 0x5f, 0x1b, 0x8c, 0x47, 0x84, 0x5f, 0x71, 0x5a, 0x78, 0xf1, 0x25, 0x29, 0x25, 0xdc, 0x01, 0x00, 0x00, 0x00, 0xc0, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb7, 0x7d, 0x5d, 0x45, 0xd1, 0x03, 0xfa, 0xfd, 0x77, 0xa5, 0x7b, 0x08,
        0xc1, 0x47, 0x9f, 0x1a, 0x44, 0x58, 0x49, 0x4c, 0xb4, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xad, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x9c, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xa4, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x80, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x03, 0x70, 0xd3, 0x70, 0xf9, 0x13, 0xf6,
        0x10, 0x92, 0xbf, 0x12, 0xd2, 0x4a, 0x4c, 0x7e, 0x71, 0xdb, 0xa8, 0x30, 0x0c, 0xc3, 0x18, 0xe6, 0x08, 0x10, 0x42, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02,
        0x05, 0x49, 0x51, 0x8e, 0x41, 0x19, 0x86, 0x61, 0x18, 0x86, 0x81, 0x96, 0x42, 0x18, 0x83, 0x81, 0x50, 0x53, 0x16, 0x60, 0x50, 0x86, 0x61, 0x30, 0x0c, 0xc3, 0x40, 0xe8, 0xb9, 0x6d, 0xb8, 0xfc,
        0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0xc9, 0xa1, 0x22, 0x81, 0x48, 0x23, 0xe7, 0x21, 0xa2, 0x09, 0x21, 0x24, 0x24, 0x0c, 0x43, 0x21, 0x94, 0x41, 0x89, 0x48, 0x3a, 0x68, 0xb8, 0xfc, 0x09, 0x7b,
        0x08, 0xc9, 0x5f, 0x09, 0x69, 0x43, 0x9a, 0x01, 0x11, 0xc3, 0x30, 0x20, 0x73, 0x04, 0x41, 0x29, 0x94, 0x81, 0x1a, 0x2a, 0xb2, 0x06, 0x02, 0x66, 0x0a, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87,
        0x79, 0x70, 0x03, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07, 0x7a, 0xf0, 0x03, 0x3d, 0xd0, 0x83, 0x76, 0x48,
        0x07, 0x78, 0x98, 0x87, 0x5f, 0xa0, 0x87, 0x7c, 0x80, 0x87, 0x72, 0x40, 0xc1, 0x30, 0x93, 0x35, 0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f,
        0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40, 0x0f, 0x7e, 0x80, 0x04, 0x23, 0x23, 0x6d, 0x18, 0x81, 0x30, 0x66, 0x22, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98,
        0x87, 0x79, 0x70, 0x03, 0x59, 0xb8, 0x05, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x7b, 0x28, 0x87, 0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e,
        0x30, 0x07, 0x76, 0x78, 0x87, 0x70, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14, 0x70, 0xd4, 0xcd, 0x11, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0,
        0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0,
        0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x79, 0x16, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x2c, 0x10, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x8a, 0xa0, 0x24,
        0x0a, 0xa1, 0x20, 0xa8, 0x1a, 0x01, 0xa0, 0xaf, 0x00, 0x01, 0x81, 0xc8, 0x9b, 0x01, 0xa0, 0x6d, 0x06, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62,
        0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d,
        0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x61, 0x8b, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x04, 0x84, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02,
        0x67, 0x82, 0xe0, 0x41, 0x13, 0x84, 0xca, 0xd9, 0xb0, 0x20, 0x90, 0x82, 0x20, 0xc3, 0x12, 0x45, 0x51, 0xb3, 0x21, 0x90, 0x36, 0x10, 0xcf, 0x04, 0x00, 0x13, 0x04, 0x01, 0x20, 0xd1, 0x16, 0x96,
        0xe6, 0x36, 0x41, 0xf8, 0x9e, 0x09, 0xc2, 0xc0, 0x4c, 0x10, 0x86, 0x66, 0xc3, 0x90, 0x65, 0xc3, 0x06, 0x02, 0xb9, 0x30, 0x6d, 0x43, 0x51, 0x59, 0x00, 0xb5, 0x55, 0x61, 0x63, 0xb3, 0x6b, 0x73,
        0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9,
        0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b,
        0x9b, 0x9b, 0x12, 0x4c, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
        0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10,
        0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03,
        0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e,
        0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b,
        0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90,
        0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e,
        0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca,
        0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82,
        0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06,
        0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0,
        0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48,
        0x63, 0xd8, 0x00, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x10, 0x05, 0x0c, 0xe8, 0xe3, 0x17, 0xb7, 0x6d, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0xe0, 0x17,
        0xb7, 0x6d, 0x04, 0xd6, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0xc1, 0x4e, 0x4e, 0x44, 0xf8, 0xc5, 0x6d, 0x5b, 0x80, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0xd3, 0x11, 0x11, 0xc0, 0x20,
        0x0e, 0x3e, 0x72, 0xdb, 0x06, 0x40, 0x30, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x34, 0xcc, 0x00, 0x94, 0x7d, 0x40, 0xc9, 0x0e, 0x94, 0x6e, 0x40, 0xc1, 0x0e, 0x94, 0x24, 0x41, 0x11, 0x06, 0x14, 0x21, 0x50, 0x49, 0x42, 0x94, 0x61, 0x00, 0x21, 0x25, 0x30, 0x02, 0x50, 0x04,
        0xe5, 0x41, 0xd1, 0x1c, 0x82, 0x72, 0xcd, 0x21, 0x64, 0x1a, 0x51, 0x73, 0x10, 0x49, 0x92, 0x70, 0x23, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x79, 0x90, 0xb0, 0x59, 0x23,
        0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x7d, 0x91, 0xc0, 0x5d, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x94, 0x41, 0xd4, 0x8d, 0x18, 0x18, 0x00, 0x08, 0x82, 0x01, 0x61, 0x06, 0x12, 0x18, 0x8c,
        0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0x34, 0x06, 0xd2, 0x80, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0xc1, 0x72, 0x06, 0x51, 0x00, 0x06, 0x83, 0x70, 0x5d, 0xd7, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06,
        0x07, 0x00, 0x82, 0x60, 0xd0, 0xa0, 0x01, 0x16, 0x38, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0xa4, 0x41, 0x26, 0x40, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0xa8, 0x81, 0x36, 0x3c, 0x23,
        0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0xa5, 0x01, 0xa6, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xf0, 0xac, 0x01, 0x15, 0x24, 0x48, 0x57, 0x10, 0x83, 0x00, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_quad_swap_2d_sm60_dxil[] =
    {
#if 0
        Texture2D<float> Tex : register(t0);
        RWTexture2D<float4> SOut : register(u0);

        [numthreads(8, 8, 1)]
        void main(uint2 thr : SV_DispatchThreadID)
        {
                float v = Tex.Load(uint3(thr, 0));
                float horiz = QuadReadAcrossX(v);
                float vert = QuadReadAcrossY(v);
                float diag = QuadReadAcrossDiagonal(v);
                SOut[thr] = float4(v, horiz, vert, diag);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0xd4, 0xec, 0xde, 0x16, 0x49, 0xbc, 0x3b, 0xf6, 0x52, 0xeb, 0x23, 0x38, 0xc8, 0x66, 0xe3, 0xd7, 0x01, 0x00, 0x00, 0x00, 0x34, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xec, 0x8e, 0x42, 0x63, 0xf0, 0x66, 0x8d, 0x06, 0xa5, 0x05, 0xb9, 0xd9,
        0x22, 0xd1, 0xcb, 0xb4, 0x44, 0x58, 0x49, 0x4c, 0x28, 0x06, 0x00, 0x00, 0x60, 0x00, 0x05, 0x00, 0x8a, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x10, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x81, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x33, 0x00, 0x37, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21,
        0xf9, 0x2b, 0x21, 0xad, 0xc4, 0xe4, 0x17, 0xb7, 0x8d, 0x0a, 0x63, 0x8c, 0x19, 0x73, 0x04, 0x08, 0xa1, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0x41, 0x2a,
        0xca, 0x19, 0x6a, 0x8c, 0x31, 0xc6, 0x18, 0xb4, 0x0a, 0x61, 0x86, 0x41, 0xd4, 0xca, 0x02, 0x86, 0x1a, 0x63, 0x18, 0x63, 0x0c, 0xa2, 0x37, 0x47, 0x10, 0x14, 0x43, 0x0d, 0x34, 0x46, 0x24, 0x39,
        0x10, 0x30, 0x53, 0x18, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0x98, 0x03, 0x3b, 0xbc,
        0x43, 0x38, 0xd0, 0x83, 0x1f, 0xe8, 0x81, 0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0xc6, 0x4c, 0xd6, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3,
        0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xf8, 0x01, 0x12, 0x4c, 0x25, 0x3b, 0x8c,
        0x40, 0x18, 0x33, 0x91, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x81, 0x2c, 0xdc, 0x02, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81,
        0x3d, 0x94, 0xc3, 0x38, 0xd0, 0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x01, 0x0a, 0x30, 0xe5, 0x39, 0x02,
        0x50, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d,
        0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10,
        0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78,
        0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x16, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50, 0x12, 0x85, 0x50, 0x10, 0x14,
        0x47, 0x00, 0x68, 0x17, 0x20, 0x20, 0x10, 0xe9, 0x19, 0x00, 0xba, 0x33, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44,
        0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a,
        0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xc1, 0x98, 0x20, 0x0c, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x03, 0xb2, 0x41, 0x18, 0x0c, 0x0a,
        0x63, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0x17, 0x44, 0x60, 0x82, 0x30, 0x24, 0x13, 0x84, 0x41, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x10, 0x3c,
        0x13, 0x04, 0xed, 0x99, 0x20, 0x44, 0xcd, 0x86, 0x85, 0x88, 0x16, 0x82, 0x18, 0x18, 0x49, 0x92, 0x9c, 0x0d, 0xc1, 0xb4, 0x81, 0x80, 0x28, 0x00, 0x98, 0x20, 0x08, 0x00, 0x89, 0xb6, 0xb0, 0x34,
        0xb7, 0x09, 0xc2, 0xe6, 0x4c, 0x10, 0x86, 0x65, 0x82, 0x30, 0x30, 0x1b, 0x06, 0x4d, 0x1b, 0x36, 0x10, 0x04, 0x96, 0x6d, 0x1b, 0x0a, 0xeb, 0x02, 0x2a, 0xae, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b,
        0x1a, 0x59, 0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57, 0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d,
        0x09, 0x8c, 0x3a, 0x64, 0x78, 0x2e, 0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53, 0x02, 0xa4, 0x0c, 0x19, 0x9e, 0x8b, 0x5c, 0xd9, 0xdc, 0x5b, 0x9d, 0xdc, 0x58, 0xd9,
        0xdc, 0x94, 0x80, 0xaa, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0xe0, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
        0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10,
        0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03,
        0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e,
        0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b,
        0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90,
        0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e,
        0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca,
        0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82,
        0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06,
        0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10,
        0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x0d, 0x40, 0xc3, 0xe5, 0x3b, 0x8f, 0x0f, 0x51, 0xc0, 0x80, 0x3e, 0x7e, 0x71, 0xdb, 0x26, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x44, 0x5c, 0x13,
        0x15, 0x11, 0xa5, 0x03, 0x0c, 0x7e, 0x71, 0xdb, 0x46, 0x60, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x44, 0x5c, 0x13, 0x15, 0x11, 0xec, 0xe4, 0x44, 0x84, 0x5f, 0xdc, 0xb6, 0x05, 0x48, 0xc3, 0xe5, 0x3b,
        0x8f, 0x3f, 0x1d, 0x11, 0x01, 0x0c, 0xe2, 0xe0, 0x23, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0xca, 0x3e, 0xa0, 0xe4, 0x4a, 0x37, 0xa0, 0x08, 0x03, 0xca, 0x30, 0x80, 0x50, 0x09, 0x8c, 0x00, 0x14, 0x41, 0x79, 0x00, 0x00,
        0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x59, 0x08, 0x41, 0x51, 0xce, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x48, 0x57, 0x42, 0x54, 0xd5, 0x33, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x04, 0x97,
        0x58, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x74, 0x0a, 0x36, 0x62, 0xd0, 0x00, 0x20, 0x08, 0x06, 0x4b, 0xa7, 0x0c, 0x98, 0x10, 0x38, 0x8e, 0xe3, 0x8c, 0x26, 0x04, 0xc0, 0x88, 0xc1, 0x01,
        0x80, 0x20, 0x18, 0x34, 0xde, 0x13, 0x24, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0x7c, 0x90, 0xb0, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0x03, 0x06, 0xd1, 0xa0, 0x8c, 0x18, 0x38, 0x00,
        0x08, 0x82, 0xc1, 0x03, 0x06, 0x4e, 0x72, 0x18, 0x13, 0x31, 0x08, 0x81, 0x82, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_quad_swap_2d_unaligned_y_sm66_dxil[] =
    {
#if 0
        Texture2D<float> Tex : register(t0);
        RWTexture2D<float4> SOut : register(u0);

        [numthreads(8, 9, 1)]
        void main(uint2 thr : SV_DispatchThreadID)
        {
                float v = Tex.Load(uint3(thr, 0));
                float horiz = QuadReadAcrossX(v);
                float vert = QuadReadAcrossY(v);
                float diag = QuadReadAcrossDiagonal(v);
                SOut[thr] = float4(v, horiz, vert, diag);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x99, 0xa0, 0x6d, 0x75, 0xbc, 0xf6, 0xde, 0x6d, 0x1c, 0xcd, 0xcf, 0x13, 0x8e, 0x92, 0xd8, 0x13, 0x01, 0x00, 0x00, 0x00, 0xc0, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x4b, 0x5d, 0x99, 0x61, 0x9c, 0xf2, 0x3c, 0x62, 0x65, 0x97, 0x3a,
        0x41, 0xa8, 0x2d, 0x4c, 0x44, 0x58, 0x49, 0x4c, 0xb4, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xad, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x9c, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xa4, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x80, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x03, 0x70, 0xd3, 0x70, 0xf9, 0x13, 0xf6,
        0x10, 0x92, 0xbf, 0x12, 0xd2, 0x4a, 0x4c, 0x7e, 0x71, 0xdb, 0xa8, 0x30, 0x0c, 0xc3, 0x18, 0xe6, 0x08, 0x10, 0x42, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02,
        0x05, 0x49, 0x51, 0x8e, 0x41, 0x19, 0x86, 0x61, 0x18, 0x86, 0x81, 0x96, 0x42, 0x18, 0x83, 0x81, 0x50, 0x53, 0x16, 0x60, 0x50, 0x86, 0x61, 0x30, 0x0c, 0xc3, 0x40, 0xe8, 0xb9, 0x6d, 0xb8, 0xfc,
        0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0xc9, 0xa1, 0x22, 0x81, 0x48, 0x23, 0xe7, 0x21, 0xa2, 0x09, 0x21, 0x24, 0x24, 0x0c, 0x43, 0x21, 0x94, 0x41, 0x89, 0x48, 0x3a, 0x68, 0xb8, 0xfc, 0x09, 0x7b,
        0x08, 0xc9, 0x5f, 0x09, 0x69, 0x43, 0x9a, 0x01, 0x11, 0xc3, 0x30, 0x20, 0x73, 0x04, 0x41, 0x29, 0x94, 0x81, 0x1a, 0x2a, 0xb2, 0x06, 0x02, 0x66, 0x0a, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87,
        0x79, 0x70, 0x03, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07, 0x7a, 0xf0, 0x03, 0x3d, 0xd0, 0x83, 0x76, 0x48,
        0x07, 0x78, 0x98, 0x87, 0x5f, 0xa0, 0x87, 0x7c, 0x80, 0x87, 0x72, 0x40, 0xc1, 0x30, 0x93, 0x35, 0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f,
        0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40, 0x0f, 0x7e, 0x80, 0x04, 0x23, 0x23, 0x6d, 0x18, 0x81, 0x30, 0x66, 0x22, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98,
        0x87, 0x79, 0x70, 0x03, 0x59, 0xb8, 0x05, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x7b, 0x28, 0x87, 0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e,
        0x30, 0x07, 0x76, 0x78, 0x87, 0x70, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14, 0x70, 0xd4, 0xcd, 0x11, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0,
        0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0,
        0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x79, 0x16, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x2c, 0x10, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x8a, 0xa0, 0x24,
        0x0a, 0xa1, 0x20, 0xa8, 0x1a, 0x01, 0xa0, 0xaf, 0x00, 0x01, 0x81, 0xc8, 0x9b, 0x01, 0xa0, 0x6d, 0x06, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62,
        0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d,
        0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x61, 0x8b, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x04, 0x84, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02,
        0x67, 0x82, 0xe0, 0x41, 0x13, 0x84, 0xca, 0xd9, 0xb0, 0x20, 0x90, 0x82, 0x20, 0xc3, 0x12, 0x45, 0x51, 0xb3, 0x21, 0x90, 0x36, 0x10, 0xcf, 0x04, 0x00, 0x13, 0x04, 0x01, 0x20, 0xd1, 0x16, 0x96,
        0xe6, 0x36, 0x41, 0xf8, 0x9e, 0x09, 0xc2, 0xc0, 0x4c, 0x10, 0x86, 0x66, 0xc3, 0x90, 0x31, 0xc3, 0x06, 0x02, 0xb9, 0x30, 0x6d, 0x43, 0x51, 0x59, 0x00, 0xb5, 0x55, 0x61, 0x63, 0xb3, 0x6b, 0x73,
        0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9,
        0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b,
        0x9b, 0x9b, 0x12, 0x4c, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
        0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10,
        0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03,
        0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e,
        0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b,
        0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90,
        0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e,
        0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca,
        0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82,
        0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06,
        0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0,
        0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48,
        0x63, 0xd8, 0x00, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x10, 0x05, 0x0c, 0xe8, 0xe3, 0x17, 0xb7, 0x6d, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0xe0, 0x17,
        0xb7, 0x6d, 0x04, 0xd6, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0xc1, 0x4e, 0x4e, 0x44, 0xf8, 0xc5, 0x6d, 0x5b, 0x80, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0xd3, 0x11, 0x11, 0xc0, 0x20,
        0x0e, 0x3e, 0x72, 0xdb, 0x06, 0x40, 0x30, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x34, 0xcc, 0x00, 0x94, 0x7d, 0x40, 0xc9, 0x0e, 0x94, 0x6e, 0x40, 0xc1, 0x0e, 0x94, 0x24, 0x41, 0x11, 0x06, 0x14, 0x21, 0x50, 0x49, 0x42, 0x94, 0x61, 0x00, 0x21, 0x25, 0x30, 0x02, 0x50, 0x04,
        0xe5, 0x41, 0xd1, 0x1c, 0x82, 0x72, 0xcd, 0x21, 0x64, 0x1a, 0x51, 0x73, 0x10, 0x49, 0x92, 0x70, 0x23, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x79, 0x90, 0xb0, 0x59, 0x23,
        0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x7d, 0x91, 0xc0, 0x5d, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x94, 0x41, 0xd4, 0x8d, 0x18, 0x18, 0x00, 0x08, 0x82, 0x01, 0x61, 0x06, 0x12, 0x18, 0x8c,
        0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0x34, 0x06, 0xd2, 0x80, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0xc1, 0x72, 0x06, 0x51, 0x00, 0x06, 0x83, 0x70, 0x5d, 0xd7, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06,
        0x07, 0x00, 0x82, 0x60, 0xd0, 0xa0, 0x01, 0x16, 0x38, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0xa4, 0x41, 0x26, 0x40, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0xa8, 0x81, 0x36, 0x3c, 0x23,
        0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0xa5, 0x01, 0xa6, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xf0, 0xac, 0x01, 0x15, 0x24, 0x48, 0x57, 0x10, 0x83, 0x00, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const D3D12_SHADER_BYTECODE cs_quad_swap_2d_sm60 = SHADER_BYTECODE(cs_quad_swap_2d_sm60_dxil);
    static const D3D12_SHADER_BYTECODE cs_quad_swap_2d_sm66 = SHADER_BYTECODE(cs_quad_swap_2d_sm66_dxil);
    static const D3D12_SHADER_BYTECODE cs_quad_swap_2d_unaligned_y_sm66 = SHADER_BYTECODE(cs_quad_swap_2d_unaligned_y_sm66_dxil);

    struct test
    {
        const D3D12_SHADER_BYTECODE *cs;
        unsigned int dispatch_x;
        unsigned int dispatch_y;
        D3D_SHADER_MODEL minimum_shader_model;
        bool quad_swap_is_2d;
    };

    /* Quad swap ops are context sensitive based on shading model and work group size (wat).
     * Pre 6.6 it's always based on lane index.
     * 6.6+ it depends on the group size. If X and Y align with 2, we get 2D semantics based on GroupThreadID.xy,
     * otherwise, it's based on GroupThread.x.
     * Testing with unaligned X is meaningless, since not even 1D quads will work reliably in that case.
     */
    static const struct test tests[] =
    {
        { &cs_quad_swap_2d_sm60, 2, 2, D3D_SHADER_MODEL_6_0, false },
        { &cs_quad_swap_2d_sm66, 2, 2, D3D_SHADER_MODEL_6_6, true },
        { &cs_quad_swap_2d_unaligned_y_sm66, 2, 2, D3D_SHADER_MODEL_6_6, false },
    };

    if (!init_compute_test_context(&context))
        return;

    device = context.device;

    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model));

    memset(&root_signature_desc, 0, sizeof(root_signature_desc));
    memset(root_parameters, 0, sizeof(root_parameters));
    memset(descriptor_ranges, 0, sizeof(descriptor_ranges));
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    root_signature_desc.pParameters = root_parameters;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = ARRAY_SIZE(descriptor_ranges);
    root_parameters[0].DescriptorTable.pDescriptorRanges = descriptor_ranges;
    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 0;
    descriptor_ranges[0].NumDescriptors = 1;
    descriptor_ranges[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;
    descriptor_ranges[1].OffsetInDescriptorsFromTableStart = 1;
    descriptor_ranges[1].NumDescriptors = 1;

    create_root_signature(device, &root_signature_desc, &root_signature);
    heap = create_gpu_descriptor_heap(device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 2);

    input_texture = create_default_texture2d(device, 16, 16, 1, 1,
            DXGI_FORMAT_R32_FLOAT, D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);
    result_texture = create_default_texture2d(device, 16, 16, 1, 1,
            DXGI_FORMAT_R32G32B32A32_FLOAT, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    {
        D3D12_SUBRESOURCE_DATA data;
        float buffer_data[16 * 16];

        for (y = 0; y < 16; y++)
            for (x = 0; x < 16; x++)
                buffer_data[16 * y + x] = 1.0f + 16.0f * (float)y + (float)x;

        data.RowPitch = 16 * sizeof(float);
        data.SlicePitch = data.RowPitch * 16;
        data.pData = buffer_data;
        upload_texture_data(input_texture, &data, 1, context.queue, context.list);
        reset_command_list(context.list, context.allocator);
        transition_resource_state(context.list, input_texture,
                D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    }

    {
        D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
        memset(&srv_desc, 0, sizeof(srv_desc));
        srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        srv_desc.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE2D;
        srv_desc.Format = DXGI_FORMAT_R32_FLOAT;
        srv_desc.Texture2D.MipLevels = 1;
        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
        ID3D12Device_CreateShaderResourceView(device, input_texture, &srv_desc, cpu_handle);
    }

    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
        memset(&uav_desc, 0, sizeof(uav_desc));
        uav_desc.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;
        uav_desc.Format = DXGI_FORMAT_R32G32B32A32_FLOAT;

        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
        cpu_handle.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
        ID3D12Device_CreateUnorderedAccessView(device, result_texture, NULL, &uav_desc, cpu_handle);
    }

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);

        if (shader_model.HighestShaderModel < tests[i].minimum_shader_model)
        {
            skip("Skipping test due to insufficient shader model support.\n");
            continue;
        }

        pso = create_compute_pipeline_state(device, root_signature, *tests[i].cs);
        ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &heap);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0,
            ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap));
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
        ID3D12GraphicsCommandList_Dispatch(context.list, tests[i].dispatch_x, tests[i].dispatch_y, 1);
        transition_resource_state(context.list, result_texture,
                   D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

        get_texture_readback_with_command_list(result_texture, 0, &rb, context.queue, context.list);

        for (y = 0; y < 16; y++)
        {
            for (x = 0; x < 16; x++)
            {
                const struct vec4 *values;
                struct vec4 expected;

                values = get_readback_vec4(&rb, x, y);

                if (tests[i].quad_swap_is_2d)
                {
                    expected.x = 1.0f + (float)y * 16.0f + (float)x;
                    expected.y = 1.0f + (float)y * 16.0f + (float)(x ^ 1);
                    expected.z = 1.0f + (float)(y ^ 1) * 16.0f + (float)x;
                    expected.w = 1.0f + (float)(y ^ 1) * 16.0f + (float)(x ^ 1);
                }
                else
                {
                    expected.x = 1.0f + (float)y * 16.0f + (float)x;
                    expected.y = 1.0f + (float)y * 16.0f + (float)(x ^ 1);
                    expected.z = 1.0f + (float)y * 16.0f + (float)(x ^ 2);
                    expected.w = 1.0f + (float)y * 16.0f + (float)(x ^ 3);
                }

                ok(compare_vec4(values, &expected, 0),
                        "Mismatch at %u, %u: Expected (%f, %f, %f, %f), got (%f %f %f %f).\n",
                        x, y, expected.x, expected.y, expected.z, expected.w,
                        values->x, values->y, values->z, values->w);
            }
        }
        reset_command_list(context.list, context.allocator);
        transition_resource_state(context.list, result_texture,
                D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12PipelineState_Release(pso);
        release_resource_readback(&rb);
    }
    vkd3d_test_set_context(NULL);

    ID3D12Resource_Release(input_texture);
    ID3D12Resource_Release(result_texture);
    ID3D12DescriptorHeap_Release(heap);
    ID3D12RootSignature_Release(root_signature);
    destroy_test_context(&context);
}

void test_sv_barycentric(void)
{
    D3D12_FEATURE_DATA_D3D12_OPTIONS3 features3;
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    float base_x, base_y, rot, off_x, off_y;
    D3D12_INPUT_LAYOUT_DESC input_layout;
    struct test_context_desc desc;
    D3D12_VERTEX_BUFFER_VIEW vbv;
    unsigned int i, x, y, offset;
    struct test_context context;
    D3D12_INDEX_BUFFER_VIEW ibv;
    struct resource_readback rb;
    unsigned int coverage_count;
    float vbo_data[16 * 4 * 4];
    ID3D12PipelineState *pso;
    UINT ibo_data[5 * 4 * 4];
    ID3D12Resource *vbo;
    ID3D12Resource *ibo;
    uint32_t value;
    HRESULT hr;

    static const BYTE vs_code[] =
    {
#if 0
        struct VOut
        {
                nointerpolation float attr0 : ATTR0;
                float attr1 : ATTR1;
                noperspective float attr2 : ATTR2;
                float4 pos : SV_Position;
        };
        VOut main(float3 pos : POSITION, float attr : ATTR)
        {
                VOut vout;
                vout.pos = float4(pos.xy * pos.z, 0.0, pos.z);
                vout.attr0 = attr;
                vout.attr1 = attr;
                vout.attr2 = attr;
                return vout;
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x60, 0xf5, 0xbc, 0x92, 0x6a, 0x42, 0x27, 0xd2, 0x68, 0x31, 0x04, 0x6c, 0xf8, 0x86, 0x12, 0xd9, 0x01, 0x00, 0x00, 0x00, 0x75, 0x07, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xa2, 0x00, 0x00, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x39, 0x02, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x56, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x41, 0x54, 0x54,
        0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0xa3, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x92, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x97, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54, 0x54, 0x52,
        0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x50, 0x53, 0x56, 0x30, 0xe4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x02, 0x04, 0x00, 0x02, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54,
        0x54, 0x52, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x41, 0x00, 0x03, 0x01, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x00, 0x03, 0x02, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
        0x00, 0x01, 0x02, 0x41, 0x00, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00,
        0x00, 0x00, 0xb0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x34, 0x05, 0x00,
        0x00, 0x61, 0x00, 0x01, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x01, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x1c, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
        0x00, 0x44, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84,
        0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90, 0x14,
        0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18, 0x00,
        0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00,
        0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22, 0xa4,
        0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x30, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02, 0x30, 0x98, 0x23, 0x40,
        0x8a, 0x31, 0x44, 0x54, 0x44, 0x56, 0x0c, 0x20, 0xa2, 0x1a, 0xc2, 0x81, 0x80, 0x3c, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72,
        0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
        0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc8, 0x02, 0x01, 0x0d, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xa2, 0x12, 0x28, 0x84, 0x62, 0x18, 0x01, 0x28,
        0x82, 0x32, 0xa0, 0x2a, 0x89, 0x11, 0x80, 0x32, 0x28, 0x81, 0x22, 0x28, 0x04, 0xda, 0xb1, 0x8e, 0x02, 0x02, 0x41, 0x02, 0x04, 0x11, 0x10, 0x64, 0x03, 0x20, 0x46, 0x00, 0x00, 0x79, 0x18, 0x00,
        0x00, 0x58, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43,
        0x10, 0x4c, 0x10, 0x88, 0x61, 0x83, 0x30, 0x0c, 0x13, 0x04, 0x82, 0xd8, 0x20, 0x0c, 0x05, 0x05, 0xbb, 0xb9, 0x09, 0x02, 0x51, 0x6c, 0x18, 0x0e, 0x64, 0x98, 0x20, 0x08, 0xc0, 0x04, 0x81, 0x30,
        0x36, 0x00, 0x1b, 0x86, 0x85, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x50, 0x9c, 0x09, 0xc2, 0xf2, 0x6c, 0x08, 0x20, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x44, 0xa0, 0x9e, 0xa6, 0x92, 0xa8, 0x92, 0x9e,
        0x9c, 0x26, 0x08, 0x45, 0x32, 0x41, 0x28, 0x94, 0x0d, 0xc1, 0x32, 0x41, 0x28, 0x96, 0x0d, 0xcb, 0x32, 0x51, 0x95, 0x55, 0x0d, 0xd7, 0x52, 0x01, 0x24, 0x82, 0xa8, 0xa8, 0xa4, 0x26, 0x08, 0x05,
        0xb3, 0x61, 0x19, 0x32, 0xaa, 0xb2, 0xaa, 0x41, 0x1b, 0x2a, 0x60, 0x83, 0x80, 0x6d, 0x1b, 0x96, 0x25, 0xa3, 0x2a, 0x4b, 0x1b, 0xb4, 0xa5, 0x02, 0x36, 0x04, 0xc3, 0x04, 0xa1, 0x68, 0x36, 0x2c,
        0x43, 0x46, 0x55, 0xde, 0x37, 0x68, 0x43, 0x05, 0x4c, 0x10, 0x88, 0x63, 0x43, 0x10, 0x06, 0x13, 0x84, 0xc2, 0xd9, 0xb0, 0x84, 0x41, 0x46, 0x55, 0x62, 0x30, 0x06, 0x83, 0x16, 0x06, 0x15, 0x30,
        0x41, 0x20, 0x10, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x1b, 0x96, 0x32, 0x30, 0x03, 0xea, 0xb2, 0xc6, 0x60, 0x18, 0x83, 0x32, 0xa8, 0x80, 0x0d, 0x44, 0x07,
        0x06, 0x64, 0x70, 0x06, 0x1b, 0x06, 0x0e, 0x0d, 0x80, 0x0d, 0x85, 0x22, 0xa5, 0x01, 0x00, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0,
        0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x43, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x41, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac,
        0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x54, 0x22, 0xc3, 0x73, 0xa1, 0xcb, 0x83, 0x2b, 0x0b, 0x72, 0x73, 0x7b, 0xa3, 0x0b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x9b, 0x12, 0x38, 0x75,
        0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x50, 0x1d, 0x32, 0x3c, 0x97, 0x32, 0x37, 0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba, 0xb9,
        0x29, 0x41, 0x1a, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80,
        0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
        0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c,
        0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
        0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
        0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c,
        0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
        0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d,
        0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d,
        0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x44, 0x33, 0x00,
        0xa5, 0x40, 0x33, 0x02, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x48, 0x0f, 0xd4, 0x24, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0x13, 0xe4, 0x38, 0x0a, 0x31, 0x62, 0x90, 0x00,
        0x20, 0x08, 0x06, 0x06, 0x15, 0x3d, 0x4f, 0x52, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x51, 0x49, 0x10, 0x94, 0x18, 0x16, 0x0c, 0xf2, 0x31, 0x61, 0x90, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20,
        0x18, 0x20, 0x17, 0x12, 0x4d, 0x8f, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x85, 0x44, 0x53, 0x13, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x72, 0x21, 0xd1, 0xc4, 0x1c, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0x80, 0x5c, 0x48, 0x34, 0x39, 0xc3, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x17, 0x32, 0x4d, 0x8f, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x85, 0x58, 0xd3,
        0x63, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x72, 0x21, 0xd2, 0xf4, 0x18, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE ps_code[] =
    {
#if 0
        struct VOut
        {
                nointerpolation float attr0 : ATTR0;
                float attr1 : ATTR1;
                noperspective float attr2 : ATTR2;
                float4 pos : SV_Position;
        };

        float4 main(VOut vout,
                float3 bary_perspective : SV_Barycentrics,
                noperspective float3 bary_noperspective : SV_Barycentrics1) : SV_Target
        {
                float reference_perspective = vout.attr1;
                float reference_noperspective = vout.attr2;

                float result_perspective =
                        GetAttributeAtVertex(vout.attr0, 0) * bary_perspective.x +
                        GetAttributeAtVertex(vout.attr0, 1) * bary_perspective.y +
                        GetAttributeAtVertex(vout.attr0, 2) * bary_perspective.z;

                float result_noperspective =
                        GetAttributeAtVertex(vout.attr0, 0) * bary_noperspective.x +
                        GetAttributeAtVertex(vout.attr0, 1) * bary_noperspective.y +
                        GetAttributeAtVertex(vout.attr0, 2) * bary_noperspective.z;

                float4 res;
                // Results should be approximately equal.
                res.x = abs(reference_perspective - result_perspective) * 20.0;
                res.y = abs(reference_noperspective - result_noperspective) * 20.0;

                // Test that we can also read the provoking vertex.
                res.z = vout.attr0 / 255.0;

                // Validate barycentrics approximately sum up as expected. Result should be 0x80 when rounded.
                res.w = (64.0 / 255.0) * (dot(bary_perspective, 1.0.xxx) + dot(bary_noperspective, 1.0.xxx));

                return res;
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0xe7, 0x9e, 0xc5, 0x10, 0x73, 0x9c, 0x3c, 0x3b, 0xf5, 0xba, 0xc7, 0x99, 0xe7, 0x9b, 0x0e, 0x3c, 0x01, 0x00, 0x00, 0x00, 0x45, 0x09, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0x79, 0x01, 0x00, 0x00, 0x85, 0x02, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0xf3, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xcd, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x17, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe3, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x17, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54,
        0x54, 0x52, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x53, 0x56, 0x5f, 0x42, 0x61, 0x72, 0x79, 0x63, 0x65, 0x6e, 0x74, 0x72, 0x69, 0x63, 0x73, 0x00, 0x4f,
        0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0x04, 0x01, 0x00,
        0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00,
        0x00, 0x06, 0x01, 0x00, 0x04, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54, 0x54, 0x52, 0x00, 0x41, 0x54, 0x54, 0x52,
        0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x41,
        0x00, 0x03, 0x01, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x00, 0x03, 0x02, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x02, 0x41,
        0x00, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03,
        0x1c, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x1c, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44,
        0x10, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xb8, 0x06, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0xae, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x01, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
        0x00, 0xa0, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xa5, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23,
        0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
        0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
        0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff,
        0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x14, 0x00, 0x00,
        0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x48, 0x23,
        0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0x62, 0x0c, 0x42, 0x48, 0x29, 0xa4, 0xca,
        0x30, 0xc8, 0x20, 0x56, 0x90, 0x41, 0xc6, 0x18, 0x63, 0x0c, 0x72, 0x03, 0x01, 0x29, 0x41, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72,
        0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
        0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x18, 0xf2, 0x28, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x61, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0xe3, 0x00,
        0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43,
        0x22, 0x25, 0x50, 0x08, 0xc5, 0x30, 0x02, 0x50, 0x04, 0x65, 0x50, 0x03, 0xa5, 0x40, 0xa5, 0x24, 0x46, 0x00, 0x4a, 0xa0, 0x08, 0x0a, 0xa1, 0x0c, 0x0a, 0xa7, 0x06, 0x0a, 0x84, 0xe2, 0x58, 0x12,
        0x44, 0x10, 0x00, 0x00, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
        0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x08, 0x63, 0x83, 0x30, 0x0c, 0x13, 0x04, 0xe2, 0xd8,
        0x20, 0x0c, 0x05, 0x05, 0xb8, 0xb9, 0x09, 0x02, 0x81, 0x6c, 0x18, 0x0e, 0x64, 0x98, 0x20, 0x08, 0xc0, 0x04, 0x81, 0x48, 0x36, 0x00, 0x1b, 0x86, 0x85, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x50,
        0x9c, 0x09, 0x42, 0x74, 0x6d, 0x08, 0x20, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x24, 0x82, 0xa8, 0xa8, 0xa4, 0x26, 0x08, 0x85, 0x33, 0x41, 0x28, 0x9e, 0x0d, 0xc1, 0x32, 0x41, 0x28, 0xa0, 0x0d, 0xcb,
        0x32, 0x51, 0x95, 0x75, 0x0d, 0xd7, 0x52, 0x01, 0x1b, 0x82, 0x61, 0x82, 0x50, 0x44, 0x1b, 0x96, 0x61, 0xa2, 0xaa, 0x4c, 0x1b, 0xae, 0xa1, 0x02, 0x26, 0x08, 0x84, 0xb2, 0x21, 0xe0, 0x26, 0x08,
        0x85, 0xb4, 0x61, 0xe1, 0x26, 0xaa, 0xea, 0xbc, 0xe1, 0xe2, 0x2a, 0x60, 0x82, 0x40, 0x2c, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0xc5, 0xb4, 0x61,
        0x01, 0x83, 0x30, 0xa0, 0xc4, 0xc0, 0xf2, 0x06, 0x0f, 0x0c, 0x2a, 0x80, 0xcf, 0x94, 0xd5, 0x97, 0x50, 0x98, 0x5c, 0xde, 0x58, 0x99, 0x1b, 0x9d, 0x5c, 0xda, 0xd8, 0xdc, 0x04, 0xa1, 0xa0, 0x26,
        0x08, 0x04, 0x33, 0x41, 0x28, 0xaa, 0x0d, 0x4b, 0x41, 0x06, 0x54, 0x19, 0x58, 0xda, 0x20, 0x06, 0x66, 0x70, 0x06, 0xc0, 0x04, 0x81, 0x68, 0x36, 0x2c, 0x69, 0x40, 0x06, 0x54, 0x19, 0x64, 0xde,
        0x20, 0x06, 0x66, 0x70, 0x06, 0xc0, 0x06, 0x03, 0xdb, 0xbe, 0x31, 0x40, 0x03, 0x35, 0x60, 0x32, 0x65, 0xf5, 0x45, 0x15, 0x26, 0x77, 0x56, 0x46, 0x37, 0x41, 0x28, 0xac, 0x0d, 0xcb, 0xc2, 0x06,
        0x54, 0x1b, 0x58, 0xd5, 0xe0, 0x2d, 0x15, 0xb0, 0x21, 0x70, 0x83, 0x0d, 0xc3, 0x1a, 0xbc, 0x01, 0xb0, 0xa1, 0x50, 0x24, 0x38, 0x00, 0x80, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65,
        0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x60, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x28, 0xea,
        0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x90, 0x4a, 0x64, 0x78, 0x2e, 0x74, 0x79, 0x70, 0x65, 0x41, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x74,
        0x69, 0x6f, 0x6e, 0x73, 0x53, 0x02, 0xa7, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12, 0xd9, 0x14, 0x5d, 0x18, 0x5d, 0xd9, 0x94, 0x00, 0xaa, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27,
        0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0x80, 0x03, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
        0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d,
        0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
        0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
        0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03,
        0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
        0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
        0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43,
        0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20, 0x00,
        0x00, 0x17, 0x00, 0x00, 0x00, 0x36, 0xb0, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0x30, 0x4d, 0x11, 0x12, 0x50, 0x13, 0xa1, 0x4d, 0x2f, 0x11, 0x4d, 0xc4, 0xe5, 0x17, 0xb7, 0x6d, 0x05, 0xce, 0x70, 0xf9,
        0xce, 0xe3, 0x0f, 0xce, 0x74, 0xfb, 0xc5, 0x6d, 0x5b, 0xc0, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62, 0xf3, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f,
        0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x11, 0x3c, 0xc3, 0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x40, 0x84, 0xf9, 0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x00,
        0x00, 0x61, 0x20, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x44, 0xca, 0xad, 0x24, 0x05, 0x66, 0x00, 0x68, 0x8c, 0x11, 0x84,
        0x24, 0x18, 0xa2, 0xdf, 0x18, 0x41, 0x48, 0x82, 0x21, 0xde, 0x8d, 0x11, 0x80, 0x20, 0x08, 0xea, 0x60, 0x30, 0x46, 0x00, 0x82, 0x20, 0x88, 0x7f, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60,
        0x78, 0x98, 0x64, 0x41, 0xc5, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0x5f, 0x36, 0x5d, 0x90, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x06, 0x18, 0x68, 0x14, 0x06, 0x1d, 0x23, 0x06, 0x09,
        0x00, 0x82, 0x60, 0x60, 0x84, 0xc1, 0xb6, 0x65, 0x13, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x18, 0x70, 0x9c, 0x36, 0x25, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x8c, 0x41, 0xd7,
        0x6d, 0x93, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x06, 0x19, 0x78, 0x1b, 0x67, 0x2d, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x94, 0xc1, 0x07, 0x06, 0xdd, 0xc5, 0x8c, 0x18, 0x24, 0x00,
        0x08, 0x82, 0x81, 0x61, 0x06, 0x60, 0xe0, 0x79, 0x58, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x4a, 0x19, 0x3c, 0xdf, 0x97, 0x65, 0x16, 0x1c, 0xf2, 0x19, 0x31, 0x48, 0x00, 0x10, 0x04, 0x03,
        0xe5, 0x0c, 0xa2, 0x30, 0x08, 0x83, 0x4d, 0xb3, 0x00, 0x91, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x28, 0x69, 0x30, 0x8d, 0xc1, 0x18, 0x74, 0x9b, 0x05, 0x89, 0x7c, 0xcc, 0x78, 0xe4, 0x63,
        0xc5, 0x23, 0x1f, 0x23, 0x1e, 0xf9, 0xd8, 0x82, 0xc4, 0xc7, 0x82, 0x23, 0x3e, 0x16, 0x18, 0xf1, 0x19, 0x31, 0x30, 0x00, 0x10, 0x04, 0x03, 0x06, 0x0e, 0xd8, 0x20, 0xb0, 0x00, 0x93, 0x8f, 0x45,
        0x48, 0x7c, 0x2c, 0x40, 0xe2, 0x63, 0x01, 0x12, 0x9f, 0x11, 0x03, 0x03, 0x00, 0x41, 0x30, 0x60, 0xea, 0x20, 0x0e, 0x02, 0x0b, 0x3a, 0xf9, 0x18, 0xf5, 0xc9, 0x67, 0xc4, 0x60, 0x01, 0x40, 0x10,
        0x0c, 0x9c, 0x3b, 0x20, 0x03, 0x2d, 0xc3, 0x3c, 0xcf, 0x1b, 0x31, 0x58, 0x00, 0x10, 0x04, 0x03, 0x07, 0x0f, 0xca, 0xc0, 0xeb, 0xb8, 0xef, 0xfb, 0x2c, 0x10, 0xe0, 0x63, 0x01, 0x19, 0xc8, 0x67,
        0xc4, 0x20, 0x01, 0x40, 0x10, 0x0c, 0x10, 0x3f, 0x98, 0x83, 0x3b, 0xb8, 0x83, 0x38, 0x58, 0x46, 0x0c, 0x12, 0x00, 0x04, 0xc1, 0x00, 0xf1, 0x83, 0x39, 0xb8, 0x83, 0x3b, 0x80, 0x03, 0x63, 0xc4,
        0x20, 0x01, 0x40, 0x10, 0x0c, 0x10, 0x3f, 0x98, 0x83, 0x3b, 0xb8, 0x83, 0x37, 0x28, 0x46, 0x0c, 0x12, 0x00, 0x04, 0xc1, 0x00, 0xf1, 0x83, 0x39, 0xb8, 0x83, 0x3b, 0x68, 0x83, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const struct D3D12_SHADER_BYTECODE vs = { vs_code, sizeof(vs_code) };
    static const struct D3D12_SHADER_BYTECODE ps = { ps_code, sizeof(ps_code) };
    static const D3D12_INPUT_ELEMENT_DESC input_elems[2] = {
        { "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 0, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0 },
        { "ATTR", 0, DXGI_FORMAT_R32_FLOAT, 0, 12, D3D12_INPUT_CLASSIFICATION_PER_VERTEX_DATA, 0 },
    };

#define BARY_RES 128

    static const D3D12_VIEWPORT vp = { 0, 0, BARY_RES, BARY_RES, 0, 1 };
    static const float white[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    static const D3D12_RECT sci = { 0, 0, BARY_RES, BARY_RES };
    static const uint8_t provoking_lut[] = {
        192, 212, 224, 244,
        128, 144, 160, 176,
        68, 80, 100, 112,
        4, 20, 36, 52,
    };
    static const float quad_lut[4][2] = {
        { -0.3f, -0.3f },
        { +0.3f, -0.3f },
        { -0.3f, +0.3f },
        { +0.3f, +0.3f },
    };

    for (i = 0; i < 16; i++)
    {
        for (x = 0; x < 4; x++)
            ibo_data[5 * i + x] = 4 * i + x;
        ibo_data[5 * i + 4] = ~0u;
    }

    for (y = 0; y < 4; y++)
    {
        for (x = 0; x < 4; x++)
        {
            offset = 16 * (4 * y + x);
            base_x = -0.75f + 0.5f * (float)x;
            base_y = -0.75f + 0.5f * (float)y;

            /* Offset by subpixels so we get consistent raster. */
            base_x += 2.0f / (64.0f * BARY_RES);
            base_y += 2.0f / (64.0f * BARY_RES);

            /* Test different rotations. */
            rot = (0.5f + ((float)(y * 4 + x) / 16.0f)) * 2.0f * M_PI;

            for (i = 0; i < 4; i++)
            {
                /* Test different winding orders. */
                off_x = quad_lut[i ^ (x & 1)][0];
                off_y = quad_lut[i ^ (x & 1)][1];

                vbo_data[offset + 4 * i + 0] = base_x + off_x * cosf(rot) - off_y * sinf(rot);
                vbo_data[offset + 4 * i + 1] = base_y + off_x * sinf(rot) + off_y * cosf(rot);

                vbo_data[offset + 4 * i + 2] = (float)(offset + 4 * i + 0.5f); /* W. Make sure different results are observed for perspective and noperspective variants. */
                vbo_data[offset + 4 * i + 3] = (float)(offset + 4 * i);
            }
        }
    }

    input_layout.NumElements = ARRAY_SIZE(input_elems);
    input_layout.pInputElementDescs = input_elems;

    memset(&desc, 0, sizeof(desc));
    desc.no_root_signature = true;
    desc.no_pipeline = true;
    desc.rt_width = BARY_RES;
    desc.rt_height = BARY_RES;
    desc.rt_format = DXGI_FORMAT_R8G8B8A8_UNORM;
    if (!init_test_context(&context, &desc))
        return;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&features3, 0, sizeof(features3));
    if (FAILED(hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS3, &features3, sizeof(features3))) || !features3.BarycentricsSupported)
    {
        skip("Context does not support barycentrics, hr #%x.\n", hr);
        destroy_test_context(&context);
        return;
    }

    vbo = create_upload_buffer(context.device, sizeof(vbo_data), vbo_data);
    ibo = create_upload_buffer(context.device, sizeof(ibo_data), ibo_data);

    context.root_signature = create_empty_root_signature(context.device, D3D12_ROOT_SIGNATURE_FLAG_ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT);

    init_pipeline_state_desc_dxil(&pso_desc, context.root_signature, DXGI_FORMAT_R8G8B8A8_UNORM, &vs, &ps, &input_layout);
    pso_desc.IBStripCutValue = D3D12_INDEX_BUFFER_STRIP_CUT_VALUE_0xFFFFFFFF;
    pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_TRIANGLE;
    pso_desc.RasterizerState.CullMode = D3D12_CULL_MODE_NONE;

    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc, &IID_ID3D12PipelineState, (void **)&pso);
    ok(SUCCEEDED(hr), "Failed to create pipeline, hr #%u.\n", hr);

    ID3D12GraphicsCommandList_OMSetRenderTargets(context.list, 1, &context.rtv, TRUE, NULL);
    ibv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(ibo);
    ibv.Format = DXGI_FORMAT_R32_UINT;
    ibv.SizeInBytes = sizeof(ibo_data);
    vbv.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(vbo);
    vbv.StrideInBytes = 4 * sizeof(float);
    vbv.SizeInBytes = sizeof(vbo_data);
    ID3D12GraphicsCommandList_IASetIndexBuffer(context.list, &ibv);
    ID3D12GraphicsCommandList_IASetVertexBuffers(context.list, 0, 1, &vbv);
    ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(context.list, context.root_signature);
    ID3D12GraphicsCommandList_RSSetViewports(context.list, 1, &vp);
    ID3D12GraphicsCommandList_RSSetScissorRects(context.list, 1, &sci);
    ID3D12GraphicsCommandList_ClearRenderTargetView(context.list, context.rtv, white, 0, NULL);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(context.list, D3D_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    ID3D12GraphicsCommandList_DrawIndexedInstanced(context.list, ARRAY_SIZE(ibo_data), 1, 0, 0, 0);

    transition_resource_state(context.list, context.render_target, D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_texture_readback_with_command_list(context.render_target, 0, &rb, context.queue, context.list);
    coverage_count = 0;
    for (y = 0; y < BARY_RES; y++)
    {
        for (x = 0; x < BARY_RES; x++)
        {
            value = get_readback_uint(&rb, x, y, 0);
            if ((value >> 24) != 0xff) /* If we have coverage.*/
            {
                ok((value & 0xffff) == 0, "Value for pixel %u, %u is 0x%x.\n", x, y, value);
                ok((value >> 24) == 0x80, "Barycentrics don't sum to 1, alpha bits = 0x%x.\n", value >> 24);
                coverage_count++;
            }
            else
                ok(value == ~0u, "Value for pixel %u, %u is 0x%x.\n", x, y, value);
        }
    }
    /* Make sure we have enough test coverage. */
    ok(coverage_count >= (BARY_RES * BARY_RES) / 4, "Coverage is too low = %u.\n", coverage_count);

    for (y = 0; y < 4; y++)
    {
        for (x = 0; x < 4; x++)
        {
            /* Sample at quad centers. Based on rotation we should sample the provoking vertex for either first or second strip tri. */
            value = get_readback_uint(&rb, (BARY_RES / 4) * x + (BARY_RES / 8), (BARY_RES / 4) * y + (BARY_RES / 8), 0);
            ok(provoking_lut[y * 4 + x] == ((value >> 16) & 0xff), "Quad %u, %u, expected %u, got %u.\n", x, y, provoking_lut[y * 4 + x], (value >> 16) & 0xff);
        }
    }

    release_resource_readback(&rb);
    ID3D12PipelineState_Release(pso);
    ID3D12Resource_Release(vbo);
    ID3D12Resource_Release(ibo);
    destroy_test_context(&context);
}

void test_shader_fp16(void)
{
    D3D12_FEATURE_DATA_D3D12_OPTIONS4 features4;
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    struct test_context_desc desc;
    struct test_context context;
    struct resource_readback rb;
    ID3D12PipelineState *pso[2];
    ID3D12Resource *src;
    ID3D12Resource *dst;
    unsigned int i, j;
    uint32_t value;
    HRESULT hr;

#if 0
    struct VOut
    {
        float4 pos : SV_Position;
        min16float2 v : V;
    };

    VOut main(uint vert : SV_VertexID)
    {
        VOut vout;
        if (vert == 0) vout.pos = float4(-1, -1, 0, 1);
        else if (vert == 1) vout.pos = float4(-1, 3, 0, 1);
        else vout.pos = float4(3, -1, 0, 1);
        vout.v = min16float2(vout.pos.xy * 0.5 + 0.5);
        return vout;
    }
#endif
    /* dxc -Tvs_6_2 -Fo vert.dxil test.vert -Qstrip_reflect -Qstrip_debug */
    static const BYTE vs_code_non_native[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xa9, 0x5e, 0xf0, 0xeb, 0x7e, 0x5f, 0x33, 0x73, 0xd4, 0x6b, 0x80, 0x4e, 0xc6, 0x4d, 0x2c, 0x6b, 0x01, 0x00, 0x00, 0x00, 0xc6, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xde, 0x00, 0x00, 0x00, 0x66, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x56, 0x65, 0x72, 0x74, 0x65, 0x78, 0x49, 0x44, 0x00,
        0x4f, 0x53, 0x47, 0x31, 0x56, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0xfc, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x56, 0x00, 0x50, 0x53,
        0x56, 0x30, 0x80, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x01, 0x02, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x41, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x06, 0x02, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x58, 0x05, 0x00, 0x00, 0x62, 0x00, 0x01, 0x00, 0x56, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x02, 0x01, 0x00, 0x00, 0x10, 0x00,
        0x00, 0x00, 0x40, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81,
        0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10,
        0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51,
        0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0,
        0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x13, 0x00,
        0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x44,
        0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x18, 0x63, 0x90, 0x41, 0x65, 0x06, 0xa0, 0x18, 0x60, 0x8c, 0x41, 0x0e, 0xa1, 0x51, 0x80, 0x62, 0x80, 0x31, 0x06,
        0x51, 0xb4, 0x06, 0x02, 0x72, 0x60, 0xcc, 0x11, 0x80, 0xc2, 0x1c, 0x41, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d,
        0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76,
        0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07,
        0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x05, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
        0xf2, 0x2c, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x05, 0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09,
        0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x04, 0x85, 0x50, 0x0c, 0x23, 0x00, 0x44, 0x4a, 0xa1, 0x04, 0x46, 0x00, 0x4a, 0xa2, 0x0c, 0x0a, 0xa1, 0x20, 0x8a, 0x80, 0x5e, 0x01, 0x91, 0x1b,
        0x6b, 0x08, 0x8c, 0x39, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26,
        0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x62, 0x82, 0x30, 0x14, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc6, 0x06, 0x61, 0x30, 0x28, 0xd8, 0xcd, 0x4d, 0x10,
        0x86, 0x63, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x26, 0x08, 0x03, 0xb2, 0x01, 0xd8, 0x30, 0x30, 0x4d, 0xb3, 0x21, 0x70, 0x36, 0x0c, 0xc3, 0xf2, 0x4c, 0x10, 0x1c, 0x69, 0x43, 0x10, 0x91,
        0x68, 0x0b, 0x4b, 0x73, 0xe3, 0x32, 0x65, 0xf5, 0x65, 0x55, 0x26, 0x47, 0x57, 0x86, 0x97, 0x44, 0x34, 0x41, 0x20, 0x92, 0x09, 0x02, 0xa1, 0x6c, 0x08, 0x98, 0x09, 0x02, 0xb1, 0x6c, 0x58, 0x18,
        0xaa, 0xb2, 0x2e, 0x6c, 0xb0, 0x18, 0x0c, 0xd8, 0x10, 0x64, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x04, 0x33, 0x41, 0x20, 0x9a, 0x09, 0x02, 0xe1,
        0x6c, 0x58, 0x98, 0x8d, 0xeb, 0x2e, 0x6f, 0xf0, 0x18, 0x0c, 0x60, 0x60, 0x35, 0x41, 0x20, 0x9e, 0x09, 0x02, 0x01, 0x6d, 0x58, 0x06, 0x30, 0x08, 0x03, 0xec, 0x12, 0x83, 0x41, 0x0c, 0x06, 0x0c,
        0xd8, 0x20, 0x7c, 0x63, 0xb0, 0x61, 0xd0, 0xc8, 0x00, 0x98, 0x20, 0x3c, 0xd1, 0x06, 0x81, 0x31, 0x83, 0x0d, 0xc5, 0x32, 0x95, 0x01, 0x70, 0x06, 0x55, 0xd8, 0xd8, 0xec, 0xda, 0x5c, 0xd2, 0xc8,
        0xca, 0xdc, 0xe8, 0xa6, 0x04, 0x41, 0x15, 0x32, 0x3c, 0x17, 0xbb, 0x32, 0xb9, 0xb9, 0xb4, 0x37, 0xb7, 0x29, 0x01, 0xd1, 0x84, 0x0c, 0xcf, 0xc5, 0x2e, 0x8c, 0xcd, 0xae, 0x4c, 0x6e, 0x4a, 0x60,
        0xd4, 0x21, 0xc3, 0x73, 0x99, 0x43, 0x0b, 0x23, 0x2b, 0x93, 0x6b, 0x7a, 0x23, 0x2b, 0x63, 0x9b, 0x12, 0x24, 0x95, 0xc8, 0xf0, 0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2,
        0xe8, 0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x04, 0x4f, 0x1d, 0x32, 0x3c, 0x17, 0xbb, 0xb4, 0xb2, 0xbb, 0x24, 0xb2, 0x29, 0xba, 0x30, 0xba, 0xb2, 0x29, 0x41, 0x54, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d,
        0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x70, 0x06, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01,
        0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1,
        0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07,
        0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0,
        0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84,
        0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05,
        0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62,
        0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20,
        0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x23, 0xb7, 0x6d, 0x03, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13,
        0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb5, 0x6e, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d,
        0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x34, 0x4a, 0x61, 0x06, 0x80, 0xce, 0x18, 0x01, 0x08, 0x82,
        0x20, 0xfc, 0x8d, 0x11, 0x80, 0x20, 0x08, 0xe2, 0xbf, 0x30, 0x46, 0x00, 0x82, 0x20, 0x48, 0x82, 0xc1, 0x08, 0xc0, 0x18, 0x01, 0x08, 0x82, 0x20, 0xfe, 0x01, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0x50, 0x64, 0x94, 0x24, 0x3d, 0xc6, 0x70, 0x43, 0x30, 0x81, 0xc1, 0x70, 0x83, 0x80, 0x81, 0xc1, 0x74, 0x83, 0x71, 0x04, 0xc3, 0x0d, 0x44, 0x46, 0x06, 0xd3, 0x0d, 0x09, 0x12, 0x4c, 0x37, 0x28,
        0x43, 0x61, 0x02, 0x23, 0x1f, 0x13, 0x1a, 0xf9, 0x98, 0xe0, 0xc0, 0xc7, 0x84, 0x07, 0x3e, 0x27, 0xa8, 0x3b, 0x41, 0xdd, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x2c, 0x64, 0x40, 0x8d, 0xc1, 0xc7,
        0x09, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xb0, 0x90, 0x01, 0x35, 0x06, 0x5f, 0x17, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x52, 0x06, 0xd4, 0xf7, 0x71, 0xc8, 0x88, 0x41, 0x02, 0x80, 0x20,
        0x18, 0x20, 0x65, 0x40, 0x7d, 0x5f, 0x77, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x52, 0x06, 0xd4, 0xf7, 0x5d, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x65, 0x40, 0x7d, 0x9f, 0xe6,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    /* dxc -Tvs_6_2 -Fo vert.dxil test.vert -Qstrip_reflect -Qstrip_debug -enable-16bit-types
     * Expected warnings:
     * test.vert:4:2: warning: min16float is promoted to float16_t
     * test.vert:4:2: warning: min16float is promoted to float16_t
     * test.vert:13:11: warning: min16float is promoted to float16_t */
    static const BYTE vs_code_native[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xa7, 0x7e, 0x73, 0xd7, 0x38, 0x3d, 0xe2, 0x9b, 0xa6, 0x82, 0xc4, 0x1a, 0x29, 0x50, 0x6d, 0x0c, 0x01, 0x00, 0x00, 0x00, 0xc6, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xde, 0x00, 0x00, 0x00, 0x66, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x56, 0x65, 0x72, 0x74, 0x65, 0x78, 0x49, 0x44, 0x00,
        0x4f, 0x53, 0x47, 0x31, 0x56, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x56, 0x00, 0x50, 0x53,
        0x56, 0x30, 0x80, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x01, 0x02, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x41, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x06, 0x02, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x58, 0x05, 0x00, 0x00, 0x62, 0x00, 0x01, 0x00, 0x56, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x02, 0x01, 0x00, 0x00, 0x10, 0x00,
        0x00, 0x00, 0x40, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81,
        0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10,
        0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51,
        0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0,
        0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x13, 0x00,
        0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x44,
        0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x18, 0x63, 0x90, 0x41, 0x65, 0x06, 0xa0, 0x18, 0x60, 0x8c, 0x41, 0x0e, 0xa1, 0x51, 0x80, 0x62, 0x80, 0x31, 0x06,
        0x51, 0xb4, 0x06, 0x02, 0x72, 0x60, 0xcc, 0x11, 0x80, 0xc2, 0x1c, 0x41, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d,
        0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0, 0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74,
        0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07,
        0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x60, 0xc8, 0x53, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x07, 0x01, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf,
        0x02, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04,
        0x43, 0x1a, 0x25, 0x50, 0x04, 0x85, 0x50, 0x0c, 0x23, 0x00, 0x44, 0x4a, 0xa1, 0x04, 0x46, 0x00, 0x4a, 0xa2, 0x0c, 0x0a, 0xa1, 0x20, 0x8a, 0x80, 0x5e, 0x01, 0x03, 0x02, 0x22, 0x90, 0x1b, 0x6b,
        0x08, 0x8c, 0x39, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26,
        0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x62, 0x82, 0x30, 0x14, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc6, 0x06, 0x61, 0x30, 0x28, 0xd8, 0xcd, 0x4d, 0x10,
        0x86, 0x63, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x26, 0x08, 0x03, 0xb2, 0x01, 0xd8, 0x30, 0x30, 0x4d, 0xb3, 0x21, 0x70, 0x36, 0x0c, 0xc3, 0xf2, 0x4c, 0x10, 0x1c, 0x69, 0x43, 0x10, 0x91,
        0x68, 0x0b, 0x4b, 0x73, 0xe3, 0x32, 0x65, 0xf5, 0x65, 0x55, 0x26, 0x47, 0x57, 0x86, 0x97, 0x44, 0x34, 0x41, 0x20, 0x92, 0x09, 0x02, 0xa1, 0x6c, 0x08, 0x98, 0x09, 0x02, 0xb1, 0x6c, 0x58, 0x18,
        0xaa, 0xb2, 0x2e, 0x6c, 0xb0, 0x18, 0x0c, 0xd8, 0x10, 0x64, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x04, 0x33, 0x41, 0x20, 0x9a, 0x09, 0x02, 0xe1,
        0x6c, 0x58, 0x98, 0x8d, 0xeb, 0x2e, 0x6f, 0xf0, 0x18, 0x0c, 0x60, 0x60, 0x35, 0x41, 0x20, 0x9e, 0x09, 0x02, 0x01, 0x6d, 0x58, 0x06, 0x30, 0x08, 0x03, 0xec, 0x12, 0x83, 0x41, 0x0c, 0x06, 0x0c,
        0xd8, 0x20, 0x7c, 0x63, 0xb0, 0x61, 0xd0, 0xc8, 0x00, 0x98, 0x20, 0x3c, 0xd1, 0x06, 0x81, 0x31, 0x83, 0x0d, 0xc5, 0x32, 0x95, 0x01, 0x70, 0x06, 0x55, 0xd8, 0xd8, 0xec, 0xda, 0x5c, 0xd2, 0xc8,
        0xca, 0xdc, 0xe8, 0xa6, 0x04, 0x41, 0x15, 0x32, 0x3c, 0x17, 0xbb, 0x32, 0xb9, 0xb9, 0xb4, 0x37, 0xb7, 0x29, 0x01, 0xd1, 0x84, 0x0c, 0xcf, 0xc5, 0x2e, 0x8c, 0xcd, 0xae, 0x4c, 0x6e, 0x4a, 0x60,
        0xd4, 0x21, 0xc3, 0x73, 0x99, 0x43, 0x0b, 0x23, 0x2b, 0x93, 0x6b, 0x7a, 0x23, 0x2b, 0x63, 0x9b, 0x12, 0x24, 0x95, 0xc8, 0xf0, 0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2,
        0xe8, 0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x04, 0x4f, 0x1d, 0x32, 0x3c, 0x17, 0xbb, 0xb4, 0xb2, 0xbb, 0x24, 0xb2, 0x29, 0xba, 0x30, 0xba, 0xb2, 0x29, 0x41, 0x54, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d,
        0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x70, 0x06, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01,
        0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1,
        0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07,
        0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0,
        0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84,
        0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05,
        0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62,
        0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00, 0x71, 0x20,
        0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x23, 0xb7, 0x6d, 0x03, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13,
        0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb5, 0x6e, 0x02, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d,
        0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x34, 0x4a, 0x61, 0x06, 0x80, 0xce, 0x18, 0x01, 0x08, 0x82,
        0x20, 0xfc, 0x8d, 0x11, 0x80, 0x20, 0x08, 0xe2, 0xbf, 0x30, 0x46, 0x00, 0x82, 0x20, 0x48, 0x82, 0xc1, 0x08, 0xc0, 0x18, 0x01, 0x08, 0x82, 0x20, 0xfe, 0x01, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0x50, 0x64, 0x94, 0x24, 0x3d, 0xc6, 0x70, 0x43, 0x30, 0x81, 0xc1, 0x70, 0x83, 0x80, 0x81, 0xc1, 0x74, 0x83, 0x71, 0x04, 0xc3, 0x0d, 0x44, 0x46, 0x06, 0xd3, 0x0d, 0x09, 0x12, 0x4c, 0x37, 0x28,
        0x43, 0x61, 0x02, 0x23, 0x1f, 0x13, 0x1a, 0xf9, 0x98, 0xe0, 0xc0, 0xc7, 0x84, 0x07, 0x3e, 0x27, 0xa8, 0x3b, 0x41, 0xdd, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x2c, 0x64, 0x40, 0x8d, 0xc1, 0xc7,
        0x09, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xb0, 0x90, 0x01, 0x35, 0x06, 0x5f, 0x17, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x52, 0x06, 0xd4, 0xf7, 0x71, 0xc8, 0x88, 0x41, 0x02, 0x80, 0x20,
        0x18, 0x20, 0x65, 0x40, 0x7d, 0x5f, 0x77, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x52, 0x06, 0xd4, 0xf7, 0x5d, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x65, 0x40, 0x7d, 0x9f, 0xe6,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

#if 0
    struct VOut
    {
        float4 pos : SV_Position;
        min16float2 v : V;
    };

    StructuredBuffer<min16float4> A : register(t0);
    RWStructuredBuffer<min16float4> B : register(u0);

    min16float4 main(VOut vout) : SV_Target
    {
            min16float4 A0 = A[0];
            min16float4 A1 = A[1];
            B[0] = A0 + A1;
            B[1] = A0 - A1;
            return vout.v.xyxy * min16float(256.0 / 255.0);
    }
#endif
    static const BYTE ps_code_non_native[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xb8, 0x72, 0xb7, 0xba, 0xed, 0x66, 0x0a, 0xb5, 0x09, 0x01, 0xe0, 0x57, 0x0d, 0xbb, 0xea, 0xe5, 0x01, 0x00, 0x00, 0x00, 0x58, 0x09, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xa2, 0x00, 0x00, 0x00, 0xdc, 0x00, 0x00, 0x00, 0x98, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x56, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x56, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30,
        0xb4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x01, 0x42, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xb8, 0x07, 0x00, 0x00,
        0x62, 0x00, 0x00, 0x00, 0xee, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x02, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xa0, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0xe5, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f,
        0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x74, 0x23, 0x00,
        0x25, 0x00, 0x14, 0x46, 0x01, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0x21, 0x2a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21,
        0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x02, 0x35, 0x47, 0x10, 0x14, 0x63, 0x91, 0x42, 0x08, 0xa3, 0x36, 0x03, 0x50, 0x16, 0x40, 0x16, 0x21, 0xef, 0xbd, 0x57, 0x08, 0xc1, 0x9b, 0x86, 0xcb, 0x9f,
        0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x7b, 0xef, 0x91, 0x72, 0x48, 0xb2, 0x08, 0x29, 0x84, 0xe6, 0x40, 0xc0, 0x30, 0x02, 0xf1, 0xcc, 0x24, 0x0a, 0xe4,
        0x80, 0x0f, 0x6e, 0x10, 0x0e, 0xec, 0x90, 0x0e, 0xe7, 0xe0, 0x0e, 0xed, 0x50, 0x0e, 0xee, 0x40, 0x0f, 0x6e, 0xc0, 0x0e, 0xe5, 0x70, 0x0e, 0xe1, 0x30, 0x0e, 0xf9, 0xe0, 0x06, 0xe3, 0xc0, 0x0e,
        0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb3, 0x40, 0x0f, 0xf2, 0x50, 0x0f, 0xe3, 0x40, 0x0f, 0xf5, 0x20, 0x0f, 0xe5, 0x40, 0x0e, 0xa2, 0x50, 0x0f, 0xe6, 0x60, 0x0e, 0xe5, 0x20, 0x0f, 0x7c, 0x60,
        0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xed, 0x90, 0x0e, 0xee, 0x10, 0x07, 0x76, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40, 0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60,
        0xe0, 0x07, 0x28, 0xb0, 0x74, 0x67, 0x22, 0x05, 0x72, 0xc0, 0x07, 0x37, 0x08, 0x07, 0x76, 0x48, 0x87, 0x73, 0x70, 0x87, 0x76, 0x28, 0x07, 0x77, 0xa0, 0x07, 0x37, 0x60, 0x87, 0x72, 0x38, 0x87,
        0x70, 0x18, 0x87, 0x7c, 0x70, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x59, 0xb8, 0x85, 0x59, 0xa0, 0x07, 0x79, 0xa8, 0x87, 0x71, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x20,
        0x07, 0x51, 0xa8, 0x07, 0x73, 0x30, 0x87, 0x72, 0x90, 0x07, 0x3e, 0xb0, 0x87, 0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x83, 0x76, 0x48, 0x07, 0x77, 0x88, 0x03, 0x3b, 0x30, 0x07, 0x76,
        0x78, 0x87, 0x70, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14, 0x58, 0xca, 0x89, 0x40, 0xe6, 0x08, 0x40, 0x01, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87,
        0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76,
        0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90,
        0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x50, 0x04, 0x85, 0x50, 0x0c, 0x23, 0x00, 0x85, 0x51, 0x20, 0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0, 0x20, 0x8a, 0xa0, 0x40, 0x88,
        0x8d, 0x00, 0x10, 0x2f, 0x30, 0xd2, 0x33, 0x00, 0xb4, 0xc7, 0x42, 0x0c, 0x02, 0x00, 0x00, 0xa0, 0x50, 0x08, 0xcf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x63, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x08, 0x63,
        0x82, 0x40, 0x1c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x10, 0xc8, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x88, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x81, 0x50, 0x26, 0x08, 0x18, 0x46, 0x60,
        0x82, 0x40, 0x2c, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x0c, 0xcf, 0x86, 0x64, 0x61, 0x9a, 0x65, 0x19, 0x9c, 0x05, 0xda, 0x10, 0x44, 0x13, 0x04, 0xcd, 0x9a, 0x20, 0x30, 0xd4, 0x86, 0x65, 0x99, 0x9a,
        0x65, 0x19, 0x1c, 0x8a, 0xa2, 0xa0, 0x0d, 0x41, 0xb5, 0x81, 0x90, 0x2c, 0x00, 0x98, 0x20, 0x08, 0xc0, 0x06, 0x60, 0xc3, 0xb0, 0x64, 0xd9, 0x86, 0x40, 0xdb, 0x30, 0x0c, 0xd8, 0x36, 0x41, 0xd8,
        0xae, 0x0d, 0x41, 0x47, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb, 0x04, 0xa1, 0x68, 0x26, 0x08, 0x85, 0xb3, 0x21, 0x58, 0x26, 0x08,
        0xc5, 0x33, 0x41, 0x28, 0xa0, 0x0d, 0xcb, 0x02, 0x06, 0x61, 0x20, 0x06, 0x63, 0x40, 0x06, 0x03, 0x19, 0x2c, 0x65, 0x00, 0x30, 0xb0, 0x9a, 0x20, 0x14, 0xd1, 0x04, 0xa1, 0x90, 0x36, 0x2c, 0xc3,
        0x19, 0xa0, 0x41, 0x19, 0x8c, 0x41, 0x1a, 0x0c, 0x69, 0x30, 0x94, 0x01, 0xb0, 0x41, 0x30, 0x03, 0x35, 0x60, 0x32, 0x65, 0xf5, 0x45, 0x15, 0x26, 0x77, 0x56, 0x46, 0x37, 0x41, 0x28, 0xa6, 0x0d,
        0xcb, 0xc2, 0x06, 0x68, 0xd0, 0x06, 0x63, 0x50, 0x06, 0x03, 0x19, 0x2c, 0x65, 0x00, 0x6c, 0x08, 0xdc, 0x60, 0xc3, 0xb0, 0x06, 0x6f, 0x00, 0x4c, 0x10, 0xb8, 0x6a, 0x83, 0xb0, 0xc4, 0xc1, 0x86,
        0x02, 0xfb, 0xe0, 0xe0, 0x92, 0x83, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94,
        0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92,
        0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x02, 0xab, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda,
        0x9b, 0xdb, 0xdc, 0x94, 0x60, 0xab, 0x43, 0x86, 0xe7, 0x62, 0x97, 0x56, 0x76, 0x97, 0x44, 0x36, 0x45, 0x17, 0x46, 0x57, 0x36, 0x25, 0xe8, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1, 0xc9, 0xe5,
        0x41, 0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0xe4, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72,
        0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x36, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x05, 0x4c, 0xc3, 0xe5,
        0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0x45, 0xad, 0x5b, 0xc1, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x5f, 0xdc, 0xb6,
        0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x7e, 0x71, 0xdb, 0x26, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81, 0x52,
        0xd3, 0x43, 0x4d, 0x7e, 0x51, 0xeb, 0x06, 0x40, 0x30, 0x00, 0xd2, 0x00, 0x61, 0x20, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x54, 0x4a, 0x80, 0x48, 0x29, 0x94, 0xdc, 0x0c, 0x40, 0x59, 0x0a, 0x14, 0xa6, 0x00, 0x95, 0xf2, 0xa0, 0x31, 0x46, 0x40, 0x82, 0x0f, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xd0, 0x78,
        0x06, 0x72, 0x5d, 0xcd, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x34, 0xdf, 0x21, 0x61, 0x98, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x18, 0x6c, 0x5d, 0x36, 0x1d, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x60, 0x8c, 0x01, 0xe7, 0x69, 0x0b, 0x32, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x13, 0x18, 0x20, 0xc3, 0xb6, 0x19, 0xdd, 0x68, 0x42, 0x00, 0x5c, 0x30, 0x6e, 0x34, 0x61, 0x08, 0x2e,
        0x18, 0x37, 0x9a, 0x50, 0x08, 0x17, 0x8c, 0x1b, 0x4d, 0x38, 0x86, 0x0b, 0xc6, 0x8d, 0x18, 0x28, 0x00, 0x08, 0x82, 0xc1, 0x94, 0x06, 0x11, 0x83, 0x06, 0x64, 0xf0, 0x98, 0xc1, 0x68, 0x42, 0x00,
        0x5c, 0x30, 0x6e, 0x34, 0x61, 0x08, 0x2e, 0x18, 0x37, 0x9a, 0x50, 0x08, 0x17, 0x8c, 0x1b, 0x4d, 0x38, 0x86, 0x0b, 0xc6, 0xd9, 0x01, 0xc1, 0xc7, 0x8c, 0x07, 0x3e, 0x56, 0x38, 0xf0, 0x31, 0xa2,
        0x81, 0xcf, 0x11, 0x0f, 0x8f, 0x78, 0x78, 0xc4, 0xc3, 0x23, 0x1e, 0x8e, 0x18, 0x38, 0x00, 0x08, 0x82, 0x01, 0xb4, 0x07, 0x61, 0xe0, 0xd5, 0x41, 0x1d, 0x10, 0x83, 0x10, 0x80, 0xc1, 0x1d, 0x18,
        0xf6, 0xc4, 0xc7, 0x2e, 0x27, 0x3e, 0x66, 0x35, 0xf1, 0xb1, 0x8a, 0x89, 0xcf, 0x11, 0x0f, 0x8f, 0x78, 0x78, 0xc4, 0xc3, 0x23, 0x1e, 0x8e, 0x18, 0x38, 0x00, 0x08, 0x82, 0x01, 0x34, 0x0a, 0x69,
        0x60, 0x06, 0xa1, 0xd0, 0x07, 0xc4, 0x20, 0x04, 0x68, 0xf0, 0x07, 0x46, 0x06, 0x67, 0x20, 0x1f, 0x23, 0x03, 0x34, 0x90, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xa7, 0xf0, 0x06, 0x7f,
        0xf0, 0x07, 0x79, 0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x72, 0x0a, 0x6f, 0xf0, 0x07, 0x7f, 0x00, 0x07, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xa7, 0xf0, 0x06, 0x7f, 0xf0,
        0x07, 0x77, 0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x72, 0x0a, 0x6f, 0xf0, 0x07, 0x7f, 0xb0, 0x07, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00,
    };

    /* Expected warnings:
     * test.frag:4:2: warning: min16float is promoted to float16_t
     * test.frag:4:2: warning: min16float is promoted to float16_t
     * test.frag:7:18: warning: min16float is promoted to float16_t
     * test.frag:8:20: warning: min16float is promoted to float16_t
     * test.frag:10:1: warning: min16float is promoted to float16_t
     * test.frag:12:2: warning: min16float is promoted to float16_t
     * test.frag:13:2: warning: min16float is promoted to float16_t
     * test.frag:16:23: warning: min16float is promoted to float16_t */
    static const BYTE ps_code_native[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xca, 0x2a, 0xfa, 0xf3, 0x59, 0x80, 0x64, 0xe6, 0x33, 0xc0, 0x96, 0x4f, 0x6d, 0x58, 0xfb, 0xf4, 0x01, 0x00, 0x00, 0x00, 0xec, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xa2, 0x00, 0x00, 0x00, 0xdc, 0x00, 0x00, 0x00, 0x98, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x56, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x56, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30,
        0xb4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x01, 0x00, 0x02, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x01, 0x42, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x4c, 0x07, 0x00, 0x00,
        0x62, 0x00, 0x00, 0x00, 0xd3, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x02, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x34, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0xca, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f,
        0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00,
        0x25, 0x00, 0x14, 0x46, 0x01, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0x21, 0x2a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21,
        0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x02, 0x55, 0x16, 0x40, 0x16, 0x21, 0x63, 0x8c, 0x51, 0x08, 0xb1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0x5a,
        0x47, 0xc5, 0x18, 0x63, 0x90, 0x72, 0x38, 0xb2, 0x08, 0x29, 0x84, 0xde, 0x1c, 0x41, 0x50, 0x8c, 0x45, 0x0a, 0x21, 0x91, 0xe4, 0x40, 0xc0, 0x30, 0x02, 0x31, 0xcc, 0xe4, 0x06, 0xe3, 0xc0, 0x0e,
        0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb3, 0x40, 0x0f, 0xf2, 0x50, 0x0f, 0xe3, 0x40, 0x0f, 0xf5, 0x20, 0x0f, 0xe5, 0x40, 0x0e, 0xa2, 0x50, 0x0f, 0xe6, 0x60, 0x0e, 0xe5, 0x20, 0x0f, 0x7c, 0x60,
        0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xed, 0x90, 0x0e, 0xee, 0x10, 0x07, 0x76, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40, 0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60,
        0xe0, 0x07, 0x28, 0xa8, 0x64, 0x67, 0x82, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x59, 0xb8, 0x85, 0x59, 0xa0, 0x07, 0x79, 0xa8, 0x87, 0x71, 0xa0, 0x87, 0x7a, 0x90, 0x87,
        0x72, 0x20, 0x07, 0x51, 0xa8, 0x07, 0x73, 0x30, 0x87, 0x72, 0x90, 0x07, 0x3e, 0xb0, 0x87, 0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x83, 0x76, 0x48, 0x07, 0x77, 0x88, 0x03, 0x3b, 0x30,
        0x07, 0x76, 0x78, 0x87, 0x70, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14, 0x54, 0xc2, 0x89, 0x40, 0xe6, 0x08, 0x40, 0x01, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0,
        0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0, 0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x6d, 0x90,
        0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xd0, 0x06, 0xe6,
        0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x63, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90,
        0x07, 0x01, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x03, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x07, 0x08, 0x80, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x81, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14,
        0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x50, 0x04, 0x85, 0x50, 0x0c, 0x23, 0x00, 0x85, 0x51, 0x10, 0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0, 0x20,
        0x8a, 0xa0, 0x40, 0x28, 0x8e, 0x00, 0xd0, 0x2e, 0x70, 0x40, 0x40, 0x04, 0xca, 0x33, 0x00, 0xa4, 0xc7, 0x42, 0x0c, 0x02, 0x00, 0x00, 0xa0, 0x50, 0xe8, 0xce, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x63, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10,
        0x4c, 0x10, 0x08, 0x63, 0x82, 0x40, 0x1c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x10, 0xc8, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x88, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x81, 0x50, 0x26,
        0x08, 0x17, 0x46, 0x60, 0x82, 0x40, 0x2c, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x0c, 0xcf, 0x86, 0x64, 0x61, 0x9a, 0x65, 0x19, 0x9c, 0x05, 0xda, 0x10, 0x44, 0x13, 0x84, 0xcc, 0x9a, 0x20, 0x44, 0xd4,
        0x86, 0x65, 0x99, 0x9a, 0x65, 0x19, 0x1c, 0x8a, 0xa2, 0xa0, 0x0d, 0x41, 0xb5, 0x81, 0x90, 0x2c, 0x00, 0x98, 0x20, 0x08, 0xc0, 0x06, 0x60, 0xc3, 0xb0, 0x64, 0xd9, 0x86, 0x40, 0xdb, 0x30, 0x0c,
        0xd8, 0x36, 0x41, 0xd0, 0xae, 0x0d, 0x41, 0x47, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb, 0x04, 0xa1, 0x68, 0x26, 0x08, 0x85, 0xb3,
        0x21, 0x58, 0x26, 0x08, 0xc5, 0x33, 0x41, 0x28, 0xa0, 0x0d, 0xcb, 0x02, 0x06, 0x61, 0x20, 0x06, 0x63, 0x40, 0x06, 0x03, 0x19, 0x2c, 0x65, 0x00, 0x30, 0xb0, 0x9a, 0x20, 0x14, 0xd1, 0x04, 0xa1,
        0x90, 0x36, 0x2c, 0xc3, 0x19, 0xa0, 0x41, 0x19, 0x8c, 0x41, 0x1a, 0x0c, 0x69, 0x30, 0x94, 0x01, 0xb0, 0x41, 0x30, 0x03, 0x35, 0x60, 0x32, 0x65, 0xf5, 0x45, 0x15, 0x26, 0x77, 0x56, 0x46, 0x37,
        0x41, 0x28, 0xa6, 0x0d, 0xcb, 0xc2, 0x06, 0x68, 0xd0, 0x06, 0x63, 0x50, 0x06, 0x03, 0x19, 0x2c, 0x65, 0x00, 0x6c, 0x08, 0xdc, 0x60, 0xc3, 0xb0, 0x06, 0x6f, 0x00, 0x4c, 0x10, 0xb6, 0x6a, 0x83,
        0xb0, 0xc4, 0xc1, 0x86, 0x02, 0xfb, 0xe0, 0xe0, 0x92, 0x83, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c,
        0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95,
        0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x02, 0xab, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b,
        0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x94, 0x60, 0xab, 0x43, 0x86, 0xe7, 0x62, 0x97, 0x56, 0x76, 0x97, 0x44, 0x36, 0x45, 0x17, 0x46, 0x57, 0x36, 0x25, 0xe8, 0xea, 0x90, 0xe1, 0xb9, 0x94,
        0xb9, 0xd1, 0xc9, 0xe5, 0x41, 0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0xe4, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66,
        0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e,
        0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b,
        0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0,
        0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83,
        0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76,
        0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30,
        0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc,
        0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87,
        0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84,
        0x05, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0x45, 0xad, 0x1b, 0xc1, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00,
        0x83, 0x5f, 0xd4, 0xba, 0x0d, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x7e, 0x51, 0xeb, 0x26, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34,
        0x39, 0x11, 0x81, 0x52, 0xd3, 0x43, 0x4d, 0x7e, 0x51, 0xeb, 0x06, 0x40, 0x30, 0x00, 0xd2, 0x00, 0x61, 0x20, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x54, 0x4a, 0x80, 0x48, 0x29, 0x94, 0xdc, 0x0c, 0x40, 0x59, 0x0a, 0x14, 0xa6, 0x00, 0x95, 0xf2, 0xa0, 0x31, 0x46, 0x40, 0x82, 0x0f, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x20, 0x71, 0x06, 0x72, 0x5d, 0xcd, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x48, 0xdd, 0x21, 0x61, 0x98, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x18, 0x6c, 0x5d, 0x36, 0x1d,
        0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x8c, 0x01, 0xe7, 0x69, 0x0b, 0x32, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x4f, 0x18, 0x20, 0xc3, 0xb6, 0x19, 0xde, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08,
        0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0x81, 0x02, 0x80, 0x20, 0x18, 0x3c, 0x66, 0xd0, 0x20, 0x64, 0x00, 0x06, 0xcb, 0x18, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26,
        0x0c, 0xc2, 0x68, 0x02, 0x31, 0x18, 0x91, 0xc0, 0xc7, 0x88, 0x04, 0x3e, 0x46, 0x24, 0xf0, 0x31, 0x22, 0x81, 0xcf, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x30, 0x70, 0x50, 0x49, 0x69, 0x90, 0x06,
        0xc4, 0x20, 0x04, 0x14, 0x1b, 0x58, 0x83, 0xc4, 0xc7, 0x1a, 0x24, 0x3e, 0xd6, 0x20, 0xf1, 0xb1, 0x06, 0x89, 0xcf, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x30, 0x74, 0x90, 0x59, 0x71, 0xd0, 0x06,
        0xc4, 0x20, 0x04, 0x18, 0x1c, 0x18, 0x75, 0xc9, 0xc7, 0x28, 0x4c, 0x3e, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xdc, 0xc1, 0xf7, 0x06, 0x6f, 0x90, 0x06, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20,
        0x18, 0x20, 0x77, 0xf0, 0xbd, 0xc1, 0x1b, 0x80, 0x41, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc8, 0x1d, 0x7c, 0x6f, 0xf0, 0x06, 0x67, 0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x72,
        0x07, 0xdf, 0x1b, 0xbc, 0xc1, 0x1a, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00,
    };

    static const struct D3D12_SHADER_BYTECODE vs_non_native = { vs_code_non_native, sizeof(vs_code_non_native) };
    static const struct D3D12_SHADER_BYTECODE ps_non_native = { ps_code_non_native, sizeof(ps_code_non_native) };
    static const struct D3D12_SHADER_BYTECODE vs_native = { vs_code_native, sizeof(vs_code_native) };
    static const struct D3D12_SHADER_BYTECODE ps_native = { ps_code_native, sizeof(ps_code_native) };

    struct test
    {
        const struct D3D12_SHADER_BYTECODE *vs;
        const struct D3D12_SHADER_BYTECODE *ps;
        bool native_fp16;
        unsigned int src_offset;
        unsigned int dst_offset;
        const float *reference_fp32;
        const uint16_t *reference_fp16;
    };

    static const float reference_fp32[] = { 6.0f, 8.0f, 10.0f, 12.0f, -4.0f, -4.0f, -4.0f, -4.0f };
    static const uint16_t reference_fp16[] = { 0x4080, 0x40c0, 0x4100, 0x4140, 0xb400, 0xb400, 0xb400, 0xb400 };
    static const struct test tests[] =
    {
        { &vs_non_native, &ps_non_native, false, 0, 0, reference_fp32, NULL },
        { &vs_native, &ps_native, true, 32, 32, NULL, reference_fp16 },
    };

    static const float white[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    D3D12_VIEWPORT vp = { 0, 0, 1, 1, 0, 1 };
    D3D12_RECT sci = { 0, 0, 1, 1 };
    uint32_t upload_data[8 + 4];

    memset(&desc, 0, sizeof(desc));
    desc.no_root_signature = true;
    desc.no_pipeline = true;
    desc.rt_width = ARRAY_SIZE(tests);
    desc.rt_height = 1;
    desc.rt_format = DXGI_FORMAT_R8G8B8A8_UNORM;
    if (!init_test_context(&context, &desc))
        return;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    {
        /* With non-native FP16 declared in the DXIL, raw buffers behave as if they are 32-bit types, despite being declared with FP16 in the shader. */
        float v;
        for (i = 0; i < 8; i++)
        {
            v = 1.0f + (float)i;
            memcpy(&upload_data[i], &v, sizeof(v));
        }
    }

    {
        /* With native FP16 (-enable-16bit-types), min16float is true FP16, so we should bake down some halfs. */
        uint16_t halfs[8];
        for (i = 0; i < 8; i++)
            halfs[i] = 0x3c00 + (i << 6);

        memcpy(upload_data + 8, halfs, sizeof(halfs));
    }
    src = create_upload_buffer(context.device, sizeof(upload_data), upload_data);
    dst = create_default_buffer(context.device, sizeof(upload_data), D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    memset(&features4, 0, sizeof(features4));
    ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS4, &features4, sizeof(features4));

    memset(&rs_desc, 0, sizeof(rs_desc));
    rs_desc.NumParameters = ARRAY_SIZE(root_parameters);
    rs_desc.pParameters = root_parameters;
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    create_root_signature(context.device, &rs_desc, &context.root_signature);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        init_pipeline_state_desc_dxil(&pso_desc, context.root_signature, DXGI_FORMAT_R8G8B8A8_UNORM, tests[i].vs, tests[i].ps, NULL);
        pso_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_TRIANGLE;
        pso_desc.RasterizerState.CullMode = D3D12_CULL_MODE_NONE;

        hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc, &IID_ID3D12PipelineState, (void **)&pso[i]);

        if (tests[i].native_fp16 && features4.Native16BitShaderOpsSupported)
            ok(SUCCEEDED(hr), "Failed to create pipeline, hr #%u.\n", hr);
        else if (tests[i].native_fp16 && !features4.Native16BitShaderOpsSupported)
        {
            ok(hr == E_INVALIDARG, "Unexpected hr: %x.\n", hr);
            skip("NativeFP16 is not supported. Failing pipeline compilation is expected.\n");
        }
        else
            ok(SUCCEEDED(hr), "Failed to create pipeline, hr #%u.\n", hr);

        if (FAILED(hr))
            pso[i] = NULL;
    }

    ID3D12GraphicsCommandList_OMSetRenderTargets(context.list, 1, &context.rtv, TRUE, NULL);
    ID3D12GraphicsCommandList_ClearRenderTargetView(context.list, context.rtv, white, 0, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(context.list, context.root_signature);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        if (pso[i])
        {
            ID3D12GraphicsCommandList_SetPipelineState(context.list, pso[i]);
            vp.TopLeftX = (float)i;
            sci.left = i;
            sci.right = i + 1;
            ID3D12GraphicsCommandList_RSSetViewports(context.list, 1, &vp);
            ID3D12GraphicsCommandList_RSSetScissorRects(context.list, 1, &sci);
            ID3D12GraphicsCommandList_IASetPrimitiveTopology(context.list, D3D_PRIMITIVE_TOPOLOGY_TRIANGLELIST);
            ID3D12GraphicsCommandList_SetGraphicsRootShaderResourceView(context.list, 0, ID3D12Resource_GetGPUVirtualAddress(src) + tests[i].src_offset);
            ID3D12GraphicsCommandList_SetGraphicsRootUnorderedAccessView(context.list, 1, ID3D12Resource_GetGPUVirtualAddress(dst) + tests[i].dst_offset);
            ID3D12GraphicsCommandList_DrawInstanced(context.list, 3, 1, 0, 0);
        }
    }

    transition_resource_state(context.list, context.render_target, D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    transition_resource_state(context.list, dst, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_texture_readback_with_command_list(context.render_target, 0, &rb, context.queue, context.list);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        unsigned int expected = pso[i] ? 0x80808080 : 0xffffffff;
        value = get_readback_uint(&rb, i, 0, 0);
        ok(value == expected, "0x%x != 0x%x", value, expected);
    }
    release_resource_readback(&rb);

    reset_command_list(context.list, context.allocator);
    get_buffer_readback_with_command_list(dst, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);
    for (i = 0; i < ARRAY_SIZE(upload_data); i++)
        upload_data[i] = get_readback_uint(&rb, i, 0, 0);
    release_resource_readback(&rb);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        if (!pso[i])
            continue;

        if (tests[i].native_fp16)
        {
            uint16_t f16_values[8];
            memcpy(f16_values, ((const uint8_t *)upload_data) + tests[i].dst_offset, sizeof(f16_values));
            for (j = 0; j < ARRAY_SIZE(f16_values); j++)
                ok(f16_values[j] == tests[i].reference_fp16[j], "Test %u, fp16 value %u, 0x%x != 0x%x\n", i, j, f16_values[j], tests[i].reference_fp16[j]);
        }
        else
        {
            float f32_values[8];
            memcpy(f32_values, ((const uint8_t *)upload_data) + tests[i].dst_offset, sizeof(f32_values));
            for (j = 0; j < ARRAY_SIZE(f32_values); j++)
                ok(f32_values[j] == tests[i].reference_fp32[j], "Test %u, fp32 value %u, %f != %f\n", i, j, f32_values[j], tests[i].reference_fp32[j]);
        }
    }

    ID3D12Resource_Release(src);
    ID3D12Resource_Release(dst);
    for (i = 0; i < ARRAY_SIZE(pso); i++)
        if (pso[i])
            ID3D12PipelineState_Release(pso[i]);
    destroy_test_context(&context);
}

void test_shader_sm62_denorm(void)
{
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    struct test_context context;
    struct resource_readback rb;
    ID3D12PipelineState *pso;
    ID3D12Resource *src;
    ID3D12Resource *dst;
    uint32_t value;
    unsigned int i;

#if 0
    RWStructuredBuffer<float> RW : register(u0);
    StructuredBuffer<float> RO : register(t0);

    [numthreads(1, 1, 1)]
    void main(uint3 id : SV_DispatchThreadID)
    {
        float v = RO[id.x];
        v += v;
        RW[id.x] = v;
    }
#endif
    /* -denorm ftz */
    static const BYTE cs_code_ftz[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xd4, 0x0a, 0xa5, 0x22, 0xa2, 0xfb, 0x4b, 0xac, 0x99, 0x15, 0xd4, 0xd6, 0x5a, 0x2d, 0x26, 0x37, 0x01, 0x00, 0x00, 0x00, 0xa0, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x5c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xd0, 0x05, 0x00, 0x00, 0x62, 0x00, 0x05, 0x00, 0x74, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x02, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0xb8, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x6b, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x1b, 0xb0, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0xc2, 0x1c, 0xe0, 0x61,
        0x0e, 0xe4, 0xa0, 0x0d, 0xc8, 0xa1, 0x1c, 0xdc, 0xe1, 0x1d, 0xe4, 0xa1, 0x1d, 0xda, 0xa0, 0x1d, 0xde, 0x81, 0x1c, 0xca, 0x01, 0x30, 0x07, 0x7a, 0xd0, 0x07, 0x60, 0x83, 0x21, 0xfc, 0xff, 0xff,
        0xff, 0xff, 0x00, 0x48, 0x00, 0xb5, 0x81, 0x18, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0x00, 0xa4, 0x0d, 0x06, 0xf1, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x15, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x86, 0x09, 0x01, 0x01, 0x00, 0x89, 0x20, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04,
        0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x60, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c,
        0x63, 0x0c, 0x22, 0x73, 0x04, 0x08, 0x99, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0xc1, 0x99, 0x01, 0x28, 0x0b, 0x18, 0x68, 0x8c, 0x94, 0x52, 0x32, 0x83,
        0xd4, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48, 0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x22, 0xa5, 0x94, 0x46, 0x39, 0xd8, 0x40, 0x63, 0x98, 0x41, 0x6d, 0x8e, 0x20, 0x28, 0x06,
        0x1a, 0x66, 0x8c, 0x47, 0x70, 0x20, 0x60, 0xa6, 0x6e, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x60, 0x16, 0xe8, 0x41, 0x1e, 0xea, 0x61, 0x1c, 0xe8, 0xa1, 0x1e, 0xe4, 0xa1, 0x1c, 0xc8,
        0x41, 0x14, 0xea, 0xc1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xcc, 0x81, 0x1d, 0xde, 0x21, 0x1c, 0xe8, 0xc1, 0x0f, 0x50, 0x90, 0x68, 0xce, 0xf4, 0x8d, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc,
        0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x81, 0x39,
        0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xf8, 0x01, 0x0a, 0x12, 0xd5, 0x39, 0x02, 0x50, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0,
        0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6,
        0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x80, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00,
        0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x04, 0x85, 0x50, 0x0c, 0x23, 0x00, 0x85, 0x41, 0x6f, 0x04,
        0x80, 0x6e, 0x81, 0x90, 0x9d, 0x01, 0x20, 0x3a, 0x03, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec,
        0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x62, 0x82, 0x30, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc7, 0x06, 0x61,
        0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x06, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x61, 0x48, 0x26, 0x08, 0x94, 0x43, 0x60, 0x82, 0x30, 0x28, 0x1b, 0x92, 0x85, 0x69, 0x96, 0x65, 0x70, 0x96, 0x63,
        0x43, 0xf0, 0x4c, 0x10, 0xac, 0x66, 0x82, 0xf0, 0x2c, 0x1b, 0x96, 0x25, 0x6a, 0x96, 0x65, 0x70, 0x24, 0x49, 0x3a, 0x36, 0x04, 0xd3, 0x06, 0x02, 0xa2, 0x00, 0x60, 0x82, 0x20, 0x00, 0x1b, 0x80,
        0x0d, 0xc3, 0x72, 0x5d, 0x1b, 0x02, 0x6c, 0xc3, 0x30, 0x58, 0x19, 0x89, 0xb6, 0xb0, 0x34, 0xb7, 0x09, 0xc2, 0xc5, 0x6c, 0x18, 0x86, 0x61, 0xd8, 0x40, 0x2c, 0x9c, 0xd1, 0x6d, 0x28, 0xac, 0x0d,
        0xa8, 0xbc, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86,
        0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e,
        0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x02, 0xaa, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x94,
        0x20, 0xab, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0xf0, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c,
        0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e,
        0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c,
        0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e,
        0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4,
        0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07,
        0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5,
        0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b,
        0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90,
        0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x46, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84,
        0x40, 0x33, 0x2c, 0x84, 0x0d, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0xc5, 0x6d, 0x9b, 0x00, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60,
        0x6d, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39, 0x11, 0xe1, 0x17, 0xb7, 0x6d, 0x01, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0x47, 0x44, 0x00, 0x83, 0x38, 0xf8, 0xc8, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48,
        0x03, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x34, 0x4a, 0xae, 0x74, 0x03, 0xca, 0x52, 0xa0,
        0x30, 0x05, 0xc8, 0x94, 0xc0, 0x08, 0x00, 0xa5, 0x19, 0x80, 0x31, 0x02, 0x10, 0x04, 0x41, 0x10, 0x0c, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x00, 0x4d, 0x08, 0xe1, 0x38, 0xcc,
        0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x40, 0x54, 0x42, 0x3c, 0x4f, 0x33, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x04, 0x96, 0x40, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xd0, 0x5c, 0x89, 0x10, 0x44,
        0xc7, 0x34, 0x9a, 0x10, 0x00, 0x16, 0x18, 0xf2, 0x19, 0x31, 0x70, 0x00, 0x10, 0x04, 0x03, 0x65, 0x5b, 0x0c, 0x82, 0x0a, 0x10, 0x04, 0x51, 0x2c, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    /* -denorm preserve */
    static const BYTE cs_code_preserve[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x7c, 0x5e, 0x67, 0x8e, 0xbc, 0xf4, 0x16, 0xc1, 0xcb, 0x9f, 0xed, 0xc2, 0xa1, 0xce, 0x48, 0x34, 0x01, 0x00, 0x00, 0x00, 0xa8, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x5c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xd8, 0x05, 0x00, 0x00, 0x62, 0x00, 0x05, 0x00, 0x76, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x02, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0xc0, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x6d, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x1b, 0xba, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0xc2, 0x1c, 0xe0, 0x61,
        0x0e, 0xe4, 0xa0, 0x0d, 0xc8, 0xa1, 0x1c, 0xdc, 0xe1, 0x1d, 0xe4, 0xa1, 0x1d, 0xda, 0xa0, 0x1d, 0xde, 0x81, 0x1c, 0xca, 0x01, 0x80, 0x07, 0x79, 0x28, 0x87, 0x79, 0x28, 0x07, 0x79, 0xb0, 0x87,
        0x72, 0x00, 0x36, 0x18, 0xc2, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x80, 0x04, 0x50, 0x1b, 0x88, 0xe1, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0xda, 0x60, 0x10, 0xff, 0xff, 0xff, 0xff, 0x3f, 0x00, 0x12,
        0x50, 0x01, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x86, 0x09, 0x01, 0x01, 0x00, 0x89, 0x20, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x60, 0x23, 0x00,
        0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x08, 0x99, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0xc1, 0x99, 0x01, 0x28,
        0x0b, 0x18, 0x68, 0x8c, 0x94, 0x52, 0x32, 0x83, 0xd4, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48, 0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x22, 0xa5, 0x94, 0x46, 0x39, 0xd8, 0x40,
        0x63, 0x98, 0x41, 0x6d, 0x8e, 0x20, 0x28, 0x06, 0x1a, 0x66, 0x8c, 0x47, 0x70, 0x20, 0x60, 0xa6, 0x6e, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x60, 0x16, 0xe8, 0x41, 0x1e, 0xea, 0x61,
        0x1c, 0xe8, 0xa1, 0x1e, 0xe4, 0xa1, 0x1c, 0xc8, 0x41, 0x14, 0xea, 0xc1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xcc, 0x81, 0x1d, 0xde, 0x21, 0x1c, 0xe8, 0xc1, 0x0f, 0x50, 0x90, 0x68, 0xce,
        0xf4, 0x8d, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98,
        0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xf8, 0x01, 0x0a, 0x12, 0xd5, 0x39, 0x02, 0x50, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87,
        0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76,
        0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20,
        0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x04, 0x85,
        0x50, 0x0c, 0x23, 0x00, 0x85, 0x41, 0x6f, 0x04, 0x80, 0x6e, 0x81, 0x90, 0x9d, 0x01, 0x20, 0x3a, 0x03, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x62, 0x82, 0x30, 0x18, 0x1b,
        0x84, 0x81, 0x98, 0x20, 0x0c, 0xc7, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x06, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x61, 0x48, 0x26, 0x08, 0x94, 0x43, 0x60, 0x82, 0x30, 0x28, 0x1b,
        0x92, 0x85, 0x69, 0x96, 0x65, 0x70, 0x96, 0x63, 0x43, 0xf0, 0x4c, 0x10, 0xac, 0x66, 0x82, 0xf0, 0x2c, 0x1b, 0x96, 0x25, 0x6a, 0x96, 0x65, 0x70, 0x24, 0x49, 0x3a, 0x36, 0x04, 0xd3, 0x06, 0x02,
        0xa2, 0x00, 0x60, 0x82, 0x20, 0x00, 0x1b, 0x80, 0x0d, 0xc3, 0x72, 0x5d, 0x1b, 0x02, 0x6c, 0xc3, 0x30, 0x58, 0x19, 0x89, 0xb6, 0xb0, 0x34, 0xb7, 0x09, 0xc2, 0xc5, 0x6c, 0x18, 0x86, 0x61, 0xd8,
        0x40, 0x2c, 0x9c, 0xd1, 0x6d, 0x28, 0xac, 0x0d, 0xa8, 0xbc, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c,
        0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95,
        0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x02, 0xaa, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b,
        0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x94, 0x20, 0xab, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0xf0, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87,
        0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x46, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x0d, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0xc5, 0x6d, 0x9b,
        0x00, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39, 0x11, 0xe1, 0x17, 0xb7, 0x6d, 0x01, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0x47, 0x44, 0x00, 0x83, 0x38,
        0xf8, 0xc8, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x34, 0x4a, 0xae, 0x74, 0x03, 0xca, 0x52, 0xa0, 0x30, 0x05, 0xc8, 0x94, 0xc0, 0x08, 0x00, 0xa5, 0x19, 0x80, 0x31, 0x02, 0x10, 0x04, 0x41, 0x10, 0x0c, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x00, 0x4d, 0x08, 0xe1, 0x38, 0xcc, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x40, 0x54, 0x42, 0x3c, 0x4f, 0x33, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x04, 0x96, 0x40, 0x23, 0x06, 0x0a,
        0x00, 0x82, 0x60, 0xd0, 0x5c, 0x89, 0x10, 0x44, 0xc7, 0x34, 0x9a, 0x10, 0x00, 0x16, 0x18, 0xf2, 0x19, 0x31, 0x70, 0x00, 0x10, 0x04, 0x03, 0x65, 0x5b, 0x0c, 0x82, 0x0a, 0x10, 0x04, 0x51, 0x2c,
        0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const struct D3D12_SHADER_BYTECODE cs_ftz = { cs_code_ftz, sizeof(cs_code_ftz) };
    static const struct D3D12_SHADER_BYTECODE cs_preserve = { cs_code_preserve, sizeof(cs_code_preserve) };

    struct test
    {
        const struct D3D12_SHADER_BYTECODE *cs;
        uint32_t input_fp32;
        uint32_t output_fp32;
    };

    static const struct test tests[] =
    {
        /* In denorms, additions work like uint32_t. */
        { &cs_ftz, 10, 0 },
        { &cs_preserve, 10, 20 },
    };

    if (!init_compute_test_context(&context))
        return;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&shader_model, 0, sizeof(shader_model));
    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_2;
    if (FAILED(ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model))) ||
        shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_2)
    {
        skip("Device does not support SM 6.2.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&rs_desc, 0, sizeof(rs_desc));
    rs_desc.NumParameters = ARRAY_SIZE(root_parameters);
    rs_desc.pParameters = root_parameters;
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    create_root_signature(context.device, &rs_desc, &context.root_signature);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);
        pso = create_compute_pipeline_state(context.device, context.root_signature, *tests[i].cs);
        src = create_upload_buffer(context.device, sizeof(tests[i].input_fp32), &tests[i].input_fp32);
        dst = create_default_buffer(context.device, sizeof(tests[i].input_fp32), D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, context.root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
        ID3D12GraphicsCommandList_SetComputeRootShaderResourceView(context.list, 0, ID3D12Resource_GetGPUVirtualAddress(src));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, ID3D12Resource_GetGPUVirtualAddress(dst));
        ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);

        transition_resource_state(context.list, dst, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
        get_buffer_readback_with_command_list(dst, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

        value = get_readback_uint(&rb, 0, 0, 0);
        ok(value == tests[i].output_fp32, "0x%x != 0x%x\n", value, tests[i].output_fp32);

        ID3D12Resource_Release(src);
        ID3D12Resource_Release(dst);
        ID3D12PipelineState_Release(pso);
        release_resource_readback(&rb);
        reset_command_list(context.list, context.allocator);
    }

    vkd3d_test_set_context(NULL);
    destroy_test_context(&context);
}

void test_shader_sm66_packed(void)
{
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_FEATURE_DATA_D3D12_OPTIONS4 features4;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    struct test_context context;
    struct resource_readback rb;
    ID3D12PipelineState *pso;
    ID3D12Resource *src;
    ID3D12Resource *dst;
    unsigned int i, j;
    uint32_t value;

    static const BYTE cs_code_unpack_u8_u32[] =
    {
#if 0
        StructuredBuffer<uint> RO : register(t0);
        RWStructuredBuffer<uint4> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = unpack_u8u32(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0xd2, 0x34, 0x75, 0x27, 0xab, 0x37, 0x91, 0x0b, 0x5a, 0x16, 0x74, 0x64, 0xb3, 0x50, 0x87, 0x62, 0x01, 0x00, 0x00, 0x00, 0x9c, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x5c, 0x92, 0x7e, 0x15, 0x47, 0x02, 0xd6, 0x6a, 0xd6, 0xd3, 0xb8,
        0x4f, 0x75, 0xbe, 0xeb, 0x44, 0x58, 0x49, 0x4c, 0x90, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xa4, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x78, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x9b, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x07, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x85,
        0x43, 0x45, 0xc8, 0x6d, 0x23, 0x62, 0x8c, 0x31, 0xe6, 0x08, 0x90, 0x42, 0x98, 0x71, 0x06, 0xa1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x91, 0xdb, 0x46,
        0xc5, 0x18, 0x63, 0x0c, 0x3a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82, 0x55, 0x0e, 0x35, 0xd8, 0x18, 0x67, 0x50, 0x2b, 0x0b, 0x18, 0x6c, 0x8c, 0x31,
        0xc6, 0x38, 0x83, 0xde, 0x6d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48, 0xfe, 0x4a, 0x48, 0x0e, 0x15, 0x09, 0x44, 0x1a, 0x39, 0x0f, 0x11, 0x4d, 0x08, 0x21, 0x21, 0x61, 0x8c, 0x42, 0xb0, 0xc1, 0x22,
        0xc9, 0x83, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x36, 0xa4, 0x19, 0x10, 0x31, 0xc6, 0x38, 0x73, 0x04, 0x41, 0x29, 0xd8, 0xa0, 0xa3, 0x92, 0x1d, 0x08, 0x98, 0x89, 0x0c, 0xc6,
        0x81, 0x1d, 0xc2, 0x61, 0x1e, 0xe6, 0xc1, 0x0d, 0x66, 0x81, 0x1e, 0xe4, 0xa1, 0x1e, 0xc6, 0x81, 0x1e, 0xea, 0x41, 0x1e, 0xca, 0x81, 0x1c, 0x44, 0xa1, 0x1e, 0xcc, 0xc1, 0x1c, 0xca, 0x41, 0x1e,
        0xf8, 0xa0, 0x1e, 0xdc, 0x61, 0x1e, 0xd2, 0xe1, 0x1c, 0xdc, 0xa1, 0x1c, 0xc8, 0x01, 0x0c, 0xd2, 0xc1, 0x1d, 0xe8, 0xc1, 0x0f, 0x50, 0x30, 0x28, 0x0f, 0x23, 0x10, 0xc3, 0x4c, 0x72, 0x30, 0x0e,
        0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x20, 0x0b, 0xb7, 0x30, 0x0b, 0xf4, 0x20, 0x0f, 0xf5, 0x30, 0x0e, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0xe4, 0x20, 0x0a, 0xf5, 0x60, 0x0e, 0xe6, 0x50,
        0x0e, 0xf2, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40, 0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x50, 0x0f, 0xee, 0x30, 0x0f, 0xe9, 0x70, 0x0e, 0xee, 0x50, 0x0e, 0xe4, 0x00, 0x06, 0xe9, 0xe0, 0x0e, 0xf4,
        0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00, 0x06, 0x7e, 0x80, 0x82, 0x4d, 0x7c, 0x8e, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79,
        0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60,
        0xc8, 0x23, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0a, 0x00, 0x00, 0x00,
        0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23, 0x00, 0x85, 0x51, 0x08, 0x05, 0x42, 0x75, 0x04, 0x80, 0x7c, 0x81, 0x50,
        0x9f, 0x01, 0x20, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1,
        0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73,
        0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x41, 0x83, 0x08, 0x4c,
        0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xd0, 0x3d, 0x13, 0x84, 0xaa, 0x99, 0x20, 0x0c, 0xcc, 0x06,
        0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d, 0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3c, 0x67,
        0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x18, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c,
        0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c,
        0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4,
        0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f,
        0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37,
        0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0xd8, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54,
        0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d,
        0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x0f, 0x20, 0x28, 0xf8, 0xc5, 0xfb, 0xc8, 0x6d, 0x1b, 0x00, 0xc1,
        0x00, 0x48, 0x03, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x34, 0x4a, 0x76, 0xa0, 0x60, 0x07, 0x4a, 0x37,
        0xa0, 0x2c, 0x05, 0xca, 0x76, 0xa0, 0x30, 0x81, 0x0a, 0x53, 0x80, 0x4e, 0x09, 0x8c, 0x00, 0x94, 0x07, 0xc5, 0x39, 0x04, 0x65, 0x99, 0x43, 0xb0, 0x18, 0xa2, 0x73, 0x10, 0x49, 0x92, 0x60, 0x23,
        0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x65, 0x8e, 0x60, 0x49, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x69, 0x8f, 0x70, 0x4d, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60,
        0x40, 0x80, 0x81, 0x83, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0xd4, 0x41, 0xc2, 0x31, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x0d, 0x18, 0x3c, 0x81, 0xa0, 0x2d, 0xd8, 0x68, 0x42, 0x00, 0x8c,
        0x18, 0x1c, 0x00, 0x08, 0x82, 0x01, 0x32, 0x06, 0x10, 0x13, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0x54,
        0x06, 0xd8, 0xe2, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0xc1, 0x73, 0x06, 0x54, 0xa0, 0x88, 0x41, 0x41, 0x0c, 0x42, 0x04, 0x06, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_unpack_u8_u16[] =
    {
#if 0
        StructuredBuffer<uint> RO : register(t0);
        RWStructuredBuffer<uint16_t4> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = unpack_u8u16(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x37, 0x20, 0xa8, 0x3a, 0x3c, 0xae, 0xb8, 0x1f, 0x3f, 0xb9, 0xe9, 0xef, 0x71, 0x41, 0x07, 0xaa, 0x01, 0x00, 0x00, 0x00, 0xa4, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0xb0, 0x70, 0xcf, 0xbb, 0x36, 0x06, 0xee, 0x0d, 0x78, 0xb0, 0xfe,
        0xf3, 0xf4, 0xe1, 0xc6, 0x44, 0x58, 0x49, 0x4c, 0x98, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xa6, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x80, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x9d, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x80, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x47, 0x00, 0x1d, 0x34, 0x5c, 0xfe, 0x84,
        0x3d, 0x84, 0xe4, 0x17, 0x0e, 0x15, 0x21, 0xb5, 0x8e, 0x08, 0xc3, 0x30, 0x8c, 0x39, 0x02, 0xa4, 0x10, 0xc7, 0x80, 0x0c, 0x94, 0xdc, 0x34, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0xaf, 0x84, 0xb4,
        0x12, 0x93, 0x8f, 0xdc, 0x36, 0x2a, 0x86, 0x61, 0x18, 0x06, 0x42, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0x4c, 0x39, 0x96, 0xa1, 0x19, 0x06, 0x64,
        0x20, 0xa7, 0x2c, 0xc0, 0xd0, 0x0c, 0x83, 0x61, 0x18, 0x06, 0x32, 0x10, 0x74, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43, 0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13,
        0x42, 0x48, 0x48, 0x18, 0x86, 0x42, 0x34, 0x43, 0x23, 0xd1, 0x74, 0xd0, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x86, 0x34, 0x03, 0x22, 0x86, 0x61, 0x40, 0xe6, 0x08, 0x82, 0x52,
        0x34, 0x43, 0x35, 0x58, 0x74, 0x0d, 0x04, 0xcc, 0x44, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb3, 0x40, 0x0f, 0xf2, 0x50, 0x0f, 0xe3, 0x40, 0x0f, 0xf5, 0x20, 0x0f, 0xe5,
        0x40, 0x0e, 0xa2, 0x50, 0x0f, 0xe6, 0x60, 0x0e, 0xe5, 0x20, 0x0f, 0x7c, 0x50, 0x0f, 0xee, 0x30, 0x0f, 0xe9, 0x70, 0x0e, 0xee, 0x50, 0x0e, 0xe4, 0x00, 0x06, 0xe9, 0xe0, 0x0e, 0xf4, 0xe0, 0x07,
        0x28, 0x18, 0x48, 0x1b, 0x46, 0x20, 0x8c, 0x99, 0xdc, 0x60, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x40, 0x16, 0x6e, 0x61, 0x16, 0xe8, 0x41, 0x1e, 0xea, 0x61, 0x1c, 0xe8, 0xa1, 0x1e,
        0xe4, 0xa1, 0x1c, 0xc8, 0x41, 0x14, 0xea, 0xc1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xec, 0xa1, 0x1c, 0xc6, 0x81, 0x1e, 0xde, 0x41, 0x1e, 0xf8, 0xa0, 0x1e, 0xd2, 0xc1, 0x1d, 0xe8, 0x21,
        0x0e, 0xec, 0xe0, 0x17, 0xe8, 0x81, 0x0d, 0xc0, 0x80, 0x0e, 0xfc, 0x00, 0x0c, 0xfc, 0x00, 0x05, 0x1c, 0x75, 0x73, 0x04, 0xa0, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87,
        0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0, 0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07,
        0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74,
        0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x27, 0x01, 0x02, 0x20,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f, 0x03, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x08, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x86, 0x3c, 0x13, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x2e, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x0a, 0xa3, 0x10, 0x0a, 0x82, 0xac, 0x11,
        0x00, 0xfa, 0x0a, 0x1c, 0x10, 0x10, 0x81, 0xbc, 0x19, 0x00, 0xda, 0x66, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44,
        0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a,
        0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98,
        0x20, 0x0c, 0xc9, 0x04, 0x61, 0x83, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xe0, 0x3d,
        0x13, 0x04, 0xab, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d, 0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48,
        0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3e, 0x67, 0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x18, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73,
        0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87,
        0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54,
        0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c,
        0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e,
        0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c,
        0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e,
        0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4,
        0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07,
        0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5,
        0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b,
        0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2,
        0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c,
        0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0xd8, 0xc0, 0x35,
        0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93,
        0x13, 0x11, 0x3e, 0x52, 0xeb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x0f, 0x20, 0x28,
        0xf8, 0xc5, 0xfb, 0x48, 0xad, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x34, 0x94, 0xec, 0x40, 0xc1, 0x0e, 0x94, 0x6e, 0x40, 0x59, 0x0a, 0x94, 0xed, 0x40, 0x61, 0x02, 0x15, 0xa6, 0x40, 0x11, 0x10, 0x52, 0x02, 0x23, 0x00, 0xe5, 0x41, 0xd2, 0x1c, 0x82, 0xb2, 0xcc,
        0x21, 0x58, 0x0c, 0x55, 0x73, 0x10, 0x49, 0x92, 0x64, 0x23, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x69, 0x8f, 0x70, 0x4d, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x6d, 0x90,
        0x80, 0x51, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x84, 0xc1, 0x93, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0xe4, 0x45, 0xc2, 0x31, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x4e, 0x18, 0x40,
        0x81, 0xb0, 0x2d, 0xd9, 0x68, 0x42, 0x00, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0x42, 0x06, 0x11, 0x13, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31,
        0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0x64, 0x06, 0xd9, 0xe2, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0x01, 0x84, 0x06, 0x55, 0xa0, 0x8c, 0x41, 0x41, 0x0c, 0x42, 0x44, 0x21, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_unpack_s8_s32[] =
    {
#if 0
        StructuredBuffer<int> RO : register(t0);
        RWStructuredBuffer<int4> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = unpack_s8s16(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x74, 0x74, 0xa4, 0x6a, 0x53, 0xcf, 0x07, 0xee, 0xa9, 0xee, 0xc4, 0xdd, 0x17, 0xea, 0x52, 0xa0, 0x01, 0x00, 0x00, 0x00, 0x90, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x61, 0x0d, 0xc7, 0x75, 0x21, 0x5d, 0xd4, 0x13, 0xbe, 0xb1, 0xd8,
        0xa0, 0x81, 0x96, 0x5a, 0x44, 0x58, 0x49, 0x4c, 0x84, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xa1, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x6c, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x98, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x80, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x47, 0x00, 0x1d, 0x34, 0x5c, 0xfe, 0x84,
        0x3d, 0x84, 0xe4, 0x17, 0x0e, 0x15, 0x21, 0xb5, 0x8e, 0x08, 0xc3, 0x30, 0x8c, 0x39, 0x02, 0xa4, 0x10, 0xc7, 0x80, 0x0c, 0x94, 0xdc, 0x34, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0xaf, 0x84, 0xb4,
        0x12, 0x93, 0x8f, 0xdc, 0x36, 0x2a, 0x86, 0x61, 0x18, 0x06, 0x42, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0x4c, 0x39, 0x96, 0xa1, 0x19, 0x06, 0x64,
        0x20, 0xa7, 0x2c, 0xc0, 0xd0, 0x0c, 0xc3, 0x30, 0x0c, 0x03, 0x32, 0x10, 0x74, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43, 0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13,
        0x42, 0x48, 0x48, 0x18, 0x86, 0x42, 0x34, 0x43, 0x23, 0xd1, 0x74, 0xd0, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x86, 0x34, 0x03, 0x22, 0x86, 0x61, 0x40, 0xe6, 0x08, 0x82, 0x52,
        0x34, 0x43, 0x35, 0x58, 0x74, 0x0d, 0x04, 0xcc, 0xb4, 0x8d, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03,
        0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3a, 0xb8, 0x03, 0x3d, 0xf8, 0x01, 0x0a, 0x06, 0xd2, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x34, 0x18, 0x07, 0x76, 0x08, 0x87,
        0x79, 0x98, 0x07, 0x37, 0x90, 0x85, 0x5b, 0x98, 0x05, 0x7a, 0x90, 0x87, 0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0,
        0x03, 0x7b, 0x28, 0x87, 0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e, 0x48, 0x07, 0x77, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14, 0x70, 0xd4, 0xcd, 0x11, 0x80, 0x02,
        0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d,
        0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10,
        0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78,
        0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x12, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x38, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xe7, 0x02,
        0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a,
        0x4a, 0xa0, 0x18, 0x46, 0x00, 0x0a, 0xa3, 0x10, 0x0a, 0x84, 0xac, 0x11, 0x00, 0xfa, 0x0a, 0x8c, 0xbc, 0x19, 0x00, 0xda, 0x66, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x61, 0x83, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xe0, 0x3d, 0x13, 0x04, 0xab, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d,
        0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3e, 0x67, 0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x18, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61,
        0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
        0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b,
        0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20,
        0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97,
        0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30,
        0x1b, 0xd2, 0x0c, 0x48, 0x63, 0xd8, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f,
        0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x4c, 0xc3,
        0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x0f, 0x20, 0x28, 0xf8, 0xc5, 0xfb, 0x48, 0xad, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x34, 0x94, 0xec, 0x40, 0xc1, 0x0e, 0x94, 0x6e, 0x40, 0x59, 0x0a, 0x94, 0xed, 0x40, 0x61, 0x02, 0x15, 0xa6, 0x00, 0x21, 0x25, 0x50, 0x1e, 0x24,
        0xcd, 0x21, 0x28, 0xcb, 0x1c, 0x82, 0xc5, 0x50, 0x35, 0x07, 0x91, 0x24, 0x09, 0x36, 0x02, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x61, 0x8d, 0x50, 0x45, 0x23, 0x06, 0x08, 0x00, 0x82,
        0x60, 0x70, 0x65, 0x8e, 0x60, 0x49, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x7c, 0xcd, 0x35, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x13, 0xf7, 0x08, 0xc7, 0x88, 0x81, 0x02, 0x80, 0x20, 0x18,
        0x38, 0x9f, 0x13, 0x08, 0x99, 0x72, 0x8d, 0x26, 0x04, 0xc0, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x24, 0x62, 0xf0, 0x30, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c,
        0x26, 0x10, 0xc3, 0x11, 0x83, 0x1c, 0x31, 0xc8, 0x11, 0x83, 0x1c, 0x31, 0xc8, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x68, 0xb0, 0x3d, 0xd2, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x40, 0x6a,
        0x70, 0x05, 0x4e, 0x19, 0x14, 0xc4, 0x20, 0x54, 0x63, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_unpack_s8_s16[] =
    {
#if 0
        StructuredBuffer<int> RO : register(t0);
        RWStructuredBuffer<int16_t4> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = unpack_s8s16(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0xf5, 0xcb, 0x38, 0xb3, 0xa0, 0x1d, 0xf5, 0x59, 0x42, 0xb4, 0xf4, 0xf0, 0x4d, 0x52, 0x3f, 0x36, 0x01, 0x00, 0x00, 0x00, 0x94, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xda, 0xda, 0x45, 0x03, 0x57, 0x37, 0x5e, 0x81, 0xbc, 0x90, 0x2c,
        0x39, 0x70, 0x89, 0xf6, 0x44, 0x58, 0x49, 0x4c, 0x88, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xa2, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x70, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x99, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x80, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x31, 0x47, 0x00, 0x1d, 0x34, 0x5c, 0xfe, 0x84,
        0x3d, 0x84, 0xe4, 0x17, 0x0e, 0x15, 0x21, 0xb5, 0x8e, 0x08, 0xc3, 0x30, 0x8c, 0x39, 0x02, 0xa4, 0x10, 0xc7, 0x80, 0x0c, 0x94, 0xdc, 0x34, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0xaf, 0x84, 0xb4,
        0x12, 0x93, 0x8f, 0xdc, 0x36, 0x2a, 0x86, 0x61, 0x18, 0x06, 0x42, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0x4c, 0x39, 0x96, 0xa1, 0x19, 0x06, 0x64,
        0x20, 0xa7, 0x2c, 0xc0, 0xd0, 0x0c, 0x83, 0x61, 0x18, 0x06, 0x32, 0x10, 0x74, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43, 0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13,
        0x42, 0x48, 0x48, 0x18, 0x86, 0x42, 0x34, 0x43, 0x23, 0xd1, 0x74, 0xd0, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x86, 0x34, 0x03, 0x22, 0x86, 0x61, 0x40, 0xe6, 0x08, 0x82, 0x52,
        0x34, 0x43, 0x35, 0x58, 0x74, 0x0d, 0x04, 0xcc, 0xb4, 0x8d, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03,
        0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3a, 0xb8, 0x03, 0x3d, 0xf8, 0x01, 0x0a, 0x06, 0xd2, 0x86, 0x11, 0x08, 0x63, 0xa6, 0x36, 0x18, 0x07, 0x76, 0x08, 0x87,
        0x79, 0x98, 0x07, 0x37, 0x90, 0x85, 0x5b, 0x98, 0x05, 0x7a, 0x90, 0x87, 0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0,
        0x03, 0x7b, 0x28, 0x87, 0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e, 0x48, 0x07, 0x77, 0xa0, 0x87, 0x38, 0xb0, 0x83, 0x5f, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14,
        0x70, 0xd4, 0xcd, 0x11, 0x80, 0x02, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0,
        0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d,
        0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06,
        0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43,
        0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x27, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f, 0x03, 0x04, 0x80, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x08, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x13, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c,
        0x79, 0x2e, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x0a, 0xa3, 0x10, 0x0a, 0x82, 0xac, 0x11, 0x00, 0xfa, 0x0a, 0x1c, 0x10, 0x10, 0x81, 0xbc, 0x19, 0x00, 0xda, 0x66, 0x00, 0x00, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99,
        0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84,
        0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x61, 0x83, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84,
        0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xe0, 0x3d, 0x13, 0x04, 0xab, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8,
        0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d, 0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3e, 0x67, 0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x18,
        0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6,
        0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60,
        0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6,
        0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19,
        0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13,
        0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0xd8, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc,
        0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x52, 0xeb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30,
        0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0x4f, 0x35, 0x0f, 0x20, 0x28, 0xf8, 0xc5, 0xfb, 0x48, 0xad, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x61, 0x20, 0x00, 0x00,
        0x2b, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x34, 0x94, 0xec, 0x40, 0xc1, 0x0e, 0x94, 0x6e, 0x40, 0x59, 0x0a, 0x94, 0xed, 0x40, 0x61, 0x02,
        0x15, 0xa6, 0x40, 0x11, 0x10, 0x52, 0x02, 0xe5, 0x41, 0xd2, 0x1c, 0x82, 0xb2, 0xcc, 0x21, 0x58, 0x0c, 0x55, 0x73, 0x10, 0x49, 0x92, 0x64, 0x23, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00,
        0x82, 0x60, 0x70, 0x65, 0x8e, 0x60, 0x49, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x70, 0x69, 0x8f, 0x70, 0x4d, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x80, 0x81, 0x83, 0x8d, 0x18, 0x1c, 0x00,
        0x08, 0x82, 0xc1, 0xd4, 0x41, 0xc2, 0x31, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x0e, 0x18, 0x3c, 0x81, 0xa0, 0x29, 0xd8, 0x68, 0x42, 0x00, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0x32, 0x06,
        0x10, 0x13, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0x54, 0x06, 0xd8, 0xe2, 0x8c, 0x18, 0x38, 0x00, 0x08,
        0x82, 0x01, 0x74, 0x06, 0x54, 0xa0, 0x88, 0x41, 0x41, 0x0c, 0x42, 0x34, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_pack_u32_u8[] =
    {
#if 0
        StructuredBuffer<uint4> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = pack_u8(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x9a, 0x52, 0x04, 0xec, 0x23, 0xc7, 0xfd, 0x2c, 0xd0, 0x55, 0x84, 0xb0, 0xce, 0x5e, 0xfa, 0x02, 0x01, 0x00, 0x00, 0x00, 0x88, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9c, 0x3e, 0x28, 0xfd, 0x32, 0x22, 0x44, 0x7d, 0x22, 0x25, 0xc7, 0x01,
        0x7b, 0x06, 0xf7, 0x2c, 0x44, 0x58, 0x49, 0x4c, 0x7c, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x9f, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x64, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x96, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x48, 0x39, 0xc6, 0x30, 0x63, 0x8c, 0x41,
        0xe7, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0x7c, 0xe4, 0xb6, 0x51, 0x31, 0xc6, 0x18, 0x83, 0xcc, 0x3d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48, 0x7e, 0x08, 0x34,
        0xc3, 0x42, 0xa0, 0x40, 0x95, 0x23, 0x8d, 0x35, 0x86, 0x19, 0xc4, 0xca, 0x02, 0xc6, 0x1a, 0x63, 0x8c, 0x31, 0xcc, 0x20, 0x77, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43,
        0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13, 0x42, 0x48, 0x48, 0x18, 0xa3, 0x10, 0x6b, 0x2c, 0x48, 0xf1, 0xa0, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x0d, 0x69, 0x06, 0x44,
        0x8c, 0x31, 0xcc, 0x1c, 0x41, 0x50, 0x8a, 0x35, 0xe6, 0xa0, 0x54, 0x07, 0x02, 0x86, 0x11, 0x88, 0x61, 0x26, 0x38, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x98, 0x05, 0x7a, 0x90,
        0x87, 0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x03, 0x7b, 0x28, 0x87, 0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e,
        0xa8, 0x07, 0x77, 0x98, 0x87, 0x74, 0x38, 0x07, 0x77, 0x28, 0x07, 0x72, 0x00, 0x83, 0x74, 0x70, 0x07, 0x7a, 0x60, 0x03, 0x30, 0xa0, 0x03, 0x3f, 0x00, 0x03, 0x3f, 0x40, 0x01, 0xa6, 0x3c, 0x93,
        0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98,
        0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03, 0x18, 0xa4, 0x83, 0x3b, 0xd0, 0x83, 0x1f, 0xa0, 0x60, 0xd0, 0x9e, 0x23, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d,
        0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10,
        0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78,
        0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x13, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02,
        0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a,
        0x25, 0x50, 0x0c, 0x23, 0x00, 0x85, 0x51, 0x20, 0x85, 0x40, 0x74, 0x04, 0x80, 0x7a, 0x81, 0x10, 0x9f, 0x01, 0x20, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x41, 0x83, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xc0, 0x3d, 0x13, 0x04, 0xaa, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d,
        0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3a, 0x67, 0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x24, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61,
        0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
        0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b,
        0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20,
        0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97,
        0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30,
        0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x98, 0x40, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x0f, 0x20, 0x28, 0xf8, 0xc5, 0xfb, 0xc8, 0x6d, 0xdb, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51,
        0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0x76, 0xa0, 0x60, 0x07, 0x4a, 0x37, 0xa0, 0x2c, 0x05, 0x0a, 0x77, 0xa0, 0x30, 0x81, 0x0a, 0x53, 0x80, 0x4c, 0x09, 0x94,
        0xc7, 0x08, 0x00, 0xc1, 0x39, 0x04, 0x65, 0x99, 0x43, 0xb8, 0x18, 0x9a, 0x73, 0x10, 0x49, 0x92, 0x64, 0x23, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x50, 0x69, 0x8e, 0x70, 0x4d, 0x23,
        0x06, 0x08, 0x00, 0x82, 0x60, 0x50, 0x6d, 0x8f, 0x80, 0x51, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x84, 0x81, 0x93, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0xe4, 0x41, 0xc2, 0x31, 0x62,
        0xa0, 0x00, 0x20, 0x08, 0x06, 0x4c, 0x18, 0x3c, 0x81, 0xb0, 0x29, 0xd8, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0x81, 0x02, 0x80, 0x20, 0x18,
        0x1c, 0x67, 0x30, 0x39, 0xc4, 0x20, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x10, 0x95, 0xc1, 0xa5, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xe0, 0x9c, 0xc1, 0x14, 0x24, 0x62, 0x20, 0x68,
        0x9a, 0x26, 0x7d, 0x08, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_pack_u16_u8[] =
    {
#if 0
        StructuredBuffer<uint16_t4> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = pack_u8(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x57, 0xfa, 0xaa, 0xf2, 0x31, 0x47, 0x9c, 0x54, 0xa5, 0x40, 0x99, 0x32, 0x18, 0x2e, 0x55, 0x38, 0x01, 0x00, 0x00, 0x00, 0x90, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0xb1, 0x0f, 0x35, 0xd0, 0x80, 0x19, 0x97, 0xc8, 0x0f, 0xd4, 0x38,
        0xdf, 0xd7, 0xa3, 0x16, 0x44, 0x58, 0x49, 0x4c, 0x84, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xa1, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x6c, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x98, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0xc8, 0x1c, 0x01, 0x54, 0x8e, 0x31, 0xcc,
        0x39, 0xe7, 0x10, 0xba, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x25, 0x26, 0x1f, 0xa9, 0x75, 0x54, 0x9c, 0x73, 0xce, 0x20, 0x73, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92,
        0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x58, 0xe5, 0x50, 0x83, 0x8d, 0x61, 0x06, 0xb5, 0xb2, 0x80, 0xc1, 0xc6, 0x18, 0x63, 0x0c, 0x33, 0xe8, 0xdd, 0x36, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0xaf,
        0x84, 0xe4, 0x50, 0x91, 0x40, 0xa4, 0x91, 0xf3, 0x10, 0xd1, 0x84, 0x10, 0x12, 0x12, 0xc6, 0x28, 0x04, 0x1b, 0x2c, 0x92, 0x3c, 0x68, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x43,
        0x9a, 0x01, 0x11, 0x63, 0x0c, 0x33, 0x47, 0x10, 0x94, 0x82, 0x0d, 0x3a, 0x2a, 0xd9, 0x81, 0x80, 0x61, 0x04, 0xe2, 0x98, 0x89, 0x0d, 0xc6, 0x81, 0x1d, 0xc2, 0x61, 0x1e, 0xe6, 0xc1, 0x0d, 0x66,
        0x81, 0x1e, 0xe4, 0xa1, 0x1e, 0xc6, 0x81, 0x1e, 0xea, 0x41, 0x1e, 0xca, 0x81, 0x1c, 0x44, 0xa1, 0x1e, 0xcc, 0xc1, 0x1c, 0xca, 0x41, 0x1e, 0xf8, 0xc0, 0x1e, 0xca, 0x61, 0x1c, 0xe8, 0xe1, 0x1d,
        0xe4, 0x81, 0x0f, 0xea, 0x21, 0x1d, 0xdc, 0x81, 0x1e, 0xe2, 0xc0, 0x0e, 0x7e, 0x81, 0x1e, 0xd8, 0x00, 0x0c, 0xe8, 0xc0, 0x0f, 0xc0, 0xc0, 0x0f, 0x50, 0x90, 0x49, 0xcf, 0x64, 0x06, 0xe3, 0xc0,
        0x0e, 0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb2, 0x70, 0x0b, 0xb3, 0x40, 0x0f, 0xf2, 0x50, 0x0f, 0xe3, 0x40, 0x0f, 0xf5, 0x20, 0x0f, 0xe5, 0x40, 0x0e, 0xa2, 0x50, 0x0f, 0xe6, 0x60, 0x0e, 0xe5,
        0x20, 0x0f, 0x7c, 0x50, 0x0f, 0xee, 0x30, 0x0f, 0xe9, 0x70, 0x0e, 0xee, 0x50, 0x0e, 0xe4, 0x00, 0x06, 0xe9, 0xe0, 0x0e, 0xf4, 0xe0, 0x07, 0x28, 0x18, 0xc4, 0xe7, 0x08, 0x40, 0x01, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0, 0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71,
        0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07,
        0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x90, 0x07, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x03, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x07,
        0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x2c, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23,
        0x00, 0x85, 0x51, 0x10, 0x85, 0x40, 0x75, 0x04, 0x80, 0x7c, 0x81, 0x03, 0x02, 0x22, 0x50, 0x9f, 0x01, 0xa0, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x61, 0x83, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xd0, 0x3d, 0x13, 0x84, 0xaa, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d,
        0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3c, 0x67, 0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x24, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61,
        0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
        0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b,
        0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20,
        0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97,
        0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30,
        0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x98, 0x40, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x0f, 0x20, 0x28, 0xf8, 0xc5, 0xfb, 0x48, 0xad, 0xdb, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51,
        0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xd4, 0xba, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0x76, 0xa0, 0x60, 0x07, 0x4a, 0x37, 0xa0, 0x2c, 0x05, 0x8a, 0xa0, 0x70, 0x07, 0x0a, 0x13, 0xa8, 0x30, 0x05, 0xc8, 0x94,
        0x40, 0x79, 0x8c, 0x00, 0x50, 0x9c, 0x43, 0x50, 0x96, 0x39, 0x04, 0x8c, 0x21, 0x3a, 0x07, 0x91, 0x24, 0x89, 0x36, 0x02, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x6d,
        0x8f, 0x80, 0x51, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x71, 0x90, 0x90, 0x55, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x88, 0xc1, 0xa3, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0xf4,
        0x45, 0xc2, 0x31, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x8d, 0x18, 0x40, 0x81, 0xc0, 0x29, 0xcf, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0x81,
        0x02, 0x80, 0x20, 0x18, 0x20, 0x68, 0x30, 0x39, 0xc4, 0x20, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0x99, 0x01, 0xa6, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xf0, 0xa0, 0xc1, 0x14,
        0x24, 0x63, 0x20, 0x6c, 0xdb, 0x26, 0x81, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_pack_s32_u8_clamp[] =
    {
#if 0
        StructuredBuffer<int4> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = pack_clamp_u8(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x0c, 0xdd, 0xca, 0xbc, 0x8a, 0xb7, 0x82, 0xbb, 0xb6, 0xe4, 0x37, 0x04, 0x21, 0x95, 0xef, 0x02, 0x01, 0x00, 0x00, 0x00, 0x78, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x7c, 0x1a, 0xa0, 0x4c, 0x0e, 0x2d, 0x91, 0x8b, 0x90, 0xef, 0xb0,
        0xdc, 0xf1, 0x0d, 0x55, 0x44, 0x58, 0x49, 0x4c, 0x6c, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x9b, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x54, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x92, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x48, 0x39, 0xc6, 0x30, 0x63, 0x8c, 0x41,
        0xe7, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0x7c, 0xe4, 0xb6, 0x51, 0x31, 0xc6, 0x18, 0x83, 0xcc, 0x3d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48, 0x7e, 0x08, 0x34,
        0xc3, 0x42, 0xa0, 0x40, 0x95, 0x23, 0x8d, 0x35, 0x86, 0x19, 0xc4, 0xca, 0x02, 0xc6, 0x1a, 0x63, 0x8c, 0x31, 0xcc, 0x20, 0x77, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43,
        0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13, 0x42, 0x48, 0x48, 0x18, 0xa3, 0x10, 0x6b, 0x2c, 0x48, 0xf1, 0xa0, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x0d, 0x69, 0x06, 0x44,
        0x8c, 0x31, 0xcc, 0x1c, 0x41, 0x50, 0x8a, 0x35, 0xe6, 0xa0, 0x54, 0x07, 0x02, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x33, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x98, 0x05, 0x7a, 0x90,
        0x87, 0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x03, 0x7b, 0x28, 0x87, 0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e,
        0x48, 0x07, 0x77, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14, 0x60, 0xca, 0x33, 0x99, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x81, 0x2c, 0xdc, 0xc2,
        0x2c, 0xd0, 0x83, 0x3c, 0xd4, 0xc3, 0x38, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0x90, 0x83, 0x28, 0xd4, 0x83, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x03, 0x1f, 0xd4, 0x83, 0x3b, 0xcc, 0x43, 0x3a, 0x9c,
        0x83, 0x3b, 0x94, 0x03, 0x39, 0x80, 0x41, 0x3a, 0xb8, 0x03, 0x3d, 0xf8, 0x01, 0x0a, 0x06, 0xed, 0x39, 0x02, 0x50, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79,
        0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60,
        0xc8, 0x13, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0a, 0x00, 0x00, 0x00,
        0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23, 0x00, 0x85, 0x51, 0x20, 0x85, 0x40, 0x74, 0x04, 0x80, 0x7a, 0x81, 0x10,
        0x9f, 0x01, 0x20, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1,
        0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73,
        0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x41, 0x83, 0x08, 0x4c,
        0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xc0, 0x3d, 0x13, 0x04, 0xaa, 0x99, 0x20, 0x0c, 0xcc, 0x06,
        0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d, 0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3a, 0x67,
        0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x24, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c,
        0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c,
        0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4,
        0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f,
        0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37,
        0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x98, 0x40, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x0f, 0x20, 0x28, 0xf8,
        0xc5, 0xfb, 0xc8, 0x6d, 0xdb, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01,
        0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80,
        0x34, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0x76, 0xa0, 0x60, 0x07,
        0x4a, 0x37, 0xa0, 0x2c, 0x05, 0x0a, 0x77, 0xa0, 0x30, 0x81, 0x0a, 0x53, 0x80, 0x4c, 0x09, 0x94, 0x07, 0xc1, 0x39, 0x04, 0x65, 0x99, 0x43, 0xb8, 0x18, 0x9a, 0x73, 0x10, 0x49, 0x92, 0x64, 0x23,
        0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x50, 0x65, 0x8d, 0x60, 0x49, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x50, 0x69, 0x8e, 0x70, 0x4d, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60,
        0x40, 0x80, 0x41, 0x83, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0xd4, 0x3d, 0xc2, 0x31, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x0c, 0x18, 0x38, 0x81, 0xa0, 0x25, 0xd7, 0x68, 0x42, 0x00, 0x8c,
        0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0x81, 0x02, 0x80, 0x20, 0x18, 0x1c, 0x66, 0x20, 0x3d, 0xc4, 0x20, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x10, 0x91, 0x81,
        0xa5, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xe0, 0x98, 0x81, 0x14, 0x24, 0x61, 0x20, 0x64, 0x59, 0x16, 0x79, 0x08, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_pack_s32_s8_clamp[] =
    {
#if 0
        StructuredBuffer<int4> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = pack_clamp_s8(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x84, 0x73, 0x3b, 0xa2, 0x4d, 0xdc, 0x2b, 0x70, 0x5e, 0xa3, 0x54, 0x0c, 0x9c, 0xbb, 0x65, 0x4a, 0x01, 0x00, 0x00, 0x00, 0x78, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x71, 0x65, 0x01, 0xa3, 0xca, 0xcb, 0xf6, 0x35, 0xaf, 0xc0, 0xa9, 0x16,
        0x71, 0x14, 0x71, 0x87, 0x44, 0x58, 0x49, 0x4c, 0x6c, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x9b, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x54, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x92, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x48, 0x39, 0xc6, 0x30, 0x63, 0x8c, 0x41,
        0xe7, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0x7c, 0xe4, 0xb6, 0x51, 0x31, 0xc6, 0x18, 0x83, 0xcc, 0x3d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48, 0x7e, 0x08, 0x34,
        0xc3, 0x42, 0xa0, 0x40, 0x95, 0x23, 0x8d, 0x35, 0x86, 0x19, 0xc4, 0xca, 0x02, 0xc6, 0x1a, 0x63, 0x8c, 0x31, 0xcc, 0x20, 0x77, 0xdb, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0x92, 0x43,
        0x45, 0x02, 0x91, 0x46, 0xce, 0x43, 0x44, 0x13, 0x42, 0x48, 0x48, 0x18, 0xa3, 0x10, 0x6b, 0x2c, 0x48, 0xf1, 0xa0, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x0d, 0x69, 0x06, 0x44,
        0x8c, 0x31, 0xcc, 0x1c, 0x41, 0x50, 0x8a, 0x35, 0xe6, 0xa0, 0x54, 0x07, 0x02, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x33, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x98, 0x05, 0x7a, 0x90,
        0x87, 0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72, 0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x03, 0x7b, 0x28, 0x87, 0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e,
        0x48, 0x07, 0x77, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x14, 0x60, 0xca, 0x33, 0x99, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x81, 0x2c, 0xdc, 0xc2,
        0x2c, 0xd0, 0x83, 0x3c, 0xd4, 0xc3, 0x38, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0x90, 0x83, 0x28, 0xd4, 0x83, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x03, 0x1f, 0xd4, 0x83, 0x3b, 0xcc, 0x43, 0x3a, 0x9c,
        0x83, 0x3b, 0x94, 0x03, 0x39, 0x80, 0x41, 0x3a, 0xb8, 0x03, 0x3d, 0xf8, 0x01, 0x0a, 0x06, 0xed, 0x39, 0x02, 0x50, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79,
        0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x0e, 0x20, 0x00, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60,
        0xc8, 0x13, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0a, 0x00, 0x00, 0x00,
        0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23, 0x00, 0x85, 0x51, 0x20, 0x85, 0x40, 0x74, 0x04, 0x80, 0x7a, 0x81, 0x10,
        0x9f, 0x01, 0x20, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1,
        0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73,
        0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x41, 0x83, 0x08, 0x4c,
        0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xc0, 0x3d, 0x13, 0x04, 0xaa, 0x99, 0x20, 0x0c, 0xcc, 0x06,
        0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d, 0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3a, 0x67,
        0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x24, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c,
        0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c,
        0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4,
        0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f,
        0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37,
        0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x98, 0x40, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x0f, 0x20, 0x28, 0xf8,
        0xc5, 0xfb, 0xc8, 0x6d, 0xdb, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01,
        0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80,
        0x34, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0x76, 0xa0, 0x60, 0x07,
        0x4a, 0x37, 0xa0, 0x2c, 0x05, 0x0a, 0x77, 0xa0, 0x30, 0x81, 0x0a, 0x53, 0x80, 0x4c, 0x09, 0x94, 0x47, 0x11, 0x10, 0x9c, 0x43, 0x50, 0x96, 0x39, 0x84, 0x8b, 0xa1, 0x39, 0x07, 0x91, 0x24, 0x49,
        0x36, 0x02, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x50, 0x69, 0x8e, 0x70, 0x4d, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x50, 0x6d, 0x8f, 0x80, 0x51, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60,
        0x40, 0x84, 0x81, 0x93, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0xe4, 0x41, 0xc2, 0x31, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x4c, 0x18, 0x3c, 0x81, 0xb0, 0x29, 0xd8, 0x68, 0x42, 0x00, 0x8c,
        0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0x81, 0x02, 0x80, 0x20, 0x18, 0x1c, 0x67, 0x30, 0x39, 0xc4, 0x20, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x10, 0x95, 0xc1,
        0xa5, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xe0, 0x9c, 0xc1, 0x14, 0x24, 0x62, 0x20, 0x68, 0x9a, 0x26, 0x7d, 0x08, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_pack_s16_u8_clamp[] =
    {
#if 0
        StructuredBuffer<int16_t4> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = pack_clamp_u8(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x8d, 0x58, 0x81, 0x65, 0xda, 0x19, 0xe9, 0x54, 0x70, 0x12, 0x06, 0x1a, 0x8d, 0x64, 0xcb, 0xff, 0x01, 0x00, 0x00, 0x00, 0x88, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0x43, 0x1c, 0x8a, 0xd2, 0x69, 0x35, 0x7a, 0x2c, 0xbf, 0xe4, 0x4b,
        0x28, 0x3c, 0x3f, 0x62, 0x44, 0x58, 0x49, 0x4c, 0x7c, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x9f, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x64, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x96, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0xc8, 0x1c, 0x01, 0x54, 0x8e, 0x31, 0xcc,
        0x39, 0xe7, 0x10, 0xba, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x25, 0x26, 0x1f, 0xa9, 0x75, 0x54, 0x9c, 0x73, 0xce, 0x20, 0x73, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92,
        0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x58, 0xe5, 0x50, 0x83, 0x8d, 0x61, 0x06, 0xb5, 0xb2, 0x80, 0xc1, 0xc6, 0x18, 0x63, 0x0c, 0x33, 0xe8, 0xdd, 0x36, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0xaf,
        0x84, 0xe4, 0x50, 0x91, 0x40, 0xa4, 0x91, 0xf3, 0x10, 0xd1, 0x84, 0x10, 0x12, 0x12, 0xc6, 0x28, 0x04, 0x1b, 0x2c, 0x92, 0x3c, 0x68, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x43,
        0x9a, 0x01, 0x11, 0x63, 0x0c, 0x33, 0x47, 0x10, 0x94, 0x82, 0x0d, 0x3a, 0x2a, 0xd9, 0x81, 0x80, 0x61, 0x04, 0xe2, 0x98, 0x69, 0x0d, 0xc6, 0x81, 0x1d, 0xc2, 0x61, 0x1e, 0xe6, 0xc1, 0x0d, 0x66,
        0x81, 0x1e, 0xe4, 0xa1, 0x1e, 0xc6, 0x81, 0x1e, 0xea, 0x41, 0x1e, 0xca, 0x81, 0x1c, 0x44, 0xa1, 0x1e, 0xcc, 0xc1, 0x1c, 0xca, 0x41, 0x1e, 0xf8, 0xc0, 0x1e, 0xca, 0x61, 0x1c, 0xe8, 0xe1, 0x1d,
        0xe4, 0x81, 0x0f, 0xd2, 0xc1, 0x1d, 0xe8, 0x21, 0x0e, 0xec, 0xe0, 0x17, 0xe8, 0x81, 0x0d, 0xc0, 0x80, 0x0e, 0xfc, 0x00, 0x0c, 0xfc, 0x00, 0x05, 0x99, 0xf4, 0x4c, 0x66, 0x30, 0x0e, 0xec, 0x10,
        0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x20, 0x0b, 0xb7, 0x30, 0x0b, 0xf4, 0x20, 0x0f, 0xf5, 0x30, 0x0e, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0xe4, 0x20, 0x0a, 0xf5, 0x60, 0x0e, 0xe6, 0x50, 0x0e, 0xf2,
        0xc0, 0x07, 0xf5, 0xe0, 0x0e, 0xf3, 0x90, 0x0e, 0xe7, 0xe0, 0x0e, 0xe5, 0x40, 0x0e, 0x60, 0x90, 0x0e, 0xee, 0x40, 0x0f, 0x7e, 0x80, 0x82, 0x41, 0x7c, 0x8e, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0, 0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71,
        0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07,
        0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x90, 0x07, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x03, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x07,
        0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x2c, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23,
        0x00, 0x85, 0x51, 0x10, 0x85, 0x40, 0x75, 0x04, 0x80, 0x7c, 0x81, 0x03, 0x02, 0x22, 0x50, 0x9f, 0x01, 0xa0, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x61, 0x83, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xd0, 0x3d, 0x13, 0x84, 0xaa, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d,
        0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3c, 0x67, 0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x24, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61,
        0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
        0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b,
        0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20,
        0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97,
        0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30,
        0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x98, 0x40, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x0f, 0x20, 0x28, 0xf8, 0xc5, 0xfb, 0x48, 0xad, 0xdb, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51,
        0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xd4, 0xba, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0x76, 0xa0, 0x60, 0x07, 0x4a, 0x37, 0xa0, 0x2c, 0x05, 0x8a, 0xa0, 0x70, 0x07, 0x0a, 0x13, 0xa8, 0x30, 0x05, 0xc8, 0x94,
        0x40, 0x79, 0x50, 0x9c, 0x43, 0x50, 0x96, 0x39, 0x04, 0x8c, 0x21, 0x3a, 0x07, 0x91, 0x24, 0x89, 0x36, 0x02, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x69, 0x8e, 0x70, 0x4d, 0x23,
        0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x6d, 0x8f, 0x80, 0x51, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x84, 0x81, 0x93, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0xe4, 0x41, 0xc2, 0x31, 0x62,
        0xa0, 0x00, 0x20, 0x08, 0x06, 0x4d, 0x18, 0x3c, 0x81, 0xb0, 0x25, 0xce, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0x81, 0x02, 0x80, 0x20, 0x18,
        0x20, 0x67, 0x20, 0x3d, 0xc4, 0x20, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0x95, 0xc1, 0xa5, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xf0, 0x9c, 0x81, 0x14, 0x24, 0x62, 0x20, 0x68,
        0x9a, 0x16, 0x7d, 0x08, 0x00, 0x00, 0x00, 0x00,
    };
    static const BYTE cs_code_pack_s16_s8_clamp[] =
    {
#if 0
        StructuredBuffer<int16_t4> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(1, 1, 1)]
        void main(uint thr : SV_DispatchThreadID)
        {
                RW[thr] = pack_clamp_s8(RO[thr]);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x58, 0x68, 0xab, 0xb7, 0x27, 0xaa, 0xc8, 0x7e, 0x74, 0x4a, 0x3f, 0x48, 0xb8, 0x0b, 0x48, 0xbd, 0x01, 0x00, 0x00, 0x00, 0x90, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x6b, 0x46, 0xf6, 0x1a, 0xc3, 0x1b, 0x89, 0xa8, 0x2c, 0xfa, 0xf6,
        0xd8, 0x8a, 0xcb, 0x15, 0x44, 0x58, 0x49, 0x4c, 0x84, 0x06, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0xa1, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x6c, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x98, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0xc8, 0x1c, 0x01, 0x54, 0x8e, 0x31, 0xcc,
        0x39, 0xe7, 0x10, 0xba, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x25, 0x26, 0x1f, 0xa9, 0x75, 0x54, 0x9c, 0x73, 0xce, 0x20, 0x73, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92,
        0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x58, 0xe5, 0x50, 0x83, 0x8d, 0x61, 0x06, 0xb5, 0xb2, 0x80, 0xc1, 0xc6, 0x18, 0x63, 0x0c, 0x33, 0xe8, 0xdd, 0x36, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0xaf,
        0x84, 0xe4, 0x50, 0x91, 0x40, 0xa4, 0x91, 0xf3, 0x10, 0xd1, 0x84, 0x10, 0x12, 0x12, 0xc6, 0x28, 0x04, 0x1b, 0x2c, 0x92, 0x3c, 0x68, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x43,
        0x9a, 0x01, 0x11, 0x63, 0x0c, 0x33, 0x47, 0x10, 0x94, 0x82, 0x0d, 0x3a, 0x2a, 0xd9, 0x81, 0x80, 0x61, 0x04, 0xe2, 0x98, 0x69, 0x0d, 0xc6, 0x81, 0x1d, 0xc2, 0x61, 0x1e, 0xe6, 0xc1, 0x0d, 0x66,
        0x81, 0x1e, 0xe4, 0xa1, 0x1e, 0xc6, 0x81, 0x1e, 0xea, 0x41, 0x1e, 0xca, 0x81, 0x1c, 0x44, 0xa1, 0x1e, 0xcc, 0xc1, 0x1c, 0xca, 0x41, 0x1e, 0xf8, 0xc0, 0x1e, 0xca, 0x61, 0x1c, 0xe8, 0xe1, 0x1d,
        0xe4, 0x81, 0x0f, 0xd2, 0xc1, 0x1d, 0xe8, 0x21, 0x0e, 0xec, 0xe0, 0x17, 0xe8, 0x81, 0x0d, 0xc0, 0x80, 0x0e, 0xfc, 0x00, 0x0c, 0xfc, 0x00, 0x05, 0x99, 0xf4, 0x4c, 0x66, 0x30, 0x0e, 0xec, 0x10,
        0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x20, 0x0b, 0xb7, 0x30, 0x0b, 0xf4, 0x20, 0x0f, 0xf5, 0x30, 0x0e, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0xe4, 0x20, 0x0a, 0xf5, 0x60, 0x0e, 0xe6, 0x50, 0x0e, 0xf2,
        0xc0, 0x07, 0xf5, 0xe0, 0x0e, 0xf3, 0x90, 0x0e, 0xe7, 0xe0, 0x0e, 0xe5, 0x40, 0x0e, 0x60, 0x90, 0x0e, 0xee, 0x40, 0x0f, 0x7e, 0x80, 0x82, 0x41, 0x7c, 0x8e, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0, 0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71,
        0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07,
        0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x90, 0x07, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x03, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x07,
        0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x2c, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0c, 0x23,
        0x00, 0x85, 0x51, 0x10, 0x85, 0x40, 0x75, 0x04, 0x80, 0x7c, 0x81, 0x03, 0x02, 0x22, 0x50, 0x9f, 0x01, 0xa0, 0x3d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b,
        0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20,
        0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xc9, 0x04, 0x61, 0x83, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x84, 0x81, 0xd9, 0x90, 0x20, 0x89, 0x82, 0x20, 0xc3,
        0x82, 0x34, 0x1b, 0x02, 0x67, 0x82, 0xd0, 0x3d, 0x13, 0x84, 0xaa, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x90, 0x36, 0x2c, 0x08, 0xa4, 0x20, 0xc8, 0xb0, 0x44, 0x51, 0x34, 0x6d, 0x08, 0xa8, 0x0d,
        0xc4, 0x53, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x3c, 0x67, 0xc3, 0x30, 0x0c, 0xc3, 0x06, 0x02, 0xc9, 0x24, 0x6d, 0x43, 0x71, 0x61, 0x80, 0xb5, 0x55, 0x61,
        0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
        0x36, 0xbb, 0x32, 0xb9, 0x29, 0x01, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x60, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b,
        0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20,
        0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0x40, 0x0d, 0x97,
        0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30,
        0x1b, 0xd2, 0x0c, 0x48, 0x63, 0x98, 0x40, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x0f, 0x20, 0x28, 0xf8, 0xc5, 0xfb, 0x48, 0xad, 0xdb, 0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51,
        0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xd4, 0xba, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0x76, 0xa0, 0x60, 0x07, 0x4a, 0x37, 0xa0, 0x2c, 0x05, 0x8a, 0xa0, 0x70, 0x07, 0x0a, 0x13, 0xa8, 0x30, 0x05, 0xc8, 0x94,
        0x40, 0x79, 0x14, 0x01, 0xc5, 0x39, 0x04, 0x65, 0x99, 0x43, 0xc0, 0x18, 0xa2, 0x73, 0x10, 0x49, 0x92, 0x68, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x6d,
        0x8f, 0x80, 0x51, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x60, 0x71, 0x90, 0x90, 0x55, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x88, 0xc1, 0xa3, 0x8d, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0xf4,
        0x45, 0xc2, 0x31, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x8d, 0x18, 0x40, 0x81, 0xc0, 0x29, 0xcf, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0x81,
        0x02, 0x80, 0x20, 0x18, 0x20, 0x68, 0x30, 0x39, 0xc4, 0x20, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0x99, 0x01, 0xa6, 0x34, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xf0, 0xa0, 0xc1, 0x14,
        0x24, 0x63, 0x20, 0x6c, 0xdb, 0x26, 0x81, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

#define PACK_SHADER_BYTECODE(x) static const struct D3D12_SHADER_BYTECODE cs_##x = SHADER_BYTECODE(cs_code_##x)
    PACK_SHADER_BYTECODE(unpack_u8_u32);
    PACK_SHADER_BYTECODE(unpack_u8_u16);
    PACK_SHADER_BYTECODE(unpack_s8_s32);
    PACK_SHADER_BYTECODE(unpack_s8_s16);
    PACK_SHADER_BYTECODE(pack_u32_u8);
    PACK_SHADER_BYTECODE(pack_u16_u8);
    PACK_SHADER_BYTECODE(pack_s32_u8_clamp);
    PACK_SHADER_BYTECODE(pack_s32_s8_clamp);
    PACK_SHADER_BYTECODE(pack_s16_u8_clamp);
    PACK_SHADER_BYTECODE(pack_s16_s8_clamp);

    struct test
    {
        const struct D3D12_SHADER_BYTECODE *cs;
        uint32_t input[4];
        uint32_t output[4];
        bool requires_16bit;
    };

    static const struct test tests[] =
    {
        { &cs_unpack_u8_u32, { 0xfedba917 }, { 0x17, 0xa9, 0xdb, 0xfe }, false },
        { &cs_unpack_u8_u16, { 0xfedba917 }, { 0x00a90017, 0x00fe00db }, true },
        { &cs_unpack_s8_s32, { 0xfedba917 }, { 0x17, 0xffffffa9, 0xffffffdb, 0xfffffffe }, false },
        { &cs_unpack_s8_s16, { 0xfedba917 }, { 0xffa90017, 0xfffeffdb }, true },
        { &cs_pack_u32_u8, { 0x101, 0x02, 0xfffffffe, 0x10005 }, { 0x05fe0201 }, false },
        { &cs_pack_u16_u8, { 0x00020101, 0x0005fffe }, { 0x05fe0201 }, true },
        { &cs_pack_s32_u8_clamp, { 0xff, 0x100, 0xffffffff, 0 }, { 0x0000ffff }, false },
        { &cs_pack_s32_s8_clamp, { 0xff, 0x100, 0xffffffff, 0xffff8000 }, { 0x80ff7f7f }, false },
        { &cs_pack_s16_u8_clamp, { 0x010000ff, 0x0000ffff }, { 0x0000ffff }, true },
        { &cs_pack_s16_s8_clamp, { 0x010000ff, 0x8000ffff }, { 0x80ff7f7f }, true },
    };

    if (!init_compute_test_context(&context))
        return;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&shader_model, 0, sizeof(shader_model));
    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    if (FAILED(ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model))) ||
            shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_6)
    {
        skip("Device does not support SM 6.6.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&rs_desc, 0, sizeof(rs_desc));
    rs_desc.NumParameters = ARRAY_SIZE(root_parameters);
    rs_desc.pParameters = root_parameters;
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    create_root_signature(context.device, &rs_desc, &context.root_signature);

    memset(&features4, 0, sizeof(features4));
    ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS4, &features4, sizeof(features4));

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);

        if (tests[i].requires_16bit && !features4.Native16BitShaderOpsSupported)
        {
            skip("Skipping unsupported test for 16-bit native operations.\n");
            continue;
        }

        pso = create_compute_pipeline_state(context.device, context.root_signature, *tests[i].cs);
        src = create_upload_buffer(context.device, sizeof(tests[i].input), tests[i].input);
        dst = create_default_buffer(context.device, sizeof(tests[i].output), D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, context.root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
        ID3D12GraphicsCommandList_SetComputeRootShaderResourceView(context.list, 0, ID3D12Resource_GetGPUVirtualAddress(src));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, ID3D12Resource_GetGPUVirtualAddress(dst));
        ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);

        transition_resource_state(context.list, dst, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
        get_buffer_readback_with_command_list(dst, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

        for (j = 0; j < 4; j++)
        {
            value = get_readback_uint(&rb, j, 0, 0);
            ok(value == tests[i].output[j], "Value %u mismatch -> 0x%x != 0x%x\n", j, value, tests[i].output[j]);
        }

        ID3D12Resource_Release(src);
        ID3D12Resource_Release(dst);
        ID3D12PipelineState_Release(pso);
        release_resource_readback(&rb);
        reset_command_list(context.list, context.allocator);
    }

    vkd3d_test_set_context(NULL);
    destroy_test_context(&context);
}

void test_shader_sm64_packed(void)
{
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_FEATURE_DATA_D3D12_OPTIONS4 features4;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    struct test_context context;
    struct resource_readback rb;
    ID3D12PipelineState *pso;
    ID3D12Resource *src;
    ID3D12Resource *dst;
    uint32_t value;
    unsigned int i;

#if 0
    /* ./dxc -Tcs_6_4 -Fo test.dxil test.comp -Qstrip_reflect -Qstrip_debug -E main_dot2add -enable-16bit-types */
    RWStructuredBuffer<uint> RW : register(u0);

    struct Inputs { uint a, b, acc; };
    StructuredBuffer<Inputs> RO : register(t0);

    [numthreads(1, 1, 1)]
    void main_u8dot(uint3 id : SV_DispatchThreadID)
    {
        Inputs v = RO[id.x];
        uint acc = dot4add_u8packed(v.a, v.b, v.acc);
        RW[id.x] = acc;
    }

    [numthreads(1, 1, 1)]
    void main_i8dot(uint3 id : SV_DispatchThreadID)
    {
        Inputs v = RO[id.x];
        uint acc = dot4add_i8packed(v.a, v.b, v.acc);
        RW[id.x] = acc;
    }

    half2 ashalf2(uint v)
    {
        // Apparently there is no ashalf(), sigh ...
        return half2(f16tof32(uint16_t(v)), f16tof32(uint16_t(v >> 16)));
    }

    uint asuint(half2 v)
    {
        return uint(asuint16(v.x)) | (uint(asuint16(v.y)) << 16);
    }

    [numthreads(1, 1, 1)]
    void main_dot2add(uint3 id : SV_DispatchThreadID)
    {
        Inputs v = RO[id.x];
        precise float result = dot2add(ashalf2(v.a), ashalf2(v.b), asfloat(v.acc));
        uint acc = asuint(result);
        RW[id.x] = acc;
    }
#endif

    static const BYTE cs_code_u8dot[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x0c, 0x2d, 0x8d, 0x38, 0xe0, 0x83, 0xb3, 0x9d, 0x82, 0xda, 0x35, 0xb8, 0x34, 0xd3, 0xad, 0x54, 0x01, 0x00, 0x00, 0x00, 0xe4, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x5c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x14, 0x06, 0x00, 0x00, 0x64, 0x00, 0x05, 0x00, 0x85, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x04, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0xfc, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x7c, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x68, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x37, 0x0d, 0x97, 0x3f, 0x61,
        0x0f, 0x21, 0xf9, 0x2b, 0x21, 0xad, 0xc4, 0xe4, 0x23, 0xb7, 0x8d, 0x8a, 0x31, 0xc6, 0x18, 0x73, 0x04, 0x08, 0x9d, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40,
        0x01, 0x2a, 0x87, 0x19, 0x69, 0x8c, 0x33, 0x48, 0x95, 0x62, 0x8c, 0x31, 0x06, 0xb1, 0xb2, 0x80, 0x91, 0xc6, 0x18, 0x63, 0x8c, 0x33, 0xc8, 0xcd, 0x11, 0x04, 0xc5, 0x48, 0xe3, 0x8c, 0x01, 0x29,
        0x0e, 0x04, 0x5c, 0x23, 0x4d, 0x11, 0x25, 0x4c, 0xbe, 0xd8, 0x3c, 0xd4, 0x24, 0xa1, 0x61, 0x8c, 0x31, 0x93, 0x37, 0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x30, 0x0b, 0xf4, 0x20, 0x0f,
        0xf5, 0x30, 0x0e, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0xe4, 0x20, 0x0a, 0xf5, 0x60, 0x0e, 0xe6, 0x50, 0x0e, 0xf2, 0xc0, 0x07, 0xa9, 0xe0, 0x0e, 0xf0, 0x50, 0x0f, 0xf4, 0x30, 0x0f, 0x7e, 0x80,
        0x02, 0x4a, 0x75, 0x26, 0x33, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x90, 0x85, 0x5b, 0x98, 0x05, 0x7a, 0x90, 0x87, 0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72,
        0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x83, 0x7a, 0x70, 0x87, 0x79, 0x48, 0x87, 0x73, 0x70, 0x87, 0x72, 0x20, 0x07, 0x30, 0x48, 0x07, 0x77, 0xa0, 0x07, 0x3f, 0x40, 0xc1,
        0xa0, 0x3b, 0x47, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0,
        0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d,
        0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06,
        0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43,
        0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x47, 0x01, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x03, 0x04, 0x40, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x07, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x11, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64,
        0x81, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x08, 0xc5, 0x30, 0x02, 0x50, 0x18,
        0x04, 0x47, 0x00, 0x28, 0x17, 0x28, 0x20, 0x20, 0x02, 0xe1, 0x19, 0x00, 0xb2, 0x33, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4,
        0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x63, 0x82, 0x30, 0x1c, 0x1b, 0x84, 0x81, 0xa0, 0x30,
        0x36, 0x37, 0x41, 0x18, 0x90, 0x0d, 0x83, 0x71, 0x10, 0x13, 0x84, 0x21, 0x99, 0x20, 0x58, 0x0e, 0x81, 0x09, 0xc2, 0xa0, 0x6c, 0x10, 0x06, 0x66, 0x43, 0x92, 0x28, 0x4b, 0x92, 0x0c, 0x4c, 0xd2,
        0x6c, 0x08, 0x9c, 0x09, 0x02, 0xd6, 0x4c, 0x10, 0xa0, 0x65, 0xc3, 0x92, 0x40, 0x4b, 0x92, 0x0c, 0x4c, 0x14, 0x45, 0xc5, 0x86, 0x40, 0xda, 0x40, 0x3c, 0x13, 0x00, 0x4c, 0x10, 0x04, 0x60, 0x03,
        0xb0, 0x61, 0x48, 0x2c, 0x6b, 0x43, 0x70, 0x6d, 0x18, 0x86, 0x0a, 0xa3, 0xd2, 0x16, 0x96, 0xe6, 0xf6, 0x55, 0x87, 0x43, 0xf6, 0x46, 0x37, 0x41, 0xc8, 0x98, 0x0d, 0xc3, 0x30, 0x0c, 0x1b, 0x88,
        0x64, 0x23, 0xb8, 0x0d, 0x45, 0xa5, 0x01, 0x54, 0x57, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b,
        0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x44, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2, 0x32, 0xb6,
        0x29, 0xc1, 0x51, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0x30, 0x55, 0x22, 0xc3, 0x73, 0xa1, 0xcb, 0x83, 0x2b, 0x0b, 0x72, 0x73, 0x7b, 0xa3, 0x0b,
        0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x9b, 0x12, 0x60, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1d, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87,
        0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x0d, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x38, 0x13, 0xae, 0x0d, 0x43, 0x0a, 0x08, 0x0a, 0x31, 0xf8, 0xc8, 0x6d, 0x9b,
        0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11,
        0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x28, 0x0c, 0x80, 0x34, 0x3f, 0xc5, 0x0f, 0xce,
        0x04, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x34, 0xca, 0x52, 0x60, 0x06, 0xa0, 0xe4, 0x4a,
        0x37, 0xa0, 0x20, 0x0a, 0x59, 0xa0, 0x30, 0x05, 0xe8, 0x94, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x10, 0x4d, 0x87, 0xf0, 0x3c, 0xcd, 0x88, 0x41, 0x02, 0x80,
        0x20, 0x18, 0x44, 0x14, 0x22, 0x40, 0x90, 0x33, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x44, 0x86, 0x44, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x64, 0x8c, 0x10, 0x48, 0x05, 0x35, 0x9a, 0x10,
        0x00, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x6c, 0x0e, 0x31, 0x58, 0x87, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x74, 0x90, 0x51, 0x30, 0x09, 0x36, 0x9a, 0x10,
        0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xc0, 0x78, 0x4d, 0x50, 0x0c, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xe0, 0x78, 0x8d, 0x82, 0x64, 0x81, 0x24, 0x49, 0xcc, 0x86, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_code_i8dot[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x7e, 0x52, 0x99, 0x8e, 0x1c, 0xa8, 0x0e, 0x90, 0xdd, 0x8b, 0x56, 0xfb, 0xf8, 0x2a, 0xb4, 0xf2, 0x01, 0x00, 0x00, 0x00, 0xe4, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x5c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x14, 0x06, 0x00, 0x00, 0x64, 0x00, 0x05, 0x00, 0x85, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x04, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0xfc, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x7c, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x68, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x37, 0x0d, 0x97, 0x3f, 0x61,
        0x0f, 0x21, 0xf9, 0x2b, 0x21, 0xad, 0xc4, 0xe4, 0x23, 0xb7, 0x8d, 0x8a, 0x31, 0xc6, 0x18, 0x73, 0x04, 0x08, 0x9d, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40,
        0x01, 0x2a, 0x87, 0x19, 0x69, 0x8c, 0x33, 0x48, 0x95, 0x62, 0x8c, 0x31, 0x06, 0xb1, 0xb2, 0x80, 0x91, 0xc6, 0x18, 0x63, 0x8c, 0x33, 0xc8, 0xcd, 0x11, 0x04, 0xc5, 0x48, 0xe3, 0x8c, 0x01, 0x29,
        0x0e, 0x04, 0x5c, 0x23, 0x4d, 0x11, 0x25, 0x4c, 0xbe, 0xd8, 0x3c, 0xd4, 0x24, 0xa1, 0x61, 0x8c, 0x31, 0x93, 0x37, 0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x30, 0x0b, 0xf4, 0x20, 0x0f,
        0xf5, 0x30, 0x0e, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0xe4, 0x20, 0x0a, 0xf5, 0x60, 0x0e, 0xe6, 0x50, 0x0e, 0xf2, 0xc0, 0x07, 0xa9, 0xe0, 0x0e, 0xf0, 0x50, 0x0f, 0xf4, 0x30, 0x0f, 0x7e, 0x80,
        0x02, 0x4a, 0x75, 0x26, 0x33, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0x90, 0x85, 0x5b, 0x98, 0x05, 0x7a, 0x90, 0x87, 0x7a, 0x18, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x72,
        0x10, 0x85, 0x7a, 0x30, 0x07, 0x73, 0x28, 0x07, 0x79, 0xe0, 0x83, 0x7a, 0x70, 0x87, 0x79, 0x48, 0x87, 0x73, 0x70, 0x87, 0x72, 0x20, 0x07, 0x30, 0x48, 0x07, 0x77, 0xa0, 0x07, 0x3f, 0x40, 0xc1,
        0xa0, 0x3b, 0x47, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x78, 0xd0,
        0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d,
        0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06,
        0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43,
        0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x47, 0x01, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x03, 0x04, 0x40, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x07, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x11, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64,
        0x81, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x08, 0xc5, 0x30, 0x02, 0x50, 0x18,
        0x04, 0x47, 0x00, 0x28, 0x17, 0x28, 0x20, 0x20, 0x02, 0xe1, 0x19, 0x00, 0xb2, 0x33, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4,
        0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x63, 0x82, 0x30, 0x1c, 0x1b, 0x84, 0x81, 0xa0, 0x30,
        0x36, 0x37, 0x41, 0x18, 0x90, 0x0d, 0x83, 0x71, 0x10, 0x13, 0x84, 0x21, 0x99, 0x20, 0x58, 0x0e, 0x81, 0x09, 0xc2, 0xa0, 0x6c, 0x10, 0x06, 0x66, 0x43, 0x92, 0x28, 0x4b, 0x92, 0x0c, 0x4c, 0xd2,
        0x6c, 0x08, 0x9c, 0x09, 0x02, 0xd6, 0x4c, 0x10, 0xa0, 0x65, 0xc3, 0x92, 0x40, 0x4b, 0x92, 0x0c, 0x4c, 0x14, 0x45, 0xc5, 0x86, 0x40, 0xda, 0x40, 0x3c, 0x13, 0x00, 0x4c, 0x10, 0x04, 0x60, 0x03,
        0xb0, 0x61, 0x48, 0x2c, 0x6b, 0x43, 0x70, 0x6d, 0x18, 0x86, 0x0a, 0xa3, 0xd2, 0x16, 0x96, 0xe6, 0xf6, 0x95, 0x86, 0x43, 0xf6, 0x46, 0x37, 0x41, 0xc8, 0x98, 0x0d, 0xc3, 0x30, 0x0c, 0x1b, 0x88,
        0x64, 0x23, 0xb8, 0x0d, 0x45, 0xa5, 0x01, 0x54, 0x57, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b,
        0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x44, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2, 0x32, 0xb6,
        0x29, 0xc1, 0x51, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0x30, 0x55, 0x22, 0xc3, 0x73, 0xa1, 0xcb, 0x83, 0x2b, 0x0b, 0x72, 0x73, 0x7b, 0xa3, 0x0b,
        0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x9b, 0x12, 0x60, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x1d, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87,
        0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x0d, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x38, 0x13, 0xae, 0x0d, 0x43, 0x0a, 0x08, 0x0a, 0x31, 0xf8, 0xc8, 0x6d, 0x9b,
        0xc0, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0xb6, 0x11, 0x60, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11,
        0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x28, 0x0c, 0x80, 0x34, 0x3f, 0xc2, 0x0f, 0xce,
        0x04, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x34, 0xca, 0x52, 0x60, 0x06, 0xa0, 0xe4, 0x4a,
        0x37, 0xa0, 0x20, 0xca, 0x58, 0xa0, 0x30, 0x05, 0xe8, 0x94, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x10, 0x4d, 0x87, 0xf0, 0x3c, 0xcd, 0x88, 0x41, 0x02, 0x80,
        0x20, 0x18, 0x44, 0x14, 0x22, 0x40, 0x90, 0x33, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x44, 0x86, 0x44, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x64, 0x8c, 0x10, 0x48, 0x05, 0x35, 0x9a, 0x10,
        0x00, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x6c, 0x0e, 0x31, 0x58, 0x87, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x74, 0x90, 0x51, 0x30, 0x09, 0x36, 0x9a, 0x10,
        0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xc0, 0x78, 0x4d, 0x50, 0x0c, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xe0, 0x78, 0x8d, 0x82, 0x64, 0x81, 0x24, 0x49, 0xcc, 0x86, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_code_dot2add[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x0c, 0x19, 0xd3, 0xb0, 0x8e, 0xa2, 0x61, 0x43, 0xee, 0x5f, 0x7a, 0xc6, 0x5e, 0x8f, 0x70, 0xfd, 0x01, 0x00, 0x00, 0x00, 0xc4, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8e, 0x06, 0xf9, 0x08, 0x21, 0x17, 0x8f, 0xcf, 0x10, 0xe4, 0xfe, 0xb1,
        0xe2, 0x70, 0x58, 0xa7, 0x44, 0x58, 0x49, 0x4c, 0xb8, 0x06, 0x00, 0x00, 0x64, 0x00, 0x05, 0x00, 0xae, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x04, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0xa0, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xa5, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff,
        0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x37, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x2b,
        0x21, 0xad, 0xc4, 0xe4, 0x23, 0xb7, 0x8d, 0x8a, 0x31, 0xc6, 0x18, 0x73, 0x04, 0x08, 0x9d, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0x01, 0x2a, 0x87, 0x19,
        0x69, 0x8c, 0x33, 0x48, 0xcd, 0x00, 0x8c, 0x02, 0x94, 0x83, 0x0d, 0xd6, 0x5a, 0x6b, 0xe4, 0xca, 0xc0, 0xc6, 0x20, 0x58, 0x16, 0x30, 0xd2, 0x18, 0x63, 0x8c, 0x71, 0x06, 0xc9, 0x39, 0x82, 0xa0,
        0x18, 0x69, 0x9c, 0x31, 0x28, 0xd5, 0x81, 0x80, 0x6b, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x17, 0x9b, 0x87, 0x9a, 0x24, 0x34, 0x8c, 0x31, 0x66, 0xf2, 0xc6, 0x81, 0x1d, 0xc2, 0x61, 0x1e, 0xe6, 0xc1,
        0x0d, 0x66, 0x81, 0x1e, 0xe4, 0xa1, 0x1e, 0xc6, 0x81, 0x1e, 0xea, 0x41, 0x1e, 0xca, 0x81, 0x1c, 0x44, 0xa1, 0x1e, 0xcc, 0xc1, 0x1c, 0xca, 0x41, 0x1e, 0xf8, 0x20, 0x15, 0xdc, 0x01, 0x1e, 0xea,
        0x81, 0x1e, 0xe6, 0xc1, 0x0f, 0x50, 0x80, 0x29, 0xcf, 0x64, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb2, 0x70, 0x0b, 0xb3, 0x40, 0x0f, 0xf2, 0x50, 0x0f, 0xe3, 0x40, 0x0f,
        0xf5, 0x20, 0x0f, 0xe5, 0x40, 0x0e, 0xa2, 0x50, 0x0f, 0xe6, 0x60, 0x0e, 0xe5, 0x20, 0x0f, 0x7c, 0x50, 0x0f, 0xee, 0x30, 0x0f, 0xe9, 0x70, 0x0e, 0xee, 0x50, 0x0e, 0xe4, 0x00, 0x06, 0xe9, 0xe0,
        0x0e, 0xf4, 0xe0, 0x07, 0x28, 0x18, 0xb4, 0xe7, 0x08, 0x40, 0x01, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xae, 0x50,
        0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78,
        0xa0, 0x07, 0x78, 0xd0, 0x06, 0xe9, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x6d, 0x90, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x60, 0x07, 0x74, 0xa0, 0x07,
        0x76, 0x40, 0x07, 0x6d, 0x60, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xd0, 0x06, 0xe6, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60,
        0x07, 0x74, 0xd0, 0x06, 0xee, 0x80, 0x07, 0x7a, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x7a, 0x60, 0x07, 0x74, 0x30, 0xe4, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x60, 0xc8, 0x43, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x47, 0x01, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f, 0x03, 0x04,
        0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x08, 0x08, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x0c, 0x79, 0x2a, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90,
        0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x08, 0xc5, 0x30, 0x02, 0x50, 0x18, 0x44, 0x47, 0x00, 0xa8, 0x17, 0x38, 0x20, 0x20, 0x02, 0xf1, 0x19, 0x00, 0xd2, 0x33, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99,
        0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84,
        0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x26, 0x08, 0x43, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x63, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0x83, 0x32, 0x41, 0xd0, 0x1e, 0x02, 0x13, 0x84,
        0x61, 0xd9, 0x20, 0x0c, 0xcd, 0x86, 0x44, 0x59, 0x18, 0x45, 0x19, 0x1a, 0xc5, 0xd9, 0x10, 0x3c, 0x13, 0x04, 0xce, 0x99, 0x20, 0x50, 0xcc, 0x86, 0x45, 0x89, 0x18, 0x45, 0x19, 0x1a, 0x49, 0x92,
        0x8a, 0x0d, 0xc1, 0xb4, 0x81, 0x80, 0x28, 0x00, 0x98, 0x20, 0x08, 0x00, 0x99, 0xb6, 0xb0, 0x34, 0xb7, 0x2f, 0xb2, 0x37, 0x3a, 0x99, 0x30, 0x32, 0xb2, 0x09, 0x42, 0xd7, 0x6c, 0x18, 0x86, 0x61,
        0xd8, 0x40, 0x28, 0x18, 0x91, 0x6d, 0x28, 0xac, 0x0b, 0xa8, 0xb4, 0x0d, 0xc1, 0x50, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1,
        0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x46, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9,
        0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x01, 0x52, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0x40, 0xd5, 0x21, 0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb,
        0x83, 0x7a, 0x4b, 0x73, 0xa3, 0x9b, 0x9b, 0x12, 0x68, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xcb, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xf0, 0x20, 0x0f, 0xe5, 0x30, 0x0e, 0xe9, 0x30, 0x0f, 0xe5, 0x30, 0x23, 0x82, 0xc8, 0x01, 0x1f, 0xdc, 0x80,
        0x1c, 0xc4, 0xe1, 0x1c, 0xdc, 0xc0, 0x1e, 0xc2, 0x41, 0x1e, 0xd8, 0x21, 0x1c, 0xf2, 0xe1, 0x1d, 0xea, 0x81, 0x1e, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x66, 0x20, 0x0d, 0x97,
        0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x0d, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x38, 0x93, 0xad, 0x0d, 0x43, 0x08, 0x2c, 0x85, 0x5f, 0xdc, 0xb6, 0x11, 0x50,
        0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x10, 0x06, 0x20, 0x60, 0x5f, 0xad, 0xb7, 0xce, 0x77, 0xdb, 0x26, 0x70, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x04, 0x58, 0x1b, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0xe0,
        0x23, 0xb7, 0x6d, 0x05, 0xd8, 0x70, 0xf9, 0xce, 0xe3, 0x47, 0x80, 0xb5, 0x51, 0x45, 0x41, 0x44, 0xec, 0xe4, 0x44, 0x84, 0x8f, 0xdc, 0xb6, 0x05, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x1d, 0x11,
        0x01, 0x0c, 0xe2, 0xe0, 0x23, 0xb7, 0x6d, 0x00, 0x0c, 0x03, 0x20, 0xcd, 0x3f, 0x38, 0x93, 0x0d, 0x0c, 0x03, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x34, 0xca, 0x50, 0xa0, 0x2c, 0x05, 0x66, 0x00, 0x4a, 0xae, 0xfc, 0xff, 0x0f, 0x0a, 0xa4, 0x74, 0x03, 0x0a, 0xa2, 0x88, 0x05, 0x0a, 0x53, 0x80,
        0x4e, 0x09, 0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x50, 0x59, 0x89, 0x20, 0x49, 0xd0, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x54, 0x97, 0x22, 0x4c, 0x53, 0x34,
        0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x44, 0x87, 0x50, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x74, 0x8e, 0x10, 0x54, 0xc5, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0,
        0x7c, 0x10, 0x31, 0x64, 0x47, 0x36, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x84, 0x81, 0x64, 0x14, 0x4c, 0xb2, 0x8d, 0x26, 0x04, 0xc0, 0x05, 0x6c, 0x8b, 0x90, 0x74, 0xc4,
        0xc0, 0x00, 0x40, 0x10, 0x0c, 0xa0, 0x31, 0xb8, 0x82, 0x0b, 0xda, 0xd5, 0x41, 0xe1, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x40, 0x66, 0xa0, 0x05, 0x17, 0xb4, 0x2b, 0x06, 0xd3, 0x11, 0x03, 0x03,
        0x00, 0x41, 0x30, 0x80, 0xd2, 0xa0, 0x0b, 0x2e, 0x68, 0x57, 0x8f, 0x86, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x00, 0xb1, 0x01, 0x18, 0x04, 0x17, 0xb4, 0x1b, 0x31, 0x50, 0x00, 0x10, 0x04, 0x03,
        0xe7, 0x0d, 0xb4, 0x86, 0x08, 0x94, 0xe3, 0x82, 0xb1, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0x20, 0xbd, 0xc1, 0x86, 0x59, 0x6a, 0x10, 0x88, 0x81, 0x18, 0x88, 0x81, 0xc6, 0x06, 0x18, 0x10, 0x03,
        0x06, 0x00, 0x00, 0x00, 0x5b, 0x86, 0xe5, 0xd9, 0xb6, 0x0c, 0xce, 0xb3, 0x6d, 0x19, 0xa2, 0x67, 0xdb, 0x32, 0x50, 0xcf, 0xb6, 0x65, 0xb0, 0x9e, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
    };

    static const struct D3D12_SHADER_BYTECODE cs_u8dot = { cs_code_u8dot, sizeof(cs_code_u8dot) };
    static const struct D3D12_SHADER_BYTECODE cs_i8dot = { cs_code_i8dot, sizeof(cs_code_i8dot) };
    static const struct D3D12_SHADER_BYTECODE cs_dot2add = { cs_code_dot2add, sizeof(cs_code_dot2add) };

    struct test
    {
        const struct D3D12_SHADER_BYTECODE *cs;
        uint32_t input[3];
        uint32_t output;
    };

#define FP16_EXP2(x) ((15u + (x)) << 10)
    static const struct test tests[] =
    {
        { &cs_u8dot, { 0x010203ff, 0x05ff0708, 1000 }, 1000 + 1 * 5 + 2 * 255 + 3 * 7 + 255 * 8 },
        { &cs_i8dot, { 0x010203ff, 0x05ff0708, 1000 }, 1000 + 1 * 5 + 2 * (-1) + 3 * 7 + (-1) * 8 },
        { &cs_u8dot, { 0x01010101, 0x01010101, 0xffffffffu }, 0x3u },
        { &cs_i8dot, { 0x01010101, 0x01010101, 0x7fffffffu }, 0x80000003u },
        { &cs_i8dot, { 0x01010101, 0x01010101, 0xffffffffu }, 0x00000003u },
        { &cs_i8dot, { 0xffffffffu, 0x01010101, 0x80000003u }, 0x7fffffffu },
        { &cs_i8dot, { 0xffffffffu, 0x01010101, 0x3u }, 0xffffffffu },
        { &cs_dot2add, { FP16_EXP2(0), FP16_EXP2(0), 0x3f800000 }, 0x40000000 }, /* 2 = 1 + (1 * 1 + 0 * 0) */
        { &cs_dot2add, { FP16_EXP2(0) | (FP16_EXP2(1) << 16), FP16_EXP2(2) | (FP16_EXP2(3) << 16), 0x3f800000 }, 0x41a80000 }, /* 21 = 1 + (1 * 4 + 2 * 8) */
        /* Carefully test inf behavior. Verify that the operation is acc += float(a.x * b.x) + float(a.y * b.y).
         * I.e., in precise mode, we must observe FP16 infs from multiply, expand to FP32, then add.
         * Based on careful AMD testing, dot2_f16_f32 is generated from this intrinsic,
         * and FADD is supposed to happen in > FP16, not FP16 as the docs might suggest. */
        { &cs_dot2add, { FP16_EXP2(0), FP16_EXP2(15), 0x3f800000 }, 0x47000100 }, /* 32769 = 1 + 2^15 * 1 + 0 * 0 */
        { &cs_dot2add, { FP16_EXP2(0), FP16_EXP2(16), 0x3f800000 }, 0x7f800000 }, /* inf = 1 + inf * 1 + 0 * 0 */
        { &cs_dot2add, { FP16_EXP2(1), FP16_EXP2(15), 0x3f800000 }, 0x7f800000 }, /* inf = 1 + fp16 inf + 0 * 0 */
        { &cs_dot2add, { FP16_EXP2(0) * 0x10001u, FP16_EXP2(15) * 0x10001u, 0x3f800000 }, 0x47800080 }, /* 2^16 + 1 = 1 + 2^15 + 2^15. This will inf if dot product is completed in FP16. */
        /* Verify addition order. The dot part must be completed before accumulating. If this is not done, both accumulations will round to original value. */
        { &cs_dot2add, { FP16_EXP2(0) * 0x10001u, 0x3c00 * 0x10001u, 0x4b800000 }, 0x4b800001 },
    };
#undef FP16_EXP2

    if (!init_compute_test_context(&context))
        return;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&shader_model, 0, sizeof(shader_model));
    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_4;
    if (FAILED(ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model))) ||
        shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_4)
    {
        skip("Device does not support SM 6.4.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&rs_desc, 0, sizeof(rs_desc));
    rs_desc.NumParameters = ARRAY_SIZE(root_parameters);
    rs_desc.pParameters = root_parameters;
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    create_root_signature(context.device, &rs_desc, &context.root_signature);

    memset(&features4, 0, sizeof(features4));
    ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS4, &features4, sizeof(features4));

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);

        if (tests[i].cs == &cs_dot2add && !features4.Native16BitShaderOpsSupported)
        {
            skip("Skipping unsupported test dot2add.\n");
            continue;
        }

        pso = create_compute_pipeline_state(context.device, context.root_signature, *tests[i].cs);
        src = create_upload_buffer(context.device, sizeof(tests[i].input), tests[i].input);
        dst = create_default_buffer(context.device, sizeof(tests[i].input), D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, context.root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
        ID3D12GraphicsCommandList_SetComputeRootShaderResourceView(context.list, 0, ID3D12Resource_GetGPUVirtualAddress(src));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, ID3D12Resource_GetGPUVirtualAddress(dst));
        ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);

        transition_resource_state(context.list, dst, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
        get_buffer_readback_with_command_list(dst, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

        value = get_readback_uint(&rb, 0, 0, 0);
        ok(value == tests[i].output, "0x%x != 0x%x\n", value, tests[i].output);

        ID3D12Resource_Release(src);
        ID3D12Resource_Release(dst);
        ID3D12PipelineState_Release(pso);
        release_resource_readback(&rb);
        reset_command_list(context.list, context.allocator);
    }

    vkd3d_test_set_context(NULL);
    destroy_test_context(&context);
}

void test_shader_sm65_wave_intrinsics(void)
{
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    struct test_context context;
    struct resource_readback rb;
    ID3D12PipelineState *pso;
    unsigned int i, j, k;
    ID3D12Resource *src;
    ID3D12Resource *dst;
    uint32_t value;

    static const BYTE cs_code_wave_match[] =
    {
#if 0
        StructuredBuffer<uint> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(16, 1, 1)]
        void main()
        {
                RW[WaveGetLaneIndex()] = WaveMatch(RO[WaveGetLaneIndex()]).x;
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x45, 0x71, 0xfc, 0x9d, 0xe2, 0xb8, 0x0d, 0x2c, 0x8b, 0xfe, 0xf7, 0x43, 0x4c, 0xb6, 0xd5, 0x70, 0x01, 0x00, 0x00, 0x00, 0xf8, 0x06, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xab, 0x09, 0x3e, 0xf1, 0xcf, 0x2f, 0x68, 0xdc, 0xc1, 0xab, 0xb3, 0x78,
        0xd2, 0x1d, 0xda, 0xd9, 0x44, 0x58, 0x49, 0x4c, 0xec, 0x05, 0x00, 0x00, 0x65, 0x00, 0x05, 0x00, 0x7b, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x05, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0xd4, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x88, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0xda, 0x60, 0x08, 0xff, 0xff, 0xff, 0xff,
        0x3f, 0x00, 0x12, 0x50, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa0, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x68, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xe0, 0xa0, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0xbf, 0x70, 0xa8, 0x08, 0xb9, 0x6d,
        0x44, 0x8c, 0x31, 0x46, 0x19, 0xc8, 0x18, 0x54, 0xe6, 0x08, 0x10, 0x3a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x02, 0x54, 0x16, 0x30, 0xd2, 0x18, 0x63,
        0x8c, 0x71, 0x06, 0xa9, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x91, 0xdb, 0x46, 0xc5, 0x18, 0x63, 0x8c, 0x72, 0xb0, 0x91, 0xc6, 0x38, 0x83, 0x5a, 0x11,
        0xc6, 0xa0, 0x37, 0x47, 0x10, 0x14, 0x23, 0x8d, 0x33, 0x46, 0x24, 0x39, 0x10, 0x30, 0x13, 0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
        0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03, 0x18,
        0xa4, 0x83, 0x3b, 0xd0, 0x83, 0x1f, 0xa0, 0x60, 0x50, 0x9d, 0xc9, 0x0c, 0xc6, 0x81, 0x1d, 0xc2, 0x61, 0x1e, 0xe6, 0xc1, 0x0d, 0x64, 0xe1, 0x16, 0x66, 0x81, 0x1e, 0xe4, 0xa1, 0x1e, 0xc6, 0x81,
        0x1e, 0xea, 0x41, 0x1e, 0xca, 0x81, 0x1c, 0x44, 0xa1, 0x1e, 0xcc, 0xc1, 0x1c, 0xca, 0x41, 0x1e, 0xf8, 0xa0, 0x1e, 0xdc, 0x61, 0x1e, 0xd2, 0xe1, 0x1c, 0xdc, 0xa1, 0x1c, 0xc8, 0x01, 0x0c, 0xd2,
        0xc1, 0x1d, 0xe8, 0xc1, 0x0f, 0x50, 0x30, 0xe8, 0xce, 0x11, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0,
        0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6,
        0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x05, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a,
        0x25, 0x50, 0x0a, 0xc5, 0x30, 0x02, 0x50, 0x18, 0x85, 0x50, 0x20, 0x14, 0x47, 0x00, 0x28, 0x17, 0x28, 0x20, 0x10, 0xe1, 0x19, 0x00, 0xb2, 0x33, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41,
        0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xc1, 0x98, 0x20, 0x0c,
        0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x03, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x63, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0x43, 0x32, 0x41, 0xb0, 0x20, 0x02, 0x13, 0x84, 0x41, 0x99, 0x20, 0x0c,
        0xcb, 0x06, 0x61, 0x70, 0x36, 0x24, 0xca, 0xc2, 0x28, 0xca, 0xd0, 0x28, 0xcf, 0x86, 0x00, 0x9a, 0x20, 0x60, 0xcf, 0x04, 0x21, 0x6a, 0x36, 0x2c, 0x8a, 0xc4, 0x28, 0xca, 0xd0, 0x4c, 0xd3, 0xf4,
        0x6c, 0x08, 0xa8, 0x0d, 0x44, 0x54, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0x32, 0x67, 0x82, 0x30, 0x30, 0x1b, 0x06, 0x6d, 0x18, 0x36, 0x10, 0x4a, 0xe6, 0x6c,
        0x1b, 0x8a, 0x0b, 0x03, 0x2c, 0xae, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57, 0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25,
        0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a, 0x64, 0x78, 0x2e, 0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53, 0x02, 0xa4,
        0x0c, 0x19, 0x9e, 0x8b, 0x5c, 0xd9, 0xdc, 0x5b, 0x9d, 0xdc, 0x58, 0xd9, 0xdc, 0x94, 0xa0, 0xaa, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25,
        0xe0, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b,
        0x08, 0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10,
        0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x0d, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0xc8, 0x6d, 0x9b, 0x00, 0x36, 0x5c, 0xbe,
        0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39, 0x11, 0xe1, 0x23, 0xb7, 0x6d, 0x04, 0xd6, 0x70, 0xf9, 0xce, 0xe3, 0x5b, 0x40, 0x45, 0x80, 0xc4, 0x54, 0x02, 0x0d, 0x21, 0x36,
        0x03, 0x71, 0x59, 0xc0, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x16, 0x50, 0x11, 0x26, 0x30, 0x09, 0x87, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00,
        0x1c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0xae, 0xbc, 0x03, 0xca, 0x52, 0xa0, 0x94, 0x05, 0x0a, 0x53, 0x80, 0x4e,
        0x09, 0x8c, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x51, 0x87, 0x00, 0x41, 0xcc, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x48, 0x15, 0x22, 0x44, 0x51, 0x33, 0x62, 0x50, 0x00, 0x20,
        0x08, 0x06, 0xcf, 0x85, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0x41, 0x93, 0x21, 0x42, 0x30, 0x15, 0xd1, 0x68, 0x42, 0x00, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0x41, 0xd1, 0x25, 0xc1, 0x68, 0x42,
        0x00, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0x81, 0xe2, 0x29, 0x47, 0x71, 0x05, 0xcf, 0xf3, 0x24, 0x15, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const BYTE cs_code_wave_multiprefix[] =
    {
#if 0
        StructuredBuffer<uint> RO : register(t0);
        RWStructuredBuffer<uint> RW : register(u0);

        [numthreads(16, 1, 1)]
        void main_summed()
        {
                uint4 mask = uint4(RO[WaveGetLaneIndex()], 0, 0, 0);
                RW[WaveGetLaneIndex() + 16 * 0] = WaveMultiPrefixSum(WaveGetLaneIndex() + 1, mask);
                RW[WaveGetLaneIndex() + 16 * 1] = WaveMultiPrefixProduct(WaveGetLaneIndex() + 1, mask);
                RW[WaveGetLaneIndex() + 16 * 2] = WaveMultiPrefixCountBits((mask.x & 0xaaaa) != 0, mask);
                RW[WaveGetLaneIndex() + 16 * 3] = WaveMultiPrefixBitAnd(WaveGetLaneIndex() + 1, mask);
                RW[WaveGetLaneIndex() + 16 * 4] = WaveMultiPrefixBitOr(WaveGetLaneIndex() + 1, mask);
                RW[WaveGetLaneIndex() + 16 * 5] = WaveMultiPrefixBitXor(WaveGetLaneIndex() + 1, mask);
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0x1f, 0x46, 0x84, 0x8f, 0xa2, 0xc4, 0xd6, 0xa3, 0x20, 0x00, 0xc0, 0xa4, 0x5c, 0xe2, 0x64, 0x26, 0x01, 0x00, 0x00, 0x00, 0x08, 0x08, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x78, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x9e, 0x05, 0x43, 0xd4, 0x4b, 0xcf, 0x8b, 0xb7, 0x0b, 0xf7, 0xe7,
        0xc6, 0x73, 0x97, 0x8b, 0x44, 0x58, 0x49, 0x4c, 0xfc, 0x06, 0x00, 0x00, 0x65, 0x00, 0x05, 0x00, 0xbf, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x05, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0xe4, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xb6, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91,
        0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14,
        0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c,
        0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x88, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0xda, 0x60, 0x08, 0xff, 0xff, 0xff, 0xff,
        0x3f, 0x00, 0x12, 0x50, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa0, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14,
        0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x6c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x21, 0x72, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02,
        0xcd, 0xb0, 0x10, 0x28, 0x28, 0x65, 0x01, 0xc3, 0x8c, 0x31, 0xc6, 0x18, 0x64, 0xd0, 0xb9, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x25, 0x26, 0x1f, 0xb9, 0x6d, 0x54, 0x8c,
        0x31, 0xc6, 0x28, 0x47, 0x1a, 0x66, 0x0c, 0x32, 0x48, 0x15, 0x61, 0x0c, 0x62, 0x25, 0x19, 0x63, 0x8c, 0x31, 0x06, 0x21, 0xe4, 0xe6, 0x08, 0x82, 0x72, 0x8c, 0x01, 0xc7, 0x18, 0x83, 0x62, 0x31,
        0xcc, 0x20, 0x63, 0x40, 0x9a, 0x03, 0x01, 0x33, 0x91, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0xc1, 0x2c, 0xd0, 0x83, 0x3c, 0xd4, 0xc3, 0x38, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39,
        0x90, 0x83, 0x28, 0xd4, 0x83, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x03, 0x1f, 0xd4, 0x83, 0x3b, 0xcc, 0x43, 0x3a, 0x9c, 0x83, 0x3b, 0x94, 0x03, 0x39, 0x80, 0x41, 0x3a, 0xb8, 0x03, 0x3d, 0xf8, 0x01,
        0x0a, 0x06, 0xd9, 0x99, 0xcc, 0x60, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x40, 0x16, 0x6e, 0x61, 0x16, 0xe8, 0x41, 0x1e, 0xea, 0x61, 0x1c, 0xe8, 0xa1, 0x1e, 0xe4, 0xa1, 0x1c, 0xc8,
        0x41, 0x14, 0xea, 0xc1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xea, 0xc1, 0x1d, 0xe6, 0x21, 0x1d, 0xce, 0xc1, 0x1d, 0xca, 0x81, 0x1c, 0xc0, 0x20, 0x1d, 0xdc, 0x81, 0x1e, 0xfc, 0x00, 0x05,
        0x83, 0xf0, 0x1c, 0x01, 0x28, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74,
        0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07,
        0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86,
        0x3c, 0x07, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x13, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x90, 0x67, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x1a, 0x25, 0x50, 0x0a, 0xc5, 0x30, 0x02, 0x50, 0x18, 0x85, 0x50, 0x20, 0x04, 0x47, 0x00, 0x48, 0x17, 0x28, 0x20, 0x10, 0xe5, 0x19, 0x00, 0xba, 0x33, 0x00, 0x00, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99,
        0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84,
        0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x26, 0x08, 0x43, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x63, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0x83, 0x32, 0x41, 0xb8, 0x22, 0x02, 0x13, 0x84,
        0x61, 0x99, 0x20, 0x0c, 0xcc, 0x06, 0x61, 0x70, 0x36, 0x24, 0xca, 0xc2, 0x28, 0xca, 0xd0, 0x28, 0xcf, 0x86, 0x00, 0x9a, 0x20, 0x64, 0xd0, 0x04, 0x01, 0x72, 0x36, 0x2c, 0x8a, 0xc4, 0x28, 0xca,
        0xd0, 0x4c, 0xd3, 0xf4, 0x6c, 0x08, 0xa8, 0x0d, 0x44, 0x54, 0x01, 0xc0, 0x04, 0x41, 0x00, 0xb8, 0xb4, 0x85, 0xa5, 0xb9, 0x7d, 0xcd, 0xd5, 0xb5, 0xb5, 0x95, 0x91, 0x4d, 0x10, 0xb4, 0x67, 0x82,
        0x30, 0x34, 0x1b, 0x06, 0x6d, 0x18, 0x36, 0x10, 0x4a, 0xe6, 0x6c, 0x1b, 0x8a, 0x0b, 0x03, 0x2c, 0xae, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42,
        0x86, 0xe7, 0x62, 0x57, 0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a, 0x64, 0x78, 0x2e, 0x73, 0x68, 0x61,
        0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53, 0x02, 0xa4, 0x0c, 0x19, 0x9e, 0x8b, 0x5c, 0xd9, 0xdc, 0x5b, 0x9d, 0xdc, 0x58, 0xd9, 0xdc, 0x94, 0xa0, 0xaa, 0x43, 0x86, 0xe7, 0x52, 0xe6,
        0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66,
        0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e,
        0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b,
        0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0,
        0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83,
        0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76,
        0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30,
        0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8,
        0x21, 0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08, 0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x21, 0x00, 0x00, 0x00, 0x66, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x09, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05,
        0x11, 0x95, 0x0e, 0x30, 0xf8, 0xc8, 0x6d, 0x5b, 0x00, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39, 0x11, 0xe1, 0x23, 0xb7, 0x6d, 0x03, 0xd6, 0x70, 0xf9,
        0xce, 0xe3, 0x5b, 0x40, 0x45, 0x80, 0xc4, 0x54, 0x02, 0x0d, 0x21, 0x36, 0x03, 0x71, 0x59, 0x41, 0x37, 0x5c, 0xbe, 0xf3, 0xf8, 0x16, 0x50, 0x11, 0x26, 0xb5, 0x4c, 0x48, 0x1a, 0x11, 0x05, 0x72,
        0x6d, 0xc8, 0xc4, 0x39, 0x54, 0x33, 0x19, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x16, 0x50, 0x11, 0x26, 0xb5, 0x4c, 0x48, 0x1a, 0x11, 0x05, 0x72, 0xa1, 0x8f, 0x8f, 0xdc, 0xb6, 0x01, 0x2c, 0x0c,
        0x80, 0x34, 0xbf, 0x44, 0x31, 0x0c, 0x31, 0x00, 0x61, 0x20, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x0a,
        0x53, 0xa0, 0x98, 0x05, 0x4a, 0xae, 0xbc, 0x03, 0xca, 0x52, 0xa0, 0xa8, 0x55, 0x0b, 0xca, 0x59, 0xa0, 0x80, 0x0a, 0xac, 0x00, 0x03, 0x0a, 0x34, 0x80, 0x48, 0x09, 0x8c, 0x00, 0x14, 0x42, 0x11,
        0x94, 0x01, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x30, 0x75, 0x4e, 0x91, 0x65, 0xd5, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x4c, 0xde, 0x53, 0x68, 0x9a, 0x35, 0x62, 0x50, 0x00, 0x20,
        0x08, 0x06, 0x8c, 0x18, 0x3c, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0xa0, 0x90, 0xc1, 0x23, 0x04, 0x1c, 0xa2, 0x8d, 0x26, 0x04, 0x40, 0x0d, 0x61, 0x00, 0x23, 0x06, 0x0c, 0x00, 0x82, 0x60, 0xe0,
        0x94, 0x41, 0x15, 0x08, 0xdf, 0xf7, 0x29, 0xcb, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x1c, 0x69, 0x70, 0x1d, 0x05, 0x18, 0x04, 0x18, 0x86, 0x31, 0xde, 0x88, 0x01, 0x03, 0x80, 0x20, 0x18, 0x38,
        0x66, 0x60, 0x09, 0x03, 0x18, 0x80, 0x01, 0x18, 0x28, 0x4c, 0x19, 0x1e, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0xc1, 0xb1, 0x06, 0x59, 0x12, 0x88, 0x81, 0xa0, 0x69, 0x9a, 0x03, 0x06, 0x55, 0x50,
        0x3a, 0xdc, 0x10, 0x8c, 0x41, 0x18, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0x41, 0x94, 0x06, 0x55, 0x70, 0x90, 0x01, 0x19, 0x90, 0x41, 0x29, 0x15, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0xc1, 0xf1,
        0x06, 0x5d, 0x13, 0x98, 0x81, 0xe0, 0x79, 0x9e, 0x44, 0x06, 0x23, 0x06, 0x0c, 0x00, 0x82, 0x60, 0xe0, 0xb0, 0x01, 0x87, 0x24, 0x66, 0x60, 0x06, 0x66, 0x20, 0x45, 0xc5, 0x58, 0x30, 0x62, 0xe0,
        0x00, 0x20, 0x08, 0x06, 0x47, 0x1c, 0x7c, 0x4f, 0x80, 0x06, 0x02, 0x18, 0x80, 0x01, 0x18, 0x50, 0x66, 0x30, 0x62, 0xc0, 0x00, 0x20, 0x08, 0x06, 0x8e, 0x1b, 0x78, 0xca, 0x82, 0x06, 0x68, 0x80,
        0x06, 0xd1, 0x54, 0xce, 0x05, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0x70, 0xcc, 0x41, 0x18, 0x44, 0x81, 0x1a, 0x08, 0x62, 0x20, 0x06, 0x62, 0x60, 0xa1, 0xc1, 0x88, 0x01, 0x03, 0x80, 0x20, 0x18,
        0x38, 0x70, 0x00, 0x06, 0x4c, 0xa3, 0x06, 0x6a, 0xa0, 0x06, 0x52, 0x55, 0x10, 0x06, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0x70, 0xd4, 0xc1, 0x18, 0x4c, 0x01, 0x1b, 0x08, 0x64, 0x40, 0x06, 0x64,
        0x80, 0xa9, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00,
    };

    static const struct D3D12_SHADER_BYTECODE cs_wave_match = { cs_code_wave_match, sizeof(cs_code_wave_match) };
    static const struct D3D12_SHADER_BYTECODE cs_wave_multiprefix = { cs_code_wave_multiprefix, sizeof(cs_code_wave_multiprefix) };

    struct test
    {
        const struct D3D12_SHADER_BYTECODE *cs;
        uint32_t input[16];
        uint32_t output[6][16];
        unsigned int instances;
    };

    /* Wave match is fairly basic. It groups elements which are equal into partitions.
     * That partition gets a ballot which is the mask of all active invocations in that partition.
     * The ballot is then broadcast to all active participants of the partition.
     * This is implemented as a scalarization loop on AMD, and probably maps to something more optimal on NV
     * if SPV_NV_shader_subgroup_partitioned is anything to go by ... */

    /* Multiprefix works on partitions, and each partition performs its own prefix sum operation.
     * There are various restrictions:
     * - An invocation must only appear in one partition.
     * - Undocumented, but obvious requirement: For invocation N, bit N must be set in mask.
     * Additionally, the mask is ANDed by active invocation mask.
     *
     * NV SPIR-V extension is more precise here:
        Add: "The ballot parameter to the partitioned operations must form a valid partition of the active invocations in the subgroup.
        The values of ballot are a valid partition if:
        for each active invocation i, the bit corresponding to i is set in i's value of ballot, and
        for any two active invocations i and j, if the bit corresponding to invocation j is set in invocation i's value of ballot,
        then invocation j's value of ballot must equal invocation i's value of ballot, and
        bits not corresponding to any invocation in the subgroup are ignored.
     */
    static const struct test tests[] =
    {
        { &cs_wave_match,
            { 20, 50, 80, 100, 40, 20, 90, 20, 10, 0, 20, 50, 80, 90, 110, 120 },
            {{ 0x4a1, 0x802, 0x1004, 0x8, 0x10, 0x4a1, 0x2040, 0x4a1, 0x100, 0x200, 0x4a1, 0x802, 0x1004, 0x2040, 0x4000, 0x8000 }}, 1 },

        /* First, test identities as a sanity check. */
        { &cs_wave_multiprefix,
            { 0x1, 0x2, 0x4, 0x8, 0x10, 0x20, 0x40, 0x80, 0x100, 0x200, 0x400, 0x800, 0x1000, 0x2000, 0x4000, 0x8000 },
            {
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
                { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, },
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
                { ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, },
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
            }, 6 },

        /* Everything in same group. */
        { &cs_wave_multiprefix,
            { ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, },
            {
                { 0, 0x1, 0x3, 0x6, 0xa, 0xf, 0x15, 0x1c, 0x24, 0x2d, 0x37, 0x42, 0x4e, 0x5b, 0x69, 0x78, },
                { 0x1, 0x1, 0x2, 0x6, 0x18, 0x78, 0x2d0, 0x13b0, 0x9d80, 0x58980, 0x375f00, 0x2611500, 0x1c8cfc00, 0x7328cc00, 0x4c3b2800, 0x77775800, },
                { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, },
                { ~0u, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
                { 0, 1, 3, 3, 7, 7, 7, 7, 0xf, 0xf, 0xf, 0xf, 0xf, 0xf, 0xf, 0xf },
                { 0, 1, 3, 0, 4, 1, 7, 0, 8, 1, 0xb, 0, 0xc, 0x1, 0xf, 0, },
            }, 6 },

        /* Everything in same group, still. Need to mask ballot before checking partitions. */
        { &cs_wave_multiprefix,
            { ~0u, ~0u, ~0u, 0xffff, ~0u, 0x1ffff, ~0u, ~0u, ~0u, 0x8000ffff, ~0u, ~0u, ~0u, ~0u, ~0u, ~0u, },
            {
                { 0, 0x1, 0x3, 0x6, 0xa, 0xf, 0x15, 0x1c, 0x24, 0x2d, 0x37, 0x42, 0x4e, 0x5b, 0x69, 0x78, },
                { 0x1, 0x1, 0x2, 0x6, 0x18, 0x78, 0x2d0, 0x13b0, 0x9d80, 0x58980, 0x375f00, 0x2611500, 0x1c8cfc00, 0x7328cc00, 0x4c3b2800, 0x77775800, },
                { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, },
                { ~0u, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
                { 0, 1, 3, 3, 7, 7, 7, 7, 0xf, 0xf, 0xf, 0xf, 0xf, 0xf, 0xf, 0xf },
                { 0, 1, 3, 0, 4, 1, 7, 0, 8, 1, 0xb, 0, 0xc, 0x1, 0xf, 0, },
            }, 6 },

        /* The real test ... */
        { &cs_wave_multiprefix,
            { 0x4a1, 0x802, 0x1004, 0x8, 0x10, 0x4a1, 0x2040, 0x4a1, 0x100, 0x200, 0x4a1, 0x802, 0x1004, 0x2040, 0x4000, 0x8000 },
            {
                { 0, 0, 0, 0, 0, 1, 0, 7, 0, 0, 0xf, 2, 3, 7, },
                { 1, 1, 1, 1, 1, 1, 1, 6, 1, 1, 0x30, 0x2, 0x3, 0x7, 0x1, 0x1, },
                { 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 3, 1, 0, 1, },
                { ~0u, ~0u, ~0u, ~0u, ~0u, 1, ~0u, 0, ~0u, ~0u, 0, 2, 3, 7, ~0u, ~0u },
                { 0, 0, 0, 0, 0, 1, 0, 7, 0, 0, 0xf, 0x2, 0x3, 0x7, },
                { 0, 0, 0, 0, 0, 1, 0, 7, 0, 0, 0xf, 0x2, 0x3, 0x7, },
            }, 6 },

        /* With inactive lane handling ... */
        { &cs_wave_multiprefix,
            { 0x4a1, 0x800802, 0x1004, 0x8, 0x10, 0x4a1, 0x8002040, 0x4a1, 0x100, 0x200, 0x4a1, 0x802, 0x1004, 0x2040, 0x4000, 0x8000 },
            {
                { 0, 0, 0, 0, 0, 1, 0, 7, 0, 0, 0xf, 2, 3, 7, },
                { 1, 1, 1, 1, 1, 1, 1, 6, 1, 1, 0x30, 0x2, 0x3, 0x7, 0x1, 0x1, },
                { 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 3, 1, 0, 1, },
                { ~0u, ~0u, ~0u, ~0u, ~0u, 1, ~0u, 0, ~0u, ~0u, 0, 2, 3, 7, ~0u, ~0u },
                { 0, 0, 0, 0, 0, 1, 0, 7, 0, 0, 0xf, 0x2, 0x3, 0x7, },
                { 0, 0, 0, 0, 0, 1, 0, 7, 0, 0, 0xf, 0x2, 0x3, 0x7, },
            }, 6 }
    };

    if (!init_compute_test_context(&context))
        return;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&shader_model, 0, sizeof(shader_model));
    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_5;
    if (FAILED(ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model))) ||
        shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_5)
    {
        skip("Device does not support SM 6.5.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&rs_desc, 0, sizeof(rs_desc));
    rs_desc.NumParameters = ARRAY_SIZE(root_parameters);
    rs_desc.pParameters = root_parameters;
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_SRV;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    create_root_signature(context.device, &rs_desc, &context.root_signature);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);
        pso = create_compute_pipeline_state(context.device, context.root_signature, *tests[i].cs);
        src = create_upload_buffer(context.device, sizeof(tests[i].input), tests[i].input);
        dst = create_default_buffer(context.device, sizeof(tests[i].output), D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, context.root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pso);
        ID3D12GraphicsCommandList_SetComputeRootShaderResourceView(context.list, 0, ID3D12Resource_GetGPUVirtualAddress(src));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, ID3D12Resource_GetGPUVirtualAddress(dst));
        ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);

        transition_resource_state(context.list, dst, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
        get_buffer_readback_with_command_list(dst, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

        for (k = 0; k < tests[i].instances; k++)
        {
            for (j = 0; j < 16; j++)
            {
                value = get_readback_uint(&rb, k * 16 + j, 0, 0);
                ok(value == tests[i].output[k][j], "Index %u, instance %u: 0x%x != 0x%x\n", j, k, value, tests[i].output[k][j]);
            }
        }

        ID3D12Resource_Release(src);
        ID3D12Resource_Release(dst);
        ID3D12PipelineState_Release(pso);
        release_resource_readback(&rb);
        reset_command_list(context.list, context.allocator);
    }

    vkd3d_test_set_context(NULL);
    destroy_test_context(&context);
}

void test_shader_sm66_is_helper_lane(void)
{
    /* Oh, hi there. */
    static const float alpha_keys[4] = { 0.75f, 2.25f, 3.25f, 3.75f };
    static const struct vec4 white = { 1.0f, 1.0f, 1.0f, 1.0f };
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
    ID3D12GraphicsCommandList *command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    ID3D12Resource *readback_buffer;
    struct test_context_desc desc;
    ID3D12Resource *atomic_buffer;
    struct resource_readback rb;
    struct test_context context;
    ID3D12Resource *input_keys;
    ID3D12DescriptorHeap *heap;
    ID3D12CommandQueue *queue;
    unsigned int x, y, i;
    HRESULT hr;

    static const BYTE ps_discard_atomic_loop[] =
    {
#if 0
        float build_alpha_shuffle(float v)
        {
                v = IsHelperLane() ? 8.0 : round(v);
                float4 helpers;
                helpers.x = QuadReadLaneAt(v, 0);
                helpers.y = QuadReadLaneAt(v, 1);
                helpers.z = QuadReadLaneAt(v, 2);
                helpers.w = QuadReadLaneAt(v, 3);
                return dot(helpers, float4(1, 10, 100, 1000));
        }

        float4 main(float4 pos : SV_Position) : SV_Target
        {
                int2 coord = int2(pos.xy);
                int linear_coord = coord.y * 2 + coord.x;

                StructuredBuffer<float> alpha = ResourceDescriptorHeap[0];
                float alpha_value = alpha[linear_coord];
                float mask0 = build_alpha_shuffle(alpha_value);
                // Lane 1 and 2 should be nuked by this.
                if (frac(alpha_value) < 0.5)
                        discard;

                float mask1 = build_alpha_shuffle(alpha_value);

                RWStructuredBuffer<uint> atomics = ResourceDescriptorHeap[1];
                uint last_value = 0;
                InterlockedAdd(atomics[linear_coord], 101, last_value);
                if (linear_coord == 3 || last_value > 1000)
                        discard;

                float mask2 = build_alpha_shuffle(alpha_value);
                float4 color = float4(1.0, mask0, mask1, mask2);
                return color;
        }
#endif
        0x44, 0x58, 0x42, 0x43, 0xdc, 0x07, 0xff, 0x8a, 0xa1, 0x45, 0x3d, 0x04, 0x48, 0x33, 0x69, 0xd0, 0xdb, 0x46, 0x4b, 0xbe, 0x01, 0x00, 0x00, 0x00, 0x0e, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0x42, 0x01, 0x00, 0x00, 0x5e, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53,
        0x56, 0x30, 0x7c, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x98, 0x4c, 0xa1, 0x3c, 0x2c, 0xf1, 0x80, 0x5f, 0xfa, 0x05, 0x63, 0x14, 0x3a, 0xfd, 0xe2, 0x7c, 0x44, 0x58,
        0x49, 0x4c, 0xa8, 0x08, 0x00, 0x00, 0x66, 0x00, 0x00, 0x00, 0x2a, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x90, 0x08, 0x00, 0x00, 0x42, 0x43,
        0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x21, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10,
        0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32,
        0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31,
        0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30,
        0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20,
        0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84,
        0xc4, 0x4c, 0x10, 0x94, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x40, 0x10, 0x44, 0x41, 0x90, 0x51, 0x0c, 0x80, 0x20, 0x88, 0x62, 0x20,
        0x64, 0x8e, 0x20, 0x28, 0x03, 0x40, 0x28, 0xb4, 0x50, 0x71, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x68, 0x4a, 0xe1, 0x10, 0x84, 0xa2, 0xd0, 0x73, 0xd3,
        0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x4a, 0x4c, 0x7e, 0x71, 0xdb, 0xa8, 0x18, 0x86, 0x61, 0x20, 0xca, 0x11, 0x11, 0x0e, 0x41, 0x14, 0x04, 0x49, 0x65, 0x18, 0x88, 0x81, 0xa8,
        0x82, 0x10, 0x84, 0x43, 0x10, 0x04, 0x41, 0x90, 0x55, 0x04, 0x85, 0x20, 0xac, 0x10, 0x03, 0x31, 0x10, 0xa4, 0x15, 0x65, 0x20, 0x86, 0x61, 0x18, 0x86, 0x61, 0x18, 0x88, 0xbb, 0x6d, 0xb8, 0xfc,
        0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0xc9, 0xa1, 0x22, 0x81, 0x48, 0x23, 0xe7, 0x21, 0xa2, 0x09, 0x21, 0x24, 0x24, 0x10, 0x44, 0x21, 0x1c, 0xc2, 0xf1, 0xe8, 0x1b, 0x08, 0x48, 0x03, 0x62, 0x8e,
        0x00, 0x14, 0xa6, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50,
        0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06,
        0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0,
        0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06,
        0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x2c, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90,
        0x87, 0x02, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f, 0x05, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x0c, 0x08, 0x80, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x1a, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x38, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x18, 0xf2, 0x7c, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x05, 0x02, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09,
        0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0xca, 0xa0, 0x3c, 0xa8, 0x28, 0x89, 0x32, 0x28, 0x84, 0x11, 0x80, 0x02, 0xa1, 0xb1, 0x00, 0x01, 0x81, 0x01, 0x11, 0x48, 0x1c,
        0xcb, 0x20, 0x08, 0xc7, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b,
        0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31,
        0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x82, 0x99, 0x20, 0x10, 0xcd, 0x06, 0x61, 0x20, 0x28, 0xc0, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x88, 0xc1, 0xb5, 0x21, 0x40,
        0x26, 0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x6d, 0x82, 0x40, 0x38, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x45, 0x34, 0x41, 0x28, 0xa4, 0x0d,
        0x01, 0x33, 0x41, 0x28, 0xa6, 0x09, 0x42, 0x41, 0x4d, 0x10, 0x88, 0x67, 0x83, 0x30, 0x4d, 0x1b, 0x16, 0xa6, 0x71, 0x1e, 0x28, 0x1a, 0x22, 0x46, 0xa2, 0x36, 0x04, 0x15, 0x93, 0x29, 0xab, 0x2f,
        0xaa, 0x30, 0xb9, 0xb3, 0x32, 0xba, 0x09, 0x42, 0x51, 0x4d, 0x10, 0x08, 0x68, 0x83, 0x30, 0x65, 0x1b, 0x16, 0xe6, 0x72, 0x30, 0x48, 0x1a, 0x22, 0x46, 0xd2, 0x36, 0x04, 0xdb, 0x86, 0xc1, 0xe2,
        0x80, 0x09, 0xc2, 0x18, 0x58, 0x1b, 0x04, 0xc6, 0xdb, 0x50, 0x28, 0x4b, 0x07, 0x7c, 0x55, 0xd8, 0xd8, 0xec, 0xda, 0x5c, 0xd2, 0xc8, 0xca, 0xdc, 0xe8, 0xa6, 0x04, 0x41, 0x15, 0x32, 0x3c, 0x17,
        0xbb, 0x32, 0xb9, 0xb9, 0xb4, 0x37, 0xb7, 0x29, 0x01, 0xd1, 0x84, 0x0c, 0xcf, 0xc5, 0x2e, 0x8c, 0xcd, 0xae, 0x4c, 0x6e, 0x4a, 0x40, 0xd4, 0x21, 0xc3, 0x73, 0x99, 0x43, 0x0b, 0x23, 0x2b, 0x93,
        0x6b, 0x7a, 0x23, 0x2b, 0x63, 0x9b, 0x12, 0x18, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x48, 0x1d, 0x32, 0x3c, 0x97, 0x32, 0x37,
        0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba, 0xb9, 0x29, 0xc1, 0x07, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01,
        0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1,
        0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07,
        0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0,
        0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84,
        0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05,
        0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62,
        0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07,
        0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef,
        0x50, 0x0f, 0xf4, 0x30, 0x83, 0x81, 0xc8, 0x01, 0x1f, 0xdc, 0x40, 0x1c, 0xe4, 0xa1, 0x1c, 0xc2, 0x61, 0x1d, 0xdc, 0x40, 0x1c, 0xe4, 0x01, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x31, 0x00,
        0x00, 0x00, 0xb6, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0x41, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xc0, 0xe4, 0x30, 0x88, 0xb0, 0x21,
        0x0d, 0xfa, 0xf8, 0xc8, 0x6d, 0x1b, 0x81, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x5f, 0xe4, 0x30, 0x21, 0x01, 0x3c, 0x36, 0xd0, 0x0c, 0x97, 0xef,
        0x3c, 0xfe, 0x80, 0x48, 0x02, 0x10, 0x0d, 0xa6, 0xe0, 0x0c, 0x97, 0xef, 0x3c, 0xfe, 0xe0, 0x4c, 0xb8, 0x5f, 0xdc, 0xb6, 0x21, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x8f, 0x48, 0x21, 0xb1, 0x3c, 0x44,
        0x54, 0x02, 0x0d, 0xe1, 0x23, 0xb5, 0x05, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x5b, 0x02, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x10, 0x05, 0x0c,
        0x2b, 0x01, 0x0c, 0x25, 0xd0, 0x10, 0xda, 0xe4, 0x17, 0xb7, 0x6d, 0x05, 0xd7, 0x70, 0xf9, 0xce, 0xe3, 0x47, 0x80, 0xb5, 0x51, 0x45, 0x41, 0x44, 0xa5, 0x03, 0x0c, 0x7e, 0x71, 0xdb, 0x26, 0x50,
        0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81, 0x52, 0xd3, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x66, 0xf0, 0x0c, 0x97, 0xef, 0x3c, 0x3e, 0xd5, 0x00, 0x11, 0xe6, 0x17, 0xb7, 0x6d, 0x00, 0x04,
        0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x9f, 0x00, 0x00, 0x00, 0x13, 0x04, 0x45, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x00, 0xa4, 0xd4, 0xc0, 0x08, 0x00, 0x15,
        0x45, 0x50, 0x02, 0x44, 0x14, 0xf4, 0x43, 0x29, 0x07, 0x14, 0x7d, 0x40, 0x21, 0x94, 0xc2, 0x0c, 0x40, 0xe9, 0x0e, 0x14, 0x41, 0xc1, 0x15, 0xed, 0x40, 0xc1, 0x0e, 0x14, 0x69, 0x40, 0x61, 0x94,
        0xa5, 0x40, 0xd1, 0x14, 0x4b, 0x61, 0x02, 0x15, 0x67, 0x00, 0x0d, 0x63, 0x04, 0x20, 0x08, 0x82, 0xf0, 0x37, 0x46, 0x00, 0x82, 0x20, 0x7d, 0x8a, 0xc1, 0x18, 0x01, 0x08, 0x82, 0x30, 0x1b, 0x06,
        0x63, 0x04, 0x20, 0x08, 0x82, 0x32, 0x18, 0x8c, 0x11, 0x80, 0x20, 0x08, 0xc2, 0x60, 0x30, 0x46, 0x00, 0x82, 0x20, 0x88, 0x7f, 0xe4, 0xcd, 0x21, 0xa0, 0xc1, 0x37, 0x87, 0xc0, 0x06, 0x1f, 0x00,
        0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xd4, 0xc1, 0x85, 0x06, 0x68, 0x20, 0x06, 0xd5, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0x76, 0x80, 0xa5, 0x41, 0x1a, 0x70, 0xd6, 0x09,
        0x04, 0x9d, 0x40, 0x90, 0x05, 0x6e, 0x70, 0x02, 0x0b, 0x06, 0x10, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0xc1, 0x83, 0x07, 0x97, 0x1b, 0x8c, 0xc1, 0x18, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1,
        0x27, 0x07, 0x57, 0x90, 0x8c, 0x18, 0x28, 0x00, 0x08, 0x82, 0x81, 0x94, 0x07, 0x55, 0x30, 0xc0, 0xc1, 0x18, 0x7c, 0xa3, 0x09, 0x01, 0x30, 0x62, 0x50, 0x00, 0x20, 0x08, 0x06, 0x18, 0x1e, 0x78,
        0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0xed, 0xc1, 0x25, 0x4c, 0x37, 0x40, 0x81, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x9a, 0x1e, 0x94, 0x41, 0x50, 0x07, 0x23, 0x06, 0x07, 0x00, 0x82,
        0x60, 0xa0, 0xed, 0x81, 0x19, 0x08, 0x78, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x1a, 0x1f, 0x9c, 0xc1, 0x20, 0x06, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xa0, 0xf5, 0x01, 0x1a, 0x10, 0x77,
        0x30, 0x62, 0xd0, 0x00, 0x20, 0x08, 0x06, 0x5c, 0x1f, 0x8c, 0x01, 0x31, 0x08, 0x01, 0x65, 0x5d, 0xd8, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x50, 0xa2, 0xd0, 0x25, 0xc3, 0x11, 0xc1, 0x46, 0x7c,
        0xb3, 0x0c, 0x81, 0x10, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0xc1, 0x72, 0x0a, 0x63, 0x20, 0x07, 0xb3, 0x04, 0xc2, 0x88, 0x41, 0x01, 0x80, 0x20, 0x18, 0x60, 0xa2, 0x80, 0x06, 0xd3, 0x0d, 0x99,
        0x12, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0x36, 0x0a, 0x6e, 0x10, 0xf8, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x68, 0xa4, 0xf0, 0x06, 0x42, 0x28, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82,
        0x81, 0x56, 0x0a, 0x70, 0x30, 0xac, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x68, 0xa6, 0x10, 0x07, 0x04, 0x28, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0x01, 0x67, 0x0a, 0x6c, 0x40, 0x0c, 0x42,
        0xd0, 0x7d, 0x60, 0x10, 0x06, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xf0, 0xb4, 0x02, 0x1b, 0x94, 0x02, 0x1e, 0xe0, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x7c, 0xa7, 0xc0, 0x06, 0x41, 0x37,
        0x62, 0xb0, 0x00, 0x20, 0x08, 0x06, 0x16, 0x2b, 0x98, 0x41, 0x50, 0x0a, 0x58, 0x29, 0xc8, 0x81, 0x1d, 0x0c, 0x37, 0x64, 0xa5, 0x00, 0x06, 0xc3, 0x0d, 0x42, 0x1e, 0x88, 0x41, 0x09, 0xc1, 0xce,
        0x32, 0x0c, 0x44, 0x30, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x0b, 0x2d, 0xc0, 0xc1, 0x1f, 0xcc, 0x12, 0x10, 0x23, 0x06, 0x05, 0x00, 0x82, 0x60, 0x80, 0xbd, 0x42, 0x1d, 0x4c, 0x37, 0x98, 0xc1,
        0x15, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0x06, 0x0b, 0x7b, 0x10, 0xac, 0xc2, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x68, 0xb1, 0xc0, 0x07, 0x82, 0x2b, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82,
        0x81, 0x26, 0x0b, 0x7d, 0x30, 0xe0, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x68, 0xb3, 0xe0, 0x07, 0x44, 0x2b, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0x01, 0x37, 0x0b, 0x79, 0x40, 0x0c, 0x42,
        0xa0, 0x06, 0x6c, 0xd0, 0x06, 0x6e, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x2f, 0xf8, 0x01, 0x2c, 0xc0, 0x82, 0x2a, 0xac, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xbc, 0xe0,
        0x07, 0xb0, 0x00, 0x0b, 0xa3, 0x70, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x0b, 0x7e, 0x00, 0x0b, 0xb0, 0x40, 0x0a, 0xce, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0xbc, 0xe0, 0x07,
        0xb0, 0x00, 0x0b, 0xac, 0x10, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    const D3D12_SHADER_BYTECODE ps = SHADER_BYTECODE(ps_discard_atomic_loop);

    memset(&desc, 0, sizeof(desc));
    desc.rt_format = DXGI_FORMAT_R32G32B32A32_FLOAT;
    desc.rt_width = 2;
    desc.rt_height = 2;
    desc.no_pipeline = true;
    desc.no_root_signature = true;
    if (!init_test_context(&context, &desc))
        return;
    command_list = context.list;
    queue = context.queue;

    if (!context_supports_dxil(&context))
    {
        skip("Context does not support DXIL.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&shader_model, 0, sizeof(shader_model));
    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    if (FAILED(ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_SHADER_MODEL, &shader_model, sizeof(shader_model))) ||
        shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_6)
    {
        skip("Device does not support SM 6.6.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&rs_desc, 0, sizeof(rs_desc));
    rs_desc.Flags = D3D12_ROOT_SIGNATURE_FLAG_CBV_SRV_UAV_HEAP_DIRECTLY_INDEXED;
    create_root_signature(context.device, &rs_desc, &context.root_signature);

    init_pipeline_state_desc_dxil(&pso_desc, context.root_signature, DXGI_FORMAT_R32G32B32A32_FLOAT, NULL, &ps, NULL);
    hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pso_desc,
            &IID_ID3D12PipelineState, (void **)&context.pipeline_state);
    ok(hr == S_OK, "Failed to create state, hr %#x.\n", hr);

    input_keys = create_upload_buffer(context.device, sizeof(alpha_keys), alpha_keys);
    atomic_buffer = create_default_buffer(context.device, 4 * sizeof(uint32_t),
            D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    readback_buffer = create_readback_buffer(context.device, 4 * sizeof(uint32_t));

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 2);

    memset(&srv_desc, 0, sizeof(srv_desc));
    srv_desc.Format = DXGI_FORMAT_UNKNOWN;
    srv_desc.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
    srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
    srv_desc.Buffer.NumElements = 4;
    srv_desc.Buffer.StructureByteStride = 4;
    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    ID3D12Device_CreateShaderResourceView(context.device, input_keys, &srv_desc, cpu_handle);

    memset(&uav_desc, 0, sizeof(uav_desc));
    uav_desc.Format = DXGI_FORMAT_UNKNOWN;
    uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
    uav_desc.Buffer.NumElements = 4;
    uav_desc.Buffer.StructureByteStride = 4;

    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    cpu_handle.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
    ID3D12Device_CreateUnorderedAccessView(context.device, atomic_buffer, NULL, &uav_desc, cpu_handle);

    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 1, &heap);
    ID3D12GraphicsCommandList_ClearRenderTargetView(command_list, context.rtv, &white.x, 0, NULL);
    ID3D12GraphicsCommandList_OMSetRenderTargets(command_list, 1, &context.rtv, false, NULL);
    ID3D12GraphicsCommandList_SetGraphicsRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_IASetPrimitiveTopology(command_list, D3D_PRIMITIVE_TOPOLOGY_TRIANGLELIST);
    set_viewport(&context.viewport, 0.0f, 0.0f, 2.0f, 2.0f, 0.0f, 1.0f);
    ID3D12GraphicsCommandList_RSSetViewports(command_list, 1, &context.viewport);
    ID3D12GraphicsCommandList_RSSetScissorRects(command_list, 1, &context.scissor_rect);
    ID3D12GraphicsCommandList_DrawInstanced(command_list, 3, 1, 0, 0);

    transition_resource_state(command_list, atomic_buffer,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    ID3D12GraphicsCommandList_CopyBufferRegion(command_list, readback_buffer, 0, atomic_buffer, 0, 4 * sizeof(uint32_t));
    transition_resource_state(command_list, context.render_target,
            D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_texture_readback_with_command_list(context.render_target, 0, &rb, queue, command_list);

    {
        uint32_t *ptr;
        hr = ID3D12Resource_Map(readback_buffer, 0, NULL, (void**)&ptr);
        ok(SUCCEEDED(hr), "Failed to map buffer, hr #%x.\n", hr);
        if (SUCCEEDED(hr))
        {
            static const uint32_t expected[] = { 101, 0, 0, 101 };
            for (i = 0; i < ARRAY_SIZE(expected); i++)
                ok(ptr[i] == expected[i], "Atomic value %u, expected %u, got %u.\n", i, expected[i], ptr[i]);
            ID3D12Resource_Unmap(readback_buffer, 0, NULL);
        }
    }

    for (y = 0; y < 2; y++)
    {
        for (x = 0; x < 2; x++)
        {
            const struct vec4 *value;
            struct vec4 expected;

            value = get_readback_vec4(&rb, x, y);

            if (x == 0 && y == 0)
            {
                expected.x = 1.0f;
                expected.y = 4321.0f;
                expected.z = 4881.0f;
                expected.w = 8881.0f;
            }
            else
                expected = white;

            ok(compare_vec4(value, &expected, 0), "Mismatch pixel %u, %u, (%f %f %f %f) != (%f %f %f %f).\n",
                    x, y, expected.x, expected.y, expected.z, expected.w,
                    value->x, value->y, value->z, value->w);
        }
    }

    ID3D12DescriptorHeap_Release(heap);
    release_resource_readback(&rb);
    ID3D12Resource_Release(readback_buffer);
    ID3D12Resource_Release(input_keys);
    ID3D12Resource_Release(atomic_buffer);
    destroy_test_context(&context);
}

