/* uogetopt.c: somewhat GNU compatible reimplementation of getopt(_long) */


/*
   Copyright (C) 1999 Uwe Ohse

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*
 * main differences:
 * + new feature: --version 
 * + new feature: --help  (standard help text, one line per option)
 * + new feature: --help OPTION (maybe somewhat more)
 * + new feature: --longhelp (all --help OPTION)
 * - no "return in order"
 * + really no support for i18n.
 * + small code, about 60% of GNU C library stuff.
 * + it doesn't do black magic, like that GNU stuff.
 *
 * Note that the following statement from the GNU getopt.c was the cause
 * for reinventing the wheel:
 *    Bash 2.0 puts a special variable in the environment for each
 *    command it runs, specifying which ARGV elements are the results of
 *    file name wildcard expansion and therefore should not be
 *    considered as options.
 * i decided that this wheel is to broken to be reused. Think of that
 * "-i" trick.
 */

#include "config.h"
#include <string.h>
#include <errno.h>
#include <unistd.h> /* write */
#include <stdlib.h> /* getenv */
#include "uogetopt.h"
#include "str2num.h"
#include "uostr.h"


#define L longname
#define S shortname
#define LLEN 80
#define OFFSET 29 /* keep me < LLEN */

void 
uogetopt_out(int iserr,const char *s)
{
	int fd=1;
	if (iserr)
		fd=2;
	write(fd,s,strlen(s));
}

static void 
uogetopt_outn(void (*out)(int iserr,const char *),
	int iserr,const char *s,const char *t,const char *u,const char *v)
{ out(iserr,s); if (t) out(iserr,t); if (u) out(iserr,u); if (v) out(iserr,v); }
#define O2(E,s,t) do { uogetopt_outn(out,(E),(s),(t),0,0); } while (0)
#define O3(E,s,t,u) do { uogetopt_outn(out,(E),(s),(t),(u),0); } while (0)
#define O4(E,s,t,u,v) do { uogetopt_outn(out,(E),(s),(t),(u),(v)); } while (0)

static void
uogetopt_hlong(void (*out)(int iserr,const char *),const char *argv0, const char *s,struct uogetopt *o)
{
	int len;
	long l;
	len=str2long(s,&l,0);
	if (len<=0) { O4(1,argv0,": ",s," is not an signed long\n");_exit(2);} 
	*(long *)o->var=l;
}

static void
uogetopt_hulong(void (*out)(int iserr,const char *),const char *argv0, const char *s,struct uogetopt *o)
{
	int len;
	unsigned long ul;
	len=str2ulong(s,&ul,0);
	if (len<=0) { O4(1,argv0,": ",s," is not an unsigned long\n");_exit(2);} 
	*(unsigned long *)o->var=ul;
}

static void
uogetopt_describe(void (*out)(int iserr,const char *),struct uogetopt *o, int full)
{
/* 
  -s, --size                 print size of each file, in blocks
  -S                         sort by file size
      --sort=WORD            ctime -c, extension -X, none -U, size -S,
123456789012345678901234567890
I don't really know for what the spaces at the start at the line are
for, but i suppose someone will need them ...
*/
	int l;
	const char *p;
	char *q;
	char buf[LLEN+1];
	char minbuf[2];
	out(0,"  ");
	if (o->S) { buf[0]='-'; buf[1]=o->S; buf[2]=0; out(0,buf); } /* -X */
	else out(0,"  ");

	if (o->S && o->L) out(0,", --");
	else if (o->L)    out(0,"  --");
	else              out(0,"    ");

	l=8;

	if (o->L) {l+=strlen(o->L);out(0,o->L);}

	if (o->argtype!=UOGO_FLAG && o->argtype!=UOGO_FLAGOR) {
		if (o->L) out(0,"=");
		else      out(0," ");
		l++;
		if (o->paraname) {out(0,o->paraname);l+=strlen(o->paraname);}
		else if (o->argtype==UOGO_STRING) {out(0,"STRING");l+=6;}
		else {out(0,"NUMBER");l+=6;}
	}
	/* fill up with spaces (at least one) */
	q=buf; do { *q++=' '; l++; } while (l<OFFSET); *q=0; 
	out(0,buf);

	/* 1. line of help */
	if (!o->help) {out(0,"???\n"); return; }
	p=strchr(o->help,'\n');
	if (!p || !p[1]) { out(0,o->help); if (!p) out(0,"\n"); return;}
	minbuf[1]=0;
	p=o->help;
	do {minbuf[0]=*p;out(0,minbuf); } while (*p++!='\n');

	if (!full) return;
	memset(buf,' ',OFFSET); buf[OFFSET]=0;
	do { 
		out(0,buf);
		do {minbuf[0]=*p++;out(0,minbuf);} while (*p && p[-1]!='\n');
		if (!*p) out(0,"\n");
	} while (*p && p[1]);
}

static void	
uogetopt_printver(void (*out)(int iserr,const char *),
	const char *prog, const char *package, const char *version)
{
		out(0,prog);
		if (package) { O3(0, " (",package,")"); }
		O3(0," ",version,"\n");
}

void 
uogetopt(const char *prog, const char *package, const char *version, 
	int *argc, char **argv, void (*out) P__((int iserr,const char *)), 
	const char *head, struct uogetopt *opts, const char *tail)
{
	int i;
	int posix;
	int newargc;
	int ocount;
	int is_longhelp;
	if (!argv[1]) return;
	for (i=0;opts[i].S || opts[i].L;i++) {
		if (!opts[i].var) {
			/* no use to waste code for detailed error messages, this only
			 * happens during development.
			 */
			out(1,"NULL variable address in uogetopt()\n");
			_exit(1);
		}
	}
	ocount=i;
	if (argv[1] && 0==strcmp(argv[1],"--version")) {
		uogetopt_printver(out,prog,package,version);
		_exit(0);
	}
	is_longhelp=(0==strcmp(argv[1],"--longhelp"));
	if (argv[1] && (is_longhelp || 0==strcmp(argv[1],"--help"))) {
		if (argv[2]) { 
			if (argv[2][0]=='-') argv[2]++;
			if (argv[2][0]=='-' && argv[2][1]) argv[2]++;
			for (i=0;i<ocount;i++) {
				if (opts[i].L && 0==strcmp(opts[i].L,argv[2])) break;
				if (opts[i].S && !argv[2][1] && argv[2][0]==opts[i].S) break;
			}
			if (i==ocount) { (out)(0,"no such option\n"); _exit(1); }
			if (!opts[i].help) { (out)(0,"no help available\n"); _exit(1); }
			uogetopt_describe(out,&opts[i],1);
			_exit(0);
		}
		uogetopt_printver(out,prog,package,version);
		if (head) {out(0,head); out(0,"\n");}
		else { O3(0,"usage: ",prog," [options]\n");}
		for (i=0;i<ocount;i++)
			uogetopt_describe(out,&opts[i],is_longhelp);
		if (tail) {out(0,tail); out(0,"\n");}
		_exit(0);
	}

	posix=!!getenv("POSIXLY_CORRECT");
	newargc=1;
	for (i=1;i<*argc;i++) {
		if (*argv[i]!='-') {
			if (posix) { 
		  copyrest:
				while (argv[i]) argv[newargc++]=argv[i++];
				argv[newargc]=0;
				*argc=newargc;
				return;
			}
			argv[newargc++]=argv[i];
			continue;
		}
		if (argv[i][1]=='-') { 
			int j;
			int ioff=1;
			char *para;
			struct uogetopt *o;
			para=strchr(argv[i],'=');
			if (para) { *para++=0;ioff=0;}
			else {para=argv[i+1];}
			if (!argv[i][2]) { i++; goto copyrest; }
			o=opts;
			for (j=0;j<ocount;o++,j++)
				if (o->L && 0==strcmp(o->L,argv[i]+2)) break;
			if (j==ocount) {O4(1,argv[0],": illegal option -- ",argv[i],"\n");_exit(2);}
			if ((o->argtype==UOGO_FLAG || o->argtype==UOGO_FLAGOR) && ioff==0) /* --x=y */
				{ O4(1,argv[0],": option doesn't allow an argument -- ",argv[i],"\n");_exit(2);}
			if (o->argtype==UOGO_FLAG) { *((int*)o->var)=o->val; continue;}
			if (o->argtype==UOGO_FLAGOR) { *((int*)o->var)|=o->val; continue;}
			if (!para) {O4(1,argv[0],": option requires an argument -- ",argv[i]+2,"\n");_exit(2);}
			if (o->argtype==UOGO_STRING) { *((char **)o->var)=para; i+=ioff; continue;}
			if (o->argtype==UOGO_ULONG) { uogetopt_hulong(out,argv[0],para,o); i+=ioff; continue; }
			if (o->argtype==UOGO_LONG) { uogetopt_hlong(out,argv[0],para,o); i+=ioff; continue; }
			if (o->argtype==UOGO_CALLBACK) { 
				void (*fn)(struct uogetopt *,const char *)=o->var;
				(*fn)(o,para);
				i+=ioff; continue;
			}
			O4(1,argv[0],": internal problem, unknown option type for `",argv[i],"'\n");
			_exit(2);
		} else { 
			int j;
			for (j=1;argv[i][j];j++) { /* all chars */
				char c=argv[i][j];
				int k;
				int nexti;
				char *p;
				char optstr[2];
				struct uogetopt *o;
				optstr[0]=c;optstr[1]=0;
				o=opts;
				for (k=0;k<ocount;k++,o++)
					if (o->S && o->S==c) break;
				if (k==ocount) {
					O4(1,argv[0],": illegal option -- ",optstr,"\n");
					_exit(2);
				}
				if (o->argtype==UOGO_FLAG) { *((int*)o->var)=o->val; continue;}
				if (o->argtype==UOGO_FLAGOR) { *((int*)o->var)|=o->val; continue;}
				/* options with arguments, first get arg */
				nexti=i; p=argv[i]+j+1; 
				if (!*p) { p=argv[i+1];
					if (!p) {O4(1,argv[0],": option requires an argument -- ",optstr,"\n");_exit(2);}
					nexti=i+1; }
				if (o->argtype==UOGO_STRING) { *((char **)o->var)=p; i=nexti; break; }
				if (o->argtype==UOGO_ULONG) { uogetopt_hulong(out,argv[0],p,o); i=nexti; break; }
				if (o->argtype==UOGO_LONG) { uogetopt_hlong(out,argv[0],p,o); i=nexti; break; }
				if (o->argtype==UOGO_CALLBACK) { 
					void (*fn)(struct uogetopt *,const char *)=o->var;
					(*fn)(o,p);
					i+=nexti; break;
				}
				O4(1,argv[0],": internal problem, unknown option type for `",optstr,"'\n");
				_exit(2);
			}
		}
	}
	*argc=newargc;
	argv[newargc]=0;
}

#ifdef TEST

int flag=0;
char *string;
unsigned long ul=17;
signed long lo=18;
struct uogetopt myopts[]={
	{'t',"test",UOGO_FLAG,&flag,1,"test option\n"},
	{'l',"longtest",UOGO_FLAGOR,&flag,2,"test option with some characters\nand more than one line, of which one is really long, of course, just to test some things, like crashs or not crash, life or 42\n"},
	{'L',0,UOGO_FLAG,&flag,2,"test option with some characters\nand more than two lines\nwith not much information\nbut a missing newline at the end: 43"},
	{'M',0,UOGO_FLAG,&flag,2,"test option with some characters\nand two lines: 44\n"},
	{'N',0,UOGO_FLAG,&flag,2,"test option with some characters\nand two lines, no nl at end: 45"},
	{'u',"ulong",UOGO_ULONG,&ul,2,"set a ulong variable\n","Ulong"},
	{'n',"nulong",UOGO_LONG,&lo,2,"set a long variable\n","long"},
	{'s',"string",UOGO_STRING,&string,2,"set a string variable\n","String"},
	{0,0}
};
int 
main(int argc, char **argv)
{

	printf("old argc=%d\n",argc);
	uogetopt("uogetopt","misc","0.1",
		&argc, argv,
		writeerr,
		"head\n",myopts,"tail\n");
	printf("flag=%d\n",flag);
	printf("string=%s\n",string ? string : "NULL");
	printf("ulong=%lu\n",ul);
	printf("long=%ld\n",lo);
	printf("new argc=%d\n",argc);
	{ int i; for (i=0;argv[i];i++) printf("argv%d: %s\n",i,argv[i]);}
	exit(0);
}
#endif
