/* pdv.c
   $Id: pdv.c,v 1.10 2001/09/30 01:45:35 gwiley Exp $
   Glen Wiley <gwiley@ieee.org>
  
   this is the stub-prefix for the payload delivery vehicle, it is
	appended with the payload and meta-data - it can then be executed
	to "deliver" the payload

   Copyright (c)1999,2000,2001, Glen Wiley
  
	Permission is hereby granted, free of charge, to any person
	obtaining a copy of this software and associated documentation
	files (the "Software"), to deal in the Software without
	restriction, including without limitation the rights to use, copy,
	modify, merge, publish, distribute, sublicense, and/or sell copies
	of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions: The above
	copyright notice and this permission notice shall be included in
	all copies or substantial portions of the Software.  THE SOFTWARE
	IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
	NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
	HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
	WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.
*/

#if HAVE_CONFIG_H
 #include "config.h"
#endif
#include <errno.h>
#if HAVE_FCNTL_H
 #include <fcntl.h>
#endif
#if HAVE_GETOPT_H
 #include <getopt.h>
#endif
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
 #include <string.h>
#else
 #include <strings.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#if HAVE_UNISTD_H
 #include <unistd.h>
#endif
#include "pdv.h"
#include "version.h"

/* basename of this program - used in error messages */
char *g_prg = NULL;

/* if !=0 then progress messages are printed */
int g_verbose = 0;

/* length of the buffer used to extract the payload */
#define EXTR_BUF_LEN 4096

/* assumed maximum length of a command line */
#define LONGEST_CMD_LINE 1024

/* print error message and exit */
void fatalError(int err, const char *fmt, ...);

/* print usage message, or user supplied help message */
void usage(char *hlpmsg);

void readPayloadData(const char *fn, struct payload_st *pld);
void deliverPayload(const char *fn, struct payload_st *pld);
int  agreementPrompt(const char *msg);

/*---------------------------------------- main */
int
main(int argc, char *argv[])
{
	char   opt;
	int    retval   = 0;
	int    showhelp = 0;
	struct payload_st pld_data;

	/* get our basename for the sake of error reporting */

	g_prg  = strrchr(argv[0], '/');
	if(g_prg == NULL)
		g_prg = argv[0];
	else
		g_prg++;

	/* get command line options */

	while((opt=getopt(argc, argv, "?hvV")) != EOF)
	{
		switch(opt)
		{
			case 'v':
				printf("%s\n", g_versionstr);
				exit(0);
				break;
				
			case 'V':
				g_verbose = 1;
				break;
				
			case '?':
			case 'h':
				showhelp = 1;
				break;

			default:
				showhelp = 1;
				fatalError(EINVAL, "");
		}
	}

	readPayloadData(argv[0], &pld_data);

	if(showhelp)
	{
		usage(pld_data.hlpmsg);
		exit(1);
	}

	/* only deliver the payload if the user agrees to the agreement or if
	   there is no agreement message */

	if(pld_data.agrmsg == NULL 
	 || strlen(pld_data.agrmsg) == 0
	 || agreementPrompt(pld_data.agrmsg) == 0)
	{
		deliverPayload(argv[0], &pld_data);

		if(pld_data.cmd && pld_data.cmd[0] != '\0')
		{
			system(pld_data.cmd);
		}
	}

	return retval;
} /* main */

/*---------------------------------------- fatalError
  display error message and exit */
void
fatalError(int err, const char *fmt, ...)
{
	char    *newfmt;
	va_list alist;

	fflush(stdout);

	newfmt = (char *) malloc(strlen(fmt) + 80);
	if(err)
		sprintf(newfmt, "\n%s: %s, %s\n", g_prg, fmt, strerror(err));
	else
		sprintf(newfmt, "\n%s: %s\n", g_prg, fmt);

	va_start(alist, fmt);
	vfprintf(stderr, newfmt, alist);
	va_end(alist);

	free(newfmt);

	exit((err ? err : 1));
} /* fatalError */

/*---------------------------------------- readPayloadData
  read meta-data regarding the payload from the end of the file */
void
readPayloadData(const char *fn, struct payload_st *pld)
{
	int     fd;
	int     off_tsz;
	ssize_t rdres;
	off_t   flen;
	char    *buf;
	char    *p;
	long    bytes;

	if(g_verbose)
		printf("reading payload data...");

	off_tsz = sizeof(off_t);

	fd = open(fn, O_RDONLY);
	if(fd == -1)
		fatalError(errno, "unable to read package file %s", fn);

	/* locate the meta data offset */

	flen = lseek(fd, -1 * off_tsz, SEEK_END) + off_tsz;
	if(read(fd, &(pld->metadatastart), off_tsz) != off_tsz)
		fatalError(errno, "error reading meta data", fn);

	if(pld->metadatastart == 0L)
		fatalError(0, "no payload has been packaged, please run pdvmkpg");

	/* read the meta data into buffer */

	bytes = flen - pld->metadatastart;
	buf = (char *) malloc(bytes);
	if(buf == NULL)
		fatalError(errno, "error reading meta data", fn);

	p = buf;
	lseek(fd, pld->metadatastart, SEEK_SET);
	while((rdres = read(fd, p, bytes)) > 0)
		p += rdres;
	if(rdres == -1)
		fatalError(errno, "error reading meta data", fn);

	/* extract the output file name and permissions from the buffer */

	pld->fn_out = buf;
	p = buf + strlen(buf) + 1;

	memcpy(&(pld->payldmode), p, sizeof(mode_t));
	p = p + sizeof(mode_t);

	/* extract the executable file name from the buffer */

	pld->cmd = p;
	p = p + strlen(p) + 1;

	/* extract the help message from the buffer */

	pld->hlpmsg = p;
	p = p + strlen(p) + 1;

	/* extract the agreement message from the buffer */

	pld->agrmsg = p;
	p = p + strlen(p) + 1;

	/* extract flags from the buffer */

	pld->iscompressed = (*p == 0 ? 0 : 1);
	p++;
	pld->isatar       = (*p == 0 ? 0 : 1);
	p++;

	/* extract payload file offset from the buffer */

	memcpy(&(pld->payloadstart), p, off_tsz);

	/* we don't free(buf) because it is now used by our payload data struct */
	close(fd);

	if(g_verbose)
		printf("done\n");

	return;
} /* readPayloadData */

/*---------------------------------------- deliverPayload
*/
void
deliverPayload(const char *fn, struct payload_st *pld)
{
	int    done = 0;
	int    fd_in;
	FILE   *fh_out;
	int    nbytes;
	off_t  curpos = 0;
	char   *buf;
	char   cmd[LONGEST_CMD_LINE];

	printf("extracting %s...", pld->fn_out);

	buf = (char *) malloc(EXTR_BUF_LEN);
	if(buf == NULL)
		fatalError(errno, "error extracting payload");

	fd_in = open(fn, O_RDONLY);
	if(fd_in == -1)
		fatalError(errno, "error extracting payload");

	/* build the command we will use to help extract the payload */

	strcpy(cmd, "");

	if(pld->iscompressed)
		strcat(cmd, "uncompress -c ");

	if(pld->isatar)
	{
		if(pld->iscompressed)
			strcat(cmd, "|");
		strcat(cmd, "tar -xf -");
	}
	else
	{
		strcat(cmd, "> ");
		strcat(cmd, pld->fn_out);
	}

	/* open pipe to our command */

	if(pld->isatar || pld->iscompressed)
		fh_out = popen(cmd, "w");
	else
		fh_out = fopen(pld->fn_out, "w+");

	if(fh_out == NULL)
		fatalError(errno, "error extracting payload");

	fd_in = open(fn, O_RDONLY);
	if(fd_in != -1)
	{
		curpos = lseek(fd_in, pld->payloadstart, SEEK_SET);
		if(curpos != pld->payloadstart)
			fatalError(errno, "error extracting payload");

		while(!done && (nbytes = read(fd_in, buf, EXTR_BUF_LEN)) > 0)
		{
			curpos += nbytes;
			if(curpos >= pld->metadatastart)
			{
				nbytes -= curpos - pld->metadatastart;
				done = 1;
			}
			fwrite(buf, nbytes, 1, fh_out);
		}

		if(nbytes < 0)
			fatalError(errno, "error extracting payload");

		close(fd_in);
	}
	else
		fatalError(errno, "error extracting payload");

	if(pld->isatar || pld->iscompressed)
		pclose(fh_out);
	else
		fclose(fh_out);

	/* if the payload was not filtered through tar then we need to
	   restore its mode (in case it will be executed */

	if(! pld->isatar)
		chmod(pld->fn_out, pld->payldmode);

	printf("done\n");

	return;
} /* deliverPayload */

/*---------------------------------------- usage
  if hlpmsg != NULL then it will be used in place of this
  usage message
*/
void
usage(char *hlpmsg)
{
	FILE *fh;

	if(hlpmsg)
	{
		fh = popen("more", "w");
		if(fh == NULL)
			fatalError(errno, "error attempting to pipe to more");

		fprintf(fh, hlpmsg);
		pclose(fh);
	}
	else
	{
		printf("%s\n"
 		 "USAGE: %s [-hv]\n"
		 "\n"
		 "This is the PDV (Payload Delivery Vehicle) executable stub.\n"
		 "If this stub has been used to create a package file then running\n"
		 "this program with no arguments will usually install the package\n"
		 "that this was meant to distribute.\n"
		 "\n"
		 "-h  print this message and exit\n"
		 "-v  print version and exit\n"
		 "\n"
	 , g_versionstr, g_prg);
	}

	return;
} /* usage */

/*---------------------------------------- agreementPrompt
  display agreement message (assumed to be non-NULL) and prompt user to 
  accept the agreement

  return 0 on success (user accepts agreement)
*/
int
agreementPrompt(const char *msg)
{
	int  retval = 0;
	int  done   = 0;
	char resp[5];
	char *p;

	/* display the message */
	/* TODO pipe message through more/less */

	printf("%s\n", msg);

	/* prompt for user acceptance (must be "yes") */
	while(!done)
	{
		printf("Enter 'yes' to accept this agreement, 'no' to decline: ");
		fgets(resp, 5, stdin);
		p = strchr(resp, '\n');
		if(p != NULL)
			*p = '\0';

		if(strcasecmp(resp, "yes") == 0)
			done = 1;
		if(strcasecmp(resp, "no") == 0)
			done = retval = 1;
	}

	return retval;
} /* agreementPrompt */

/* pdv.c */
