/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.exporter;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.namespace.QName;
import org.jabref.logic.exporter.Exporter;
import org.jabref.logic.exporter.SaveException;
import org.jabref.logic.importer.fileformat.bibtexml.Article;
import org.jabref.logic.importer.fileformat.bibtexml.Book;
import org.jabref.logic.importer.fileformat.bibtexml.Booklet;
import org.jabref.logic.importer.fileformat.bibtexml.Conference;
import org.jabref.logic.importer.fileformat.bibtexml.Entry;
import org.jabref.logic.importer.fileformat.bibtexml.File;
import org.jabref.logic.importer.fileformat.bibtexml.Inbook;
import org.jabref.logic.importer.fileformat.bibtexml.Incollection;
import org.jabref.logic.importer.fileformat.bibtexml.Inproceedings;
import org.jabref.logic.importer.fileformat.bibtexml.Manual;
import org.jabref.logic.importer.fileformat.bibtexml.Mastersthesis;
import org.jabref.logic.importer.fileformat.bibtexml.Misc;
import org.jabref.logic.importer.fileformat.bibtexml.Phdthesis;
import org.jabref.logic.importer.fileformat.bibtexml.Proceedings;
import org.jabref.logic.importer.fileformat.bibtexml.Techreport;
import org.jabref.logic.importer.fileformat.bibtexml.Unpublished;
import org.jabref.logic.util.StandardFileType;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BibTeXMLExporter
extends Exporter {
    private static final String BIBTEXML_NAMESPACE_URI = "http://bibtexml.sf.net/";
    private static final Locale ENGLISH = Locale.ENGLISH;
    private static final Logger LOGGER = LoggerFactory.getLogger(BibTeXMLExporter.class);
    private JAXBContext context;

    public BibTeXMLExporter() {
        super("bibtexml", "BibTexXML", StandardFileType.XML);
    }

    @Override
    public void export(BibDatabaseContext databaseContext, Path resultFile, Charset encoding, List<BibEntry> entries) throws SaveException {
        Objects.requireNonNull(databaseContext);
        Objects.requireNonNull(entries);
        if (entries.isEmpty()) {
            return;
        }
        File file = new File();
        for (BibEntry bibEntry : entries) {
            String type;
            Entry entry = new Entry();
            bibEntry.getCiteKeyOptional().ifPresent(entry::setId);
            switch (type = bibEntry.getType().toLowerCase(ENGLISH)) {
                case "article": {
                    this.parse(new Article(), bibEntry, entry);
                    break;
                }
                case "book": {
                    this.parse(new Book(), bibEntry, entry);
                    break;
                }
                case "booklet": {
                    this.parse(new Booklet(), bibEntry, entry);
                    break;
                }
                case "conference": {
                    this.parse(new Conference(), bibEntry, entry);
                    break;
                }
                case "inbook": {
                    this.parseInbook(new Inbook(), bibEntry, entry);
                    break;
                }
                case "incollection": {
                    this.parse(new Incollection(), bibEntry, entry);
                    break;
                }
                case "inproceedings": {
                    this.parse(new Inproceedings(), bibEntry, entry);
                    break;
                }
                case "mastersthesis": {
                    this.parse(new Mastersthesis(), bibEntry, entry);
                    break;
                }
                case "manual": {
                    this.parse(new Manual(), bibEntry, entry);
                    break;
                }
                case "misc": {
                    this.parse(new Misc(), bibEntry, entry);
                    break;
                }
                case "phdthesis": {
                    this.parse(new Phdthesis(), bibEntry, entry);
                    break;
                }
                case "proceedings": {
                    this.parse(new Proceedings(), bibEntry, entry);
                    break;
                }
                case "techreport": {
                    this.parse(new Techreport(), bibEntry, entry);
                    break;
                }
                case "unpublished": {
                    this.parse(new Unpublished(), bibEntry, entry);
                    break;
                }
                default: {
                    LOGGER.warn("unexpected type appeared");
                }
            }
            file.getEntry().add(entry);
        }
        this.createMarshallerAndWriteToFile(file, resultFile);
    }

    private void createMarshallerAndWriteToFile(File file, Path resultFile) throws SaveException {
        try {
            if (this.context == null) {
                this.context = JAXBContext.newInstance((Class[])new Class[]{File.class});
            }
            Marshaller marshaller = this.context.createMarshaller();
            marshaller.setProperty("jaxb.formatted.output", (Object)Boolean.TRUE);
            marshaller.marshal((Object)file, resultFile.toFile());
        }
        catch (JAXBException e) {
            throw new SaveException(e);
        }
    }

    private void parseInbook(Inbook inbook, BibEntry bibEntry, Entry entry) {
        Map<String, String> fieldMap = bibEntry.getFieldMap();
        for (Map.Entry<String, String> entryField : fieldMap.entrySet()) {
            String value = entryField.getValue();
            String key = entryField.getKey();
            if ("year".equals(key)) {
                try {
                    XMLGregorianCalendar calendar = DatatypeFactory.newInstance().newXMLGregorianCalendar(value);
                    JAXBElement year = new JAXBElement(new QName(BIBTEXML_NAMESPACE_URI, "year"), XMLGregorianCalendar.class, (Object)calendar);
                    inbook.getContent().add(year);
                }
                catch (DatatypeConfigurationException e) {
                    LOGGER.error("A configuration error occured");
                }
                continue;
            }
            if ("number".equals(key)) {
                JAXBElement number = new JAXBElement(new QName(BIBTEXML_NAMESPACE_URI, "number"), BigInteger.class, (Object)new BigInteger(value));
                inbook.getContent().add(number);
                continue;
            }
            JAXBElement element = new JAXBElement(new QName(BIBTEXML_NAMESPACE_URI, key), String.class, (Object)value);
            inbook.getContent().add(element);
        }
        entry.setInbook(inbook);
    }

    private <T> void parse(T entryType, BibEntry bibEntry, Entry entry) {
        List<Method> declaredSetMethods = this.getListOfSetMethods(entryType);
        Map<String, String> fieldMap = bibEntry.getFieldMap();
        for (Map.Entry<String, String> entryField : fieldMap.entrySet()) {
            String value = entryField.getValue();
            String key = entryField.getKey();
            for (Method method : declaredSetMethods) {
                String methodNameWithoutSet = method.getName().replace("set", "").toLowerCase(ENGLISH);
                try {
                    if ("year".equals(key) && key.equals(methodNameWithoutSet)) {
                        try {
                            XMLGregorianCalendar calendar = DatatypeFactory.newInstance().newXMLGregorianCalendar(value);
                            method.invoke(entryType, calendar);
                        }
                        catch (DatatypeConfigurationException e) {
                            LOGGER.error("A configuration error occured");
                        }
                        break;
                    }
                    if ("number".equals(key) && key.equals(methodNameWithoutSet)) {
                        method.invoke(entryType, new BigInteger(value));
                        break;
                    }
                    if (!key.equals(methodNameWithoutSet)) continue;
                    method.invoke(entryType, value);
                    break;
                }
                catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                    LOGGER.error("Could not invoke method", e);
                }
            }
            List<Method> entryMethods = this.getListOfSetMethods(entry);
            for (Method method : entryMethods) {
                String simpleClassName;
                String methodWithoutSet = method.getName().replace("set", "");
                if (!methodWithoutSet.equals(simpleClassName = entryType.getClass().getSimpleName())) continue;
                try {
                    method.invoke((Object)entry, entryType);
                }
                catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                    LOGGER.warn("Could not set the type to the entry");
                }
            }
        }
    }

    private <T> List<Method> getListOfSetMethods(T entryType) {
        return Arrays.stream(entryType.getClass().getDeclaredMethods()).filter(method -> method.getName().startsWith("set")).collect(Collectors.toList());
    }
}

