//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4VParticleChange
//
// Class description:
//
// Abstract class for "Particle Change".
//
// The ParticleChange class contains the results after invocation
// of a physics process. This includes final states of parent particle
// (momentum, energy, etc) and secondary particles generated by the
// interaction.
// The tracking assumes that all the values of energy and momentum are
// in global reference system, therefore all the needed Lorentz
// transformations must have been already computed when filling the
// data-members of this class.
//
// This abstract class has following four virtual methods:
//     virtual G4Step* UpdateStepForAtRest(G4Step* Step);
//     virtual G4Step* UpdateStepForAlongStep(G4Step* Step);
//     virtual G4Step* UpdateStepForPostStep(G4Step* Step);
//     virtual void Initialize(const G4Track&);
// The UpdateStep methods return the pointer to the G4Step after updating
// the given Step information by using final state information of the track
// given by a physics process.
// User must add methods to keep the final state information in the derived
// class as well as implement UpdateStep() methods wished to use.
//
// The Initialize() methods are provided to refresh the final state
// information and should be called by each process at the beginning of DoIt().

// Author: Hisaya Kurashige, 23 March 1998
// --------------------------------------------------------------------
#ifndef G4VParticleChange_hh
#define G4VParticleChange_hh 1

#include <vector>
#include "globals.hh"
#include "G4ios.hh"
#include "G4TrackStatus.hh"
#include "G4SteppingControl.hh"
#include "G4Step.hh"
#include "G4Track.hh"

class G4VParticleChange
{
  public:

    G4VParticleChange();

    virtual ~G4VParticleChange() = default;

    G4VParticleChange(const G4VParticleChange& right) = delete;
    G4VParticleChange& operator=(const G4VParticleChange& right) = delete;

  // --- the following methods are for updating G4Step ---

    virtual G4Step* UpdateStepForAtRest(G4Step* Step);
    virtual G4Step* UpdateStepForAlongStep(G4Step* Step);
    virtual G4Step* UpdateStepForPostStep(G4Step* Step);
      // Return the pointer to the G4Step after updating the Step information
      // by using final state information of the track given by a physics
      // process

    virtual void Initialize(const G4Track&);
      // This methods will be called by each process at the beginning
      // of DoIt() if necessary

  // --- the following methods are for TruePathLength ---

    inline G4double GetTrueStepLength() const;
    inline void ProposeTrueStepLength(G4double truePathLength);
      // Get/Propose theTrueStepLength

  // --- the following methods are for LocalEnergyDeposit ---

    inline G4double GetLocalEnergyDeposit() const;
    inline void ProposeLocalEnergyDeposit(G4double anEnergyPart);
      // Get/Propose the locally deposited energy

  // --- the following methods are for nonIonizingEnergyDeposit  ---

    inline G4double GetNonIonizingEnergyDeposit() const;
    inline void ProposeNonIonizingEnergyDeposit(G4double anEnergyPart);
      // Get/Propose the non-ionizing deposited energy

  // --- the following methods are for TrackStatus ---

    inline G4TrackStatus GetTrackStatus() const;
    inline void ProposeTrackStatus(G4TrackStatus status);
      // Get/Propose the final TrackStatus of the current particle

    inline const G4Track* GetCurrentTrack() const;
      // Get primary track pointer

  // --- the following methods are for management of SteppingControl ---

    inline G4SteppingControl GetSteppingControl() const;
    inline void ProposeSteppingControl(G4SteppingControl StepControlFlag);
      // Set/Propose a flag to control stepping manager behaviour

  // --- the following methods are for management of initial/last step

    inline G4bool GetFirstStepInVolume() const;
    inline G4bool GetLastStepInVolume() const;
    inline void ProposeFirstStepInVolume(G4bool flag);
    inline void ProposeLastStepInVolume(G4bool flag);

  // --- the following methods are for management of secondaries ---

    inline void Clear();
      // Clear the contents of this objects
      // This method should be called after the Tracking(Stepping)
      // manager removes all secondaries in theListOfSecondaries

    inline void SetNumberOfSecondaries(G4int totSecondaries);
      // SetNumberOfSecondaries must be called just before AddSecondary()
      // in order to secure memory space for theListOfSecondaries
      // This method resets theNumberOfSecondaries to zero
      // (that will be incremented at every AddSecondary() call)

    inline G4int GetNumberOfSecondaries() const;
      // Returns the number of secondaries current stored in G4TrackFastVector

    inline G4Track* GetSecondary(G4int anIndex) const;
      // Returns the pointer to the generated secondary particle,
      // which is specified by an Index, no check on boundary is performed

    void AddSecondary(G4Track* aSecondary);
      // Adds a secondary particle to theListOfSecondaries

  // --- the following methods are for management of weights ---

    inline G4double GetWeight() const;
    inline G4double GetParentWeight() const;
      // Get weight of the parent (i.e. current) track

    inline void ProposeWeight(G4double finalWeight);
    inline void ProposeParentWeight(G4double finalWeight);
      // Propose new weight of the parent (i.e. current) track
      // As for AlongStepDoIt, the parent weight will be set
      // in accumulated manner, i.e. - If two processes propose
      // weight of W1 and W2 respectively for the track with initial
      // weight of W0 the final weight is set to: (W1/W0) * (W2/W0) * W0

    inline void SetSecondaryWeightByProcess(G4bool);
    inline G4bool IsSecondaryWeightSetByProcess() const;
      // In default (fSecondaryWeightByProcess flag is false),
      // the weight of secondary tracks will be set to the parent weight
      // If fSecondaryWeightByProcess flag is true, the weight of secondary
      // tracks will not be changed by the ParticleChange (i.e. the process
      // determine the secondary weight)
      // NOTE:
      // Make sure that only one process in AlongStepDoIt() proposes the
      // parent weight; if several processes in AlongStepDoIt() propose
      // the parent weight and add secondaties with fSecondaryWeightByProcess
      // is set to false, secondary weights may be wrong

    void SetParentWeightByProcess(G4bool);
    G4bool IsParentWeightSetByProcess() const;
      // Obsolete

  // --- Dump and debug methods ---

    virtual void DumpInfo() const;
      // Print out information

    inline void SetVerboseLevel(G4int vLevel);
    inline G4int GetVerboseLevel() const;

    virtual G4bool CheckIt(const G4Track&);
      // CheckIt method for general control in debug regime

    inline void ClearDebugFlag();
    inline void SetDebugFlag();
    inline G4bool GetDebugFlag() const;
      // CheckIt method is activated if debug flag is set
      // and 'G4VERBOSE' is defined

  protected:

    G4Step* UpdateStepInfo(G4Step* Step);
      // Update the G4Step specific attributes
      // (i.e. SteppingControl, LocalEnergyDeposit, and TrueStepLength)

    inline void InitializeLocalEnergyDeposit();
    inline void InitializeSteppingControl();
    inline void InitializeParentWeight(const G4Track&);
    inline void InitializeStatusChange(const G4Track&);
    inline void InitializeSecondaries();
    inline void InitializeFromStep(const G4Step*);

    inline G4double ComputeBeta(G4double kinEnergy); 

    G4bool CheckSecondary(G4Track&);
      // CheckSecondary method is provided to control secondary track 
      // in debug regime

    G4double GetAccuracyForWarning() const;
    G4double GetAccuracyForException() const;

  protected:

    static const G4double accuracyForWarning;
    static const G4double accuracyForException;
    static const G4int maxError;
      // accuracy levels

    const G4Track* theCurrentTrack = nullptr;

    G4TrackStatus theStatusChange = fAlive;
      // The changed (final) track status of a given particle

    G4SteppingControl theSteppingControlFlag = NormalCondition;
      // A flag to control stepping manager behavior

    G4double theLocalEnergyDeposit = 0.0;
      // It represents the part of the energy lost for discrete
      // or semi-continuous processes which is due to secondaries
      // not generated because they would have been below their cut
      // threshold.
      // The sum of the locally deposited energy + the delta-energy
      // coming from the continuous processes gives the
      // total energy loss localized in the current Step

    G4double theNonIonizingEnergyDeposit = 0.0;
      // Non-ionizing energu deposit is defined as a part of local
      // energy deposit, which does not cause ionization of atoms

    G4double theTrueStepLength = 0.0;
      // The value of "True" Step Length

    G4double theParentWeight = 1.0;
      // Weight ofparent track

    G4double theParentGlobalTime = 0.0;
      // Global time of the parent.
      // This is used only for checking

    G4int theNumberOfSecondaries = 0;
      // The total number of secondaries produced by each process.

    G4int theSizeOftheListOfSecondaries = 0;
      // TheSizeOftheListOfSecondaries;

    G4int verboseLevel = 1;
      // The Verbose level

    G4int nError = 0;

    G4bool theFirstStepInVolume = false;
    G4bool theLastStepInVolume = false;
      // Flag for initial/last step

    G4bool isParentWeightProposed = false;
      // Flag for Weight of parent track
    G4bool fSetSecondaryWeightByProcess = false;
      // Flag for setting weight of secondaries

    G4bool debugFlag = false;

    std::vector<G4Track*> theListOfSecondaries;
      // The vector of secondaries
};

#include "G4VParticleChange.icc"

#endif
